/* 
   Copyright  1998, 1999 Enbridge Pipelines Inc. 
   Copyright  1999 Dave Carrigan
   All rights reserved.

   This module is free software; you can redistribute it and/or modify
   it under the same terms as Apache itself. This module is
   distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE. The copyright holder of this
   module can not be held liable for any general, special, incidental
   or consequential damages arising out of the use of the module.

   $Id: auth_ldap_config.c,v 1.1 1999/07/05 21:22:50 dave Exp $

*/

#include "auth_ldap.h"

extern module MODULE_VAR_EXPORT auth_ldap_module;

void *
create_auth_ldap_dir_config(pool *p, char *d)
{
  auth_ldap_config_rec *sec = 
    (auth_ldap_config_rec *)ap_pcalloc(p, sizeof(auth_ldap_config_rec));

  sec->auth_authoritative = 1;

  sec->have_ldap_url = 0;
  sec->ldc = NULL;
  sec->host = NULL;
  sec->url = "";
  sec->binddn = NULL;
  sec->bindpw = NULL;
  sec->needbind = 1;

#ifdef AUTH_LDAP_FRONTPAGE_HACK
  sec->frontpage_hack = 0;
  sec->frontpage_hack_pwfile = NULL;
#endif

  sec->dn = "";
  sec->user_is_dn = 0;

  return sec;
}

/* 
   Use the ldap url parsing routines to break up the ldap url into
   host and port, and find/make a pointer to a LDAPconnection struct
*/
const char *
parse_auth_ldap_url(cmd_parms *cmd, auth_ldap_config_rec *sec, char *url)
{
  int result;
  LDAPURLDesc *urld;

  result = ldap_url_parse(url, &(urld));
  if (result != 0) {
    switch (result) {
    case LDAP_URL_ERR_NOTLDAP:
      return "LDAP URL does not begin with ldap://";
    case LDAP_URL_ERR_NODN:
      return "LDAP URL does not have a DN";
    case LDAP_URL_ERR_BADSCOPE:
      return "LDAP URL has an invalid scope";
    case LDAP_URL_ERR_MEM:
      return "Out of memory parsing LDAP URL";
    default:
      return "Could not parse LDAP URL";
    }
  }
  sec->url = ap_pstrdup(cmd->pool, url);

  /* Set all the values, or at least some sane defaults */
  if (sec->host) {
    char *p = ap_palloc(cmd->pool, strlen(sec->host) + strlen(urld->lud_host) + 2);
    strcpy(p, urld->lud_host);
    strcat(p, " ");
    strcat(p, sec->host);
    sec->host = p;
  } else {
    sec->host = urld->lud_host? ap_pstrdup(cmd->pool, urld->lud_host) : "localhost";
  }
  sec->basedn = urld->lud_dn? ap_pstrdup(cmd->pool, urld->lud_dn) : "";
  sec->attribute = urld->lud_attrs? ap_pstrdup(cmd->pool, urld->lud_attrs[0]) : "uid";
  sec->scope = urld->lud_scope? urld->lud_scope : LDAP_SCOPE_SUBTREE;
  if (urld->lud_filter[0] == '(') {
    /* 
       Get rid of the surrounding parens; later on when generating the
       filter, they'll be put back.
    */
    sec->filter = ap_pstrdup(cmd->pool, urld->lud_filter+1);
    sec->filter[strlen(sec->filter)-1] = '\0';
  } else {
    sec->filter = ap_pstrdup(cmd->pool, urld->lud_filter);
  }
  if (strncmp(url, "ldaps", 5) == 0) {
    ap_log_error(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, 
		 cmd->server, "{%d} requesting secure LDAP", (int)getpid());
#ifdef WITH_SSL
    sec->port = urld->lud_port? urld->lud_port : LDAPS_PORT;
    sec->secure = 1;
#else
    return "Secure LDAP (ldaps://) not supported. Rebuild auth_ldap";
#endif
  } else {
    ap_log_error(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, 
		 cmd->server, "{%d} not requesting secure LDAP", (int)getpid());
#ifdef WITH_SSL
    sec->secure = 0;
#endif
    sec->port = urld->lud_port? urld->lud_port : LDAP_PORT;
  }

  ap_log_error(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, 
	       cmd->server, "version %s: Successfully parsed an url %s", 
	       auth_ldap_version, url);
  sec->have_ldap_url = 1;
  return NULL;
}

const char *
auth_ldap_set_cache_ttl(cmd_parms *cmd, void *dummy, char *ttl)
{
  auth_ldap_server_conf *c = 
    (auth_ldap_server_conf *)ap_get_module_config(cmd->server->module_config, 
						  &auth_ldap_module);
  c->cache_ttl = atol(ttl);
  return NULL;
}

const char *
auth_ldap_set_cache_size(cmd_parms *cmd, void *dummy, char *size)
{
  auth_ldap_server_conf *c = 
    (auth_ldap_server_conf *)ap_get_module_config(cmd->server->module_config, 
						  &auth_ldap_module);
  c->cache_size = atol(size);
  return NULL;
}

const char *
auth_ldap_set_opcache_ttl(cmd_parms *cmd, void *dummy, char *ttl)
{
  auth_ldap_server_conf *c = 
    (auth_ldap_server_conf *)ap_get_module_config(cmd->server->module_config, 
						  &auth_ldap_module);
  c->opcache_ttl = atol(ttl);
  return NULL;
}

const char *
auth_ldap_set_opcache_size(cmd_parms *cmd, void *dummy, char *size)
{
  auth_ldap_server_conf *c = 
    (auth_ldap_server_conf *)ap_get_module_config(cmd->server->module_config, 
						  &auth_ldap_module);
  c->opcache_size = atol(size);
  return NULL;
}

const char *
auth_ldap_set_compare_flag(cmd_parms *cmd, void *dummy, char *flag)
{
  auth_ldap_server_conf *c = 
    (auth_ldap_server_conf *)ap_get_module_config(cmd->server->module_config, 
						  &auth_ldap_module);
  c->cache_compares = 1;
  if (strcasecmp(flag, "off") == 0 ||
      strcasecmp(flag, "no") == 0)
    c->cache_compares = 0;
  return NULL;
}

#ifdef WITH_SSL
const char *
auth_ldap_set_certdbpath(cmd_parms *cmd, void *dummy, char *path)
{
  auth_ldap_server_conf *c = 
    (auth_ldap_server_conf *)ap_get_module_config(cmd->server->module_config, 
						  &auth_ldap_module);
  c->have_certdb = 1;
  if (ldapssl_client_init(path, NULL) != 0)
    return "Could not initialize SSL client";
  else
    return NULL;
}
#endif

#ifdef AUTH_LDAP_FRONTPAGE_HACK
/* 
   When using the frontpage hack, we need to know the name of the auth
   password file. So, we handle the AuthUserFile directive so that we
   can get the filename, but we always decline so that mod_auth gets
   to use the directive as well.
*/
const char 
*auth_ldap_snarf_pwfile(cmd_parms *cmd, auth_ldap_config_rec *sec, char *f)
{
  sec->frontpage_hack_pwfile = ap_pstrdup(cmd->pool, f);
  return DECLINE_CMD;
}

#endif

void *
create_auth_ldap_config(pool *p, server_rec *s)
{
  auth_ldap_server_conf *c = 
    (auth_ldap_server_conf *)ap_pcalloc(p, sizeof(auth_ldap_server_conf));
  c->cache_ttl = 600;
  c->cache_size = 1024 * 10;
  c->opcache_ttl = 600;
  c->opcache_size = 1024;
  c->cache_compares = 1;
  c->mtx = ap_create_mutex(NULL);
  c->ldapconnections = NULL;
#ifdef WITH_SSL
  c->have_certdb = 0;
#endif
  return c;
}

