;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;
;;;                Centre for Speech Technology Research                  ;;
;;;                     University of Edinburgh, UK                       ;;
;;;                         Copyright (c) 1998                            ;;
;;;                        All Rights Reserved.                           ;;
;;;                                                                       ;;
;;;  Permission to use, copy, modify, distribute this software and its    ;;
;;;  documentation for research, educational and individual use only, is  ;;
;;;  hereby granted without fee, subject to the following conditions:     ;;
;;;   1. The code must retain the above copyright notice, this list of    ;;
;;;      conditions and the following disclaimer.                         ;;
;;;   2. Any modifications must be clearly marked as such.                ;;
;;;   3. Original authors' names are not deleted.                         ;;
;;;  This software may not be used for commercial purposes without        ;;
;;;  specific prior written permission from the authors.                  ;;
;;;                                                                       ;;
;;;  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        ;;
;;;  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      ;;
;;;  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   ;;
;;;  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     ;;
;;;  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    ;;
;;;  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   ;;
;;;  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          ;;
;;;  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       ;;
;;;  THIS SOFTWARE.                                                       ;;
;;;                                                                       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Support for cluster unit selection method
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  This has some quite special requirements at present and is not
;;;  a general usable system yet.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  There are four stages to the test run
;;;     generate distance tables
;;;     generate features used for clustered (doesn't change much)
;;;     generate cluster trees
;;;     generate test examples
;;;  There are four functions to do these steps
;;;     (do_disttabs)
;;;     (do_features)
;;;     (do_cluster_trees)
;;;     (do_test_synth)
;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Distance table functions
(define (get_stdweights stds weights)
"generate the actual weights from standard deviations and weights."
  (mapcar
   (lambda (a b)
     (* b (/ 1.0 a)))
   stds weights))

(define (do_disttabs)
"Generate distance tables."
  (the_voice)
  (do_disttabs_list (ducs.remove_nulltypes (ducs.unittypes))))

(define (ducs.remove_nulltypes types)
"(ducs.remove_nulltypes types)
Remove types which have zero occurrences."
  (cond
   ((null types) nil)
   ((ducs.occurs (car types))
    (cons (car types) (ducs.remove_nulltypes (cdr types))))
   (t
    (ducs.remove_nulltypes (cdr types)))))

(define (do_disttabs_list phones)
"Gener distance tables for named phones."
  (the_voice)
  (ducs.make_unit_disttabs
   phones
   (list
    (list 'dir disttab_dir)
;    (list 'weights (get_stdweights ac_stds ac_weights))
    (list 'weights ac_weights)
    (list 'get_stds_per_unit t)
    (list 'ac_left_context ac_left_context_size)
    (list 'dur_pen_weight ac_dur_pen_weight))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Generating feature files

(define (find_utt_stream unit_desc)
"Find the utterance and stream that goes with this unit."
  (let ((utt (car (cdr (assoc (car unit_desc) all-utts))))
	(rname (car (cdr (cdr (cdr unit_desc)))))
	(rstart (car (cdr (cdr (cdr (cdr unit_desc))))))
	(streamitem nil))
    (if utt
	(begin
	  (mapcar
	   (lambda (s)
	     (if (and (string-equal rname (utt.streamitem.feat utt s "name"))
		      (approx-equal? rstart (utt.streamitem.feat utt s "start") 0.01))
		 (set! streamitem s)))
	   (utt.stream utt 'Segment))
	  (if streamitem
	      (list utt streamitem)
	      (begin
		(format t "Can't find streamitem %l\n" unit_desc)
		nil)))
	(begin
	  (format t "Can't find utt %l\n" unit_desc)
	  nil))))

(defvar utts_skeleton "utts/seg/%s.seg.utt")

(define (load-all-utts files)
  "(load-all-utts files)
Load in all the utterances so they may be searched."
  (let ((utt_seg))
    (set! all-utts
	  (mapcar
	   (lambda (fname)
	     (set! utt_seg (Utterance Text fname))
	     (utt.load utt_seg 
		       (string-append db_dir 
				      (format nil utts_skeleton fname)))
	     (list fname utt_seg))
	   files))
    t))
    
(define (output_features unit pos fd)
  "Lookup an output features"
  (let ((utt_stream (find_utt_stream (ducs.unit unit))))
    (if utt_stream
	(begin
	  (format fd "%d " pos)
	  (mapcar
	   (lambda (feat)
	     (format fd "%s " (utt.streamitem.feat (car utt_stream) 
					       (car (cdr utt_stream))
					       feat)))
	   feats)
	  (format fd "\n")))))

(define (save_features_for_unittype ut fname)
  "Save features for all occurrences of ut into fname"
  (let ((fd (fopen fname "wb"))
	(pos 0))
    (mapcar
     (lambda (u) 
       (output_features u pos fd)
       (set! pos (+ 1 pos)))
     (ducs.occurs ut))
    (fclose fd)))

(define (do_features)
"Generate the feature files of the wagon clustering code"
  (the_voice)
  (load-all-utts database_files)
  (mapcar
   (lambda (unit)
     (format t "Outputing unit %s\n" unit)
     (save_features_for_unittype 
      unit 
      (string-append feat_dir unit ".feats")))
   (ducs.remove_nulltypes (ducs.unittypes))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Tree building functions

(defvar wagon-balance-size 0)

(define (build_tree unit)
"Build tree with Wagon for this unit."
  (let ((command 
	 (format nil "%s -desc %s -data %s -balance %s -distmatrix %s -stop %s -output %s"
		 wagon-progname
		 wagon_field_desc
		 (string-append feat_dir unit ".feats")	
		 wagon-balance-size
		 (string-append disttab_dir unit ".disttab")
		 wagon-cluster-size
		 (string-append tree_dir unit ".tree")
		 )))
    (format t "%s\n" command)
    (system command)))

(define (build_trees unittypes)
"Build a tree for each unittype and save them all in named file."
  (mapcar
   (lambda (unit)
     (build_tree unit))
   unittypes))

(defvar cluster_prune_limit 0)

(define (collect_trees unittypes)
"Collect the trees into one file as an assoc list"
  (let ((fd (fopen (string-append tree_dir "all.trees." run-name) "wb")))
    (format fd ";; Autogenerated list of selection trees\n")
    (format fd ";; run-name %s\n" run-name)
    (format fd ";; ac_std %l\n" ac_stds)
    (format fd ";; ac_weight %l\n" ac_weights)
    (format fd ";; dur_pen %l\n" ac_dur_pen_weight)
    (format fd ";; feats %l\n" feats)
    (format fd ";; wagon-cluster-size %d\n" wagon-cluster-size)
    (format fd "(set! ducs_selection_trees '(\n")
    (mapcar
     (lambda (unit)
       (set! tree (car (load (string-append tree_dir unit ".tree") t)))
       (if (> cluster_prune_limit 0)
	   (set! tree (cluster_tree_prune tree cluster_prune_limit)))
       (pprintf (list unit tree) fd))
     unittypes)
    (format fd "))\n")
    (fclose fd)))

(define (do_cluster_trees)
  (the_voice)
  (build_trees (ducs.remove_nulltypes (ducs.unittypes)))
  (collect_trees (ducs.remove_nulltypes (ducs.unittypes))))

(define (cluster_tree_prune_in_line prune_limit)
"(cluster_tree_prune_in_line)
Prune number of units in each cluster in each tree *by* prune_limit,
if negative, or *to* prune_limit, if positive."
  (set! sucs_select_trees 
        (mapcar
	 (lambda (t)
	     (cluster_tree_prune t prune_limit))
	 sucs_select_trees)))

(define (cluster_tree_prune tree prune_limit)
"(cluster_tree_prune TREE PRUNE_LIMIT)
Reduce the number of elements in the (CART) tree leaves to PRUNE_LIMIT
removing the ones further from the cluster centre.  Maybe later this should
have guards on minimum number of units that must remain in the tree and
a per unit type limit."
  (cond
   ((cdr tree)  ;; a question
    (list
     (car tree)
     (cluster_tree_prune (car (cdr tree)) prune_limit)
     (cluster_tree_prune (car (cdr (cdr tree))) prune_limit)))
   (t           ;; tree leave
    (list 
     (list
      (remove_n_worst 
       (car (car tree))
       (if (< prune_limit 0)
	   (* -1 prune_limit)
	   (- (length (car (car tree))) prune_limit)))
      (car (cdr (car tree))))))))

(define (remove_n_worst lll togo)
"(remove_n_worst lll togo)
Remove togo worst items from lll."
  (cond
   ((< togo 0)
    lll)
   ((equal? 0 togo)
    lll)
   (t
    (remove_n_worst
     (remove (worst_unit (cdr lll) (car lll)) lll)
     (- togo 1)))))

(define (worst_unit lll worst_so_far)
"(worst_unit lll worst_so_far)
Returns unit with worst score in list."
  (cond
   ((null lll)
    worst_so_far)
   ((< (car (cdr worst_so_far)) (car (cdr (car lll))))
    (worst_unit (cdr lll) (car lll)))
   (t
    (worst_unit (cdr lll) worst_so_far))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Test a number of utterances

(define (test_a_sent name)
"Do a resynthesis of name."
  (set! utt1 (utt.synth (utt.load.segf0 
			 (string-append db_dir "festival/streams/Segment/" name ".Segment")
			 (string-append db_dir "f0/" name ".f0"))))
)  

(define (test_n_save_sent name)
  "Synthesize a natural utterance and save it"
  (let ((utt (test_a_sent name)))
    (utt.save.wave 
     utt
     (string-append test_save_dir run-name "/" name ".wav"))
    (utt.save.units
     utt
     (string-append test_save_dir run-name "/" name ".units"))))

(define (test_n_save_tts name)
  "Synthesize a test sentence and save it"
  (let ((utt (utt.synth (eval (list 'Utterance 'Text (car (cdr name)))))))
    (utt.save.wave 
     utt
     (string-append test_save_dir run-name "/" (car name) ".wav"))
    (utt.save.units
     utt
     (string-append test_save_dir run-name "/" (car name) ".units"))))

(define (do_voice_select)
"Select the newly trained voice"
   (the_voice)
   (load (string-append tree_dir "all.trees." run-name))
   (set! sucs_select_trees ducs_selection_trees)
   (Parameter.set 'SUCS_Method 'TS_select))

(defvar prune_reduce 0)

(define (do_test_synth)
"Test a number of difference utterances and say the result."
   (do_voice_select)
   (if (null (equal? prune_reduce 0))
       (cluster_tree_prune_in_line prune_reduce))
   (system (string-append "mkdir -p " test_save_dir run-name))
   (mapcar test_n_save_sent testset)
   (mapcar test_n_save_tts tts_testset)
   t)

(provide 'clunits)
