/* 
** Copyright (C) 1994, 1995 Enterprise Integration Technologies Corp.
**         VeriFone Inc./Hewlett-Packard. All Rights Reserved.
** Kevin Hughes, kev@kevcom.com 3/11/94
** Kent Landfield, kent@landfield.com 4/6/97
** 
** This program and library is free software; you can redistribute it and/or 
** modify it under the terms of the GNU (Library) General Public License 
** as published by the Free Software Foundation; either version 2 
** of the License, or any later version. 
** 
** This program is distributed in the hope that it will be useful, 
** but WITHOUT ANY WARRANTY; without even the implied warranty of 
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
** GNU (Library) General Public License for more details. 
** 
** You should have received a copy of the GNU (Library) General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
*/

#include "hypermail.h"
#include <errno.h>

#ifdef TM_IN_SYS_TIME
#include <sys/time.h>
#else
#include <time.h>
#endif

/*
** Does a file exist?
*/

int isfile(char *path)
{
    struct stat stbuf;

    if (stat(path, &stbuf))
        return 0;
    return ((stbuf.st_mode & S_IFMT) == S_IFREG) ? 1 : 0;
}

/*
** This tries to create and chmod a path to a directory.
*/

void checkdir(char *dir)
{
    register char *p;
    struct stat sbuf;
    int ch;

    p = dir;

    if (*p && * p == '/')  /* get off root */
        p++;

    for (;; ++p) {
        if (!*p || *p  == '/') {
            ch = *p;
            *p = '\0';
            if (stat(dir, &sbuf)) {
                if (errno != ENOENT || mkdir(dir, dirmode) < 0) {
                    sprintf(errmsg,"Can't create directory \"%s\".", dir);
                    progerr(NULL);
                    return;
                }
                else if (showprogress)
                    printf("Creating directory \"%s\", mode %o.\n",dir,dirmode);

                if (chmod(dir, dirmode) == -1) {
                    sprintf(errmsg,"Couldn't chmod \"%s\" to %o.",dir,dirmode);
                    progerr(NULL);
                }
            }
            *p = ch;
            if (!*p)
                break;
        }
    }
}

char *getfilecontents(char *filename)
{
    FILE *infile;
    struct stat finfo;
    char *retval;

    if ((infile = fopen(filename, "r")) == NULL)
        return(NULL);

    if (fstat(fileno(infile), &finfo) != 0) {
        (void) fclose(infile);
        return (NULL);
    }

    if (finfo.st_size == 0) {
        (void) fclose(infile);
        return (NULL);
    }
    
    else {
        if (!(retval = (char *)malloc((unsigned) finfo.st_size + 1))) {
            (void) fclose(infile);
            return (NULL);
        }
        if (!fread(retval, (size_t) finfo.st_size, 1, infile)) {
            (void) fclose(infile);
            free(retval);
            return (NULL);
        }
        *(retval + finfo.st_size) = '\0';
    }
    (void) fclose(infile);
    return (retval);
}

/*
** add_char - add supplied character to output string
** Returns:    pointer to next position to add to.
*/

char *add_char(char *ptr, char member_char)
{
    *ptr++ = member_char;
    return(ptr);
}

/*
** add_string - add supplied string to output string
** Returns:    pointer to next position to add to.
*/

char *add_string(char *ptr, char *member_str)
{
    while (*member_str)
        *ptr++ = *member_str++;
    return(ptr);
}

/*
** expand_path - fill in values substituting for magic cookies
**
**      Substitution cookies supported
**
**              %d - two digit day of month (1-28/30/31)
**              %D - three letter day of the week
**              %m - two digit month of year (1-12)
**              %M - three letter month of year (Jan, Feb, ..., Dec)
**              %y - four digit year (1990,..2001)
**
** Returns:   expanded path string
*/

char *dirpath(char *frmptr)
{
    static char format[MAXLINE];
    register char *aptr;
    register char *cp;
    char dtstr[DATESTRLEN];
    char c;
    struct tm *now;
    long clk;

    clk = time((long *)0);
    now = localtime(&clk); 
 
    for (cp = format; cp < format + sizeof(format); cp++)
         *cp = '\0';
 
    aptr = frmptr;
    cp = format;

    while ((c = *aptr++)) {
        if (c == '%') {
           switch (*aptr++) {
               case '%':  /* Add the % character */
                   cp = add_char(cp,'%');
                   continue;
               case 'd':     /* two digit day of month (1-31) */
                   sprintf(dtstr,"%.2d",now->tm_mday);
                   cp = add_string(cp, dtstr);
                   continue;
               case 'D':     /* three letter day of week */
                   cp = add_string(cp, days[now->tm_wday]);
                   continue;
               case 'j':     /* julian date */
                   sprintf(dtstr,"%.3d",now->tm_yday);
                   cp = add_string(cp, dtstr);
                   continue;
               case 'm':     /* two digit month of year (1-12) */
                   sprintf(dtstr,"%.2d",now->tm_mon+1);
                   cp = add_string(cp, dtstr);
                   continue;
               case 'M':     /* three letter month of year */
                   cp = add_string(cp, months[now->tm_mon]);
                   continue;
               case 'y':     /* 4 digit year */
                   sprintf(dtstr,"%.4d",now->tm_year+1900);
                   cp = add_string(cp, dtstr);
                   continue;
               default:
                   cp = add_char(cp,'%');
                   cp = add_char(cp,'?');
                   continue;
           }    /* end switch */
        }
        *cp++ = c;
     }  /* end while */
     return(format);
}

void dump_config(char *mbox, char *label, char *dir, char *archives, 
            char *about, int overwrite, int increment, char *defaultindex)
{
    register char *cp;

    printf("Configuration Values:\n");
    printf("mbox = %s\n",*mbox ? mbox : "Not Used");
    printf("label = %s\n",*label ? label : "Not set");
    printf("archives = %s\n",*archives ? archives : "Not set");
    printf("about = %s\n",*about ? about : "Not set");
    printf("dir = %s\n",*dir ? dir : "Not set");
    printf("defaultindex = %s\n",*defaultindex ? defaultindex : "Not set");
    printf("overwrite = %d\n",overwrite);
    printf("increment = %d\n",increment);
    printf("reverse = %d\n",reverse);
    printf("showprogress = %d\n",showprogress);
    printf("showheaders = %d\n",showheaders);
    printf("showhtml = %d\n",showhtml);
    printf("showbr = %d\n",showbr);
    printf("showhr = %d\n",showhr);
    printf("showreplies = %d\n",showreplies);
    printf("show_msg_links = %d\n",show_msg_links);
    printf("usetable = %d\n",usetable);
    printf("iquotes = %d\n",iquotes);
    printf("eurodate = %d\n",eurodate);
    printf("thrdlevels = %d\n",thrdlevels);
    printf("dirmode = %04o\n",dirmode);
    printf("filemode = %04o\n",filemode);
    printf("mailcommand = %s\n",*mailcommand ? mailcommand : "Not set");
    printf("mailto = %s\n",*mailto ? mailto : "Not set");
    printf("hmail = %s\n",*hmail ? hmail : "Not set");
    printf("domainaddr = %s\n",*domainaddr ? domainaddr : "Not set");
    printf("htmlbody = %s\n",*htmlbody ? htmlbody : "Not set");

    if (!ihtmlheader)
        printf("ihtmlheader = Not set\n");
    else {
        printf("ihtmlheader = ");
        for (cp = ihtmlheader; *cp; cp++)
             putchar(*cp);
    }

    if (!ihtmlfooter)
        printf("ihtmlfooter = Not set\n");
    else {
        printf("ihtmlfooter = ");
        for (cp = ihtmlfooter; *cp; cp++)
             putchar(*cp);
    }

    if (!mhtmlheader)
        printf("mhtmlheader = Not set\n");
    else {
        printf("mhtmlheader = ");
        for (cp = mhtmlheader; *cp; cp++)
             putchar(*cp);
    }

    if (!mhtmlfooter)
        printf("mhtmlfooter = Not set\n");
    else {
        printf("mhtmlfooter = ");
        for (cp = mhtmlfooter; *cp; cp++)
             putchar(*cp);
    }

    print_list("inline_types", inline_types);
    print_list("prefered_types",prefered_types);
    print_list("ignore_types",ignore_types);

    print_list("show_headers", show_headers);

}


/*
** Reads a configuration file if it exists and puts all the right
** values into the right variables.
*/

void readconfigs(char *path, char *mbox, char *label, char *dir, 
                 char *archives, char *about, int *overwrite, 
                 int *increment, char *defaultindex)
{
    char *tmpfpath;
    char tmppath[MAXFILELEN], line[MAXLINE], value[MAXLINE];
    struct passwd *pp;
    FILE *fp;

    if (!strcasecmp(path, "NONE"))
        return;

    if (path[0] == '~') {
        pp = getpwuid(getuid());
        sprintf(tmppath, "%s%s", pp->pw_dir, path + 1);
        if ((fp = fopen(tmppath, "r")) == NULL)
            return;
    }
    else {
        if ((fp = fopen(path, "r")) == NULL)
            return;

        if (showprogress) 
            printf("path: %s\n", path);
    }

    while (fgets(line, MAXLINE, fp) != NULL) {
        if (line[0] == '#' || line[0] == '\n')
            continue;
        if (getconfvalue(line, "hm_mbox", value) != NULL)
            strcpy(mbox, value);
        else if (getconfvalue(line, "hm_label", value) != NULL)
            strcpy(label, value);
        else if (getconfvalue(line, "hm_archives", value) != NULL)
            strcpy(archives, value);
        else if (getconfvalue(line, "hm_about", value) != NULL)
            strcpy(about, value);
        else if (getconfvalue(line, "hm_dir", value) != NULL)
            strcpy(dir, dirpath(value));
        else if (getconfvalue(line, "hm_defaultindex", value) != NULL)
            strcpy(defaultindex, value);
        else if (getconfvalue(line, "hm_progress", value) != NULL)
            showprogress = atoi(value);
        else if (getconfvalue(line, "hm_overwrite", value) != NULL)
            *overwrite = atoi(value);
        else if (getconfvalue(line, "hm_increment", value) != NULL)
            *increment = atoi(value);
        else if (getconfvalue(line, "hm_reverse", value) != NULL)
            reverse = atoi(value);
        else if (getconfvalue(line, "hm_showheaders", value) != NULL)
            showheaders = atoi(value);
        else if (getconfvalue(line, "hm_showhtml", value) != NULL)
            showhtml = atoi(value);
        else if (getconfvalue(line, "hm_thrdlevels", value) != NULL)
            thrdlevels = atoi(value);
        else if (getconfvalue(line, "hm_dirmode", value) != NULL)
            dirmode = strtol(value, (char **) NULL, 0);
        else if (getconfvalue(line, "hm_filemode", value) != NULL)
            filemode = strtol(value, (char **) NULL, 0);
        else if (getconfvalue(line, "hm_showbr", value) != NULL)
            showbr = atoi(value);
        else if (getconfvalue(line, "hm_iquotes", value) != NULL)
            iquotes = atoi(value);
        else if (getconfvalue(line, "hm_showhr", value) != NULL)
            showhr = atoi(value);
        else if (getconfvalue(line, "hm_usetable", value) != NULL)
            usetable = atoi(value);
        else if (getconfvalue(line, "hm_eurodate", value) != NULL)
            eurodate = atoi(value);
        else if (getconfvalue(line, "hm_showreplies", value) != NULL)
            showreplies = atoi(value);
        else if (getconfvalue(line, "hm_show_msg_links", value) != NULL)
            show_msg_links = atoi(value);
        else if (getconfvalue(line, "hm_mailcommand", value) != NULL)
            strcpy(mailcommand, value);
        else if (getconfvalue(line, "hm_mailto", value) != NULL)
            strcpy(mailto, value);
        else if (getconfvalue(line, "hm_hmail", value) != NULL)
            strcpy(hmail, value);
        else if (getconfvalue(line, "hm_domainaddr", value) != NULL)
            strcpy(domainaddr, value);
        else if (getconfvalue(line, "hm_body", value) != NULL) 
            strcpy(htmlbody, value);

        /*
        ** potentially get the header and footer files. Check to make
        ** sure the value is a valid file or else assure the original
        ** value is there.
        */

        else if (getconfvalue(line, "hm_ihtmlheaderfile", value) != NULL) {
            if (isfile(value)) {  /* valid file ? */
                if ((tmpfpath = getfilecontents(value)) != NULL) {
                    if (ihtmlheader)
                        free(ihtmlheader);
                    ihtmlheader = tmpfpath;
                }
            }
        } 
        else if (getconfvalue(line, "hm_ihtmlfooterfile", value) != NULL) {
            if (isfile(value)) {  /* valid file ? */
                if ((tmpfpath = getfilecontents(value)) != NULL) {
                    if (ihtmlfooter)
                        free(ihtmlfooter);
                    ihtmlfooter = tmpfpath;
                }
            }
        } 
        else if (getconfvalue(line, "hm_mhtmlheaderfile", value) != NULL) {
            if (isfile(value)) {  /* valid file ? */
                if ((tmpfpath = getfilecontents(value)) != NULL) {
                    if (mhtmlheader)
                        free(mhtmlheader);
                    mhtmlheader = tmpfpath;
                }
            }
        } 
        else if (getconfvalue(line, "hm_mhtmlfooterfile", value) != NULL) {
            if (isfile(value)) {  /* valid file ? */
                if ((tmpfpath = getfilecontents(value)) != NULL) {
                    if (mhtmlfooter)
                        free(mhtmlfooter);
                    mhtmlfooter = tmpfpath;
                }
            }
        } 

        /*
        ** Multi-lined variables 
        */
        else if (getconfvalue(line, "hm_prefered_types", value) != NULL) 
            prefered_types = add_list(prefered_types, value);

        else if (getconfvalue(line, "hm_ignore_types", value) != NULL) 
            ignore_types = add_list(ignore_types, value);

        else if (getconfvalue(line, "hm_inline_types", value) != NULL) 
            inline_types = add_list(inline_types, value);

        else if (getconfvalue(line, "hm_show_headers", value) != NULL) 
            show_headers = add_list(show_headers, value);
    }

#if 0
    dump_config(mbox, label, dir, archives, about, *overwrite, 
                 *increment, defaultindex);
#endif

    fclose(fp);
}
