#include "rstoret.h"
#include "mmglue.h"
#include "compress.h"
#include "swizzler.h"

/* #define DEBUG_LOADING */

struct PHeapHdr *first_on_first( struct VMPageRecord *page );

/*
 *   Functions to load pages from the store on demand
 */

struct page_ref_t {
    UINT_32	base_page_num;	/* the referred-to (base) page */
    unsigned	nth_page;	/* which page, as in a large object */
    int		indirect_q;
    void	*vm_addr;
};

/*  read a page into memory  */

static struct VMPageRecord *(page_ref_table[2100]);

#ifdef DEBUG_LOADING
static obj dns_( struct Decompressor *s, 
		struct VMPageRecord **prt, 
		int l )
{
  obj x = decompress_word(s);
  printf( "swizzling word: %08x (line %d)\n", VAL(x), l );  
  return map_pers_to_local( prt, x );
}

obj decompress_and_swiz( struct Decompressor *s, 
			 struct VMPageRecord **prt )
{
  return dns_( s, prt, 0 );
}

#define decompress_and_swiz(s,prt) dns_(s,prt,__LINE__)

#else
obj decompress_and_swiz( struct Decompressor *s, 
			 struct VMPageRecord **prt )
{
  obj x = OBJ(decompress_word(s));
  return map_pers_to_local( prt, x );
}
#endif

static struct LocationRef decompress_LR( struct Decompressor *s )
{
  union { 
    struct LocationRef lr;
    UINT_32 w[2];
  } overload;

  overload.w[0] = decompress_word(s);
  overload.w[1] = decompress_word(s);
  return overload.lr;
}
/* 
 *  note that the header (hdr) may not be in the same page
 *  as the data being loaded
 */

static void decompress_and_swizzle_pob( RStore *store,
				        struct Decompressor *s,
				        struct VMPageRecord **prt, 
				        struct PHeapHdr *hdr,
				        UINT_32 *dst,
				        UINT_32 from_start,
				        UINT_32 N )
{
  UINT_32 i;
  enum SwizzleMode m = mode_for_object(hdr);

#ifdef DEBUG_LOADING
  printf( "decompress & swizzle pob "
	  "(%p (mode %d), hdr at %p, %u from start, N = %u)\n",
	  dst, m, hdr, from_start, N );
#endif

  switch (m)
    {
	  /* note that a template deswizzles exactly like
	   * a gvec, because the only interesting things
	   * in a template, the code pointer and function
	   * descriptor pointer, are stored as indirect
	   * pointers, which the usual swizzling process
	   * will translate into the appropriate ids.  Note
	   * that this is note a reversible process, because the
	   * translated (swizzled) value looks like an integer.
	   * That's why SWIZ_MODE_TEMPLATE is handled specially
	   * on the write side.
	   */
    case SWIZ_MODE_TEMPLATE:
    case SWIZ_MODE_GVEC:
      {
	obj *d = (obj *)dst;

	for (i=0; i<N; i+=SLOT(1))
	  {
	    *d++ = decompress_and_swiz( s, prt );
	  }
      }
      break;

    case SWIZ_MODE_FLOAT:
    case SWIZ_MODE_UINT32:
    case SWIZ_MODE_BVEC:
      for (i=0; i<N; i+=SLOT(1))
	{
	  *dst++ = decompress_word( s );
	}
      break;

    case SWIZ_MODE_PADDR_VEC:
      i = 0;
      if (from_start == 0)
	{
	  struct PAddrVec *pv = (void *)dst;
	  pv->owner = store;
	  pv->spare = 0;
	  i += sizeof( struct PAddrVec ) - sizeof( struct LocationRef );
	  dst += 2;
	}
      for (; i<N; i+=SLOT(1))
	{
	  *dst++ = decompress_word( s );
	}
      break;
      
    case SWIZ_MODE_ALLOC_AREA:
      assert( from_start == 0 );
      {
	PAllocArea *aa = (PAllocArea *)dst;

	aa->allocfn = parea_alloc;
	aa->entry = decompress_and_swiz( s, prt );
	aa->reserved = decompress_and_swiz( s, prt );
	aa->current_LR = decompress_LR( s );
	aa->parent_LR = decompress_LR( s );

	i = 2 * sizeof(struct LocationRef) + 2 * sizeof(obj);
	dst += i/sizeof(UINT_32);
	for (; i<N; i+=SLOT(1))
	  {
	    *dst++ = decompress_word(s);
	  }

	/* fill in non-stored fields */

	aa->current = NULL;
	aa->owner = store;
      }
      break;
      
    default:
      {
	struct swiz_mode_handler *h = get_swiz_mode_handler( store, m );

	h->swizzle( h, store, s, prt, hdr, dst, from_start, N );
	break;
      }
    }
}

/* our caller has already made sure the first
   page is loaded
*/

static void decompress_and_swizzle_interior_page( RStore *store, 
						  struct VMPageRecord *vmp, 
						  struct VMPageRecord **prt, 
						  struct Decompressor *s,
						  struct VMPageRecord *first_vmp )
{
  struct PHeapHdr *phh;
  UINT_32 N, from_start;
  void *limit;

  phh = first_on_first(first_vmp);

  /* where does the object end in virtual memory? */

  limit = (char *)(phh + 1) + phh->rs_header.pob_size;

  /* and how far into it's data are we going to start? */

  from_start = (char*)vmp->mem_address - (char *)(phh + 1);

  /* try to do a whole page of translation */
     
  N = MM_PAGE_SIZE;

  /* but do less if that would put us past the end of the object */

  if ((char *)vmp->mem_address + N > (char *)limit)
    {
      N = (char *)limit - (char *)vmp->mem_address;
    }

#ifdef DEBUG_LOADING
  printf( "load interior page %08x: at %08x (first is at %08x)\n",
	  vmp->ref.base_page_num + vmp->ref.nth_page,
	  vmp->mem_address,
	  first_vmp->mem_address );
  printf( "\ttranslating %#x bytes, at +%u from start of object\n", 
	  N, from_start );
#endif

  decompress_and_swizzle_pob( store, s, prt, phh, 
				  vmp->mem_address,
				  from_start, N );
}

static void decompress_first_page_hdr( RStore *store,
				       struct VMPageRecord *vmp,
				       struct VMPageRecord **prt,
				       struct Decompressor *s )
{
  struct FirstPageHdr *fph = vmp->mem_address;

  fph->area = PTR_TO_DATAPTR(decompress_and_swiz( s, prt ));
  fph->vmpr = vmp;
  fph->spare1 = fph->spare2 = 0;
}

/* do the first page of a large object,
 * which is just like doing a page of objects except
 *  (1) we know there is only one object on the page
 *  (2) we don't do the WHOLE object, only PAGE_SIZE worth of it
 */

static void decompress_and_swizzle_first_page( RStore *store, 
					       struct VMPageRecord *vmp, 
					       struct VMPageRecord **prt, 
					       struct Decompressor *s )
{
  struct PHeapHdr *phh;
  UINT_32 N;

#ifdef DEBUG_LOADING
  printf( "load first page %08x: at %08x\n",
	  vmp->ref.base_page_num + vmp->ref.nth_page,
	  vmp->mem_address );
#endif

  decompress_first_page_hdr( store, vmp, prt, s );
  phh = first_on_first(vmp);

  phh->mem_size = decompress_word( s );
  phh->pstore_flags = decompress_word( s );
  phh->gc_flag_bits = 0xF;
  phh->rs_header.pob_class = decompress_and_swiz( s, prt );
  phh->rs_header.pob_size = decompress_word( s );
  phh->size_class = &store->the_size_class;

#ifdef DEBUG_LOADING
  printf( "    pob_size = %08x\n", phh->rs_header.pob_size );
#endif

  N = phh->rs_header.pob_size;
  if (N + sizeof(struct FirstPageHdr)
        + sizeof(struct PHeapHdr) 
      > MM_PAGE_SIZE)
    {
      /* quite likely.  The only exception would be if
         there was a lot of unused space at the end
	 of the large object */
      N = MM_PAGE_SIZE - sizeof(struct PHeapHdr)
	  - sizeof(struct FirstPageHdr);
    }

  decompress_and_swizzle_pob( store, s, prt, phh, 
			      (UINT_32 *)(phh+1), 
			      0, N );
}

static void decompress_and_swizzle_page( RStore *store, 
					 struct VMPageRecord *vmp, 
					 struct VMPageRecord **prt, 
					 struct Decompressor *s )
{
  struct PHeapHdr *phh, *limit;

#ifdef DEBUG_LOADING
  printf( "load single page %08x: at %08x\n",
	  vmp->ref.base_page_num + vmp->ref.nth_page,
	  vmp->mem_address );
#endif

  decompress_first_page_hdr( store, vmp, prt, s );

  phh = first_on_first( vmp );
  limit = (struct PHeapHdr *)((char *)vmp->mem_address + MM_PAGE_SIZE);

  while (phh < limit)
    {
      phh->mem_size = decompress_word( s );
      if (phh->mem_size == 0)
	{
	  /* indicates the end of the page */
	  break;
	}

      phh->pstore_flags = decompress_word( s );
      phh->gc_flag_bits = 0xF;
      phh->rs_header.pob_class = decompress_and_swiz( s, prt );
      phh->rs_header.pob_size = decompress_word( s );
      phh->size_class = &store->the_size_class;

      decompress_and_swizzle_pob( store,
				  s,
				  prt,
				  phh,
				  (UINT_32 *)((obj *)(phh+1)),
				  0,
				  phh->rs_header.pob_size );

      /* go on to the next object */
      phh = (struct PHeapHdr *)((char *)phh + (phh->mem_size));
    }
}

 
void load_page( RStore *store, struct VMPageRecord *vmp )
{
    access_t a;
    struct Decompressor s;
    struct VMPageRecord **prt = page_ref_table;
    unsigned i, num_page_refs;
    struct VMPageRecord *fvmp = NULL;

   assert( !vmp->ref.loaded );

    /* make sure that the first page is loaded */

    if (!vmp->ref.first)
      {
	struct PageRef fpr;

	fpr = vmp->ref;
	fpr.first = 1;

	fvmp = get_vmpr( store, &fpr );
	if (!fvmp->ref.loaded)
	  load_page( store, fvmp );
      }

    i = vmp->ref.first 
        ? vmp->ref.base_page_num
	: vmp->ref.base_page_num + vmp->ref.nth_page;
    lss_access( store->lss, i, &a );
#ifdef DEBUG_LOADING
    printf( "accessing page %08x at: %08x for %u bytes\n",
	   i, a.addr, a.bytes );
    printf( "  VMPR base %08x, first %c, indirect %c, nth %u\n",
	    vmp->ref.base_page_num,
	    vmp->ref.first ? 'Y' : 'N', 
	    vmp->ref.indirect ? 'Y' : 'N',
	    vmp->ref.nth_page );
    if (!vmp->ref.first)
      printf( "  FVMPR at %08x\n", fvmp->mem_address );

    {
      unsigned *z = a.addr;
      unsigned *l = (void *)((char *)a.addr + a.bytes);
      
      while (z < l)
	{
	  printf( "%08x: %08x %08x %08x %08x\n", 
		  (char*)z - (char *)a.addr, z[0], z[1], z[2], z[3] );
	  z += 4;
	}
    }
#endif

    if (!a.addr)
      {
	scheme_error( "RStore error: page ~04x_~04x could not be loaded",
		      2,
		      int2fx( i >> 16 ),
		      int2fx( i & 0xFFFF ) );
      }
    init_decompressor( &s, a.addr, a.bytes );

    /* read the page ref table,
       accessing, possibly reserving, the indicated pages
    */
    
    num_page_refs = decompress_word(&s);

#ifdef DEBUG_LOADING
    printf( "%u page refs\n", num_page_refs );
#endif
    for (i=0; i<num_page_refs; i++)
    {
    struct PageRef pr;
    UINT_32 w;
    
	pr.base_page_num = decompress_word( &s );
	w = decompress_word( &s );
	pr.first = (w & 1) ? 1 : 0;
        pr.indirect = (w & 2) ? 1 : 0;

        pr.loaded = 0;
        pr.dirty = 0;
	pr.nth_page = w >> 2;
	prt[i] = get_vmpr( store, &pr );
#ifdef DEBUG_LOADING
        printf( "  page ref[%u] %08x ==> vm %08x\n", 
	        i, pr.base_page_num + (pr.first ? 0 : pr.nth_page),
	        prt[i]->mem_address );
#endif
    }

    /* mark the page writable */
    
    mm_set_prot( vmp->mem_address, MM_PAGE_SIZE, MM_MODE_READ_WRITE );

    /* read the page into memory, decompressing */

    if (vmp->ref.first)
      if (vmp->ref.nth_page > 1)
	decompress_and_swizzle_first_page( store, vmp, prt, &s );
      else
	decompress_and_swizzle_page( store, vmp, prt, &s );
    else
      decompress_and_swizzle_interior_page( store, vmp, prt, &s, fvmp );

    /* halt the reading process and release the resources */
    
    close_decompressor( &s );
    lss_release( store->lss, &a );

    /* clean up the protections, etc. */
    
    mm_set_prot( vmp->mem_address, MM_PAGE_SIZE, MM_MODE_READ_ONLY );

    vmp->ref.loaded = 1;
    vmp->next_loaded = store->first_loaded;
    store->first_loaded = vmp;
}

