/*
 *  arch/ppc/kernel/feature.c
 *
 *  Copyright (C) 1996 Paul Mackerras (paulus@cs.anu.edu.au)
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version
 *  2 of the License, or (at your option) any later version.
 *
 */
#include <linux/types.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/kernel.h>
#include <asm/feature.h>
#include <asm/ohare.h>
#include <asm/io.h>
#include <asm/prom.h>

static u32 feature_bits_pbook[] = {
	0,			/* FEATURE_null */
	OH_SCC_RESET,		/* FEATURE_Serial_reset */
	OH_SCC_ENABLE,		/* FEATURE_Serial_enable */
	OH_SCCA_IO,		/* FEATURE_Serial_IO_A */
	OH_SCCB_IO,		/* FEATURE_Serial_IO_B */
	OH_FLOPPY_ENABLE,	/* FEATURE_SWIM3_enable */
	OH_MESH_ENABLE,		/* FEATURE_MESH_enable */
	OH_IDE_ENABLE,		/* FEATURE_IDE_enable */
	OH_VIA_ENABLE,		/* FEATURE_VIA_enable */
	OH_IDECD_POWER,		/* FEATURE_CD_power */
	OH_BAY_RESET,		/* FEATURE_Mediabay_reset */
	OH_BAY_ENABLE,		/* FEATURE_Mediabay_enable */
	OH_BAY_PCI_ENABLE,	/* FEATURE_Mediabay_PCI_enable */
	OH_BAY_IDE_ENABLE,	/* FEATURE_Mediabay_IDE_enable */
	OH_BAY_FLOPPY_ENABLE,	/* FEATURE_Mediabay_floppy_enable */
	0,			/* FEATURE_BMac_reset */
	0,			/* FEATURE_BMac_IO_enable */
};

/* assume these are the same as the ohare until proven otherwise */
static u32 feature_bits_heathrow[] = {
	0,			/* FEATURE_null */
	OH_SCC_RESET,		/* FEATURE_Serial_reset */
	OH_SCC_ENABLE,		/* FEATURE_Serial_enable */
	OH_SCCA_IO,		/* FEATURE_Serial_IO_A */
	OH_SCCB_IO,		/* FEATURE_Serial_IO_B */
	OH_FLOPPY_ENABLE,	/* FEATURE_SWIM3_enable */
	OH_MESH_ENABLE,		/* FEATURE_MESH_enable */
	OH_IDE_ENABLE,		/* FEATURE_IDE_enable */
	OH_VIA_ENABLE,		/* FEATURE_VIA_enable */
	OH_IDECD_POWER,		/* FEATURE_CD_power */
	OH_BAY_RESET,		/* FEATURE_Mediabay_reset */
	OH_BAY_ENABLE,		/* FEATURE_Mediabay_enable */
	OH_BAY_PCI_ENABLE,	/* FEATURE_Mediabay_PCI_enable */
	OH_BAY_IDE_ENABLE,	/* FEATURE_Mediabay_IDE_enable */
	OH_BAY_FLOPPY_ENABLE,	/* FEATURE_Mediabay_floppy_enable */
	0x80000000,		/* FEATURE_BMac_reset */
	0x60000000,		/* FEATURE_BMac_IO_enable */
};

static u32 *feature_bits = NULL;
static volatile u32 *feature_reg = NULL;

void
feature_init(void)
{
	struct device_node *np;

	np = find_devices("mac-io");
	if (np != NULL) {
		feature_bits = feature_bits_heathrow;
	} else {
		np = find_devices("ohare");
		if (np == NULL)
			return;
		if (find_devices("via-pmu") != NULL) {
			feature_bits = feature_bits_pbook;
		}
		/* else not sure; maybe this is a Starmax? */
	}

	if (np->n_addrs == 0) {
		printk(KERN_ERR "No addresses for %s\n", np->full_name);
		return;
	}

	feature_reg = (volatile u32 *)
		ioremap(np->addrs[0].address + OHARE_FEATURE_REG, 4);

	if (feature_bits == NULL) {
		printk(KERN_INFO "Twiddling the magic ohare bits\n");
		out_le32(feature_reg, STARMAX_FEATURES);
		return;
	}

	feature_clear(FEATURE_Serial_reset);
	feature_set(FEATURE_Serial_enable);
	feature_set(FEATURE_Serial_IO_A);
	feature_set(FEATURE_Serial_IO_B);
	feature_set(FEATURE_MESH_enable);
	feature_set(FEATURE_IDE_enable);
}

void
feature_set(enum system_feature f)
{
	if (f >= FEATURE_last || feature_reg == NULL || feature_bits == NULL)
		return;
	st_le32(feature_reg, ld_le32(feature_reg) | feature_bits[f]);
	udelay(10);
}

void
feature_clear(enum system_feature f)
{
	if (f >= FEATURE_last || feature_reg == NULL || feature_bits == NULL)
		return;
	st_le32(feature_reg, ld_le32(feature_reg) & ~feature_bits[f]);
	udelay(10);
}

int
feature_test(enum system_feature f)
{
	if (f >= FEATURE_last || feature_reg == NULL || feature_bits == NULL)
		return 0;
	return (ld_le32(feature_reg) & feature_bits[f]) != 0;
}
