/* blue6.cxx
     $Id: blue6.cxx,v 1.23 1998/09/07 00:49:30 elf Exp $

   written by Marc Singer
   22 February 1998

   This file is part of the project CurVeS.  See the file README for
   more information.

   Copyright (C) 1997,1998 Marc Singer

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   in a file called COPYING along with this program; if not, write to
   the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA
   02139, USA.

   -----------
   DESCRIPTION
   -----------

   Some classes to handle the ugly details of window management and
   attribute switching on text output devices.  The intention is to
   perform some output optimzation for queues output.  This may or may
   not be implemented depending on the need for more efficient
   output.  The most obvious area for optimization is in knowing that
   the screen does not need to be redrawn.  Already, we can see
   evidence of duplicate erasures, for example.

   Constraining
   ------------

   It looks like I am constraining only after I pull an Op from the
   queue and perform it.  I think that, if this is truly the case, I
   should make it a known policy that constraining is done only at
   this time.

   Attributes
   ----------

   I think that the attribute handle is a little broken when it comes
   to Panes and Windows.  It is OK to set the attr() for a Pane since
   this value is stored in an Op and reconciled with the Window's
   attr.  If we were to set the attr for the window directly, we'd be
   in trouble because we would be setting the variable responsible for
   keeping track of the current attribute.  I think we should store
   the 'current' attribute in a Window variable that is *not* part of
   the Pane.  This means we can use all Pane functions in a Window
   without defining Pane's within that Window.

*/

#include "std.h"
#include <stdarg.h>
#include <termios.h>
#include <sys/time.h>
#include <sys/types.h>
#include "blue6.h"
#include "termcap.h"

inline int min (int a, int b) { if (a < b) return a; else return b; }
inline int max (int a, int b) { if (a > b) return a; else return b; }

Attribute Attribute::normal = Attribute (Attribute::BgDefault
					 | Attribute::FgDefault,
					 Attribute::All);

inline void send (const char* sz) {
  fprintf (stdout, sz);
  fflush (stdout);
}

Attribute& Drawable::attr (void)
{
  return m_attribute;
}

Attribute& Drawable::attrBg (void)
{
  return m_attributeBg;
}

Attribute Attribute::merge (const Attribute& attrOther) const 
{
  Attribute attr (*this);

  attr.m_value = (  (attr.m_value      & ~attrOther.m_mask)
		  | (attrOther.m_value &  attrOther.m_mask));
  attr.m_mask |= attrOther.m_mask;

  return attr;
}

/* Attribute::reconcile

   reconciles the differences between two attributes.  The termcap
   string to make the change from this Attribute to attr are coded
   into the string and the value of this attribute is set.  The return
   value is the number of characters coded in the string.

*/

int Attribute::reconcile (const Termcap& termcap, Attribute attr,
			  char* sz, int cbMax)
{
  int cb = 0;
				// Normalize the new Attribute
  attr.m_value = (  (     m_value & ~attr.m_mask)
		  | (attr.m_value &  attr.m_mask));

				// Map bright colors to emboldend ones
  if (attr.m_mask & FgBright)
    attr.m_value = (   (attr.m_value & ~Bold)
		    | ((attr.m_value &  FgBright) ? Bold : 0));

				// Disable attributes first
  if (!(attr.m_value & Bold) && (m_value & Bold)) {
    cb += termcap.compose (termcapEndMode, sz + cb);
    m_value &= ~(Bold | FgMask | BgMask);
  }
  if (!(attr.m_value & Underline) && (m_value & Underline)) {
    if (termcap.has (termcapEndUnderline))
      cb += termcap.compose (termcapEndUnderline, sz + cb);
    else {
      cb += termcap.compose (termcapEndMode, sz + cb);
      m_value &= ~(Underline | FgMask | BgMask);
    }
  }
  if (!(attr.m_value & Inverse) && (m_value & Inverse)) {
    cb += termcap.compose (termcapEndMode, sz + cb);
    m_value &= ~(Inverse | FgMask | BgMask);
  }
  if (!(attr.m_value & Standout) && (m_value & Standout)) {
    if (termcap.has (termcapEndStandout))
      cb += termcap.compose (termcapEndStandout, sz + cb);
    else {
      cb += termcap.compose (termcapEndMode, sz + cb);
      m_value &= ~(Standout | FgMask | BgMask);
    }
  }
  if (!(attr.m_value & Blink) && (m_value & Blink)) {
    cb += termcap.compose (termcapEndMode, sz + cb);
    m_value &= ~(AllMode & ~(Blink | FgMask | BgMask));
  }

				// Enable missing attributes
  if ((attr.m_value & Bold) && !(m_value & Bold))
    cb += termcap.compose (termcapStartBold, sz + cb);
  if ((attr.m_value & Underline) && !(m_value & Underline))
    cb += termcap.compose (termcapStartUnderline, sz + cb);
  if ((attr.m_value & Inverse) && !(m_value & Inverse))
    cb += termcap.compose (termcapStartInverse, sz + cb);
  if ((attr.m_value & Standout) && !(m_value & Standout))
    cb += termcap.compose (termcapStartStandout, sz + cb);
  if ((attr.m_value & Blink) && !(m_value & Blink))
    cb += termcap.compose (termcapStartBlink, sz + cb);

				// Set color attribute
  int idFg = termcap.has (termcapSetAnsiForeground)
    ? termcapSetAnsiForeground : termcapSetForeground;
  int idBg = termcap.has (termcapSetAnsiBackground)
    ? termcapSetAnsiBackground : termcapSetBackground;
  if ((attr.m_value & FgMask) != (m_value & FgMask)) {
    unsigned16 color = (attr.m_value & FgMask);
    if (color == FgDefault)
      cb += termcap.compose (idFg, sz + cb, 9);
    else
      cb += termcap.compose (idFg, sz + cb, (color - FgFirst) >> FgShift);
  }
  if ((attr.m_value & BgMask) != (m_value & BgMask)) {
    unsigned16 color = (attr.m_value & BgMask);
    if (color == BgDefault)
      cb += termcap.compose (idBg, sz + cb, 9);
    else
      cb += termcap.compose (idBg, sz + cb, (color - BgFirst) >> BgShift);
  }

  assert_ (cb < cbMax);		// *** FIXME *** need something better

  m_value = attr.m_value;

  return cb;
}

Position& Drawable::pos (void)
{
  return m_position;
}

void Drawable::_draw_text (const Position& pos, const char* szText)
{
  Drawable::pos () = pos;
  _draw_text (szText);
}

void Drawable::draw_text (const char* szFormat, ...)
{
  va_list ap;
  va_start (ap, szFormat);
  
  static char sz[2048];
  vsnprintf (sz, sizeof (sz), szFormat, ap);
  va_end (ap);

  _draw_text (sz);
}


void Drawable::draw_text (const Position& pos, const char* szFormat, ...)
{
  Drawable::pos () = pos;
  
  va_list ap;
  va_start (ap, szFormat);
  
  static char sz[2048];
  vsnprintf (sz, sizeof (sz), szFormat, ap);
  va_end (ap);

  _draw_text (sz);
}


void Op::init (eOp op, const Rectangle& rc, const Attribute& attr)
{
  m_op = op;
  m_rc = rc;
  m_attr = attr;
}


void Op::init (eOp op, const Position& pos, const Attribute& attr,
	       const char* sz, int cb)
{
  m_op = op;
  m_pos = pos;
  m_attr = attr;
  assert_ (m_sz == NULL);
  cb = min (cb, strlen (sz));
  m_sz = new char [cb + 1];
  memcpy (m_sz, sz, cb);
  m_sz[cb] = 0;
}


void Op::init (eOp op, const Position& pos)
{
  m_op = op;
  m_pos = pos;
}


void Pane::draw_vertical_line (int x, int y, int dy)
{
  Rectangle rc (x + m_rc.m_x, y + m_rc.m_y, 1, dy);
  rc.intersect (m_rc);
  m_pWindow->queue (new Op (Op::opDrawLine, rc, attrBg ()));
}

/* Pane::_draw_text

   emits the given string at the current position in the window.  The
   string will be constrained to fit within the current Pane's
   bounding rectangle.  The real task of this routine is to queue
   drawing operations to the Window containing this Pane.

*/

void Pane::_draw_text (const char* szText)
{
  if (!szText)
    return;

  Position& pos = Drawable::pos ();
  int cbText = strlen (szText);
  
				// Generate Op's for each line
  while (*szText) {
    pos.constrain (m_rc);
    int cb = min (m_rc.m_dx - pos.m_x, cbText);
    m_pWindow->queue (new Op (Op::opDrawText,
			      Position (pos.m_x + m_rc.m_x, 
					pos.m_y + m_rc.m_y),
			      attrBg ().merge (attr ()),
			      szText, cb));
    szText += cb;
    pos.m_x += cb;
  }
}

void Pane::erase (void)
{
  m_pWindow->queue (new Op (Op::opErase, m_rc, attrBg ()));
}

void Pane::erase (const Rectangle& rcErase)
{
  Rectangle rc (rcErase);
  rc.offset (m_rc.m_x, m_rc.m_y);	// Offset to parent's coordinates
  rc.intersect (m_rc);			// Intersect with pane's rectangle
  if (rc.is_empty ())
    return;

  m_pWindow->queue (new Op (Op::opErase, rc, attrBg ()));
}

int Pane::getch (const Position& pos)
{
  move_to (pos);
  return m_pWindow->Window::getch ();
}

void Pane::move_to (const Position& pos)
{
  Drawable::pos () = pos; 
  m_pWindow->queue (new Op (Op::opMoveTo, Position (pos.m_x + m_rc.m_x,
						    pos.m_y + m_rc.m_y)));
}

/* Pane::prompt

   fetches a string from the user in a nice, emacs sort of way.  The
   return valuse is zero if the user input is accepted.  It is !0 if
   the user pressed escape.

   There is potential for badness if this function is called on a
   Window instead of a Pane (not the implicit Pane for a Window).
   Attributes are not properly handled in these situations.

*/

int Pane::prompt (Position pos, const char* szPrompt, 
		  char* szResponse, size_t cchResponseMax, 
		  const char* szDefault)
{
  if (szDefault) {
    if (szResponse != szDefault)
      strcpy (szResponse, szDefault);
  }
  else
    *szResponse = 0;

  //  attr ().toggle (Attribute::Inverse);
  draw_text (pos, szPrompt);
  pos.m_x += strlen (szPrompt);

  int ibSelect = 0, cbSelect = strlen (szResponse);
  attr ().invert (Attribute::Inverse);
  // on (A_REVERSE);
  draw_text (pos, szResponse);
  attr ().ignore (Attribute::Inverse);
  //  off (A_REVERSE);

  int point = 0;
  int cch = strlen (szResponse);
  int ch;
  int scroll = 0;
  int scrollDelta = (dx () - pos.m_x)/2;
  int scrollZone = 10;
  while (1) {
    switch (ch = getch (Position (point + pos.m_x - scroll, pos.m_y))) {
    case '\n':
    case '\r':
      return 0;			// Ok
      break;

    case -1:			// Getstr happening off screen
      assert_ (ch != -1);
      break;
    case 27:			// ESC
    case 7:			// ^G -- abort
      return -1;
    case 1:			// ^A -- beginning of line
    case KEYCODE (termcapKeyHome):
      point = 0;
      cbSelect = 0;
      break;
    case 5:			// ^E -- end of line
    case KEYCODE (termcapKeyEnd):
      point = cch;
      cbSelect = 0; 
      break;
    case 4:			// ^D -- delete
    case KEYCODE (termcapKeyDelete):
      if (point < cch) {
	memmove (szResponse + point, szResponse + point + 1, 
		 cch - point);
	--cch;
      }
      break;

    case 8:			// ^H -- backspace
    case KEYCODE (termcapKeyBackspace):
      if (cbSelect) {
	*szResponse = 0;
	cbSelect = 0;
	point = 0;
	cch = 0;
      }
      else if (point) {
	memmove (szResponse + point - 1, szResponse + point,
		 cch - point + 1);
	--point;
	--cch;
      }
      break;

    case 11:			// ^K -- delete to end of line
      if (cbSelect) {
	memmove (szResponse + ibSelect,
		 szResponse + ibSelect + cbSelect,
		 cch - ibSelect - cbSelect + 1);
	cch -= cbSelect;
	point = ibSelect;
	cbSelect = 0;
      }
      else {
	szResponse[point] = 0;
	cch = point;
      }
      break;

    case 2:			// ^B -- backward character
    case KEYCODE (termcapKeyLeft):
      if (point)
	--point;
      cbSelect = 0;
      break;

    case 6:			// ^F -- forward character
    case KEYCODE (termcapKeyRight):
      if (point < cch)
	++point;
      cbSelect = 0;
      break;

    default:
      if (isprint (ch)) {
	if (cbSelect) {		// Replace selection
	  memmove (szResponse + ibSelect + 1, 
		   szResponse + ibSelect + cbSelect,
		   cch - ibSelect - cbSelect + 1);
	  cch -= cbSelect;
	  point = ibSelect;
	  cbSelect = 0;
	}
	else {
	  if (cch >= (int) cchResponseMax - 1) // Leave room for nul, right?
	    continue;		// FIXME: how about a beep here
	  if (point != cch)
	    memmove (szResponse + point + 1, szResponse + point,
		     cch - point + 1);
	  else {
	    assert_ (point < (int) cchResponseMax - 1);
	    szResponse[point + 1] = 0;
	  }
	}
	szResponse[point++] = ch;
	++cch;
      }
      break;			// FIXME: how about a beep on bad input
    }
    // FIXME: We need to break-up the display so we show the
    //        highlighted portion AND the unhighlighted portion.
    if (cbSelect)
      attr ().invert (Attribute::Inverse);
    //      on (A_REVERSE);
    else
      attr ().ignore (Attribute::Inverse);
    //      off (A_REVERSE);
				// Check for scroll changes
    while (point - scroll >= dx () - pos.m_x)
      scroll += scrollDelta;
    while (scroll && point < scroll + scrollZone)
      scroll -= scrollDelta;

    draw_text (pos, "%-*.*s", dx () - pos.m_x, dx () - pos.m_x,
	       szResponse + scroll);
//    printfxy (x, y, "%-*.*s", dx () - x, dx () - x, szResponse + scroll);
    if (cbSelect)
      attr ().ignore (Attribute::Inverse);
    //      off (A_REVERSE);
  }

				// *** this can never be called

				// *** FIXME why do we have to do this?
  //  Drawable::_draw_text (pos, szPrompt);
  //  window ().flush ();
  //  return 0;
}


void Position::constrain (Rectangle& rc)
{
  if (m_x < 0)
    m_x = 0;
  if (m_y < 0)
    m_y = 0;
  if (m_x >= rc.m_dx) {
    m_x = 0;
    ++m_y;
  }
  if (m_y >= rc.m_dy)
    m_y = 0;
}


Rectangle& Rectangle::intersect (const Rectangle& rc)
{
				// Check for disjoint
  if (   (rc.m_x >    m_x +    m_dx)
      || (   m_x > rc.m_x + rc.m_dx)
      || (rc.m_y >    m_y +    m_dy)
      || (   m_y > rc.m_y + rc.m_dy))
    zero ();
  else {
    m_x = max (rc.m_x, m_x);
    m_y = max (rc.m_y, m_y);
    m_dx = min (rc.m_x + rc.m_dx, m_x + m_dx) - m_x;
    m_dy = min (rc.m_y + rc.m_dy, m_y + m_dy) - m_y;
  }
  return *this;
}

Rectangle& Rectangle::intersect (const Rectangle& rc1, const Rectangle& rc2)
{
  *this = rc1;
  return intersect (rc2);
}

void Window::activate (bool fActivate)
{
  static struct termios tiOriginal;

  if (fActivate) {
    if (!tiOriginal.c_iflag && !tiOriginal.c_oflag && !tiOriginal.c_cflag)
      tcgetattr (1, &tiOriginal);
    struct termios ti = tiOriginal;
    cfmakeraw (&ti);
    ti.c_oflag &= ~ONLCR;	// Permit ^J (\n) to perform newline only
    tcsetattr (1, TCSANOW, &ti);
    m_pTermcap->activate (fActivate);
  }
  else {
    m_pTermcap->activate (fActivate);
    struct termios ti = tiOriginal;
    tcsetattr (1, TCSANOW, &ti);
  }
}

void Window::init (Termcap* pTermcap)
{
  m_pTermcap = pTermcap;

  m_rc.m_dx = m_pTermcap->num (termcapNumColumns);
  m_rc.m_dy = m_pTermcap->num (termcapNumLines);

  if (!m_pListOp)
    m_pListOp = new list<Op*>;

  activate (true);
}

void Window::erase (const Rectangle& rcErase)
{
  queue (new Op (Op::opErase, rcErase, Attribute ()));
}

void Window::flush (void)
{
  TRACE ((T_UI_TRACE, "Window::flush"));
  while (!m_pListOp->empty ()) {
    Op* pOp = m_pListOp->front ();
    m_pListOp->pop_front ();
    switch (pOp->op ()) {
    case Op::opErase:
      do_erase (pOp);
      break;
    case Op::opDrawText:
      do_draw_text (pOp);
      break;
    case Op::opDrawLine:
      do_draw_line (pOp);
      break;
    case Op::opMoveTo:
      do_move_to (pOp);
      break;
    default:
      assert_ (false);
      break;
    }
    delete pOp;
  }
}


int Window::getch (void)
{
  m_pWindow->flush ();		// We always want to flush before
				// asking user for input.

  if (!m_cbInput) {
    do {
      fd_set rfds;
      struct timeval tv;

      FD_ZERO(&rfds);
      FD_SET(0, &rfds);
      tv.tv_sec = 0;
      tv.tv_usec = 20000;	// 20 ms

      switch (select (1, &rfds, NULL, NULL, &tv)) {
      case 1:			// read available
	{
	  char ch;
	  read (0, &ch, 1);
	  TRACE ((T_UI_TRACE, "Window::getch '%c'", ch));
	  m_rgbInput[m_cbInput++] = ch;
	  int id = m_pTermcap->keyof (m_rgbInput, m_cbInput);
	  if (id == termcapCloseMatch)
	    continue;		// continue looking
	  if (id == termcapNul)
	    goto normalkey;
	  m_cbInput = 0;
	  return KEYCODE (id);
	  break;
	}
      case -1:			// error
	break;
      default:			// timeout
	if (m_cbInput)
	  goto normalkey;
	break;
      }
    } while (1);
  }
normalkey:

  assert_ (m_cbInput);
  if (m_cbInput) {
    char ch = m_rgbInput[0];
    memmove (m_rgbInput, m_rgbInput + 1, --m_cbInput);
    return ch;
  }

  return 0;
} 


void Window::do_draw_line (Op* pOp)
{
  static char sz[2048];
  int cb = 0;

  Rectangle rc (pOp->rc ());
  rc.intersect (m_rc);

  if (rc.is_empty ())
    return;

  TRACE ((T_UI_TRACE, "Window::do_draw_line [%d %d %d %d]", 
	  rc.m_x, rc.m_y, rc.m_dx, rc.m_dy));

  cb += attr ().reconcile (termcap (), pOp->attr (), 
			   sz + cb, sizeof (sz) - cb);

  bool fGraphics = !m_fNoGraphicCharset && termcap ().has (termcapACS);

  if (fGraphics)
    cb += termcap ().compose (termcapStartACS, sz + cb);
  cb += termcap ().compose (termcapMoveTo, sz + cb, rc.m_y, rc.m_x);
  if (rc.m_dx == 1) {
    while (rc.m_dy) {
      if (fGraphics)
	cb += termcap ().translate (termcapACS, sz + cb, "x");
      else
	cb += sprintf (sz + cb, "|");
				// Note that the following behavior is
				// determined by the behavior of ^J
				// (\n) to perform a newline without a
				// carriage return.
      ++rc.m_y;
      --rc.m_dy;
      if (rc.m_dy) {
	cb += termcap ().compose (termcapDownOne, sz + cb);
	cb += termcap ().compose (termcapLeftOne, sz + cb);
      }
    }
  }
  if (!m_fNoGraphicCharset)
    cb += termcap ().compose (termcapEndACS, sz + cb);

  sz[cb] = 0;
  TRACE ((T_UI_TRACE, "Window::do_draw_line '%s'", sz));
  send (sz);
}


void Window::do_draw_text (Op* pOp)
{
  if (!pOp->sz () || !*pOp->sz ())
    return;

  static char sz[2048];
  int cb = 0;
  Position pos = pOp->pos ();
  const char* szText = pOp->sz ();
  
  pos.constrain (m_rc);
  
  TRACE ((T_UI_TRACE, "Window::do_draw_text [%d %d]"
	  " attributes 0x%x -> 0x%x",
	  pos.m_x, pos.m_y, attr ().value (), pOp->attr ().value ()));

  cb += attr ().reconcile (termcap (), pOp->attr (), sz + cb, 
			   sizeof (sz) - cb);
  if (cb)
    TRACE ((T_UI_TRACE,
	    "Window::do_draw_text reconciled %d bytes for attribute", cb));

  bool fAddress = true;
  for (; *szText; ++szText) {
    if (fAddress) {
      cb += termcap ().compose (termcapMoveTo, sz + cb, pos.m_y, pos.m_x);
      fAddress = false;
    }

    if (isprint (*szText)) {
      sz[cb++] = *szText;
      ++pos.m_x;
    }
    else {
				// *** FIXME we ignore these for now
      continue;
    }
				// Perform simple wrapping
    if (pos.m_x < m_rc.m_dx)
      continue;
    pos.m_x = 0;
    ++pos.m_y;
    if (pos.m_y >= m_rc.m_dy)
      pos.m_y = 0;
    fAddress = true;
  }
  sz[cb] = 0;
  TRACE ((T_UI_TRACE, "Window::do_draw_text '%s'", sz));
  send (sz);
}


void Window::do_erase (Op* pOp)
{
  static char sz[2048];
  int cb = 0;

  Rectangle rc (pOp->rc ());
  rc.intersect (m_rc);		// Constrain the original

  if (rc.is_empty ())
    return;

  TRACE ((T_UI_TRACE, "Window::do_erase performing [%d %d %d %d] "
	  "within [%d %d %d %d]", 
	  rc.m_x, rc.m_y, rc.m_dx, rc.m_dy,
	  m_rc.m_x, m_rc.m_y, m_rc.m_dx, m_rc.m_dy));

				// Reconcile attributes, first
  TRACE ((T_UI_TRACE, "Window::do_erase attributes 0x%x -> 0x%x",
	  attr ().value (), pOp->attr ().value ()));

  cb += attr ().reconcile (termcap (), pOp->attr (), 
			   sz + cb, sizeof (sz) - cb);
  if (cb)
    TRACE ((T_UI_TRACE, "Window::do_erase reconciled %d bytes for attribute", 
	    cb));

				// Erase screen all at once
  if (rc == m_rc) {
    TRACE ((T_UI_TRACE, "Window::do_erase CLS"));
    cb += termcap ().compose (termcapClearScreen, sz + cb);
  }
				// Erase to end of each line
  else if (rc.m_x + rc.m_dx == m_rc.m_x + m_rc.m_dx) {
    TRACE ((T_UI_TRACE, "Window::do_erase EOL"));
    cb += termcap ().compose (termcapMoveTo, sz + cb, rc.m_y, rc.m_x);
    while (rc.m_dy > 0) {
      cb += termcap ().compose (termcapClearEOL, sz + cb);
      --rc.m_dy;
      if (rc.m_dy)
	cb += termcap ().compose (termcapDownOne, sz + cb);
    }
  }
				// Erase within the lines
  else {
    while (rc.m_dy) {
      cb += termcap ().compose (termcapMoveTo, sz + cb, rc.m_y, rc.m_x);
      sprintf (sz + cb, "%*.*s", rc.m_dx, rc.m_dx, "");
      cb += rc.m_dx;
      ++rc.m_y;
      --rc.m_dy;
    }
  }
  sz[cb] = 0;
  TRACE ((T_UI_TRACE, "Window::do_erase '%s'", sz));
  send (sz);			// Perform the erase
}


void Window::do_move_to (Op* pOp)
{
  static char sz[2048];
  int cb = 0;
  Position pos = pOp->pos ();
  
  pos.constrain (m_rc);
  
  TRACE ((T_UI_TRACE, "Window::do_move_to [%d %d]",
	  pos.m_x, pos.m_y));

  cb += termcap ().compose (termcapMoveTo, sz + cb, pos.m_y, pos.m_x);

  sz[cb] = 0;
  TRACE ((T_UI_TRACE, "Window::do_move_to '%s'", sz));
  send (sz);
}


/* Window::push_attribute

   saves the current attribute and sets the Window to normal.  It is
   very important to flush before pushing so that the current
   attribute reflects the correct value.  We could make an operation
   for this, but that seems to be extreme for something we use when
   executing system programs.

*/

void Window::push_attribute (void)
{
  flush ();			// Guarantee that attr() is accurate

  m_attrSave = attr ();

  static char sz[2048];
  int cb = 0;
  cb += attr ().reconcile (termcap (), Attribute::normal,
			   sz + cb, sizeof (sz) - cb);

  sz[cb] = 0;
  TRACE ((T_UI_TRACE, "Window::push_attribute '%s'", sz));
  send (sz);
}

/* Window::pop_attribute

   restores the Window attribute from a saved value.  *** FIXME this
   may be inadequate if the excursion set arbitrary attributes.
   Perhaps we should perform some other procedure to put the Window in
   a known state?

*/

void Window::pop_attribute (void)
{
  static char sz[2048];
  int cb = 0;
  cb += attr ().reconcile (termcap (), m_attrSave,
			   sz + cb, sizeof (sz) - cb);

  sz[cb] = 0;
  TRACE ((T_UI_TRACE, "Window::pop_attribute '%s'", sz));
  send (sz);
}


void Window::queue (Op* pOp)
{    
  TRACE ((T_UI_TRACE, "Window::queue Op '%d'", pOp->op ()));
  m_pListOp->push_back (pOp);
}


/* Window::release_this

   cleanup.  Primarily, we are concerned with fixing the attributes.

*/

void Window::release_this (void)
{
  static char sz[2048];
  int cb = 0;
				// ***FIXME We should have some
				// knowledge of the 'default'
				// attributes. 
  cb += attr ().reconcile (termcap (), Attribute::normal,
			   sz + cb, sizeof (sz) - cb);

  sz[cb] = 0;
  TRACE ((T_UI_TRACE, "Window::reconciling final attributes '%s'", sz));
  send (sz);
  flush ();

  if (m_pListOp)
    delete m_pListOp, m_pListOp = NULL;
  if (m_rgb)
    delete m_rgb, m_rgb = NULL;

  activate (false);		// Return to previously scheduled programming
}

void Window::reset (void)
{
  static char sz[2048];
  int cb = 0;

  cb += termcap ().compose (termcapReset, sz + cb);
  cb += termcap ().compose (termcapReset1, sz + cb);
  sz[cb] = 0;
  TRACE ((T_UI_TRACE, "Window::reset '%s'", sz));
  send (sz);
}
