/*
Copyright (c) 1989-1990, University of Illinois board of trustees.  All
rights reserved.  Written by Michael Maciukenas at the Center for Prokaryote
Genome Analysis.  Design and implementation guidance by Steven Smith, Carl
Woese.
*/
/* File picker by Mike Maciukenas
** Allows the user to search up and down the directory tree, and to specify
**  a file name for saving in the currently shown directory.
** "Open Dir" descends down into a directory.  (the user may also type return
**  after choosing a directory in the list, to do the same thing).
** "Up Dir" ascends to the parent directory.  
** The user may also type a directory into the "Directory:" field.  Pressing
**  return causes the directory typed in to be shown in the list.
** Save returns the file name (with full directory path), and exits.
** Caller specifies an initial string for placement into the "File Name:" 
**  field.  For no initial string, pass NULL.
*/
#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <errno.h>

#include <xview/xview.h>
#include <xview/canvas.h>
#include <xview/scrollbar.h>
#include <xview/win_input.h>
#include "list.h"

#include "generic.h"

#include "canvas.h"



typedef enum {
	save_reg,
	save_link,
	save_exec,
	save_sock,
	save_dir
	} save_file_type;

typedef struct {
	ltgenericd g;
	tframe f;
	tlist l;
	tptext dirtext, filetext;
	tpanel p;
	char dirname[1024];
	tpbutton openb, saveb, upb, cancelb;
	tpmsg bmsg1, bmsg2, bmsg3;
	int lastchosen;
	unsigned long lasttime;
	int (*fun)();
	} *ltsave, ltsaved;

tsave_show(s)
ltsave s;
{
	tsave_make_filelist(s);
	tptext_set_current(s->filetext);
	tframe_push_pin(s->f);
	tframe_show(s->f);
}

tsave_hide(s)
ltsave s;
{
	tframe_pull_pin(s->f);
	tframe_hide(s->f);
}

tsave_push_pin(s)
ltsave s;
{
	tframe_push_pin(s->f);
}

tsave_pull_pin(s)
ltsave s;
{
	tframe_pull_pin(s->f);
}

tsave_change_filename(s, string)
ltsave s;
char *string;
{
	if(string!=NULL)
		tptext_set_val(s->filetext, string);
	tptext_set_current(s->filetext);
}

int tsave_get_filename(s, buf)
ltsave s;
char *buf;
{
	strcpy(buf, tptext_get_val(s->filetext));
	return(1);
}

int tsave_move(s, win, x, y)
ltsave s;
Xv_Window win;
int x,y;
{
    if(s==NULL)
        return(0);
    if(titem_type(s)!=lt_save)
        return(0);

    tframe_move(s->f, win, x, y);
	return(1);
}

int tsave_checkdir(s)
char *s;
{
	DIR *dirp;
	dirp=opendir(s);
	if(dirp==NULL)
		return(0);
	else
	{
		closedir(dirp);
		return(1);
	}
}

save_file_type tsave_readln(dirp, buf, dir)
DIR *dirp;
char *buf;
char *dir;
{
	struct dirent *entry;
	struct stat sbuf;
	int i;
	save_file_type result;

	result=save_reg;
	i=strlen(dir);
	entry=readdir(dirp);
	if(entry==NULL)
		strcpy(buf, "");
	else
	{
		strcpy(buf, entry->d_name);
		if(dir[i-1]!='/')
			strcat(dir, "/");
		strcat(dir, buf);
		if(stat(dir, &sbuf)!=0)
			strcpy(buf, "");
		else
		{
			if(S_ISDIR(sbuf.st_mode))
			{
				strcat(buf, "/");
				result=save_dir;
			}
			else if(S_ISLNK(sbuf.st_mode))
			{
				strcat(buf, "@");
				result=save_link;
			}
			else if(S_ISSOCK(sbuf.st_mode))
			{
				strcat(buf, "=");
				result=save_sock;
			}
			else if(S_ISREG(sbuf.st_mode))
			{
				if(sbuf.st_mode&(S_IXUSR|S_IXGRP|S_IXOTH))
				{
					strcat(buf, "*");
					result=save_exec;
				}
				else
					result=save_reg;
			}
		}
		dir[i]='\0';
	}
	return(result);
}

tsave_make_filelist(s)
ltsave s;
{
	char tempbuf[1024];
	char tmpname[1024];
	DIR *dirp;
	save_file_type type;


	strcpy(tmpname, s->dirname);
	dirp=opendir(tmpname);
	if(dirp==NULL)
	{
		fprintf(stderr, "tsave_make_filelist got bad directory name\n:(\n");
		return;
	}
	else
	{
		tframe_set_busy(s->f);
		tlist_remove_all(s->l);
		for(type=tsave_readln(dirp, tempbuf, s->dirname);tempbuf[0]!='\0';
				type=tsave_readln(dirp, tempbuf, s->dirname))
		{
			if((strcmp(tempbuf, "./")!=0) &&
					(strcmp(tempbuf, "../")!=0))
			{
				if(type==save_dir)
					tlist_add(s->l, tempbuf, -2, 0, type);
				else
					tlist_add(s->l, tempbuf, -2, 1, type);
			}
		}
		closedir(dirp);
		tframe_set_notbusy(s->f);
		tlist_refresh(s->l);
	}
}

return_file(s, e)
ltsave s;
tevent e;
{
	char namebuf[1024];
	int x, y;

	tevent_location(e, &x, &y);

	strcpy(namebuf, tptext_get_val(s->dirtext));
	if(tsave_checkdir(namebuf))
	{
		if(strlen(tptext_get_val(s->filetext))>0)
		{
			strcpy(s->dirname, namebuf);
			chdir(namebuf);
			if(namebuf[strlen(namebuf)-1]!='/')
				strcat(namebuf, "/");
			strcat(namebuf, tptext_get_val(s->filetext));
			if((*s->fun)(namebuf, e))
				tsave_hide(s);
		}
		else
		{
			/* invalid file specified */
			tnotice_no_choice_2line(s->f,
				x, y,
				"Invalid file name specified.",
				"Please choose another.");
		}
	}
	else
	{
		/* invalid directory */
		if(errno==EACCES)
		{
			tnotice_no_choice_2line(s->f,
				x, y,
				"You don't have access to that directory.",
				"Please choose another.");
		}
		else
		{
			tnotice_no_choice_2line(s->f,
				x, y,
				"Invalid directory specified.",
				"Please choose another.");
		}
	}
}

int tsave_choose_open(b, e)
tpbutton b;
tevent e;
{
	ltsave s;
	int selected;
	char namebuf[1024], thestr[1024];
	int x, y;

	tevent_location(e, &x, &y);
	s=(ltsave)titem_get_intdata(b);

	selected=tlist_selected(s->l);
	if(selected>=1)
	{
		strcpy(namebuf, tlist_nth_name(s->l, selected));
		strcpy(thestr, s->dirname);
		if(thestr[strlen(thestr)-1]!='/')
			strcat(thestr, "/");
		strcat(thestr, namebuf);
		thestr[strlen(thestr)-1]='\0';
		if(tsave_checkdir(thestr))
		{
			strcpy(s->dirname, thestr);
			tsave_make_filelist(s);
			tptext_set_val(s->dirtext, s->dirname);
			return(1);
		}
		else
		{
			if(errno==EACCES)
			{
				tnotice_no_choice_2line(s->f,
					x, y,
					"You don't have access to that directory.",
					"Please choose another.");
			}
			else
			{
				tnotice_no_choice_2line(s->f,
					x, y,
					"Invalid directory specified.",
					"Please choose another.");
			}
		}
	}
}

int tsave_choose_up(b, e)
tpbutton b;
tevent e;
{
	ltsave s;
	int i;

	s=(ltsave)titem_get_intdata(b);
	
	if(strlen(s->dirname)==1)
		return(1);
	if(s->dirname[strlen(s->dirname)-1]=='/')
		i=strlen(s->dirname)-2;
	else
		i=strlen(s->dirname)-1;
	for(;s->dirname[i]!='/' && i>0;i--);
	if(i==0) i++;
	s->dirname[i]='\0';
	tsave_make_filelist(s);
	tptext_set_val(s->dirtext, s->dirname);
	return(1);
}

int tsave_choose_cancel(b, e)
tpbutton b;
tevent e;
{
	ltsave s;
	int i;

	s=(ltsave)titem_get_intdata(b);

	if((*s->fun)(NULL, e))
		tsave_hide(s);
	return(0);
}

int tsave_choose_dirname(b, e)
tptext b;
tevent e;
{
	ltsave s;
	char *keys;
	int x, y;

	tevent_location(e, &x, &y);
	s=(ltsave)titem_get_intdata(b);

	if(tevent_type(e)==te_keydown)
	{
		keys=tevent_string(e);
		if(keys[0]=='\n' || keys[0]=='\r')
		{
			if(tsave_checkdir(tptext_get_val(s->dirtext)))
			{
				strcpy(s->dirname, tptext_get_val(s->dirtext));
				tsave_make_filelist(s);
				return(tptext_next);
			}
			else
			{
				if(errno==EACCES)
				{
					tnotice_no_choice_2line(s->f,
						x, y,
						"You don't have access to that directory.",
						"Please choose another.");
				}
				else
				{
					tnotice_no_choice_2line(s->f,
						x, y,
						"Invalid directory specified.",
						"Please choose another.");
				}
			}
		}
		else if(keys[0]=='\t')
		{
			return(tptext_next);
		}
		else
			return(tptext_insert);
	}
}

int tsave_choose_filename(b, e)
tptext b;
tevent e;
{
	ltsave s;
	char namebuf[1024];
	char *keys;
	int x, y;

	tevent_location(e, &x, &y);
	s=(ltsave)titem_get_intdata(b);

	if(tevent_type(e)==te_keydown)
	{
		keys=tevent_string(e);
		if(keys[0]=='\n' || keys[0]=='\r')
		{
			return_file(s, e);
		}
		else if(keys[0]=='\t')
		{
			return(tptext_next);
		}
		else
		return(tptext_insert);
	}
}

int tsave_choose_save(b, e)
tpbutton b;
tevent e;
{
	ltsave s;

	s=(ltsave)titem_get_intdata(b);
	return_file(s, e);
}

int handle_save_frame(f, e)
tframe f;
tevent e;
{
	int w, h;
	ltsave s;

	s=(ltsave)titem_get_intdata(f);

	if(tevent_type(e)==te_resize)
	{
		tevent_size(e, &w, &h);
		tpanel_set_width_to_frame(s->p, s->f);
		tlist_resize(s->l, 0, 0, w, h-tpanel_height(s->p), NULL, s->p);
		return(1);
	}
	else if(tevent_type(e)==te_done)
	{
		printf("done\n");
		return(1);
	}
	else
		return(0);
}

int tsave_choose_filelist(l, i, e)
tlist l;
int i;
tevent e;
{
	ltsave s;
	char *keys;
	static unsigned long thistime;

	s=(ltsave)titem_get_intdata(l);

	switch(tevent_type(e))
	{
		case te_mouseup:
			if(i>0)
				tpbutton_enable(s->openb);
			else
				tpbutton_disable(s->openb);
			if(s->lastchosen==i && i!=-1)
			{
				/* may be double clicking */
				thistime=tevent_time(e);
				if(thistime-s->lasttime<500)
				{
					/* double clicking */
					tsave_choose_open(s->openb, e);
					s->lastchosen=-1;
					return(0);
				}
				else
				{
					s->lastchosen=i;
					s->lasttime=thistime;
				}
			}
			else
			{
				s->lastchosen=i;
				s->lasttime=tevent_time(e);
			}
			break;
		case te_keydown:
			keys=tevent_string(e);
			if(keys[0]=='\n' || keys[0]=='\r')
				tsave_choose_open(s->openb, e);
			s->lastchosen=-1;
			break;
		default:
			break;
	}
	return(1);
}

tsave tsave_new(parent, x, y, listfont, fun, args, title, dirname, init_file)
/* create a frame for choosing where to save a file */
tframe parent;
int x, y;
tfont listfont;
int (*fun)();
char *title;
char *dirname;
char *init_file;
targs args;
{
	ltsave tmp;

	if(parent==NULL || listfont==NULL)
		return(NULL);
	if(titem_type(parent)!=lt_frame || titem_type(listfont)!=lt_font)
		return(NULL);

	tmp=(ltsave)titem_new(parent, lt_save, sizeof(ltsaved));
	if(tmp==NULL)
	{
		printf("Not enough memory for save frame\n");
		return(NULL);
	}

	tmp->f=tframe_new(parent, 50,50,500,500,0,1,0,title,args);
	if(tmp->f==NULL)
	{
		titem_free(tmp);
		return(NULL);
	}
	titem_set_intdata(tmp->f, tmp);
	tframe_set_event_procedure(tmp->f, handle_save_frame);

	tmp->p=tframe_dialog_panel(tmp->f);

	tmp->openb=tpbutton_new(tmp->p, 10, 10, 0, 0, NULL, NULL, "Open Dir");
	tpbutton_set_event_procedure(tmp->openb, tsave_choose_open);
	titem_set_intdata(tmp->openb, tmp);
	tpbutton_disable(tmp->openb);

	tmp->upb=tpbutton_new(tmp->p, 10, 10, 0, 0, tmp->openb, NULL, "Up Dir");
	tpbutton_set_event_procedure(tmp->upb, tsave_choose_up);
	titem_set_intdata(tmp->upb, tmp);

	tmp->saveb=tpbutton_new(tmp->p, 10, 10, 0, 0, tmp->upb, NULL, "Save");
	tpbutton_set_event_procedure(tmp->saveb, tsave_choose_save);
	titem_set_intdata(tmp->saveb, tmp);

	tmp->cancelb=tpbutton_new(tmp->p, 10, 10, 0, 0, tmp->saveb, NULL,
		"Cancel");
	tpbutton_set_event_procedure(tmp->cancelb, tsave_choose_cancel);
	titem_set_intdata(tmp->cancelb, tmp);

	tmp->bmsg1=(tpmsg)tpmsg_new(tmp->p,
		10, 10, 0, 0, NULL, tmp->openb, "    ");

	tmp->dirtext=tptext_new(tmp->p, 10, 60, 0, 0, NULL, tmp->bmsg1,
		"Directory:");
	tptext_set_displayed_characters(tmp->dirtext, 30);
	tptext_set_event_procedure(tmp->dirtext, tsave_choose_dirname);
	titem_set_intdata(tmp->dirtext, tmp);
	tptext_set_val(tmp->dirtext, dirname);

	tmp->bmsg3=(tpmsg)tpmsg_new(tmp->p,
		10, 10, 0, 0, NULL, tmp->dirtext, "    ");

	tmp->filetext=tptext_new(tmp->p, 10, 60, 0, 0, NULL, tmp->bmsg3,
		"File name:");
	tptext_set_displayed_characters(tmp->filetext, 30);
	tptext_set_event_procedure(tmp->filetext, tsave_choose_filename);
	titem_set_intdata(tmp->filetext, tmp);
	tptext_set_val(tmp->filetext, init_file);

	tmp->bmsg2=(tpmsg)tpmsg_new(tmp->p,
		10, 10, 0, 0, NULL, tmp->filetext, "    ");

	tpanel_fit(tmp->p);

	tmp->fun=fun;
	strcpy(tmp->dirname, dirname);
	tmp->lastchosen=-1;
	tmp->lasttime=0;

	tmp->l=(tlist)tlist_new(tmp->f,
		0, 0, tpanel_width(tmp->p), 15*tfont_height(listfont),
		NULL, tmp->p, listfont, 0, args);
	titem_set_intdata(tmp->l, tmp);
	tlist_set_event_procedure(tmp->l, tsave_choose_filelist);

	tsave_make_filelist(tmp);

	tframe_fit(tmp->f);
	tframe_hide(tmp->f);
	tframe_pull_pin(tmp->f);

	return((tsave)tmp);
}

void *tsave_xview(s)
ltsave s;
{
	if(s==NULL)
		return(NULL);
	if(titem_type(s)!=lt_save)
		return(NULL);
	
	return((void *)tframe_xview(s->f));
}

int tsave_get_panel(s, p, b)
/* returns save file panel and bottom item */
ltsave s;
tpanel *p;
titem *b;
{
	if(s==NULL)
		return(0);
	if(titem_type(s)!=lt_save)
		return(0);
	
	*p=s->p;
	*b=s->bmsg2;
	return(1);
}

int tsave_refit_panel(s)
/* refits panel and moves canvas */
ltsave s;
{
	if(s==NULL)
		return(0);
	if(titem_type(s)!=lt_save)
		return(0);
	

	tpanel_fit(s->p);
	tlist_move(s->l, 0, 0, NULL, s->p);
	tframe_fit(s->f);
	tpanel_set_width_to_frame(s->p, s->f);
	return(1);
}

int tsave_set_busy(s)
/* refits panel and moves canvas */
ltsave s;
{
	if(s==NULL)
		return(0);
	if(titem_type(s)!=lt_save)
		return(0);
	
	tframe_set_busy(s->f);
	return(1);
}

int tsave_set_notbusy(s)
/* refits panel and moves canvas */
ltsave s;
{
	if(s==NULL)
		return(0);
	if(titem_type(s)!=lt_save)
		return(0);
	
	tframe_set_notbusy(s->f);
	return(1);
}
