/*
 * dnode.c - Linux node functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dnode.c,v 1.12 99/06/22 08:18:38 abe Exp $";
#endif


#include "lsof.h"


/*
 * Local definitions
 */

#define	OFFSET_MAX	((off_t)0x7fffffff)	/* this is defined in
						 * .../src/fs/locks.c and not
						 * in a header file */

/*
 * process_inode() - process inode
 */

void
process_inode(fi, fop)
	struct inode *fi;		/* inode kernel space address 
					 * from file structure */
	struct file_operations *fop;	/* inode kernel file operations pointer
					 * from file structure */
{
	dev_t dev;
	char dev_ch[32];
	static int ft = 1;
	struct inode is;
	struct file_lock lk, *lkf, *lkp;
	short lo, lt;
	struct mounts *mp;
	static unsigned long ndops;
	static unsigned long nfops;
	char *pd, *tn;
	static unsigned long rpfops;
	static unsigned long rwpfops;
	umode_t type;
	static unsigned long wpfops;
/*
 * Do first-time only operations;
 */
	if (ft) {
	    if (get_Nl_value("ndops", Drive_Nl, &ndops) < 0)
		ndops = (unsigned long)0;
	    if (get_Nl_value("nfops", Drive_Nl, &nfops) < 0)
		nfops = (unsigned long)0;
	    if (get_Nl_value("rpfops", Drive_Nl, &rpfops) < 0)
		rpfops = (unsigned long)0;
	    if (get_Nl_value("rwpfops", Drive_Nl, &rwpfops) < 0)
		rwpfops = (unsigned long)0;
	    if (get_Nl_value("wpfops", Drive_Nl, &wpfops) < 0)
		wpfops = (unsigned long)0;
	    ft = 0;
	}
/*
 * Read inode.
 */
	if (readinode((KA_T)fi, &is)) {
	    enter_nm(Namech);
	    return;
	}

#if	defined(HASNCACHE)
	Lf->na = (KA_T)fi;
#endif	/* defined(HASNCACHE) */

#if	defined(HASFSTRUCT)
	if (Fsv & FSV_NI) {
	    Lf->fna = (KA_T)fi;
	    Lf->fsv |= FSV_NI;
	}
#endif	/* defined(HASFSTRUCT) */

/*
 * Process node type.
 */
	type = is.i_mode & S_IFMT;
	switch (type) {
	case S_IFBLK:
	    Ntype = N_BLK;
	    break;
	case S_IFCHR:
	    Ntype = N_CHR;
	    break;
	case S_IFIFO:
	    Ntype = N_FIFO;
	    if (rpfops && rpfops == (unsigned long)fop)
		pd = "<-";
	    else if (wpfops && wpfops == (unsigned long)fop)
		pd = "->";
	    else if (rwpfops && rwpfops == (unsigned long)fop)
		pd = "<->";
	    else
		pd = NULL;
	    break;
	case S_IFSOCK:
	    process_isock(&is);
	    return;
	}
	if (Selinet)
	    return;
	if (Ntype == N_REGLR && HasNFS) {
	    if ((ndops && ndops == (unsigned long)is.i_op)
	    ||  (nfops && nfops == (unsigned long)is.i_op))
		Ntype = N_NFS;
	}
/*
 * Obtain lock information.
 */

	if ((lkf = is.i_flock) && Cfd >= 0 && Cfp && Tsp) {
	    lkp = lkf;
	    do {
		if (kread((KA_T)lkp, (char *)&lk, sizeof(lk)))
		    break;
		if (lk.fl_owner && lk.fl_owner == Tsp)
		    lo = 1;
		else
		    lo = 0;

#if	defined(HAS_FL_FILE)
		if (lk.fl_file && lk.fl_file == Cfp)
		    lo++;
#endif	/* defined(HAS_FL_FILE) */

#if	defined(HAS_FL_FD)
		if (lk.fl_fd && lk.fl_fd == Cfd)
		    lo++;
#endif	/* defined(HAS_FL_FD) */

		if (!lo)
		    continue;
		if (lk.fl_start == 0

#if	defined(HAS_FL_WHENCE)
		&&  lk.fl_whence == 0
#endif	/* defined(HAS_FL_WHENCE) */

		&&  lk.fl_end == OFFSET_MAX)
		    lt = 1;
		else
		    lt = 0;
		switch (lk.fl_type & (F_RDLCK | F_WRLCK)) {
		case F_RDLCK:
		    Lf->lock = lt ? 'R' : 'r';
		    break;
		case F_WRLCK:
		    Lf->lock = lt ? 'W' : 'w';
		    break;
		}
		break;
	    } while ((lkp = lk.fl_next) && lkp != lkf);
	}
/*
 * Determine the device.  If it's an NFS device, change Ntype.
 */
	switch (Ntype) {
	case N_FIFO:
	    if (pd) {
		(void) sprintf(dev_ch, "0x%08x", is.u.pipe_i.base);
		enter_dev_ch(dev_ch);
		break;
	    }
		/* fall through */
	default:
	    Lf->dev = dev = (Ntype == N_CHR || Ntype == N_BLK) ? is.i_rdev
							       : is.i_dev;
	    Lf->dev_def = 1;
	}
	if (Ntype == N_REGLR && HasNFS && (!ndops || !nfops)) {
	    for (mp = readmnt(); mp; mp = mp->next) {
		if (mp->ty == N_NFS && dev == mp->dev) {
		    Ntype = N_NFS;
		    switch (type) {
		    case S_IFDIR:
			if (!ndops)
			    ndops = (unsigned long)is.i_op;
			break;
		    default:
			if (!nfops)
			    nfops = (unsigned long)is.i_op;
		    }
		    break;
		}
	    }
	}
/*
 * Determine the inode number.
 */
	switch (Ntype) {
	case N_FIFO:
	    if (pd)
		break;
	    /* fall through */
	default:
	    if (Ntype != N_BLK) {
		Lf->inode = (unsigned long)is.i_ino;
		Lf->inp_ty = 1;
	    }
	}
/*
 * Determine the file size.
 */
	if (Foffset)
	    Lf->off_def = 1;
	else {
	    switch (Ntype) {
	    case N_FIFO:

#if	LINUXV<21057
		if (is.i_pipe) {
		    Lf->sz = (SZOFFTYPE)is.u.pipe_i.len;
		    Lf->sz_def = 1;
		} else if (!Fsize)
#endif	/& LINUXV<21057 */

		    Lf->off_def = 1;
		break;
	    default:
		if (Ntype == N_CHR || Ntype == N_BLK) {
		    if (!Fsize)
			Lf->off_def = 1;
		} else {
		    Lf->sz = (SZOFFTYPE)is.i_size;
		    Lf->sz_def = 1;
		}
	    }
	}
/*
 * Record the link count.
 */
	if (Fnlink) {
	    Lf->nlink = (long)is.i_nlink;
	    Lf->nlink_def = 1;
	    if (Nlink && (Lf->nlink < Nlink))
		Lf->sf |= SELNLINK;
	}
/*
 * Format the type name.
 */
	switch (type) {
	case S_IFBLK:
	    tn = "BLK";
	    break;
	case S_IFCHR:
	    tn = "CHR";
	    break;
	case S_IFDIR:
	    tn = "DIR";
	    break;
	case S_IFIFO:
	    tn = "FIFO";
	    break;
	case S_IFREG:
	    tn = "REG";
	    break;
	case S_IFLNK:
	    tn = "LINK";
	    break;
	default:
	    if (type > 9999)
		(void) sprintf(Lf->type, "*%03d", type % 1000);
	    else
		(void) sprintf(Lf->type, "%04d", type);
	    (void) strcpy(Namech, "unknown inode type");
	    tn = NULL;
	}
	if (tn)
	    (void) strcpy(Lf->type, tn);
	Lf->ntype = Ntype;
/*
 * Record an NFS file selection.
 */
	if (Ntype == N_NFS && Fnfs)
	    Lf->sf |= SELNFS;
/*
 * Look up names.
 */
	switch (Ntype) {
	case N_FIFO:
	    if (pd) {
			(void) strcpy(Lf->iproto, "PIPE");
			Lf->inp_ty = 2;
			(void) strcpy(Namech, pd);
			break;
	    }
	    /* fall through */
	case N_BLK:
	case N_NFS:
	case N_REGLR:
	    if (Lf->dev_def) {
		for (mp = readmnt(); mp; mp = mp->next) {
		    if (dev == mp->dev) {
			Lf->fsdir = mp->dir;
			Lf->fsdev = mp->fsname;

#if	defined(HASFSINO)
			Lf->fs_ino = mp->inode;
#endif	/* defined(HASFSINO) */

			break;
		    }
		}
	    }
	    break;
	}
	Lf->ntype = Ntype;

#if	defined(HASBLKDEV)
/*
 * If this is a S_IFBLK file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && Ntype == N_BLK && Lf->dev_def)
	    find_bl_ino();
#endif	/* defined(HASBLKDEV) */

/*
 * If this is a S_IFCHR file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && Ntype == N_CHR && Lf->dev_def)
	    find_ch_ino();
/*
 * Test for specified file.
 */
	if (Sfile && is_file_named(NULL, (type == S_IFCHR) ? 1 : 0))
	    Lf->sf |= SELNM;
/*
 * Enter name characters.
 */
	if (Namech[0])
	    enter_nm(Namech);
}
