/*
 * dproc.c - AIX process access functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dproc.c,v 1.13 99/10/22 08:21:43 abe Exp $";
#endif


#include "lsof.h"


_PROTOTYPE(static void get_kernel_access,(void));
_PROTOTYPE(static struct le *getle,(long a, long sid, char **err));

#if	AIXV>=4110
_PROTOTYPE(static void getlenm,(struct le *le));
#endif	/* AIXV>=4110 */

_PROTOTYPE(static void process_text,(unsigned long sid));


/*
 * Local definitions
 */

#if	AIXV<4300
#define	PROCINFO	procinfo
#else	/* AIXV>=4300 */
#define	PROCINFO	procsinfo
#define	PROCINFO_INCR	256
#endif	/* AIXV<4300 */

#define	PROCSIZE	sizeof(struct PROCINFO)


/*
 * Local static values
 */

static int Np = 0;			/* number of processes */
static struct PROCINFO *P = (struct PROCINFO *)NULL;
					/* the process table */
static struct user *Up;			/* user structure */

#if	AIXV>=4110
static unsigned long Soff;		/* shared library VM offset */
int Soff_stat = 0;			/* Soff-available status */
static unsigned long Uoff;		/* user area VM offset */
#endif	/* AIXV>=4110 */


/*
 * ckkv() - check kernel version
 */

void
ckkv(d, er, ev, ea)
	char *d;			/* dialect */
	char *er;			/* expected release */
	char *ev;			/* expected version */
	char *ea;			/* expected architecture */
{

#if	defined(HASKERNIDCK)
	int br, p[2], pid;
	char buf[128], *cp;
	struct stat sb;

	if (Fwarn)
	    return;
/*
 * Make sure we can execute OSLEVEL.  If OSLEVEL doesn't exist and the AIX
 * version is below 4.1, return quietly.
 */

#define	OSLEVEL		"oslevel"
#define	OSLEVELPATH	"/usr/bin/oslevel"
	
	if (stat(OSLEVELPATH, &sb)) {

#if	AIXV<4100
	    if (errno == ENOENT)
		return;
#endif	/* AIXV<4100 */

	    (void) fprintf(stderr, "%s: can't execure %s: %s\n",
		Pn, OSLEVELPATH, strerror(errno));
	    Exit(1);
	}
	if ((sb.st_mode & (S_IROTH | S_IXOTH)) != (S_IROTH | S_IXOTH)) {
	    (void) fprintf(stderr, "%s: can't execute %s, modes: %o\n",
		Pn, OSLEVELPATH, sb.st_mode);
	    Exit(1);
	}
/*
 * Open a pipe for receiving the version number from OSLEVEL.  Fork a
 * child to run OSLEVEL.  Retrieve the OSLEVEL output.
 */
	if (pipe(p)) {
	    (void) fprintf(stderr, "%s: can't create pipe to: %s\n",
		Pn, OSLEVELPATH);
	    Exit(1);
	}
	if ((pid = fork()) == 0) {
	    (void) close(1);
	    (void) close(2);
	    (void) close(p[0]);
	    dup2(p[1], 1);
	    dup2(p[1], 2);
	    (void) close(p[1]);
	    execl(OSLEVELPATH, OSLEVEL, NULL);
	    _exit(0);
	}
	if (pid < 0) {
	    (void) fprintf(stderr, "%s: can't fork a child for %s: %s\n",
		Pn, OSLEVELPATH, strerror(errno));
	    Exit(1);
	}
	(void) close(p[1]);
	br = read(p[0], buf, sizeof(buf) - 1);
	(void) close(p[0]);
	(void) wait(NULL);
/*
 * Warn if the actual and expected versions don't match.
 */
	if (br > 0) {
	    buf[br] = '\0';
	    if ((cp = strrchr(buf, '\n')))
		*cp = '\0';
	} else
	    (void) strcpy(buf, "UNKNOWN");
	if (!ev || strcmp(buf, ev))
	    (void) fprintf(stderr,
		"%s: WARNING: compiled for %s version %s; this is %s.\n",
		Pn, d, ev ? ev : "UNKNOWN", buf);
#endif	/* defined(HASKERNIDCK) */

}


/*
 * gather_proc_info() - gather process information
 */

void
gather_proc_info()
{
	KA_T cdir, fp, pdir, rdir;
	char *cmd;
	int hl, i, nf, np;
	struct PROCINFO *p;
	short pss, sf;
	struct user us;

#if	AIXV>=4300
	static struct fdsinfo *fds = (struct fdsinfo *)NULL;
	MALLOC_S msz;
	pid_t pid;
#endif	/* AIXV>=4300 */

/*
 * Read the process table.
 */

#if	AIXV<4300
	if (!P) {
	    if (!(P = (struct PROCINFO *)malloc((size_t)PROCSIZE))) {
		(void) fprintf(stderr,
		    "%s: can't allocate space for 1 proc\n", Pn);
		Exit(1);
	    }
	    Np = 1;
	}
	while (((np = getproc(P, Np, PROCSIZE)) == -1) && errno == ENOSPC) {
	    Np = P->p_pid + 10;
	    if (!(P = (struct PROCINFO *)realloc((MALLOC_P *)P,
					 (size_t)(Np * PROCSIZE))))
	    {
		(void) fprintf(stderr, "%s: no space for %d procinfo's\n",
		    Pn, Np);
		Exit(1);
	    }
	}
#else	/* AIXV>=4300 */
	if (!P) {
	    msz = (MALLOC_S)(PROCSIZE * PROCINFO_INCR);
	    if (!(P = (struct PROCINFO *)malloc(msz))) {
		(void) fprintf(stderr,
		    "%s: can't allocate space for %d procs\n",
		    Pn, PROCINFO_INCR);
		Exit(1);
	    }
	    Np = PROCINFO_INCR;
	}
	np = pid = 0;
	p = P;
	while ((i = getprocs(p, PROCSIZE, (struct fdsinfo *)NULL, 0, &pid,
			     PROCINFO_INCR))
	== PROCINFO_INCR) {
	    np += PROCINFO_INCR;
	    if (np >= Np) {
	        msz = (MALLOC_S)(PROCSIZE * (Np + PROCINFO_INCR));
		if (!(P = (struct PROCINFO *)realloc((MALLOC_P *)P, msz))) {
		    (void) fprintf(stderr,
			"%s: no more space for proc storage\n", Pn);
		    Exit(1);
		}
		Np += PROCINFO_INCR;
	    }
	    p = (struct PROCINFO *)((char *)P + (np * PROCSIZE));
	}
	if (i > 0)
	    np += i;
#endif	/* AIXV<4300 */

/*
 * Loop through processes.
 */
	for (p = P, Up = &us; np > 0; np--, p++) {
	    if (p->p_stat == 0 || p->p_stat == SZOMB)
		continue;
	    if (is_proc_excl(p->p_pid, (int)p->p_pgrp, (UID_ARG)p->p_uid,
			     &pss, &sf))
		continue;

#if	AIXV<4300
	/*
	 * Get user structure for AIX < 4.3.
	 *
	 * If AIX version is below 4.1.1, use getuser().
	 *
	 * If AIX version is 4.1.1 or above: if readx() is disabled (no -X
	 * option, use  getuser(); if readx() is enabled (-X), use readx().
	 */

# if	AIXV>=4110
	    if (Fxopt
	    &&  kreadx(Uoff, (char *)Up, U_SIZE, p->pi_adspace) == 0)
		i = 1;
	    else
		i = 0;
	    if (i == 0) {
		if (getuser(p, PROCSIZE, Up, U_SIZE) != 0)
		    continue;
	    }
	    hl = i;
# else	/* AIXV<4110 */
	    if (getuser(p, PROCSIZE, Up, U_SIZE) != 0)
		continue;
	    hl = 1;
# endif	/* AIXV>=4110 */
	/*
	 * Save directory vnode addresses, command name address, and open file
	 * count from user structure.
	 *
	 * Skip processes excluded by the user structure command name.
	 */
	    cdir = (KA_T)Up->u_cdir;

# if	AIXV<4100
	    pdir = (KA_T)Up->u_pdir;
# endif	/* AIXV<4100 */

	    rdir = (KA_T)Up->u_rdir;
	    cmd = Up->u_comm;
	    nf = Up->u_maxofile;
	    if (is_cmd_excl(cmd, &pss, &sf))
		continue;
#else	/* AIXV>=4300 */
	/*
	 * For AIX 4.3 and above, skip processes excluded by the procsinfo
	 * command name.  Use getprocs() to get the file descriptors for
	 * included processes.
	 *
	 * If readx is enabled (-X), use it to get the loader_anchor structure.
	 */
	    if (is_cmd_excl(p->pi_comm, &pss, &sf))
		continue;
	    if (!fds) {
		if (!(fds = (struct fdsinfo *)malloc((MALLOC_S)
					      sizeof(struct fdsinfo))))
		{
		    (void) fprintf(stderr,
			"%s: can't allocate fdsinfo struct for PID %d\n",
			Pn, pid);
		    Exit(1);
		}
	    }
	    pid = p->p_pid;
	    if (getprocs((struct procsinfo *)NULL, PROCSIZE, fds,
			 sizeof(struct fdsinfo), &pid, 1)
	    != 1)
		continue;
	    if (Fxopt
	    &&  kreadx((off_t)(Uoff + offsetof(struct user, U_loader)),
		       (char *)&Up->U_loader, sizeof(struct la), p->pi_adspace)
	    == 0)
		hl = 1;
	    else
		hl = 0;
	/*
	 * Save directory vnode addresses, command name, and open file count
	 * from procinfo structure.
	 */
	    cdir = (KA_T)p->pi_cdir;
	    pdir = (KA_T)NULL;
	    rdir = (KA_T)p->pi_rdir;
	    cmd = p->pi_comm;
	    nf = p->pi_maxofile;
#endif	/* AIXV<4300 */

	/*
	 * Allocate a local process structure and start filling it.
	 */
	    alloc_lproc(p->p_pid, (int)p->p_pgrp, (int)p->p_ppid,
		(UID_ARG)p->p_uid, cmd, (int)pss, (int)sf);
	    Plf = (struct lfile *)NULL;
	/*
	 * Save current working directory information.
	 */
	    if (cdir) {
		alloc_lfile(CWD, -1);
		process_node(cdir);
		if (Lf->sf)
		    link_lfile();
	    }
	/*
	 * Save root directory information.
	 */
	    if (rdir) {
		alloc_lfile(RTD, -1);
		process_node(rdir);
		if (Lf->sf)
		    link_lfile();
	    }

#if	AIXV<4100
	/*
	 * Save parent directory information.
	 */
	    if (pdir) {
		alloc_lfile("  pd", -1);
		process_node(pdir);
		if (Lf->sf)
		    link_lfile();
	    }
#endif	/* AIXV<4100 */

	/*
	 * Save information on text files.
	 */
	    if (hl)
		process_text(p->pi_adspace);
	/*
	 * Save information on file descriptors.
	 */
	    for (i = 0; i < nf; i++) {

#if	AIXV<4300
		fp = (KA_T)Up->u_ufd[i].fp;
#else	/* AIXV>=4300 */
		fp = (KA_T)fds->pi_ufd[i].fp;
#endif	/* AIXV<4300 */

		if (fp) {
		    alloc_lfile((char *)NULL, i);
		    process_file(fp);
		    if (Lf->sf) {

#if	defined(HASFSTRUCT)
			if (Fsv & FSV_FG)

# if	AIXV<4300
			    Lf->pof = (long)(Up->u_ufd[i].flags & 0x7f);
#else	/* AIXV>=4300 */
			    Lf->pof = (long)(fds->pi_ufd[i].flags & 0x7f);
#endif	/* AIXV<4300 */
#endif	/* defined(HASFSTRUCT) */

			link_lfile();
		    }
		}
	    }
	/*
	 * Examine results.
	 */
	    if (examine_lproc())
		return;
	}
}


/*
 * get_kernel_access() - get access to kernel memory
 */

static void
get_kernel_access()
{
	int oe = 0;
/*
 * Access /dev/mem.
 */
	if ((Km = open("/dev/mem", O_RDONLY, 0)) < 0) {
	    (void) fprintf(stderr, "%s: can't open /dev/mem: %s\n",
		Pn, strerror(errno));
	    oe++;
	}

#if	defined(WILLDROPGID)
/*
 * If kernel memory isn't coming from KMEM, drop setgid permission
 * before attempting to open the (Memory) file.
 */
	if (Memory)
	    (void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the non-KMEM memory file is readable.
 */
	if (Memory && !is_readable(Memory, 1))
	    Exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Access kernel memory file.
 */
	if ((Kd = open(Memory ? Memory : KMEM, O_RDONLY, 0)) < 0) {
	    (void) fprintf(stderr, "%s: can't open %s: %s\n", Pn,
		Memory ? Memory : KMEM, strerror(errno));
	    oe++;
	}
	if (oe)
	    Exit(1);

#if	defined(WILLDROPGID)
/*
 * Drop setgid permission, if necessary.
 */
	if (!Memory)
	    (void) dropgid();
#endif	/* defined(WILLDROPGID) */

/*
 * Get kernel symbols.
 */
	if (knlist(Nl, X_NL_NUM, sizeof(struct nlist))
	||  Nl[X_UADDR].n_value == (long)0) {
	    (void) fprintf(stderr, "%s: can't get kernel's %s address\n",
		Pn, Nl[X_UADDR].n_name);
	    Exit(1);
	}

#if	defined(HAS_AFS)
	(void) knlist(AFSnl, X_AFSNL_NUM, sizeof(struct nlist));
#endif	/* defined(HAS_AFS) */

#if	AIXV>=4110
/*
 * Get user area and shared library VM offsets for AIX 4.1.1 and above.
 */
	if (Fxopt) {

	    struct nlist ll[] = {
		{ "library_anchor"	},

# if	AIXV>=4330
		{ "library_le_handle"	},
# else	/* AIXV<4330 */
		{ "library_data_handle"	},
# endif	/* AIXV>=4330 */

		{ (char *)NULL		}
	    };
	    struct ublock *ub;

	    if (nlist(N_UNIX, ll) == 0
	    &&  ll[0].n_value != (long)0 && ll[1].n_value != (long)0
	    &&  kreadx((off_t)(ll[1].n_value & RDXMASK), (char *)&Soff,
			sizeof(Soff), (unsigned long)0)
	    == 0)
		Soff_stat++;
	    ub = (struct ublock *)Nl[X_UADDR].n_value;
	    Uoff = (unsigned long)((int)&ub->ub_user & RDXMASK);
	}
#endif	/* AIXV>=4110 */

/*
 * Check the kernel version number.
 */
	(void) ckkv("AIX", (char *)NULL, LSOF_VSTR, (char *)NULL);
}


/*
 * getle() - get loader entry structure
 */

static struct le *
getle(a, sid, err)
	long a;				/* loader entry kernel address */
	long sid;			/* user structure segment ID */
	char **err;			/* error message (if return is NULL) */
{
	static struct le le;

#if	AIXV<4110
	if (a < Nl[X_UADDR].n_value) {
	    *err = "address too small";
	    return((struct le *)NULL);
	}
	if (((char *)a + sizeof(le)) <= ((char *)Nl[X_UADDR].n_value + U_SIZE))
	    return((struct le *)((char *)Up + a - Nl[X_UADDR].n_value));
#endif	/* AIXV<4110 */

	if (!Fxopt) {
	    *err = "readx() disabled for Stale Segment ID bug (see -X)";
	    return((struct le *)NULL);
	}

#if	AIXV>=4110
	if (kreadx((off_t)(a & RDXMASK), (char *)&le, sizeof(le), sid) == 0)
#else	/* AIXV<4110 */
	if (kreadx((off_t)a, (char *)&le, sizeof(le), sid) == 0)
#endif	/* AIXV>=4110 */

	    return(&le);
	*err = "can't readx()";
	return((struct le *)NULL);
}


#if	AIXV>=4110
/*
 * getlenm() - get loader entry file name for AIX >= 4.1.1 and < 4.3
 */

static void
getlenm(le)
	struct le *le;
{
	char buf[LIBNMLN];
	int i;

# if	AIXV<4300
	if ((le->ft & LIBMASK) != LIBNMCOMP)
	    return;
#endif	/* AIXV<4300 */

	if (!kreadx((off_t)le->nm & RDXMASK, buf, LIBNMLN, Soff)) {
	    buf[LIBNMLN - 1] = '\0';
	    i = strlen(buf);
	    if (i < (LIBNMLN - 3) && buf[i+1])
		enter_nm(&buf[i+1]);
	    else if (buf[0])
		enter_nm(buf);
	}
}
#endif	/* AIXV>=4110 */


/*
 * initialize() - perform all initialization
 */

void
initialize()
{
	get_kernel_access();
}


/*
 * kread() - read from kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel memory address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	int br;

	if (lseek(Kd, addr, L_SET) == (off_t)-1L)
		return(-1);
	br = read(Kd, buf, len);
	return((br == len) ? 0 : 1);
}


/*
 * kreadx() - read kernel segmented memory
 */

int
kreadx(addr, buf, len, sid)
	off_t addr;			/* kernel address */
	char *buf;			/* destination buffer */
	int len;			/* length */
	long sid;			/* segment ID */
{
	int br;

	if (lseek(Km, addr, L_SET) == (off_t)-1)
		return(1);
	br = readx(Km, buf, len, sid);
	return (br == len ? 0 : 1);
}


/*
 * process_text() - process text file information for AIX < 4.3
 */

static void
process_text(sid)
	unsigned long sid;		/* user area segment ID */
{
	static struct file **f = (struct file **)NULL;
	char *err, fd[8];
	int i, j, n;
	struct le *le;
	long ll;
	MALLOC_S msz;
	static MALLOC_S nf = 0;
	struct file *xf = (struct file *)NULL;

#if	AIXV>=4300
	struct la *la = (struct la *)&Up->U_loader;
#endif	/* AIXV>=4300 */

/*
 * Display information on the exec'd entry.
 */

#if	AIXV<4300
	if ((ll = Up->u_loader[1]))
#else	/* AIXV>=4300 */
	if ((ll = (long)la->exec))
#endif	/* AIXV<4300 */

	{
	    alloc_lfile(" txt", -1);
	    if ((le = getle(ll, sid, &err))) {
		if ((xf = le->fp)) {
		    process_file((KA_T)xf);
		    if (Lf->sf) {

#if	AIXV>=4110 && AIXV<4300
			if (!Lf->nm || !Lf->nm[0])
			    getlenm(le);
#endif	/* AIXV>=4110 && AIXV<4300 */

			link_lfile();
		    }
		}
	    } else {
		(void) sprintf(Namech, "text entry at %s: %s",
		    print_kptr((KA_T)ll, (char *)NULL), err);
		enter_nm(Namech);
		if (Lf->sf)
		    link_lfile();
	    }
	}
/*
 * Display the loader list.
 */

	for (i = n = 0,

#if	AIXV<4300
	     ll = Up->u_loader[0];
#else	/* AIXV>=4300 */
	     ll = (long)la->list;
#endif	/* AIXV<4300 */

	     ll;
	     ll = (long)le->next)
	{
	    (void) sprintf(fd, " L%02d", i);
	    alloc_lfile(fd, -1);
	    if (!(le = getle(ll, sid, &err))) {
		(void) sprintf(Namech, "loader entry at %s: %s",
		    print_kptr((KA_T)ll, (char *)NULL), err);
		enter_nm(Namech);
		if (Lf->sf)
		    link_lfile();
		return;
	    }
	/*
	 * Skip entries with no file pointers, the exec'd file, and entries
	 * that have already been processed.
	 */
	    if (!le->fp || le->fp == xf)
		continue;
	    for (j = 0; j < n; j++) {
		if (f[j] == le->fp)
		    break;
	    }
	    if (j < n)
		continue;
	    if (n >= nf) {
		nf += 10;
		msz = (MALLOC_S)(nf * sizeof(struct file *));
		if (f)
		    f = (struct file **)realloc((MALLOC_P *)f, msz);
		else
		    f = (struct file **)malloc(msz);
		if (!f) {
		    (void) fprintf(stderr,
			"%s: no space for text file pointers\n", Pn);
		    Exit(1);
		}
	    }
	    f[n++] = le->fp;
	/*
	 * Save the loader entry.
	 */
	    process_file((KA_T)le->fp);
	    if (Lf->sf) {

#if	AIXV>=4110
		if (!Lf->nm || !Lf->nm[0])
		    getlenm(le);
#endif	/* AIXV>=4110 */

		link_lfile();
		i++;
	    }
	}
}
