/*
 * dnode.c - AIX node reading functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dnode.c,v 1.9 10/.0/.1 .0:.0:.4 abe Exp $";
#endif


#include "lsof.h"


/*
 * isglocked() - is a gnode locked
 */

char
isglocked(ga)
	struct gnode *ga;		/* local gnode address */
{

	struct filock *cfp, f, *ffp;
	int l;

	if (!(ffp = ga->gn_filocks))
	    return(' ');
	cfp = ffp;

#if	AIXV>=4140
	do {
#endif	/* AIXV>=4140 */

	    if (kread((KA_T)cfp, (char *)&f, sizeof(f)))
		return(' ');

#if	AIXV>=4140
	    if (f.set.l_sysid || f.set.l_pid != (pid_t)Lp->pid)
		continue;
#endif	/* AIXV>=4140 */

	    if (f.set.l_whence == 0 && f.set.l_start == 0

#if	AIXV>=4200
	    &&  f.set.l_end == 0x7fffffffffffffffLL
#else	/* AIXV<4200 */
	    &&  f.set.l_end == 0x7fffffff
#endif	/* AIXV>=4200 */

	    )
		l = 1;
	    else
		l = 0;
	    switch (f.set.l_type & (F_RDLCK | F_WRLCK)) {

	    case F_RDLCK:
		return((l) ? 'R' : 'r');
	    case F_WRLCK:
		return((l) ? 'W' : 'w');
	    case (F_RDLCK + F_WRLCK):
		return('u');
	    }
	    return(' ');

#if	AIXV>=4140
	} while ((cfp = f.next) && cfp != ffp);
	return(' ');
#endif	/* AIXV>=4140 */

}


/*
 * process_node() - process vnode
 */

void
process_node(va)
	KA_T va;			/* vnode kernel space address */
{
	struct cdrnode c;
	dev_t dev;
	struct gnode g;
	struct inode i;
	int ic = 0;
	int ins = 0;
	struct vfs *la = NULL;
	char tbuf[32], *ty;
	enum vtype type;
	struct l_vfs *vfs;
	static struct vnode *v = (struct vnode *)NULL;

#if	AIXV>=3200
	struct devnode dn;
	struct gnode pg;
	struct specnode sn;
	struct fifonode f;
#endif	/* AIXV>=3200 */

#if	defined(HAS_AFS)
	static int afs = 0;		/* AFS test status: -1 = no AFS
					 *		     0 = not tested
					 *		     1 = AFS present */
	struct afsnode an;
#endif	/* defined(HAS_AFS) */

#if	defined(HAS_NFS)
	struct rnode r;
#endif	/* defined(HAS_NFS) */

#if	AIXV>=4140
	struct clone *cl;
#endif	/* AIXV>=4140 */

/*
 * Read the vnode.
 */
	if (!va) {
	    enter_nm("no vnode address");
	    return;
	}
	if (!v) {

	/*
	 * Allocate space for the vnode or AFS vcache structure.
	 */

#if	defined(HAS_AFS)
	    v = alloc_vcache();
#else	/* !defined(HAS_AFS) */
	    v = (struct vnode *)malloc(sizeof(struct vnode));
#endif	/* defined(HAS_AFS) */

	    if (!v) {
		(void) fprintf(stderr, "%s: can't allocate %s space\n", Pn,

#if	defined(HAS_AFS)
			       "vcache"
#else	/* !defined(HAS_AFS) */
			       "vnode"
#endif	/* defined(HAS_AFS) */

			      );
		Exit(1);
	    }
	}
/*
 * Read the vnode.
 */
	if (readvnode(va, v)) {
	    enter_nm(Namech);
	    return;
	}

#if	defined(HASFSTRUCT)
	Lf->fsv |= FSV_NI;
	Lf->fna = va;
#endif	/* defined(HASFSTRUCT) */

/*
 * Read the gnode.
 */
	if (!v->v_gnode || readgnode((KA_T)v->v_gnode, &g)) {
	    (void) sprintf(Namech, "vnode at %s has no gnode\n",
		print_kptr(va, (char *)NULL));
	    enter_nm(Namech);
	    return;
	}

#if	AIXV>=3200

/*
 * Under AIX 3.2 and above, if the vnode type is ISVDEV, then there is a
 * special node and a fifonode or devnode.  Behind them are the "real" gnode,
 * inode and vnode.
 */
	if (ISVDEV(g.gn_type)) {
	    switch (g.gn_type) {
	    case VBLK:
		Ntype = N_BLK;
		break;
	    case VCHR:
		Ntype = N_CHR;
		break;
	    case VFIFO:
		Ntype = N_FIFO;
		break;
	    case VMPC:
		Ntype = N_MPC;
		break;
	    default:
		(void) sprintf(Namech, "vnode at %s: unknown ISVDEV(%#x)",
		    print_kptr(va, (char *)NULL), g.gn_type);
		enter_nm(Namech);
		return;
	    }
	/*
	 * Read the special node.
	 */
	    if (!g.gn_data || kread((KA_T)g.gn_data, (char *)&sn, sizeof(sn))) {
		(void) sprintf(Namech, "vnode at %s: can't read specnode (%s)",
		    print_kptr(va, tbuf),
		    print_kptr((KA_T)g.gn_data, (char *)NULL));
		enter_nm(Namech);
		return;
	   }
	/*
	 * Read the PFS gnode and its inode and vnode.
	 */
	    if (sn.sn_pfsgnode) {
		if (readgnode((KA_T)sn.sn_pfsgnode, &g)) {
		    (void) sprintf(Namech,
			"vnode at %s: can't read pfsgnode (%s)",
			print_kptr(va, tbuf),
			print_kptr((KA_T)sn.sn_pfsgnode, (char *)NULL));
		    enter_nm(Namech);
		    return;
		}
		if (!g.gn_data || readinode((KA_T)g.gn_data, &i)) {
		    (void) sprintf(Namech,
			"pfsgnode at %s: can't read inode (%s)",
			print_kptr((KA_T)sn.sn_pfsgnode, tbuf),
			print_kptr((KA_T)g.gn_data, (char *)NULL));
		    enter_nm(Namech);
		    return;
		}
		ins = 1;
		if (!g.gn_vnode || readvnode((KA_T)g.gn_vnode, v)) {
		    (void) sprintf(Namech,
			"pfsgnode at %s: can't read vnode (%s)",
			print_kptr((KA_T)sn.sn_pfsgnode, tbuf),
			print_kptr((KA_T)g.gn_vnode, (char *)NULL));
		    enter_nm(Namech);
		    return;
		}
	    } else {
		(void) zeromem((char *)&i, sizeof(i));

#if	AIXV>=4140
	    /*
	     * See if this is a clone device:
	     *
	     *     the clone major device number must be known;
	     *     the specnode must have a devnode pointer;
	     *     and the devnode must have a private clone data pointer.
	     */
		if (CloneMaj >= 0
		&&  sn.sn_devnode
		&&  kread((KA_T)sn.sn_devnode, (char *)&dn, sizeof(dn)) == 0
		&&  dn.dv_pdata)
		{

# if	defined(HASDCACHE)

process_clone_again:

# endif	/* defined(HASDCACHE) */

		    for (cl = Clone; cl; cl = cl->next) {
			if (major(g.gn_rdev) == minor(cl->cd.rdev)) {

# if	defined(HASDCACHE)
			    if (DCunsafe && !cl->cd.v && !vfy_dev(&cl->cd))
				goto process_clone_again;
# endif	/* defined(HASDCACHE) */

			    ic = 1;
			    (void) strcpy(Namech, cl->cd.name);
			    if (ClonePtc >= 0
			    &&  major(g.gn_rdev) == ClonePtc)
				(void) sprintf(endnm(),"/%d",minor(g.gn_rdev));
			    Lf->inode = cl->cd.inode;
			    Lf->inp_ty = 1;
			    break;
			}
		    }
		}
#endif	/* AIXV>=4140 */

	    }
	/*
	 * If it's a FIFO, read its fifonode.
	 */
	    if (Ntype == N_FIFO) {
		if (!sn.sn_fifonode ||readfifonode((KA_T)sn.sn_fifonode, &f)) {
		    (void) sprintf(Namech,
			"vnode at %s: can't read fifonode (%s)",
			print_kptr(va, tbuf),
			print_kptr((KA_T)sn.sn_fifonode, (char *)NULL));
		    enter_nm(Namech);
		    return;
		}
	/*
	 * Otherwise, read the devnode and its gnode.
	 */
	    } else {
		if (!sn.sn_devnode
		|| kread((KA_T)sn.sn_devnode,(char *)&dn,sizeof(dn))) {
		    (void) sprintf(Namech,
			"vnode at %s: can't read devnode (%s)",
			print_kptr(va, tbuf),
			print_kptr((KA_T)sn.sn_devnode, (char *)NULL));
		    enter_nm(Namech);
		    return;
		}
		g = dn.dv_gnode;
	    }
	}
#endif	/* AIXV>=3200 */

/*
 * Read the AIX virtual file system structure.
 */
	if (Ntype != N_AFS && g.gn_rdev == NODEVICE) {
	    vfs = (struct l_vfs *)NULL;
	    enter_dev_ch(print_kptr(va, (char *)NULL));
	} else {
	    if (!(vfs = readvfs(v))) {
		(void) sprintf(Namech, "can't read vfs for %s at %s",
		    print_kptr(va, tbuf),
		    print_kptr((KA_T)v->v_vfsp, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	}
/*
 * Set the type for an NFS vnode.
 * Get the lock status.
 */

#if	defined(HAS_NFS)
	if (vfs && vfs->vmt_flags & MNT_REMOTE) {

# if	defined(HAS_AFS)
	    if (!AFSVfsp || (KA_T)v->v_vfsp != AFSVfsp)
# endif	/* defined(HAS_AFS) */

		Ntype = N_NFS;
	}
#endif	/* defined(HAS_NFS) */

	Lf->lock = isglocked(&g);
	switch (Ntype) {

#if	defined(HAS_NFS)
/*
 * Read an NFS rnode.
 */
	case N_NFS:
	    if (!g.gn_data || readrnode((KA_T)g.gn_data, &r)) {
		(void) sprintf(Namech, "remote gnode at %s has no rnode",
		    print_kptr((KA_T)v->v_gnode, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	    break;
#endif	/* defined(HAS_NFS) */

/*
 * Read N_REGLR nodes.
 */
	case N_REGLR:
	    if (vfs && vfs->vmt_gfstype == MNT_CDROM) {

	    /*
	     * Read a CD-ROM cdrnode.
	     */
		if (!g.gn_data || readcdrnode((KA_T)g.gn_data, &c)) {
		    (void) sprintf(Namech, "gnode at %s has no cdrnode",
			print_kptr((KA_T)v->v_gnode, (char *)NULL));
		    enter_nm(Namech);
		    return;
		}
		(void) zeromem((char *)&i, sizeof(i));
		i.i_number = c.cn_inumber;
		i.i_size = (off_t)c.cn_size;
	    /*
	     * Otherwise, read the inode.
	     */

	    } else if (g.gn_data) {
		if (Ntype != N_BLK) {
		    if (readinode((KA_T)g.gn_data, &i)) {
			(void) sprintf(Namech,
			    "gnode at %s can't read inode: %s",
			    print_kptr((KA_T)v->v_gnode, tbuf),
			    print_kptr((KA_T)g.gn_data, (char *)NULL));
			enter_nm(Namech);
			return;
		    }
		    ins = 1;
		}
	    }

#if	defined(HAS_AFS)
	    else {

	    /*
	     * See if this is an AFS node.
	     */
		if (AFSVfsp && (KA_T)v->v_vfsp == AFSVfsp)
		    Ntype = N_AFS;
		else if (v->v_vfsp) {
		    switch (afs) {
		    case -1:
			break;
		    case 0:
			if (!hasAFS(v)) {
			    afs = 1;
			    break;
			}
			afs = 1;
			Ntype = N_AFS;
			break;
		    case 1:
			if ((KA_T)v->v_vfsp == AFSVfsp)
			    Ntype = N_AFS;
		     }
		}
	    /*
	     * If this is an AFS node, read the afsnode.
	     */
		if (Ntype == N_AFS) {
		    if (readafsnode(va, v, &an))
			return;
		} else {
		    (void) sprintf(Namech, "gnode at %s has no inode",
			print_kptr((KA_T)v->v_gnode, (char *)NULL));
		    enter_nm(Namech);
		    return;
		}
	    }
#else	/* !defined(HAS_AFS) */

	    else {
		(void) sprintf(Namech, "gnode at %s has no inode",
		    print_kptr((KA_T)v->v_gnode, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
#endif	/* defined(HAS_AFS) */

	}
/*
 * Get device and type for printing.
 */

#if	defined(HAS_NFS)
	if (Ntype == N_NFS)
	    dev = vfs ? vfs->dev : 0;
	else
#endif	/* defined(HAS_NFS) */

#if	defined(HAS_AFS)
	if (Ntype == N_AFS)
	    dev = an.dev;
	else
#endif	/* defined(HAS_AFS) */

	    dev = g.gn_rdev;

#if	AIXV>=3200
	if (Ntype == N_MPC)
	    type = VMPC;
	else
#endif	/* AIXV>=3200 */

	    type = g.gn_type;
/*
 * Obtain the inode number.
 */
	switch (Ntype) {

#if	defined(HAS_AFS)
	case N_AFS:
	    if (an.ino_st) {
		Lf->inode = an.inode;
		Lf->inp_ty = 1;
	    }
	    break;
#endif	/* defined(HAS_AFS) */

#if	defined(HAS_NFS)
	case N_NFS:
	    Lf->inode = (unsigned long) r.r_attr.va_serialno;
	    Lf->inp_ty = 1;
	    break;
#endif	/* defined(HAS_NFS) */

# if	AIXV>=3200
	case N_BLK:
	    break;
	case N_CHR:
	case N_FIFO:
	case N_MPC:
# endif	/* AIXV>=3200 */

	case N_REGLR:
	    if (ins) {
		Lf->inode = (unsigned long) i.i_number;
		Lf->inp_ty = 1;
	    }
	}
/*
 * Obtain the file size.
 */
	if (Foffset)
	    Lf->off_def = 1;
	else {
	    switch (Ntype) {

#if	defined(HAS_AFS)
	    case N_AFS:
		Lf->sz = (SZOFFTYPE)an.size;
		Lf->sz_def = 1;
		break;
#endif	/* defined(HAS_AFS) */

#if	AIXV>=3200
	    case N_FIFO:
		Lf->sz = (SZOFFTYPE)f.ff_size;
		Lf->sz_def = 1;
		break;
#endif	/* AIXV>=3200 */

#if	defined(HAS_NFS)
	    case N_NFS:
		Lf->sz = (SZOFFTYPE)r.r_attr.va_size;
		Lf->sz_def = 1;
		break;
#endif	/* defined(HAS_NFS) */

#if	 AIXV>=3200
	    case N_BLK:
		if (!Fsize)
		    Lf->off_def = 1;
		break;
	    case N_CHR:
	    case N_MPC:
		if (!Fsize)
		    Lf->off_def = 1;
		break;
#endif	/* AIXV>=3200 */

	    case N_REGLR:
		if (type == VREG || type == VDIR) {
		    if (ins) {
			Lf->sz = (SZOFFTYPE)i.i_size;
			Lf->sz_def = 1;
		    }
		} else if ((type == VCHR || type == VMPC) && !Fsize)
		    Lf->off_def = 1;
		break;
	    }
	}
/*
 * Record link count.
 */
	if (Fnlink) {
	    switch(Ntype) {

# if	defined(HAS_AFS)
	    case N_AFS:
		Lf->nlink = an.nlink;
		Lf->nlink_def = an.nlink_st;
		break;
# endif	/* defined(HAS_AFS) */

# if	defined(HAS_NFS)
	    case N_NFS:
		Lf->nlink = (long)r.r_attr.va_nlink;
		Lf->nlink_def = 1;
		break;
# endif	/* defined(HAS_NFS) */

# if	AIXV>=3200
	    case N_BLK:				/* no link count? */
		break;
	    case N_CHR:
	    case N_FIFO:
	    case N_MPC:
# endif	/* AIXV>=3200 */

	    case N_REGLR:
		if (ins) {
		    Lf->nlink = (long)i.i_nlink;
		    Lf->nlink_def = 1;
		}
		break;
	    }
	    if (Nlink && Lf->nlink_def && (Lf->nlink < Nlink))
		Lf->sf |= SELNLINK;
	}
/*
 * Record an NFS file selection.
 */

#if	defined(HAS_NFS)
	if (Ntype == N_NFS && Fnfs)
	    Lf->sf |= SELNFS;
#endif	/* defined(HAS_NFS) */

/*
 * Save the file system names.
 */
	if (vfs) {
	    Lf->fsdir = vfs->dir;
	    Lf->fsdev = vfs->fsname;
	}
/*
 * Format the vnode type.
 */
	switch (type) {

	case VNON:
	    ty ="VNON";
	    break;
	case VREG:
	case VDIR:
	    ty = (type == VREG) ? "VREG" : "VDIR";
	    Lf->dev_def = 1;
	    Lf->dev = dev;
	    break;
	case VBLK:
	    ty = "VBLK";
	    Lf->dev_def = 1;
	    Lf->dev = dev;
	    Ntype = N_BLK;
	    break;
	case VCHR:
	    ty = "VCHR";
	    Lf->dev = dev;
	    Lf->dev_def = 1;
	    Ntype = N_CHR;
	    break;
	case VLNK:
	    ty = "VLNK";
	    break;

#if	defined(VSOCK)
	case VSOCK:
	    ty = "SOCK";
	    break;
#endif

	case VBAD:
	    ty = "VBAD";
	    break;
	case VFIFO:
	    if (!Lf->dev_ch || Lf->dev_ch[0] == '\0') {
		Lf->dev = dev;
		Lf->dev_def = 1;
	    }
	    ty = "FIFO";
	    break;
	case VMPC:
	    Lf->dev = g.gn_rdev;
	    Lf->ch = g.gn_chan;

#if	AIXV<3200
	    Lf->inp_ty = 0;
#endif	/* AIXV<3200 */

	    Lf->dev_def = 1;
	    Ntype = N_CHR;
	    ty = "VMPC";
	    break;
	default:
	    if (type > 9999)
		(void) sprintf(Lf->type, "*%03d", type % 1000);
	    else
		(void) sprintf(Lf->type, "%4d", type);
	    (void) strcpy(Namech, "unknown type");
	    ty = (char *)NULL;
	}
	if (ty)
	    (void) strcpy(Lf->type, ty);
	Lf->ntype = Ntype;

#if	defined(HASBLKDEV)
/*
 * If this is a VBLK file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VBLK && Lf->dev_def)
	    find_bl_ino();
#endif	/* defined(HASBLKDEV) */

/*
 * If this is a VCHR file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VCHR && Lf->dev_def)
	    find_ch_ino();
/*
 * Test for specified file.
 */
	if (Sfile && is_file_named(NULL, type, g.gn_chan, ic))
	    Lf->sf |= SELNM;
/*
 * Enter name characters.
 */
	if (Namech[0])
	    enter_nm(Namech);
}


#if	defined(HASPRIVFILETYPE)
/*
 * process_shmt() -- process shared memory transport file
 */

void
process_shmt(sa)
	KA_T sa;			/* shared memory transport node struct
					 * address ??? */
{
	struct shmtnode {		/* shared memory transport node
					 * struct ??? */

	    struct shmtnode *peer;	/* peer shmtnode struct */
	    caddr_t d1[2];		/* dummy to fill space */
	    int sz;			/* buffer size */
	    caddr_t d2[3];		/* dyummy to fill space */
	    int free;			/* free bytes in buffer */
	    caddr_t d3[17];		/* dummy to fill space */
	    pid_t pid;			/* process ID */
	} mn, pn;
/*
 * Ignore this file if only Internet files are selected.
 */
	if (Selinet)
	    return;
/*
 * Set type to " SMT" and put shmtnode structure address in device column.
 */
	(void) strcpy(Lf->type, " SMT");
	if (!sa || kread((KA_T)sa, (char *)&mn, sizeof(mn))) {
	    (void) sprintf(Namech, "can't read shmtnode: %s",
		print_kptr(sa, (char *)NULL));
	    enter_nm(Namech);
	    return;
	}
	enter_dev_ch(print_kptr(sa, (char *)NULL));
/*
 * If offset display has been requested or if buffer size less free bytes is
 * negative, enable offset display.  Otherwise set the  file size as buffer
 * size less free bytes.
 */
	if (Foffset || mn.free > mn.sz)
	    Lf->off_def = 1;
	else {
	    Lf->sz = (SZOFFTYPE)(mn.sz - mn.free);
	    Lf->sz_def = 1;
	}
/*
 * If there is a peer, read its shmtnode structure.
 */
	if (!mn.peer)
	    (void) strcpy(Namech, "->(unknown)");
	else {
	    if (kread((KA_T)mn.peer, (char *)&pn, sizeof(pn)))
		(void) sprintf(Namech, "can't read peer shmtnode: %s",
		    print_kptr((KA_T)mn.peer, (char *)NULL));
	    else {
		if (pn.pid)
		    (void) sprintf(Namech, "->%s (PID %d)",
			print_kptr((KA_T)mn.peer, (char *)NULL), pn.pid);
		else
		    (void) sprintf(Namech, "->%s",
			print_kptr((KA_T)mn.peer, (char *)NULL));
	    }
	}
	enter_nm(Namech);
}
#endif	/* AIXV>=4200 */
