/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/* 
 * models.c
 */

/* 	code to setup a new model        */


#include <stdio.h>

#include <modellib.h>
#include <constants.h>
#include <pm.h>
#include <model_headers.h>

void
load_model()
{
  int n = *((int *) pm(GET, "Model.Load_Number", NULL));
  int parser_ds_init();
  int lorenz_init(void);
  
  if (n >= N_DS + N_USER_DS) {
    system_mess_proc(1, "Invalid dynamical system requested.");
    return;
  }

  /* clean up old model */
  pm(EXEC, "Model.Clean", NULL);
  pm(INIT, "Model.Clean", NULL);

  /* clear custom installation routine */
  pm(INIT, "Model.Install", NULL);
  
  pm(INIT, "Model.Name", MAX_LEN_DS_TITLE,
     INIT, "Model.Initialization", 
     NULL);
 
  if (n<0)
    {
      /* parsed model */
      pm(PUT, "Model.Name", "Parsed dynamical system",
	 PUT, "Model.Initialization", parser_ds_init,
	 NULL);
    }
  else if (n < N_DS)
    {
      /* system model
      fprintf( stdout, "\nsystem model: %s", DS_Sel[n].DS_Name);
      */

      pm(PUT,  "Model.Name", DS_Sel[n].DS_Name, 
	 PUT,  "Model.Initialization", DS_Sel[n].ds_init,
	 NULL);
    }
  else
    {
      /* user model
      fprintf( stdout, "\nuser model: %s", USER_DS_Sel[n].DS_Name);
      */

      n -= N_DS;
      pm(PUT,  "Model.Name", USER_DS_Sel[n].DS_Name, 
	 PUT,  "Model.Initialization", USER_DS_Sel[n].ds_init,
	 NULL);
    }
  
  pm(EXEC, "Model.Initialization", NULL);
  pm_exec_all("Reset");
  pm(EXEC, "Model.Install", NULL);
}


/* ------------------------------------------------------------------------ 
   fetches the currently installed functional values.
   ------------------------------------------------------------------------ */
void
get_ds_func(double *func, double *varb, double *param)
{
  int 	(*aux_f)( double* f, double* x, double* p);
  
  aux_f = (void *) pm( GET, "Model.Aux_Function", NULL );
  if (aux_f)
    aux_f(func, varb, param);
}

/* ------------------------------------------------------------------------ 
   fetches the currently installed geomview values.
   ------------------------------------------------------------------------ */
void
get_gv_func(double *func, double *varb, double *param)
{
  int 	(*aux_f)( double* f, double* x, double* p);
  
  aux_f = (void *) pm( GET, "Model.GV_Function", NULL );
  if (aux_f)
    aux_f(func, varb, param);
}



/* ------------------------------------------------------------------------ 
   returns TRUE if the currently installed dynamical system has
   functions associated with it and FALSE otherwise.

   ------------------------------------------------------------------------ */
int
valid_func()
{
	if( (void *) pm ( GET, "Model.Aux_Function", NULL ) ) 
	   return(TRUE);
	else
	   return(FALSE);
}



/* -----------------------------------------------------------------------------  
   returns a ptr to the name of the current dynamical system  

   ----------------------------------------------------------------------------- */
static char the_name[MAX_LEN_DS_TITLE];
char *
get_ds_name()
{
  pm (GET, "Model.Name", the_name, NULL);
  return ( the_name );
}



/* 
   initialize variables for the new model
*/
void
varb_setup(int n_varb, char **variable_names,
	   double *variables, double *variable_min, double *variable_max,
	   char *indep_varb_name, double indep_varb_min, double indep_varb_max)
{
  int	i;
  
  pm( PUT, "Model.Varb_Dim", n_varb+1, NULL);
  pm( INIT, "Model.Varb_Names", n_varb+1, MAX_LEN_VARB_NAME, NULL);
  for (i=0;i<n_varb; i++)
    pm( PUT, "Model.Varb_Names", i, variable_names[i], NULL);
  pm( PUT, "Model.Varb_Names", n_varb, indep_varb_name, NULL);
  pm( INIT, "Model.Varb_Ic", n_varb+1,
     INIT, "Selected.Varb_Ic", n_varb+1,
     INIT, "Selected.Varb_Fc", n_varb+1,
     PUT_LIST, "Model.Varb_Ic", 0, n_varb-1, variables,
     PUT_LIST, "Selected.Varb_Ic", 0, n_varb-1, variables, 
     PUT_LIST, "Selected.Varb_Fc", 0, n_varb-1, variables, 
     NULL);
  pm( INIT, "Model.Varb_Min", n_varb+1, NULL);
  pm( INIT, "Model.Varb_Max", n_varb+1, NULL);
  pm( PUT_LIST, "Model.Varb_Min", 0, n_varb-1, variable_min, NULL);
  pm( PUT, "Model.Varb_Min", n_varb, indep_varb_min, NULL);
  pm( PUT_LIST, "Model.Varb_Max", 0, n_varb-1, variable_max, NULL);
  pm( PUT, "Model.Varb_Max", n_varb, indep_varb_max, NULL);
}


/*
  initialize parameters and auxiliary functions
*/
void
param_func_setup(int n_param, int n_funct,
		 char **parameter_names, char **funct_names,
		 double *parameters,
		 double *parameter_min, double *parameter_max,
		 double *funct_min, double *funct_max )
{
  int i;

  pm( PUT, "Model.Param_Dim", n_param, NULL);
  pm( INIT, "Model.Param_Names", n_param, MAX_LEN_VARB_NAME, NULL);
  for( i=0; i<n_param; i++)
    pm( PUT, "Model.Param_Names", i, parameter_names[i], NULL);
 
  pm( PUT, "Model.Funct_Dim", n_funct, NULL);
  pm( INIT, "Model.Funct_Names", n_funct, MAX_LEN_VARB_NAME, NULL);
  for( i=0; i<n_funct; i++ )
    pm( PUT, "Model.Funct_Names", i, funct_names[i], NULL);

  pm( INIT, "Model.Param_Ic", n_param,
      INIT, "Selected.Param_Ic", n_param,
      INIT, "Selected.Param_Fc", n_param,
      PUT_LIST, "Model.Param_Ic", 0, n_param-1, parameters, 
      PUT_LIST, "Selected.Param_Ic", 0, n_param-1, parameters, 
      PUT_LIST, "Selected.Param_Fc", 0, n_param-1, parameters, 
      NULL);

  pm( INIT, "Model.Param_Min", n_param,
      INIT, "Model.Param_Max", n_param,
      INIT, "Model.Funct_Min", n_funct,
      INIT, "Model.Funct_Max", n_funct,
      PUT_LIST, "Model.Param_Min", 0, n_param-1, parameter_min,
      PUT_LIST, "Model.Param_Max", 0, n_param-1, parameter_max,
      PUT_LIST, "Model.Funct_Min", 0, n_funct-1, funct_min, 
      PUT_LIST, "Model.Funct_Max", 0, n_funct-1, funct_max, NULL);
}

/* 
   initialize phase space for the new model
*/
void
phase_sp_setup(int n_varb, int manifold_type,
	       int *periodic_varb, double *period_start, double *period_end )
{

  pm( INIT, "Manifold.Type", 
      INIT, "Manifold.Periodic_Varb", n_varb, 
      INIT, "Manifold.Period_Start", n_varb,
      INIT, "Manifold.Period_End", n_varb,
      PUT, "Manifold.Type", manifold_type,
      PUT_LIST, "Manifold.Periodic_Varb", 0, n_varb-1, periodic_varb,
      PUT_LIST, "Manifold.Period_Start", 0, n_varb-1, period_start,
      PUT_LIST, "Manifold.Period_End", 0, n_varb-1, period_end, NULL );
}

void
geomview_setup(int gv_n, char *gv_filename)
{
  pm(PUT, "Model.GV_N", gv_n,
     PUT, "Model.GV_Filename", gv_filename,
     NULL);
}
