/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/* 
 * dbl_pend_def.c
 */

#include <model_headers.h>

/*
        This vector field describes the motion of a double pendulum.
        The vector field is given by
	theta' = thetadot
	phi' = phidot
	thetadot' = ( sin(phi) cos(theta-phi) - mu sin(theta) - sin(theta-phi)
	              (l1 thetadot^2 cos(theta-phi) + l2 phidot^2) )
		      / ( l1 (mu - cos(theta-phi)^2 ) )
	phidot' = ( mu sin(theta) cos(theta-phi) - mu sin(phi) + sin(theta-phi)
	            (mu l1 thetadot^2 + l2 phidot^2 cos(theta-phi) ) )
		    / (l2 (mu - cos(theta-phi)^2 ) )
	where
	mu = 1-m1/m2

	Translation table:

		x[0] <--> theta
		x[1] <--> phi
		x[2] <--> thetadot
		x[3] <--> phidot
		x[4] <--> time
		p[0] <--> m1/m2         ratio of masses of bobs
		p[1] <--> l1            length of first rod
		p[2] <--> l2            length of second rod

*/
/* ------------------------------------------------------------------------
   proc used to define the vector field 
   ------------------------------------------------------------------------ */
int 
dblpend( double* f, double* x, double* p )
{
  double TEMP[5], mu;

  mu = 1+p[0];
  TEMP[0] = cos(x[0]-x[1]);
  TEMP[1] = sin(x[0]-x[1]);
  TEMP[2] = sin(x[0]);
  TEMP[3] = sin(x[1]);
  TEMP[4] = mu - TEMP[0]*TEMP[0];

  f[0] = x[2];
  f[1] = x[3];
  f[2] = ( TEMP[3]*TEMP[0] - mu*TEMP[2] - TEMP[1]*(p[1]*x[2]*x[2]*TEMP[0]+p[2]*x[3]*x[3]) )/
    (p[1]*TEMP[4]);
  f[3] = ( mu*TEMP[2]*TEMP[0]-mu*TEMP[3] + TEMP[1]*(mu*p[1]*x[2]*x[2]+p[2]*x[3]*x[3]) )/
    (p[2]*TEMP[4]);

  return(0);

}

/* ------------------------------------------------------------------------
   proc used to define functions of the variable, parameters or derivatives
   ------------------------------------------------------------------------ */
int 
dblpend_func(double* f, double* x, double* p)
{
  double mu = 1+p[0];

  f[0] = mu*p[1]*p[1]*x[2]*x[2]/2 + p[2]*p[2]*x[3]*x[3]/2
    + p[1]*p[2]*x[2]*x[3]*cos(x[0]-x[1])
      - mu*p[1]*cos(x[0]) - p[2]*cos(x[1]);

  return 0;
}
/* ------------------------------------------------------------------------
   proc used to define jacobian
   ------------------------------------------------------------------------ */
int 
dblpend_jac(double** m, double* x, double* p)
{
  return 0;
}

/* ---------------------------------
  proc for geomview
  ---------------------------------  */
int 
dblpend_gv(double* f, double* x, double* p)
{
  int i;
  double a,b,c0,c1,s0,s1,fac;

  a = p[1]+p[2];
  b = p[1]/p[2];
  c0 = cos(x[0]);
  c1 = cos(x[1]);
  s0 = sin(x[0]);
  s1 = sin(x[1]);

  for (i=0; i<64; i++) f[i] = 0;

  /* rotate and scale rod 0 */
  f[0] = f[5] = c0;
  f[1] = s0;
  f[4] = -s0;
  f[10] = 1.0;
  f[15] = a/p[1];

  /* rotate, scale, and shift rod 1 */
  f[16] = f[21] = c1;
  f[17] = s1;
  f[20] = -s1;
  f[26] = 1.0;
  f[28] = s0*b;
  f[29] = -c0*b;
  f[31] = a/p[2];

  /* rotate and shift bob 0 */
  f[32] = f[37] = c0;
  f[33] = s0;
  f[36] = -s0;
  f[42] = 1.0;
  f[44] = p[1]/a*s0;
  f[45] = -p[1]/a*c0;
  f[47] = 1.0;

  /* rotate and shift bob 1 */
  f[48] = f[53] = c1;
  f[49] = s1;
  f[52] = -s1;
  f[58] = 1.0;
  f[60] = p[1]/a*s0 + p[2]/a*s1;
  f[61] = -p[1]/a*c0 - p[2]/a*c1;
  f[63] = 1.0;

  /* scale bobs for relative mass */
  if (p[0]>=1.0) {
    fac = sqrt(p[0]);
    f[32] = fac*f[32];
    f[33] = fac*f[33];
    f[42] = fac*f[42];
  } else {
    fac = sqrt(1/p[0]);
    f[48] = fac*f[48];
    f[49] = fac*f[49];
    f[58] = fac*f[58];
  }

  return 0;
}

/* ------------------------------------------------------------------------
   proc to define the default data for the dynamical system
   Note: You may change the entries for each variable but
	 DO NOT change the list of items.  If a variable is
	 unused, NULL or zero the entry, as appropriate.
   ------------------------------------------------------------------------ */
int 
dblpend_init()
{

  /* define the dynamical system in this segment 
     ------------------------------------------------------------------------------------------- */
  int	 n_varb=4;
  static char	*variable_names[]={"theta","phi", "thetadot", "phidot"};
  static double	variables[]={0.9,0.7,1.4,1.0};
  static double	variable_min[]={-3.2,-3.2,-5.0,-5.0};
  static double	variable_max[]={3.2,3.2,5.0,5.0};
  static char   *indep_varb_name="time";
  double indep_varb_min=0.0;
  double indep_varb_max=100.0;

  int	 n_param=3;
  static char	*parameter_names[]={"m1/m2","l1","l2"};
  static double	parameters[]={3.0,1.5,1.0};
  static double	parameter_min[]={0,0.,0.,0.};
  static double	parameter_max[]={5.,2.,2.,2.};

  int	 n_funct=1;
  static char	*funct_names[]={"Energy"};
  static double	funct_min[]={-1};
  static double	funct_max[]={1};

  int	 manifold_type = PERIODIC;
  static int	periodic_varb[] = {TRUE, TRUE, FALSE, FALSE};
  static double period_start[] = {-PI,-PI,0.0,0.0};
  static double period_end[] = {PI, PI, 0.0, 0.0};

  int 	 mapping_toggle=FALSE;
  int	 inverse_toggle=FALSE;

  int           (*def_name)()=dblpend;		  /* the eqns of motion */
  int           (*jac_name)()=NULL;		  /* the jacobian (deriv w.r.t. space) */
  int           (*aux_func_name)()=dblpend_func;  /* the auxiliary functions */
  int           (*inv_name)()=NULL;			  /* the inverse or approx inverse */
  int           (*dfdt_name)()=NULL;			  /* the deriv w.r.t time */
  int           (*dfdparam_name)()=NULL;		  /* the derivs w.r.t. parameters */

#define GV
  int (*gv_funct)()=dblpend_gv;   /* transformations for geomview */
  int (*gv_custom_funct)()=NULL;    /* nontransformation geomview commands */
  int gv_n=4;                       /* number of transformations */
  static char *gv_filename = "dbl_pendulum.oogl";

  c_filename = __FILE__;  /* display this file for help with this system */

/* ------------------ end of dynamical system definition ------------------ */
#include <ds_define.c>

  return 0;
}

