/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/* 
 * gv_install.c
 */

/*
 * Create GEOMVIEW postmaster object
 */
#include <stdio.h>
#include <string.h>

#include "constants.h"
#include "defaults.h"
#include "pm.h"
#include "utilities.h"
#include "math_utils.h"
#include "view.h"
#include "modellib.h"
#include "init_local.h"

static char *GEOMVW_OBJ_NAME = "Geomview";

static char *GEOMVW[] = {
  "Geomview.Open",
  "Geomview.Close",
  "Geomview.Active",
  "Geomview.Directory",
  "Geomview.Filename",
  "Geomview.Selected",
  "Geomview.Sendfile",
  "Geomview.Clear",
  "Geomview.Reset",
  "Geomview.Index1",
  "Geomview.Index2",
  "Geomview.Index3",
  "Geomview.Send_Memory",
  "Geomview.Wait",
  "Geomview.PickColor"
  };

typedef enum {
  OPEN=0, CLOSE, ACTIVE, DIRECTORY, FILENAME, SELECTED,
  SENDFILE, GV_CLEAR, RESET, INDEX1, INDEX2, INDEX3,
  SEND_MEMORY, WAIT, PICKCOLOR
  } GEOMVIEW_t;

static void geomview_open(void);
static void geomview_close(void);
static void geomview_selected(void);
static void geomview_sendfile(void);
static void geomview_clear(void);
static void geomview_reset(void);
static void geomview_send_memory(void);

void
geomview_install(void)
{
  char dirname[SIZE_OF_DIR];

  pm(CREATE_OBJ, GEOMVW_OBJ_NAME,
     CREATE_ELEM, GEOMVW[OPEN], FNCT,
     CREATE_ELEM, GEOMVW[CLOSE], FNCT,
     CREATE_ELEM, GEOMVW[ACTIVE], INT,
     CREATE_ELEM, GEOMVW[DIRECTORY], STRNG,
     CREATE_ELEM, GEOMVW[FILENAME], STRNG,
     CREATE_ELEM, GEOMVW[SELECTED], FNCT,
     CREATE_ELEM, GEOMVW[SENDFILE], FNCT,
     CREATE_ELEM, GEOMVW[GV_CLEAR], FNCT,
     CREATE_ELEM, GEOMVW[RESET], FNCT,
     CREATE_ELEM, GEOMVW[INDEX1], INT,
     CREATE_ELEM, GEOMVW[INDEX2], INT,
     CREATE_ELEM, GEOMVW[INDEX3], INT,
     CREATE_ELEM, GEOMVW[SEND_MEMORY], FNCT,
     CREATE_ELEM, GEOMVW[WAIT], FNCT,
     CREATE_ELEM, GEOMVW[PICKCOLOR], INT,
     NULL);

  get_dstool_path(dirname, DSTOOL_OOGL_DIR);

  pm(INIT, GEOMVW[OPEN],
     PUT, GEOMVW[OPEN], geomview_open,
     INIT, GEOMVW[CLOSE],
     PUT, GEOMVW[CLOSE], geomview_close,
     PUT, GEOMVW[ACTIVE], FALSE,
     INIT, GEOMVW[DIRECTORY], SIZE_OF_DIR,
     PUT, GEOMVW[DIRECTORY], dirname,
     INIT, GEOMVW[FILENAME], SIZE_OF_FNAME,
     INIT, GEOMVW[SELECTED],
     PUT, GEOMVW[SELECTED], geomview_selected,
     INIT, GEOMVW[SENDFILE],
     PUT, GEOMVW[SENDFILE], geomview_sendfile,
     INIT, GEOMVW[GV_CLEAR],
     PUT, GEOMVW[GV_CLEAR], geomview_clear,
     INIT, GEOMVW[RESET],
     PUT, GEOMVW[RESET], geomview_reset,
     INIT, GEOMVW[SEND_MEMORY],
     PUT, GEOMVW[SEND_MEMORY], geomview_send_memory,
     INIT, GEOMVW[WAIT],
     PUT, GEOMVW[WAIT], geomview_wait,
     PUT, GEOMVW[PICKCOLOR], FALSE,
     NULL);
}

static void
geomview_reset(void)
{
  char dirname[SIZE_OF_DIR];
  char fname[SIZE_OF_FNAME];

  /* load up new geomview filename */
  pm(GET, "Model.GV_Filename", fname,
     PUT, GEOMVW[FILENAME], fname,
     NULL);
  /* choose directory */
  if (*((int *) pm(GET, "Model.Load_Number", NULL)) <
      *((int *) pm(GET, "Model.N_Sys_Models", NULL)))
    get_dstool_path(dirname, DSTOOL_OOGL_DIR);
  else get_dstool_path(dirname, MY_DSTOOL_OOGL_DIR);
  pm(PUT, GEOMVW[DIRECTORY], dirname, NULL);
  
  pm(EXEC, GEOMVW[GV_CLEAR],
     EXEC, GEOMVW[SENDFILE],
     EXEC, GEOMVW[SELECTED],
     NULL);

  pm(PUT, GEOMVW[INDEX1], 0,
     PUT, GEOMVW[INDEX2], 1,
     PUT, GEOMVW[INDEX3], 1,
     PUT, GEOMVW[PICKCOLOR], FALSE,
     NULL);
  if (2 < *((int *) pm(GET, "Model.Varb_Dim", NULL)) + 
      *((int *) pm(GET, "Model.Param_Dim", NULL)) +
      *((int *) pm(GET, "Model.Funct_Dim", NULL)))
    pm(PUT, GEOMVW[INDEX3], 2, NULL);

}

FILE *gv_to = NULL;
FILE *gv_from = NULL;

static char gv_init_string[] = "(progn\n(normalization allgeoms none))\n";

static void
geomview_open(void)
{
  if ( gv_to == NULL) 
    {  
      open_geomview(&gv_to, &gv_from);
      if ( NULL == gv_to)
	{
	  fprintf(stderr, "DsTool: Failed to start geomview\n");
	  return;
	}
    }
  geomview_send(gv_init_string);

  pm(PUT, GEOMVW[ACTIVE], TRUE, NULL);

}

static void
geomview_close(void)
{
  fclose(gv_to);
  fclose(gv_from);
  pm(PUT, GEOMVW[ACTIVE], FALSE, NULL);
  gv_to = NULL;
  gv_from = NULL;
}

void
geomview_clear(void)
{
  geomview_send("(delete DsTool)\n");
}


static void
geomview_selected(void)
{

  int gv_n = *(( int *) pm(GET, "Model.GV_N", NULL));
  int n_varb = *(( int *) pm(GET, "Model.Varb_Dim", NULL));
  int n_param = *(( int *) pm(GET, "Model.Param_Dim", NULL));
  int (*gv_custom)(double *, double *);

  double *v = dvector(0, n_varb-1);
  double *p = dvector(0, n_param-1);
  pm(GET_LIST, "Selected.Varb_Ic", 0, n_varb-1, v, NULL);
  pm(GET_LIST, "Selected.Param_Ic", 0, n_param-1, p, NULL);

  if (gv_n > 0)
    {
      double *f = dvector(0, 16*gv_n-1);
      get_gv_func(f,v,p);
      geomview_transform(gv_n, f);
      free_dvector(f,0,16*gv_n-1);
    }  

  gv_custom = (int (*)(double *, double *)) pm(GET, "Model.GV_Custom", NULL);
  if (NULL != gv_custom)
    gv_custom(v,p);
  free_dvector(v,0,n_varb-1);
  free_dvector(p,0,n_param-1);
}

void
geomview_sendfile(void)
{

  FILE *fp;
  char pc[MAX_LONG_STR];

  /* if no filename specified then return with no error message */
  pm(GET, GEOMVW[FILENAME], pc, NULL);
  if (strlen(pc) == 0) return;

  if ( gv_to == NULL )
    {
      system_mess_proc(0,"Geomview is not active.");
      return;
    }

  fp = open_pm_file(GEOMVW[FILENAME], GEOMVW[DIRECTORY]);
  if (fp == NULL)
    {
      system_mess_proc(0,"Cannot open file for Geomview.");
    }
  else
    {
      while (fgets(pc, MAX_LONG_STR, fp) != NULL)
	{
	  geomview_send(pc);
	}
    }
  
}

void
geomview_send_memory(void)
{
  int x[3], xtypes[3], xindices[3], i;
  double mins[3], maxs[3];

#ifdef DEBUG
  fprintf(stderr, "In geomview_send_memory.\n");
#endif

  x[0] = *((int *) pm(GET, GEOMVW[INDEX1], NULL));
  x[1] = *((int *) pm(GET, GEOMVW[INDEX2], NULL));
  x[2] = *((int *) pm(GET, GEOMVW[INDEX3], NULL));

  for (i=0; i<3; i++)
    {
      xtypes[i] = index_to_type(x[i]);
      xindices[i] = index_to_count(x[i]);
      switch(xtypes[i])
	{
	case PHASE_SPACE_VARB:
	  mins[i] = *((double *) pm(GET, "Defaults.Varb_Min", 
				    xindices[i], NULL));
	  maxs[i] = *((double *) pm(GET, "Defaults.Varb_Max", 
				    xindices[i], NULL));
	  break;
	case PARAMETER_VARB:
	  mins[i] = *((double *) pm(GET, "Defaults.Param_Min", 
				    xindices[i], NULL));
	  maxs[i] = *((double *) pm(GET, "Defaults.Param_Max", 
				    xindices[i], NULL));
	  break;
	case FUNCTION_VARB:
	  mins[i] = *((double *) pm(GET, "Defaults.Funct_Min", 
				    xindices[i], NULL));
	  maxs[i] = *((double *) pm(GET, "Defaults.Funct_Max", 
				    xindices[i], NULL));
	  break;
	default:
	  /* other selection so this coordinate will have value 0 */
	  mins[i] = 0.0;
	  maxs[i] = 0.0;
	  break;
	}
    }

  gv_allmem(xtypes,xindices,mins, maxs);
}






