package com.jclark.xsl.sax;

import com.jclark.xsl.tr.*;
import com.jclark.xsl.om.*;
import org.xml.sax.*;

public abstract class ResultBase implements Result, AttributeList {

  private DocumentHandler documentHandler;
  private CommentHandler commentHandler;
  static private final int INITIAL_BUF_SIZE = 8192;
  private char[] buf = new char[INITIAL_BUF_SIZE];
  private int bufUsed = 0;
  private Name[] attributeNames = new Name[10];
  private String[] attributeValues = new String[10];
  private int nAttributes;
  private Name pendingElementType;
  private NamespacePrefixMap pendingNamespacePrefixMap;
  
  protected ResultBase(DocumentHandler handler) {
    this.documentHandler = handler;
    setCommentHandler();
  }

  private void setCommentHandler() {
    if (documentHandler instanceof CommentHandler)
      commentHandler = (CommentHandler)documentHandler;
    else
      commentHandler = null;
  }

  protected void flush() throws XSLException {
    if (pendingElementType != null) {
      startElementContent(pendingElementType, pendingNamespacePrefixMap);
      pendingElementType = null;
    }
    else if (bufUsed > 0) {
      try {
	documentHandler.characters(buf, 0, bufUsed);
	bufUsed = 0;
      }
      catch (SAXException e) {
	throwXSLException(e);
      }
    }
  }

  public void characters(String str) throws XSLException {
    if (pendingElementType != null)
      flush();
    int strLength = str.length();
    if (bufUsed + strLength > buf.length) {
      char[] oldBuf = buf;
      int newLen = oldBuf.length * 2;
      while (newLen < bufUsed + strLength)
	newLen *= 2;
      buf = new char[newLen];
      if (bufUsed > 0)
	System.arraycopy(oldBuf, 0, buf, 0, bufUsed);
    }
    str.getChars(0, strLength, buf, bufUsed);
    bufUsed += strLength;
  }

  public void comment(String str) throws XSLException {
    if (commentHandler != null) {
      try {
	flush();
	commentHandler.comment(fixComment(str));
      }
      catch (SAXException e) {
	throwXSLException(e);
      }
    }
  }

  private static final String fixComment(String str) {
    int i = str.indexOf('-');
    while (i++ >= 0) {
      int len = str.length();
      if (i == len)
	return str + " ";
      if (str.charAt(i) == '-')
	str = str.substring(0, i) + " " + str.substring(i);
      i = str.indexOf('-', i);
    }
    return str;
  }

  public void processingInstruction(String target, String data)
    throws XSLException {
    try {
      flush();
      documentHandler.processingInstruction(target,
					    fixProcessingInstruction(data));
    }
    catch (SAXException e) {
      throwXSLException(e);
    }
  }

  private static final String fixProcessingInstruction(String str) {
    int i = str.indexOf('?');
    while (i++ >= 0) {
      int len = str.length();
      if (i == len)
	break;
      if (str.charAt(i) == '>')
	str = str.substring(0, i) + " " + str.substring(i);
      i = str.indexOf('?', i);
    }
    return str;
  }

  public void startElement(Name elementType, NamespacePrefixMap nsMap) 
    throws XSLException {
    flush();
    pendingElementType = elementType;
    pendingNamespacePrefixMap = nsMap;
    nAttributes = 0;
  }

  public void endElement(Name elementType) throws XSLException {
    flush();
    endElementContent(elementType);
  }

  protected final DocumentHandler getDocumentHandler() {
    return documentHandler;
  }

  public int getLength() {
    return nAttributes;
  }

  protected final Name getAttributeName(int i) {
    return attributeNames[i];
  }

  public String getValue(int i) {
    return attributeValues[i];
  }

  public String getType(int i) {
    return "CDATA";
  }

  public String getType(String name) {
    return "CDATA";
  }

  public String getValue(String name) {
    int len = getLength();
    for (int i = 0; i < len; i++) {
      if (name.equals(getName(i)))
	return getValue(i);
    }
    return null;
  }

  protected abstract void startElementContent(Name elementType,
					      NamespacePrefixMap nsMap)
    throws XSLException;

  protected abstract void endElementContent(Name elementType)
    throws XSLException;

  public void attribute(Name name, String value) throws XSLException {
    if (pendingElementType == null)
      return;
    for (int i = 0; i < nAttributes; i++)
      if (attributeNames[i].equals(name)) {
	attributeValues[i] = value;
	return;
      }
    if (nAttributes == attributeNames.length) {
      attributeNames = grow(attributeNames);
      attributeValues = grow(attributeValues);
    }
    attributeNames[nAttributes] = name;
    attributeValues[nAttributes] = value;
    nAttributes++;
  }

  static String[] grow(String[] v) {
    String[] old = v;
    v = new String[old.length * 2];
    System.arraycopy(old, 0, v, 0, old.length);
    return v;
  }

  static Name[] grow(Name[] v) {
    Name[] old = v;
    v = new Name[old.length * 2];
    System.arraycopy(old, 0, v, 0, old.length);
    return v;
  }

  public void start() throws XSLException {
    try {
      documentHandler.startDocument();
    }
    catch (SAXException e) {
      throwXSLException(e);
    }
  }

  public void end() throws XSLException {
    try {
      flush();
      documentHandler.endDocument();
    }
    catch (SAXException e) {
      throwXSLException(e);
    }
  }

  protected void throwXSLException(SAXException e) throws XSLException {
    Exception wrapped = e.getException();
    if (wrapped != null)
      throw new XSLException(wrapped);
    else
      throw new XSLException(e.getMessage());
  }

  public abstract void resultTreeFragment(ResultTreeFragment frag) throws XSLException;
}
