/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*             Author :  Paul Taylor and Alan Black                      */
/*             Date   :  June 1996                                       */
/*-----------------------------------------------------------------------*/
/*             more EST_Wave class methods                                   */
/*                                                                       */
/*=======================================================================*/

#include <string.h>
#include "EST_unix.h"
#include <stdlib.h>
#include "EST_cutils.h"
#include "EST_string_aux.h"
#include "EST_Wave.h"
#include "EST_wave_utils.h"
#include "EST_wave_aux.h"
#include "EST_io_aux.h"
#include "EST_Stream.h"
#include "EST_stream_aux.h"

int wave_extract_channel(EST_Wave &single, const EST_Wave &multi, int channel)
{
    int i, k;
    int n = multi.num_samples();
    int c = multi.num_channels();
    short *sig, *msig;
    
    if (channel >= c)
    {
	cerr << "Can't extract channel " << channel << " from " << 
	    c << " channel waveform\n";
	return -1;
    }

    sig = new short[n];
    msig = multi.data();

    for (i = k = 0; i < n; ++i, k+=c)
	sig[i] =  msig[k + channel];

    single.set_data(sig,n,multi.sample_rate(),1);

    return 0;
}

void wave_combine_channels(EST_Wave &s,const EST_Wave &m)
{
    // Combine all channels into a single channel
    short *c,*d;
    int i,k,sum;

    d = m.data();

    c = new short[m.num_samples()];
    for (i=0; i < (m.num_samples()*m.num_channels()); i+=m.num_channels())
    {
	for (sum=k=0; k < m.num_channels(); k++)
	    sum += d[i+k];
	c[i/m.num_channels()] = sum / m.num_channels();
    }

    s.set_data(c,m.num_samples(),m.sample_rate(),1);

}

void invert_waveform(EST_Wave &sig)
{
    for (int i = 0; i < sig.num_samples(); ++i)
	sig.a(i) = sig.a(i) * -1;
}

int wave_subwave(EST_Wave &subsig,EST_Wave &sig, float offset, float length)
{
    return wave_subwave(subsig, sig, (int)(offset *(float)sig.sample_rate()),
			(int)(length *(float)sig.sample_rate()));
}

int wave_subwave(EST_Wave &subsig,EST_Wave &sig,int offset,int length)
{
    // take out a subpart of sig and put it in subsig
    int ns;
    short *s,*d;

    if (length == -1)
	ns = sig.num_samples() - offset;
    else
	ns = length;
    
    if ((offset+ns) > sig.num_samples())
    {
	cerr << "Subset past end of signal\n";
	return -1;
    }

    s = new short[ns*sig.num_channels()];
    d = sig.data();
    memmove(s,&d[offset*sig.num_channels()],
	    ns*sig.num_channels()*sizeof(short));

    subsig.set_data(s,ns,sig.sample_rate(),sig.num_channels());

    return 0;
}

static int parse_esps_r_option(EST_String arg, int &offset, int &length)
{
    EST_String s, e;

    if (arg.contains("-"))
    {
	s = arg.before("-");
	e = arg.after("-");
    }
    else if (arg.contains(":"))
    {
	s = arg.before(":");
	e = arg.after(":");
    }
    else
    {
	cerr << "Argument to -r is illformed " << arg << endl;
	return -1;
    }
    
    if (!s.matches(RXint))
    {
	cerr << "First argument to -r must be an integer " << arg << endl;
	return -1;
    }

    offset = atoi(s);
    if (e.contains("+"))
    {	
	e = e.after("+");
	length = atoi(e);
    }
    else
	length = atoi(e) - offset;

    if (length <= 0)
    {
	cerr << "length is negative or zero " << arg << endl;
	return -1;
    }

    return 0;
}

int wave_divide(EST_WaveList &wl, EST_Wave &sig, EST_Stream &keylab,
		const EST_String &ext)
{
    wl.clear();
    EST_Wave a;
    EST_Stream_Item *k;
    EST_String filename;
    
    for (k = keylab.head(); k; k = next(k))
    {
	a.clear();
	if (k->dur() < 0)
	    continue;
	wave_subwave(a, sig, k->start(), k->dur());
	find_filename(k->fields, filename);
	a.set_name(filename + ext);
	wl.append(a);
    }

    return 0;
}

int wave_extract(EST_Wave &part, EST_Wave &sig, EST_Stream &keylab, 
		 const EST_String &file)
{
    EST_Wave a;
    EST_Stream_Item *k;
    EST_String key_file_name;
    
    for (k = keylab.head(); k; k = next(k))
    {
	find_filename(k->fields, key_file_name);
	if (key_file_name == file)
	{
	    wave_subwave(part, sig, k->start(), k->dur());
	    return (0);
	}
    }
    cerr << "Couldn't locate file fragment " << file << " in keylab file\n";
    return -1;
}

int read_wave(EST_Wave &sig, const EST_String &in_file, EST_Option &al)
{
    char *sr;
    EST_String fname, file_type, sample_type;
    int sample_rate;
    EST_read_status rval;
    int num_channels;
    int offset, length;
    int bo;
    
    if (in_file == "-")
	fname = stdin_to_file();
    else
	fname = in_file;

    if (al.present("-n"))
	num_channels = al.ival("-n", 0);
    else 
	num_channels = 1;

    if (al.present("-ulaw"))
    {
	al.add_item("-itype","ulaw");
	al.add_item("-f","8000");
    }
    if (al.present("-iswap"))
	al.add_item("-ibo","other");

    if (al.present("-istype"))
	sample_type = al.val("-istype");
    else 
	sample_type = sig.sample_type(); // else default type;

    if (al.present("-itype"))
	file_type = al.val("-itype");  // else default type;
    else
	file_type = "undef";

    if (al.present("-f"))
	sample_rate = al.ival("-f", 0);
    else if ((sr = getenv("NA_PLAY_SAMPLE_RATE")) != NULL)
    {
	sample_rate = atoi(sr);
	cerr << "Warning: no sample rate specified, " <<
	    " using NA_PLAY_SAMPLE_RATE environment variable\n";
    }
    else
    {
	sample_rate = Wave_DEFAULT_SR;
	if (file_type == "raw")
	    cerr << "Warning: no sample rate specified - using default " << 
		sample_rate << endl;
    }
    
    if (file_type == "ulaw")
    {
	sample_rate = 8000;
	sample_type = "mulaw";
    }
	
    if (al.present("-r")) // only load in part of waveform
    {
	if (parse_esps_r_option(al.val("-r"), offset, length) != 0)
	    return -1;
    }
    else
	offset = length = 0;

    if (al.present("-iswap"))
	bo = str_to_bo("swap");
    else
	bo = str_to_bo("native");
    if (al.present("-ibo"))   // can override -iswap
	bo = str_to_bo(al.val("-ibo"));

    rval = sig.load_file(fname,file_type, sample_rate,
			 sample_type, bo, num_channels, offset, length);
    if ((rval == wrong_format) && (al.present("-basic")))
    {
	// For HTML audio/basic, it seems to mean headered or ulaw 8k
	// so try to load it again as ulaw 8k.
	rval = sig.load_file(fname, "raw", 8000,
			     "mulaw", bo, 1, offset, length);
    }
    if (rval != format_ok)
    {
	if (in_file == "-") unlink(fname);
	cerr << "Cannot recognize file format or cannot access file: \"" << in_file << "\"\n";
	return -1;
    }

    if (in_file == "-") unlink(fname);
    return 0;
}

int write_wave(EST_Wave &sig, const EST_String &out_file, EST_Option &al)
{
    EST_String file_type, sample_type;
    int bo;

    if (al.present("-otype"))
	file_type = al.val("-otype");
    else
	file_type = "undef";

    if (al.present("-ostype"))
	sample_type = al.val("-ostype");
    else
	sample_type = "undef";

    if (al.present("-oswap"))
	bo = str_to_bo("swap");
    else
	bo = str_to_bo("native");

    if (al.present("-obo"))   // can over ride -oswap
	bo = str_to_bo(al.val("-obo"));

    if (sig.save_file(out_file, file_type,
		      sample_type, bo, 0,
		      sig.num_samples()) != write_ok)
    {
	cerr << "Cannot write file: \"" << out_file << "\"\n";
	return -1;
    }

    return 0;
}

void wave_info(EST_Wave &w)
{
    EST_String t;
    cout << "Duration: " << 
	ftoString((float)w.num_samples()/(float)w.sample_rate(),4,1) << endl;

    cout << "Sample rate: " << w.sample_rate() << endl;
    cout << "Number of samples: " << w.num_samples() << endl;
    cout << "Number of channels: " << w.num_channels() << endl;
    cout << "Header type: " << w.file_type() << endl;
    cout << "Data type: " << w.sample_type() << endl;
}

static EST_String options_wave_filetypes(void)
{
    // Returns list of currently support wave file types
    // Should be extracted from the list in EST_Wave (but that's
    // not very clear :-(

    return "nist, esps, snd, riff, aiff, audlab, raw, ascii";
}

EST_String options_wave_input(void)
{
    // The standard waveform input options 
    return
	EST_String("")+
        "-itype <string>  Input file type (optional).  If no type is\n"+
        "                 specified type is automatically derived from\n"+
        "                 file's header, in unheaders and not itype file\n"+
        "                 is assumed to be raw (shorts).  Supported types\n"+
        "                 are: "+options_wave_filetypes()+"\n"+
        "-c <int>         Select channel from input data (starts from 0)\n"+
        "-n <int>         Number of channels of raw data\n"+
        "-f <int>         Input file sample rate in Hertz\n"+
        "-ibo <string>    Input byte order: MSB, LSB, native, nonnative\n"+
        "                 Suns, HP, SGI Mips, M68000 are MSB (big endian)\n"+
        "                 Intel, Alpha, DEC Mips, Vax are LSB (little endian)\n"+
        "-iswap           Swap bytes in input\n"+
        "-istype <string> Sample type: short, ulaw, byte, ascii\n"+
        "-ulaw            Assume unheadered input is 8k ulaw\n"+
        "-r*              Select subrange of file. (ESPS compatible)\n"+
        "-basic           HTML audio/basic format, if unheadered treat\n"+
        "                 as ulaw 8K\n";
}

EST_String options_wave_output(void)
{
    return 
	EST_String("")+
	"-o <ofile>       Output filename, if not specified output is\n"+
	"                 to stdout\n"+
        "-otype <string>  Output file type, (optional).  If no type is\n"+
        "                 Specified the type of the input file is assumed\n"+
        "                 types are: "+options_wave_filetypes()+"\n"+
        "-F <int>         Output sample rate in Hertz, will cause resampling\n"+
        "                 if value is different from input sample rate.\n"+
        "-obo <string>    Output byte order MSB, LSB, native, nonnative\n"+
        "                 Suns, HP, SGI Mips, M68000 are MSB (big endian)\n"+
        "                 Intel, Alpha, DEC Mips, Vax are LSB (little endian)\n"+
        "-oswap           Swap bytes when saving to output\n"+
        "-ostype <string> Output sample type: short, ulaw, byte, ascii\n";

}
