/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                    Copyright (c) 1994,1995,1996                       */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Paul Taylor                            */
/*                      Date   :  December 96                            */
/*-----------------------------------------------------------------------*/
/*                    Spectrogram Generation                             */
/*                                                                       */
/*=======================================================================*/
#include <math.h>
#include <limits.h>
#include <float.h>
#include "EST_sigpr.h"

void raw_spectrogram(EST_Track &sp, EST_Wave &sig, float length, float shift,
		    int order);
void scale_spectrogram(EST_Track &s, float range, float b, float w);
float *Float_hamming(int size);

EST_Track make_spectrogram(EST_Wave &sig, EST_Option &op)
{
    EST_Track sp;
    int i;

    // apply pre-emphasis
    for(i = 1; i < sig.num_samples(); i++)
	sig.a(i) = sig(i) - (short((float)sig(i - 1) * 0.95));

    // calculate raw spectrogram
    raw_spectrogram(sp, sig, op.fval("bc_frame_length", 1), 
			      op.fval("bc_frame_shift", 1), 
			      op.ival("bc_frame_order", 1));

    // coerce the values so as to emphasis important features
    scale_spectrogram(sp, op.fval("sp_range"), op.fval("sp_wcut"),
		      op.fval("sp_bcut"));

    return sp;
}

void scale_spectrogram(EST_Track &sp, float range, float wcut, float bcut)
{
    float max, min, scale, v;
    int i, j;

    max = -FLT_MIN;
    min = FLT_MAX;

    // find min and max values
    for (i = 0; i < sp.num_frames(); ++i)
	for (j = 0; j < sp.num_channels(); ++j)
	{
	    if (sp(i, j) > max)
		max = sp(i, j);
	    if (sp(i, j) < min)
		min = sp(i, j);
	}
    scale = (max - min);

    // for every value:
    // 1. Effectively scale in range 0 to 1
    // 2. Impose white and black cut offs
    // 3. Rescale to 0 and 1
    // 4. scale to fit in "range"
    // this can obviously be done more efficiently
    for (i = 0; i < sp.num_frames(); ++i)
	for (j = 0; j < sp.num_channels(); ++j)
	{
	    v = ((((sp(i, j) - min) / scale) - wcut) / (bcut - wcut)) *range;
	    if (v > range) v = range;
	    if (v < 0.0) v = 0.0;
	    sp.a(i, j) = v;
	}
}	    

void raw_spectrogram(EST_Track &sp, EST_Wave &sig, float length, float shift, int order)
{
    int frame_length = (int) (length * (float) sig.sample_rate() );
    int frame_shift = (int) (shift * (float) sig.sample_rate() );
    float *hamming = Float_hamming(frame_length);
    int i, k, pos;
    short *tmp_sig, *window_start;
    int copy_size;

    EST_FVector window(order);
    EST_FVector imag(order);

    copy_size = (frame_length > order) ? order : frame_length;
    
    int num_frames = (int)ceil((float)sig.num_samples() / frame_shift);
    int num_samples = ((num_frames - 1) * frame_shift) + frame_length;

    tmp_sig = new short[num_samples];
    for (i = 0; i < sig.num_samples(); ++i)
	tmp_sig[i] = sig.a(i);

    for (; i < num_samples; ++i)
	tmp_sig[i] = 0;
	
    sp.resize(num_frames, order/2);
    
    for (k = 0 ; k < num_frames; ++k)
    {
	pos = frame_shift * k;
	window_start = tmp_sig + pos;
		
	for(i = 0; i < order; i++)
	    imag(i) = 0.0;
	for(i = 0; i < copy_size; i++)
	    window(i) = hamming[i] * window_start[i];
	if (frame_length < order)
	    for(; i < order; i++)
		window(i) = 0.0;
	
	power_spectrum(window, imag);
	for (i = 0; i < order /2; ++i)
	    sp.a(k, i) = sqrt(window(i));
    }
    sp.fill_time((float)sig.sample_rate() / (float)order);
}
