/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1996,1997                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify and distribute this software and its */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                     Author :  Alan W Black                            */
/*                     Date   :  May 1996                                */
/*-----------------------------------------------------------------------*/
/*  A Classification and Regression Tree (CART) Program                  */
/*  A basic implementation of many of the techniques in                  */
/*  Briemen et al. 1984                                                  */
/*                                                                       */
/*  Added decision list support, Feb 1997                                */
/*                                                                       */
/*=======================================================================*/

#include <stdlib.h>
#include <iostream.h>
#include <fstream.h>
#include <string.h>
#include "EST.h"
#include "EST_Wagon.h"

char *wagon_version = "0.7.0 beta August 1997";

enum wn_strategy_type {wn_decision_list, wn_decision_tree};

static wn_strategy_type wagon_type = wn_decision_tree;
float wgn_float_range_split = 10.0;
int wgn_csubset=FALSE;
int wgn_vfold=1;
int wgn_prune=TRUE;
int wgn_held_out = 0;
float wgn_balance = 0.0;
int wgn_quiet = FALSE;

WDataSet wgn_dataset;
EST_String wgn_test_file("");
EST_String wgn_desc_file("");
int wgn_min_cluster_size = 50;
Discretes wgn_discretes;
ostream *wgn_coutput = 0;
EST_FMatrix wgn_DistMatrix;

static int wagon_main(int argc, char **argv);

int main(int argc, char **argv)
{

    wagon_main(argc,argv);

    exit(0);
    return 0;
}

static int wagon_main(int argc, char **argv)
{
    // Top level function sets up data and creates a tree
    EST_Option al;
    EST_StrList files;

    parse_command_line2(argc, argv,
       EST_String("Usage:\n")+
       "wagon  <options>\n"+
       "Wagon CART building program (defaults in {})\n"+
       "-desc <ifile>     Field description file\n"+
       "-data <ifile>     Datafile, one vector per line\n"+
       "-stop <int> {50}  Minimum number of examples for leaf nodes\n"+
       "-test <ifile>     Datafile to test tree on\n"+
       "-frs <float> {10} Float range split, number of partitions to\n"+
       "                  split a float feature range into\n"+
       "-distmatrix <ifile>\n"+
       "                  A distance matrix for clustering\n"+
       "-dlist            Build a decision list (rather than tree)\n"+
       "-dtree            Build a decision tree (rather than list) default\n"+
       "-output <ofile>   File to save output tree in\n"+
       "-v                Print version number and exit\n"+
       "-quiet            No questions printed during building\n"+
       "The following are *provisional* and may not work\n"+
       "-vfold <int> {1}  Number of folds (not standard vfolds)\n"+
       "-balance <float>  For derived stop size, if dataset at node, divided\n"+
       "                  by balance is greater than stop it is used as stop\n"+
       "                  if balance is 0 (default) always use stop as is.\n"+
       "-held_out <int>   Percent to hold out for pruning\n"+
       "-noprune          No (same class) pruning required\n",
		       files, al);

    if (al.present("-v"))
    {
	printf("%s: %s\n",argv[0],wagon_version);
	exit(0);
    }

    if (al.present("-subset"))
	wgn_csubset = TRUE;
    if (al.present("-held_out"))
	wgn_held_out = al.ival("-held_out");
    if (al.present("-balance"))
	wgn_balance = al.fval("-balance");
    if ((!al.present("-desc")) || ((!al.present("-data"))))
    {
	cerr << argv[0] << ": missing description and/or datafile" << endl;
	cerr << "use -h for description of arguments" << endl;
    }

    if (al.present("-quiet"))
	wgn_quiet = TRUE;

    if (al.present("-test"))
    {
	wgn_desc_file = al.val("-desc");
	wgn_test_file = al.val("-test");
    }
    if (al.present("-stop"))
	wgn_min_cluster_size = atoi(al.val("-stop"));
    if (al.present("-vfold"))
	wgn_vfold = atoi(al.val("-vfold"));
    if (al.present("-noprune"))
	wgn_prune = FALSE;
    if (al.present("-frs"))   // number of partitions to try in floats
	wgn_float_range_split = atof(al.val("-frs"));
    if (al.present("-output"))
    {
	wgn_coutput = new ofstream(al.val("-output"));
	if (!(*wgn_coutput))
	{
	    cerr << "Wagon: can't open file \"" << al.val("-output") <<
		"\" for output " << endl;
	    exit(-1);
	}
    }
    else
	wgn_coutput = &cout;
    if (al.present("-distmatrix"))
    {
	if (wgn_DistMatrix.binload(al.val("-distmatrix")) != 0)
	{
	    cerr << "Wagon: failed to load Distance Matrix from \"" <<
		al.val("-distmatrix") << "\"\n" << endl;
	    exit(-1);
	}
    }
    if (al.present("-dlist"))
	wagon_type = wn_decision_list;

    // Load in the data
    wgn_load_datadescription(al.val("-desc"));
    wgn_load_dataset(al.val("-data"));

    // Build and test the model 
    if (wagon_type == wn_decision_tree)
	wgn_build_tree();  // default operation
    else if (wagon_type == wn_decision_list)
	wgn_build_dlist();
    else
    {
	cerr << "Wagon: unknown operation, not tree or list" << endl;
	exit(-1);
    }

    if (wgn_coutput != &cout)
	delete wgn_coutput;
    return 0;
}
