 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Tue Apr 22 1997                              */
 /************************************************************************/
 /*                                                                      */
 /* Perform LPC analysis on a waveform, producing an analysis track      */
 /* file and optionally a residual.                                      */
 /*                                                                      */
 /************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include "EST.h"
#include "EST_sigpr.h"
#include "EST_Wave.h"
#include "EST_wave_aux.h"
#include "EST_Track.h"
#include "EST_error.h"

#define DEFAULT_FRAME_SIZE 0.01
#define DEFAULT_ORDER 20
#define DEFAULT_WINDOW "hamming"
#define DEFAULT_TIME_SCALE 0.001
#define DEFAULT_TIME_CHANNEL "track0"
#define DEFAULT_PM_AT 0.0

void pm_adjust(EST_Track &pms, EST_Wave &wave);
void pm_trans(EST_Track &pms, EST_Wave &wave, int region);
void pm_min(EST_Track &pms, EST_Wave &wave, int region);
void pm_shift(EST_Track &pms, float pm_at, EST_Wave &wave);

int main(int argc, char *argv[]) 
{ 
    EST_StrList files;
    EST_Option settings, cmd_line;

    parse_command_line2(argc, argv, 
       EST_String("Usage:   ")+
       "lpc_analysis  <options>  ... \n"+
       "Summary; perform LPC analysis\n"+
       "-h               options help\n"+
       "-lpc <string>    dunno\n"+
       "-r <ofile>       Residual file\n"+
       "-rtype <string>  Residual file type\n"+
       "-shift <float>   Frame shift in seconds\n"+
       "-length <float>  Frame length in seconds\n"+
       "-o <ofile>       Ouptut file of LPC coefficients\n"+
       "-otype <string>  Output file type\n"+
       "-order <int>     Order of coeffients\n"+
       "-window <string> Type of window rectangular/hanning/hamming\n"+
       "-reflection      Output reflection coefficients\n"+
       "-pm <ifile>      Pitch mark file name (ascii)\n"+
       "-time_channel <string>\n"+
       "		 Which track in pm file holds pitchmark times\n"+
       "-time_scale <float>    \n"+
       "		 Scale of pitchmarks (default 0.001 = milliseconds)\n"+
       "-pm_adjust_to_peak\n"+
       "                 Move pitchmark to closest peak in waveform\n"+
       "-pm_at <int>     Pitchmark this many samples into frame\n"+
       "-pm_at_perc <int>\n"+
       "                 Pitchmark this percentage of way into frame\n"+
       "-pm_at_trans <int>\n"+
       "                 Move pitchmark back to nearest -ve to +ve transition\n"+
       "-pm_at_min <int> Move pitchmark to min abs value within this many samples\n"+
       "-labels <string> File to save channel names to\n"+
       "-no_length       Don't include length channel\n"+
       "-power           Include rms power channel\n"+
       "-voiced          Include voicing channel\n"+
       "-time            Include time channel\n",
		       files, cmd_line);

    init_lib_ops(cmd_line, settings);

    if (files.length() > 2 )
    {
	cerr << argv[0] << ": too many input files" << endl;
	cerr << "Type " << argv[0] << " -h  for help " << endl;
	exit(1);
    }

    EST_String out_file (cmd_line.present("-o") ? 
			 cmd_line.val("-o") : (EST_String)"-");
    EST_String res_file (cmd_line.present("-r") ? 
			 cmd_line.val("-r") : (EST_String)"");
    EST_String lab_file (cmd_line.present("-labels") ? 
			 cmd_line.val("-labels") : (EST_String)"");
    EST_String pm_file (cmd_line.present("-pm") ? 
			cmd_line.val("-pm") : (EST_String)"");
    EST_String lpcs_file (cmd_line.present("-lpcs") ? 
			  cmd_line.val("-lpcs") : (EST_String)"");
  
    EST_Wave signal;

    if (read_wave(signal, files.first(), settings))
	exit(1);

    EST_Track lpcs;
    EST_Track pms;

    // create the track description
    
    EST_TrackMap lpc_mapping;
    int ch = 0;
    
    if (settings.present("length"))
	lpc_mapping.set(channel_length, ch++);
    if (settings.present("power"))
	lpc_mapping.set(channel_power, ch++);
    if (settings.present("voiced"))
	lpc_mapping.set(channel_voiced, ch++);
    if (settings.present("time"))
	lpc_mapping.set(channel_time, ch++);
    
    lpc_mapping.set(channel_coef0, ch++);
    lpc_mapping.set(channel_coefN, ch + settings.ival("order", 1)-1);
    
    EST_WindowType window =  
	EST_WindowTypeMap.token(settings.sval("window_type", 1));
    
    if (window == EST_WindowTypeMap.unknown_enum())
      EST_error("unknown window type '%s'", 
	      (const char *)settings.sval("window_type", 1));
    
    cerr << 
	settings.fval("frame_length", 1) << ", " << 
	    settings.fval("frame_shift", 1) << ", " << 
		EST_WindowTypeMap.name(window) << ", " << 
		    settings.ival("order", 1) << "\n";
    
    if (pm_file != "")
    {
	if (read_track(pms, pm_file, settings))
	    exit(1);
	
	if (settings.present("pm_adjust_to_peak"))
	    pm_adjust(pms, signal);
	
	if (settings.present("pm_at"))
	    pm_shift(pms, settings.fval("pm_at"), signal);
	if (settings.present("pm_adjust_to_trans"))
	    pm_trans(pms, signal, settings.ival("pm_adjust_to_trans"));      
	if (settings.present("pm_adjust_to_min"))
	    pm_min(pms, signal, settings.ival("pm_adjust_to_min"));
    }
    
    if (lpcs_file != "")
    {
    }
    else
    {
	cerr <<
	    "Computing LPCs\n";
	
	if (pm_file == "")
	    lpc_analyse(lpcs, signal, window,
			settings.fval("frame_length", 1), 
			settings.fval("frame_shift", 1),
			&lpc_mapping);
	else
	    lpc_analyse(lpcs, signal, window,
			settings.fval("frame_length", 1), 
			settings.fval("frame_shift", 1), 
			&lpc_mapping,
			pms);
	
	cerr <<
	    lpcs.num_channels() << ", " << 
		lpcs.num_frames() << "\n";

	if (cmd_line.present("-reflection"))
	{
	    EST_Track rlpcs(lpcs);  // copy lpcs
	    lpc_to_reflection(rlpcs);
	    rlpcs.save(out_file, cmd_line.val("-otype", 0));
	}
	else
	    lpcs.save(out_file, cmd_line.val("-otype", 0));
	
	if (lab_file != "")
	    if (lpcs.save_channel_names(lab_file) != write_ok)
	      EST_error("error writing labels");
    }
    
    if (res_file != "")
    {
	EST_Wave residual;
	
	cerr <<
	    "Computing residual\n";
	
	lpc_residual(residual, lpcs, signal); 
	residual.save(res_file, cmd_line.val("-rtype", 0));
    }
    
    return 0;
}

void override_lib_ops(EST_Option &settings, EST_Option &cmd_line)
{
    settings.override_fval("frame_length", DEFAULT_FRAME_SIZE);
    settings.override_fval("frame_shift", DEFAULT_FRAME_SIZE);
    settings.override_ival("order", DEFAULT_ORDER);
    settings.override_val("window_type", DEFAULT_WINDOW);
    settings.override_fval("time_scale", DEFAULT_TIME_SCALE);
    settings.override_val("time_channel", DEFAULT_TIME_CHANNEL);
    
    if (cmd_line.present("-length"))
	settings.override_fval("frame_length", cmd_line.fval("-length", 1));
    if (cmd_line.present("-shift"))
	settings.override_fval("frame_shift", cmd_line.fval("-shift", 1));
    if (cmd_line.present("-pm_at_perc"))
	settings.override_fval("pm_at", cmd_line.ival("-pm_at_perc", 1)/100.0);
    if (cmd_line.present("-pm_at"))
	settings.override_fval("pm_at", -cmd_line.ival("-pm_at", 1));
    if (cmd_line.present("-order"))
	settings.override_ival("order", cmd_line.ival("-order", 1));
    if (cmd_line.present("-window"))
	settings.override_val("window_type", cmd_line.sval("-window", 1));
    if (cmd_line.present("-time_scale"))
	settings.override_fval("time_scale", cmd_line.fval("-time_scale", 1));
    if (cmd_line.present("-time_channel"))
	settings.override_val("time_channel", cmd_line.sval("time_channel", 1));
    if (cmd_line.present("-pm_adjust_to_peak"))
	settings.override_val("pm_adjust_to_peak", "yes");
    if (cmd_line.present("-pm_at_trans"))
	settings.override_ival("pm_adjust_to_trans", cmd_line.ival("-pm_at_trans", 1));
    if (cmd_line.present("-pm_at_min"))
	settings.override_ival("pm_adjust_to_min", cmd_line.ival("-pm_at_min", 1));
    if (cmd_line.present("-power"))
	settings.override_val("power", "yes");
    if  (cmd_line.present("-voiced"))
	settings.override_val("voiced", "yes");
    if  (cmd_line.present("-time"))
	settings.override_val("time", "yes");
    if  (!cmd_line.present("-no_length"))
	settings.override_val("length", "yes");
}

int is_transition(int t, EST_Wave &wave)
{
    return wave.a(t-1) < 0 && wave.a(t) >=0;
}

int is_peak(int t, EST_Wave &wave)
{
    return wave.a(t-1) <=  wave.a(t)&& wave.a(t) >= wave.a(t+1);
}

void pm_adjust(EST_Track &pms, EST_Wave &wave)
{
    
    for(int i=0; i<pms.num_frames(); i++)
    {
	int t = (int)(pms.t(i) * wave.sample_rate());
	int pos=t, val= wave.a(t);
	
	for(int j=1; j<20; j++)
	{
	    if (t-j >=0 && is_peak(t-j, wave))
	    {
		if (wave.a(t-j) > val)
		{
		    val = wave.a(t-j);
		    pos=t-j;
		}
	    }
	    if (t+j < wave.length() && is_peak(t+j, wave))
	    {
		if (wave.a(t+j) > val)
		{
		    val = wave.a(t+j);
		    pos=t+j;
		}
	    }
	}
	pms.t(i) = (pos+0.0)/wave.sample_rate();
	// cout << "adjust " << t << " to " << pos << " = " << pos-t << "\n";
    }
}

void pm_trans(EST_Track &pms, EST_Wave &wave, int region)
{
    
    for(int i=0; i<pms.num_frames(); i++)
    {
	int t = (int)(pms.t(i) * wave.sample_rate());
	int pos=t;
	for(int j=0; j<region; j++)
	{
	    if (t-j >=0 && is_transition(t-j, wave))
	    {
		pos = t-j;
		break;
	    }
	    /*
	       if (t+j <  wave.length() && is_transition(t+j, wave))
	       {
	       pos = t+j;
	       break;
	       }
	       */
	}
	pms.t(i) = (pos+0.0)/wave.sample_rate();
	// cout << "adjust " << t << " to " << pos << " = " << pos-t << "\n";
    }
}

void pm_min(EST_Track &pms, EST_Wave &wave, int region)
{
    
    for(int i=0; i<pms.num_frames(); i++)
    {
	int t = (int)(pms.t(i) * wave.sample_rate());
	int pos=t, val=abs(wave.a(t));
	for(int j=-region/2; j<region/2; j++)
	{
	    if (t+j >=0 && t+j < wave.length() && abs(wave.a(t+j)) < val)
	    {
		pos = t+j;
		val = wave.a(t+j);
	    }
	}
	pms.t(i) = (pos+0.0)/wave.sample_rate();
	// cout << "adjust " << t << " to " << pos << " = " << pos-t << "\n";
    }
}

void pm_shift(EST_Track &pms, float pm_at, EST_Wave &wave)
{
    for(int i=0; i<pms.num_frames(); i++)
    {
	int t = (int)(pms.t(i) * wave.sample_rate());
	int t1 = (i < pms.num_frames()-1)?(int)(pms.t(i+1) * wave.sample_rate()):wave.length();
	
	int pos;
	
	if (pm_at > 0)
	    pos = (int)(t + (t1-t)*(1.0-pm_at) + 0.5);
	else 
	    pos = (int)(t + pm_at + 0.5);
	
	pms.t(i) = (pos+0.0)/wave.sample_rate();
    }
}
