 /*************************************************************************/
 /*                                                                       */
 /*                Centre for Speech Technology Research                  */
 /*                     University of Edinburgh, UK                       */
 /*                      Copyright (c) 1995,1996                          */
 /*                        All Rights Reserved.                           */
 /*  Permission to use, copy, modify, distribute this software and its    */
 /*  documentation for research, educational and individual use only, is  */
 /*  hereby granted without fee, subject to the following conditions:     */
 /*   1. The code must retain the above copyright notice, this list of    */
 /*      conditions and the following disclaimer.                         */
 /*   2. Any modifications must be clearly marked as such.                */
 /*   3. Original authors' names are not deleted.                         */
 /*  This software may not be used for commercial purposes without        */
 /*  specific prior written permission from the authors.                  */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
 /*  THIS SOFTWARE.                                                       */
 /*                                                                       */
 /*************************************************************************/
 /*                                                                       */
 /*                       Author :  Paul Taylor                           */
 /*                       Date   :  April 1995                            */
 /* --------------------------------------------------------------------- */
 /*                     Double linked list class                          */
 /*                                                                       */
 /* Modified by RJC, 21/7/97. Now much of the working code is in the      */
 /* UList class, this template class provides a type safe front end to    */
 /* the untyped list.                                                     */
 /*                                                                       */
 /*************************************************************************/

#ifndef __Tlist_H__
#define __Tlist_H__

#include <iostream.h>
#include "EST_common.h"
#include "EST_String.h"
#include "EST_UList.h"
#include "EST_TSortable.h"

template<class T> class EST_TItem : public EST_UItem {
public:
    T val;

    EST_TItem(const T &v) 
	    { init(); val=v; };
    EST_TItem() 
	    { init();};
};

// Legacy name

typedef EST_UItem EST_TBI;

// pretty name

typedef EST_UItem EST_Litem;

template <class T> class EST_TList : public EST_UList {

public:
  static void free_item(EST_UItem *item) 
    { delete (EST_TItem<T> *)item; };

    void init() { EST_UList::init(); };

    EST_TList() { };
    EST_TList(EST_TList &l);
    ~ EST_TList() { clear_and_free(free_item); }

    T &item(EST_Litem *p)     // return the value associated with the pointer
	    { return ((EST_TItem<T> *)p) -> val; };
    // Some C++ compilers don't allow overload on const alone so
    // we introduce _C functions for the const form
    const T &item_C(const EST_Litem *p) const	
	    { return ((EST_TItem<T> *)p) -> val; };
    T &ritem(EST_Litem *p) 	
	    { return item(p); };

    T &item(int n)	   // return the Nth value
	    { return item(nth_pointer(n)); };
    const T &item(int n) const
	    { return item_C(nth_pointer(n)); };
    T &ritem(int n) 	
	    { return item(n); };

    const T &first() const			// return first item in list
            { return item_C(head()); };
    const T &last() const			//  return last item in list
	    { return item_C(tail()); };
    T &first()					// return first item in list
            { return item(head()); };
    T &last()					//  return last item in list
	    { return item(tail()); };

    const T  &operator () (EST_Litem *ptr) const	// access operator
	    { return item_C(ptr); };
    T  &operator () (EST_Litem *ptr)		// access operator
 	    { return item(ptr); };

    // remove single item, return pointer to previous
    EST_Litem *remove(EST_Litem *ptr)
	    { return EST_UList::remove(ptr, free_item); };
    EST_Litem *remove(int n)
	    { return EST_UList::remove(n, free_item); };
    EST_Litem *remove_nth(int n)
	    { return EST_UList::remove(n, free_item); };

    // returns pointer to inserted item
    EST_Litem *insert_after(EST_Litem *ptr, const T &item)
	    { return EST_UList::insert_after(ptr, new EST_TItem<T>(item)); };
    // returns pointer to item after inserted item
    EST_Litem *insert_before(EST_Litem *ptr, const T &item)
	    { return EST_UList::insert_before(ptr, new EST_TItem<T>(item)); };

    void exchange(EST_Litem *a, EST_Litem *b)
	    { EST_UList::exchange(a, b); };
    void exchange(int i, int j)
	    { EST_UList::exchange(i,j); };
    
   static void exchange_contents(EST_Litem *a, EST_Litem *b);
	
    void clear(void) 
	    { clear_and_free(free_item); };
    void append(const T &item)		// add item onto end of list
	    { EST_UList::append(new EST_TItem<T>(item)); };

    EST_TList<T> &operator=(const EST_TList<T> &a); // full copy of list
//    T  &operator () (EST_TItem<T> *ptr); // full copy of list


    EST_TList<T> &operator +=(const EST_TList<T> &a);

    bool operator >(const EST_TList<T> &a);

    friend ostream& operator << (ostream &st, const EST_TList<T> &list);
};

// functions in EST_tlist_aux.C

template<class T> 
ostream& operator << (ostream &st, const EST_TList<T> &list)
{   
    EST_Litem *ptr;
    for (ptr = list.head(); ptr != 0; ptr = next(ptr))
	st << list.item_C(ptr) << " ";
    return st;
}

template<class T> 
bool operator==(const EST_TList<T> &a, const EST_TList<T> &b)
{ 
    return EST_UList::operator_eq(a, b, EST_TSortable<T>::items_eq); 
}

template<class T> 
void sort(EST_TList<T> &a, bool (*gt)(const EST_UItem *, const EST_UItem *) = NULL)
{ 
    EST_UList::sort(a, gt?gt:EST_TSortable<T>::items_gt); 
}

template<class T> 
void ptr_sort(EST_TList<T> &a)
{ 
    return EST_UList::sort(a, EST_TSortable<T *>::items_gt); 
}

template<class T> 
void qsort(EST_TList<T> &a, bool (*gt)(const EST_UItem *, const EST_UItem *) = NULL)
{ 
    EST_UList::qsort(a, gt?gt:EST_TSortable<T>::items_gt, EST_TList<T>::exchange_contents); 
}

template<class T> 
void ptr_qsort(EST_TList<T> &a)
{ 
    EST_UList::qsort(a, EST_TSortable<T *>::items_gt, EST_TList<T>::exchange_contents); 
}

template<class T> 
void sort_unique(EST_TList<T> &l)
{ 
    EST_UList::sort_unique(l, 
			   EST_TSortable<T>::items_eq, 
			   EST_TSortable<T>::items_gt,
			   EST_TList<T>::free_item); 
}

template<class T> 
void merge_sort_unique(EST_TList<T> &l, EST_TList<T> &m)
{  
    EST_UList::merge_sort_unique(l, m,
				 EST_TSortable<T>::items_eq, 
				 EST_TSortable<T>::items_gt,
				 EST_TList<T>::free_item); 
}

#endif
