#!/bin/bash
#
# bug @VER@: Generates a bug report (needs bash >=2.0)
#
# Christoph Lameter, October 26, 1996
#
# Modified by Nicols Lichtmaier
#
set -e

if [ "$BASH_VERSION" \< "2.01.1" ] ; then
	echo "$0: Needs Bash >= 2.01.1" >&2
	exit 1
fi

export TEXTDOMAIN=bug
YESNO=$"yYnN"

if [ -z "$EMAIL" ] ; then
	if  [ ! -r /etc/mailname ] ; then
		echo $"$0: Please set the environment variable EMAIL to contain your fully qualified e-mail address and try again." >&2
		exit 1
	fi
	CC="`id -un`"
	CC=`sed -ne "s/^$CC:[^:]*:[^:]*:[^:]*:\\([^,:]*\\),*:.*/\\1/p" \
		< /etc/passwd`" <$CC@`cat /etc/mailname`>"
else
	CC="$EMAIL"
fi

FROM="$CC"
if [ "$(type -t realpath 2> /dev/null)" = "" ]; then REALPATH=realp; else REALPATH=realpath ; fi

realp()
{
	fname="$1"
	while [ -L "$fname" ]; do
		fname="$(command ls -l $fname)"
		fname="${fname#*\> }"
	done
	echo $fname
}

sendhelp()
{
echo "bug v@VER@"
echo $"Usage: $1 [options] <packagename> [<text>]
A script to ease the reporting of bugs.

Options:
    -c        Do  not  include  configuration  files  in  the bug
              report.
    -d        Debug.  Dont send the mail to the bug tracking sys
              tem. Instead sent to postmaster@localhost.
    -f        <packagename> is a file belonging to the package
              and bug will search for it.
    -H special_header
              Adds  a  custom  header  to  generated mail. Can be
              repeated to add multiple custom headers.
    -m        Submit a \"maintonly\" bug report. Only sent  to  the
              maintainer not to the mailing list.
    -p        print. Do not send the bug report instead output it
              on stdout so that it can be redirected into a file.
    -q        Submit a quiet bug report (Just register. No e-mail
              forwarding to anyone)
    -s text   Set the Subject of the bug  report.  Otherwise  bug
              will ask for a short description of the problem.
    -x        Do not send a CC to the submitter of the bugreport
    -z        Do not compress the configfiles by  removing  empty
              lines and comments.
    -h        Shows this text.

Please don't report several unrelated bugs - especially ones in
different packages - in one message."
	exit 0
}

if [ "$1" = "--help" ]; then sendhelp ; fi

while getopts "cdfhH:mpqs:vzx-" OPT; do
	case $OPT in
		c)	NOCONFIG=1;;
		d)	DEBUG=1;;
		f)	SEARCH=1;;
		h)	sendhelp $0;;
		H)	CUSTOM_HEADER[${#CUSTOM_HEADER[*]}]=$OPTARG;;
		m)	MAINT=1;;
		p)	PRINT=1;;
		q)	QUIET=1;;
		s)	SUBJECT="$OPTARG";;
		v)	echo "bug @VER@" >&2
			exit 1;;
		x)	CC="";;
		z)	NOZAP=1;;
	        -)	break;;
	  	*)	echo $"Usage: $0 [options] <packagename> [<text>] .." >&2
		    	exit 1;;
        esac
done
shift $(($OPTIND - 1))

if [ "$UID" = "0" ]; then
	echo $"$0: Filing bug reports with superuser priviledges is insecure and not supported." >&2
	exit 1
fi

PACKAGE="$1"

if [ "$SEARCH" ]; then
	if [ -z "$PACKAGE" ]; then
		echo $"$0: Must specify file name when using -f." 2>&2
		exit 1
	fi
	FILE=$($REALPATH "$PACKAGE")
	test "$PRINT" || echo -n $"Searching for package containing ${FILE}..."
	if [ "$FILE" = "" ]; then
		echo
		echo $"$0: Must include parameter when using -f." 2>&2
		exit 1
	fi
	if [ ! -e "$FILE" ]; then
		echo
		echo $"$0: File \`$FILE' not found." 2>&2
		exit 1	
	fi
	PACKAGE=$(grep -l ^${FILE}\$ /var/lib/dpkg/info/*.list | head -1)
	if [ -z "$PACKAGE" ]; then
		echo
		echo $"$0: File \`$FILE' can't be found in any installed package." >&2
		exit 1
	fi
	PACKAGE=${PACKAGE##*/}
	PACKAGE=${PACKAGE%%.list}
	test "$PRINT" || echo $"done."
	if [ -z "$PRINT" ]; then
		echo $"\`$FILE' found in package \`$PACKAGE'"
		read -erp $"Submit bug for this package? [Y|n] "
		if [ "${REPLY:0:1}" = "${YESNO:2:1}" -o "${REPLY:0:1}" = "${YESNO:3:1}" ]; then
			exit 0
		fi
	fi
fi

if [ "$PACKAGE" = "" ]; then
	echo $"Please state in which package you have found a problem, or...
type one of these bug categories:

base             General bugs in the base system
bootdisk         Bugs in the bootdisk
rootdisk         Bugs in the rootdisk
bugs.debian.org  The bug tracking system, @bugs.debian.org
ftp.debian.org   Problems with the FTP site
nonus.debian.org Problems with the non-us FTP site
www.debian.org   Problems with the WWW site
manual           Bugs in the manual
project          Problems related to Project administration
general          General problems (e.g., that many manpages are mode 755)
kernel           Problems with the Linux kernel, or that shipped with Debian
lists.debian.org The mailing lists debian-*@lists.debian.org."
	echo
	read -p $"package? " -er PACKAGE
	if [ "$PACKAGE" = "" ]; then
		echo $"Canceled!"
		exit 0
	fi
fi

i=$((${#PACKAGE}-1))
while [ $i -gt 0 -o $i -eq 0 ]; do
	case ${PACKAGE:$i:1} in
		" ")
			echo $"$0: Package name goes first, bug report last." >&2
			exit 1
			;;
		[A-Z])
			echo $"$0: Packages names are formed with lower case letters." >&2
			exit 1
			;;
		[^-a-z0-9.+])
			echo $"$0: Invalid package name." >&2
			exit 1
			;;
	esac
	i=$(($i - 1))
done

shift || true

if [ "$EDITOR" = "" ]; then
	EDITOR="$VISUAL"
	if [ "$EDITOR" = "" ]; then
		if [ -x /usr/bin/editor ]; then
			EDITOR="$($REALPATH /usr/bin/editor)"
		else
			if [ ! -x /bin/ae ]; then
				echo $"$0: Cannot locate a texteditor." >&2
				exit 1
			fi
			EDITOR="ae"
		fi
	fi
fi

checkconf()
{ 
# Takes two parameters filename and md5sum and checks if the file was modified
# If so outputs the filename
	if [ "$1" = "/etc/passwd" ]; then exit; fi
	if [ -r "$1" ]; then
		SUM=`md5sum $1`
		if [ "${SUM:0:32}" != "$2" ]; then
			echo "$1"
		fi
	else
		echo $1
	fi
}

checkconfs()
{
	read X
	while [ "$X" ]; do
		checkconf $X
		if ! read X ; then
			return
		fi
	done
}

pinfo()
{
	sed -n </var/lib/dpkg/status -e "/^Package: $1$/,/^Description: /p"
}

setsubject()
{
	if [ "$SUBJECT" = "" ]; then
		while [ "$SUBJECT" = "" ]; do
#			dialog --backtitle "Bug Reporting Tool - Debian/GNU Linux" \
#				--inputbox "Please describe your problems using $PACKAGE" \
#				8 78 2> /tmp/bugtmp...
#			SUBJECT=`cat /tmp/bugtmp...`
			echo $"Please describe your problems using $PACKAGE in one line."
			read -er SUBJECT ;
		done
		SUBJECT="$PACKAGE: $SUBJECT"
	fi
}

template()
{
 	cat <<-EOF
		Subject: $SUBJECT
		Package: $PACKAGE
		Version: $VERSION

		$*

		-- System Information
		Debian Release: `cat /etc/debian_version`
		Kernel Version: `uname -a`

	EOF
}

trap 'echo $"$0 interrupted." ; rm -f $TEMPS ; exit 0' SIGINT

TEMP1=$(tempfile -pbug)
TEMPS=$TEMP1
if [ -f /var/lib/dpkg/info/$PACKAGE.list ]; then
	test "$PRINT" || echo -n $"Getting package's info..."
	pinfo $PACKAGE >$TEMP1
	VERSION=`grep ^Version: $TEMP1`
	VERSION=${VERSION#Version: }
	DEPS=`grep Depends: $TEMP1` || DEPS=""
	if [ "$NOCONFIG" = "" ]; then
		if grep -q "Conffiles:" $TEMP1; then
			MODCONF=`sed -n <$TEMP1  -e '/Conffiles:/,/^[^ ].*\|^$/p'| egrep -v :\|^$ | checkconfs`
		fi
	fi
	test "$PRINT" || echo $"done."
	setsubject
	template "$*" >$TEMP1

	if [ "$DEPS" != "" ]; then
		test "$PRINT" || echo -n $"Checking ${PACKAGE}'s dependencies..."
		echo "Versions of the packages $PACKAGE depends on:" >>$TEMP1
		DEPS=`echo "$DEPS"|tr "|,\n" "  "|sed -e "s/ ([^)]*)//g" -e "s/Pre-Depends://g" -e "s/Depends://g" -e "s/  / /g" `
		pkgs=""
		vpkgs=""
		for i in $DEPS; do
			if [ -f /var/lib/dpkg/info/$i.list ]; then
				pkgs="$pkgs $i"
			else
				vpkgs="$vpkgs $i"
			fi
		done
		dpkg -l $pkgs | tail +6 >>$TEMP1
		unset pkgs
		for i in $vpkgs; do
			X=${i//+/\\\\+}
			TEMP2=$(tempfile -pbug)
			TEMPS="$TEMPS $TEMP2"
			perl -n000e "print if /^Provides:.* $X/mo && !/^Status: +[^ ]+ +[^ ]+ +config-files/mo" /var/lib/dpkg/status >$TEMP2
			if [ -s $TEMP2 ]; then
				REALPACKAGE=`head -1 $TEMP2`
				REALPACKAGE=${REALPACKAGE#Package: }
#				echo "$REALPACKAGE	$(grep '^Version:' $TEMP2) (Provides virtual package $i)" >>$TEMP1
				dpkg -l $REALPACKAGE | tail +6 >>$TEMP1
				echo "	^^^ (Provides virtual package $i)" >>$TEMP1
			else
				echo "$i	Not installed or no info" >>$TEMP1
			fi
			rm $TEMP2
		done
		unset vpkgs
		test "$PRINT" || echo $"done."
	fi

	if [ "$MODCONF" != "" ]; then
		test "$PRINT" || echo -n $"Including modified configuration files..."
		for i in $MODCONF; do
			if ls -lL "$i" | grep -qv ^.......r ; then
				continue
			fi
			echo -e "\n--- Begin $i (modified conffile)" >>$TEMP1
			if [ -r $i ]; then
				if file -L $i | grep -q "text"; then
					if [ "$NOZAP" ]; then
						cp $i $TEMP1
					else
						sed -e '/^#/d' -e '/^$/d' <$i >>$TEMP1
					fi
				else
					echo "*** Contents not text ***" >>$TEMP1
				fi
			else
				echo "Config file not present or no permissions for access" >>$TEMP1
			fi
			echo -e "\n--- End $i" >>$TEMP1
		done
		test "$PRINT" || echo $"done."
	fi
else
	VIRT=nop
	for i in base bootdisk rootdisk bugs.debian.org ftp.debian.org www.debian.org manual project general kernel lists.debian.org ; do
		if [ "$PACKAGE" = "$i" ]; then
			PSEUDO=yep
		fi
	done
	if [ "$PSEUDO" != "yep" ] && ! grep -q "^Package: $PACKAGE\>" /var/lib/dpkg/available; then
		echo $"Package $PACKAGE doesn't seem to exist!"
		echo -n $"Submit bug anyway? [y|N] "
		read
		if [ ! "${REPLY:0:1}" = "${YESNO:0:1}" -a ! "${REPLY:0:1}" = "${YESNO:1:1}" ]; then
			echo $"Bug report not sent"
			exit 1
		fi
	fi
	VERSION="N/A"
	setsubject
	template "$*" >$TEMP1
fi

if [ -z "$1" -a -z "$PRINT" ]; then
	X=""
	while [ "$X" = "" ]; do
		TEMPBACKUP=$(tempfile -pbug)
		TEMPS="$TEMPS $TEMPBACKUP"
		cp $TEMP1 $TEMPBACKUP
		EDNAME="${EDITOR%% *}"
		EDNAME="${EDNAME##*/}"
		if [ "$EDNAME" = "joe" ]; then
			IFJOE="-skiptop 1"
			echo $"Reporting bug for package: $PACKAGE $VERSION"
			echo -ne \\033[2J\\033[1\;1H
		fi
		if [ "$EDNAME" = "joe" -o \
			"$EDNAME" = "jed" -o \
			"$EDNAME" = "emacs" -o \
			"$EDNAME" = "vi" ]; then
			EDITARGS="+5"
		fi
		$EDITOR $IFJOE $EDITARGS $TEMP1 ||
		{
			echo $"Editor Aborted: No Bug report sent"
			rm $TEMP1 $TEMPBACKUP
			exit 1
		}

		if cmp $TEMP1 $TEMPBACKUP >/dev/null; then
			echo $"No changes: No Bug report sent."
			rm $TEMP1 $TEMPBACKUP
			exit 1;
		fi
		echo -n $"Do you really want to submit this bug report? [Y|n] "
		stty -icanon
		KEY=$(dd bs=1 count=1 2> /dev/null)
		stty icanon
		echo
		if [ "${KEY:0:1}" = "${YESNO:2:1}" -o "${KEY:0:1}" = "${YESNO:3:1}" ]; then
			exit 0
		fi
		rm $TEMPBACKUP
		X="`head -1 $TEMP1`"
		X=${X#Subject: }
	done
	SUBJECT=$X
else
	if [ "$SUBJECT" = "" ]; then
		SUBJECT="$PACKAGE $VERSION"
	fi
fi

if [ "$PRINT" ]; then
	cat $TEMP1
	rm $TEMP1
	exit 0
fi

MADDR="submit@bugs.debian.org"

if [ "$QUIET" ]; then
	MADDR="quiet@bugs.debian.org"
fi

if [ "$MAINT" ]; then
	MADDR="maintonly@bugs.debian.org"
fi

if [ "$DEBUG" ]; then
	MADDR="postmaster@localhost"
fi

{
	echo "From: $FROM"
	echo "Subject: $SUBJECT"
	echo "To: $MADDR"
	test "$CC" && echo "Bcc: $CC"
	echo "X-Mailer: bug @VER@"
	test "$REPLYTO" && echo "Reply-To: $REPLYTO"
	i=0
	while [ $i -lt ${#CUSTOM_HEADER[*]} ] ; do
		echo ${CUSTOM_HEADER[$i]}
		i=$((i+1))
	done
	echo
	sed <$TEMP1 -e "1d"
} |
{
	{
		test -x /usr/sbin/sendmail && /usr/sbin/sendmail -t
	} ||
	{ 
		tee /var/tmp/${PACKAGE}.bug | grep -q zazazaz
		echo $"Bug report saved in /var/tmp/${PACKAGE}.bug"
	}
}

rm $TEMP1
echo $"Bug Report submitted to $MADDR $CC

If you want to submit further information about this bug please wait
to receive the bug number by e-mail. Then mail any extra information to
nnnn@bugs.debian.org, where nnnn is the bug number."
