/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for the MIDI interface - GF1 MIDI v1.0 emulation
 */

#include "driver.h"
#include "midi.h"

#ifdef GUSCFG_MIDI_DEVICES

#if 0
#define DEBUG_SYSEX
#endif

int gus_gf1_midi_engine_reset( gus_card_t *card )
{
  return gus_engine_reset( card, card -> gf1.midi_mix_voices, 0xffffffff );
}

static int gus_gf1_midi_init_write( gus_card_t *card )
{
  int result;

  if ( ( result = gus_engine_open( card ) ) < 0 )
    return result;
  if ( ( result = gus_gf1_midi_engine_reset( card ) ) < 0 )
    {
      gus_engine_close( card );
      return result;
    }
  card -> midi[ GUS_MIDID_SYNTH ].tx_prev_cmd = 0x00;
  return 0;
}

static int gus_gf1_midi_init_read( gus_card_t *card )
{
  return -EIO;		/* always fail */
}

static void gus_gf1_midi_done_write( gus_card_t *card )
{
  gus_engine_close( card );
}

static void gus_gf1_midi_done_read( gus_card_t *card )
{
  /* nothing */
}

/* --- */

static void gus_gf1_midi_reset( gus_card_t *card )
{
  int i;
  gus_channel_t *channel;

  for ( i = 0; i < GUS_CHANNELS; i++ )
    {
      channel = card -> gf1.syn_channels + i;
      gus_engine_midi_control( card, channel, GUS_MCTL_ALL_NOTES_OFF, 0 );
      gus_engine_midi_control( card, channel, GUS_MCTL_RESET_CONTROLLERS, 0 );
      gus_engine_midi_control( card, channel, GUS_MCTL_REGIST_PARM_NUM_MSB, 0 );
      gus_engine_midi_control( card, channel, GUS_MCTL_REGIST_PARM_NUM_LSB, 0 );
      gus_engine_midi_control( card, channel, GUS_MCTL_MSB_DATA_ENTRY, 2 );
      gus_engine_midi_control( card, channel, GUS_MCTL_LSB_DATA_ENTRY, 0 );
      gus_engine_midi_control( card, channel, GUS_MCTL_REGIST_PARM_NUM_MSB, 0x7f );
      gus_engine_midi_control( card, channel, GUS_MCTL_REGIST_PARM_NUM_LSB, 0x7f );
      gus_engine_channel_program( card, channel, 0 );
    }
}

static void gus_gf1_midi_sysex( gus_card_t *card, unsigned char *buffer, int count )
{
  unsigned char id_number;
  unsigned char dev_number;
  
  if ( count < 3 ) return;
  id_number = *buffer++;
  dev_number = *buffer++;
  count -= 2;
  if ( dev_number != 0x10 &&		/* default device number by Roland */
       dev_number != 0x7f ) return;	/* broadcast */
  switch ( id_number ) {
    case 0x41:		/* Roland */
      {
        unsigned char model_number;
        unsigned char command_id;
        unsigned int address;
        unsigned char checksum;
        int i;
        
        if ( count < 5 ) return;
        model_number = *buffer++;
        command_id = *buffer++;
        checksum = buffer[ 0 ] + buffer[ 1 ] + buffer[ 2 ];
        address = *buffer++ << 16;
        address |= *buffer++ << 8;
        address |= *buffer++;
        count -= 5;
        for ( i = 0; i < count; i++ )
          checksum += buffer[ i ];
        if ( ( checksum & 0x7f ) != 0 )
          {
            PRINTK( "gus_gf1_midi_sysex: GS SysEx - checksum failed - 0x%x\n", checksum );
            return;
          }
        if ( address == 0x40007f )	/* Mode set register */
          {
            if ( count < 2 ) return;
            if ( *buffer == 0x00 ||	/* GS reset */
                 *buffer == 0x7f )	/* Exit GS mode */
              {
                gus_gf1_midi_reset( card );
                return;
              }
          }
      }
      break;
    case 0x7e:		/* Universal Non-realtime Message */
      {
        unsigned char sub_id_1, sub_id_2;
        
        if ( count < 2 ) return;
        sub_id_1 = *buffer++;
        sub_id_2 = *buffer++;
        count -= 2;
        switch ( sub_id_1 ) {
          case 0x06:	/* ??? */
            if ( sub_id_2 == 0x01 ||	/* Inquiry Request */
                 sub_id_2 == 0x02 )	/* Inquiry Reply */
              return;
            break;
          case 0x09:	/* General MIDI message */
            if ( sub_id_2 == 0x01 ||	/* General MIDI On */
                 sub_id_2 == 0x02 ) 	/* General MIDI Off */
              {
                gus_gf1_midi_reset( card );
                return;
              }
            break;
        }
      }
      break;
    case 0x7f:		/* Universal Realtime Message */
      {
        unsigned char sub_id_1, sub_id_2;
      
        if ( count < 2 ) return;
        sub_id_1 = *buffer++;
        sub_id_2 = *buffer++;
        count -= 2;
        switch ( sub_id_1 ) {
          case 0x04:			/* Device Control messages */
            switch ( sub_id_2 ) {
              case 0x01:		/* Master Volume */
                {
                  int volume;
                  
                  if ( count < 2 ) return;
                  volume = ( (int)( buffer[ 1 ] & 0x7f ) * 100 ) / 127;
                  gus_mixer_update_level( card, MIX_GF1_MASTER, volume, volume, -1, -1 );
                }
                return;
            }
        }
      }
      break;
  }
#ifdef DEBUG_SYSEX
  {
    int count1 = count;
    unsigned char *ptr = buffer;
    
    printk( "UNKNOWN SYSEX: " );
    while ( count1-- > 0 )
      printk( "%02x:", *ptr++ );
    printk( "\n" );
  }
#endif
}

/* --- */

static int gus_gf1_midi_putcmd( gus_card_t *card, unsigned char *buffer, unsigned int count )
{
  unsigned char cmd, channel;
  gus_channel_t *pchannel;
  int i;

#if 0
  printk( "gf1 put cmd called - count = %i\n", count );
#endif
  if ( !(card -> midi[ GUS_MIDID_SYNTH ].flags & GUS_MIDIF_USED_OUT) )
    return -EIO;
  while ( count > 0 )
    {
      if ( *buffer & 0x80 )
        {
          cmd = *buffer++;
          count--;
        }
       else 
      	cmd = card -> midi[ GUS_MIDID_SYNTH ].tx_prev_cmd;
      card -> midi[ GUS_MIDID_SYNTH ].tx_prev_cmd = cmd;
#if 0
      printk( "gf1 put cmd - 0x%x - count = %i\n", cmd, count );
#endif
      if ( !cmd ) return -EINVAL;
      if ( (cmd & GUS_MCMD_COMMON_SYSEX) != GUS_MCMD_COMMON_SYSEX )
        {
          channel = cmd & 0x0f;
          pchannel = &card -> gf1.syn_channels[ channel ];
        }
       else
        {
          channel = 0;
          pchannel = NULL;
        }
      switch ( cmd & 0xf0 ) {
        case GUS_MCMD_NOTE_OFF:		/* note off */
          gus_engine_midi_note_off( card, pchannel, buffer[ 0 ], buffer[ 1 ] );
          buffer += 2;
          count -= 2;
          break;
        case GUS_MCMD_NOTE_ON:		/* note on */
          gus_engine_midi_note_on( card, pchannel, buffer[ 0 ], buffer[ 1 ], 32 );
          buffer += 2;
          count -= 2;
          break;
        case GUS_MCMD_NOTE_PRESSURE:	/* note pressure */
          buffer += 2;
          count -= 2;
          break;
        case GUS_MCMD_CONTROL:		/* seq control */
          gus_engine_midi_control( card, pchannel, buffer[ 0 ], buffer[ 1 ] );
          buffer += 2;
          count -= 2;
          break;
        case GUS_MCMD_PGM_CHANGE:	/* seq set instrument */
          gus_engine_channel_program_drum( card, pchannel, buffer[ 0 ] );
          count--;
          break;
        case GUS_MCMD_CHANNEL_PRESSURE:	/* channel pressure */
          count--;
          break;
        case GUS_MCMD_BENDER:		/* bender */
          gus_engine_midi_pitchbend( card, pchannel, ( buffer[ 0 ] & 0x7f ) | ( buffer[ 1 ] << 7 ) );
          buffer += 2;
          count -= 2;
          break;
        case GUS_MCMD_COMMON_SYSEX:	/* common message */
          switch ( cmd ) {
            case GUS_MCMD_COMMON_SYSEX:	/* sysex */
              for ( i = 0; buffer[ i ] != GUS_MCMD_COMMON_SYSEX_END && count; i++, count-- );
              gus_gf1_midi_sysex( card, buffer, i );
              buffer += i;
              if ( count )
                {
                  buffer++;
                  count--;
                }
              break;
            case GUS_MCMD_COMMON_MTC_QUARTER:
            case GUS_MCMD_COMMON_SONG_SELECT:
              if ( count )
                {
                  buffer++;
                  count--;
                }
              break;
            case GUS_MCMD_COMMON_SONG_POS:
              if ( count > 1 )
                {
                  buffer += 2;
                  count -= 2;
                }
               else
                count = 0;
              break;
          }
      }
    }
  return 0;
}

void gus_init_gf1_midi( gus_card_t *card )
{
  struct GUS_STRU_MIDI *midi;

  midi = &card -> midi[ GUS_MIDID_SYNTH ];
  midi -> flags = 0;
  midi -> init_write = gus_gf1_midi_init_write;
  midi -> init_read = gus_gf1_midi_init_read;
  midi -> done_write = gus_gf1_midi_done_write;
  midi -> done_read = gus_gf1_midi_done_read;
  midi -> putcmd = gus_gf1_midi_putcmd;
  card -> gf1.midi_mix_voices = 32;
}

#endif /* GUSCFG_MIDI_DEVICES */
