/*================================================================
 * parsesf.c
 *	parse SoundFonr layers and convert it to AWE driver patch
 *
 * Copyright (C) 1996,1997 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *================================================================*/

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#ifdef linux
#include <linux/soundcard.h>
#include <linux/awe_voice.h>
#else
#include <machine/soundcard.h>
#include <awe_voice.h>
#endif
#include "awe_parm.h"
#include "itypes.h"
#include "sffile.h"
#include "sflayer.h"
#include "sfitem.h"
#include "aweseq.h"
#include "util.h"
#include "sfopts.h"
#include "dynload.h"


/*----------------------------------------------------------------
 * prototypes
 *----------------------------------------------------------------*/

#define P_GLOBAL	1
#define P_LAYER		2

typedef int (*Loader)(SFInfo *sf, int sample);
typedef int (*Mapper)(SFPatchRec *pat, SFPatchRec *map, LayerTable *tbl);

static int awe_load_patch(void *patch, int len);
static int awe_load_data(SFInfo *sf, FILE *fp, int sample, int *memsize);

static LoadList *is_matched_preset(LoadList *list, SFPatchRec *pat);
static int buffer_map(SFPatchRec *pat, SFPatchRec *map, LayerTable *tbl);
static int buffer_sample(SFInfo *sf, int sample);
static int buffer_load(SFInfo *sf, int sample);
static int mark_samples_in_list(SFInfo *sf);

static int dynamic_map(SFPatchRec *pat, SFPatchRec *map, LayerTable *tbl);
static int dynamic_load(SFInfo *sf, int sample);

static int load_font(SFInfo *sf, SFPresetHdr *preset, SFPatchRec *pat,
		     Mapper map, Loader loader);
static void search_def_drum_inst(SFInfo *sf);

static int parse_layer(SFInfo *sf, SFPatchRec *pat, LayerTable *tbl,
		       Mapper mapper, Loader loader, int level);

static void set_to_table(SFInfo *sf, LayerTable *tbl, SFGenLayer *lay, int level);
static void add_item_to_table(LayerTable *tbl, int oper, int amount, int level);
static int sanity_range(LayerTable *tbl);
static int in_range(LayerTable *tbl, int key);

static void clear_table(LayerTable *tbl);
static void merge_table(SFInfo *sf, LayerTable *dst, LayerTable *src);
static void init_and_merge_table(SFInfo *sf, LayerTable *dst, LayerTable *src);

static int make_patch(SFInfo *sf, SFPatchRec *pat, LayerTable *tbl);
static void make_info(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl);
static void set_sample_info(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl);
static void set_init_info(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl);
static void set_rootkey(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl);
static void set_modenv(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl);
static void set_volenv(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl);
static void set_lfo1(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl);
static void set_lfo2(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl);

static int find_inst(SFGenLayer *layer);
static int is_global(SFGenLayer *layer);


/*----------------------------------------------------------------
 * local common variables
 *----------------------------------------------------------------*/

static int def_drum_inst;
static FILE *sample_fd;
static int mem_avail;
static SFPatchRec *sample_map;
static LoadList *part_list, *exclude_list;


/*================================================================
 * open / close patch
 *================================================================*/

int awe_open_patch(char *name, int type, int locked)
{
	struct open_patch_rec {
		awe_patch_info head;
		awe_open_parm parm;
	} rec;

	rec.head.type = AWE_OPEN_PATCH;
	rec.head.len = AWE_OPEN_PARM_SIZE;
	rec.parm.type = type;
	if (locked)
		rec.parm.type |= AWE_PAT_LOCKED;
	strncpy(rec.parm.name, name, sizeof(rec.parm.name));
	return awe_load_patch(&rec, sizeof(rec));
}

int awe_open_font(SFInfo *sf, FILE *fp, int locked)
{
	def_drum_inst = -1;
	sample_fd = fp;
	mem_avail = awe_dev;
	ioctl(seqfd, SNDCTL_SYNTH_MEMAVL, &mem_avail);
	sample_map = NULL;
	part_list = NULL;
	awe_init_marks();

	/* set zero attenuation level; this doesn't use seqbuf */
	_AWE_CMD_NOW(seqfd, awe_dev, 0, _AWE_MISC_MODE,
		     AWE_MD_ZERO_ATTEN, awe_option.default_atten);

	return awe_open_patch(sf->sf_name, AWE_PAT_TYPE_GS, locked);
}

int awe_close_patch(void)
{
	awe_patch_info head;

	head.key = AWE_PATCH;
	head.type = AWE_CLOSE_PATCH;
	head.len = 0;
	return awe_load_patch(&head, sizeof(head));
}

void awe_close_font(SFInfo *sf)
{
	def_drum_inst = -1;
	sample_fd = NULL;
	sample_map = NULL;
	part_list = NULL;
	awe_free_marks();
	awe_close_patch();
}


/*----------------------------------------------------------------*/

int awe_is_ram_fonts(SFInfo *sf)
{
	int i;
	for (i = 0; i < sf->nsamples; i++) {
		if (sf->sample[i].size > 0)
			return TRUE;
	}
	return FALSE;
}

/*----------------------------------------------------------------
 * load a preset map
 *----------------------------------------------------------------*/

int awe_load_map(LoadList *lp)
{
	struct open_patch_rec {
		awe_patch_info head;
		awe_voice_map parm;
	} rec;

	rec.head.type = AWE_MAP_PRESET;
	rec.head.len = sizeof(awe_voice_map);
	rec.parm.src_bank = lp->pat.bank;
	rec.parm.src_instr = lp->pat.preset;
	rec.parm.src_key = lp->pat.keynote;
	rec.parm.map_bank = lp->map.bank;
	rec.parm.map_instr = lp->map.preset;
	rec.parm.map_key = lp->map.keynote;
	if (awe_load_patch(&rec, sizeof(rec)) == -1)
		return AWE_RET_ERR;
	return AWE_RET_OK;
}

/*----------------------------------------------------------------
 * load a patch data
 *----------------------------------------------------------------*/

static int awe_load_patch(void *patch, int len)
{
	awe_patch_info *p;
	p = (awe_patch_info*)patch;
	p->key = AWE_PATCH;
	p->device_no = awe_dev;
	p->sf_id = 0;
	return write(seqfd, patch, len);
}


/*----------------------------------------------------------------
 * load sample wave data on the driver
 *----------------------------------------------------------------*/

typedef struct patch_rec {
	awe_patch_info patch;
	awe_sample_info hdr;
	unsigned short data[1];
} patch_rec;

static int awe_load_data(SFInfo *sf, FILE *fp, int sample, int *memsize)
{
	SFSampleInfo *sp;
	patch_rec *rec;

	sp = sf->sample + sample;
	if (*memsize < sp->size * 2)
		return AWE_RET_NOMEM;
	*memsize -= sp->size * 2;

	/* allocate temporary buffer for all the sample */
	rec = (patch_rec*)safe_malloc(sizeof(*rec) + sp->size * 2);

	/* set sample info */
	rec->hdr.sf_id = 0;
	rec->hdr.sample = sample;
	rec->hdr.start = sp->startsample;
	rec->hdr.end = sp->endsample;
	rec->hdr.loopstart = sp->startloop;
	rec->hdr.loopend = sp->endloop;
	rec->hdr.size = sp->size;
	rec->hdr.checksum_flag = 0;
	rec->hdr.mode_flags = 0;
	rec->hdr.checksum = 0;
		
	/* if this is not ROM sample, load it */
	if (rec->hdr.size > 0) {
		int pos = rec->hdr.start * 2;
		if (fp == NULL)
			return AWE_RET_ERR;
		if (pos < 0 || pos > sf->samplesize) {
			fprintf(stderr, "awe: illegal file pos %d\n", pos);
			return AWE_RET_ERR;
		}
		pos += sf->samplepos;
		fseek(fp, pos, SEEK_SET);
		fread(rec->data, rec->hdr.size, 2, fp);
		/* clear the blank data at tail */
		memset(rec->data + rec->hdr.end - rec->hdr.start, 0,
		       (rec->hdr.size - rec->hdr.end + rec->hdr.start) * 2);
	}
	/* ok, loading the patch.. */
	rec->patch.optarg = 0;
	rec->patch.len = AWE_SAMPLE_INFO_SIZE + rec->hdr.size * 2;
	/* not including the patch header size */
	rec->patch.type = AWE_LOAD_DATA;
	rec->patch.reserved = 0;
	if (awe_load_patch(rec, AWE_PATCH_INFO_SIZE + rec->patch.len) == -1) {
		safe_free(rec);
		return AWE_RET_ERR;
	}

	safe_free(rec);
	return AWE_RET_OK;
}

/*================================================================
 * load font info
 *================================================================*/

/* load the whole file at once */
int awe_load_font_buffered(SFInfo *sf, LoadList *plist, LoadList *elist, int load_alt)
{
	int rc, i, nsamples;
	int nomem;
	LoadList *p;
	int missing;

	part_list = plist;
	exclude_list = elist;
	nomem = 0;

	if (plist == NULL)
		search_def_drum_inst(sf);

	if (plist || elist) {
		if ((rc = mark_samples_in_list(sf)) == AWE_RET_ERR)
			return rc;
	}

	if (load_alt) {
		/* replace to alternative fonts if the fonts not exist */
		missing = FALSE;
		for (p = part_list; p; p = p->next) {
			if (p->loaded)
				continue;
			if (p->pat.bank == 128) {
				if (p->pat.preset != 0) {
					p->pat.preset = 0;
					missing = TRUE;
				}
			} else if (p->pat.bank != 0) {
				p->pat.bank = 0;
				missing = TRUE;
			}
		}
		if (missing) {
			if ((rc = mark_samples_in_list(sf)) == AWE_RET_ERR)
				return rc;
		}
	}

	/* corret number of samples; SF2 includes EOS at the last sample
	   which is a dummy sample meaning End Of Sample.
	   SBK doens't have it.
	   */
	if (sf->version == 1)
		nsamples = sf->nsamples;
	else
		nsamples = sf->nsamples - 1;

	for (i = 0; i < nsamples; i++) {
		if ((plist||elist) && !awe_is_listed_sample(i))
			continue;
		rc = awe_load_data(sf, sample_fd, i, &mem_avail);
		if (rc == AWE_RET_ERR)
			return rc;
		else if (rc == AWE_RET_NOMEM)
			nomem = 1;
		else {
			if (plist||elist)
				awe_mark_sample(i);
			else
				awe_add_marked_sample(i);
		}
	}


	for (i = 0; i < sf->npresets; i++) {
		SFPatchRec tmp;
		tmp.preset = sf->preset[i].preset;
		tmp.bank = sf->preset[i].bank;
		tmp.keynote = -1;
		if (plist == NULL || is_matched_preset(plist, &tmp)) {
			rc = load_font(sf, &sf->preset[i], &tmp,
				       buffer_map, buffer_load);
			if (rc == AWE_RET_ERR)
				return rc;
		}
	}

	if (nomem)
		return AWE_RET_NOMEM;
	return AWE_RET_OK;
}

static int mark_samples_in_list(SFInfo *sf)
{
	int i, rc;
	LoadList *p = NULL;

	for (i = 0; i < sf->npresets; i++) {
		SFPatchRec tmp;
		tmp.preset = sf->preset[i].preset;
		tmp.bank = sf->preset[i].bank;
		tmp.keynote = -1;
		if (part_list == NULL ||
		    (p = is_matched_preset(part_list, &tmp)) != NULL) {
			rc = load_font(sf, &sf->preset[i], &tmp,
				       buffer_map, buffer_sample);
			if (rc == AWE_RET_ERR)
				return rc;
			if (p)
				p->loaded = TRUE;
		}
	}
	return AWE_RET_OK;
}

static int buffer_map(SFPatchRec *pat, SFPatchRec *map, LayerTable *tbl)
{
	LoadList *p;
	for (p = exclude_list; p; p = p->next) {
		if (! awe_match_preset(pat, &p->map))
			continue;
		if (p->map.keynote == -1)
			return AWE_RET_SKIP;
		if (in_range(tbl, p->map.keynote))
			return AWE_RET_SKIP;
	}
	if (part_list == NULL) {
		*map = *pat;
		return AWE_RET_OK;
	}
	for (p = part_list; p; p = p->next) {
		if (! awe_match_preset(pat, &p->pat))
			continue;
		if (p->pat.keynote != -1) {
			if (! in_range(tbl, p->pat.keynote))
				continue;
			if (p->map.keynote != -1) {
				tbl->val[SF_keyRange] = RANGE(p->map.keynote,
							      p->map.keynote);
				if (! tbl->set[SF_keynum]) {
					tbl->val[SF_keynum] = p->pat.keynote;
					tbl->set[SF_keynum] = P_LAYER;
				}
			}
		}
		*map = p->map;
		if (p->map.bank == -1)
			map->bank = pat->bank;
		return AWE_RET_OK;
	}
	return AWE_RET_SKIP;
}

/* check if the specified preset matches the preset list
 */
static LoadList *is_matched_preset(LoadList *list, SFPatchRec *pat)
{
	LoadList *p;
	for (p = list; p; p = p->next) {
		if (awe_match_preset(&p->pat, pat))
			return p;
	}
	return NULL;
}

static int buffer_sample(SFInfo *sf, int sample)
{
	awe_add_sample(sample);
	return AWE_RET_SKIP; /* do not send info */
}

static int buffer_load(SFInfo *sf, int sample)
{
	if (! awe_is_marked(sample))
		return AWE_RET_SKIP;
	return AWE_RET_OK;
}


/*----------------------------------------------------------------*/

/* load each preset */
int awe_load_font(SFInfo *sf, SFPatchRec *pat, SFPatchRec *map)
{
	int rc, i;

	sample_map = map;
	for (i = 0; i < sf->npresets; i++) {
		SFPatchRec tmp;
		tmp.preset = sf->preset[i].preset;
		tmp.bank = sf->preset[i].bank;
		tmp.keynote = -1;
		if (awe_match_preset(&tmp, pat)) {
			rc = load_font(sf, &sf->preset[i], pat,
				       dynamic_map, dynamic_load);
			if (rc == AWE_RET_ERR || rc == AWE_RET_NOMEM)
				return rc;
			return AWE_RET_OK;
		}
	}
	return AWE_RET_SKIP;
}

static int dynamic_map(SFPatchRec *pat, SFPatchRec *map, LayerTable *tbl)
{
	if (sample_map == NULL) {
		*map = *pat;
		return AWE_RET_OK;
	}
	*map = *sample_map;
	if (pat->keynote == -1)
		return AWE_RET_OK;
	if (! in_range(tbl, pat->keynote))
		return AWE_RET_SKIP;
	tbl->val[SF_keyRange] = RANGE(sample_map->keynote,
				      sample_map->keynote);
	if (! tbl->set[SF_keynum]) {
		tbl->val[SF_keynum] = pat->keynote;
		tbl->set[SF_keynum] = P_LAYER;
	}
	return AWE_RET_OK;
}

static int dynamic_load(SFInfo *sf, int sample)
{
	int rc;
	if (awe_is_marked(sample))
		return AWE_RET_OK;
	if ((rc = awe_load_data(sf, sample_fd, sample, &mem_avail)) != AWE_RET_OK)
		return rc;
	awe_mark_sample(sample);
	return AWE_RET_OK;
}


/*----------------------------------------------------------------*/

static int load_font(SFInfo *sf, SFPresetHdr *preset, SFPatchRec *pat,
		     Mapper mapper, Loader loader)
{
	int rc, j, nlayers;
	SFGenLayer *layp, *globalp;

	/* if layer is empty, skip it */
	if ((nlayers = preset->hdr.nlayers) <= 0 ||
	    (layp = preset->hdr.layer) == NULL)
		return AWE_RET_SKIP;
	/* check global layer */
	globalp = NULL;
	if (is_global(layp)) {
		globalp = layp;
		layp++;
		nlayers--;
	}
	/* parse for each preset layer */
	for (j = 0; j < nlayers; j++, layp++) {
		LayerTable tbl;

		/* set up table */
		clear_table(&tbl);
		if (globalp)
			set_to_table(sf, &tbl, globalp, P_GLOBAL);
		set_to_table(sf, &tbl, layp, P_LAYER);
		
		/* parse the instrument */
		rc = parse_layer(sf, pat, &tbl, mapper, loader, 0);
		/* fatal error */
		if (rc == AWE_RET_ERR || rc == AWE_RET_NOMEM)
			return rc;
	}

	return AWE_RET_OK;
}


/*----------------------------------------------------------------*/

/* find instrument id of default drumset #0:
 * the standard drumset is often included in other drumsets.
 */
static void search_def_drum_inst(SFInfo *sf)
{
	int i;
	SFGenLayer *layp;

	def_drum_inst = -1;
	for (i = 0; i < sf->npresets; i++) {
		if (sf->preset[i].bank == 128 && sf->preset[i].preset == 0) {
			/* check the first layer */
			if ((layp = sf->preset[i].hdr.layer) == NULL)
				continue;
			if (!is_global(layp))
				def_drum_inst = find_inst(layp);
			break;
		}
	}
}

/*----------------------------------------------------------------*/

/* parse a preset layer and convert it to the patch structure */
static int parse_layer(SFInfo *sf, SFPatchRec *pat, LayerTable *tbl,
		       Mapper mapper, Loader loader, int level)
{
	SFInstHdr *inst;
	int rc, i, nlayers;
	SFGenLayer *lay, *globalp;

	if (level >= 2) {
		fprintf(stderr, "parse_layer: too deep instrument level\n");
		return AWE_RET_ERR;
	}

	/* instrument must be defined */
	if (!tbl->set[SF_instrument])
		return AWE_RET_SKIP;

	/* if non-standard drumset includes standard drum instruments,
	   skip it to avoid duplicate the data */
	inst = &sf->inst[tbl->val[SF_instrument]];
	if (def_drum_inst >= 0 && pat->bank == 128 && pat->preset != 0 &&
	    tbl->val[SF_instrument] == def_drum_inst)
			return AWE_RET_SKIP;

	/* if layer is empty, skip it */
	if ((nlayers = inst->hdr.nlayers) <= 0 ||
	    (lay = inst->hdr.layer) == NULL)
		return AWE_RET_SKIP;

	/* check global layer */
	globalp = NULL;
	if (is_global(lay)) {
		globalp = lay;
		lay++;
		nlayers--;
	}

	/* parse for each layer */
	for (i = 0; i < nlayers; i++, lay++) {
		LayerTable ctbl;
		clear_table(&ctbl);
		if (globalp)
			set_to_table(sf, &ctbl, globalp, P_GLOBAL);
		set_to_table(sf, &ctbl, lay, P_LAYER);

		if (!ctbl.set[SF_sampleId]) {
			/* recursive loading */
			merge_table(sf, &ctbl, tbl);
			if (! sanity_range(&ctbl))
				continue;
			if ((rc = parse_layer(sf, pat, &ctbl, mapper, loader,
					      level+1)) != AWE_RET_OK) {
				if (rc != AWE_RET_SKIP)
					return rc;
			}
		} else {
			SFPatchRec map;

			init_and_merge_table(sf, &ctbl, tbl);
			if (! sanity_range(&ctbl))
				continue;
			/* if the keynote is specified, check and remap it */
			if ((rc = mapper(pat, &map, &ctbl)) != AWE_RET_OK)
				continue;

			/* load or mark up the sample */
			rc = loader(sf, ctbl.val[SF_sampleId]);
			if (rc == AWE_RET_ERR || rc == AWE_RET_NOMEM)
				return rc;
			else if (rc == AWE_RET_SKIP)
				continue;

			/* load the info data */
			if ((rc = make_patch(sf, &map, &ctbl)) != AWE_RET_OK)
				return rc;
		}
	}
	return AWE_RET_OK;
}


/*----------------------------------------------------------------
 * find instrument
 *----------------------------------------------------------------*/

static int find_inst(SFGenLayer *layer)
{
	int i;
	for (i = 0; i < layer->nlists; i++) {
		if (layer->list[i].oper == SF_instrument)
			return layer->list[i].amount;
	}
	return -1;
}

static int is_global(SFGenLayer *layer)
{
	int i;
	for (i = 0; i < layer->nlists; i++) {
		if (layer->list[i].oper == SF_instrument ||
		    layer->list[i].oper == SF_sampleId)
			return 0;
	}
	return 1;
}


/*----------------------------------------------------------------
 * layer table handlers
 *----------------------------------------------------------------*/

/* initialize layer table */
static void clear_table(LayerTable *tbl)
{
	memset(tbl->val, 0, sizeof(tbl->val));
	memset(tbl->set, 0, sizeof(tbl->set));
}

/* set items in a layer to the table */
static void set_to_table(SFInfo *sf, LayerTable *tbl, SFGenLayer *lay, int level)
{
	int i;
	for (i = 0; i < lay->nlists; i++) {
		SFGenRec *gen = &lay->list[i];
		/* copy the value regardless of its copy policy */
		tbl->val[gen->oper] = gen->amount;
		tbl->set[gen->oper] = level;
	}
}

/* add an item to the table */
static void add_item_to_table(LayerTable *tbl, int oper, int amount, int level)
{
	LayerItem *item = &layer_items[oper];
	int o_lo, o_hi, lo, hi;

	switch (item->copy) {
	case L_INHRT:
		tbl->val[oper] += amount;
		break;
	case L_OVWRT:
		tbl->val[oper] = amount;
		break;
	case L_PRSET:
	case L_INSTR:
		/* do not overwrite */
		if (!tbl->set[oper])
			tbl->val[oper] = amount;
		break;
	case L_RANGE:
		if (!tbl->set[oper]) {
			tbl->val[oper] = amount;
		} else {
			o_lo = LOWNUM(tbl->val[oper]);
			o_hi = HIGHNUM(tbl->val[oper]);
			lo = LOWNUM(amount);
			hi = HIGHNUM(amount);
			if (lo < o_lo) lo = o_lo;
			if (hi > o_hi) hi = o_hi;
			tbl->val[oper] = RANGE(lo, hi);
		}
		break;
	}
}

/* merge two tables */
static void merge_table(SFInfo *sf, LayerTable *dst, LayerTable *src)
{
	int i;
	for (i = 0; i < SF_EOF; i++) {
		if (src->set[i]) {
			if (sf->version == 1) {
				if (!dst->set[i] ||
				    i == SF_keyRange || i == SF_velRange)
					/* just copy it */
					dst->val[i] = src->val[i];
			}
			else
				add_item_to_table(dst, i, src->val[i], P_GLOBAL);
			dst->set[i] = P_GLOBAL;
		}
	}
}

/* merge and set default values */
static void init_and_merge_table(SFInfo *sf, LayerTable *dst, LayerTable *src)
{
	int i;

	/* default value is not zero */
	if (sf->version == 1) {
		layer_items[SF_sustainEnv1].defv = 1000;
		layer_items[SF_sustainEnv2].defv = 1000;
		layer_items[SF_freqLfo1].defv = -725;
		layer_items[SF_freqLfo2].defv = -15600;
	} else {
		layer_items[SF_sustainEnv1].defv = 0;
		layer_items[SF_sustainEnv2].defv = 0;
		layer_items[SF_freqLfo1].defv = 0;
		layer_items[SF_freqLfo2].defv = 0;
	}

	/* set default */
	for (i = 0; i < SF_EOF; i++) {
		if (!dst->set[i])
			dst->val[i] = layer_items[i].defv;
	}
	merge_table(sf, dst, src);
	/* convert from SBK to SF2 */
	if (sf->version == 1) {
		for (i = 0; i < SF_EOF; i++) {
			if (dst->set[i])
				dst->val[i] = sbk_to_sf2(i, dst->val[i]);
		}
	}
}


/*----------------------------------------------------------------
 * check key and velocity range
 *----------------------------------------------------------------*/

static int sanity_range(LayerTable *tbl)
{
	int lo, hi;

	lo = LOWNUM(tbl->val[SF_keyRange]);
	hi = HIGHNUM(tbl->val[SF_keyRange]);
	if (lo < 0 || lo > 127 || hi < 0 || hi > 127 || hi < lo)
		return 0;

	lo = LOWNUM(tbl->val[SF_velRange]);
	hi = HIGHNUM(tbl->val[SF_velRange]);
	if (lo < 0 || lo > 127 || hi < 0 || hi > 127 || hi < lo)
		return 0;

	return 1;
}

static int in_range(LayerTable *tbl, int key)
{
	int lo, hi;

	lo = LOWNUM(tbl->val[SF_keyRange]);
	hi = HIGHNUM(tbl->val[SF_keyRange]);
	if (key >= lo && key <= hi)
		return TRUE;
	return FALSE;
}


/*----------------------------------------------------------------
 * create patch record from the stored data table
 *----------------------------------------------------------------*/

static int make_patch(SFInfo *sf, SFPatchRec *pat, LayerTable *tbl)
{
	static awe_voice_rec_patch vrec;
	
	/* set voice header */
	if (pat->bank > 0 || awe_option.default_bank < 0)
		vrec.hdr.bank = pat->bank;
	else
		vrec.hdr.bank = awe_option.default_bank;
	vrec.hdr.instr = pat->preset;
	vrec.hdr.nvoices = 1;
	vrec.hdr.write_mode = AWE_WR_APPEND;

	make_info(sf, &vrec.info, tbl);

	/* set patch header */
	vrec.patch.optarg = 0;
	vrec.patch.len = AWE_VOICE_REC_SIZE + AWE_VOICE_INFO_SIZE;
	vrec.patch.type = AWE_LOAD_INFO;
	vrec.patch.reserved = 0;

	/* then, put it to sequencer */
	if (awe_load_patch(&vrec, sizeof(vrec)) < 0)
		return AWE_RET_ERR;

	return AWE_RET_OK;
}


/*----------------------------------------------------------------*/

/* conver to awe_voice_info parameter */
static void make_info(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl)
{
	set_sample_info(sf, vp, tbl);
	set_init_info(sf, vp, tbl);
	set_rootkey(sf, vp, tbl);
	set_modenv(sf, vp, tbl);
	set_volenv(sf, vp, tbl);
	set_lfo1(sf, vp, tbl);
	set_lfo2(sf, vp, tbl);

	memset(vp->parm.reserved, 0, sizeof(vp->parm.reserved));
}

/*----------------------------------------------------------------*/

/* set sample address */
static void set_sample_info(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl)
{
	SFSampleInfo *sp;

	vp->sf_id = 0;
	vp->sample = tbl->val[SF_sampleId];
	sp = &sf->sample[vp->sample];

	vp->start = (tbl->val[SF_startAddrsHi] << 15)
		+ tbl->val[SF_startAddrs];
	vp->end = (tbl->val[SF_endAddrsHi] << 15)
		+ tbl->val[SF_endAddrs];
	vp->loopstart = (tbl->val[SF_startloopAddrsHi] << 15)
		+ tbl->val[SF_startloopAddrs];
	vp->loopend = (tbl->val[SF_endloopAddrsHi] << 15)
		+ tbl->val[SF_endloopAddrs];

	vp->rate_offset = awe_calc_rate_offset(sp->samplerate);

	/* sample mode */
	vp->mode = 0;
	if (sp->sampletype & 0x8000)
		vp->mode |= AWE_MODE_ROMSOUND;
	if (tbl->val[SF_sampleFlags] == 1 || tbl->val[SF_sampleFlags] == 3)
		/* looping */
		vp->mode |= AWE_MODE_LOOPING;
	else {
		/* short-shot; set a small blank loop at the tail */
		if (sp->loopshot > 8) {
			vp->loopstart = sp->endsample + 8 - sp->startloop;
			vp->loopend = sp->endsample + sp->loopshot - 8 - sp->endloop;
		} else {
			fprintf(stderr, "loop size is too short: %d\n", sp->loopshot);
			exit(1);
		}
	}
}

/*----------------------------------------------------------------*/

/* set global information */
static void set_init_info(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl)
{
	/* key range */
	vp->low = LOWNUM(tbl->val[SF_keyRange]);
	vp->high = HIGHNUM(tbl->val[SF_keyRange]);

	/* velocity range */
	vp->vellow = LOWNUM(tbl->val[SF_velRange]);
	vp->velhigh = HIGHNUM(tbl->val[SF_velRange]);

	/* fixed key & velocity */
	vp->fixkey = tbl->val[SF_keynum];
	vp->fixvel = tbl->val[SF_velocity];
	
	/* panning position */
	vp->pan = awe_calc_pan(tbl->val[SF_panEffectsSend]);
	vp->fixpan = -1;

	/* initial volume */

	vp->amplitude = awe_option.default_volume * 127 / 100;
	/* this is not a centibel? */
	vp->attenuation = awe_calc_attenuation((int)(tbl->val[SF_initAtten] / awe_option.atten_sense));
	
	/* chorus & reverb effects */
	if (tbl->set[SF_chorusEffectsSend])
		vp->parm.chorus = awe_calc_chorus(tbl->val[SF_chorusEffectsSend]);
	else
		vp->parm.chorus = awe_option.default_chorus * 255 / 100;
	if (tbl->set[SF_reverbEffectsSend])
		vp->parm.reverb = awe_calc_reverb(tbl->val[SF_chorusEffectsSend]);
	else
		vp->parm.reverb = awe_option.default_reverb * 255 / 100;

	/* initial cutoff & resonance */
	vp->parm.cutoff = awe_calc_cutoff(tbl->val[SF_initialFilterFc]);
	vp->parm.filterQ = awe_calc_filterQ(tbl->val[SF_initialFilterQ]);

	/* exclusive class key */
	vp->exclusiveClass = tbl->val[SF_keyExclusiveClass];
}

/*----------------------------------------------------------------*/

/* calculate root key & fine tune */
static void set_rootkey(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl)
{
	SFSampleInfo *sp = &sf->sample[vp->sample];

	/* scale tuning */
	vp->scaleTuning = tbl->val[SF_scaleTuning];

	/* set initial root key & fine tune */
	if (sf->version == 1 && tbl->set[SF_samplePitch]) {
		/* set from sample pitch */
		vp->root = tbl->val[SF_samplePitch] / 100;
		vp->tune = -tbl->val[SF_samplePitch] % 100;
		if (vp->tune <= -50) {
			vp->root++;
			vp->tune = 100 + vp->tune;
		}
		if (vp->scaleTuning == 50)
			vp->tune /= 2;
	} else {
		/* from sample info */
		vp->root = sp->originalPitch;
		vp->tune = sp->pitchCorrection;
	}

	/* orverride root key */
	if (tbl->set[SF_rootKey])
		vp->root += tbl->val[SF_rootKey] - sp->originalPitch;

	/* tuning */
	if (sf->version == 1)
		vp->tune += tbl->val[SF_coarseTune] * vp->scaleTuning +
			(int)tbl->val[SF_fineTune] * (int)vp->scaleTuning / 100;
	else
		vp->tune += tbl->val[SF_coarseTune] * 100 + tbl->val[SF_fineTune];

	/* correct too high pitch */
	if (vp->root >= vp->high + 60)
		vp->root -= 60;
}

/*----------------------------------------------------------------*/

#define TO_WORD(hi,lo) (((unsigned short)(hi) << 8) | (unsigned short)(lo))

/* modulation envelope parameters */
static void set_modenv(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl)
{
	/* delay */
	vp->parm.moddelay = awe_calc_delay(tbl->val[SF_delayEnv1]);

	/* attack & hold */
	vp->parm.modatkhld = awe_calc_atkhld(tbl->val[SF_attackEnv1],
					     tbl->val[SF_holdEnv1]);


	/* decay & sustain */
	vp->parm.moddcysus =
		TO_WORD(awe_calc_mod_sustain(tbl->val[SF_sustainEnv1]),
			awe_calc_decay(tbl->val[SF_decayEnv1]));

	/* release */
	vp->parm.modrelease =
		TO_WORD(0x80, awe_calc_decay(tbl->val[SF_releaseEnv1]));

	/* key hold/decay */
	vp->parm.modkeyhold = tbl->val[SF_autoHoldEnv1];
	vp->parm.modkeydecay = tbl->val[SF_autoDecayEnv1];

	/* pitch / cutoff shift */
	vp->parm.pefe =
		TO_WORD(awe_calc_pitch_shift(tbl->val[SF_env1ToPitch]),
			awe_calc_cutoff_shift(tbl->val[SF_env1ToFilterFc], 6));
}

/* volume envelope parameters */
static void set_volenv(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl)
{
	/* delay */
	vp->parm.voldelay = awe_calc_delay(tbl->val[SF_delayEnv2]);

	/* attack & hold */
	vp->parm.volatkhld = awe_calc_atkhld(tbl->val[SF_attackEnv2],
					     tbl->val[SF_holdEnv2]);
	/* decay & sustain */
	vp->parm.voldcysus =
		TO_WORD(awe_calc_sustain(tbl->val[SF_sustainEnv2]),
			awe_calc_decay(tbl->val[SF_decayEnv2]));

	/* release */
	vp->parm.volrelease =
		TO_WORD(0x80, awe_calc_decay(tbl->val[SF_releaseEnv2]));

	/* key hold/decay */
	vp->parm.volkeyhold = tbl->val[SF_autoHoldEnv2];
	vp->parm.volkeydecay = tbl->val[SF_autoDecayEnv2];
}

/* lfo1 parameters (tremolo & vibrato) */
static void set_lfo1(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl)
{
	vp->parm.lfo1delay = awe_calc_delay(tbl->val[SF_delayLfo1]);
	vp->parm.fmmod =
		TO_WORD(awe_calc_pitch_shift(tbl->val[SF_lfo1ToPitch]),
			awe_calc_cutoff_shift(tbl->val[SF_lfo1ToFilterFc], 3));
	vp->parm.tremfrq =
		TO_WORD(awe_calc_tremolo(tbl->val[SF_lfo1ToVolume]),
			awe_calc_freq(tbl->val[SF_freqLfo1]));
}

/* lfo2 parameters (vibrato only) */
static void set_lfo2(SFInfo *sf, awe_voice_info *vp, LayerTable *tbl)
{
	vp->parm.lfo2delay = awe_calc_delay(tbl->val[SF_delayLfo2]);
	vp->parm.fm2frq2 =
		TO_WORD(awe_calc_pitch_shift(tbl->val[SF_lfo2ToPitch]),
			awe_calc_freq(tbl->val[SF_freqLfo2]));
}


