#!/usr/local/bin/wish -f
#
# MIDI Option Editor for AWE MIDI Player
# Copyright (c) 1996,1997  Takashi Iwai
#

#
# create editor window and edit options
# w is the window name, midi is the midi file name to be edited
#
proc EditIndexFile {w {midi ""}} {
    global Edit EditSet EditState

    if {[winfo exists $w]} {
	return
    }

    set EditState(default_set) 0
    if {$midi == ""} {
	set EditState(default_set) 1
	set title "default options"
    } else {
	set last [string last "/" $midi]
	if {$last >= 0} {
	    set title [string range $midi [expr $last + 1] end]
	} else {
	    set title $midi
	}
    }

    ReadDefaultIndex
    if {$midi != ""} {
	set file [SearchIndexFile $midi]
	ReadIndexFile $file
    }

    set EditState(canceled) 0
    MakeIndexEditor $w $title
    tkwait visibility $w
    #push-grab $w
    tkwait window $w
    #pop-grab

    if {! $EditState(canceled)} {
	if {$EditState(default_set)} {
	    SaveDefaultIndex
	} else {
	    SaveIndexFile $file
	}
    }
}


#----------------------------------------------------------------
# boolean options
set AWEBoolIndex {
    "realpan" "autoskip" "tuning" "samecsec" "gsmacro" "xgmacro"
    "xgmap" "acceptall" "parsetitle" "chnprior" "multipart" "trace"
    "usefx" "seqecho" "seqbuf"
}

set AWEBoolDefault {
    1 0 1 1 1 1
    0 1 1 0 0 0
    0 1 1
}

# integer options
set AWEIntIndex {
    "chorus" "reverb" "chorusdepth" "reverbdepth" "volume"
    "drumflag" "tracks" "volscale" "mt32" "offset" "verbose"
}

set AWEIntDefault {
    2 4 0 0 100
    0x02000200 16 100 0 0 0
}

# string options
set AWEStrIndex {
    "title" "dynamic" "subsf" "xgload" "mode"
    "fx_cutoff" "fx_resonance" "fx_attack" "fx_release"
    "fx_vibrate" "fx_vibdepth" "fx_vibdelay"
}

set AWEStrDefault {
    "" "" "" "" ""
    170 6 50 50 30 4 1500
}

# get boolean (0 or 1) from a string argument
proc BoolVal {val} {
    set val [string tolower $val]
    if {$val == "on" || $val == "true" || $val == "yes" || $val == 1} {
	return 1
    } else {
	return 0
    }
}

# convert to on/off from boolean value
proc BoolStr {val} {
    if {$val} {
	return "on"
    } else {
	return "off"
    }
}

# read system resource file and parse options
proc ReadDefaultIndex {} {
    global Edit EditSet EditDefault EditState
    global AWEBoolIndex AWEBoolDefault
    global AWEIntIndex AWEIntDefault
    global AWEStrIndex AWEStrDefault
    global env

    set file "$env(HOME)/.drvmidirc"

    catch {unset Edit}
    catch {unset EditSet}
    catch {unset EditDefault}
    
    for {set i 0} {$i < [llength $AWEBoolIndex]} {incr i} {
	set type [lindex $AWEBoolIndex $i]
	set Edit($type) [lindex $AWEBoolDefault $i]
	set EditSet($type) 0
    }
    for {set i 0} {$i < [llength $AWEIntIndex]} {incr i} {
	set type [lindex $AWEIntIndex $i]
	set Edit($type) [lindex $AWEIntDefault $i]
	set EditSet($type) 0
    }
    for {set i 0} {$i < [llength $AWEStrIndex]} {incr i} {
	set type [lindex $AWEStrIndex $i]
	set Edit($type) [lindex $AWEStrDefault $i]
	set EditSet($type) 0
    }

    set Edit(convert) {}
    set EditSet(convert) 0

    foreach i [array names Edit] {
	set EditDefault($i) $Edit($i)
    }

    if {! [file readable $file]} {
	set file "/etc/drvmidirc"
	if {! [file readable $file]} {
	    return
	}
    }

    set fid [open $file "r"]
    if {$fid == ""} {
	return
    }

    while {[gets $fid line] != -1} {
	set elm {}
	foreach i [splittoken $line] {
	    if {$i != {}} {
		lappend elm $i
	    }
	}
	set type [lindex $elm 0]
	if {[string match "#*" $type]} {
	    continue
	}
	set elm [lreplace $elm 0 0]

	for {set idx 0} {$idx < [llength $elm]} {incr idx} {
	    set i [lindex $elm $idx]
	    if {[regexp "^--\(\[a-zA-Z_\]+\)=*\(.*\)$" $i dummy type arg]} {
		if {$EditState(default_set)} {
		    set EditSet($type) 1
		}
		if {$type == "drum"} {
		    if {$arg > 0} {
			set Edit(drumflag) [expr $Edit(drumflag)|(1 << ($arg-1))]
		    } elseif {$arg < 0} {
			set Edit(drumflag) [expr $Edit(drumflag)&~(1<<(-$arg+1))]
		    }

		} elseif {$type == "convert"} {
		    lappend Edit($type) $arg
		    
		} elseif {[lsearch $AWEBoolIndex $type] != -1} {
		    if {$arg != ""} {
			set Edit($type) [BoolVal $arg]
		    } else {
			set Edit($type) [expr !$Edit($type)]
		    }
		} elseif {[lsearch $AWEIntIndex $type] != -1 ||\
			[lsearch $AWEStrIndex $type] != -1} {
		    set Edit($type) $arg
		}
	    }
	}
    }

    foreach i [array names Edit] {
	set EditDefault($i) $Edit($i)
    }

    close $fid
}


# read a midi option file and parse options
proc ReadIndexFile {file} {
    global Edit EditSet
    global AWEBoolIndex AWEBoolDefault
    global AWEIntIndex AWEIntDefault
    global AWEStrIndex AWEStrDefault


    ReadDefaultIndex

    if {! [file readable $file]} {
	return
    }

    set fid [open $file "r"]
    if {$fid == ""} {
	return
    }

    while {[gets $fid line] != -1} {
	set elm {}
	foreach i [split $line] {
	    if {$i != {}} {
		lappend elm $i
	    }
	}
	set type [lindex $elm 0]
	if {[string match "#*" $type]} {
	    continue
	}
	set arg [lindex $elm 1]
	set EditSet($type) 1
	if {$type == "drum"} {
	    if {$arg > 0} {
		set Edit(drumflag) [expr $Edit(drumflag)|(1 << ($arg-1))]
	    } elseif {$arg < 0} {
		set Edit(drumflag) [expr $Edit(drumflag)&~(1<<(-$arg+1))]
	    }
	    set EditSet(drumflag) 1
	    continue
	}
	if {[lsearch $AWEBoolIndex $type] != -1} {
	    set Edit($type) [BoolVal $arg]
	} elseif {[lsearch $AWEIntIndex $type] != -1} {
	    set Edit($type) $arg
	} elseif {[lsearch $AWEStrIndex $type] != -1} {
	    set Edit($type) [lrange $elm 1 end]
	}
    }

    close $fid
}

# save to the midi option file
proc SaveIndexFile {file} {
    global Edit EditSet EditDefault
    global AWEBoolIndex AWEBoolDefault
    global AWEIntIndex AWEIntDefault
    global AWEStrIndex AWEStrDefault

    set fid [open $file "w"]
    if {$fid == ""} {
	return
    }
    foreach i [array names Edit] {
	if {[lsearch $AWEStrIndex $i] != -1} {
	    if {$Edit($i) != $EditDefault($i)} {
		puts $fid "$i $Edit($i)"
	    }
	    continue
	}

	if {! $EditSet($i)} {
	    continue
	}
	if {$i == "drumflag"} {
	    puts $fid "$i 0x[format %x $Edit($i)]"
	} elseif {[lsearch $AWEBoolIndex $i] != -1} {
	    puts $fid "$i [BoolStr $Edit($i)]"
	} elseif {[lsearch $AWEIntIndex $i] != -1} {
	    puts $fid "$i $Edit($i)"
	}
    }

    close $fid
}

# save to the system resource file
proc SaveDefaultIndex {} {
    global Edit EditSet EditDefault
    global AWEBoolIndex AWEBoolDefault
    global AWEIntIndex AWEIntDefault
    global AWEStrIndex AWEStrDefault
    global env

    set file "$env(HOME)/.drvmidirc"

    set fid [open $file "w"]
    if {$fid == ""} {
	return
    }

    foreach i [array names Edit] {
	if {[lsearch $AWEStrIndex $i] != -1} {
	    if {$Edit($i) != $EditDefault($i)} {
		puts $fid "default --$i=$Edit($i)"
	    }
	    continue
	}

	if {! $EditSet($i)} {
	    continue
	}
	if {$i == "drumflag"} {
	    puts $fid "default --$i=0x[format %x $Edit($i)]"
	} elseif {$i == "convert"} {
	    foreach clist $Edit(convert) {
		puts $fid "default --convert=\"[escstring $clist]\""
	    }
	} elseif {[lsearch $AWEBoolIndex $i] != -1} {
	    puts $fid "default --$i=[BoolStr $Edit($i)]"
	} elseif {[lsearch $AWEIntIndex $i] != -1} {
	    puts $fid "default --$i=$Edit($i)"
	}
    }
    close $fid
}

# make a valid option file name from the midi file
proc SearchIndexFile {midi} {
    set root [file rootname $midi]
    set file "$root.id"
    if [file readable $file] {
	return $file
    }
    set dir [file dirname $file]
    set tail [file tail $file]
    return "$dir/.$tail"
}

#----------------------------------------------------------------
# make editor window

proc MakeIndexEditor {w title} {
    global Edit EditSet EditState

    set f [my-dialog $w "Edit MIDI Option: $title" 0 1 [list\
	    [list {  OK  } "destroy $w"]\
	    [list "Cancel" "set EditState(canceled) 1; destroy $w"]\
	    [list "Advanced.." "DoAdvancedSettings $w.adv"]]]

    frame $f.title -relief ridge -bd 1
    checkbutton $f.title.parsetitle -text "Parse Title in File"\
	    -variable Edit(parsetitle)\
	    -command "SetState parsetitle"
    pack $f.title.parsetitle -side top
    if {! $EditState(default_set)} {
	frame $f.title.string
	label $f.title.string.label -text "Title"
	entry $f.title.string.val -width 40 -textvariable Edit(title)
	bind $f.title.string.val <Return> ""
	pack $f.title.string.label $f.title.string.val -side left
	pack $f.title.string -side top
    }
    pack $f.title -side top -fill x

    global Stat Config

    frame $f.chorus -relief ridge -bd 1
    frame $f.chorus.in
    menubutton $f.chorus.in.mode -text "Chorus"\
	    -menu $f.chorus.in.mode.m\
	    -relief raised -bd 2 -indicatoron 1
    menu $f.chorus.in.mode.m -tearoff 0
    foreach i $Stat(ChorusList) {
	$f.chorus.in.mode.m add radio -label [lindex $i 1]\
		-variable Edit(chorus) -value [lindex $i 0]\
		-command "SetChorus"
    }
    foreach i $Config(UserChorus) {
	$f.chorus.in.mode.m add radio -label [lindex $i 1]\
		-variable Edit(chorus) -value [lindex $i 0]\
		-command "SetChorus"
    }
    scale $f.chorus.in.depth -length 167\
	    -orient horizontal -from 0 -to 127 -showvalue true\
	    -variable Edit(chorusdepth)\
	    -command "SetState chorusdepth"
    pack $f.chorus.in.mode $f.chorus.in.depth -side left
    pack $f.chorus.in -side top
    pack $f.chorus -side top -fill x

    frame $f.reverb -relief ridge -bd 1
    frame $f.reverb.in
    menubutton $f.reverb.in.mode -text "Reverb"\
	    -menu $f.reverb.in.mode.m\
	    -relief raised -bd 2 -indicatoron 1
    menu $f.reverb.in.mode.m -tearoff 0
    foreach i $Stat(ReverbList) {
	$f.reverb.in.mode.m add radio -label [lindex $i 1]\
		-variable Edit(reverb) -value [lindex $i 0]\
		-command "SetReverb"
    }
    foreach i $Config(UserReverb) {
	$f.reverb.in.mode.m add radio -label [lindex $i 1]\
		-variable Edit(reverb) -value [lindex $i 0]\
		-command "SetReverb"
    }
    scale $f.reverb.in.depth -length 167\
	    -orient horizontal -from 0 -to 127 -showvalue true\
	    -variable Edit(reverbdepth)\
	    -command "SetState reverbdepth"
    pack $f.reverb.in.mode $f.reverb.in.depth -side left
    pack $f.reverb.in -side top
    pack $f.reverb -side top -fill x

    frame $f.vol -relief ridge -bd 1
    frame $f.vol.in
    label $f.vol.in.label -text "Volume Scale (%)"
    scale $f.vol.in.scale -length 140\
	    -orient horizontal -from 0 -to 200 -showvalue true\
	    -variable Edit(volscale)\
	    -command "SetState volscale"
    pack $f.vol.in.label $f.vol.in.scale -side left
    pack $f.vol.in -side top
    pack $f.vol -side top -fill x

    frame $f.mt32 -relief ridge -bd 1
    label $f.mt32.label -text "MIDI Playing Mode"
    pack $f.mt32.label -side top -fill x
    frame $f.mt32.in
    radiobutton $f.mt32.in.normal -text "GM/GS/XG"\
	    -variable Edit(mt32) -value 0\
	    -command "SetState mt32"
    radiobutton $f.mt32.in.font -text "MT32"\
	    -variable Edit(mt32) -value 1\
	    -command "SetState mt32"
    radiobutton $f.mt32.in.emu -text "MT32 Emulation"\
	    -variable Edit(mt32) -value 2\
	    -command "SetState mt32"
    pack $f.mt32.in.normal $f.mt32.in.font $f.mt32.in.emu -side left
    pack $f.mt32.in -side top
    pack $f.mt32 -side top -fill x

    frame $f.drum -relief ridge -bd 1
    label $f.drum.label -text "Drum Channel Status"
    pack $f.drum.label -side top -fill x
    canvas $f.drum.c -width 320 -height 60 -bd 0 -bg black
    for {set i 0} {$i < 16} {incr i} {
	set x [expr $i * 20]
	$f.drum.c create text [expr $x + 10] 2 -anchor n -fill white\
		-text [expr $i+1]
	$f.drum.c create rect $x 21 [expr $x + 18] 38 -fill green\
		-tags drum$i
	set j [expr $i + 16]
	$f.drum.c create rect $x 41 [expr $x + 18] 58 -fill green\
		-tags drum$j
	$f.drum.c bind drum$i <1> [list ToggleDrumState $f.drum.c $i]
	$f.drum.c bind drum$j <1> [list ToggleDrumState $f.drum.c $j]
	SetDrumState $f.drum.c $i
	SetDrumState $f.drum.c $j
    }
    pack $f.drum.c -side top
    pack $f.drum -side top -fill x

    frame $f.bool -relief ridge -bd 1
    checkbutton $f.bool.multipart -text "Play with Double MIDI Modules"\
	    -variable Edit(multipart)\
	    -command "SetState multipart"
    pack $f.bool.multipart -side top
    checkbutton $f.bool.realpan -text "Allow Realtime Pan Change"\
	    -variable Edit(realpan)\
	    -command "SetState realpan"
    pack $f.bool.realpan -side top
    checkbutton $f.bool.autoskip -text "Skip Blank Events on Head"\
	    -variable Edit(autoskip)\
	    -command "SetState autoskip"
    pack $f.bool.autoskip -side top
    pack $f.bool -side top -fill x

    if {$EditState(default_set)} {
	button $f.convert -text "Edit Conversion Rules"\
		-command "DoEditConversion $w.conv"
	pack $f.convert -side top -fill x
    }
}

proc ToggleDrumState {w i} {
    global Edit EditSet
    if {$Edit(drumflag) & (1 << $i)} {
	set Edit(drumflag) [expr $Edit(drumflag) & ~(1 << $i)]
    } else {
	set Edit(drumflag) [expr $Edit(drumflag) | (1 << $i)]
    }
    SetDrumState $w $i
    set EditSet(drumflag) 1
}

proc SetDrumState {w i} {
    global Edit
    if {$Edit(drumflag) & (1 << $i)} {
	$w itemconfigure drum$i -fill red
    } else {
	$w itemconfigure drum$i -fill green
    }
}

proc SetState {type {state 1}} {
    global Edit EditSet EditDefault
    if {$Edit($type) != $EditDefault($type)} {
	set EditSet($type) 1
    }
}

#----------------------------------------------------------------
# advanced options window

proc DoAdvancedSettings {w} {
    if {[winfo exists $w]} {
	return
    }
    MakeAdvancedSettings $w
    tkwait visibility $w
    #push-grab $w
    tkwait window $w
    #pop-grab
}

proc MakeAdvancedSettings {w} {
    global Edit EditSet EditState

    set f [my-dialog $w "Advanced Settings" 0 1\
	    [list [list {  OK  } "destroy $w"]]]

    frame $f.bool -relief ridge -bd 1
    checkbutton $f.bool.gsmacro -text "Allow GS SYSEX Macros"\
	    -variable Edit(gsmacro)\
	    -command "SetState gsmacro"
    pack $f.bool.gsmacro -side top
    checkbutton $f.bool.xgmacro -text "Allow XG Bank Controls"\
	    -variable Edit(xgmacro)\
	    -command "SetState xgmacro"
    pack $f.bool.xgmacro -side top
    checkbutton $f.bool.samecsec -text "Check Note-On/Off Timing"\
	    -variable Edit(samecsec)\
	    -command "SetState samecsec"
    pack $f.bool.samecsec -side top
    checkbutton $f.bool.acceptall -text "Allow All Notes/Sounds Off"\
	    -variable Edit(acceptall)\
	    -command "SetState acceptall"
    pack $f.bool.acceptall -side top
    checkbutton $f.bool.chnprior -text "Channel Priority Mode"\
	    -variable Edit(chnprior)\
	    -command "SetState chnprior"
    pack $f.bool.chnprior -side top
    checkbutton $f.bool.tuning -text "Allow Tuning via RPN"\
	    -variable Edit(tuning)\
	    -command "SetState tuning"
    pack $f.bool.tuning -side top
    checkbutton $f.bool.seqecho -text "Use Sequencer Echo Back"\
	    -variable Edit(seqecho)\
	    -command "SetState seqecho"
    pack $f.bool.seqecho -side top
    checkbutton $f.bool.seqbuf -text "Enable Sequencer Buffering"\
	    -variable Edit(seqbuf)\
	    -command "SetState seqbuf"
    pack $f.bool.seqbuf -side top
    pack $f.bool -side top -fill x

    if {$EditState(default_set)} {
	frame $f.verbose -relief ridge -bd 1
	label $f.verbose.label -text "Verbosity"
	radiobutton $f.verbose.lq -text "Quiet"\
		-variable Edit(verbose) -value -1\
		-command "SetState verbose"
	radiobutton $f.verbose.l0 -text "0"\
		-variable Edit(verbose) -value 0\
		-command "SetState verbose"
	radiobutton $f.verbose.l1 -text "1"\
		-variable Edit(verbose) -value 1\
		-command "SetState verbose"
	radiobutton $f.verbose.l2 -text "2"\
		-variable Edit(verbose) -value 2\
		-command "SetState verbose"
	pack $f.verbose.label $f.verbose.lq $f.verbose.l0 $f.verbose.l1 $f.verbose.l2\
		-side left
	pack $f.verbose -side top -fill x
    }

    frame $f.font -relief ridge -bd 1
    label $f.font.dlabel -text "Dynamic Font Loading"
    entry $f.font.dynamic -width 20\
	    -textvariable Edit(dynamic)
    pack $f.font.dlabel $f.font.dynamic -side top
    checkbutton $f.font.xgmap -text "XG Drum/SFX Mapping"\
	    -variable Edit(xgmap)\
	    -command "SetState xgmap"
    label $f.font.xlabel -text "Dynamic XG Font Loading"
    entry $f.font.xgload -width 20\
	    -textvariable Edit(xgload)
    pack $f.font.xgmap $f.font.xlabel $f.font.xgload -side top
    if {! $EditState(default_set)} {
	label $f.font.slabel -text "Load Additional Fonts"
	entry $f.font.subsf -width 20\
		-textvariable Edit(subsf)
	pack $f.font.slabel $f.font.subsf -side top
    }
    pack $f.font -side top -fill x

    frame $f.fx -relief ridge -bd 1
    checkbutton $f.fx.use -text "Enable Effect Controls"\
	    -variable Edit(usefx)\
	    -command "SetState usefx"
    pack $f.fx.use -side top
    frame $f.fx.cutoff
    label $f.fx.cutoff.label -width 15 -text "Cut Off"
    entry $f.fx.cutoff.entry -width 12 -textvariable Edit(fx_cutoff)
    #numeric-bind $f.fx.cutoff.entry
    pack $f.fx.cutoff.label $f.fx.cutoff.entry -side left
    pack $f.fx.cutoff -side top
    frame $f.fx.resonance
    label $f.fx.resonance.label -width 15 -text "Resonance"
    entry $f.fx.resonance.entry -width 12 -textvariable Edit(fx_resonance)
    #numeric-bind $f.fx.resonance.entry
    pack $f.fx.resonance.label $f.fx.resonance.entry -side left
    pack $f.fx.resonance -side top
    frame $f.fx.attack
    label $f.fx.attack.label -width 15 -text "Attack Time"
    entry $f.fx.attack.entry -width 12 -textvariable Edit(fx_attack)
    #numeric-bind $f.fx.attack.entry
    pack $f.fx.attack.label $f.fx.attack.entry -side left
    pack $f.fx.attack -side top
    frame $f.fx.release
    label $f.fx.release.label -width 15 -text "Release Time"
    entry $f.fx.release.entry -width 12 -textvariable Edit(fx_release)
    #numeric-bind $f.fx.release.entry
    pack $f.fx.release.label $f.fx.release.entry -side left
    pack $f.fx.release -side top
    frame $f.fx.vibrate
    label $f.fx.vibrate.label -width 15 -text "Vibrato Rate"
    entry $f.fx.vibrate.entry -width 12 -textvariable Edit(fx_vibrate)
    #numeric-bind $f.fx.vibrate.entry
    pack $f.fx.vibrate.label $f.fx.vibrate.entry -side left
    pack $f.fx.vibrate -side top
    frame $f.fx.vibdepth
    label $f.fx.vibdepth.label -width 15 -text "Vibrato Depth"
    entry $f.fx.vibdepth.entry -width 12 -textvariable Edit(fx_vibdepth)
    #numeric-bind $f.fx.vibdepth.entry
    pack $f.fx.vibdepth.label $f.fx.vibdepth.entry -side left
    pack $f.fx.vibdepth -side top
    frame $f.fx.vibdelay
    label $f.fx.vibdelay.label -width 15 -text "Vibrato Delay"
    entry $f.fx.vibdelay.entry -width 12 -textvariable Edit(fx_vibdelay)
    #numeric-bind $f.fx.vibdelay.entry
    pack $f.fx.vibdelay.label $f.fx.vibdelay.entry -side left
    pack $f.fx.vibdelay -side top
    pack $f.fx -side top -fill x
}

proc DoEditConversion {w} {
    global Edit EditSet EditState
    set f [my-dialog $w "Edit Conversion Rules" 0 0 [list\
	    [list {  OK  } "destroy $w"]]]
    frame $f.list -relief ridge -bd 2
    set lw [make-listbox $f.list 30 10]
    pack $f.list -side top -fill x

    $lw delete 0 end
    foreach i $Edit(convert) {
	$lw insert end $i
    }

    frame $f.edit -relief ridge -bd 2
    frame $f.edit.ext
    label $f.edit.ext.label -text "Extension"
    entry $f.edit.ext.entry -width 9 -textvariable EditState(extrule)
    bind $f.edit.ext.entry <Return> "ModifyRule $lw"
    pack $f.edit.ext.label $f.edit.ext.entry -side top
    frame $f.edit.fmt
    label $f.edit.fmt.label -text "Conversion"
    entry $f.edit.fmt.entry -width 22 -textvariable EditState(extformat)
    bind $f.edit.fmt.entry <Return> "ModifyRule $lw"
    pack $f.edit.fmt.label $f.edit.fmt.entry -side top
    pack $f.edit $f.edit.ext $f.edit.fmt -side left -expand 1
    pack $f.edit -side top -fill x
    
    bind $lw <Button-1> [list SetRule $lw %y]

    frame $f.btn -relief ridge -bd 2
    button $f.btn.add -text "Add" -command "AddNewRule $lw"
    button $f.btn.modify -text "Modify" -command "ModifyRule $lw"
    button $f.btn.delete -text "Delete" -command "DeleteRule $lw"
    button $f.btn.clear -text "Clear" -command "ClearRule $lw"
    pack $f.btn.add $f.btn.modify $f.btn.delete $f.btn.clear -side left\
	    -expand 1
    pack $f.btn -side top -fill x

    set EditState(convert) -1
    set EditState(extrule) ""
    set EditState(extformat) ""

    tkwait window $w
}

proc SetRule {lw y} {
    global Edit EditState
    set idx [$lw nearest $y]
    if {$idx >= 0 && $idx < [llength $Edit(convert)]} {
	set EditState(convert) $idx
	regexp "^\(\[^/\]+\)+/\(.+\)$" [lindex $Edit(convert) $idx] dummy\
		EditState(extrule) EditState(extformat)
    }
}

proc ClearRule {lw} {
    global EditState
    $lw selection clear 0 end
    set EditState(extrule) ""
    set EditState(extformat) ""
    set EditState(convert) -1
}

proc AddNewRule {lw} {
    global Edit EditSet EditState
    if {$EditState(extrule) == {} || $EditState(extformat) == {}} {
	return
    }
    set arg "$EditState(extrule)/$EditState(extformat)"
    set EditSet(convert) 1
    lappend Edit(convert) $arg
    $lw insert end $arg
    ClearRule $lw
}

proc DeleteRule {lw} {
    global Edit EditSet EditState
    if {$EditState(convert) != -1} {
	set EditSet(convert) 1
	set Edit(convert) [lreplace $Edit(convert) $EditState(convert) $EditState(convert)]
	$lw delete $EditState(convert) $EditState(convert)
	ClearRule $lw
    }
}

proc ModifyRule {lw} {
    global Edit EditSet EditState
    if {$EditState(convert) != -1} {
	set arg "$EditState(extrule)/$EditState(extformat)"
	set EditSet(convert) 1
	set Edit(convert) [lreplace $Edit(convert) $EditState(convert) $EditState(convert) $arg]
	$lw delete $EditState(convert) $EditState(convert)
	$lw insert $EditState(convert) $arg
	ClearRule $lw
    }
}

