//
// /home/ms/sidplay/qtsidplay/RCS/sidplay.cpp,v
//

#include <iostream.h>
#include <iomanip.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>

#include <qapp.h>
#include <qbttngrp.h>
#include <qchkbox.h>
#include <qdialog.h>
#include <qdict.h>
#include <qdir.h>
#include <qfiledlg.h>
#include <qmsgbox.h>
#include <qpushbt.h>
#include <qpixmap.h>
#include <qradiobt.h>
#include <qsignal.h>

#include "AudioDriver.h"
#include "LinePrompt.h"
#include "MainDialog.h"
#include "WaveViewDialog.h"

#include "xsidplay.h"
#include "player.h"  // libsidplay

MainDialog* myMainDlg;
extern WaveViewDialog* myWaveViewDlg;

bool expertMode = false;
udword usage = 0;  // listening mileage in seconds

// --------------------------------------------------------------------------

// Instantiate emulator engine.
emuEngine myEmuEngine;
emuConfig myEmuConfig;
emuConfig oldEmuConfig;  // a 2nd copy to allow UNDO

// Instantiate audio driver.
audioDriver myAudio;

// Instantiate sidtune loader.
sidTune mySidTune(0);
sidTuneInfo mySidTuneInfo;

// --------------------------------------------------------------------------

bool initAudio();
void closeAudio();

void parseCommandLine(int argc, char **argv);

static int commandLineSelectedSong = 0;

// Audio defaults.
static const bufferConfig default_bufferConfig = 
{
	2048, 64, 10
};

// Audio settings.
bufferConfig myBufferConfig = default_bufferConfig;
bufferConfig bakBufferConfig = default_bufferConfig;

// Multi-buffering system.
static const maxBuffers = 256;  // worst case: 256*256 = 64 KB
static int buffers = 2;
static int multiBufferSize = 2048;
static bool bufferFlag[maxBuffers];
static ubyte* pBuffer[maxBuffers];
static int bufferCount[maxBuffers];
static int lastBufferCount;
static int nextBuffer;
static int currentBuffer;

// --------------------------------------------------------------------------

#include "btnstate.h"
#include "mytimer.h"

int playbackState = IS_STOPPED;
bool isReadyToPlay = false;

static myTimer playbackTimer;

myTimer::myTimer( QObject* parent, const char* name ) : QTimer( parent, name )
{
    connect(this,SIGNAL(timeout()),this,SLOT(processJob()));
}


static int timeInSecs;
static char timeString[] = "00:00";
static const int timeDigitMax[] =
{
	// 60 sec, 60 min
	// '9'++, '6', '9'++, '6'
	0x3a,0x36,0x3a,0x36,-1
};
static const int timeDigitIndex[] =
{ 
	4,3,1,0,-1  // order in which to increase the digits
};

void initTimeDisplay()
{	
	timeInSecs = 0;
	myEmuEngine.resetSecondsThisSong();
	
	timeString[0] = ' ';
	timeString[1] = '0';
	timeString[2] = ':';
	timeString[3] = '0';
	timeString[4] = '0';
	myMainDlg->drawTime(timeString);
}

inline void updateTimeDisplay(int secs)
{
	if (timeInSecs != secs)
	{
		while (timeInSecs < secs)
		{
			timeInSecs++;
			int i = 0;
			while (timeDigitIndex[i] != (-1))
			{
				int d = timeDigitIndex[i];  // get digit number
				if (timeString[d] == ' ')   // digit not used so far
					timeString[d] = '1';
				else
					timeString[d]++;        // increase digit
				if (timeString[d] == timeDigitMax[i])
				{
					timeString[d] = '0';
					i++;  // proceed to next digit only if necessary
				}
				else
					break;
			};
		};
		myMainDlg->drawTime(timeString);
	}
}


void myTimer::processJob()
{
	static isActive = false;  // semaphore
	if (!isActive)
	{
		isActive = true;
		if (playbackState & IS_PLAYING)
		{
			int secs = myEmuEngine.getSecondsThisSong();

			// Multi-buffering system unavailable.
			//
			// Due to a bug in the OSS sound driver in Linux 2.0.x 
			// it is impossible to determine the current playback
			// position in an accurate way.
			
			sidEmuFillBuffer(myEmuEngine,mySidTune,pBuffer[currentBuffer],multiBufferSize);
			myAudio.Play(pBuffer[currentBuffer],multiBufferSize);
			bufferFlag[currentBuffer] = true;
			lastBufferCount += multiBufferSize;
//				bufferCount[nextBuffer] = lastBufferCount;
//				if (++nextBuffer >= buffers)
//					nextBuffer = 0;
//				if (++currentBuffer >= buffers)
//					currentBuffer = 0;
			
			updateTimeDisplay(secs);
			
			if (myWaveViewDlg->isVisible())
			{
				if ( bufferFlag[currentBuffer] )
				{
					bufferFlag[currentBuffer] = false;
					myWaveViewDlg->paintWaveformFunc(pBuffer[currentBuffer]);
				}
			}
		}
		isActive = false;
	}
}

void preFillBuffers()
{
	currentBuffer = 0;
	nextBuffer = currentBuffer+1;
	
	lastBufferCount = 0;
	for (int i = 0; i < buffers; i++)
	{
		sidEmuFillBuffer(myEmuEngine,mySidTune,pBuffer[i],multiBufferSize);
		bufferFlag[i] = true;
		lastBufferCount += multiBufferSize;
		bufferCount[i] = lastBufferCount;
	}
	for (int i = 0; i < buffers; i++)
	{
		myAudio.Play(pBuffer[i],multiBufferSize);
	}
}

// --------------------------------------------------------------------------

void stopPlaying()
{
	playbackState = IS_STOPPED;
	playbackTimer.stop();
	myAudio.Reset();
	closeAudio();
}

void pausePlaying()
{
	playbackState = IS_PAUSED;
	myAudio.Reset();
}

void resumePlaying()
{
	playbackState = IS_PLAYING;
}

void startPlaying()
{
	stopPlaying();
	initTimeDisplay();
	
	if ( initAudio() )
    {
		sidEmuInitializeSong(myEmuEngine,mySidTune,mySidTuneInfo.currentSong);
		extern bool sidEmuFastForwardReplay(int percent);
		sidEmuFastForwardReplay(100);
		if ( !mySidTune )
		{
			myMainDlg->buttonsOff();
		}
		else
		{
			myMainDlg->showSidTuneInfo();
			myMainDlg->setPrevBtnState();
			myMainDlg->setNextBtnState();
				
			preFillBuffers();
			// Enable timer callback.
			playbackTimer.changeInterval(0);
			playbackTimer.start(playbackTimer.interval,false);
			playbackState = IS_PLAYING;
		}
    }
}

// ---------------------------------------------------------------------------

void loadSidTune(const char* fileName)
{
	// Stop any kind of audio playback.
	stopPlaying();
	initTimeDisplay();

	if (!mySidTune.open(fileName))
    {
		mySidTune.returnInfo(mySidTuneInfo);
		myMainDlg->buttonsOff();
		myMainDlg->clearSidTuneInfo();
		
		QMessageBox error;
		error.setIcon(QMessageBox::Warning);
		error.setText(mySidTuneInfo.statusString);
		error.adjustSize();
		error.show();
    }
	else
    {
		if ( mySidTune )
		{
			mySidTune.returnInfo(mySidTuneInfo);
			if (commandLineSelectedSong != 0)
			{
				sidEmuInitializeSong(myEmuEngine,mySidTune,commandLineSelectedSong);
				commandLineSelectedSong = 0;
			}
			else
				sidEmuInitializeSong(myEmuEngine,mySidTune,mySidTuneInfo.startSong);
			mySidTune.returnInfo(mySidTuneInfo);
			isReadyToPlay = true;
			startPlaying();
			myMainDlg->buttonsForceInitialState();
		}
    }
}

// --------------------------------------------------------------------------

static bufferConfig argBufferConfig = {0,0,0};
static unsigned long inFile = 0;

int main( int argc, char ** argv )
{
    QApplication a( argc, argv );

	// The first thing we do, so that dialog constructors can initialize
	// checkboxes and buttons with defaults.
	myEmuEngine.getConfig(myEmuConfig);
	myEmuEngine.getConfig(oldEmuConfig);

	myMainDlg = new MainDialog;
    a.setMainWidget( myMainDlg );
	
	extern void readConfigFile();
	readConfigFile();
	
	myEmuEngine.setConfig(myEmuConfig);
	myEmuEngine.getConfig(myEmuConfig);
	
	extern void readDefaultVoiceVolume();
	readDefaultVoiceVolume();

	myMainDlg->show();

	// Check initial state of Emulator Engine.
	if (!myEmuEngine)
	{
		QMessageBox error;
		error.setIcon(QMessageBox::Critical);
		error.setText("SID Emulator Engine\nis out of memory.");
		error.adjustSize();
		error.show();
		exit(-1);
	}
	
	// Check state of audio device.
	if (!initAudio())
	{
		QMessageBox error;
		error.setIcon(QMessageBox::Warning);
		error.setText("Cannot open audio device.");
		error.adjustSize();
		error.show();
	}
	closeAudio();

	// Reset multi-buffering system.
	lastBufferCount = 0;
	for (int i = 0; i < maxBuffers; i++)
	{
		bufferFlag[i] = false;
		bufferCount[i] = 0;
		pBuffer[i] = 0;
	}

	parseCommandLine(argc,argv);
	bakBufferConfig = myBufferConfig;
	if (expertMode)
	{
		myMainDlg->enableWaveViewDlg(true);
		myBufferConfig.bufferSize = 2048;
		myBufferConfig.fragments = 2;
		myBufferConfig.fragSizeExp = 11;
		buffers = 2;
	}
	else
	{
		myMainDlg->enableWaveViewDlg(false);
	}
	// Check whether we take the argument line settings.
	// Need to do it this way because of defaults when no
	// sidplayrc file is available and expert mode.
	if (argBufferConfig.fragments != 0)
		myBufferConfig.fragments = argBufferConfig.fragments;
	if (argBufferConfig.fragSizeExp != 0)
		myBufferConfig.fragSizeExp = argBufferConfig.fragSizeExp;
	if (myBufferConfig.bufferSize < 4*280)  // waveView w=280!
		myBufferConfig.bufferSize = 4*280;
	cout 
		<< "SIDPLAY: OSS: "
		<< "Fragments = " << myBufferConfig.fragments << ", "
		<< "Fragment size = " << (1L<<myBufferConfig.fragSizeExp) << ", "
		<< "Buffer size = " << myBufferConfig.bufferSize
		<< endl;

	// Init multi-buffering system.
	multiBufferSize = myBufferConfig.bufferSize;
	bool success = true;
	for (int i = 0; i < buffers; i++)
	{
		if ((pBuffer[i] = new ubyte[multiBufferSize]) == 0)
		{
			success = false;
			break;
		}
	}
	if ( !success )
	{
		QMessageBox error;
		error.setIcon(QMessageBox::Critical);
		error.setText("Out of memory.");
		error.adjustSize();
		error.show();
		exit(-1);
	}

	// HVSC/STIL stuff.
	extern bool stilInfoDesired;
	bool handleStil();
	if ( stilInfoDesired )
		myMainDlg->setStilInfoEnabled(handleStil());
		
	if (inFile != 0)
	{
		cout << "Loading sidtune ..." << endl;
		loadSidTune(argv[inFile]);
	}
	
	// Give control to Qt.
    return a.exec();
}


void parseCommandLine(int argc, char **argv)
{
	inFile = 0;
	for (int a = 1; a < argc; a++)  
    {
		// Check for options prefix "-".
		if (argv[a][0] == '-')  
		{
			// Single "-" means reading from stdin.
			if (strlen(argv[a]) == 1)
			{
				if ( inFile == 0 )
				{
					inFile = a;
					break;
				}
				else
				{
					cerr << "Command line syntax error." << endl;
				}
			}
			else
			{
				if (strncasecmp(&argv[a][1],"-expert",7) == 0)
					expertMode = true;
				else if (strncasecmp(&argv[a][1],"b",1) == 0)
					argBufferConfig.bufferSize = atoi(&argv[a][1+1]);
				else if (strncasecmp(&argv[a][1],"fn",2) == 0)
					argBufferConfig.fragments = atoi(&argv[a][1+2]);
				else if (strncasecmp(&argv[a][1],"fs",2) == 0)
					argBufferConfig.fragSizeExp = atoi(&argv[a][1+2]);
				else if (strncasecmp(&argv[a][1],"o",1) == 0)
					commandLineSelectedSong = atoi(&argv[a][1+1]);
			}
		}
		else
		{
			// No prefix. Thus is considered a filename argument.
			if ( inFile == 0 )
			{
				inFile = a;
			}
			else 
			{
				cerr << "Command line syntax error." << endl;
				break;
			}
		}
    }
}

// ------------------------------------------------------------- Audio driver

static bool isOpen = false;

void closeAudio()
{
	if (isOpen)
    {
		myAudio.Close();  // Close audio driver.
		isOpen = false;
    }
}

bool initAudio()
{
	closeAudio();
	if (!myAudio.Open(myEmuConfig.frequency,myEmuConfig.bitsPerSample,
					  myEmuConfig.channels,myBufferConfig.fragments,
					  myBufferConfig.fragSizeExp))
    {
		isOpen = false;
    }
	else
    {
		isOpen = true;
		// Get sample encoding (format) and used frequency from driver.
		myEmuConfig.sampleFormat = myAudio.GetSampleEncoding();
		myEmuConfig.bitsPerSample = myAudio.GetSamplePrecision();
		myEmuConfig.frequency = myAudio.GetFrequency();
		// Set the emuEngine to supported settings.
		myEmuEngine.setConfig(myEmuConfig);
    }
	return isOpen;
}

// --------------------------------------------------------------------------

#include "STIL.h"

static const char hvscStilFile[] = "/DOCUMENTS/STIL.txt";
STIL* stilTxt;
QDict<char>* stilDict;

bool handleStil()
{
	// If HVSC path has not been specified, prompt for it,
	extern QDir* hvscPath;
	if (hvscPath == 0)
	{
		LinePrompt myPrompt;
		myPrompt.setLabelText("Please specify HVSC v2.x root directory:");
		myPrompt.setLineText("/usr/lib/c64music");  // default
		myPrompt.show();
		hvscPath = new QDir(myPrompt.getLineText());
	}
	
	// Do existence check on STIL database file.
	QString stilFileName = hvscPath->absPath();
	stilFileName += hvscStilFile;
	if ( !hvscPath->exists(stilFileName,true) )
	{		
		QMessageBox error;
		error.setIcon(QMessageBox::Warning);
		error.setText("Specified directory is incorrect.\nCould not find STIL text file.");
		error.adjustSize();
		error.show();
		// Don't accept input.
		delete hvscPath;
		hvscPath = 0;
		return false;
	}
	// Load STIL database and put it into hashtable.
	else
	{
		stilTxt = new STIL(stilFileName);
		stilDict = new QDict<char>(49999,false,true);
		if (stilTxt)
		{
			const char* entry;
			do
			{
				entry = stilTxt->getNextEntry();
				if (entry != 0)
				{
					// Cut X:XX time at end of line.
					QString tmp(entry);
					int pos = tmp.findRev('#');
					if (pos != (-1))
					{
						pos = tmp.find(' ',pos);
						if (pos != (-1))
							tmp.resize(pos);
					}
					
					const char* comment = stilTxt->getFormattedString();
					if (comment != 0)
					{
						stilDict->insert(tmp,comment);
					}
				}
			}
			while (entry != 0);
			return true;
		}
		else
			return false;
	}
}
