/*
 * dnode.c - FreeBSD node functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dnode.c,v 1.14 96/01/11 12:54:27 abe Exp $";
#endif


#include "lsof.h"


#if	_FREEBSDV>=200
# if	defined(HASPROCFS)
_PROTOTYPE(static void getmemsz,(pid_t pid));


/*
 * getmemsz() - get memory size of a /proc/<n>/mem entry
 */

static void
getmemsz(pid)
	pid_t pid;
{
	int n;
	struct kinfo_proc *p;
	struct vmspace vm;

	for (n = 0, p = P; n < Np; n++, p++) {
		if (p->P_PID == pid) {
			if (p->P_VMSPACE == NULL
			||  kread((KA_T)p->P_VMSPACE, (char *)&vm, sizeof(vm)))
				return;
			Lf->sz = (unsigned long)ctob(vm.vm_tsize + vm.vm_dsize
						     + vm.vm_ssize);
			Lf->sz_def = 1;
			return;
		}
	}
}
# endif	/* defined(HASPROCFS) */
#endif	/* _FREEBSDV>=200 */


/*
 * process_node() - process vnode
 */

void
process_node(va)
	caddr_t va;			/* vnode kernel space address */
{

	dev_t dev;
	char dev_ch[32];
	struct inode *i = NULL;
	struct lockf lf;
	int lt;
	struct mfsnode *m = NULL;
	struct nfsnode *n = NULL;
	char *ty;
	enum vtype type;
	struct vnode *v, vb;
	struct l_vfs *vfs;

#if	_FREEBSDV>=200
	struct inode ib;
	struct mfsnode mb;
	struct nfsnode nb;
#endif	/* _FREEBSDV>=200 */

#if	defined(HASPROCFS)
	struct pfsnode *p = NULL;
	struct procfsid *pfi;
	static int pgsz = -1;
	struct vmspace vm;

# if	_FREEBSDV>=200
	struct pfsnode pb;
# endif	/* _FREEBSDV>=200 */
#endif	/* defined(HASPROCFS) */

/*
 * Read the vnode.
 */
	if ( ! va) {
		enter_nm("no vnode address");
		return;
	}
	v = &vb;
	if (readvnode((caddr_t)va, v)) {
                enter_nm(Namech);
                return;
        }

# if	defined(HASNCACHE)
	Lf->id = v->v_id;
	Lf->na = (unsigned long)va;
# endif	/* defined(HASNCACHE) */

/*
 * Get the vnode type.
 */
	if (v->v_mount == NULL)
		vfs = NULL;
	else
		vfs = readvfs(v->v_mount);
	if (vfs) {
		switch (vfs->type) {
		case MOUNT_NFS:
			Ntype = N_NFS;
			break;

#if	defined(HASPROCFS)
		case MOUNT_PROCFS:
			Ntype = N_PROC;
			break;
#endif	/* defined(HASPROCFS) */

		}
	}
	if (Ntype == N_REGLR) {
		switch (v->v_type) {
		case VFIFO:
			Ntype = N_FIFO;
			break;
		}
	}
/*
 * Define the specific node pointer.
 */
	switch (v->v_tag) {
	case VT_MFS:

#if	_FREEBSDV<200
		m = (struct mfsnode *)v->v_data;
#else	/* _FREEBSDV>=200 */
		if (!v->v_data
		||  kread((KA_T)v->v_data, (char *)&mb, sizeof(mb))) {
			(void) sprintf(Namech, "no mfs node: %#x", v->v_data);
			enter_nm(Namech);
			return;
		}
		m = &mb;
#endif	/* _FREEBSDV<200 */

		break;
	case VT_NFS:

#if	_FREEBSDV<200
		n = (struct nfsnode *)v->v_data;
#else	/* _FREEBSDV>=200 */
		if (!v->v_data
		||  kread((KA_T)v->v_data, (char *)&nb, sizeof(nb))) {
			(void) sprintf(Namech, "no nfs node: %#x", v->v_data);
			enter_nm(Namech);
			return;
		}
		n = &nb;
#endif	/* _FREEBSDV<200 */

		break;

#if	defined(HASPROCFS)
	case VT_PROCFS:

# if	_FREEBSDV<200
		p = (struct pfsnode *)v->v_data;
# else	/* _FREEBSDV>=200 */
		if (!v->v_data
		||  kread((KA_T)v->v_data, (char *)&pb, sizeof(pb))) {
			(void) sprintf(Namech, "no pfs node: %#x", v->v_data);
			enter_nm(Namech);
			return;
		}
		p = &pb;
# endif	/* _FREEBSDV<200 */

		break;
#endif	/* defined(HASPROCFS) */

	case VT_UFS:

#if	_FREEBSDV<200
		i = (struct inode *)v->v_data;
#else	/* _FREEBSDV>=200 */
		if (!v->v_data
		||  kread((KA_T)v->v_data, (char *)&ib, sizeof(ib))) {
			(void) sprintf(Namech, "no ifs node: %#x", v->v_data);
			enter_nm(Namech);
			return;
		}
		i = &ib;
#endif	/* _FREEBSDV<200 */

		if (i->i_lockf != NULL
		&&  kread((KA_T)i->i_lockf, (char *)&lf, sizeof(lf)) == 0) {

		/*
		 * Determine the lock state.
		 */
			if (lf.lf_start == (off_t)0 && lf.lf_end == (off_t)-1)
				lt = 1;
			else
				lt = 0;
			if (lf.lf_type == F_RDLCK)
				Lf->lock = lt ? 'R' : 'r';
			else if (lf.lf_type == F_WRLCK)
				Lf->lock = lt ? 'W' : 'w';
			else if (lf.lf_type == (F_RDLCK | F_WRLCK))
				Lf->lock = 'u';
		}
		break;
	}
/*
 * Get device and type for printing.
 */
	type = v->v_type;
	if (n != NULL)
		dev = n->n_vattr.va_fsid;
	else if (i != NULL)
		dev = (type == VCHR) ? i->i_rdev : i->i_dev;
/*
 * Obtain the inode number.
 */
	if (i != NULL) {
		Lf->inode = (unsigned long)i->i_number;
		Lf->inp_ty = 1;
	} else if (n != NULL) {
		Lf->inode = (unsigned long)n->n_vattr.va_fileid;
		Lf->inp_ty = 1;
	}

#if	defined(HASPROCFS)
# if	_FREEBSDV>=200
	else if (p != NULL) {
		Lf->inode = (unsigned long)p->pfs_fileno;
		Lf->inp_ty = 1;
	}
# endif	/* _FREEBSDV>=200 */
#endif	/* defined(HASPROCFS) */

/*
 * Obtain the file size.
 */
	if (Foffset)
		Lf->off_def = 1;
	else {
		switch (Ntype) {
		case N_FIFO:
			if (!Fsize)
				Lf->off_def = 1;
			break;
		case N_NFS:
			if (n != NULL) {
				Lf->sz = (unsigned long)n->n_vattr.va_size;
				Lf->sz_def = 1;
			}
			break;

#if	defined(HASPROCFS)
		case N_PROC:

# if	_FREEBSDV<200
			if (type == VDIR || p == NULL || p->pfs_vs == NULL
			||  kread((KA_T)p->pfs_vs, (char *)&vm, sizeof(vm)))
				break;
			if (pgsz < 0)
				pgsz = getpagesize();
			Lf->sz = (unsigned long)((pgsz * vm.vm_tsize)
			       +         (pgsz * vm.vm_dsize)
			       +         (pgsz * vm.vm_ssize));
			Lf->sz_def = 1;
			break;
# else	/* _FREEBSD>=200 */
			if (p) {
				switch(p->pfs_type) {
				case Proot:
				case Pproc:
					Lf->sz = (unsigned long)DEV_BSIZE;
					Lf->sz_def = 1;
					break;
				case Pmem:
					(void) getmemsz(p->pfs_pid);
					break;
				case Pregs:
					Lf->sz = (unsigned long)
						 sizeof(struct reg);
					Lf->sz_def = 1;
					break;
				case Pfpregs:
					Lf->sz = (unsigned long)
						 sizeof(struct fpreg);
					Lf->sz_def = 1;
					break;
				}
			}
# endif	/* _FREEBSDV<200 */
#endif	/* defined(HASPROCFS) */

		case N_REGLR:
			if (type == VREG || type == VDIR) {
				if (i != NULL) {
					Lf->sz = (unsigned long)i->i_size;
					Lf->sz_def = 1;
				} else if (m != NULL) {
					Lf->sz = (unsigned long)m->mfs_size;
					Lf->sz_def = 1;
				}
			}
			else if (type == VCHR && !Fsize)
				Lf->off_def = 1;
			break;
		}
	}
/*
 * Record an NFS file selection.
 */
	if (Ntype == N_NFS && Fnfs)
		Lf->sf |= SELNFS;
/*
 * Save the file system names.
 */
	if (vfs) {
		Lf->fsdir = vfs->dir;
		Lf->fsdev = vfs->fsname;
	}
/*
 * Format the vnode type, and possibly the device name.
 */
	switch (type) {

	case VNON:
		ty ="VNON";
		break;
	case VREG:
	case VDIR:
		ty = (type == VREG) ? "VREG" : "VDIR";
		Lf->dev_def = 1;
		Lf->dev = dev;
		break;
	case VBLK:
		ty = "VBLK";
		break;
	case VCHR:
		Lf->dev = dev;
		Lf->dev_def = Lf->is_chr_dev = 1;
		ty = "VCHR";
		break;
	case VLNK:
		ty = "VLNK";
		break;

#if	defined(VSOCK)
	case VSOCK:
		ty = "SOCK";
		break;
#endif

	case VBAD:
		ty = "VBAD";
		break;
	case VFIFO:
		Lf->dev = dev;
		Lf->dev_def = 1;
		ty = "FIFO";
		break;
	default:
		if (type > 9999)
			(void) sprintf(Lf->type, "*%03d", type % 1000);
		else
			(void) sprintf(Lf->type, "%4d", type);
		(void) strcpy(Namech, "unknown type");
		ty = NULL;
	}
	if (ty)
		(void) strcpy(Lf->type, ty);
/*
 * Handle some special cases:
 *
 * 	ioctl(fd, TIOCNOTTY) files;
 *	memory node files;
 *	/proc files.
 */

	if (type == VBAD)
		(void) strcpy(Namech, "(revoked)");
	else if (m != NULL) {
		Lf->dev_def = 0;
		(void) sprintf(Namech, "%#x", m->mfs_baseoff);
		(void) strcpy(dev_ch, "    memory");
		enter_dev_ch(dev_ch);
	}

#if	defined(HASPROCFS)
	else if (p != NULL) {
		Lf->dev_def = 0;

# if	_FREEBSDV<200
		if (type == VDIR)
			(void) sprintf(Namech, "/%s", HASPROCFS);
		else
			(void) sprintf(Namech, "/%s/%0*d", HASPROCFS,
				PNSIZ, p->pfs_pid);
		enter_nm(Namech);
# else	/* _FREEBSDV>=200 */
		ty = NULL;
		(void) sprintf(Namech, "/%s", HASPROCFS);
		switch (p->pfs_type) {
		case Proot:
			ty = "PDIR";
			break;
		case Pproc:
			(void) sprintf(endnm(), "/%d", p->pfs_pid);
			ty = "PDIR";
			break;
		case Pfile:
			(void) sprintf(endnm(), "/%d/file", p->pfs_pid);
			ty = "PFIL";
			break;
		case Pmem:
			(void) sprintf(endnm(), "/%d/mem", p->pfs_pid);
			ty = "PMEM";
			break;
		case Pregs:
			(void) sprintf(endnm(), "/%d/regs", p->pfs_pid);
			ty = "PREG";
			break;
		case Pfpregs:
			(void) sprintf(endnm(), "/%d/fpregs", p->pfs_pid);
			ty = "PFPR";
			break;
		case Pctl:
			(void) sprintf(endnm(), "/%d/ctl", p->pfs_pid);
			ty = "PCTL";
			break;
		case Pstatus:
			(void) sprintf(endnm(), "/%d/status", p->pfs_pid);
			ty = "PSTA";
			break;
		case Pnote:
			(void) sprintf(endnm(), "/%d/note", p->pfs_pid);
			ty = "PNTF";
			break;
		case Pnotepg:
			(void) sprintf(endnm(), "/%d/notepg", p->pfs_pid);
			ty = "PGRP";
			break;
		}
		if (ty)
			(void) strcpy(Lf->type, ty);
		enter_nm(Namech);

# endif	/* _FREEBSD<200 */
	}
#endif	/* defined(HASPROCFS) */

/*
 * If this is a VCHR file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VCHR && Lf->dev_def)
		find_ch_ino();
/*
 * Test for specified file.
 */

#if	defined(HASPROCFS)
	if (Ntype == N_PROC) {
		if (Procsrch)
			Lf->sf |= SELNM;
		else {
			for (pfi = Procfsid; pfi; pfi = pfi->next) {

# if	defined(HASPINODEN)
				if (pfi->inode == p->pfs_fileno)
# else	/* !defined(HASPINODEN) */
				if (pfi->pid == p->pfs_pid)
# endif	/* defined(HASPINODEN) */

				{
					Lf->sf |= SELNM;
					break;
				}
			}
		}
	} else
#endif	/* defined(HASPROCFS) */

	{
		if (Sfile && is_file_named(NULL, type))
			Lf->sf |= SELNM;
	}
/*
 * Enter name characters.
 */
	if (Namech[0])
		enter_nm(Namech);
}
