// Copyright (c) 2015-2019 The Khronos Group Inc.
// 
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// 
//     http://www.apache.org/licenses/LICENSE-2.0
// 
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// 
// ---- Exceptions to the Apache 2.0 License: ----
// 
// As an exception, if you use this Software to generate code and portions of
// this Software are embedded into the generated code as a result, you may
// redistribute such product without providing attribution as would otherwise
// be required by Sections 4(a), 4(b) and 4(d) of the License.
// 
// In addition, if you combine or link code generated by this Software with
// software that is licensed under the GPLv2 or the LGPL v2.0 or 2.1
// ("`Combined Software`") and if a court of competent jurisdiction determines
// that the patent provision (Section 3), the indemnity provision (Section 9)
// or other Section of the License conflicts with the conditions of the
// applicable GPL or LGPL license, you may retroactively and prospectively
// choose to deem waived or otherwise exclude such Section(s) of the License,
// but only in their entirety and only with respect to the Combined Software.
//     

// This header is generated from the Khronos Vulkan XML API Registry.

#ifndef VULKAN_HPP
#define VULKAN_HPP

#include <algorithm>
#include <array>
#include <cstddef>
#include <cstdint>
#include <cstring>
#include <initializer_list>
#include <string>
#include <system_error>
#include <tuple>
#include <type_traits>
#include <vulkan/vulkan.h>
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
# include <memory>
# include <vector>
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#if !defined(VULKAN_HPP_ASSERT)
# include <cassert>
# define VULKAN_HPP_ASSERT   assert
#endif

// <tuple> includes <sys/sysmacros.h> through some other header
// this results in major(x) being resolved to gnu_dev_major(x)
// which is an expression in a constructor initializer list.
#if defined(major)
  #undef major
#endif
#if defined(minor)
  #undef minor
#endif

// Windows defines MemoryBarrier which is deprecated and collides
// with the vk::MemoryBarrier struct.
#ifdef MemoryBarrier
  #undef MemoryBarrier
#endif

static_assert( VK_HEADER_VERSION ==  97 , "Wrong VK_HEADER_VERSION!" );

// 32-bit vulkan is not typesafe for handles, so don't allow copy constructors on this platform by default.
// To enable this feature on 32-bit platforms please define VULKAN_HPP_TYPESAFE_CONVERSION
#if defined(__LP64__) || defined(_WIN64) || (defined(__x86_64__) && !defined(__ILP32__) ) || defined(_M_X64) || defined(__ia64) || defined (_M_IA64) || defined(__aarch64__) || defined(__powerpc64__)
# if !defined( VULKAN_HPP_TYPESAFE_CONVERSION )
#  define VULKAN_HPP_TYPESAFE_CONVERSION
# endif
#endif

#if !defined(VULKAN_HPP_HAS_UNRESTRICTED_UNIONS)
# if defined(__clang__)
#  if __has_feature(cxx_unrestricted_unions)
#   define VULKAN_HPP_HAS_UNRESTRICTED_UNIONS
#  endif
# elif defined(__GNUC__)
#  define GCC_VERSION (__GNUC__ * 10000 + __GNUC_MINOR__ * 100 + __GNUC_PATCHLEVEL__)
#  if 40600 <= GCC_VERSION
#   define VULKAN_HPP_HAS_UNRESTRICTED_UNIONS
#  endif
# elif defined(_MSC_VER)
#  if 1900 <= _MSC_VER
#   define VULKAN_HPP_HAS_UNRESTRICTED_UNIONS
#  endif
# endif
#endif

#if !defined(VULKAN_HPP_INLINE)
# if defined(__clang___)
#  if __has_attribute(always_inline)
#   define VULKAN_HPP_INLINE __attribute__((always_inline)) __inline__
#  else
#    define VULKAN_HPP_INLINE inline
#  endif
# elif defined(__GNUC__)
#  define VULKAN_HPP_INLINE __attribute__((always_inline)) __inline__
# elif defined(_MSC_VER)
#  define VULKAN_HPP_INLINE inline
# else
#  define VULKAN_HPP_INLINE inline
# endif
#endif

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
# define VULKAN_HPP_TYPESAFE_EXPLICIT
#else
# define VULKAN_HPP_TYPESAFE_EXPLICIT explicit
#endif

#if defined(_MSC_VER) && (_MSC_VER <= 1800)
# define VULKAN_HPP_CONSTEXPR
#else
# define VULKAN_HPP_CONSTEXPR constexpr
#endif


#if !defined(VULKAN_HPP_NAMESPACE)
#define VULKAN_HPP_NAMESPACE vk
#endif

#define VULKAN_HPP_STRINGIFY2(text) #text
#define VULKAN_HPP_STRINGIFY(text) VULKAN_HPP_STRINGIFY2(text)
#define VULKAN_HPP_NAMESPACE_STRING VULKAN_HPP_STRINGIFY(VULKAN_HPP_NAMESPACE)

namespace VULKAN_HPP_NAMESPACE
{

  template <typename FlagBitsType> struct FlagTraits
  {
    enum { allFlags = 0 };
  };

  template <typename BitType, typename MaskType = VkFlags>
  class Flags
  {
  public:
    VULKAN_HPP_CONSTEXPR Flags()
      : m_mask(0)
    {
    }

    Flags(BitType bit)
      : m_mask(static_cast<MaskType>(bit))
    {
    }

    Flags(Flags<BitType> const& rhs)
      : m_mask(rhs.m_mask)
    {
    }

    explicit Flags(MaskType flags)
      : m_mask(flags)
    {
    }

    Flags<BitType> & operator=(Flags<BitType> const& rhs)
    {
      m_mask = rhs.m_mask;
      return *this;
    }

    Flags<BitType> & operator|=(Flags<BitType> const& rhs)
    {
      m_mask |= rhs.m_mask;
      return *this;
    }

    Flags<BitType> & operator&=(Flags<BitType> const& rhs)
    {
      m_mask &= rhs.m_mask;
      return *this;
    }

    Flags<BitType> & operator^=(Flags<BitType> const& rhs)
    {
      m_mask ^= rhs.m_mask;
      return *this;
    }

    Flags<BitType> operator|(Flags<BitType> const& rhs) const
    {
      Flags<BitType> result(*this);
      result |= rhs;
      return result;
    }

    Flags<BitType> operator&(Flags<BitType> const& rhs) const
    {
      Flags<BitType> result(*this);
      result &= rhs;
      return result;
    }

    Flags<BitType> operator^(Flags<BitType> const& rhs) const
    {
      Flags<BitType> result(*this);
      result ^= rhs;
      return result;
    }

    bool operator!() const
    {
      return !m_mask;
    }

    Flags<BitType> operator~() const
    {
      Flags<BitType> result(*this);
      result.m_mask ^= FlagTraits<BitType>::allFlags;
      return result;
    }

    bool operator==(Flags<BitType> const& rhs) const
    {
      return m_mask == rhs.m_mask;
    }

    bool operator!=(Flags<BitType> const& rhs) const
    {
      return m_mask != rhs.m_mask;
    }

    explicit operator bool() const
    {
      return !!m_mask;
    }

    explicit operator MaskType() const
    {
        return m_mask;
    }

  private:
    MaskType  m_mask;
  };

  template <typename BitType>
  Flags<BitType> operator|(BitType bit, Flags<BitType> const& flags)
  {
    return flags | bit;
  }

  template <typename BitType>
  Flags<BitType> operator&(BitType bit, Flags<BitType> const& flags)
  {
    return flags & bit;
  }

  template <typename BitType>
  Flags<BitType> operator^(BitType bit, Flags<BitType> const& flags)
  {
    return flags ^ bit;
  }


  template <typename RefType>
  class Optional
  {
  public:
    Optional(RefType & reference) { m_ptr = &reference; }
    Optional(RefType * ptr) { m_ptr = ptr; }
    Optional(std::nullptr_t) { m_ptr = nullptr; }

    operator RefType*() const { return m_ptr; }
    RefType const* operator->() const { return m_ptr; }
    explicit operator bool() const { return !!m_ptr; }

  private:
    RefType *m_ptr;
  };

#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
  template <typename T>
  class ArrayProxy
  {
  public:
    VULKAN_HPP_CONSTEXPR ArrayProxy(std::nullptr_t)
      : m_count(0)
      , m_ptr(nullptr)
    {}

    ArrayProxy(T & ptr)
      : m_count(1)
      , m_ptr(&ptr)
    {}

    ArrayProxy(uint32_t count, T * ptr)
      : m_count(count)
      , m_ptr(ptr)
    {}

    template <size_t N>
    ArrayProxy(std::array<typename std::remove_const<T>::type, N> & data)
      : m_count(N)
      , m_ptr(data.data())
    {}

    template <size_t N>
    ArrayProxy(std::array<typename std::remove_const<T>::type, N> const& data)
      : m_count(N)
      , m_ptr(data.data())
    {}

    template <class Allocator = std::allocator<typename std::remove_const<T>::type>>
    ArrayProxy(std::vector<typename std::remove_const<T>::type, Allocator> & data)
      : m_count(static_cast<uint32_t>(data.size()))
      , m_ptr(data.data())
    {}

    template <class Allocator = std::allocator<typename std::remove_const<T>::type>>
    ArrayProxy(std::vector<typename std::remove_const<T>::type, Allocator> const& data)
      : m_count(static_cast<uint32_t>(data.size()))
      , m_ptr(data.data())
    {}

    ArrayProxy(std::initializer_list<T> const& data)
      : m_count(static_cast<uint32_t>(data.end() - data.begin()))
      , m_ptr(data.begin())
    {}

    const T * begin() const
    {
      return m_ptr;
    }

    const T * end() const
    {
      return m_ptr + m_count;
    }

    const T & front() const
    {
      VULKAN_HPP_ASSERT(m_count && m_ptr);
      return *m_ptr;
    }

    const T & back() const
    {
      VULKAN_HPP_ASSERT(m_count && m_ptr);
      return *(m_ptr + m_count - 1);
    }

    bool empty() const
    {
      return (m_count == 0);
    }

    uint32_t size() const
    {
      return m_count;
    }

    T * data() const
    {
      return m_ptr;
    }

  private:
    uint32_t  m_count;
    T *       m_ptr;
  };
#endif

#ifndef VULKAN_HPP_NO_SMART_HANDLE

  template <typename Type, typename Dispatch> class UniqueHandleTraits;

  template <typename Type, typename Dispatch>
  class UniqueHandle : public UniqueHandleTraits<Type,Dispatch>::deleter
  {
  private:
    using Deleter = typename UniqueHandleTraits<Type,Dispatch>::deleter;
  public:
    explicit UniqueHandle( Type const& value = Type(), Deleter const& deleter = Deleter() )
      : Deleter( deleter)
      , m_value( value )
    {}

    UniqueHandle( UniqueHandle const& ) = delete;

    UniqueHandle( UniqueHandle && other )
      : Deleter( std::move( static_cast<Deleter&>( other ) ) )
      , m_value( other.release() )
    {}

    ~UniqueHandle()
    {
      if ( m_value ) this->destroy( m_value );
    }

    UniqueHandle & operator=( UniqueHandle const& ) = delete;

    UniqueHandle & operator=( UniqueHandle && other )
    {
      reset( other.release() );
      *static_cast<Deleter*>(this) = std::move( static_cast<Deleter&>(other) );
      return *this;
    }

    explicit operator bool() const
    {
      return m_value.operator bool();
    }

    Type const* operator->() const
    {
      return &m_value;
    }

    Type * operator->()
    {
      return &m_value;
    }

    Type const& operator*() const
    {
      return m_value;
    }

    Type & operator*()
    {
      return m_value;
    }

    const Type & get() const
    {
      return m_value;
    }
    
    Type & get()
    {
      return m_value;
    }

    void reset( Type const& value = Type() )
    {
      if ( m_value != value )
      {
        if ( m_value ) this->destroy( m_value );
        m_value = value;
      }
    }

    Type release()
    {
      Type value = m_value;
      m_value = nullptr;
      return value;
    }

    void swap( UniqueHandle<Type,Dispatch> & rhs )
    {
      std::swap(m_value, rhs.m_value);
      std::swap(static_cast<Deleter&>(*this), static_cast<Deleter&>(rhs));
    }

  private:
    Type    m_value;
  };

  template <typename Type, typename Dispatch>
  VULKAN_HPP_INLINE void swap( UniqueHandle<Type,Dispatch> & lhs, UniqueHandle<Type,Dispatch> & rhs )
  {
    lhs.swap( rhs );
  }
#endif



  template <typename X, typename Y> struct isStructureChainValid { enum { value = false }; };

  template <typename P, typename T>
  struct TypeList
  {
    using list = P;
    using last = T;
  };

  template <typename List, typename X>
  struct extendCheck
  {
    static const bool valid = isStructureChainValid<typename List::last, X>::value || extendCheck<typename List::list,X>::valid;
  };

  template <typename T, typename X>
  struct extendCheck<TypeList<void,T>,X>
  {
    static const bool valid = isStructureChainValid<T, X>::value;
  };

  template <typename X>
  struct extendCheck<void,X>
  {
    static const bool valid = true;
  };

  template <class Element>
  class StructureChainElement
  {
  public:
    explicit operator Element&() { return value; }
    explicit operator const Element&() const { return value; }
  private:
    Element value;
  };

  template<typename ...StructureElements>
  class StructureChain : private StructureChainElement<StructureElements>...
  {
  public:
    StructureChain()
    {
      link<void, StructureElements...>();  
    }

    StructureChain(StructureChain const &rhs)
    {
      linkAndCopy<void, StructureElements...>(rhs);
    }

    StructureChain(StructureElements const &... elems)
    {
      linkAndCopyElements<void, StructureElements...>(elems...);
    }

    StructureChain& operator=(StructureChain const &rhs)
    {
      linkAndCopy<void, StructureElements...>(rhs);
      return *this;
    }

    template<typename ClassType> ClassType& get() { return static_cast<ClassType&>(*this);}

  private:
    template<typename List, typename X>
    void link()
    {
      static_assert(extendCheck<List, X>::valid, "The structure chain is not valid!");
    }

    template<typename List, typename X, typename Y, typename ...Z>
    void link()
    {
      static_assert(extendCheck<List,X>::valid, "The structure chain is not valid!");
      X& x = static_cast<X&>(*this);
      Y& y = static_cast<Y&>(*this);
      x.pNext = &y;
      link<TypeList<List, X>, Y, Z...>();
    }

    template<typename List, typename X>
    void linkAndCopy(StructureChain const &rhs)
    {
      static_assert(extendCheck<List, X>::valid, "The structure chain is not valid!");
      static_cast<X&>(*this) = static_cast<X const &>(rhs);
    }

    template<typename List, typename X, typename Y, typename ...Z>
    void linkAndCopy(StructureChain const &rhs)
    {
      static_assert(extendCheck<List, X>::valid, "The structure chain is not valid!");
      X& x = static_cast<X&>(*this);
      Y& y = static_cast<Y&>(*this);
      x = static_cast<X const &>(rhs);
      x.pNext = &y;
      linkAndCopy<TypeList<List, X>, Y, Z...>(rhs);
    }

    template<typename List, typename X>
    void linkAndCopyElements(X const &xelem)
    {
      static_assert(extendCheck<List, X>::valid, "The structure chain is not valid!");
      static_cast<X&>(*this) = xelem;
    }

    template<typename List, typename X, typename Y, typename ...Z>
    void linkAndCopyElements(X const &xelem, Y const &yelem, Z const &... zelem)
    {
      static_assert(extendCheck<List, X>::valid, "The structure chain is not valid!");
      X& x = static_cast<X&>(*this);
      Y& y = static_cast<Y&>(*this);
      x = xelem;
      x.pNext = &y;
      linkAndCopyElements<TypeList<List, X>, Y, Z...>(yelem, zelem...);
    }
  };

  enum class Result
  {
    eSuccess = VK_SUCCESS,
    eNotReady = VK_NOT_READY,
    eTimeout = VK_TIMEOUT,
    eEventSet = VK_EVENT_SET,
    eEventReset = VK_EVENT_RESET,
    eIncomplete = VK_INCOMPLETE,
    eErrorOutOfHostMemory = VK_ERROR_OUT_OF_HOST_MEMORY,
    eErrorOutOfDeviceMemory = VK_ERROR_OUT_OF_DEVICE_MEMORY,
    eErrorInitializationFailed = VK_ERROR_INITIALIZATION_FAILED,
    eErrorDeviceLost = VK_ERROR_DEVICE_LOST,
    eErrorMemoryMapFailed = VK_ERROR_MEMORY_MAP_FAILED,
    eErrorLayerNotPresent = VK_ERROR_LAYER_NOT_PRESENT,
    eErrorExtensionNotPresent = VK_ERROR_EXTENSION_NOT_PRESENT,
    eErrorFeatureNotPresent = VK_ERROR_FEATURE_NOT_PRESENT,
    eErrorIncompatibleDriver = VK_ERROR_INCOMPATIBLE_DRIVER,
    eErrorTooManyObjects = VK_ERROR_TOO_MANY_OBJECTS,
    eErrorFormatNotSupported = VK_ERROR_FORMAT_NOT_SUPPORTED,
    eErrorFragmentedPool = VK_ERROR_FRAGMENTED_POOL,
    eErrorOutOfPoolMemory = VK_ERROR_OUT_OF_POOL_MEMORY,
    eErrorOutOfPoolMemoryKHR = VK_ERROR_OUT_OF_POOL_MEMORY,
    eErrorInvalidExternalHandle = VK_ERROR_INVALID_EXTERNAL_HANDLE,
    eErrorInvalidExternalHandleKHR = VK_ERROR_INVALID_EXTERNAL_HANDLE,
    eErrorSurfaceLostKHR = VK_ERROR_SURFACE_LOST_KHR,
    eErrorNativeWindowInUseKHR = VK_ERROR_NATIVE_WINDOW_IN_USE_KHR,
    eSuboptimalKHR = VK_SUBOPTIMAL_KHR,
    eErrorOutOfDateKHR = VK_ERROR_OUT_OF_DATE_KHR,
    eErrorIncompatibleDisplayKHR = VK_ERROR_INCOMPATIBLE_DISPLAY_KHR,
    eErrorValidationFailedEXT = VK_ERROR_VALIDATION_FAILED_EXT,
    eErrorInvalidShaderNV = VK_ERROR_INVALID_SHADER_NV,
    eErrorInvalidDrmFormatModifierPlaneLayoutEXT = VK_ERROR_INVALID_DRM_FORMAT_MODIFIER_PLANE_LAYOUT_EXT,
    eErrorFragmentationEXT = VK_ERROR_FRAGMENTATION_EXT,
    eErrorNotPermittedEXT = VK_ERROR_NOT_PERMITTED_EXT,
    eErrorInvalidDeviceAddressEXT = VK_ERROR_INVALID_DEVICE_ADDRESS_EXT
  };

  VULKAN_HPP_INLINE std::string to_string(Result value)
  {
    switch (value)
    {
    case Result::eSuccess: return "Success";
    case Result::eNotReady: return "NotReady";
    case Result::eTimeout: return "Timeout";
    case Result::eEventSet: return "EventSet";
    case Result::eEventReset: return "EventReset";
    case Result::eIncomplete: return "Incomplete";
    case Result::eErrorOutOfHostMemory: return "ErrorOutOfHostMemory";
    case Result::eErrorOutOfDeviceMemory: return "ErrorOutOfDeviceMemory";
    case Result::eErrorInitializationFailed: return "ErrorInitializationFailed";
    case Result::eErrorDeviceLost: return "ErrorDeviceLost";
    case Result::eErrorMemoryMapFailed: return "ErrorMemoryMapFailed";
    case Result::eErrorLayerNotPresent: return "ErrorLayerNotPresent";
    case Result::eErrorExtensionNotPresent: return "ErrorExtensionNotPresent";
    case Result::eErrorFeatureNotPresent: return "ErrorFeatureNotPresent";
    case Result::eErrorIncompatibleDriver: return "ErrorIncompatibleDriver";
    case Result::eErrorTooManyObjects: return "ErrorTooManyObjects";
    case Result::eErrorFormatNotSupported: return "ErrorFormatNotSupported";
    case Result::eErrorFragmentedPool: return "ErrorFragmentedPool";
    case Result::eErrorOutOfPoolMemory: return "ErrorOutOfPoolMemory";
    case Result::eErrorInvalidExternalHandle: return "ErrorInvalidExternalHandle";
    case Result::eErrorSurfaceLostKHR: return "ErrorSurfaceLostKHR";
    case Result::eErrorNativeWindowInUseKHR: return "ErrorNativeWindowInUseKHR";
    case Result::eSuboptimalKHR: return "SuboptimalKHR";
    case Result::eErrorOutOfDateKHR: return "ErrorOutOfDateKHR";
    case Result::eErrorIncompatibleDisplayKHR: return "ErrorIncompatibleDisplayKHR";
    case Result::eErrorValidationFailedEXT: return "ErrorValidationFailedEXT";
    case Result::eErrorInvalidShaderNV: return "ErrorInvalidShaderNV";
    case Result::eErrorInvalidDrmFormatModifierPlaneLayoutEXT: return "ErrorInvalidDrmFormatModifierPlaneLayoutEXT";
    case Result::eErrorFragmentationEXT: return "ErrorFragmentationEXT";
    case Result::eErrorNotPermittedEXT: return "ErrorNotPermittedEXT";
    case Result::eErrorInvalidDeviceAddressEXT: return "ErrorInvalidDeviceAddressEXT";
    default: return "invalid";
    }
  }

#ifndef VULKAN_HPP_NO_EXCEPTIONS
#if defined(_MSC_VER) && (_MSC_VER == 1800)
# define noexcept _NOEXCEPT
#endif

  class ErrorCategoryImpl : public std::error_category
  {
    public:
    virtual const char* name() const noexcept override { return VULKAN_HPP_NAMESPACE_STRING"::Result"; }
    virtual std::string message(int ev) const override { return to_string(static_cast<Result>(ev)); }
  };

#if defined(_MSC_VER) && (_MSC_VER == 1800)
# undef noexcept
#endif

  VULKAN_HPP_INLINE const std::error_category& errorCategory()
  {
    static ErrorCategoryImpl instance;
    return instance;
  }

  VULKAN_HPP_INLINE std::error_code make_error_code(Result e)
  {
    return std::error_code(static_cast<int>(e), errorCategory());
  }

  VULKAN_HPP_INLINE std::error_condition make_error_condition(Result e)
  {
    return std::error_condition(static_cast<int>(e), errorCategory());
  }

#if defined(_MSC_VER) && (_MSC_VER == 1800)
# define noexcept _NOEXCEPT
#endif

  class Error
  {
    public:
    virtual ~Error() = default;

    virtual const char* what() const noexcept = 0;
  };

  class LogicError : public Error, public std::logic_error
  {
    public:
    explicit LogicError( const std::string& what )
      : Error(), std::logic_error(what) {}
    explicit LogicError( char const * what )
      : Error(), std::logic_error(what) {}
    virtual ~LogicError() = default;

    virtual const char* what() const noexcept { return std::logic_error::what(); }
  };

  class SystemError : public Error, public std::system_error
  {
    public:
    SystemError( std::error_code ec )
      : Error(), std::system_error(ec) {}
    SystemError( std::error_code ec, std::string const& what )
      : Error(), std::system_error(ec, what) {}
    SystemError( std::error_code ec, char const * what )
      : Error(), std::system_error(ec, what) {}
    SystemError( int ev, std::error_category const& ecat )
      : Error(), std::system_error(ev, ecat) {}
    SystemError( int ev, std::error_category const& ecat, std::string const& what)
      : Error(), std::system_error(ev, ecat, what) {}
    SystemError( int ev, std::error_category const& ecat, char const * what)
      : Error(), std::system_error(ev, ecat, what) {}
    virtual ~SystemError() = default;

    virtual const char* what() const noexcept { return std::system_error::what(); }
  };

#if defined(_MSC_VER) && (_MSC_VER == 1800)
# undef noexcept
#endif

  class OutOfHostMemoryError : public SystemError
  {
  public:
    OutOfHostMemoryError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorOutOfHostMemory ), message ) {}
    OutOfHostMemoryError( char const * message )
      : SystemError( make_error_code( Result::eErrorOutOfHostMemory ), message ) {}
  };
  class OutOfDeviceMemoryError : public SystemError
  {
  public:
    OutOfDeviceMemoryError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorOutOfDeviceMemory ), message ) {}
    OutOfDeviceMemoryError( char const * message )
      : SystemError( make_error_code( Result::eErrorOutOfDeviceMemory ), message ) {}
  };
  class InitializationFailedError : public SystemError
  {
  public:
    InitializationFailedError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorInitializationFailed ), message ) {}
    InitializationFailedError( char const * message )
      : SystemError( make_error_code( Result::eErrorInitializationFailed ), message ) {}
  };
  class DeviceLostError : public SystemError
  {
  public:
    DeviceLostError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorDeviceLost ), message ) {}
    DeviceLostError( char const * message )
      : SystemError( make_error_code( Result::eErrorDeviceLost ), message ) {}
  };
  class MemoryMapFailedError : public SystemError
  {
  public:
    MemoryMapFailedError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorMemoryMapFailed ), message ) {}
    MemoryMapFailedError( char const * message )
      : SystemError( make_error_code( Result::eErrorMemoryMapFailed ), message ) {}
  };
  class LayerNotPresentError : public SystemError
  {
  public:
    LayerNotPresentError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorLayerNotPresent ), message ) {}
    LayerNotPresentError( char const * message )
      : SystemError( make_error_code( Result::eErrorLayerNotPresent ), message ) {}
  };
  class ExtensionNotPresentError : public SystemError
  {
  public:
    ExtensionNotPresentError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorExtensionNotPresent ), message ) {}
    ExtensionNotPresentError( char const * message )
      : SystemError( make_error_code( Result::eErrorExtensionNotPresent ), message ) {}
  };
  class FeatureNotPresentError : public SystemError
  {
  public:
    FeatureNotPresentError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorFeatureNotPresent ), message ) {}
    FeatureNotPresentError( char const * message )
      : SystemError( make_error_code( Result::eErrorFeatureNotPresent ), message ) {}
  };
  class IncompatibleDriverError : public SystemError
  {
  public:
    IncompatibleDriverError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorIncompatibleDriver ), message ) {}
    IncompatibleDriverError( char const * message )
      : SystemError( make_error_code( Result::eErrorIncompatibleDriver ), message ) {}
  };
  class TooManyObjectsError : public SystemError
  {
  public:
    TooManyObjectsError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorTooManyObjects ), message ) {}
    TooManyObjectsError( char const * message )
      : SystemError( make_error_code( Result::eErrorTooManyObjects ), message ) {}
  };
  class FormatNotSupportedError : public SystemError
  {
  public:
    FormatNotSupportedError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorFormatNotSupported ), message ) {}
    FormatNotSupportedError( char const * message )
      : SystemError( make_error_code( Result::eErrorFormatNotSupported ), message ) {}
  };
  class FragmentedPoolError : public SystemError
  {
  public:
    FragmentedPoolError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorFragmentedPool ), message ) {}
    FragmentedPoolError( char const * message )
      : SystemError( make_error_code( Result::eErrorFragmentedPool ), message ) {}
  };
  class OutOfPoolMemoryError : public SystemError
  {
  public:
    OutOfPoolMemoryError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorOutOfPoolMemory ), message ) {}
    OutOfPoolMemoryError( char const * message )
      : SystemError( make_error_code( Result::eErrorOutOfPoolMemory ), message ) {}
  };
  class InvalidExternalHandleError : public SystemError
  {
  public:
    InvalidExternalHandleError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorInvalidExternalHandle ), message ) {}
    InvalidExternalHandleError( char const * message )
      : SystemError( make_error_code( Result::eErrorInvalidExternalHandle ), message ) {}
  };
  class SurfaceLostKHRError : public SystemError
  {
  public:
    SurfaceLostKHRError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorSurfaceLostKHR ), message ) {}
    SurfaceLostKHRError( char const * message )
      : SystemError( make_error_code( Result::eErrorSurfaceLostKHR ), message ) {}
  };
  class NativeWindowInUseKHRError : public SystemError
  {
  public:
    NativeWindowInUseKHRError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorNativeWindowInUseKHR ), message ) {}
    NativeWindowInUseKHRError( char const * message )
      : SystemError( make_error_code( Result::eErrorNativeWindowInUseKHR ), message ) {}
  };
  class OutOfDateKHRError : public SystemError
  {
  public:
    OutOfDateKHRError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorOutOfDateKHR ), message ) {}
    OutOfDateKHRError( char const * message )
      : SystemError( make_error_code( Result::eErrorOutOfDateKHR ), message ) {}
  };
  class IncompatibleDisplayKHRError : public SystemError
  {
  public:
    IncompatibleDisplayKHRError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorIncompatibleDisplayKHR ), message ) {}
    IncompatibleDisplayKHRError( char const * message )
      : SystemError( make_error_code( Result::eErrorIncompatibleDisplayKHR ), message ) {}
  };
  class ValidationFailedEXTError : public SystemError
  {
  public:
    ValidationFailedEXTError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorValidationFailedEXT ), message ) {}
    ValidationFailedEXTError( char const * message )
      : SystemError( make_error_code( Result::eErrorValidationFailedEXT ), message ) {}
  };
  class InvalidShaderNVError : public SystemError
  {
  public:
    InvalidShaderNVError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorInvalidShaderNV ), message ) {}
    InvalidShaderNVError( char const * message )
      : SystemError( make_error_code( Result::eErrorInvalidShaderNV ), message ) {}
  };
  class InvalidDrmFormatModifierPlaneLayoutEXTError : public SystemError
  {
  public:
    InvalidDrmFormatModifierPlaneLayoutEXTError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorInvalidDrmFormatModifierPlaneLayoutEXT ), message ) {}
    InvalidDrmFormatModifierPlaneLayoutEXTError( char const * message )
      : SystemError( make_error_code( Result::eErrorInvalidDrmFormatModifierPlaneLayoutEXT ), message ) {}
  };
  class FragmentationEXTError : public SystemError
  {
  public:
    FragmentationEXTError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorFragmentationEXT ), message ) {}
    FragmentationEXTError( char const * message )
      : SystemError( make_error_code( Result::eErrorFragmentationEXT ), message ) {}
  };
  class NotPermittedEXTError : public SystemError
  {
  public:
    NotPermittedEXTError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorNotPermittedEXT ), message ) {}
    NotPermittedEXTError( char const * message )
      : SystemError( make_error_code( Result::eErrorNotPermittedEXT ), message ) {}
  };
  class InvalidDeviceAddressEXTError : public SystemError
  {
  public:
    InvalidDeviceAddressEXTError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorInvalidDeviceAddressEXT ), message ) {}
    InvalidDeviceAddressEXTError( char const * message )
      : SystemError( make_error_code( Result::eErrorInvalidDeviceAddressEXT ), message ) {}
  };

  VULKAN_HPP_INLINE void throwResultException( Result result, char const * message )
  {
    switch ( result )
    {
    case Result::eErrorOutOfHostMemory: throw OutOfHostMemoryError ( message );
    case Result::eErrorOutOfDeviceMemory: throw OutOfDeviceMemoryError ( message );
    case Result::eErrorInitializationFailed: throw InitializationFailedError ( message );
    case Result::eErrorDeviceLost: throw DeviceLostError ( message );
    case Result::eErrorMemoryMapFailed: throw MemoryMapFailedError ( message );
    case Result::eErrorLayerNotPresent: throw LayerNotPresentError ( message );
    case Result::eErrorExtensionNotPresent: throw ExtensionNotPresentError ( message );
    case Result::eErrorFeatureNotPresent: throw FeatureNotPresentError ( message );
    case Result::eErrorIncompatibleDriver: throw IncompatibleDriverError ( message );
    case Result::eErrorTooManyObjects: throw TooManyObjectsError ( message );
    case Result::eErrorFormatNotSupported: throw FormatNotSupportedError ( message );
    case Result::eErrorFragmentedPool: throw FragmentedPoolError ( message );
    case Result::eErrorOutOfPoolMemory: throw OutOfPoolMemoryError ( message );
    case Result::eErrorInvalidExternalHandle: throw InvalidExternalHandleError ( message );
    case Result::eErrorSurfaceLostKHR: throw SurfaceLostKHRError ( message );
    case Result::eErrorNativeWindowInUseKHR: throw NativeWindowInUseKHRError ( message );
    case Result::eErrorOutOfDateKHR: throw OutOfDateKHRError ( message );
    case Result::eErrorIncompatibleDisplayKHR: throw IncompatibleDisplayKHRError ( message );
    case Result::eErrorValidationFailedEXT: throw ValidationFailedEXTError ( message );
    case Result::eErrorInvalidShaderNV: throw InvalidShaderNVError ( message );
    case Result::eErrorInvalidDrmFormatModifierPlaneLayoutEXT: throw InvalidDrmFormatModifierPlaneLayoutEXTError ( message );
    case Result::eErrorFragmentationEXT: throw FragmentationEXTError ( message );
    case Result::eErrorNotPermittedEXT: throw NotPermittedEXTError ( message );
    case Result::eErrorInvalidDeviceAddressEXT: throw InvalidDeviceAddressEXTError ( message );
    default: throw SystemError( make_error_code( result ) );
    }
  }
#endif
} // namespace VULKAN_HPP_NAMESPACE

namespace std
{
  template <>
  struct is_error_code_enum<VULKAN_HPP_NAMESPACE::Result> : public true_type
  {};
}

namespace VULKAN_HPP_NAMESPACE
{

  template <typename T>
  struct ResultValue
  {
    ResultValue( Result r, T & v )
      : result( r )
      , value( v )
    {}

    ResultValue( Result r, T && v )
      : result( r )
      , value( std::move( v ) )
    {}

    Result  result;
    T       value;

    operator std::tuple<Result&, T&>() { return std::tuple<Result&, T&>(result, value); }
  };

  template <typename T>
  struct ResultValueType
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    typedef ResultValue<T>  type;
#else
    typedef T               type;
#endif
  };

  template <>
  struct ResultValueType<void>
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    typedef Result type;
#else
    typedef void   type;
#endif
  };

  template <typename T>
  VULKAN_HPP_INLINE void ignore(T const&) {}

  VULKAN_HPP_INLINE ResultValueType<void>::type createResultValue( Result result, char const * message )
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    ignore( message );
    VULKAN_HPP_ASSERT( result == Result::eSuccess );
    return result;
#else
    if ( result != Result::eSuccess )
    {
      throwResultException( result, message );
    }
#endif
  }

  template <typename T>
  VULKAN_HPP_INLINE typename ResultValueType<T>::type createResultValue( Result result, T & data, char const * message )
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    ignore( message );
    VULKAN_HPP_ASSERT( result == Result::eSuccess );
    return ResultValue<T>( result, std::move( data ) );
#else
    if ( result != Result::eSuccess )
    {
      throwResultException( result, message );
    }
    return std::move( data );
#endif
  }

  VULKAN_HPP_INLINE Result createResultValue( Result result, char const * message, std::initializer_list<Result> successCodes )
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    ignore( message );
    VULKAN_HPP_ASSERT( std::find( successCodes.begin(), successCodes.end(), result ) != successCodes.end() );
#else
    if ( std::find( successCodes.begin(), successCodes.end(), result ) == successCodes.end() )
    {
      throwResultException( result, message );
    }
#endif
    return result;
  }

  template <typename T>
  VULKAN_HPP_INLINE ResultValue<T> createResultValue( Result result, T & data, char const * message, std::initializer_list<Result> successCodes )
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    ignore( message );
    VULKAN_HPP_ASSERT( std::find( successCodes.begin(), successCodes.end(), result ) != successCodes.end() );
#else
    if ( std::find( successCodes.begin(), successCodes.end(), result ) == successCodes.end() )
    {
      throwResultException( result, message );
    }
#endif
    return ResultValue<T>( result, data );
  }

#ifndef VULKAN_HPP_NO_SMART_HANDLE
  template <typename T, typename D>
  VULKAN_HPP_INLINE typename ResultValueType<UniqueHandle<T,D>>::type createResultValue( Result result, T & data, char const * message, typename UniqueHandleTraits<T,D>::deleter const& deleter )
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    ignore( message );
    VULKAN_HPP_ASSERT( result == Result::eSuccess );
    return ResultValue<UniqueHandle<T,D>>( result, UniqueHandle<T,D>(std::move( data ), deleter) );
#else
    if ( result != Result::eSuccess )
    {
      throwResultException( result, message );
    }
    return UniqueHandle<T,D>(data, deleter);
#endif
  }
#endif

class DispatchLoaderStatic
{
public:
  VkResult vkAcquireNextImage2KHR( VkDevice device, const VkAcquireNextImageInfoKHR* pAcquireInfo, uint32_t* pImageIndex  ) const
  {
    return ::vkAcquireNextImage2KHR( device, pAcquireInfo, pImageIndex);
  }
  VkResult vkAcquireNextImageKHR( VkDevice device, VkSwapchainKHR swapchain, uint64_t timeout, VkSemaphore semaphore, VkFence fence, uint32_t* pImageIndex  ) const
  {
    return ::vkAcquireNextImageKHR( device, swapchain, timeout, semaphore, fence, pImageIndex);
  }
#ifdef VK_USE_PLATFORM_XLIB_XRANDR_EXT
  VkResult vkAcquireXlibDisplayEXT( VkPhysicalDevice physicalDevice, Display* dpy, VkDisplayKHR display  ) const
  {
    return ::vkAcquireXlibDisplayEXT( physicalDevice, dpy, display);
  }
#endif /*VK_USE_PLATFORM_XLIB_XRANDR_EXT*/
  VkResult vkAllocateCommandBuffers( VkDevice device, const VkCommandBufferAllocateInfo* pAllocateInfo, VkCommandBuffer* pCommandBuffers  ) const
  {
    return ::vkAllocateCommandBuffers( device, pAllocateInfo, pCommandBuffers);
  }
  VkResult vkAllocateDescriptorSets( VkDevice device, const VkDescriptorSetAllocateInfo* pAllocateInfo, VkDescriptorSet* pDescriptorSets  ) const
  {
    return ::vkAllocateDescriptorSets( device, pAllocateInfo, pDescriptorSets);
  }
  VkResult vkAllocateMemory( VkDevice device, const VkMemoryAllocateInfo* pAllocateInfo, const VkAllocationCallbacks* pAllocator, VkDeviceMemory* pMemory  ) const
  {
    return ::vkAllocateMemory( device, pAllocateInfo, pAllocator, pMemory);
  }
  VkResult vkBeginCommandBuffer( VkCommandBuffer commandBuffer, const VkCommandBufferBeginInfo* pBeginInfo  ) const
  {
    return ::vkBeginCommandBuffer( commandBuffer, pBeginInfo);
  }
  VkResult vkBindAccelerationStructureMemoryNV( VkDevice device, uint32_t bindInfoCount, const VkBindAccelerationStructureMemoryInfoNV* pBindInfos  ) const
  {
    return ::vkBindAccelerationStructureMemoryNV( device, bindInfoCount, pBindInfos);
  }
  VkResult vkBindBufferMemory( VkDevice device, VkBuffer buffer, VkDeviceMemory memory, VkDeviceSize memoryOffset  ) const
  {
    return ::vkBindBufferMemory( device, buffer, memory, memoryOffset);
  }
  VkResult vkBindBufferMemory2( VkDevice device, uint32_t bindInfoCount, const VkBindBufferMemoryInfo* pBindInfos  ) const
  {
    return ::vkBindBufferMemory2( device, bindInfoCount, pBindInfos);
  }
  VkResult vkBindBufferMemory2KHR( VkDevice device, uint32_t bindInfoCount, const VkBindBufferMemoryInfo* pBindInfos  ) const
  {
    return ::vkBindBufferMemory2KHR( device, bindInfoCount, pBindInfos);
  }
  VkResult vkBindImageMemory( VkDevice device, VkImage image, VkDeviceMemory memory, VkDeviceSize memoryOffset  ) const
  {
    return ::vkBindImageMemory( device, image, memory, memoryOffset);
  }
  VkResult vkBindImageMemory2( VkDevice device, uint32_t bindInfoCount, const VkBindImageMemoryInfo* pBindInfos  ) const
  {
    return ::vkBindImageMemory2( device, bindInfoCount, pBindInfos);
  }
  VkResult vkBindImageMemory2KHR( VkDevice device, uint32_t bindInfoCount, const VkBindImageMemoryInfo* pBindInfos  ) const
  {
    return ::vkBindImageMemory2KHR( device, bindInfoCount, pBindInfos);
  }
  void vkCmdBeginConditionalRenderingEXT( VkCommandBuffer commandBuffer, const VkConditionalRenderingBeginInfoEXT* pConditionalRenderingBegin  ) const
  {
    return ::vkCmdBeginConditionalRenderingEXT( commandBuffer, pConditionalRenderingBegin);
  }
  void vkCmdBeginDebugUtilsLabelEXT( VkCommandBuffer commandBuffer, const VkDebugUtilsLabelEXT* pLabelInfo  ) const
  {
    return ::vkCmdBeginDebugUtilsLabelEXT( commandBuffer, pLabelInfo);
  }
  void vkCmdBeginQuery( VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t query, VkQueryControlFlags flags  ) const
  {
    return ::vkCmdBeginQuery( commandBuffer, queryPool, query, flags);
  }
  void vkCmdBeginQueryIndexedEXT( VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t query, VkQueryControlFlags flags, uint32_t index  ) const
  {
    return ::vkCmdBeginQueryIndexedEXT( commandBuffer, queryPool, query, flags, index);
  }
  void vkCmdBeginRenderPass( VkCommandBuffer commandBuffer, const VkRenderPassBeginInfo* pRenderPassBegin, VkSubpassContents contents  ) const
  {
    return ::vkCmdBeginRenderPass( commandBuffer, pRenderPassBegin, contents);
  }
  void vkCmdBeginRenderPass2KHR( VkCommandBuffer commandBuffer, const VkRenderPassBeginInfo* pRenderPassBegin, const VkSubpassBeginInfoKHR* pSubpassBeginInfo  ) const
  {
    return ::vkCmdBeginRenderPass2KHR( commandBuffer, pRenderPassBegin, pSubpassBeginInfo);
  }
  void vkCmdBeginTransformFeedbackEXT( VkCommandBuffer commandBuffer, uint32_t firstCounterBuffer, uint32_t counterBufferCount, const VkBuffer* pCounterBuffers, const VkDeviceSize* pCounterBufferOffsets  ) const
  {
    return ::vkCmdBeginTransformFeedbackEXT( commandBuffer, firstCounterBuffer, counterBufferCount, pCounterBuffers, pCounterBufferOffsets);
  }
  void vkCmdBindDescriptorSets( VkCommandBuffer commandBuffer, VkPipelineBindPoint pipelineBindPoint, VkPipelineLayout layout, uint32_t firstSet, uint32_t descriptorSetCount, const VkDescriptorSet* pDescriptorSets, uint32_t dynamicOffsetCount, const uint32_t* pDynamicOffsets  ) const
  {
    return ::vkCmdBindDescriptorSets( commandBuffer, pipelineBindPoint, layout, firstSet, descriptorSetCount, pDescriptorSets, dynamicOffsetCount, pDynamicOffsets);
  }
  void vkCmdBindIndexBuffer( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, VkIndexType indexType  ) const
  {
    return ::vkCmdBindIndexBuffer( commandBuffer, buffer, offset, indexType);
  }
  void vkCmdBindPipeline( VkCommandBuffer commandBuffer, VkPipelineBindPoint pipelineBindPoint, VkPipeline pipeline  ) const
  {
    return ::vkCmdBindPipeline( commandBuffer, pipelineBindPoint, pipeline);
  }
  void vkCmdBindShadingRateImageNV( VkCommandBuffer commandBuffer, VkImageView imageView, VkImageLayout imageLayout  ) const
  {
    return ::vkCmdBindShadingRateImageNV( commandBuffer, imageView, imageLayout);
  }
  void vkCmdBindTransformFeedbackBuffersEXT( VkCommandBuffer commandBuffer, uint32_t firstBinding, uint32_t bindingCount, const VkBuffer* pBuffers, const VkDeviceSize* pOffsets, const VkDeviceSize* pSizes  ) const
  {
    return ::vkCmdBindTransformFeedbackBuffersEXT( commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes);
  }
  void vkCmdBindVertexBuffers( VkCommandBuffer commandBuffer, uint32_t firstBinding, uint32_t bindingCount, const VkBuffer* pBuffers, const VkDeviceSize* pOffsets  ) const
  {
    return ::vkCmdBindVertexBuffers( commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets);
  }
  void vkCmdBlitImage( VkCommandBuffer commandBuffer, VkImage srcImage, VkImageLayout srcImageLayout, VkImage dstImage, VkImageLayout dstImageLayout, uint32_t regionCount, const VkImageBlit* pRegions, VkFilter filter  ) const
  {
    return ::vkCmdBlitImage( commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions, filter);
  }
  void vkCmdBuildAccelerationStructureNV( VkCommandBuffer commandBuffer, const VkAccelerationStructureInfoNV* pInfo, VkBuffer instanceData, VkDeviceSize instanceOffset, VkBool32 update, VkAccelerationStructureNV dst, VkAccelerationStructureNV src, VkBuffer scratch, VkDeviceSize scratchOffset  ) const
  {
    return ::vkCmdBuildAccelerationStructureNV( commandBuffer, pInfo, instanceData, instanceOffset, update, dst, src, scratch, scratchOffset);
  }
  void vkCmdClearAttachments( VkCommandBuffer commandBuffer, uint32_t attachmentCount, const VkClearAttachment* pAttachments, uint32_t rectCount, const VkClearRect* pRects  ) const
  {
    return ::vkCmdClearAttachments( commandBuffer, attachmentCount, pAttachments, rectCount, pRects);
  }
  void vkCmdClearColorImage( VkCommandBuffer commandBuffer, VkImage image, VkImageLayout imageLayout, const VkClearColorValue* pColor, uint32_t rangeCount, const VkImageSubresourceRange* pRanges  ) const
  {
    return ::vkCmdClearColorImage( commandBuffer, image, imageLayout, pColor, rangeCount, pRanges);
  }
  void vkCmdClearDepthStencilImage( VkCommandBuffer commandBuffer, VkImage image, VkImageLayout imageLayout, const VkClearDepthStencilValue* pDepthStencil, uint32_t rangeCount, const VkImageSubresourceRange* pRanges  ) const
  {
    return ::vkCmdClearDepthStencilImage( commandBuffer, image, imageLayout, pDepthStencil, rangeCount, pRanges);
  }
  void vkCmdCopyAccelerationStructureNV( VkCommandBuffer commandBuffer, VkAccelerationStructureNV dst, VkAccelerationStructureNV src, VkCopyAccelerationStructureModeNV mode  ) const
  {
    return ::vkCmdCopyAccelerationStructureNV( commandBuffer, dst, src, mode);
  }
  void vkCmdCopyBuffer( VkCommandBuffer commandBuffer, VkBuffer srcBuffer, VkBuffer dstBuffer, uint32_t regionCount, const VkBufferCopy* pRegions  ) const
  {
    return ::vkCmdCopyBuffer( commandBuffer, srcBuffer, dstBuffer, regionCount, pRegions);
  }
  void vkCmdCopyBufferToImage( VkCommandBuffer commandBuffer, VkBuffer srcBuffer, VkImage dstImage, VkImageLayout dstImageLayout, uint32_t regionCount, const VkBufferImageCopy* pRegions  ) const
  {
    return ::vkCmdCopyBufferToImage( commandBuffer, srcBuffer, dstImage, dstImageLayout, regionCount, pRegions);
  }
  void vkCmdCopyImage( VkCommandBuffer commandBuffer, VkImage srcImage, VkImageLayout srcImageLayout, VkImage dstImage, VkImageLayout dstImageLayout, uint32_t regionCount, const VkImageCopy* pRegions  ) const
  {
    return ::vkCmdCopyImage( commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions);
  }
  void vkCmdCopyImageToBuffer( VkCommandBuffer commandBuffer, VkImage srcImage, VkImageLayout srcImageLayout, VkBuffer dstBuffer, uint32_t regionCount, const VkBufferImageCopy* pRegions  ) const
  {
    return ::vkCmdCopyImageToBuffer( commandBuffer, srcImage, srcImageLayout, dstBuffer, regionCount, pRegions);
  }
  void vkCmdCopyQueryPoolResults( VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t firstQuery, uint32_t queryCount, VkBuffer dstBuffer, VkDeviceSize dstOffset, VkDeviceSize stride, VkQueryResultFlags flags  ) const
  {
    return ::vkCmdCopyQueryPoolResults( commandBuffer, queryPool, firstQuery, queryCount, dstBuffer, dstOffset, stride, flags);
  }
  void vkCmdDebugMarkerBeginEXT( VkCommandBuffer commandBuffer, const VkDebugMarkerMarkerInfoEXT* pMarkerInfo  ) const
  {
    return ::vkCmdDebugMarkerBeginEXT( commandBuffer, pMarkerInfo);
  }
  void vkCmdDebugMarkerEndEXT( VkCommandBuffer commandBuffer  ) const
  {
    return ::vkCmdDebugMarkerEndEXT( commandBuffer);
  }
  void vkCmdDebugMarkerInsertEXT( VkCommandBuffer commandBuffer, const VkDebugMarkerMarkerInfoEXT* pMarkerInfo  ) const
  {
    return ::vkCmdDebugMarkerInsertEXT( commandBuffer, pMarkerInfo);
  }
  void vkCmdDispatch( VkCommandBuffer commandBuffer, uint32_t groupCountX, uint32_t groupCountY, uint32_t groupCountZ  ) const
  {
    return ::vkCmdDispatch( commandBuffer, groupCountX, groupCountY, groupCountZ);
  }
  void vkCmdDispatchBase( VkCommandBuffer commandBuffer, uint32_t baseGroupX, uint32_t baseGroupY, uint32_t baseGroupZ, uint32_t groupCountX, uint32_t groupCountY, uint32_t groupCountZ  ) const
  {
    return ::vkCmdDispatchBase( commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY, groupCountZ);
  }
  void vkCmdDispatchBaseKHR( VkCommandBuffer commandBuffer, uint32_t baseGroupX, uint32_t baseGroupY, uint32_t baseGroupZ, uint32_t groupCountX, uint32_t groupCountY, uint32_t groupCountZ  ) const
  {
    return ::vkCmdDispatchBaseKHR( commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY, groupCountZ);
  }
  void vkCmdDispatchIndirect( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset  ) const
  {
    return ::vkCmdDispatchIndirect( commandBuffer, buffer, offset);
  }
  void vkCmdDraw( VkCommandBuffer commandBuffer, uint32_t vertexCount, uint32_t instanceCount, uint32_t firstVertex, uint32_t firstInstance  ) const
  {
    return ::vkCmdDraw( commandBuffer, vertexCount, instanceCount, firstVertex, firstInstance);
  }
  void vkCmdDrawIndexed( VkCommandBuffer commandBuffer, uint32_t indexCount, uint32_t instanceCount, uint32_t firstIndex, int32_t vertexOffset, uint32_t firstInstance  ) const
  {
    return ::vkCmdDrawIndexed( commandBuffer, indexCount, instanceCount, firstIndex, vertexOffset, firstInstance);
  }
  void vkCmdDrawIndexedIndirect( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, uint32_t drawCount, uint32_t stride  ) const
  {
    return ::vkCmdDrawIndexedIndirect( commandBuffer, buffer, offset, drawCount, stride);
  }
  void vkCmdDrawIndexedIndirectCountAMD( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride  ) const
  {
    return ::vkCmdDrawIndexedIndirectCountAMD( commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
  }
  void vkCmdDrawIndexedIndirectCountKHR( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride  ) const
  {
    return ::vkCmdDrawIndexedIndirectCountKHR( commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
  }
  void vkCmdDrawIndirect( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, uint32_t drawCount, uint32_t stride  ) const
  {
    return ::vkCmdDrawIndirect( commandBuffer, buffer, offset, drawCount, stride);
  }
  void vkCmdDrawIndirectByteCountEXT( VkCommandBuffer commandBuffer, uint32_t instanceCount, uint32_t firstInstance, VkBuffer counterBuffer, VkDeviceSize counterBufferOffset, uint32_t counterOffset, uint32_t vertexStride  ) const
  {
    return ::vkCmdDrawIndirectByteCountEXT( commandBuffer, instanceCount, firstInstance, counterBuffer, counterBufferOffset, counterOffset, vertexStride);
  }
  void vkCmdDrawIndirectCountAMD( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride  ) const
  {
    return ::vkCmdDrawIndirectCountAMD( commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
  }
  void vkCmdDrawIndirectCountKHR( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride  ) const
  {
    return ::vkCmdDrawIndirectCountKHR( commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
  }
  void vkCmdDrawMeshTasksIndirectCountNV( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride  ) const
  {
    return ::vkCmdDrawMeshTasksIndirectCountNV( commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
  }
  void vkCmdDrawMeshTasksIndirectNV( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, uint32_t drawCount, uint32_t stride  ) const
  {
    return ::vkCmdDrawMeshTasksIndirectNV( commandBuffer, buffer, offset, drawCount, stride);
  }
  void vkCmdDrawMeshTasksNV( VkCommandBuffer commandBuffer, uint32_t taskCount, uint32_t firstTask  ) const
  {
    return ::vkCmdDrawMeshTasksNV( commandBuffer, taskCount, firstTask);
  }
  void vkCmdEndConditionalRenderingEXT( VkCommandBuffer commandBuffer  ) const
  {
    return ::vkCmdEndConditionalRenderingEXT( commandBuffer);
  }
  void vkCmdEndDebugUtilsLabelEXT( VkCommandBuffer commandBuffer  ) const
  {
    return ::vkCmdEndDebugUtilsLabelEXT( commandBuffer);
  }
  void vkCmdEndQuery( VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t query  ) const
  {
    return ::vkCmdEndQuery( commandBuffer, queryPool, query);
  }
  void vkCmdEndQueryIndexedEXT( VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t query, uint32_t index  ) const
  {
    return ::vkCmdEndQueryIndexedEXT( commandBuffer, queryPool, query, index);
  }
  void vkCmdEndRenderPass( VkCommandBuffer commandBuffer  ) const
  {
    return ::vkCmdEndRenderPass( commandBuffer);
  }
  void vkCmdEndRenderPass2KHR( VkCommandBuffer commandBuffer, const VkSubpassEndInfoKHR* pSubpassEndInfo  ) const
  {
    return ::vkCmdEndRenderPass2KHR( commandBuffer, pSubpassEndInfo);
  }
  void vkCmdEndTransformFeedbackEXT( VkCommandBuffer commandBuffer, uint32_t firstCounterBuffer, uint32_t counterBufferCount, const VkBuffer* pCounterBuffers, const VkDeviceSize* pCounterBufferOffsets  ) const
  {
    return ::vkCmdEndTransformFeedbackEXT( commandBuffer, firstCounterBuffer, counterBufferCount, pCounterBuffers, pCounterBufferOffsets);
  }
  void vkCmdExecuteCommands( VkCommandBuffer commandBuffer, uint32_t commandBufferCount, const VkCommandBuffer* pCommandBuffers  ) const
  {
    return ::vkCmdExecuteCommands( commandBuffer, commandBufferCount, pCommandBuffers);
  }
  void vkCmdFillBuffer( VkCommandBuffer commandBuffer, VkBuffer dstBuffer, VkDeviceSize dstOffset, VkDeviceSize size, uint32_t data  ) const
  {
    return ::vkCmdFillBuffer( commandBuffer, dstBuffer, dstOffset, size, data);
  }
  void vkCmdInsertDebugUtilsLabelEXT( VkCommandBuffer commandBuffer, const VkDebugUtilsLabelEXT* pLabelInfo  ) const
  {
    return ::vkCmdInsertDebugUtilsLabelEXT( commandBuffer, pLabelInfo);
  }
  void vkCmdNextSubpass( VkCommandBuffer commandBuffer, VkSubpassContents contents  ) const
  {
    return ::vkCmdNextSubpass( commandBuffer, contents);
  }
  void vkCmdNextSubpass2KHR( VkCommandBuffer commandBuffer, const VkSubpassBeginInfoKHR* pSubpassBeginInfo, const VkSubpassEndInfoKHR* pSubpassEndInfo  ) const
  {
    return ::vkCmdNextSubpass2KHR( commandBuffer, pSubpassBeginInfo, pSubpassEndInfo);
  }
  void vkCmdPipelineBarrier( VkCommandBuffer commandBuffer, VkPipelineStageFlags srcStageMask, VkPipelineStageFlags dstStageMask, VkDependencyFlags dependencyFlags, uint32_t memoryBarrierCount, const VkMemoryBarrier* pMemoryBarriers, uint32_t bufferMemoryBarrierCount, const VkBufferMemoryBarrier* pBufferMemoryBarriers, uint32_t imageMemoryBarrierCount, const VkImageMemoryBarrier* pImageMemoryBarriers  ) const
  {
    return ::vkCmdPipelineBarrier( commandBuffer, srcStageMask, dstStageMask, dependencyFlags, memoryBarrierCount, pMemoryBarriers, bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers);
  }
  void vkCmdProcessCommandsNVX( VkCommandBuffer commandBuffer, const VkCmdProcessCommandsInfoNVX* pProcessCommandsInfo  ) const
  {
    return ::vkCmdProcessCommandsNVX( commandBuffer, pProcessCommandsInfo);
  }
  void vkCmdPushConstants( VkCommandBuffer commandBuffer, VkPipelineLayout layout, VkShaderStageFlags stageFlags, uint32_t offset, uint32_t size, const void* pValues  ) const
  {
    return ::vkCmdPushConstants( commandBuffer, layout, stageFlags, offset, size, pValues);
  }
  void vkCmdPushDescriptorSetKHR( VkCommandBuffer commandBuffer, VkPipelineBindPoint pipelineBindPoint, VkPipelineLayout layout, uint32_t set, uint32_t descriptorWriteCount, const VkWriteDescriptorSet* pDescriptorWrites  ) const
  {
    return ::vkCmdPushDescriptorSetKHR( commandBuffer, pipelineBindPoint, layout, set, descriptorWriteCount, pDescriptorWrites);
  }
  void vkCmdPushDescriptorSetWithTemplateKHR( VkCommandBuffer commandBuffer, VkDescriptorUpdateTemplate descriptorUpdateTemplate, VkPipelineLayout layout, uint32_t set, const void* pData  ) const
  {
    return ::vkCmdPushDescriptorSetWithTemplateKHR( commandBuffer, descriptorUpdateTemplate, layout, set, pData);
  }
  void vkCmdReserveSpaceForCommandsNVX( VkCommandBuffer commandBuffer, const VkCmdReserveSpaceForCommandsInfoNVX* pReserveSpaceInfo  ) const
  {
    return ::vkCmdReserveSpaceForCommandsNVX( commandBuffer, pReserveSpaceInfo);
  }
  void vkCmdResetEvent( VkCommandBuffer commandBuffer, VkEvent event, VkPipelineStageFlags stageMask  ) const
  {
    return ::vkCmdResetEvent( commandBuffer, event, stageMask);
  }
  void vkCmdResetQueryPool( VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t firstQuery, uint32_t queryCount  ) const
  {
    return ::vkCmdResetQueryPool( commandBuffer, queryPool, firstQuery, queryCount);
  }
  void vkCmdResolveImage( VkCommandBuffer commandBuffer, VkImage srcImage, VkImageLayout srcImageLayout, VkImage dstImage, VkImageLayout dstImageLayout, uint32_t regionCount, const VkImageResolve* pRegions  ) const
  {
    return ::vkCmdResolveImage( commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions);
  }
  void vkCmdSetBlendConstants( VkCommandBuffer commandBuffer, const float blendConstants[4]  ) const
  {
    return ::vkCmdSetBlendConstants( commandBuffer, blendConstants);
  }
  void vkCmdSetCheckpointNV( VkCommandBuffer commandBuffer, const void* pCheckpointMarker  ) const
  {
    return ::vkCmdSetCheckpointNV( commandBuffer, pCheckpointMarker);
  }
  void vkCmdSetCoarseSampleOrderNV( VkCommandBuffer commandBuffer, VkCoarseSampleOrderTypeNV sampleOrderType, uint32_t customSampleOrderCount, const VkCoarseSampleOrderCustomNV* pCustomSampleOrders  ) const
  {
    return ::vkCmdSetCoarseSampleOrderNV( commandBuffer, sampleOrderType, customSampleOrderCount, pCustomSampleOrders);
  }
  void vkCmdSetDepthBias( VkCommandBuffer commandBuffer, float depthBiasConstantFactor, float depthBiasClamp, float depthBiasSlopeFactor  ) const
  {
    return ::vkCmdSetDepthBias( commandBuffer, depthBiasConstantFactor, depthBiasClamp, depthBiasSlopeFactor);
  }
  void vkCmdSetDepthBounds( VkCommandBuffer commandBuffer, float minDepthBounds, float maxDepthBounds  ) const
  {
    return ::vkCmdSetDepthBounds( commandBuffer, minDepthBounds, maxDepthBounds);
  }
  void vkCmdSetDeviceMask( VkCommandBuffer commandBuffer, uint32_t deviceMask  ) const
  {
    return ::vkCmdSetDeviceMask( commandBuffer, deviceMask);
  }
  void vkCmdSetDeviceMaskKHR( VkCommandBuffer commandBuffer, uint32_t deviceMask  ) const
  {
    return ::vkCmdSetDeviceMaskKHR( commandBuffer, deviceMask);
  }
  void vkCmdSetDiscardRectangleEXT( VkCommandBuffer commandBuffer, uint32_t firstDiscardRectangle, uint32_t discardRectangleCount, const VkRect2D* pDiscardRectangles  ) const
  {
    return ::vkCmdSetDiscardRectangleEXT( commandBuffer, firstDiscardRectangle, discardRectangleCount, pDiscardRectangles);
  }
  void vkCmdSetEvent( VkCommandBuffer commandBuffer, VkEvent event, VkPipelineStageFlags stageMask  ) const
  {
    return ::vkCmdSetEvent( commandBuffer, event, stageMask);
  }
  void vkCmdSetExclusiveScissorNV( VkCommandBuffer commandBuffer, uint32_t firstExclusiveScissor, uint32_t exclusiveScissorCount, const VkRect2D* pExclusiveScissors  ) const
  {
    return ::vkCmdSetExclusiveScissorNV( commandBuffer, firstExclusiveScissor, exclusiveScissorCount, pExclusiveScissors);
  }
  void vkCmdSetLineWidth( VkCommandBuffer commandBuffer, float lineWidth  ) const
  {
    return ::vkCmdSetLineWidth( commandBuffer, lineWidth);
  }
  void vkCmdSetSampleLocationsEXT( VkCommandBuffer commandBuffer, const VkSampleLocationsInfoEXT* pSampleLocationsInfo  ) const
  {
    return ::vkCmdSetSampleLocationsEXT( commandBuffer, pSampleLocationsInfo);
  }
  void vkCmdSetScissor( VkCommandBuffer commandBuffer, uint32_t firstScissor, uint32_t scissorCount, const VkRect2D* pScissors  ) const
  {
    return ::vkCmdSetScissor( commandBuffer, firstScissor, scissorCount, pScissors);
  }
  void vkCmdSetStencilCompareMask( VkCommandBuffer commandBuffer, VkStencilFaceFlags faceMask, uint32_t compareMask  ) const
  {
    return ::vkCmdSetStencilCompareMask( commandBuffer, faceMask, compareMask);
  }
  void vkCmdSetStencilReference( VkCommandBuffer commandBuffer, VkStencilFaceFlags faceMask, uint32_t reference  ) const
  {
    return ::vkCmdSetStencilReference( commandBuffer, faceMask, reference);
  }
  void vkCmdSetStencilWriteMask( VkCommandBuffer commandBuffer, VkStencilFaceFlags faceMask, uint32_t writeMask  ) const
  {
    return ::vkCmdSetStencilWriteMask( commandBuffer, faceMask, writeMask);
  }
  void vkCmdSetViewport( VkCommandBuffer commandBuffer, uint32_t firstViewport, uint32_t viewportCount, const VkViewport* pViewports  ) const
  {
    return ::vkCmdSetViewport( commandBuffer, firstViewport, viewportCount, pViewports);
  }
  void vkCmdSetViewportShadingRatePaletteNV( VkCommandBuffer commandBuffer, uint32_t firstViewport, uint32_t viewportCount, const VkShadingRatePaletteNV* pShadingRatePalettes  ) const
  {
    return ::vkCmdSetViewportShadingRatePaletteNV( commandBuffer, firstViewport, viewportCount, pShadingRatePalettes);
  }
  void vkCmdSetViewportWScalingNV( VkCommandBuffer commandBuffer, uint32_t firstViewport, uint32_t viewportCount, const VkViewportWScalingNV* pViewportWScalings  ) const
  {
    return ::vkCmdSetViewportWScalingNV( commandBuffer, firstViewport, viewportCount, pViewportWScalings);
  }
  void vkCmdTraceRaysNV( VkCommandBuffer commandBuffer, VkBuffer raygenShaderBindingTableBuffer, VkDeviceSize raygenShaderBindingOffset, VkBuffer missShaderBindingTableBuffer, VkDeviceSize missShaderBindingOffset, VkDeviceSize missShaderBindingStride, VkBuffer hitShaderBindingTableBuffer, VkDeviceSize hitShaderBindingOffset, VkDeviceSize hitShaderBindingStride, VkBuffer callableShaderBindingTableBuffer, VkDeviceSize callableShaderBindingOffset, VkDeviceSize callableShaderBindingStride, uint32_t width, uint32_t height, uint32_t depth  ) const
  {
    return ::vkCmdTraceRaysNV( commandBuffer, raygenShaderBindingTableBuffer, raygenShaderBindingOffset, missShaderBindingTableBuffer, missShaderBindingOffset, missShaderBindingStride, hitShaderBindingTableBuffer, hitShaderBindingOffset, hitShaderBindingStride, callableShaderBindingTableBuffer, callableShaderBindingOffset, callableShaderBindingStride, width, height, depth);
  }
  void vkCmdUpdateBuffer( VkCommandBuffer commandBuffer, VkBuffer dstBuffer, VkDeviceSize dstOffset, VkDeviceSize dataSize, const void* pData  ) const
  {
    return ::vkCmdUpdateBuffer( commandBuffer, dstBuffer, dstOffset, dataSize, pData);
  }
  void vkCmdWaitEvents( VkCommandBuffer commandBuffer, uint32_t eventCount, const VkEvent* pEvents, VkPipelineStageFlags srcStageMask, VkPipelineStageFlags dstStageMask, uint32_t memoryBarrierCount, const VkMemoryBarrier* pMemoryBarriers, uint32_t bufferMemoryBarrierCount, const VkBufferMemoryBarrier* pBufferMemoryBarriers, uint32_t imageMemoryBarrierCount, const VkImageMemoryBarrier* pImageMemoryBarriers  ) const
  {
    return ::vkCmdWaitEvents( commandBuffer, eventCount, pEvents, srcStageMask, dstStageMask, memoryBarrierCount, pMemoryBarriers, bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers);
  }
  void vkCmdWriteAccelerationStructuresPropertiesNV( VkCommandBuffer commandBuffer, uint32_t accelerationStructureCount, const VkAccelerationStructureNV* pAccelerationStructures, VkQueryType queryType, VkQueryPool queryPool, uint32_t firstQuery  ) const
  {
    return ::vkCmdWriteAccelerationStructuresPropertiesNV( commandBuffer, accelerationStructureCount, pAccelerationStructures, queryType, queryPool, firstQuery);
  }
  void vkCmdWriteBufferMarkerAMD( VkCommandBuffer commandBuffer, VkPipelineStageFlagBits pipelineStage, VkBuffer dstBuffer, VkDeviceSize dstOffset, uint32_t marker  ) const
  {
    return ::vkCmdWriteBufferMarkerAMD( commandBuffer, pipelineStage, dstBuffer, dstOffset, marker);
  }
  void vkCmdWriteTimestamp( VkCommandBuffer commandBuffer, VkPipelineStageFlagBits pipelineStage, VkQueryPool queryPool, uint32_t query  ) const
  {
    return ::vkCmdWriteTimestamp( commandBuffer, pipelineStage, queryPool, query);
  }
  VkResult vkCompileDeferredNV( VkDevice device, VkPipeline pipeline, uint32_t shader  ) const
  {
    return ::vkCompileDeferredNV( device, pipeline, shader);
  }
  VkResult vkCreateAccelerationStructureNV( VkDevice device, const VkAccelerationStructureCreateInfoNV* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkAccelerationStructureNV* pAccelerationStructure  ) const
  {
    return ::vkCreateAccelerationStructureNV( device, pCreateInfo, pAllocator, pAccelerationStructure);
  }
#ifdef VK_USE_PLATFORM_ANDROID_KHR
  VkResult vkCreateAndroidSurfaceKHR( VkInstance instance, const VkAndroidSurfaceCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface  ) const
  {
    return ::vkCreateAndroidSurfaceKHR( instance, pCreateInfo, pAllocator, pSurface);
  }
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/
  VkResult vkCreateBuffer( VkDevice device, const VkBufferCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkBuffer* pBuffer  ) const
  {
    return ::vkCreateBuffer( device, pCreateInfo, pAllocator, pBuffer);
  }
  VkResult vkCreateBufferView( VkDevice device, const VkBufferViewCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkBufferView* pView  ) const
  {
    return ::vkCreateBufferView( device, pCreateInfo, pAllocator, pView);
  }
  VkResult vkCreateCommandPool( VkDevice device, const VkCommandPoolCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkCommandPool* pCommandPool  ) const
  {
    return ::vkCreateCommandPool( device, pCreateInfo, pAllocator, pCommandPool);
  }
  VkResult vkCreateComputePipelines( VkDevice device, VkPipelineCache pipelineCache, uint32_t createInfoCount, const VkComputePipelineCreateInfo* pCreateInfos, const VkAllocationCallbacks* pAllocator, VkPipeline* pPipelines  ) const
  {
    return ::vkCreateComputePipelines( device, pipelineCache, createInfoCount, pCreateInfos, pAllocator, pPipelines);
  }
  VkResult vkCreateDebugReportCallbackEXT( VkInstance instance, const VkDebugReportCallbackCreateInfoEXT* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDebugReportCallbackEXT* pCallback  ) const
  {
    return ::vkCreateDebugReportCallbackEXT( instance, pCreateInfo, pAllocator, pCallback);
  }
  VkResult vkCreateDebugUtilsMessengerEXT( VkInstance instance, const VkDebugUtilsMessengerCreateInfoEXT* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDebugUtilsMessengerEXT* pMessenger  ) const
  {
    return ::vkCreateDebugUtilsMessengerEXT( instance, pCreateInfo, pAllocator, pMessenger);
  }
  VkResult vkCreateDescriptorPool( VkDevice device, const VkDescriptorPoolCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDescriptorPool* pDescriptorPool  ) const
  {
    return ::vkCreateDescriptorPool( device, pCreateInfo, pAllocator, pDescriptorPool);
  }
  VkResult vkCreateDescriptorSetLayout( VkDevice device, const VkDescriptorSetLayoutCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDescriptorSetLayout* pSetLayout  ) const
  {
    return ::vkCreateDescriptorSetLayout( device, pCreateInfo, pAllocator, pSetLayout);
  }
  VkResult vkCreateDescriptorUpdateTemplate( VkDevice device, const VkDescriptorUpdateTemplateCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDescriptorUpdateTemplate* pDescriptorUpdateTemplate  ) const
  {
    return ::vkCreateDescriptorUpdateTemplate( device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate);
  }
  VkResult vkCreateDescriptorUpdateTemplateKHR( VkDevice device, const VkDescriptorUpdateTemplateCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDescriptorUpdateTemplate* pDescriptorUpdateTemplate  ) const
  {
    return ::vkCreateDescriptorUpdateTemplateKHR( device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate);
  }
  VkResult vkCreateDevice( VkPhysicalDevice physicalDevice, const VkDeviceCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDevice* pDevice  ) const
  {
    return ::vkCreateDevice( physicalDevice, pCreateInfo, pAllocator, pDevice);
  }
  VkResult vkCreateDisplayModeKHR( VkPhysicalDevice physicalDevice, VkDisplayKHR display, const VkDisplayModeCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDisplayModeKHR* pMode  ) const
  {
    return ::vkCreateDisplayModeKHR( physicalDevice, display, pCreateInfo, pAllocator, pMode);
  }
  VkResult vkCreateDisplayPlaneSurfaceKHR( VkInstance instance, const VkDisplaySurfaceCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface  ) const
  {
    return ::vkCreateDisplayPlaneSurfaceKHR( instance, pCreateInfo, pAllocator, pSurface);
  }
  VkResult vkCreateEvent( VkDevice device, const VkEventCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkEvent* pEvent  ) const
  {
    return ::vkCreateEvent( device, pCreateInfo, pAllocator, pEvent);
  }
  VkResult vkCreateFence( VkDevice device, const VkFenceCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkFence* pFence  ) const
  {
    return ::vkCreateFence( device, pCreateInfo, pAllocator, pFence);
  }
  VkResult vkCreateFramebuffer( VkDevice device, const VkFramebufferCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkFramebuffer* pFramebuffer  ) const
  {
    return ::vkCreateFramebuffer( device, pCreateInfo, pAllocator, pFramebuffer);
  }
  VkResult vkCreateGraphicsPipelines( VkDevice device, VkPipelineCache pipelineCache, uint32_t createInfoCount, const VkGraphicsPipelineCreateInfo* pCreateInfos, const VkAllocationCallbacks* pAllocator, VkPipeline* pPipelines  ) const
  {
    return ::vkCreateGraphicsPipelines( device, pipelineCache, createInfoCount, pCreateInfos, pAllocator, pPipelines);
  }
#ifdef VK_USE_PLATFORM_IOS_MVK
  VkResult vkCreateIOSSurfaceMVK( VkInstance instance, const VkIOSSurfaceCreateInfoMVK* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface  ) const
  {
    return ::vkCreateIOSSurfaceMVK( instance, pCreateInfo, pAllocator, pSurface);
  }
#endif /*VK_USE_PLATFORM_IOS_MVK*/
  VkResult vkCreateImage( VkDevice device, const VkImageCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkImage* pImage  ) const
  {
    return ::vkCreateImage( device, pCreateInfo, pAllocator, pImage);
  }
#ifdef VK_USE_PLATFORM_FUCHSIA
  VkResult vkCreateImagePipeSurfaceFUCHSIA( VkInstance instance, const VkImagePipeSurfaceCreateInfoFUCHSIA* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface  ) const
  {
    return ::vkCreateImagePipeSurfaceFUCHSIA( instance, pCreateInfo, pAllocator, pSurface);
  }
#endif /*VK_USE_PLATFORM_FUCHSIA*/
  VkResult vkCreateImageView( VkDevice device, const VkImageViewCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkImageView* pView  ) const
  {
    return ::vkCreateImageView( device, pCreateInfo, pAllocator, pView);
  }
  VkResult vkCreateIndirectCommandsLayoutNVX( VkDevice device, const VkIndirectCommandsLayoutCreateInfoNVX* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkIndirectCommandsLayoutNVX* pIndirectCommandsLayout  ) const
  {
    return ::vkCreateIndirectCommandsLayoutNVX( device, pCreateInfo, pAllocator, pIndirectCommandsLayout);
  }
  VkResult vkCreateInstance( const VkInstanceCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkInstance* pInstance  ) const
  {
    return ::vkCreateInstance( pCreateInfo, pAllocator, pInstance);
  }
#ifdef VK_USE_PLATFORM_MACOS_MVK
  VkResult vkCreateMacOSSurfaceMVK( VkInstance instance, const VkMacOSSurfaceCreateInfoMVK* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface  ) const
  {
    return ::vkCreateMacOSSurfaceMVK( instance, pCreateInfo, pAllocator, pSurface);
  }
#endif /*VK_USE_PLATFORM_MACOS_MVK*/
  VkResult vkCreateObjectTableNVX( VkDevice device, const VkObjectTableCreateInfoNVX* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkObjectTableNVX* pObjectTable  ) const
  {
    return ::vkCreateObjectTableNVX( device, pCreateInfo, pAllocator, pObjectTable);
  }
  VkResult vkCreatePipelineCache( VkDevice device, const VkPipelineCacheCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkPipelineCache* pPipelineCache  ) const
  {
    return ::vkCreatePipelineCache( device, pCreateInfo, pAllocator, pPipelineCache);
  }
  VkResult vkCreatePipelineLayout( VkDevice device, const VkPipelineLayoutCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkPipelineLayout* pPipelineLayout  ) const
  {
    return ::vkCreatePipelineLayout( device, pCreateInfo, pAllocator, pPipelineLayout);
  }
  VkResult vkCreateQueryPool( VkDevice device, const VkQueryPoolCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkQueryPool* pQueryPool  ) const
  {
    return ::vkCreateQueryPool( device, pCreateInfo, pAllocator, pQueryPool);
  }
  VkResult vkCreateRayTracingPipelinesNV( VkDevice device, VkPipelineCache pipelineCache, uint32_t createInfoCount, const VkRayTracingPipelineCreateInfoNV* pCreateInfos, const VkAllocationCallbacks* pAllocator, VkPipeline* pPipelines  ) const
  {
    return ::vkCreateRayTracingPipelinesNV( device, pipelineCache, createInfoCount, pCreateInfos, pAllocator, pPipelines);
  }
  VkResult vkCreateRenderPass( VkDevice device, const VkRenderPassCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkRenderPass* pRenderPass  ) const
  {
    return ::vkCreateRenderPass( device, pCreateInfo, pAllocator, pRenderPass);
  }
  VkResult vkCreateRenderPass2KHR( VkDevice device, const VkRenderPassCreateInfo2KHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkRenderPass* pRenderPass  ) const
  {
    return ::vkCreateRenderPass2KHR( device, pCreateInfo, pAllocator, pRenderPass);
  }
  VkResult vkCreateSampler( VkDevice device, const VkSamplerCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSampler* pSampler  ) const
  {
    return ::vkCreateSampler( device, pCreateInfo, pAllocator, pSampler);
  }
  VkResult vkCreateSamplerYcbcrConversion( VkDevice device, const VkSamplerYcbcrConversionCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSamplerYcbcrConversion* pYcbcrConversion  ) const
  {
    return ::vkCreateSamplerYcbcrConversion( device, pCreateInfo, pAllocator, pYcbcrConversion);
  }
  VkResult vkCreateSamplerYcbcrConversionKHR( VkDevice device, const VkSamplerYcbcrConversionCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSamplerYcbcrConversion* pYcbcrConversion  ) const
  {
    return ::vkCreateSamplerYcbcrConversionKHR( device, pCreateInfo, pAllocator, pYcbcrConversion);
  }
  VkResult vkCreateSemaphore( VkDevice device, const VkSemaphoreCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSemaphore* pSemaphore  ) const
  {
    return ::vkCreateSemaphore( device, pCreateInfo, pAllocator, pSemaphore);
  }
  VkResult vkCreateShaderModule( VkDevice device, const VkShaderModuleCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkShaderModule* pShaderModule  ) const
  {
    return ::vkCreateShaderModule( device, pCreateInfo, pAllocator, pShaderModule);
  }
  VkResult vkCreateSharedSwapchainsKHR( VkDevice device, uint32_t swapchainCount, const VkSwapchainCreateInfoKHR* pCreateInfos, const VkAllocationCallbacks* pAllocator, VkSwapchainKHR* pSwapchains  ) const
  {
    return ::vkCreateSharedSwapchainsKHR( device, swapchainCount, pCreateInfos, pAllocator, pSwapchains);
  }
  VkResult vkCreateSwapchainKHR( VkDevice device, const VkSwapchainCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSwapchainKHR* pSwapchain  ) const
  {
    return ::vkCreateSwapchainKHR( device, pCreateInfo, pAllocator, pSwapchain);
  }
  VkResult vkCreateValidationCacheEXT( VkDevice device, const VkValidationCacheCreateInfoEXT* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkValidationCacheEXT* pValidationCache  ) const
  {
    return ::vkCreateValidationCacheEXT( device, pCreateInfo, pAllocator, pValidationCache);
  }
#ifdef VK_USE_PLATFORM_VI_NN
  VkResult vkCreateViSurfaceNN( VkInstance instance, const VkViSurfaceCreateInfoNN* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface  ) const
  {
    return ::vkCreateViSurfaceNN( instance, pCreateInfo, pAllocator, pSurface);
  }
#endif /*VK_USE_PLATFORM_VI_NN*/
#ifdef VK_USE_PLATFORM_WAYLAND_KHR
  VkResult vkCreateWaylandSurfaceKHR( VkInstance instance, const VkWaylandSurfaceCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface  ) const
  {
    return ::vkCreateWaylandSurfaceKHR( instance, pCreateInfo, pAllocator, pSurface);
  }
#endif /*VK_USE_PLATFORM_WAYLAND_KHR*/
#ifdef VK_USE_PLATFORM_WIN32_KHR
  VkResult vkCreateWin32SurfaceKHR( VkInstance instance, const VkWin32SurfaceCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface  ) const
  {
    return ::vkCreateWin32SurfaceKHR( instance, pCreateInfo, pAllocator, pSurface);
  }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
#ifdef VK_USE_PLATFORM_XCB_KHR
  VkResult vkCreateXcbSurfaceKHR( VkInstance instance, const VkXcbSurfaceCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface  ) const
  {
    return ::vkCreateXcbSurfaceKHR( instance, pCreateInfo, pAllocator, pSurface);
  }
#endif /*VK_USE_PLATFORM_XCB_KHR*/
#ifdef VK_USE_PLATFORM_XLIB_KHR
  VkResult vkCreateXlibSurfaceKHR( VkInstance instance, const VkXlibSurfaceCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface  ) const
  {
    return ::vkCreateXlibSurfaceKHR( instance, pCreateInfo, pAllocator, pSurface);
  }
#endif /*VK_USE_PLATFORM_XLIB_KHR*/
  VkResult vkDebugMarkerSetObjectNameEXT( VkDevice device, const VkDebugMarkerObjectNameInfoEXT* pNameInfo  ) const
  {
    return ::vkDebugMarkerSetObjectNameEXT( device, pNameInfo);
  }
  VkResult vkDebugMarkerSetObjectTagEXT( VkDevice device, const VkDebugMarkerObjectTagInfoEXT* pTagInfo  ) const
  {
    return ::vkDebugMarkerSetObjectTagEXT( device, pTagInfo);
  }
  void vkDebugReportMessageEXT( VkInstance instance, VkDebugReportFlagsEXT flags, VkDebugReportObjectTypeEXT objectType, uint64_t object, size_t location, int32_t messageCode, const char* pLayerPrefix, const char* pMessage  ) const
  {
    return ::vkDebugReportMessageEXT( instance, flags, objectType, object, location, messageCode, pLayerPrefix, pMessage);
  }
  void vkDestroyAccelerationStructureNV( VkDevice device, VkAccelerationStructureNV accelerationStructure, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyAccelerationStructureNV( device, accelerationStructure, pAllocator);
  }
  void vkDestroyBuffer( VkDevice device, VkBuffer buffer, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyBuffer( device, buffer, pAllocator);
  }
  void vkDestroyBufferView( VkDevice device, VkBufferView bufferView, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyBufferView( device, bufferView, pAllocator);
  }
  void vkDestroyCommandPool( VkDevice device, VkCommandPool commandPool, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyCommandPool( device, commandPool, pAllocator);
  }
  void vkDestroyDebugReportCallbackEXT( VkInstance instance, VkDebugReportCallbackEXT callback, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyDebugReportCallbackEXT( instance, callback, pAllocator);
  }
  void vkDestroyDebugUtilsMessengerEXT( VkInstance instance, VkDebugUtilsMessengerEXT messenger, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyDebugUtilsMessengerEXT( instance, messenger, pAllocator);
  }
  void vkDestroyDescriptorPool( VkDevice device, VkDescriptorPool descriptorPool, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyDescriptorPool( device, descriptorPool, pAllocator);
  }
  void vkDestroyDescriptorSetLayout( VkDevice device, VkDescriptorSetLayout descriptorSetLayout, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyDescriptorSetLayout( device, descriptorSetLayout, pAllocator);
  }
  void vkDestroyDescriptorUpdateTemplate( VkDevice device, VkDescriptorUpdateTemplate descriptorUpdateTemplate, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyDescriptorUpdateTemplate( device, descriptorUpdateTemplate, pAllocator);
  }
  void vkDestroyDescriptorUpdateTemplateKHR( VkDevice device, VkDescriptorUpdateTemplate descriptorUpdateTemplate, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyDescriptorUpdateTemplateKHR( device, descriptorUpdateTemplate, pAllocator);
  }
  void vkDestroyDevice( VkDevice device, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyDevice( device, pAllocator);
  }
  void vkDestroyEvent( VkDevice device, VkEvent event, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyEvent( device, event, pAllocator);
  }
  void vkDestroyFence( VkDevice device, VkFence fence, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyFence( device, fence, pAllocator);
  }
  void vkDestroyFramebuffer( VkDevice device, VkFramebuffer framebuffer, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyFramebuffer( device, framebuffer, pAllocator);
  }
  void vkDestroyImage( VkDevice device, VkImage image, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyImage( device, image, pAllocator);
  }
  void vkDestroyImageView( VkDevice device, VkImageView imageView, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyImageView( device, imageView, pAllocator);
  }
  void vkDestroyIndirectCommandsLayoutNVX( VkDevice device, VkIndirectCommandsLayoutNVX indirectCommandsLayout, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyIndirectCommandsLayoutNVX( device, indirectCommandsLayout, pAllocator);
  }
  void vkDestroyInstance( VkInstance instance, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyInstance( instance, pAllocator);
  }
  void vkDestroyObjectTableNVX( VkDevice device, VkObjectTableNVX objectTable, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyObjectTableNVX( device, objectTable, pAllocator);
  }
  void vkDestroyPipeline( VkDevice device, VkPipeline pipeline, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyPipeline( device, pipeline, pAllocator);
  }
  void vkDestroyPipelineCache( VkDevice device, VkPipelineCache pipelineCache, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyPipelineCache( device, pipelineCache, pAllocator);
  }
  void vkDestroyPipelineLayout( VkDevice device, VkPipelineLayout pipelineLayout, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyPipelineLayout( device, pipelineLayout, pAllocator);
  }
  void vkDestroyQueryPool( VkDevice device, VkQueryPool queryPool, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyQueryPool( device, queryPool, pAllocator);
  }
  void vkDestroyRenderPass( VkDevice device, VkRenderPass renderPass, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyRenderPass( device, renderPass, pAllocator);
  }
  void vkDestroySampler( VkDevice device, VkSampler sampler, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroySampler( device, sampler, pAllocator);
  }
  void vkDestroySamplerYcbcrConversion( VkDevice device, VkSamplerYcbcrConversion ycbcrConversion, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroySamplerYcbcrConversion( device, ycbcrConversion, pAllocator);
  }
  void vkDestroySamplerYcbcrConversionKHR( VkDevice device, VkSamplerYcbcrConversion ycbcrConversion, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroySamplerYcbcrConversionKHR( device, ycbcrConversion, pAllocator);
  }
  void vkDestroySemaphore( VkDevice device, VkSemaphore semaphore, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroySemaphore( device, semaphore, pAllocator);
  }
  void vkDestroyShaderModule( VkDevice device, VkShaderModule shaderModule, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyShaderModule( device, shaderModule, pAllocator);
  }
  void vkDestroySurfaceKHR( VkInstance instance, VkSurfaceKHR surface, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroySurfaceKHR( instance, surface, pAllocator);
  }
  void vkDestroySwapchainKHR( VkDevice device, VkSwapchainKHR swapchain, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroySwapchainKHR( device, swapchain, pAllocator);
  }
  void vkDestroyValidationCacheEXT( VkDevice device, VkValidationCacheEXT validationCache, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkDestroyValidationCacheEXT( device, validationCache, pAllocator);
  }
  VkResult vkDeviceWaitIdle( VkDevice device  ) const
  {
    return ::vkDeviceWaitIdle( device);
  }
  VkResult vkDisplayPowerControlEXT( VkDevice device, VkDisplayKHR display, const VkDisplayPowerInfoEXT* pDisplayPowerInfo  ) const
  {
    return ::vkDisplayPowerControlEXT( device, display, pDisplayPowerInfo);
  }
  VkResult vkEndCommandBuffer( VkCommandBuffer commandBuffer  ) const
  {
    return ::vkEndCommandBuffer( commandBuffer);
  }
  VkResult vkEnumerateDeviceExtensionProperties( VkPhysicalDevice physicalDevice, const char* pLayerName, uint32_t* pPropertyCount, VkExtensionProperties* pProperties  ) const
  {
    return ::vkEnumerateDeviceExtensionProperties( physicalDevice, pLayerName, pPropertyCount, pProperties);
  }
  VkResult vkEnumerateDeviceLayerProperties( VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount, VkLayerProperties* pProperties  ) const
  {
    return ::vkEnumerateDeviceLayerProperties( physicalDevice, pPropertyCount, pProperties);
  }
  VkResult vkEnumerateInstanceExtensionProperties( const char* pLayerName, uint32_t* pPropertyCount, VkExtensionProperties* pProperties  ) const
  {
    return ::vkEnumerateInstanceExtensionProperties( pLayerName, pPropertyCount, pProperties);
  }
  VkResult vkEnumerateInstanceLayerProperties( uint32_t* pPropertyCount, VkLayerProperties* pProperties  ) const
  {
    return ::vkEnumerateInstanceLayerProperties( pPropertyCount, pProperties);
  }
  VkResult vkEnumerateInstanceVersion( uint32_t* pApiVersion  ) const
  {
    return ::vkEnumerateInstanceVersion( pApiVersion);
  }
  VkResult vkEnumeratePhysicalDeviceGroups( VkInstance instance, uint32_t* pPhysicalDeviceGroupCount, VkPhysicalDeviceGroupProperties* pPhysicalDeviceGroupProperties  ) const
  {
    return ::vkEnumeratePhysicalDeviceGroups( instance, pPhysicalDeviceGroupCount, pPhysicalDeviceGroupProperties);
  }
  VkResult vkEnumeratePhysicalDeviceGroupsKHR( VkInstance instance, uint32_t* pPhysicalDeviceGroupCount, VkPhysicalDeviceGroupProperties* pPhysicalDeviceGroupProperties  ) const
  {
    return ::vkEnumeratePhysicalDeviceGroupsKHR( instance, pPhysicalDeviceGroupCount, pPhysicalDeviceGroupProperties);
  }
  VkResult vkEnumeratePhysicalDevices( VkInstance instance, uint32_t* pPhysicalDeviceCount, VkPhysicalDevice* pPhysicalDevices  ) const
  {
    return ::vkEnumeratePhysicalDevices( instance, pPhysicalDeviceCount, pPhysicalDevices);
  }
  VkResult vkFlushMappedMemoryRanges( VkDevice device, uint32_t memoryRangeCount, const VkMappedMemoryRange* pMemoryRanges  ) const
  {
    return ::vkFlushMappedMemoryRanges( device, memoryRangeCount, pMemoryRanges);
  }
  void vkFreeCommandBuffers( VkDevice device, VkCommandPool commandPool, uint32_t commandBufferCount, const VkCommandBuffer* pCommandBuffers  ) const
  {
    return ::vkFreeCommandBuffers( device, commandPool, commandBufferCount, pCommandBuffers);
  }
  VkResult vkFreeDescriptorSets( VkDevice device, VkDescriptorPool descriptorPool, uint32_t descriptorSetCount, const VkDescriptorSet* pDescriptorSets  ) const
  {
    return ::vkFreeDescriptorSets( device, descriptorPool, descriptorSetCount, pDescriptorSets);
  }
  void vkFreeMemory( VkDevice device, VkDeviceMemory memory, const VkAllocationCallbacks* pAllocator  ) const
  {
    return ::vkFreeMemory( device, memory, pAllocator);
  }
  VkResult vkGetAccelerationStructureHandleNV( VkDevice device, VkAccelerationStructureNV accelerationStructure, size_t dataSize, void* pData  ) const
  {
    return ::vkGetAccelerationStructureHandleNV( device, accelerationStructure, dataSize, pData);
  }
  void vkGetAccelerationStructureMemoryRequirementsNV( VkDevice device, const VkAccelerationStructureMemoryRequirementsInfoNV* pInfo, VkMemoryRequirements2KHR* pMemoryRequirements  ) const
  {
    return ::vkGetAccelerationStructureMemoryRequirementsNV( device, pInfo, pMemoryRequirements);
  }
#ifdef VK_USE_PLATFORM_ANDROID_KHR
  VkResult vkGetAndroidHardwareBufferPropertiesANDROID( VkDevice device, const struct AHardwareBuffer* buffer, VkAndroidHardwareBufferPropertiesANDROID* pProperties  ) const
  {
    return ::vkGetAndroidHardwareBufferPropertiesANDROID( device, buffer, pProperties);
  }
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/
  VkDeviceAddress vkGetBufferDeviceAddressEXT( VkDevice device, const VkBufferDeviceAddressInfoEXT* pInfo  ) const
  {
    return ::vkGetBufferDeviceAddressEXT( device, pInfo);
  }
  void vkGetBufferMemoryRequirements( VkDevice device, VkBuffer buffer, VkMemoryRequirements* pMemoryRequirements  ) const
  {
    return ::vkGetBufferMemoryRequirements( device, buffer, pMemoryRequirements);
  }
  void vkGetBufferMemoryRequirements2( VkDevice device, const VkBufferMemoryRequirementsInfo2* pInfo, VkMemoryRequirements2* pMemoryRequirements  ) const
  {
    return ::vkGetBufferMemoryRequirements2( device, pInfo, pMemoryRequirements);
  }
  void vkGetBufferMemoryRequirements2KHR( VkDevice device, const VkBufferMemoryRequirementsInfo2* pInfo, VkMemoryRequirements2* pMemoryRequirements  ) const
  {
    return ::vkGetBufferMemoryRequirements2KHR( device, pInfo, pMemoryRequirements);
  }
  VkResult vkGetCalibratedTimestampsEXT( VkDevice device, uint32_t timestampCount, const VkCalibratedTimestampInfoEXT* pTimestampInfos, uint64_t* pTimestamps, uint64_t* pMaxDeviation  ) const
  {
    return ::vkGetCalibratedTimestampsEXT( device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation);
  }
  void vkGetDescriptorSetLayoutSupport( VkDevice device, const VkDescriptorSetLayoutCreateInfo* pCreateInfo, VkDescriptorSetLayoutSupport* pSupport  ) const
  {
    return ::vkGetDescriptorSetLayoutSupport( device, pCreateInfo, pSupport);
  }
  void vkGetDescriptorSetLayoutSupportKHR( VkDevice device, const VkDescriptorSetLayoutCreateInfo* pCreateInfo, VkDescriptorSetLayoutSupport* pSupport  ) const
  {
    return ::vkGetDescriptorSetLayoutSupportKHR( device, pCreateInfo, pSupport);
  }
  void vkGetDeviceGroupPeerMemoryFeatures( VkDevice device, uint32_t heapIndex, uint32_t localDeviceIndex, uint32_t remoteDeviceIndex, VkPeerMemoryFeatureFlags* pPeerMemoryFeatures  ) const
  {
    return ::vkGetDeviceGroupPeerMemoryFeatures( device, heapIndex, localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures);
  }
  void vkGetDeviceGroupPeerMemoryFeaturesKHR( VkDevice device, uint32_t heapIndex, uint32_t localDeviceIndex, uint32_t remoteDeviceIndex, VkPeerMemoryFeatureFlags* pPeerMemoryFeatures  ) const
  {
    return ::vkGetDeviceGroupPeerMemoryFeaturesKHR( device, heapIndex, localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures);
  }
  VkResult vkGetDeviceGroupPresentCapabilitiesKHR( VkDevice device, VkDeviceGroupPresentCapabilitiesKHR* pDeviceGroupPresentCapabilities  ) const
  {
    return ::vkGetDeviceGroupPresentCapabilitiesKHR( device, pDeviceGroupPresentCapabilities);
  }
  VkResult vkGetDeviceGroupSurfacePresentModesKHR( VkDevice device, VkSurfaceKHR surface, VkDeviceGroupPresentModeFlagsKHR* pModes  ) const
  {
    return ::vkGetDeviceGroupSurfacePresentModesKHR( device, surface, pModes);
  }
  void vkGetDeviceMemoryCommitment( VkDevice device, VkDeviceMemory memory, VkDeviceSize* pCommittedMemoryInBytes  ) const
  {
    return ::vkGetDeviceMemoryCommitment( device, memory, pCommittedMemoryInBytes);
  }
  PFN_vkVoidFunction vkGetDeviceProcAddr( VkDevice device, const char* pName  ) const
  {
    return ::vkGetDeviceProcAddr( device, pName);
  }
  void vkGetDeviceQueue( VkDevice device, uint32_t queueFamilyIndex, uint32_t queueIndex, VkQueue* pQueue  ) const
  {
    return ::vkGetDeviceQueue( device, queueFamilyIndex, queueIndex, pQueue);
  }
  void vkGetDeviceQueue2( VkDevice device, const VkDeviceQueueInfo2* pQueueInfo, VkQueue* pQueue  ) const
  {
    return ::vkGetDeviceQueue2( device, pQueueInfo, pQueue);
  }
  VkResult vkGetDisplayModeProperties2KHR( VkPhysicalDevice physicalDevice, VkDisplayKHR display, uint32_t* pPropertyCount, VkDisplayModeProperties2KHR* pProperties  ) const
  {
    return ::vkGetDisplayModeProperties2KHR( physicalDevice, display, pPropertyCount, pProperties);
  }
  VkResult vkGetDisplayModePropertiesKHR( VkPhysicalDevice physicalDevice, VkDisplayKHR display, uint32_t* pPropertyCount, VkDisplayModePropertiesKHR* pProperties  ) const
  {
    return ::vkGetDisplayModePropertiesKHR( physicalDevice, display, pPropertyCount, pProperties);
  }
  VkResult vkGetDisplayPlaneCapabilities2KHR( VkPhysicalDevice physicalDevice, const VkDisplayPlaneInfo2KHR* pDisplayPlaneInfo, VkDisplayPlaneCapabilities2KHR* pCapabilities  ) const
  {
    return ::vkGetDisplayPlaneCapabilities2KHR( physicalDevice, pDisplayPlaneInfo, pCapabilities);
  }
  VkResult vkGetDisplayPlaneCapabilitiesKHR( VkPhysicalDevice physicalDevice, VkDisplayModeKHR mode, uint32_t planeIndex, VkDisplayPlaneCapabilitiesKHR* pCapabilities  ) const
  {
    return ::vkGetDisplayPlaneCapabilitiesKHR( physicalDevice, mode, planeIndex, pCapabilities);
  }
  VkResult vkGetDisplayPlaneSupportedDisplaysKHR( VkPhysicalDevice physicalDevice, uint32_t planeIndex, uint32_t* pDisplayCount, VkDisplayKHR* pDisplays  ) const
  {
    return ::vkGetDisplayPlaneSupportedDisplaysKHR( physicalDevice, planeIndex, pDisplayCount, pDisplays);
  }
  VkResult vkGetEventStatus( VkDevice device, VkEvent event  ) const
  {
    return ::vkGetEventStatus( device, event);
  }
  VkResult vkGetFenceFdKHR( VkDevice device, const VkFenceGetFdInfoKHR* pGetFdInfo, int* pFd  ) const
  {
    return ::vkGetFenceFdKHR( device, pGetFdInfo, pFd);
  }
  VkResult vkGetFenceStatus( VkDevice device, VkFence fence  ) const
  {
    return ::vkGetFenceStatus( device, fence);
  }
#ifdef VK_USE_PLATFORM_WIN32_KHR
  VkResult vkGetFenceWin32HandleKHR( VkDevice device, const VkFenceGetWin32HandleInfoKHR* pGetWin32HandleInfo, HANDLE* pHandle  ) const
  {
    return ::vkGetFenceWin32HandleKHR( device, pGetWin32HandleInfo, pHandle);
  }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
  VkResult vkGetImageDrmFormatModifierPropertiesEXT( VkDevice device, VkImage image, VkImageDrmFormatModifierPropertiesEXT* pProperties  ) const
  {
    return ::vkGetImageDrmFormatModifierPropertiesEXT( device, image, pProperties);
  }
  void vkGetImageMemoryRequirements( VkDevice device, VkImage image, VkMemoryRequirements* pMemoryRequirements  ) const
  {
    return ::vkGetImageMemoryRequirements( device, image, pMemoryRequirements);
  }
  void vkGetImageMemoryRequirements2( VkDevice device, const VkImageMemoryRequirementsInfo2* pInfo, VkMemoryRequirements2* pMemoryRequirements  ) const
  {
    return ::vkGetImageMemoryRequirements2( device, pInfo, pMemoryRequirements);
  }
  void vkGetImageMemoryRequirements2KHR( VkDevice device, const VkImageMemoryRequirementsInfo2* pInfo, VkMemoryRequirements2* pMemoryRequirements  ) const
  {
    return ::vkGetImageMemoryRequirements2KHR( device, pInfo, pMemoryRequirements);
  }
  void vkGetImageSparseMemoryRequirements( VkDevice device, VkImage image, uint32_t* pSparseMemoryRequirementCount, VkSparseImageMemoryRequirements* pSparseMemoryRequirements  ) const
  {
    return ::vkGetImageSparseMemoryRequirements( device, image, pSparseMemoryRequirementCount, pSparseMemoryRequirements);
  }
  void vkGetImageSparseMemoryRequirements2( VkDevice device, const VkImageSparseMemoryRequirementsInfo2* pInfo, uint32_t* pSparseMemoryRequirementCount, VkSparseImageMemoryRequirements2* pSparseMemoryRequirements  ) const
  {
    return ::vkGetImageSparseMemoryRequirements2( device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);
  }
  void vkGetImageSparseMemoryRequirements2KHR( VkDevice device, const VkImageSparseMemoryRequirementsInfo2* pInfo, uint32_t* pSparseMemoryRequirementCount, VkSparseImageMemoryRequirements2* pSparseMemoryRequirements  ) const
  {
    return ::vkGetImageSparseMemoryRequirements2KHR( device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);
  }
  void vkGetImageSubresourceLayout( VkDevice device, VkImage image, const VkImageSubresource* pSubresource, VkSubresourceLayout* pLayout  ) const
  {
    return ::vkGetImageSubresourceLayout( device, image, pSubresource, pLayout);
  }
  PFN_vkVoidFunction vkGetInstanceProcAddr( VkInstance instance, const char* pName  ) const
  {
    return ::vkGetInstanceProcAddr( instance, pName);
  }
#ifdef VK_USE_PLATFORM_ANDROID_KHR
  VkResult vkGetMemoryAndroidHardwareBufferANDROID( VkDevice device, const VkMemoryGetAndroidHardwareBufferInfoANDROID* pInfo, struct AHardwareBuffer** pBuffer  ) const
  {
    return ::vkGetMemoryAndroidHardwareBufferANDROID( device, pInfo, pBuffer);
  }
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/
  VkResult vkGetMemoryFdKHR( VkDevice device, const VkMemoryGetFdInfoKHR* pGetFdInfo, int* pFd  ) const
  {
    return ::vkGetMemoryFdKHR( device, pGetFdInfo, pFd);
  }
  VkResult vkGetMemoryFdPropertiesKHR( VkDevice device, VkExternalMemoryHandleTypeFlagBits handleType, int fd, VkMemoryFdPropertiesKHR* pMemoryFdProperties  ) const
  {
    return ::vkGetMemoryFdPropertiesKHR( device, handleType, fd, pMemoryFdProperties);
  }
  VkResult vkGetMemoryHostPointerPropertiesEXT( VkDevice device, VkExternalMemoryHandleTypeFlagBits handleType, const void* pHostPointer, VkMemoryHostPointerPropertiesEXT* pMemoryHostPointerProperties  ) const
  {
    return ::vkGetMemoryHostPointerPropertiesEXT( device, handleType, pHostPointer, pMemoryHostPointerProperties);
  }
#ifdef VK_USE_PLATFORM_WIN32_KHR
  VkResult vkGetMemoryWin32HandleKHR( VkDevice device, const VkMemoryGetWin32HandleInfoKHR* pGetWin32HandleInfo, HANDLE* pHandle  ) const
  {
    return ::vkGetMemoryWin32HandleKHR( device, pGetWin32HandleInfo, pHandle);
  }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
#ifdef VK_USE_PLATFORM_WIN32_KHR
  VkResult vkGetMemoryWin32HandleNV( VkDevice device, VkDeviceMemory memory, VkExternalMemoryHandleTypeFlagsNV handleType, HANDLE* pHandle  ) const
  {
    return ::vkGetMemoryWin32HandleNV( device, memory, handleType, pHandle);
  }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
#ifdef VK_USE_PLATFORM_WIN32_KHR
  VkResult vkGetMemoryWin32HandlePropertiesKHR( VkDevice device, VkExternalMemoryHandleTypeFlagBits handleType, HANDLE handle, VkMemoryWin32HandlePropertiesKHR* pMemoryWin32HandleProperties  ) const
  {
    return ::vkGetMemoryWin32HandlePropertiesKHR( device, handleType, handle, pMemoryWin32HandleProperties);
  }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
  VkResult vkGetPastPresentationTimingGOOGLE( VkDevice device, VkSwapchainKHR swapchain, uint32_t* pPresentationTimingCount, VkPastPresentationTimingGOOGLE* pPresentationTimings  ) const
  {
    return ::vkGetPastPresentationTimingGOOGLE( device, swapchain, pPresentationTimingCount, pPresentationTimings);
  }
  VkResult vkGetPhysicalDeviceCalibrateableTimeDomainsEXT( VkPhysicalDevice physicalDevice, uint32_t* pTimeDomainCount, VkTimeDomainEXT* pTimeDomains  ) const
  {
    return ::vkGetPhysicalDeviceCalibrateableTimeDomainsEXT( physicalDevice, pTimeDomainCount, pTimeDomains);
  }
  VkResult vkGetPhysicalDeviceDisplayPlaneProperties2KHR( VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount, VkDisplayPlaneProperties2KHR* pProperties  ) const
  {
    return ::vkGetPhysicalDeviceDisplayPlaneProperties2KHR( physicalDevice, pPropertyCount, pProperties);
  }
  VkResult vkGetPhysicalDeviceDisplayPlanePropertiesKHR( VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount, VkDisplayPlanePropertiesKHR* pProperties  ) const
  {
    return ::vkGetPhysicalDeviceDisplayPlanePropertiesKHR( physicalDevice, pPropertyCount, pProperties);
  }
  VkResult vkGetPhysicalDeviceDisplayProperties2KHR( VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount, VkDisplayProperties2KHR* pProperties  ) const
  {
    return ::vkGetPhysicalDeviceDisplayProperties2KHR( physicalDevice, pPropertyCount, pProperties);
  }
  VkResult vkGetPhysicalDeviceDisplayPropertiesKHR( VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount, VkDisplayPropertiesKHR* pProperties  ) const
  {
    return ::vkGetPhysicalDeviceDisplayPropertiesKHR( physicalDevice, pPropertyCount, pProperties);
  }
  void vkGetPhysicalDeviceExternalBufferProperties( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalBufferInfo* pExternalBufferInfo, VkExternalBufferProperties* pExternalBufferProperties  ) const
  {
    return ::vkGetPhysicalDeviceExternalBufferProperties( physicalDevice, pExternalBufferInfo, pExternalBufferProperties);
  }
  void vkGetPhysicalDeviceExternalBufferPropertiesKHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalBufferInfo* pExternalBufferInfo, VkExternalBufferProperties* pExternalBufferProperties  ) const
  {
    return ::vkGetPhysicalDeviceExternalBufferPropertiesKHR( physicalDevice, pExternalBufferInfo, pExternalBufferProperties);
  }
  void vkGetPhysicalDeviceExternalFenceProperties( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalFenceInfo* pExternalFenceInfo, VkExternalFenceProperties* pExternalFenceProperties  ) const
  {
    return ::vkGetPhysicalDeviceExternalFenceProperties( physicalDevice, pExternalFenceInfo, pExternalFenceProperties);
  }
  void vkGetPhysicalDeviceExternalFencePropertiesKHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalFenceInfo* pExternalFenceInfo, VkExternalFenceProperties* pExternalFenceProperties  ) const
  {
    return ::vkGetPhysicalDeviceExternalFencePropertiesKHR( physicalDevice, pExternalFenceInfo, pExternalFenceProperties);
  }
  VkResult vkGetPhysicalDeviceExternalImageFormatPropertiesNV( VkPhysicalDevice physicalDevice, VkFormat format, VkImageType type, VkImageTiling tiling, VkImageUsageFlags usage, VkImageCreateFlags flags, VkExternalMemoryHandleTypeFlagsNV externalHandleType, VkExternalImageFormatPropertiesNV* pExternalImageFormatProperties  ) const
  {
    return ::vkGetPhysicalDeviceExternalImageFormatPropertiesNV( physicalDevice, format, type, tiling, usage, flags, externalHandleType, pExternalImageFormatProperties);
  }
  void vkGetPhysicalDeviceExternalSemaphoreProperties( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalSemaphoreInfo* pExternalSemaphoreInfo, VkExternalSemaphoreProperties* pExternalSemaphoreProperties  ) const
  {
    return ::vkGetPhysicalDeviceExternalSemaphoreProperties( physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties);
  }
  void vkGetPhysicalDeviceExternalSemaphorePropertiesKHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalSemaphoreInfo* pExternalSemaphoreInfo, VkExternalSemaphoreProperties* pExternalSemaphoreProperties  ) const
  {
    return ::vkGetPhysicalDeviceExternalSemaphorePropertiesKHR( physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties);
  }
  void vkGetPhysicalDeviceFeatures( VkPhysicalDevice physicalDevice, VkPhysicalDeviceFeatures* pFeatures  ) const
  {
    return ::vkGetPhysicalDeviceFeatures( physicalDevice, pFeatures);
  }
  void vkGetPhysicalDeviceFeatures2( VkPhysicalDevice physicalDevice, VkPhysicalDeviceFeatures2* pFeatures  ) const
  {
    return ::vkGetPhysicalDeviceFeatures2( physicalDevice, pFeatures);
  }
  void vkGetPhysicalDeviceFeatures2KHR( VkPhysicalDevice physicalDevice, VkPhysicalDeviceFeatures2* pFeatures  ) const
  {
    return ::vkGetPhysicalDeviceFeatures2KHR( physicalDevice, pFeatures);
  }
  void vkGetPhysicalDeviceFormatProperties( VkPhysicalDevice physicalDevice, VkFormat format, VkFormatProperties* pFormatProperties  ) const
  {
    return ::vkGetPhysicalDeviceFormatProperties( physicalDevice, format, pFormatProperties);
  }
  void vkGetPhysicalDeviceFormatProperties2( VkPhysicalDevice physicalDevice, VkFormat format, VkFormatProperties2* pFormatProperties  ) const
  {
    return ::vkGetPhysicalDeviceFormatProperties2( physicalDevice, format, pFormatProperties);
  }
  void vkGetPhysicalDeviceFormatProperties2KHR( VkPhysicalDevice physicalDevice, VkFormat format, VkFormatProperties2* pFormatProperties  ) const
  {
    return ::vkGetPhysicalDeviceFormatProperties2KHR( physicalDevice, format, pFormatProperties);
  }
  void vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX( VkPhysicalDevice physicalDevice, VkDeviceGeneratedCommandsFeaturesNVX* pFeatures, VkDeviceGeneratedCommandsLimitsNVX* pLimits  ) const
  {
    return ::vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX( physicalDevice, pFeatures, pLimits);
  }
  VkResult vkGetPhysicalDeviceImageFormatProperties( VkPhysicalDevice physicalDevice, VkFormat format, VkImageType type, VkImageTiling tiling, VkImageUsageFlags usage, VkImageCreateFlags flags, VkImageFormatProperties* pImageFormatProperties  ) const
  {
    return ::vkGetPhysicalDeviceImageFormatProperties( physicalDevice, format, type, tiling, usage, flags, pImageFormatProperties);
  }
  VkResult vkGetPhysicalDeviceImageFormatProperties2( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceImageFormatInfo2* pImageFormatInfo, VkImageFormatProperties2* pImageFormatProperties  ) const
  {
    return ::vkGetPhysicalDeviceImageFormatProperties2( physicalDevice, pImageFormatInfo, pImageFormatProperties);
  }
  VkResult vkGetPhysicalDeviceImageFormatProperties2KHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceImageFormatInfo2* pImageFormatInfo, VkImageFormatProperties2* pImageFormatProperties  ) const
  {
    return ::vkGetPhysicalDeviceImageFormatProperties2KHR( physicalDevice, pImageFormatInfo, pImageFormatProperties);
  }
  void vkGetPhysicalDeviceMemoryProperties( VkPhysicalDevice physicalDevice, VkPhysicalDeviceMemoryProperties* pMemoryProperties  ) const
  {
    return ::vkGetPhysicalDeviceMemoryProperties( physicalDevice, pMemoryProperties);
  }
  void vkGetPhysicalDeviceMemoryProperties2( VkPhysicalDevice physicalDevice, VkPhysicalDeviceMemoryProperties2* pMemoryProperties  ) const
  {
    return ::vkGetPhysicalDeviceMemoryProperties2( physicalDevice, pMemoryProperties);
  }
  void vkGetPhysicalDeviceMemoryProperties2KHR( VkPhysicalDevice physicalDevice, VkPhysicalDeviceMemoryProperties2* pMemoryProperties  ) const
  {
    return ::vkGetPhysicalDeviceMemoryProperties2KHR( physicalDevice, pMemoryProperties);
  }
  void vkGetPhysicalDeviceMultisamplePropertiesEXT( VkPhysicalDevice physicalDevice, VkSampleCountFlagBits samples, VkMultisamplePropertiesEXT* pMultisampleProperties  ) const
  {
    return ::vkGetPhysicalDeviceMultisamplePropertiesEXT( physicalDevice, samples, pMultisampleProperties);
  }
  VkResult vkGetPhysicalDevicePresentRectanglesKHR( VkPhysicalDevice physicalDevice, VkSurfaceKHR surface, uint32_t* pRectCount, VkRect2D* pRects  ) const
  {
    return ::vkGetPhysicalDevicePresentRectanglesKHR( physicalDevice, surface, pRectCount, pRects);
  }
  void vkGetPhysicalDeviceProperties( VkPhysicalDevice physicalDevice, VkPhysicalDeviceProperties* pProperties  ) const
  {
    return ::vkGetPhysicalDeviceProperties( physicalDevice, pProperties);
  }
  void vkGetPhysicalDeviceProperties2( VkPhysicalDevice physicalDevice, VkPhysicalDeviceProperties2* pProperties  ) const
  {
    return ::vkGetPhysicalDeviceProperties2( physicalDevice, pProperties);
  }
  void vkGetPhysicalDeviceProperties2KHR( VkPhysicalDevice physicalDevice, VkPhysicalDeviceProperties2* pProperties  ) const
  {
    return ::vkGetPhysicalDeviceProperties2KHR( physicalDevice, pProperties);
  }
  void vkGetPhysicalDeviceQueueFamilyProperties( VkPhysicalDevice physicalDevice, uint32_t* pQueueFamilyPropertyCount, VkQueueFamilyProperties* pQueueFamilyProperties  ) const
  {
    return ::vkGetPhysicalDeviceQueueFamilyProperties( physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);
  }
  void vkGetPhysicalDeviceQueueFamilyProperties2( VkPhysicalDevice physicalDevice, uint32_t* pQueueFamilyPropertyCount, VkQueueFamilyProperties2* pQueueFamilyProperties  ) const
  {
    return ::vkGetPhysicalDeviceQueueFamilyProperties2( physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);
  }
  void vkGetPhysicalDeviceQueueFamilyProperties2KHR( VkPhysicalDevice physicalDevice, uint32_t* pQueueFamilyPropertyCount, VkQueueFamilyProperties2* pQueueFamilyProperties  ) const
  {
    return ::vkGetPhysicalDeviceQueueFamilyProperties2KHR( physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);
  }
  void vkGetPhysicalDeviceSparseImageFormatProperties( VkPhysicalDevice physicalDevice, VkFormat format, VkImageType type, VkSampleCountFlagBits samples, VkImageUsageFlags usage, VkImageTiling tiling, uint32_t* pPropertyCount, VkSparseImageFormatProperties* pProperties  ) const
  {
    return ::vkGetPhysicalDeviceSparseImageFormatProperties( physicalDevice, format, type, samples, usage, tiling, pPropertyCount, pProperties);
  }
  void vkGetPhysicalDeviceSparseImageFormatProperties2( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceSparseImageFormatInfo2* pFormatInfo, uint32_t* pPropertyCount, VkSparseImageFormatProperties2* pProperties  ) const
  {
    return ::vkGetPhysicalDeviceSparseImageFormatProperties2( physicalDevice, pFormatInfo, pPropertyCount, pProperties);
  }
  void vkGetPhysicalDeviceSparseImageFormatProperties2KHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceSparseImageFormatInfo2* pFormatInfo, uint32_t* pPropertyCount, VkSparseImageFormatProperties2* pProperties  ) const
  {
    return ::vkGetPhysicalDeviceSparseImageFormatProperties2KHR( physicalDevice, pFormatInfo, pPropertyCount, pProperties);
  }
  VkResult vkGetPhysicalDeviceSurfaceCapabilities2EXT( VkPhysicalDevice physicalDevice, VkSurfaceKHR surface, VkSurfaceCapabilities2EXT* pSurfaceCapabilities  ) const
  {
    return ::vkGetPhysicalDeviceSurfaceCapabilities2EXT( physicalDevice, surface, pSurfaceCapabilities);
  }
  VkResult vkGetPhysicalDeviceSurfaceCapabilities2KHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo, VkSurfaceCapabilities2KHR* pSurfaceCapabilities  ) const
  {
    return ::vkGetPhysicalDeviceSurfaceCapabilities2KHR( physicalDevice, pSurfaceInfo, pSurfaceCapabilities);
  }
  VkResult vkGetPhysicalDeviceSurfaceCapabilitiesKHR( VkPhysicalDevice physicalDevice, VkSurfaceKHR surface, VkSurfaceCapabilitiesKHR* pSurfaceCapabilities  ) const
  {
    return ::vkGetPhysicalDeviceSurfaceCapabilitiesKHR( physicalDevice, surface, pSurfaceCapabilities);
  }
  VkResult vkGetPhysicalDeviceSurfaceFormats2KHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo, uint32_t* pSurfaceFormatCount, VkSurfaceFormat2KHR* pSurfaceFormats  ) const
  {
    return ::vkGetPhysicalDeviceSurfaceFormats2KHR( physicalDevice, pSurfaceInfo, pSurfaceFormatCount, pSurfaceFormats);
  }
  VkResult vkGetPhysicalDeviceSurfaceFormatsKHR( VkPhysicalDevice physicalDevice, VkSurfaceKHR surface, uint32_t* pSurfaceFormatCount, VkSurfaceFormatKHR* pSurfaceFormats  ) const
  {
    return ::vkGetPhysicalDeviceSurfaceFormatsKHR( physicalDevice, surface, pSurfaceFormatCount, pSurfaceFormats);
  }
  VkResult vkGetPhysicalDeviceSurfacePresentModesKHR( VkPhysicalDevice physicalDevice, VkSurfaceKHR surface, uint32_t* pPresentModeCount, VkPresentModeKHR* pPresentModes  ) const
  {
    return ::vkGetPhysicalDeviceSurfacePresentModesKHR( physicalDevice, surface, pPresentModeCount, pPresentModes);
  }
  VkResult vkGetPhysicalDeviceSurfaceSupportKHR( VkPhysicalDevice physicalDevice, uint32_t queueFamilyIndex, VkSurfaceKHR surface, VkBool32* pSupported  ) const
  {
    return ::vkGetPhysicalDeviceSurfaceSupportKHR( physicalDevice, queueFamilyIndex, surface, pSupported);
  }
#ifdef VK_USE_PLATFORM_WAYLAND_KHR
  VkBool32 vkGetPhysicalDeviceWaylandPresentationSupportKHR( VkPhysicalDevice physicalDevice, uint32_t queueFamilyIndex, struct wl_display* display  ) const
  {
    return ::vkGetPhysicalDeviceWaylandPresentationSupportKHR( physicalDevice, queueFamilyIndex, display);
  }
#endif /*VK_USE_PLATFORM_WAYLAND_KHR*/
#ifdef VK_USE_PLATFORM_WIN32_KHR
  VkBool32 vkGetPhysicalDeviceWin32PresentationSupportKHR( VkPhysicalDevice physicalDevice, uint32_t queueFamilyIndex  ) const
  {
    return ::vkGetPhysicalDeviceWin32PresentationSupportKHR( physicalDevice, queueFamilyIndex);
  }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
#ifdef VK_USE_PLATFORM_XCB_KHR
  VkBool32 vkGetPhysicalDeviceXcbPresentationSupportKHR( VkPhysicalDevice physicalDevice, uint32_t queueFamilyIndex, xcb_connection_t* connection, xcb_visualid_t visual_id  ) const
  {
    return ::vkGetPhysicalDeviceXcbPresentationSupportKHR( physicalDevice, queueFamilyIndex, connection, visual_id);
  }
#endif /*VK_USE_PLATFORM_XCB_KHR*/
#ifdef VK_USE_PLATFORM_XLIB_KHR
  VkBool32 vkGetPhysicalDeviceXlibPresentationSupportKHR( VkPhysicalDevice physicalDevice, uint32_t queueFamilyIndex, Display* dpy, VisualID visualID  ) const
  {
    return ::vkGetPhysicalDeviceXlibPresentationSupportKHR( physicalDevice, queueFamilyIndex, dpy, visualID);
  }
#endif /*VK_USE_PLATFORM_XLIB_KHR*/
  VkResult vkGetPipelineCacheData( VkDevice device, VkPipelineCache pipelineCache, size_t* pDataSize, void* pData  ) const
  {
    return ::vkGetPipelineCacheData( device, pipelineCache, pDataSize, pData);
  }
  VkResult vkGetQueryPoolResults( VkDevice device, VkQueryPool queryPool, uint32_t firstQuery, uint32_t queryCount, size_t dataSize, void* pData, VkDeviceSize stride, VkQueryResultFlags flags  ) const
  {
    return ::vkGetQueryPoolResults( device, queryPool, firstQuery, queryCount, dataSize, pData, stride, flags);
  }
  void vkGetQueueCheckpointDataNV( VkQueue queue, uint32_t* pCheckpointDataCount, VkCheckpointDataNV* pCheckpointData  ) const
  {
    return ::vkGetQueueCheckpointDataNV( queue, pCheckpointDataCount, pCheckpointData);
  }
#ifdef VK_USE_PLATFORM_XLIB_XRANDR_EXT
  VkResult vkGetRandROutputDisplayEXT( VkPhysicalDevice physicalDevice, Display* dpy, RROutput rrOutput, VkDisplayKHR* pDisplay  ) const
  {
    return ::vkGetRandROutputDisplayEXT( physicalDevice, dpy, rrOutput, pDisplay);
  }
#endif /*VK_USE_PLATFORM_XLIB_XRANDR_EXT*/
  VkResult vkGetRayTracingShaderGroupHandlesNV( VkDevice device, VkPipeline pipeline, uint32_t firstGroup, uint32_t groupCount, size_t dataSize, void* pData  ) const
  {
    return ::vkGetRayTracingShaderGroupHandlesNV( device, pipeline, firstGroup, groupCount, dataSize, pData);
  }
  VkResult vkGetRefreshCycleDurationGOOGLE( VkDevice device, VkSwapchainKHR swapchain, VkRefreshCycleDurationGOOGLE* pDisplayTimingProperties  ) const
  {
    return ::vkGetRefreshCycleDurationGOOGLE( device, swapchain, pDisplayTimingProperties);
  }
  void vkGetRenderAreaGranularity( VkDevice device, VkRenderPass renderPass, VkExtent2D* pGranularity  ) const
  {
    return ::vkGetRenderAreaGranularity( device, renderPass, pGranularity);
  }
  VkResult vkGetSemaphoreFdKHR( VkDevice device, const VkSemaphoreGetFdInfoKHR* pGetFdInfo, int* pFd  ) const
  {
    return ::vkGetSemaphoreFdKHR( device, pGetFdInfo, pFd);
  }
#ifdef VK_USE_PLATFORM_WIN32_KHR
  VkResult vkGetSemaphoreWin32HandleKHR( VkDevice device, const VkSemaphoreGetWin32HandleInfoKHR* pGetWin32HandleInfo, HANDLE* pHandle  ) const
  {
    return ::vkGetSemaphoreWin32HandleKHR( device, pGetWin32HandleInfo, pHandle);
  }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
  VkResult vkGetShaderInfoAMD( VkDevice device, VkPipeline pipeline, VkShaderStageFlagBits shaderStage, VkShaderInfoTypeAMD infoType, size_t* pInfoSize, void* pInfo  ) const
  {
    return ::vkGetShaderInfoAMD( device, pipeline, shaderStage, infoType, pInfoSize, pInfo);
  }
  VkResult vkGetSwapchainCounterEXT( VkDevice device, VkSwapchainKHR swapchain, VkSurfaceCounterFlagBitsEXT counter, uint64_t* pCounterValue  ) const
  {
    return ::vkGetSwapchainCounterEXT( device, swapchain, counter, pCounterValue);
  }
  VkResult vkGetSwapchainImagesKHR( VkDevice device, VkSwapchainKHR swapchain, uint32_t* pSwapchainImageCount, VkImage* pSwapchainImages  ) const
  {
    return ::vkGetSwapchainImagesKHR( device, swapchain, pSwapchainImageCount, pSwapchainImages);
  }
  VkResult vkGetSwapchainStatusKHR( VkDevice device, VkSwapchainKHR swapchain  ) const
  {
    return ::vkGetSwapchainStatusKHR( device, swapchain);
  }
  VkResult vkGetValidationCacheDataEXT( VkDevice device, VkValidationCacheEXT validationCache, size_t* pDataSize, void* pData  ) const
  {
    return ::vkGetValidationCacheDataEXT( device, validationCache, pDataSize, pData);
  }
  VkResult vkImportFenceFdKHR( VkDevice device, const VkImportFenceFdInfoKHR* pImportFenceFdInfo  ) const
  {
    return ::vkImportFenceFdKHR( device, pImportFenceFdInfo);
  }
#ifdef VK_USE_PLATFORM_WIN32_KHR
  VkResult vkImportFenceWin32HandleKHR( VkDevice device, const VkImportFenceWin32HandleInfoKHR* pImportFenceWin32HandleInfo  ) const
  {
    return ::vkImportFenceWin32HandleKHR( device, pImportFenceWin32HandleInfo);
  }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
  VkResult vkImportSemaphoreFdKHR( VkDevice device, const VkImportSemaphoreFdInfoKHR* pImportSemaphoreFdInfo  ) const
  {
    return ::vkImportSemaphoreFdKHR( device, pImportSemaphoreFdInfo);
  }
#ifdef VK_USE_PLATFORM_WIN32_KHR
  VkResult vkImportSemaphoreWin32HandleKHR( VkDevice device, const VkImportSemaphoreWin32HandleInfoKHR* pImportSemaphoreWin32HandleInfo  ) const
  {
    return ::vkImportSemaphoreWin32HandleKHR( device, pImportSemaphoreWin32HandleInfo);
  }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
  VkResult vkInvalidateMappedMemoryRanges( VkDevice device, uint32_t memoryRangeCount, const VkMappedMemoryRange* pMemoryRanges  ) const
  {
    return ::vkInvalidateMappedMemoryRanges( device, memoryRangeCount, pMemoryRanges);
  }
  VkResult vkMapMemory( VkDevice device, VkDeviceMemory memory, VkDeviceSize offset, VkDeviceSize size, VkMemoryMapFlags flags, void** ppData  ) const
  {
    return ::vkMapMemory( device, memory, offset, size, flags, ppData);
  }
  VkResult vkMergePipelineCaches( VkDevice device, VkPipelineCache dstCache, uint32_t srcCacheCount, const VkPipelineCache* pSrcCaches  ) const
  {
    return ::vkMergePipelineCaches( device, dstCache, srcCacheCount, pSrcCaches);
  }
  VkResult vkMergeValidationCachesEXT( VkDevice device, VkValidationCacheEXT dstCache, uint32_t srcCacheCount, const VkValidationCacheEXT* pSrcCaches  ) const
  {
    return ::vkMergeValidationCachesEXT( device, dstCache, srcCacheCount, pSrcCaches);
  }
  void vkQueueBeginDebugUtilsLabelEXT( VkQueue queue, const VkDebugUtilsLabelEXT* pLabelInfo  ) const
  {
    return ::vkQueueBeginDebugUtilsLabelEXT( queue, pLabelInfo);
  }
  VkResult vkQueueBindSparse( VkQueue queue, uint32_t bindInfoCount, const VkBindSparseInfo* pBindInfo, VkFence fence  ) const
  {
    return ::vkQueueBindSparse( queue, bindInfoCount, pBindInfo, fence);
  }
  void vkQueueEndDebugUtilsLabelEXT( VkQueue queue  ) const
  {
    return ::vkQueueEndDebugUtilsLabelEXT( queue);
  }
  void vkQueueInsertDebugUtilsLabelEXT( VkQueue queue, const VkDebugUtilsLabelEXT* pLabelInfo  ) const
  {
    return ::vkQueueInsertDebugUtilsLabelEXT( queue, pLabelInfo);
  }
  VkResult vkQueuePresentKHR( VkQueue queue, const VkPresentInfoKHR* pPresentInfo  ) const
  {
    return ::vkQueuePresentKHR( queue, pPresentInfo);
  }
  VkResult vkQueueSubmit( VkQueue queue, uint32_t submitCount, const VkSubmitInfo* pSubmits, VkFence fence  ) const
  {
    return ::vkQueueSubmit( queue, submitCount, pSubmits, fence);
  }
  VkResult vkQueueWaitIdle( VkQueue queue  ) const
  {
    return ::vkQueueWaitIdle( queue);
  }
  VkResult vkRegisterDeviceEventEXT( VkDevice device, const VkDeviceEventInfoEXT* pDeviceEventInfo, const VkAllocationCallbacks* pAllocator, VkFence* pFence  ) const
  {
    return ::vkRegisterDeviceEventEXT( device, pDeviceEventInfo, pAllocator, pFence);
  }
  VkResult vkRegisterDisplayEventEXT( VkDevice device, VkDisplayKHR display, const VkDisplayEventInfoEXT* pDisplayEventInfo, const VkAllocationCallbacks* pAllocator, VkFence* pFence  ) const
  {
    return ::vkRegisterDisplayEventEXT( device, display, pDisplayEventInfo, pAllocator, pFence);
  }
  VkResult vkRegisterObjectsNVX( VkDevice device, VkObjectTableNVX objectTable, uint32_t objectCount, const VkObjectTableEntryNVX* const* ppObjectTableEntries, const uint32_t* pObjectIndices  ) const
  {
    return ::vkRegisterObjectsNVX( device, objectTable, objectCount, ppObjectTableEntries, pObjectIndices);
  }
  VkResult vkReleaseDisplayEXT( VkPhysicalDevice physicalDevice, VkDisplayKHR display  ) const
  {
    return ::vkReleaseDisplayEXT( physicalDevice, display);
  }
  VkResult vkResetCommandBuffer( VkCommandBuffer commandBuffer, VkCommandBufferResetFlags flags  ) const
  {
    return ::vkResetCommandBuffer( commandBuffer, flags);
  }
  VkResult vkResetCommandPool( VkDevice device, VkCommandPool commandPool, VkCommandPoolResetFlags flags  ) const
  {
    return ::vkResetCommandPool( device, commandPool, flags);
  }
  VkResult vkResetDescriptorPool( VkDevice device, VkDescriptorPool descriptorPool, VkDescriptorPoolResetFlags flags  ) const
  {
    return ::vkResetDescriptorPool( device, descriptorPool, flags);
  }
  VkResult vkResetEvent( VkDevice device, VkEvent event  ) const
  {
    return ::vkResetEvent( device, event);
  }
  VkResult vkResetFences( VkDevice device, uint32_t fenceCount, const VkFence* pFences  ) const
  {
    return ::vkResetFences( device, fenceCount, pFences);
  }
  VkResult vkSetDebugUtilsObjectNameEXT( VkDevice device, const VkDebugUtilsObjectNameInfoEXT* pNameInfo  ) const
  {
    return ::vkSetDebugUtilsObjectNameEXT( device, pNameInfo);
  }
  VkResult vkSetDebugUtilsObjectTagEXT( VkDevice device, const VkDebugUtilsObjectTagInfoEXT* pTagInfo  ) const
  {
    return ::vkSetDebugUtilsObjectTagEXT( device, pTagInfo);
  }
  VkResult vkSetEvent( VkDevice device, VkEvent event  ) const
  {
    return ::vkSetEvent( device, event);
  }
  void vkSetHdrMetadataEXT( VkDevice device, uint32_t swapchainCount, const VkSwapchainKHR* pSwapchains, const VkHdrMetadataEXT* pMetadata  ) const
  {
    return ::vkSetHdrMetadataEXT( device, swapchainCount, pSwapchains, pMetadata);
  }
  void vkSubmitDebugUtilsMessageEXT( VkInstance instance, VkDebugUtilsMessageSeverityFlagBitsEXT messageSeverity, VkDebugUtilsMessageTypeFlagsEXT messageTypes, const VkDebugUtilsMessengerCallbackDataEXT* pCallbackData  ) const
  {
    return ::vkSubmitDebugUtilsMessageEXT( instance, messageSeverity, messageTypes, pCallbackData);
  }
  void vkTrimCommandPool( VkDevice device, VkCommandPool commandPool, VkCommandPoolTrimFlags flags  ) const
  {
    return ::vkTrimCommandPool( device, commandPool, flags);
  }
  void vkTrimCommandPoolKHR( VkDevice device, VkCommandPool commandPool, VkCommandPoolTrimFlags flags  ) const
  {
    return ::vkTrimCommandPoolKHR( device, commandPool, flags);
  }
  void vkUnmapMemory( VkDevice device, VkDeviceMemory memory  ) const
  {
    return ::vkUnmapMemory( device, memory);
  }
  VkResult vkUnregisterObjectsNVX( VkDevice device, VkObjectTableNVX objectTable, uint32_t objectCount, const VkObjectEntryTypeNVX* pObjectEntryTypes, const uint32_t* pObjectIndices  ) const
  {
    return ::vkUnregisterObjectsNVX( device, objectTable, objectCount, pObjectEntryTypes, pObjectIndices);
  }
  void vkUpdateDescriptorSetWithTemplate( VkDevice device, VkDescriptorSet descriptorSet, VkDescriptorUpdateTemplate descriptorUpdateTemplate, const void* pData  ) const
  {
    return ::vkUpdateDescriptorSetWithTemplate( device, descriptorSet, descriptorUpdateTemplate, pData);
  }
  void vkUpdateDescriptorSetWithTemplateKHR( VkDevice device, VkDescriptorSet descriptorSet, VkDescriptorUpdateTemplate descriptorUpdateTemplate, const void* pData  ) const
  {
    return ::vkUpdateDescriptorSetWithTemplateKHR( device, descriptorSet, descriptorUpdateTemplate, pData);
  }
  void vkUpdateDescriptorSets( VkDevice device, uint32_t descriptorWriteCount, const VkWriteDescriptorSet* pDescriptorWrites, uint32_t descriptorCopyCount, const VkCopyDescriptorSet* pDescriptorCopies  ) const
  {
    return ::vkUpdateDescriptorSets( device, descriptorWriteCount, pDescriptorWrites, descriptorCopyCount, pDescriptorCopies);
  }
  VkResult vkWaitForFences( VkDevice device, uint32_t fenceCount, const VkFence* pFences, VkBool32 waitAll, uint64_t timeout  ) const
  {
    return ::vkWaitForFences( device, fenceCount, pFences, waitAll, timeout);
  }
};

  struct AllocationCallbacks;

  template <typename OwnerType, typename Dispatch>
  class ObjectDestroy
  {
    public:
      ObjectDestroy( OwnerType owner = OwnerType(), Optional<const AllocationCallbacks> allocationCallbacks = nullptr, Dispatch const &dispatch = Dispatch() )
        : m_owner( owner )
        , m_allocationCallbacks( allocationCallbacks )
        , m_dispatch( &dispatch )
      {}

      OwnerType getOwner() const { return m_owner; }
      Optional<const AllocationCallbacks> getAllocator() const { return m_allocationCallbacks; }

    protected:
      template <typename T>
      void destroy(T t)
      {
        m_owner.destroy( t, m_allocationCallbacks, *m_dispatch );
      }

    private:
      OwnerType m_owner;
      Optional<const AllocationCallbacks> m_allocationCallbacks;
      Dispatch const* m_dispatch;
  };

  class NoParent;

  template <typename Dispatch>
  class ObjectDestroy<NoParent,Dispatch>
  {
    public:
      ObjectDestroy( Optional<const AllocationCallbacks> allocationCallbacks = nullptr, Dispatch const &dispatch = Dispatch() )
        : m_allocationCallbacks( allocationCallbacks )
        , m_dispatch( &dispatch )
      {}

      Optional<const AllocationCallbacks> getAllocator() const { return m_allocationCallbacks; }

    protected:
      template <typename T>
      void destroy(T t)
      {
        t.destroy( m_allocationCallbacks, *m_dispatch );
      }

    private:
      Optional<const AllocationCallbacks> m_allocationCallbacks;
      Dispatch const* m_dispatch;
  };

  template <typename OwnerType, typename Dispatch>
  class ObjectFree
  {
    public:
      ObjectFree( OwnerType owner = OwnerType(), Optional<const AllocationCallbacks> allocationCallbacks = nullptr, Dispatch const &dispatch = Dispatch() )
        : m_owner( owner )
        , m_allocationCallbacks( allocationCallbacks )
        , m_dispatch( &dispatch )
      {}

      OwnerType getOwner() const { return m_owner; }
      Optional<const AllocationCallbacks> getAllocator() const { return m_allocationCallbacks; }

    protected:
      template <typename T>
      void destroy(T t)
      {
        m_owner.free( t, m_allocationCallbacks, *m_dispatch );
      }

    private:
      OwnerType m_owner;
      Optional<const AllocationCallbacks> m_allocationCallbacks;
      Dispatch const* m_dispatch;
  };

  template <typename OwnerType, typename PoolType, typename Dispatch>
  class PoolFree
  {
    public:
      PoolFree( OwnerType owner = OwnerType(), PoolType pool = PoolType(), Dispatch const &dispatch = Dispatch() )
        : m_owner( owner )
        , m_pool( pool )
        , m_dispatch( &dispatch )
      {}

      OwnerType getOwner() const { return m_owner; }
      PoolType getPool() const { return m_pool; }

    protected:
      template <typename T>
      void destroy(T t)
      {
        m_owner.free( m_pool, t, *m_dispatch );
      }

    private:
      OwnerType m_owner;
      PoolType m_pool;
      Dispatch const* m_dispatch;
  };

  using SampleMask = uint32_t;

  using Bool32 = uint32_t;

  using DeviceSize = uint64_t;

  using DeviceAddress = uint64_t;

  enum class FramebufferCreateFlagBits
  {
  };

  using FramebufferCreateFlags = Flags<FramebufferCreateFlagBits, VkFramebufferCreateFlags>;

  enum class QueryPoolCreateFlagBits
  {
  };

  using QueryPoolCreateFlags = Flags<QueryPoolCreateFlagBits, VkQueryPoolCreateFlags>;

  enum class RenderPassCreateFlagBits
  {
  };

  using RenderPassCreateFlags = Flags<RenderPassCreateFlagBits, VkRenderPassCreateFlags>;

  enum class PipelineLayoutCreateFlagBits
  {
  };

  using PipelineLayoutCreateFlags = Flags<PipelineLayoutCreateFlagBits, VkPipelineLayoutCreateFlags>;

  enum class PipelineCacheCreateFlagBits
  {
  };

  using PipelineCacheCreateFlags = Flags<PipelineCacheCreateFlagBits, VkPipelineCacheCreateFlags>;

  enum class PipelineDepthStencilStateCreateFlagBits
  {
  };

  using PipelineDepthStencilStateCreateFlags = Flags<PipelineDepthStencilStateCreateFlagBits, VkPipelineDepthStencilStateCreateFlags>;

  enum class PipelineDynamicStateCreateFlagBits
  {
  };

  using PipelineDynamicStateCreateFlags = Flags<PipelineDynamicStateCreateFlagBits, VkPipelineDynamicStateCreateFlags>;

  enum class PipelineColorBlendStateCreateFlagBits
  {
  };

  using PipelineColorBlendStateCreateFlags = Flags<PipelineColorBlendStateCreateFlagBits, VkPipelineColorBlendStateCreateFlags>;

  enum class PipelineMultisampleStateCreateFlagBits
  {
  };

  using PipelineMultisampleStateCreateFlags = Flags<PipelineMultisampleStateCreateFlagBits, VkPipelineMultisampleStateCreateFlags>;

  enum class PipelineRasterizationStateCreateFlagBits
  {
  };

  using PipelineRasterizationStateCreateFlags = Flags<PipelineRasterizationStateCreateFlagBits, VkPipelineRasterizationStateCreateFlags>;

  enum class PipelineViewportStateCreateFlagBits
  {
  };

  using PipelineViewportStateCreateFlags = Flags<PipelineViewportStateCreateFlagBits, VkPipelineViewportStateCreateFlags>;

  enum class PipelineTessellationStateCreateFlagBits
  {
  };

  using PipelineTessellationStateCreateFlags = Flags<PipelineTessellationStateCreateFlagBits, VkPipelineTessellationStateCreateFlags>;

  enum class PipelineInputAssemblyStateCreateFlagBits
  {
  };

  using PipelineInputAssemblyStateCreateFlags = Flags<PipelineInputAssemblyStateCreateFlagBits, VkPipelineInputAssemblyStateCreateFlags>;

  enum class PipelineVertexInputStateCreateFlagBits
  {
  };

  using PipelineVertexInputStateCreateFlags = Flags<PipelineVertexInputStateCreateFlagBits, VkPipelineVertexInputStateCreateFlags>;

  enum class PipelineShaderStageCreateFlagBits
  {
  };

  using PipelineShaderStageCreateFlags = Flags<PipelineShaderStageCreateFlagBits, VkPipelineShaderStageCreateFlags>;

  enum class BufferViewCreateFlagBits
  {
  };

  using BufferViewCreateFlags = Flags<BufferViewCreateFlagBits, VkBufferViewCreateFlags>;

  enum class InstanceCreateFlagBits
  {
  };

  using InstanceCreateFlags = Flags<InstanceCreateFlagBits, VkInstanceCreateFlags>;

  enum class DeviceCreateFlagBits
  {
  };

  using DeviceCreateFlags = Flags<DeviceCreateFlagBits, VkDeviceCreateFlags>;

  enum class SemaphoreCreateFlagBits
  {
  };

  using SemaphoreCreateFlags = Flags<SemaphoreCreateFlagBits, VkSemaphoreCreateFlags>;

  enum class ShaderModuleCreateFlagBits
  {
  };

  using ShaderModuleCreateFlags = Flags<ShaderModuleCreateFlagBits, VkShaderModuleCreateFlags>;

  enum class EventCreateFlagBits
  {
  };

  using EventCreateFlags = Flags<EventCreateFlagBits, VkEventCreateFlags>;

  enum class MemoryMapFlagBits
  {
  };

  using MemoryMapFlags = Flags<MemoryMapFlagBits, VkMemoryMapFlags>;

  enum class DescriptorPoolResetFlagBits
  {
  };

  using DescriptorPoolResetFlags = Flags<DescriptorPoolResetFlagBits, VkDescriptorPoolResetFlags>;

  enum class DescriptorUpdateTemplateCreateFlagBits
  {
  };

  using DescriptorUpdateTemplateCreateFlags = Flags<DescriptorUpdateTemplateCreateFlagBits, VkDescriptorUpdateTemplateCreateFlags>;

  using DescriptorUpdateTemplateCreateFlagsKHR = DescriptorUpdateTemplateCreateFlags;

  enum class DisplayModeCreateFlagBitsKHR
  {
  };

  using DisplayModeCreateFlagsKHR = Flags<DisplayModeCreateFlagBitsKHR, VkDisplayModeCreateFlagsKHR>;

  enum class DisplaySurfaceCreateFlagBitsKHR
  {
  };

  using DisplaySurfaceCreateFlagsKHR = Flags<DisplaySurfaceCreateFlagBitsKHR, VkDisplaySurfaceCreateFlagsKHR>;

#ifdef VK_USE_PLATFORM_ANDROID_KHR
  enum class AndroidSurfaceCreateFlagBitsKHR
  {
  };
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

#ifdef VK_USE_PLATFORM_ANDROID_KHR
  using AndroidSurfaceCreateFlagsKHR = Flags<AndroidSurfaceCreateFlagBitsKHR, VkAndroidSurfaceCreateFlagsKHR>;
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

#ifdef VK_USE_PLATFORM_VI_NN
  enum class ViSurfaceCreateFlagBitsNN
  {
  };
#endif /*VK_USE_PLATFORM_VI_NN*/

#ifdef VK_USE_PLATFORM_VI_NN
  using ViSurfaceCreateFlagsNN = Flags<ViSurfaceCreateFlagBitsNN, VkViSurfaceCreateFlagsNN>;
#endif /*VK_USE_PLATFORM_VI_NN*/

#ifdef VK_USE_PLATFORM_WAYLAND_KHR
  enum class WaylandSurfaceCreateFlagBitsKHR
  {
  };
#endif /*VK_USE_PLATFORM_WAYLAND_KHR*/

#ifdef VK_USE_PLATFORM_WAYLAND_KHR
  using WaylandSurfaceCreateFlagsKHR = Flags<WaylandSurfaceCreateFlagBitsKHR, VkWaylandSurfaceCreateFlagsKHR>;
#endif /*VK_USE_PLATFORM_WAYLAND_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
  enum class Win32SurfaceCreateFlagBitsKHR
  {
  };
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
  using Win32SurfaceCreateFlagsKHR = Flags<Win32SurfaceCreateFlagBitsKHR, VkWin32SurfaceCreateFlagsKHR>;
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_XLIB_KHR
  enum class XlibSurfaceCreateFlagBitsKHR
  {
  };
#endif /*VK_USE_PLATFORM_XLIB_KHR*/

#ifdef VK_USE_PLATFORM_XLIB_KHR
  using XlibSurfaceCreateFlagsKHR = Flags<XlibSurfaceCreateFlagBitsKHR, VkXlibSurfaceCreateFlagsKHR>;
#endif /*VK_USE_PLATFORM_XLIB_KHR*/

#ifdef VK_USE_PLATFORM_XCB_KHR
  enum class XcbSurfaceCreateFlagBitsKHR
  {
  };
#endif /*VK_USE_PLATFORM_XCB_KHR*/

#ifdef VK_USE_PLATFORM_XCB_KHR
  using XcbSurfaceCreateFlagsKHR = Flags<XcbSurfaceCreateFlagBitsKHR, VkXcbSurfaceCreateFlagsKHR>;
#endif /*VK_USE_PLATFORM_XCB_KHR*/

#ifdef VK_USE_PLATFORM_IOS_MVK
  enum class IOSSurfaceCreateFlagBitsMVK
  {
  };
#endif /*VK_USE_PLATFORM_IOS_MVK*/

#ifdef VK_USE_PLATFORM_IOS_MVK
  using IOSSurfaceCreateFlagsMVK = Flags<IOSSurfaceCreateFlagBitsMVK, VkIOSSurfaceCreateFlagsMVK>;
#endif /*VK_USE_PLATFORM_IOS_MVK*/

#ifdef VK_USE_PLATFORM_MACOS_MVK
  enum class MacOSSurfaceCreateFlagBitsMVK
  {
  };
#endif /*VK_USE_PLATFORM_MACOS_MVK*/

#ifdef VK_USE_PLATFORM_MACOS_MVK
  using MacOSSurfaceCreateFlagsMVK = Flags<MacOSSurfaceCreateFlagBitsMVK, VkMacOSSurfaceCreateFlagsMVK>;
#endif /*VK_USE_PLATFORM_MACOS_MVK*/

#ifdef VK_USE_PLATFORM_FUCHSIA
  enum class ImagePipeSurfaceCreateFlagBitsFUCHSIA
  {
  };
#endif /*VK_USE_PLATFORM_FUCHSIA*/

#ifdef VK_USE_PLATFORM_FUCHSIA
  using ImagePipeSurfaceCreateFlagsFUCHSIA = Flags<ImagePipeSurfaceCreateFlagBitsFUCHSIA, VkImagePipeSurfaceCreateFlagsFUCHSIA>;
#endif /*VK_USE_PLATFORM_FUCHSIA*/

  enum class CommandPoolTrimFlagBits
  {
  };

  using CommandPoolTrimFlags = Flags<CommandPoolTrimFlagBits, VkCommandPoolTrimFlags>;

  using CommandPoolTrimFlagsKHR = CommandPoolTrimFlags;

  enum class PipelineViewportSwizzleStateCreateFlagBitsNV
  {
  };

  using PipelineViewportSwizzleStateCreateFlagsNV = Flags<PipelineViewportSwizzleStateCreateFlagBitsNV, VkPipelineViewportSwizzleStateCreateFlagsNV>;

  enum class PipelineDiscardRectangleStateCreateFlagBitsEXT
  {
  };

  using PipelineDiscardRectangleStateCreateFlagsEXT = Flags<PipelineDiscardRectangleStateCreateFlagBitsEXT, VkPipelineDiscardRectangleStateCreateFlagsEXT>;

  enum class PipelineCoverageToColorStateCreateFlagBitsNV
  {
  };

  using PipelineCoverageToColorStateCreateFlagsNV = Flags<PipelineCoverageToColorStateCreateFlagBitsNV, VkPipelineCoverageToColorStateCreateFlagsNV>;

  enum class PipelineCoverageModulationStateCreateFlagBitsNV
  {
  };

  using PipelineCoverageModulationStateCreateFlagsNV = Flags<PipelineCoverageModulationStateCreateFlagBitsNV, VkPipelineCoverageModulationStateCreateFlagsNV>;

  enum class ValidationCacheCreateFlagBitsEXT
  {
  };

  using ValidationCacheCreateFlagsEXT = Flags<ValidationCacheCreateFlagBitsEXT, VkValidationCacheCreateFlagsEXT>;

  enum class DebugUtilsMessengerCreateFlagBitsEXT
  {
  };

  using DebugUtilsMessengerCreateFlagsEXT = Flags<DebugUtilsMessengerCreateFlagBitsEXT, VkDebugUtilsMessengerCreateFlagsEXT>;

  enum class DebugUtilsMessengerCallbackDataFlagBitsEXT
  {
  };

  using DebugUtilsMessengerCallbackDataFlagsEXT = Flags<DebugUtilsMessengerCallbackDataFlagBitsEXT, VkDebugUtilsMessengerCallbackDataFlagsEXT>;

  enum class PipelineRasterizationConservativeStateCreateFlagBitsEXT
  {
  };

  using PipelineRasterizationConservativeStateCreateFlagsEXT = Flags<PipelineRasterizationConservativeStateCreateFlagBitsEXT, VkPipelineRasterizationConservativeStateCreateFlagsEXT>;

  enum class PipelineRasterizationStateStreamCreateFlagBitsEXT
  {
  };

  using PipelineRasterizationStateStreamCreateFlagsEXT = Flags<PipelineRasterizationStateStreamCreateFlagBitsEXT, VkPipelineRasterizationStateStreamCreateFlagsEXT>;

  class DeviceMemory
  {
  public:
    VULKAN_HPP_CONSTEXPR DeviceMemory()
      : m_deviceMemory(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DeviceMemory( std::nullptr_t )
      : m_deviceMemory(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DeviceMemory( VkDeviceMemory deviceMemory )
      : m_deviceMemory( deviceMemory )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DeviceMemory & operator=(VkDeviceMemory deviceMemory)
    {
      m_deviceMemory = deviceMemory;
      return *this; 
    }
#endif

    DeviceMemory & operator=( std::nullptr_t )
    {
      m_deviceMemory = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DeviceMemory const & rhs ) const
    {
      return m_deviceMemory == rhs.m_deviceMemory;
    }

    bool operator!=(DeviceMemory const & rhs ) const
    {
      return m_deviceMemory != rhs.m_deviceMemory;
    }

    bool operator<(DeviceMemory const & rhs ) const
    {
      return m_deviceMemory < rhs.m_deviceMemory;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDeviceMemory() const
    {
      return m_deviceMemory;
    }

    explicit operator bool() const
    {
      return m_deviceMemory != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_deviceMemory == VK_NULL_HANDLE;
    }

  private:
    VkDeviceMemory m_deviceMemory;
  };

  static_assert( sizeof( DeviceMemory ) == sizeof( VkDeviceMemory ), "handle and wrapper have different size!" );

  class CommandPool
  {
  public:
    VULKAN_HPP_CONSTEXPR CommandPool()
      : m_commandPool(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR CommandPool( std::nullptr_t )
      : m_commandPool(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT CommandPool( VkCommandPool commandPool )
      : m_commandPool( commandPool )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    CommandPool & operator=(VkCommandPool commandPool)
    {
      m_commandPool = commandPool;
      return *this; 
    }
#endif

    CommandPool & operator=( std::nullptr_t )
    {
      m_commandPool = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( CommandPool const & rhs ) const
    {
      return m_commandPool == rhs.m_commandPool;
    }

    bool operator!=(CommandPool const & rhs ) const
    {
      return m_commandPool != rhs.m_commandPool;
    }

    bool operator<(CommandPool const & rhs ) const
    {
      return m_commandPool < rhs.m_commandPool;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkCommandPool() const
    {
      return m_commandPool;
    }

    explicit operator bool() const
    {
      return m_commandPool != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_commandPool == VK_NULL_HANDLE;
    }

  private:
    VkCommandPool m_commandPool;
  };

  static_assert( sizeof( CommandPool ) == sizeof( VkCommandPool ), "handle and wrapper have different size!" );

  class Buffer
  {
  public:
    VULKAN_HPP_CONSTEXPR Buffer()
      : m_buffer(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Buffer( std::nullptr_t )
      : m_buffer(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Buffer( VkBuffer buffer )
      : m_buffer( buffer )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Buffer & operator=(VkBuffer buffer)
    {
      m_buffer = buffer;
      return *this; 
    }
#endif

    Buffer & operator=( std::nullptr_t )
    {
      m_buffer = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Buffer const & rhs ) const
    {
      return m_buffer == rhs.m_buffer;
    }

    bool operator!=(Buffer const & rhs ) const
    {
      return m_buffer != rhs.m_buffer;
    }

    bool operator<(Buffer const & rhs ) const
    {
      return m_buffer < rhs.m_buffer;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkBuffer() const
    {
      return m_buffer;
    }

    explicit operator bool() const
    {
      return m_buffer != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_buffer == VK_NULL_HANDLE;
    }

  private:
    VkBuffer m_buffer;
  };

  static_assert( sizeof( Buffer ) == sizeof( VkBuffer ), "handle and wrapper have different size!" );

  class BufferView
  {
  public:
    VULKAN_HPP_CONSTEXPR BufferView()
      : m_bufferView(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR BufferView( std::nullptr_t )
      : m_bufferView(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT BufferView( VkBufferView bufferView )
      : m_bufferView( bufferView )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    BufferView & operator=(VkBufferView bufferView)
    {
      m_bufferView = bufferView;
      return *this; 
    }
#endif

    BufferView & operator=( std::nullptr_t )
    {
      m_bufferView = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( BufferView const & rhs ) const
    {
      return m_bufferView == rhs.m_bufferView;
    }

    bool operator!=(BufferView const & rhs ) const
    {
      return m_bufferView != rhs.m_bufferView;
    }

    bool operator<(BufferView const & rhs ) const
    {
      return m_bufferView < rhs.m_bufferView;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkBufferView() const
    {
      return m_bufferView;
    }

    explicit operator bool() const
    {
      return m_bufferView != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_bufferView == VK_NULL_HANDLE;
    }

  private:
    VkBufferView m_bufferView;
  };

  static_assert( sizeof( BufferView ) == sizeof( VkBufferView ), "handle and wrapper have different size!" );

  class Image
  {
  public:
    VULKAN_HPP_CONSTEXPR Image()
      : m_image(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Image( std::nullptr_t )
      : m_image(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Image( VkImage image )
      : m_image( image )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Image & operator=(VkImage image)
    {
      m_image = image;
      return *this; 
    }
#endif

    Image & operator=( std::nullptr_t )
    {
      m_image = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Image const & rhs ) const
    {
      return m_image == rhs.m_image;
    }

    bool operator!=(Image const & rhs ) const
    {
      return m_image != rhs.m_image;
    }

    bool operator<(Image const & rhs ) const
    {
      return m_image < rhs.m_image;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkImage() const
    {
      return m_image;
    }

    explicit operator bool() const
    {
      return m_image != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_image == VK_NULL_HANDLE;
    }

  private:
    VkImage m_image;
  };

  static_assert( sizeof( Image ) == sizeof( VkImage ), "handle and wrapper have different size!" );

  class ImageView
  {
  public:
    VULKAN_HPP_CONSTEXPR ImageView()
      : m_imageView(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR ImageView( std::nullptr_t )
      : m_imageView(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT ImageView( VkImageView imageView )
      : m_imageView( imageView )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    ImageView & operator=(VkImageView imageView)
    {
      m_imageView = imageView;
      return *this; 
    }
#endif

    ImageView & operator=( std::nullptr_t )
    {
      m_imageView = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( ImageView const & rhs ) const
    {
      return m_imageView == rhs.m_imageView;
    }

    bool operator!=(ImageView const & rhs ) const
    {
      return m_imageView != rhs.m_imageView;
    }

    bool operator<(ImageView const & rhs ) const
    {
      return m_imageView < rhs.m_imageView;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkImageView() const
    {
      return m_imageView;
    }

    explicit operator bool() const
    {
      return m_imageView != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_imageView == VK_NULL_HANDLE;
    }

  private:
    VkImageView m_imageView;
  };

  static_assert( sizeof( ImageView ) == sizeof( VkImageView ), "handle and wrapper have different size!" );

  class ShaderModule
  {
  public:
    VULKAN_HPP_CONSTEXPR ShaderModule()
      : m_shaderModule(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR ShaderModule( std::nullptr_t )
      : m_shaderModule(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT ShaderModule( VkShaderModule shaderModule )
      : m_shaderModule( shaderModule )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    ShaderModule & operator=(VkShaderModule shaderModule)
    {
      m_shaderModule = shaderModule;
      return *this; 
    }
#endif

    ShaderModule & operator=( std::nullptr_t )
    {
      m_shaderModule = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( ShaderModule const & rhs ) const
    {
      return m_shaderModule == rhs.m_shaderModule;
    }

    bool operator!=(ShaderModule const & rhs ) const
    {
      return m_shaderModule != rhs.m_shaderModule;
    }

    bool operator<(ShaderModule const & rhs ) const
    {
      return m_shaderModule < rhs.m_shaderModule;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkShaderModule() const
    {
      return m_shaderModule;
    }

    explicit operator bool() const
    {
      return m_shaderModule != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_shaderModule == VK_NULL_HANDLE;
    }

  private:
    VkShaderModule m_shaderModule;
  };

  static_assert( sizeof( ShaderModule ) == sizeof( VkShaderModule ), "handle and wrapper have different size!" );

  class Pipeline
  {
  public:
    VULKAN_HPP_CONSTEXPR Pipeline()
      : m_pipeline(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Pipeline( std::nullptr_t )
      : m_pipeline(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Pipeline( VkPipeline pipeline )
      : m_pipeline( pipeline )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Pipeline & operator=(VkPipeline pipeline)
    {
      m_pipeline = pipeline;
      return *this; 
    }
#endif

    Pipeline & operator=( std::nullptr_t )
    {
      m_pipeline = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Pipeline const & rhs ) const
    {
      return m_pipeline == rhs.m_pipeline;
    }

    bool operator!=(Pipeline const & rhs ) const
    {
      return m_pipeline != rhs.m_pipeline;
    }

    bool operator<(Pipeline const & rhs ) const
    {
      return m_pipeline < rhs.m_pipeline;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkPipeline() const
    {
      return m_pipeline;
    }

    explicit operator bool() const
    {
      return m_pipeline != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_pipeline == VK_NULL_HANDLE;
    }

  private:
    VkPipeline m_pipeline;
  };

  static_assert( sizeof( Pipeline ) == sizeof( VkPipeline ), "handle and wrapper have different size!" );

  class PipelineLayout
  {
  public:
    VULKAN_HPP_CONSTEXPR PipelineLayout()
      : m_pipelineLayout(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR PipelineLayout( std::nullptr_t )
      : m_pipelineLayout(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT PipelineLayout( VkPipelineLayout pipelineLayout )
      : m_pipelineLayout( pipelineLayout )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    PipelineLayout & operator=(VkPipelineLayout pipelineLayout)
    {
      m_pipelineLayout = pipelineLayout;
      return *this; 
    }
#endif

    PipelineLayout & operator=( std::nullptr_t )
    {
      m_pipelineLayout = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( PipelineLayout const & rhs ) const
    {
      return m_pipelineLayout == rhs.m_pipelineLayout;
    }

    bool operator!=(PipelineLayout const & rhs ) const
    {
      return m_pipelineLayout != rhs.m_pipelineLayout;
    }

    bool operator<(PipelineLayout const & rhs ) const
    {
      return m_pipelineLayout < rhs.m_pipelineLayout;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkPipelineLayout() const
    {
      return m_pipelineLayout;
    }

    explicit operator bool() const
    {
      return m_pipelineLayout != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_pipelineLayout == VK_NULL_HANDLE;
    }

  private:
    VkPipelineLayout m_pipelineLayout;
  };

  static_assert( sizeof( PipelineLayout ) == sizeof( VkPipelineLayout ), "handle and wrapper have different size!" );

  class Sampler
  {
  public:
    VULKAN_HPP_CONSTEXPR Sampler()
      : m_sampler(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Sampler( std::nullptr_t )
      : m_sampler(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Sampler( VkSampler sampler )
      : m_sampler( sampler )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Sampler & operator=(VkSampler sampler)
    {
      m_sampler = sampler;
      return *this; 
    }
#endif

    Sampler & operator=( std::nullptr_t )
    {
      m_sampler = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Sampler const & rhs ) const
    {
      return m_sampler == rhs.m_sampler;
    }

    bool operator!=(Sampler const & rhs ) const
    {
      return m_sampler != rhs.m_sampler;
    }

    bool operator<(Sampler const & rhs ) const
    {
      return m_sampler < rhs.m_sampler;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkSampler() const
    {
      return m_sampler;
    }

    explicit operator bool() const
    {
      return m_sampler != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_sampler == VK_NULL_HANDLE;
    }

  private:
    VkSampler m_sampler;
  };

  static_assert( sizeof( Sampler ) == sizeof( VkSampler ), "handle and wrapper have different size!" );

  class DescriptorSet
  {
  public:
    VULKAN_HPP_CONSTEXPR DescriptorSet()
      : m_descriptorSet(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DescriptorSet( std::nullptr_t )
      : m_descriptorSet(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DescriptorSet( VkDescriptorSet descriptorSet )
      : m_descriptorSet( descriptorSet )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DescriptorSet & operator=(VkDescriptorSet descriptorSet)
    {
      m_descriptorSet = descriptorSet;
      return *this; 
    }
#endif

    DescriptorSet & operator=( std::nullptr_t )
    {
      m_descriptorSet = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DescriptorSet const & rhs ) const
    {
      return m_descriptorSet == rhs.m_descriptorSet;
    }

    bool operator!=(DescriptorSet const & rhs ) const
    {
      return m_descriptorSet != rhs.m_descriptorSet;
    }

    bool operator<(DescriptorSet const & rhs ) const
    {
      return m_descriptorSet < rhs.m_descriptorSet;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDescriptorSet() const
    {
      return m_descriptorSet;
    }

    explicit operator bool() const
    {
      return m_descriptorSet != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_descriptorSet == VK_NULL_HANDLE;
    }

  private:
    VkDescriptorSet m_descriptorSet;
  };

  static_assert( sizeof( DescriptorSet ) == sizeof( VkDescriptorSet ), "handle and wrapper have different size!" );

  class DescriptorSetLayout
  {
  public:
    VULKAN_HPP_CONSTEXPR DescriptorSetLayout()
      : m_descriptorSetLayout(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DescriptorSetLayout( std::nullptr_t )
      : m_descriptorSetLayout(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DescriptorSetLayout( VkDescriptorSetLayout descriptorSetLayout )
      : m_descriptorSetLayout( descriptorSetLayout )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DescriptorSetLayout & operator=(VkDescriptorSetLayout descriptorSetLayout)
    {
      m_descriptorSetLayout = descriptorSetLayout;
      return *this; 
    }
#endif

    DescriptorSetLayout & operator=( std::nullptr_t )
    {
      m_descriptorSetLayout = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DescriptorSetLayout const & rhs ) const
    {
      return m_descriptorSetLayout == rhs.m_descriptorSetLayout;
    }

    bool operator!=(DescriptorSetLayout const & rhs ) const
    {
      return m_descriptorSetLayout != rhs.m_descriptorSetLayout;
    }

    bool operator<(DescriptorSetLayout const & rhs ) const
    {
      return m_descriptorSetLayout < rhs.m_descriptorSetLayout;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDescriptorSetLayout() const
    {
      return m_descriptorSetLayout;
    }

    explicit operator bool() const
    {
      return m_descriptorSetLayout != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_descriptorSetLayout == VK_NULL_HANDLE;
    }

  private:
    VkDescriptorSetLayout m_descriptorSetLayout;
  };

  static_assert( sizeof( DescriptorSetLayout ) == sizeof( VkDescriptorSetLayout ), "handle and wrapper have different size!" );

  class DescriptorPool
  {
  public:
    VULKAN_HPP_CONSTEXPR DescriptorPool()
      : m_descriptorPool(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DescriptorPool( std::nullptr_t )
      : m_descriptorPool(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DescriptorPool( VkDescriptorPool descriptorPool )
      : m_descriptorPool( descriptorPool )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DescriptorPool & operator=(VkDescriptorPool descriptorPool)
    {
      m_descriptorPool = descriptorPool;
      return *this; 
    }
#endif

    DescriptorPool & operator=( std::nullptr_t )
    {
      m_descriptorPool = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DescriptorPool const & rhs ) const
    {
      return m_descriptorPool == rhs.m_descriptorPool;
    }

    bool operator!=(DescriptorPool const & rhs ) const
    {
      return m_descriptorPool != rhs.m_descriptorPool;
    }

    bool operator<(DescriptorPool const & rhs ) const
    {
      return m_descriptorPool < rhs.m_descriptorPool;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDescriptorPool() const
    {
      return m_descriptorPool;
    }

    explicit operator bool() const
    {
      return m_descriptorPool != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_descriptorPool == VK_NULL_HANDLE;
    }

  private:
    VkDescriptorPool m_descriptorPool;
  };

  static_assert( sizeof( DescriptorPool ) == sizeof( VkDescriptorPool ), "handle and wrapper have different size!" );

  class Fence
  {
  public:
    VULKAN_HPP_CONSTEXPR Fence()
      : m_fence(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Fence( std::nullptr_t )
      : m_fence(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Fence( VkFence fence )
      : m_fence( fence )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Fence & operator=(VkFence fence)
    {
      m_fence = fence;
      return *this; 
    }
#endif

    Fence & operator=( std::nullptr_t )
    {
      m_fence = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Fence const & rhs ) const
    {
      return m_fence == rhs.m_fence;
    }

    bool operator!=(Fence const & rhs ) const
    {
      return m_fence != rhs.m_fence;
    }

    bool operator<(Fence const & rhs ) const
    {
      return m_fence < rhs.m_fence;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkFence() const
    {
      return m_fence;
    }

    explicit operator bool() const
    {
      return m_fence != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_fence == VK_NULL_HANDLE;
    }

  private:
    VkFence m_fence;
  };

  static_assert( sizeof( Fence ) == sizeof( VkFence ), "handle and wrapper have different size!" );

  class Semaphore
  {
  public:
    VULKAN_HPP_CONSTEXPR Semaphore()
      : m_semaphore(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Semaphore( std::nullptr_t )
      : m_semaphore(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Semaphore( VkSemaphore semaphore )
      : m_semaphore( semaphore )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Semaphore & operator=(VkSemaphore semaphore)
    {
      m_semaphore = semaphore;
      return *this; 
    }
#endif

    Semaphore & operator=( std::nullptr_t )
    {
      m_semaphore = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Semaphore const & rhs ) const
    {
      return m_semaphore == rhs.m_semaphore;
    }

    bool operator!=(Semaphore const & rhs ) const
    {
      return m_semaphore != rhs.m_semaphore;
    }

    bool operator<(Semaphore const & rhs ) const
    {
      return m_semaphore < rhs.m_semaphore;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkSemaphore() const
    {
      return m_semaphore;
    }

    explicit operator bool() const
    {
      return m_semaphore != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_semaphore == VK_NULL_HANDLE;
    }

  private:
    VkSemaphore m_semaphore;
  };

  static_assert( sizeof( Semaphore ) == sizeof( VkSemaphore ), "handle and wrapper have different size!" );

  class Event
  {
  public:
    VULKAN_HPP_CONSTEXPR Event()
      : m_event(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Event( std::nullptr_t )
      : m_event(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Event( VkEvent event )
      : m_event( event )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Event & operator=(VkEvent event)
    {
      m_event = event;
      return *this; 
    }
#endif

    Event & operator=( std::nullptr_t )
    {
      m_event = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Event const & rhs ) const
    {
      return m_event == rhs.m_event;
    }

    bool operator!=(Event const & rhs ) const
    {
      return m_event != rhs.m_event;
    }

    bool operator<(Event const & rhs ) const
    {
      return m_event < rhs.m_event;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkEvent() const
    {
      return m_event;
    }

    explicit operator bool() const
    {
      return m_event != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_event == VK_NULL_HANDLE;
    }

  private:
    VkEvent m_event;
  };

  static_assert( sizeof( Event ) == sizeof( VkEvent ), "handle and wrapper have different size!" );

  class QueryPool
  {
  public:
    VULKAN_HPP_CONSTEXPR QueryPool()
      : m_queryPool(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR QueryPool( std::nullptr_t )
      : m_queryPool(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT QueryPool( VkQueryPool queryPool )
      : m_queryPool( queryPool )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    QueryPool & operator=(VkQueryPool queryPool)
    {
      m_queryPool = queryPool;
      return *this; 
    }
#endif

    QueryPool & operator=( std::nullptr_t )
    {
      m_queryPool = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( QueryPool const & rhs ) const
    {
      return m_queryPool == rhs.m_queryPool;
    }

    bool operator!=(QueryPool const & rhs ) const
    {
      return m_queryPool != rhs.m_queryPool;
    }

    bool operator<(QueryPool const & rhs ) const
    {
      return m_queryPool < rhs.m_queryPool;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkQueryPool() const
    {
      return m_queryPool;
    }

    explicit operator bool() const
    {
      return m_queryPool != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_queryPool == VK_NULL_HANDLE;
    }

  private:
    VkQueryPool m_queryPool;
  };

  static_assert( sizeof( QueryPool ) == sizeof( VkQueryPool ), "handle and wrapper have different size!" );

  class Framebuffer
  {
  public:
    VULKAN_HPP_CONSTEXPR Framebuffer()
      : m_framebuffer(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Framebuffer( std::nullptr_t )
      : m_framebuffer(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Framebuffer( VkFramebuffer framebuffer )
      : m_framebuffer( framebuffer )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Framebuffer & operator=(VkFramebuffer framebuffer)
    {
      m_framebuffer = framebuffer;
      return *this; 
    }
#endif

    Framebuffer & operator=( std::nullptr_t )
    {
      m_framebuffer = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Framebuffer const & rhs ) const
    {
      return m_framebuffer == rhs.m_framebuffer;
    }

    bool operator!=(Framebuffer const & rhs ) const
    {
      return m_framebuffer != rhs.m_framebuffer;
    }

    bool operator<(Framebuffer const & rhs ) const
    {
      return m_framebuffer < rhs.m_framebuffer;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkFramebuffer() const
    {
      return m_framebuffer;
    }

    explicit operator bool() const
    {
      return m_framebuffer != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_framebuffer == VK_NULL_HANDLE;
    }

  private:
    VkFramebuffer m_framebuffer;
  };

  static_assert( sizeof( Framebuffer ) == sizeof( VkFramebuffer ), "handle and wrapper have different size!" );

  class RenderPass
  {
  public:
    VULKAN_HPP_CONSTEXPR RenderPass()
      : m_renderPass(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR RenderPass( std::nullptr_t )
      : m_renderPass(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT RenderPass( VkRenderPass renderPass )
      : m_renderPass( renderPass )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    RenderPass & operator=(VkRenderPass renderPass)
    {
      m_renderPass = renderPass;
      return *this; 
    }
#endif

    RenderPass & operator=( std::nullptr_t )
    {
      m_renderPass = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( RenderPass const & rhs ) const
    {
      return m_renderPass == rhs.m_renderPass;
    }

    bool operator!=(RenderPass const & rhs ) const
    {
      return m_renderPass != rhs.m_renderPass;
    }

    bool operator<(RenderPass const & rhs ) const
    {
      return m_renderPass < rhs.m_renderPass;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkRenderPass() const
    {
      return m_renderPass;
    }

    explicit operator bool() const
    {
      return m_renderPass != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_renderPass == VK_NULL_HANDLE;
    }

  private:
    VkRenderPass m_renderPass;
  };

  static_assert( sizeof( RenderPass ) == sizeof( VkRenderPass ), "handle and wrapper have different size!" );

  class PipelineCache
  {
  public:
    VULKAN_HPP_CONSTEXPR PipelineCache()
      : m_pipelineCache(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR PipelineCache( std::nullptr_t )
      : m_pipelineCache(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT PipelineCache( VkPipelineCache pipelineCache )
      : m_pipelineCache( pipelineCache )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    PipelineCache & operator=(VkPipelineCache pipelineCache)
    {
      m_pipelineCache = pipelineCache;
      return *this; 
    }
#endif

    PipelineCache & operator=( std::nullptr_t )
    {
      m_pipelineCache = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( PipelineCache const & rhs ) const
    {
      return m_pipelineCache == rhs.m_pipelineCache;
    }

    bool operator!=(PipelineCache const & rhs ) const
    {
      return m_pipelineCache != rhs.m_pipelineCache;
    }

    bool operator<(PipelineCache const & rhs ) const
    {
      return m_pipelineCache < rhs.m_pipelineCache;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkPipelineCache() const
    {
      return m_pipelineCache;
    }

    explicit operator bool() const
    {
      return m_pipelineCache != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_pipelineCache == VK_NULL_HANDLE;
    }

  private:
    VkPipelineCache m_pipelineCache;
  };

  static_assert( sizeof( PipelineCache ) == sizeof( VkPipelineCache ), "handle and wrapper have different size!" );

  class ObjectTableNVX
  {
  public:
    VULKAN_HPP_CONSTEXPR ObjectTableNVX()
      : m_objectTableNVX(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR ObjectTableNVX( std::nullptr_t )
      : m_objectTableNVX(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT ObjectTableNVX( VkObjectTableNVX objectTableNVX )
      : m_objectTableNVX( objectTableNVX )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    ObjectTableNVX & operator=(VkObjectTableNVX objectTableNVX)
    {
      m_objectTableNVX = objectTableNVX;
      return *this; 
    }
#endif

    ObjectTableNVX & operator=( std::nullptr_t )
    {
      m_objectTableNVX = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( ObjectTableNVX const & rhs ) const
    {
      return m_objectTableNVX == rhs.m_objectTableNVX;
    }

    bool operator!=(ObjectTableNVX const & rhs ) const
    {
      return m_objectTableNVX != rhs.m_objectTableNVX;
    }

    bool operator<(ObjectTableNVX const & rhs ) const
    {
      return m_objectTableNVX < rhs.m_objectTableNVX;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkObjectTableNVX() const
    {
      return m_objectTableNVX;
    }

    explicit operator bool() const
    {
      return m_objectTableNVX != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_objectTableNVX == VK_NULL_HANDLE;
    }

  private:
    VkObjectTableNVX m_objectTableNVX;
  };

  static_assert( sizeof( ObjectTableNVX ) == sizeof( VkObjectTableNVX ), "handle and wrapper have different size!" );

  class IndirectCommandsLayoutNVX
  {
  public:
    VULKAN_HPP_CONSTEXPR IndirectCommandsLayoutNVX()
      : m_indirectCommandsLayoutNVX(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR IndirectCommandsLayoutNVX( std::nullptr_t )
      : m_indirectCommandsLayoutNVX(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT IndirectCommandsLayoutNVX( VkIndirectCommandsLayoutNVX indirectCommandsLayoutNVX )
      : m_indirectCommandsLayoutNVX( indirectCommandsLayoutNVX )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    IndirectCommandsLayoutNVX & operator=(VkIndirectCommandsLayoutNVX indirectCommandsLayoutNVX)
    {
      m_indirectCommandsLayoutNVX = indirectCommandsLayoutNVX;
      return *this; 
    }
#endif

    IndirectCommandsLayoutNVX & operator=( std::nullptr_t )
    {
      m_indirectCommandsLayoutNVX = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( IndirectCommandsLayoutNVX const & rhs ) const
    {
      return m_indirectCommandsLayoutNVX == rhs.m_indirectCommandsLayoutNVX;
    }

    bool operator!=(IndirectCommandsLayoutNVX const & rhs ) const
    {
      return m_indirectCommandsLayoutNVX != rhs.m_indirectCommandsLayoutNVX;
    }

    bool operator<(IndirectCommandsLayoutNVX const & rhs ) const
    {
      return m_indirectCommandsLayoutNVX < rhs.m_indirectCommandsLayoutNVX;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkIndirectCommandsLayoutNVX() const
    {
      return m_indirectCommandsLayoutNVX;
    }

    explicit operator bool() const
    {
      return m_indirectCommandsLayoutNVX != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_indirectCommandsLayoutNVX == VK_NULL_HANDLE;
    }

  private:
    VkIndirectCommandsLayoutNVX m_indirectCommandsLayoutNVX;
  };

  static_assert( sizeof( IndirectCommandsLayoutNVX ) == sizeof( VkIndirectCommandsLayoutNVX ), "handle and wrapper have different size!" );

  class DescriptorUpdateTemplate
  {
  public:
    VULKAN_HPP_CONSTEXPR DescriptorUpdateTemplate()
      : m_descriptorUpdateTemplate(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DescriptorUpdateTemplate( std::nullptr_t )
      : m_descriptorUpdateTemplate(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DescriptorUpdateTemplate( VkDescriptorUpdateTemplate descriptorUpdateTemplate )
      : m_descriptorUpdateTemplate( descriptorUpdateTemplate )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DescriptorUpdateTemplate & operator=(VkDescriptorUpdateTemplate descriptorUpdateTemplate)
    {
      m_descriptorUpdateTemplate = descriptorUpdateTemplate;
      return *this; 
    }
#endif

    DescriptorUpdateTemplate & operator=( std::nullptr_t )
    {
      m_descriptorUpdateTemplate = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DescriptorUpdateTemplate const & rhs ) const
    {
      return m_descriptorUpdateTemplate == rhs.m_descriptorUpdateTemplate;
    }

    bool operator!=(DescriptorUpdateTemplate const & rhs ) const
    {
      return m_descriptorUpdateTemplate != rhs.m_descriptorUpdateTemplate;
    }

    bool operator<(DescriptorUpdateTemplate const & rhs ) const
    {
      return m_descriptorUpdateTemplate < rhs.m_descriptorUpdateTemplate;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDescriptorUpdateTemplate() const
    {
      return m_descriptorUpdateTemplate;
    }

    explicit operator bool() const
    {
      return m_descriptorUpdateTemplate != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_descriptorUpdateTemplate == VK_NULL_HANDLE;
    }

  private:
    VkDescriptorUpdateTemplate m_descriptorUpdateTemplate;
  };

  static_assert( sizeof( DescriptorUpdateTemplate ) == sizeof( VkDescriptorUpdateTemplate ), "handle and wrapper have different size!" );

  using DescriptorUpdateTemplateKHR = DescriptorUpdateTemplate;

  class SamplerYcbcrConversion
  {
  public:
    VULKAN_HPP_CONSTEXPR SamplerYcbcrConversion()
      : m_samplerYcbcrConversion(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR SamplerYcbcrConversion( std::nullptr_t )
      : m_samplerYcbcrConversion(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT SamplerYcbcrConversion( VkSamplerYcbcrConversion samplerYcbcrConversion )
      : m_samplerYcbcrConversion( samplerYcbcrConversion )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    SamplerYcbcrConversion & operator=(VkSamplerYcbcrConversion samplerYcbcrConversion)
    {
      m_samplerYcbcrConversion = samplerYcbcrConversion;
      return *this; 
    }
#endif

    SamplerYcbcrConversion & operator=( std::nullptr_t )
    {
      m_samplerYcbcrConversion = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( SamplerYcbcrConversion const & rhs ) const
    {
      return m_samplerYcbcrConversion == rhs.m_samplerYcbcrConversion;
    }

    bool operator!=(SamplerYcbcrConversion const & rhs ) const
    {
      return m_samplerYcbcrConversion != rhs.m_samplerYcbcrConversion;
    }

    bool operator<(SamplerYcbcrConversion const & rhs ) const
    {
      return m_samplerYcbcrConversion < rhs.m_samplerYcbcrConversion;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkSamplerYcbcrConversion() const
    {
      return m_samplerYcbcrConversion;
    }

    explicit operator bool() const
    {
      return m_samplerYcbcrConversion != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_samplerYcbcrConversion == VK_NULL_HANDLE;
    }

  private:
    VkSamplerYcbcrConversion m_samplerYcbcrConversion;
  };

  static_assert( sizeof( SamplerYcbcrConversion ) == sizeof( VkSamplerYcbcrConversion ), "handle and wrapper have different size!" );

  using SamplerYcbcrConversionKHR = SamplerYcbcrConversion;

  class ValidationCacheEXT
  {
  public:
    VULKAN_HPP_CONSTEXPR ValidationCacheEXT()
      : m_validationCacheEXT(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR ValidationCacheEXT( std::nullptr_t )
      : m_validationCacheEXT(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT ValidationCacheEXT( VkValidationCacheEXT validationCacheEXT )
      : m_validationCacheEXT( validationCacheEXT )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    ValidationCacheEXT & operator=(VkValidationCacheEXT validationCacheEXT)
    {
      m_validationCacheEXT = validationCacheEXT;
      return *this; 
    }
#endif

    ValidationCacheEXT & operator=( std::nullptr_t )
    {
      m_validationCacheEXT = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( ValidationCacheEXT const & rhs ) const
    {
      return m_validationCacheEXT == rhs.m_validationCacheEXT;
    }

    bool operator!=(ValidationCacheEXT const & rhs ) const
    {
      return m_validationCacheEXT != rhs.m_validationCacheEXT;
    }

    bool operator<(ValidationCacheEXT const & rhs ) const
    {
      return m_validationCacheEXT < rhs.m_validationCacheEXT;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkValidationCacheEXT() const
    {
      return m_validationCacheEXT;
    }

    explicit operator bool() const
    {
      return m_validationCacheEXT != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_validationCacheEXT == VK_NULL_HANDLE;
    }

  private:
    VkValidationCacheEXT m_validationCacheEXT;
  };

  static_assert( sizeof( ValidationCacheEXT ) == sizeof( VkValidationCacheEXT ), "handle and wrapper have different size!" );

  class AccelerationStructureNV
  {
  public:
    VULKAN_HPP_CONSTEXPR AccelerationStructureNV()
      : m_accelerationStructureNV(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR AccelerationStructureNV( std::nullptr_t )
      : m_accelerationStructureNV(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT AccelerationStructureNV( VkAccelerationStructureNV accelerationStructureNV )
      : m_accelerationStructureNV( accelerationStructureNV )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    AccelerationStructureNV & operator=(VkAccelerationStructureNV accelerationStructureNV)
    {
      m_accelerationStructureNV = accelerationStructureNV;
      return *this; 
    }
#endif

    AccelerationStructureNV & operator=( std::nullptr_t )
    {
      m_accelerationStructureNV = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( AccelerationStructureNV const & rhs ) const
    {
      return m_accelerationStructureNV == rhs.m_accelerationStructureNV;
    }

    bool operator!=(AccelerationStructureNV const & rhs ) const
    {
      return m_accelerationStructureNV != rhs.m_accelerationStructureNV;
    }

    bool operator<(AccelerationStructureNV const & rhs ) const
    {
      return m_accelerationStructureNV < rhs.m_accelerationStructureNV;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkAccelerationStructureNV() const
    {
      return m_accelerationStructureNV;
    }

    explicit operator bool() const
    {
      return m_accelerationStructureNV != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_accelerationStructureNV == VK_NULL_HANDLE;
    }

  private:
    VkAccelerationStructureNV m_accelerationStructureNV;
  };

  static_assert( sizeof( AccelerationStructureNV ) == sizeof( VkAccelerationStructureNV ), "handle and wrapper have different size!" );

  class DisplayKHR
  {
  public:
    VULKAN_HPP_CONSTEXPR DisplayKHR()
      : m_displayKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DisplayKHR( std::nullptr_t )
      : m_displayKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DisplayKHR( VkDisplayKHR displayKHR )
      : m_displayKHR( displayKHR )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DisplayKHR & operator=(VkDisplayKHR displayKHR)
    {
      m_displayKHR = displayKHR;
      return *this; 
    }
#endif

    DisplayKHR & operator=( std::nullptr_t )
    {
      m_displayKHR = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DisplayKHR const & rhs ) const
    {
      return m_displayKHR == rhs.m_displayKHR;
    }

    bool operator!=(DisplayKHR const & rhs ) const
    {
      return m_displayKHR != rhs.m_displayKHR;
    }

    bool operator<(DisplayKHR const & rhs ) const
    {
      return m_displayKHR < rhs.m_displayKHR;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDisplayKHR() const
    {
      return m_displayKHR;
    }

    explicit operator bool() const
    {
      return m_displayKHR != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_displayKHR == VK_NULL_HANDLE;
    }

  private:
    VkDisplayKHR m_displayKHR;
  };

  static_assert( sizeof( DisplayKHR ) == sizeof( VkDisplayKHR ), "handle and wrapper have different size!" );

  class DisplayModeKHR
  {
  public:
    VULKAN_HPP_CONSTEXPR DisplayModeKHR()
      : m_displayModeKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DisplayModeKHR( std::nullptr_t )
      : m_displayModeKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DisplayModeKHR( VkDisplayModeKHR displayModeKHR )
      : m_displayModeKHR( displayModeKHR )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DisplayModeKHR & operator=(VkDisplayModeKHR displayModeKHR)
    {
      m_displayModeKHR = displayModeKHR;
      return *this; 
    }
#endif

    DisplayModeKHR & operator=( std::nullptr_t )
    {
      m_displayModeKHR = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DisplayModeKHR const & rhs ) const
    {
      return m_displayModeKHR == rhs.m_displayModeKHR;
    }

    bool operator!=(DisplayModeKHR const & rhs ) const
    {
      return m_displayModeKHR != rhs.m_displayModeKHR;
    }

    bool operator<(DisplayModeKHR const & rhs ) const
    {
      return m_displayModeKHR < rhs.m_displayModeKHR;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDisplayModeKHR() const
    {
      return m_displayModeKHR;
    }

    explicit operator bool() const
    {
      return m_displayModeKHR != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_displayModeKHR == VK_NULL_HANDLE;
    }

  private:
    VkDisplayModeKHR m_displayModeKHR;
  };

  static_assert( sizeof( DisplayModeKHR ) == sizeof( VkDisplayModeKHR ), "handle and wrapper have different size!" );

  class SurfaceKHR
  {
  public:
    VULKAN_HPP_CONSTEXPR SurfaceKHR()
      : m_surfaceKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR SurfaceKHR( std::nullptr_t )
      : m_surfaceKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT SurfaceKHR( VkSurfaceKHR surfaceKHR )
      : m_surfaceKHR( surfaceKHR )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    SurfaceKHR & operator=(VkSurfaceKHR surfaceKHR)
    {
      m_surfaceKHR = surfaceKHR;
      return *this; 
    }
#endif

    SurfaceKHR & operator=( std::nullptr_t )
    {
      m_surfaceKHR = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( SurfaceKHR const & rhs ) const
    {
      return m_surfaceKHR == rhs.m_surfaceKHR;
    }

    bool operator!=(SurfaceKHR const & rhs ) const
    {
      return m_surfaceKHR != rhs.m_surfaceKHR;
    }

    bool operator<(SurfaceKHR const & rhs ) const
    {
      return m_surfaceKHR < rhs.m_surfaceKHR;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkSurfaceKHR() const
    {
      return m_surfaceKHR;
    }

    explicit operator bool() const
    {
      return m_surfaceKHR != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_surfaceKHR == VK_NULL_HANDLE;
    }

  private:
    VkSurfaceKHR m_surfaceKHR;
  };

  static_assert( sizeof( SurfaceKHR ) == sizeof( VkSurfaceKHR ), "handle and wrapper have different size!" );

  class SwapchainKHR
  {
  public:
    VULKAN_HPP_CONSTEXPR SwapchainKHR()
      : m_swapchainKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR SwapchainKHR( std::nullptr_t )
      : m_swapchainKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT SwapchainKHR( VkSwapchainKHR swapchainKHR )
      : m_swapchainKHR( swapchainKHR )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    SwapchainKHR & operator=(VkSwapchainKHR swapchainKHR)
    {
      m_swapchainKHR = swapchainKHR;
      return *this; 
    }
#endif

    SwapchainKHR & operator=( std::nullptr_t )
    {
      m_swapchainKHR = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( SwapchainKHR const & rhs ) const
    {
      return m_swapchainKHR == rhs.m_swapchainKHR;
    }

    bool operator!=(SwapchainKHR const & rhs ) const
    {
      return m_swapchainKHR != rhs.m_swapchainKHR;
    }

    bool operator<(SwapchainKHR const & rhs ) const
    {
      return m_swapchainKHR < rhs.m_swapchainKHR;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkSwapchainKHR() const
    {
      return m_swapchainKHR;
    }

    explicit operator bool() const
    {
      return m_swapchainKHR != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_swapchainKHR == VK_NULL_HANDLE;
    }

  private:
    VkSwapchainKHR m_swapchainKHR;
  };

  static_assert( sizeof( SwapchainKHR ) == sizeof( VkSwapchainKHR ), "handle and wrapper have different size!" );

  class DebugReportCallbackEXT
  {
  public:
    VULKAN_HPP_CONSTEXPR DebugReportCallbackEXT()
      : m_debugReportCallbackEXT(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DebugReportCallbackEXT( std::nullptr_t )
      : m_debugReportCallbackEXT(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DebugReportCallbackEXT( VkDebugReportCallbackEXT debugReportCallbackEXT )
      : m_debugReportCallbackEXT( debugReportCallbackEXT )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DebugReportCallbackEXT & operator=(VkDebugReportCallbackEXT debugReportCallbackEXT)
    {
      m_debugReportCallbackEXT = debugReportCallbackEXT;
      return *this; 
    }
#endif

    DebugReportCallbackEXT & operator=( std::nullptr_t )
    {
      m_debugReportCallbackEXT = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DebugReportCallbackEXT const & rhs ) const
    {
      return m_debugReportCallbackEXT == rhs.m_debugReportCallbackEXT;
    }

    bool operator!=(DebugReportCallbackEXT const & rhs ) const
    {
      return m_debugReportCallbackEXT != rhs.m_debugReportCallbackEXT;
    }

    bool operator<(DebugReportCallbackEXT const & rhs ) const
    {
      return m_debugReportCallbackEXT < rhs.m_debugReportCallbackEXT;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDebugReportCallbackEXT() const
    {
      return m_debugReportCallbackEXT;
    }

    explicit operator bool() const
    {
      return m_debugReportCallbackEXT != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_debugReportCallbackEXT == VK_NULL_HANDLE;
    }

  private:
    VkDebugReportCallbackEXT m_debugReportCallbackEXT;
  };

  static_assert( sizeof( DebugReportCallbackEXT ) == sizeof( VkDebugReportCallbackEXT ), "handle and wrapper have different size!" );

  class DebugUtilsMessengerEXT
  {
  public:
    VULKAN_HPP_CONSTEXPR DebugUtilsMessengerEXT()
      : m_debugUtilsMessengerEXT(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DebugUtilsMessengerEXT( std::nullptr_t )
      : m_debugUtilsMessengerEXT(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DebugUtilsMessengerEXT( VkDebugUtilsMessengerEXT debugUtilsMessengerEXT )
      : m_debugUtilsMessengerEXT( debugUtilsMessengerEXT )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DebugUtilsMessengerEXT & operator=(VkDebugUtilsMessengerEXT debugUtilsMessengerEXT)
    {
      m_debugUtilsMessengerEXT = debugUtilsMessengerEXT;
      return *this; 
    }
#endif

    DebugUtilsMessengerEXT & operator=( std::nullptr_t )
    {
      m_debugUtilsMessengerEXT = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DebugUtilsMessengerEXT const & rhs ) const
    {
      return m_debugUtilsMessengerEXT == rhs.m_debugUtilsMessengerEXT;
    }

    bool operator!=(DebugUtilsMessengerEXT const & rhs ) const
    {
      return m_debugUtilsMessengerEXT != rhs.m_debugUtilsMessengerEXT;
    }

    bool operator<(DebugUtilsMessengerEXT const & rhs ) const
    {
      return m_debugUtilsMessengerEXT < rhs.m_debugUtilsMessengerEXT;
    }



    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDebugUtilsMessengerEXT() const
    {
      return m_debugUtilsMessengerEXT;
    }

    explicit operator bool() const
    {
      return m_debugUtilsMessengerEXT != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_debugUtilsMessengerEXT == VK_NULL_HANDLE;
    }

  private:
    VkDebugUtilsMessengerEXT m_debugUtilsMessengerEXT;
  };

  static_assert( sizeof( DebugUtilsMessengerEXT ) == sizeof( VkDebugUtilsMessengerEXT ), "handle and wrapper have different size!" );

  struct Offset2D
  {
    Offset2D( int32_t x_ = 0,
              int32_t y_ = 0 )
      : x( x_ )
      , y( y_ )
    {
    }

    Offset2D( VkOffset2D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Offset2D ) );
    }

    Offset2D& operator=( VkOffset2D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Offset2D ) );
      return *this;
    }
    Offset2D& setX( int32_t x_ )
    {
      x = x_;
      return *this;
    }

    Offset2D& setY( int32_t y_ )
    {
      y = y_;
      return *this;
    }

    operator VkOffset2D const&() const
    {
      return *reinterpret_cast<const VkOffset2D*>(this);
    }

    operator VkOffset2D &()
    {
      return *reinterpret_cast<VkOffset2D*>(this);
    }

    bool operator==( Offset2D const& rhs ) const
    {
      return ( x == rhs.x )
          && ( y == rhs.y );
    }

    bool operator!=( Offset2D const& rhs ) const
    {
      return !operator==( rhs );
    }

    int32_t x;
    int32_t y;
  };
  static_assert( sizeof( Offset2D ) == sizeof( VkOffset2D ), "struct and wrapper have different size!" );

  struct Offset3D
  {
    Offset3D( int32_t x_ = 0,
              int32_t y_ = 0,
              int32_t z_ = 0 )
      : x( x_ )
      , y( y_ )
      , z( z_ )
    {
    }

    explicit Offset3D( Offset2D const& offset2D,
                       int32_t z_ = 0 )
      : x( offset2D.x )
      , y( offset2D.y )
      , z( z_ )
    {}

    Offset3D( VkOffset3D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Offset3D ) );
    }

    Offset3D& operator=( VkOffset3D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Offset3D ) );
      return *this;
    }
    Offset3D& setX( int32_t x_ )
    {
      x = x_;
      return *this;
    }

    Offset3D& setY( int32_t y_ )
    {
      y = y_;
      return *this;
    }

    Offset3D& setZ( int32_t z_ )
    {
      z = z_;
      return *this;
    }

    operator VkOffset3D const&() const
    {
      return *reinterpret_cast<const VkOffset3D*>(this);
    }

    operator VkOffset3D &()
    {
      return *reinterpret_cast<VkOffset3D*>(this);
    }

    bool operator==( Offset3D const& rhs ) const
    {
      return ( x == rhs.x )
          && ( y == rhs.y )
          && ( z == rhs.z );
    }

    bool operator!=( Offset3D const& rhs ) const
    {
      return !operator==( rhs );
    }

    int32_t x;
    int32_t y;
    int32_t z;
  };
  static_assert( sizeof( Offset3D ) == sizeof( VkOffset3D ), "struct and wrapper have different size!" );

  struct Extent2D
  {
    Extent2D( uint32_t width_ = 0,
              uint32_t height_ = 0 )
      : width( width_ )
      , height( height_ )
    {
    }

    Extent2D( VkExtent2D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Extent2D ) );
    }

    Extent2D& operator=( VkExtent2D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Extent2D ) );
      return *this;
    }
    Extent2D& setWidth( uint32_t width_ )
    {
      width = width_;
      return *this;
    }

    Extent2D& setHeight( uint32_t height_ )
    {
      height = height_;
      return *this;
    }

    operator VkExtent2D const&() const
    {
      return *reinterpret_cast<const VkExtent2D*>(this);
    }

    operator VkExtent2D &()
    {
      return *reinterpret_cast<VkExtent2D*>(this);
    }

    bool operator==( Extent2D const& rhs ) const
    {
      return ( width == rhs.width )
          && ( height == rhs.height );
    }

    bool operator!=( Extent2D const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t width;
    uint32_t height;
  };
  static_assert( sizeof( Extent2D ) == sizeof( VkExtent2D ), "struct and wrapper have different size!" );

  struct Extent3D
  {
    Extent3D( uint32_t width_ = 0,
              uint32_t height_ = 0,
              uint32_t depth_ = 0 )
      : width( width_ )
      , height( height_ )
      , depth( depth_ )
    {
    }

    explicit Extent3D( Extent2D const& extent2D,
                       uint32_t depth_ = 0 )
      : width( extent2D.width )
      , height( extent2D.height )
      , depth( depth_ )
    {}

    Extent3D( VkExtent3D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Extent3D ) );
    }

    Extent3D& operator=( VkExtent3D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Extent3D ) );
      return *this;
    }
    Extent3D& setWidth( uint32_t width_ )
    {
      width = width_;
      return *this;
    }

    Extent3D& setHeight( uint32_t height_ )
    {
      height = height_;
      return *this;
    }

    Extent3D& setDepth( uint32_t depth_ )
    {
      depth = depth_;
      return *this;
    }

    operator VkExtent3D const&() const
    {
      return *reinterpret_cast<const VkExtent3D*>(this);
    }

    operator VkExtent3D &()
    {
      return *reinterpret_cast<VkExtent3D*>(this);
    }

    bool operator==( Extent3D const& rhs ) const
    {
      return ( width == rhs.width )
          && ( height == rhs.height )
          && ( depth == rhs.depth );
    }

    bool operator!=( Extent3D const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t width;
    uint32_t height;
    uint32_t depth;
  };
  static_assert( sizeof( Extent3D ) == sizeof( VkExtent3D ), "struct and wrapper have different size!" );

  struct Viewport
  {
    Viewport( float x_ = 0,
              float y_ = 0,
              float width_ = 0,
              float height_ = 0,
              float minDepth_ = 0,
              float maxDepth_ = 0 )
      : x( x_ )
      , y( y_ )
      , width( width_ )
      , height( height_ )
      , minDepth( minDepth_ )
      , maxDepth( maxDepth_ )
    {
    }

    Viewport( VkViewport const & rhs )
    {
      memcpy( this, &rhs, sizeof( Viewport ) );
    }

    Viewport& operator=( VkViewport const & rhs )
    {
      memcpy( this, &rhs, sizeof( Viewport ) );
      return *this;
    }
    Viewport& setX( float x_ )
    {
      x = x_;
      return *this;
    }

    Viewport& setY( float y_ )
    {
      y = y_;
      return *this;
    }

    Viewport& setWidth( float width_ )
    {
      width = width_;
      return *this;
    }

    Viewport& setHeight( float height_ )
    {
      height = height_;
      return *this;
    }

    Viewport& setMinDepth( float minDepth_ )
    {
      minDepth = minDepth_;
      return *this;
    }

    Viewport& setMaxDepth( float maxDepth_ )
    {
      maxDepth = maxDepth_;
      return *this;
    }

    operator VkViewport const&() const
    {
      return *reinterpret_cast<const VkViewport*>(this);
    }

    operator VkViewport &()
    {
      return *reinterpret_cast<VkViewport*>(this);
    }

    bool operator==( Viewport const& rhs ) const
    {
      return ( x == rhs.x )
          && ( y == rhs.y )
          && ( width == rhs.width )
          && ( height == rhs.height )
          && ( minDepth == rhs.minDepth )
          && ( maxDepth == rhs.maxDepth );
    }

    bool operator!=( Viewport const& rhs ) const
    {
      return !operator==( rhs );
    }

    float x;
    float y;
    float width;
    float height;
    float minDepth;
    float maxDepth;
  };
  static_assert( sizeof( Viewport ) == sizeof( VkViewport ), "struct and wrapper have different size!" );

  struct Rect2D
  {
    Rect2D( Offset2D offset_ = Offset2D(),
            Extent2D extent_ = Extent2D() )
      : offset( offset_ )
      , extent( extent_ )
    {
    }

    Rect2D( VkRect2D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Rect2D ) );
    }

    Rect2D& operator=( VkRect2D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Rect2D ) );
      return *this;
    }
    Rect2D& setOffset( Offset2D offset_ )
    {
      offset = offset_;
      return *this;
    }

    Rect2D& setExtent( Extent2D extent_ )
    {
      extent = extent_;
      return *this;
    }

    operator VkRect2D const&() const
    {
      return *reinterpret_cast<const VkRect2D*>(this);
    }

    operator VkRect2D &()
    {
      return *reinterpret_cast<VkRect2D*>(this);
    }

    bool operator==( Rect2D const& rhs ) const
    {
      return ( offset == rhs.offset )
          && ( extent == rhs.extent );
    }

    bool operator!=( Rect2D const& rhs ) const
    {
      return !operator==( rhs );
    }

    Offset2D offset;
    Extent2D extent;
  };
  static_assert( sizeof( Rect2D ) == sizeof( VkRect2D ), "struct and wrapper have different size!" );

  struct ClearRect
  {
    ClearRect( Rect2D rect_ = Rect2D(),
               uint32_t baseArrayLayer_ = 0,
               uint32_t layerCount_ = 0 )
      : rect( rect_ )
      , baseArrayLayer( baseArrayLayer_ )
      , layerCount( layerCount_ )
    {
    }

    ClearRect( VkClearRect const & rhs )
    {
      memcpy( this, &rhs, sizeof( ClearRect ) );
    }

    ClearRect& operator=( VkClearRect const & rhs )
    {
      memcpy( this, &rhs, sizeof( ClearRect ) );
      return *this;
    }
    ClearRect& setRect( Rect2D rect_ )
    {
      rect = rect_;
      return *this;
    }

    ClearRect& setBaseArrayLayer( uint32_t baseArrayLayer_ )
    {
      baseArrayLayer = baseArrayLayer_;
      return *this;
    }

    ClearRect& setLayerCount( uint32_t layerCount_ )
    {
      layerCount = layerCount_;
      return *this;
    }

    operator VkClearRect const&() const
    {
      return *reinterpret_cast<const VkClearRect*>(this);
    }

    operator VkClearRect &()
    {
      return *reinterpret_cast<VkClearRect*>(this);
    }

    bool operator==( ClearRect const& rhs ) const
    {
      return ( rect == rhs.rect )
          && ( baseArrayLayer == rhs.baseArrayLayer )
          && ( layerCount == rhs.layerCount );
    }

    bool operator!=( ClearRect const& rhs ) const
    {
      return !operator==( rhs );
    }

    Rect2D rect;
    uint32_t baseArrayLayer;
    uint32_t layerCount;
  };
  static_assert( sizeof( ClearRect ) == sizeof( VkClearRect ), "struct and wrapper have different size!" );

  struct ExtensionProperties
  {
    operator VkExtensionProperties const&() const
    {
      return *reinterpret_cast<const VkExtensionProperties*>(this);
    }

    operator VkExtensionProperties &()
    {
      return *reinterpret_cast<VkExtensionProperties*>(this);
    }

    bool operator==( ExtensionProperties const& rhs ) const
    {
      return ( memcmp( extensionName, rhs.extensionName, VK_MAX_EXTENSION_NAME_SIZE * sizeof( char ) ) == 0 )
          && ( specVersion == rhs.specVersion );
    }

    bool operator!=( ExtensionProperties const& rhs ) const
    {
      return !operator==( rhs );
    }

    char extensionName[VK_MAX_EXTENSION_NAME_SIZE];
    uint32_t specVersion;
  };
  static_assert( sizeof( ExtensionProperties ) == sizeof( VkExtensionProperties ), "struct and wrapper have different size!" );

  struct LayerProperties
  {
    operator VkLayerProperties const&() const
    {
      return *reinterpret_cast<const VkLayerProperties*>(this);
    }

    operator VkLayerProperties &()
    {
      return *reinterpret_cast<VkLayerProperties*>(this);
    }

    bool operator==( LayerProperties const& rhs ) const
    {
      return ( memcmp( layerName, rhs.layerName, VK_MAX_EXTENSION_NAME_SIZE * sizeof( char ) ) == 0 )
          && ( specVersion == rhs.specVersion )
          && ( implementationVersion == rhs.implementationVersion )
          && ( memcmp( description, rhs.description, VK_MAX_DESCRIPTION_SIZE * sizeof( char ) ) == 0 );
    }

    bool operator!=( LayerProperties const& rhs ) const
    {
      return !operator==( rhs );
    }

    char layerName[VK_MAX_EXTENSION_NAME_SIZE];
    uint32_t specVersion;
    uint32_t implementationVersion;
    char description[VK_MAX_DESCRIPTION_SIZE];
  };
  static_assert( sizeof( LayerProperties ) == sizeof( VkLayerProperties ), "struct and wrapper have different size!" );

  struct AllocationCallbacks
  {
    AllocationCallbacks( void* pUserData_ = nullptr,
                         PFN_vkAllocationFunction pfnAllocation_ = nullptr,
                         PFN_vkReallocationFunction pfnReallocation_ = nullptr,
                         PFN_vkFreeFunction pfnFree_ = nullptr,
                         PFN_vkInternalAllocationNotification pfnInternalAllocation_ = nullptr,
                         PFN_vkInternalFreeNotification pfnInternalFree_ = nullptr )
      : pUserData( pUserData_ )
      , pfnAllocation( pfnAllocation_ )
      , pfnReallocation( pfnReallocation_ )
      , pfnFree( pfnFree_ )
      , pfnInternalAllocation( pfnInternalAllocation_ )
      , pfnInternalFree( pfnInternalFree_ )
    {
    }

    AllocationCallbacks( VkAllocationCallbacks const & rhs )
    {
      memcpy( this, &rhs, sizeof( AllocationCallbacks ) );
    }

    AllocationCallbacks& operator=( VkAllocationCallbacks const & rhs )
    {
      memcpy( this, &rhs, sizeof( AllocationCallbacks ) );
      return *this;
    }
    AllocationCallbacks& setPUserData( void* pUserData_ )
    {
      pUserData = pUserData_;
      return *this;
    }

    AllocationCallbacks& setPfnAllocation( PFN_vkAllocationFunction pfnAllocation_ )
    {
      pfnAllocation = pfnAllocation_;
      return *this;
    }

    AllocationCallbacks& setPfnReallocation( PFN_vkReallocationFunction pfnReallocation_ )
    {
      pfnReallocation = pfnReallocation_;
      return *this;
    }

    AllocationCallbacks& setPfnFree( PFN_vkFreeFunction pfnFree_ )
    {
      pfnFree = pfnFree_;
      return *this;
    }

    AllocationCallbacks& setPfnInternalAllocation( PFN_vkInternalAllocationNotification pfnInternalAllocation_ )
    {
      pfnInternalAllocation = pfnInternalAllocation_;
      return *this;
    }

    AllocationCallbacks& setPfnInternalFree( PFN_vkInternalFreeNotification pfnInternalFree_ )
    {
      pfnInternalFree = pfnInternalFree_;
      return *this;
    }

    operator VkAllocationCallbacks const&() const
    {
      return *reinterpret_cast<const VkAllocationCallbacks*>(this);
    }

    operator VkAllocationCallbacks &()
    {
      return *reinterpret_cast<VkAllocationCallbacks*>(this);
    }

    bool operator==( AllocationCallbacks const& rhs ) const
    {
      return ( pUserData == rhs.pUserData )
          && ( pfnAllocation == rhs.pfnAllocation )
          && ( pfnReallocation == rhs.pfnReallocation )
          && ( pfnFree == rhs.pfnFree )
          && ( pfnInternalAllocation == rhs.pfnInternalAllocation )
          && ( pfnInternalFree == rhs.pfnInternalFree );
    }

    bool operator!=( AllocationCallbacks const& rhs ) const
    {
      return !operator==( rhs );
    }

    void* pUserData;
    PFN_vkAllocationFunction pfnAllocation;
    PFN_vkReallocationFunction pfnReallocation;
    PFN_vkFreeFunction pfnFree;
    PFN_vkInternalAllocationNotification pfnInternalAllocation;
    PFN_vkInternalFreeNotification pfnInternalFree;
  };
  static_assert( sizeof( AllocationCallbacks ) == sizeof( VkAllocationCallbacks ), "struct and wrapper have different size!" );

  struct MemoryRequirements
  {
    operator VkMemoryRequirements const&() const
    {
      return *reinterpret_cast<const VkMemoryRequirements*>(this);
    }

    operator VkMemoryRequirements &()
    {
      return *reinterpret_cast<VkMemoryRequirements*>(this);
    }

    bool operator==( MemoryRequirements const& rhs ) const
    {
      return ( size == rhs.size )
          && ( alignment == rhs.alignment )
          && ( memoryTypeBits == rhs.memoryTypeBits );
    }

    bool operator!=( MemoryRequirements const& rhs ) const
    {
      return !operator==( rhs );
    }

    DeviceSize size;
    DeviceSize alignment;
    uint32_t memoryTypeBits;
  };
  static_assert( sizeof( MemoryRequirements ) == sizeof( VkMemoryRequirements ), "struct and wrapper have different size!" );

  struct DescriptorBufferInfo
  {
    DescriptorBufferInfo( Buffer buffer_ = Buffer(),
                          DeviceSize offset_ = 0,
                          DeviceSize range_ = 0 )
      : buffer( buffer_ )
      , offset( offset_ )
      , range( range_ )
    {
    }

    DescriptorBufferInfo( VkDescriptorBufferInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorBufferInfo ) );
    }

    DescriptorBufferInfo& operator=( VkDescriptorBufferInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorBufferInfo ) );
      return *this;
    }
    DescriptorBufferInfo& setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    DescriptorBufferInfo& setOffset( DeviceSize offset_ )
    {
      offset = offset_;
      return *this;
    }

    DescriptorBufferInfo& setRange( DeviceSize range_ )
    {
      range = range_;
      return *this;
    }

    operator VkDescriptorBufferInfo const&() const
    {
      return *reinterpret_cast<const VkDescriptorBufferInfo*>(this);
    }

    operator VkDescriptorBufferInfo &()
    {
      return *reinterpret_cast<VkDescriptorBufferInfo*>(this);
    }

    bool operator==( DescriptorBufferInfo const& rhs ) const
    {
      return ( buffer == rhs.buffer )
          && ( offset == rhs.offset )
          && ( range == rhs.range );
    }

    bool operator!=( DescriptorBufferInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

    Buffer buffer;
    DeviceSize offset;
    DeviceSize range;
  };
  static_assert( sizeof( DescriptorBufferInfo ) == sizeof( VkDescriptorBufferInfo ), "struct and wrapper have different size!" );

  struct SubresourceLayout
  {
    operator VkSubresourceLayout const&() const
    {
      return *reinterpret_cast<const VkSubresourceLayout*>(this);
    }

    operator VkSubresourceLayout &()
    {
      return *reinterpret_cast<VkSubresourceLayout*>(this);
    }

    bool operator==( SubresourceLayout const& rhs ) const
    {
      return ( offset == rhs.offset )
          && ( size == rhs.size )
          && ( rowPitch == rhs.rowPitch )
          && ( arrayPitch == rhs.arrayPitch )
          && ( depthPitch == rhs.depthPitch );
    }

    bool operator!=( SubresourceLayout const& rhs ) const
    {
      return !operator==( rhs );
    }

    DeviceSize offset;
    DeviceSize size;
    DeviceSize rowPitch;
    DeviceSize arrayPitch;
    DeviceSize depthPitch;
  };
  static_assert( sizeof( SubresourceLayout ) == sizeof( VkSubresourceLayout ), "struct and wrapper have different size!" );

  struct BufferCopy
  {
    BufferCopy( DeviceSize srcOffset_ = 0,
                DeviceSize dstOffset_ = 0,
                DeviceSize size_ = 0 )
      : srcOffset( srcOffset_ )
      , dstOffset( dstOffset_ )
      , size( size_ )
    {
    }

    BufferCopy( VkBufferCopy const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferCopy ) );
    }

    BufferCopy& operator=( VkBufferCopy const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferCopy ) );
      return *this;
    }
    BufferCopy& setSrcOffset( DeviceSize srcOffset_ )
    {
      srcOffset = srcOffset_;
      return *this;
    }

    BufferCopy& setDstOffset( DeviceSize dstOffset_ )
    {
      dstOffset = dstOffset_;
      return *this;
    }

    BufferCopy& setSize( DeviceSize size_ )
    {
      size = size_;
      return *this;
    }

    operator VkBufferCopy const&() const
    {
      return *reinterpret_cast<const VkBufferCopy*>(this);
    }

    operator VkBufferCopy &()
    {
      return *reinterpret_cast<VkBufferCopy*>(this);
    }

    bool operator==( BufferCopy const& rhs ) const
    {
      return ( srcOffset == rhs.srcOffset )
          && ( dstOffset == rhs.dstOffset )
          && ( size == rhs.size );
    }

    bool operator!=( BufferCopy const& rhs ) const
    {
      return !operator==( rhs );
    }

    DeviceSize srcOffset;
    DeviceSize dstOffset;
    DeviceSize size;
  };
  static_assert( sizeof( BufferCopy ) == sizeof( VkBufferCopy ), "struct and wrapper have different size!" );

  struct SpecializationMapEntry
  {
    SpecializationMapEntry( uint32_t constantID_ = 0,
                            uint32_t offset_ = 0,
                            size_t size_ = 0 )
      : constantID( constantID_ )
      , offset( offset_ )
      , size( size_ )
    {
    }

    SpecializationMapEntry( VkSpecializationMapEntry const & rhs )
    {
      memcpy( this, &rhs, sizeof( SpecializationMapEntry ) );
    }

    SpecializationMapEntry& operator=( VkSpecializationMapEntry const & rhs )
    {
      memcpy( this, &rhs, sizeof( SpecializationMapEntry ) );
      return *this;
    }
    SpecializationMapEntry& setConstantID( uint32_t constantID_ )
    {
      constantID = constantID_;
      return *this;
    }

    SpecializationMapEntry& setOffset( uint32_t offset_ )
    {
      offset = offset_;
      return *this;
    }

    SpecializationMapEntry& setSize( size_t size_ )
    {
      size = size_;
      return *this;
    }

    operator VkSpecializationMapEntry const&() const
    {
      return *reinterpret_cast<const VkSpecializationMapEntry*>(this);
    }

    operator VkSpecializationMapEntry &()
    {
      return *reinterpret_cast<VkSpecializationMapEntry*>(this);
    }

    bool operator==( SpecializationMapEntry const& rhs ) const
    {
      return ( constantID == rhs.constantID )
          && ( offset == rhs.offset )
          && ( size == rhs.size );
    }

    bool operator!=( SpecializationMapEntry const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t constantID;
    uint32_t offset;
    size_t size;
  };
  static_assert( sizeof( SpecializationMapEntry ) == sizeof( VkSpecializationMapEntry ), "struct and wrapper have different size!" );

  struct SpecializationInfo
  {
    SpecializationInfo( uint32_t mapEntryCount_ = 0,
                        const SpecializationMapEntry* pMapEntries_ = nullptr,
                        size_t dataSize_ = 0,
                        const void* pData_ = nullptr )
      : mapEntryCount( mapEntryCount_ )
      , pMapEntries( pMapEntries_ )
      , dataSize( dataSize_ )
      , pData( pData_ )
    {
    }

    SpecializationInfo( VkSpecializationInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SpecializationInfo ) );
    }

    SpecializationInfo& operator=( VkSpecializationInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SpecializationInfo ) );
      return *this;
    }
    SpecializationInfo& setMapEntryCount( uint32_t mapEntryCount_ )
    {
      mapEntryCount = mapEntryCount_;
      return *this;
    }

    SpecializationInfo& setPMapEntries( const SpecializationMapEntry* pMapEntries_ )
    {
      pMapEntries = pMapEntries_;
      return *this;
    }

    SpecializationInfo& setDataSize( size_t dataSize_ )
    {
      dataSize = dataSize_;
      return *this;
    }

    SpecializationInfo& setPData( const void* pData_ )
    {
      pData = pData_;
      return *this;
    }

    operator VkSpecializationInfo const&() const
    {
      return *reinterpret_cast<const VkSpecializationInfo*>(this);
    }

    operator VkSpecializationInfo &()
    {
      return *reinterpret_cast<VkSpecializationInfo*>(this);
    }

    bool operator==( SpecializationInfo const& rhs ) const
    {
      return ( mapEntryCount == rhs.mapEntryCount )
          && ( pMapEntries == rhs.pMapEntries )
          && ( dataSize == rhs.dataSize )
          && ( pData == rhs.pData );
    }

    bool operator!=( SpecializationInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t mapEntryCount;
    const SpecializationMapEntry* pMapEntries;
    size_t dataSize;
    const void* pData;
  };
  static_assert( sizeof( SpecializationInfo ) == sizeof( VkSpecializationInfo ), "struct and wrapper have different size!" );

  union ClearColorValue
  {
    ClearColorValue( const std::array<float,4>& float32_ = { {0} } )
    {
      memcpy( &float32, float32_.data(), 4 * sizeof( float ) );
    }

    ClearColorValue( const std::array<int32_t,4>& int32_ )
    {
      memcpy( &int32, int32_.data(), 4 * sizeof( int32_t ) );
    }

    ClearColorValue( const std::array<uint32_t,4>& uint32_ )
    {
      memcpy( &uint32, uint32_.data(), 4 * sizeof( uint32_t ) );
    }

    ClearColorValue& setFloat32( std::array<float,4> float32_ )
    {
      memcpy( &float32, float32_.data(), 4 * sizeof( float ) );
      return *this;
    }

    ClearColorValue& setInt32( std::array<int32_t,4> int32_ )
    {
      memcpy( &int32, int32_.data(), 4 * sizeof( int32_t ) );
      return *this;
    }

    ClearColorValue& setUint32( std::array<uint32_t,4> uint32_ )
    {
      memcpy( &uint32, uint32_.data(), 4 * sizeof( uint32_t ) );
      return *this;
    }

    operator VkClearColorValue const&() const
    {
      return *reinterpret_cast<const VkClearColorValue*>(this);
    }

    operator VkClearColorValue &()
    {
      return *reinterpret_cast<VkClearColorValue*>(this);
    }

    float float32[4];
    int32_t int32[4];
    uint32_t uint32[4];
  };

  struct ClearDepthStencilValue
  {
    ClearDepthStencilValue( float depth_ = 0,
                            uint32_t stencil_ = 0 )
      : depth( depth_ )
      , stencil( stencil_ )
    {
    }

    ClearDepthStencilValue( VkClearDepthStencilValue const & rhs )
    {
      memcpy( this, &rhs, sizeof( ClearDepthStencilValue ) );
    }

    ClearDepthStencilValue& operator=( VkClearDepthStencilValue const & rhs )
    {
      memcpy( this, &rhs, sizeof( ClearDepthStencilValue ) );
      return *this;
    }
    ClearDepthStencilValue& setDepth( float depth_ )
    {
      depth = depth_;
      return *this;
    }

    ClearDepthStencilValue& setStencil( uint32_t stencil_ )
    {
      stencil = stencil_;
      return *this;
    }

    operator VkClearDepthStencilValue const&() const
    {
      return *reinterpret_cast<const VkClearDepthStencilValue*>(this);
    }

    operator VkClearDepthStencilValue &()
    {
      return *reinterpret_cast<VkClearDepthStencilValue*>(this);
    }

    bool operator==( ClearDepthStencilValue const& rhs ) const
    {
      return ( depth == rhs.depth )
          && ( stencil == rhs.stencil );
    }

    bool operator!=( ClearDepthStencilValue const& rhs ) const
    {
      return !operator==( rhs );
    }

    float depth;
    uint32_t stencil;
  };
  static_assert( sizeof( ClearDepthStencilValue ) == sizeof( VkClearDepthStencilValue ), "struct and wrapper have different size!" );

  union ClearValue
  {
    ClearValue( ClearColorValue color_ = ClearColorValue() )
    {
      color = color_;
    }

    ClearValue( ClearDepthStencilValue depthStencil_ )
    {
      depthStencil = depthStencil_;
    }

    ClearValue& setColor( ClearColorValue color_ )
    {
      color = color_;
      return *this;
    }

    ClearValue& setDepthStencil( ClearDepthStencilValue depthStencil_ )
    {
      depthStencil = depthStencil_;
      return *this;
    }

    operator VkClearValue const&() const
    {
      return *reinterpret_cast<const VkClearValue*>(this);
    }

    operator VkClearValue &()
    {
      return *reinterpret_cast<VkClearValue*>(this);
    }

#ifdef VULKAN_HPP_HAS_UNRESTRICTED_UNIONS
    ClearColorValue color;
    ClearDepthStencilValue depthStencil;
#else
    VkClearColorValue color;
    VkClearDepthStencilValue depthStencil;
#endif  // VULKAN_HPP_HAS_UNRESTRICTED_UNIONS
  };

  struct PhysicalDeviceFeatures
  {
    PhysicalDeviceFeatures( Bool32 robustBufferAccess_ = 0,
                            Bool32 fullDrawIndexUint32_ = 0,
                            Bool32 imageCubeArray_ = 0,
                            Bool32 independentBlend_ = 0,
                            Bool32 geometryShader_ = 0,
                            Bool32 tessellationShader_ = 0,
                            Bool32 sampleRateShading_ = 0,
                            Bool32 dualSrcBlend_ = 0,
                            Bool32 logicOp_ = 0,
                            Bool32 multiDrawIndirect_ = 0,
                            Bool32 drawIndirectFirstInstance_ = 0,
                            Bool32 depthClamp_ = 0,
                            Bool32 depthBiasClamp_ = 0,
                            Bool32 fillModeNonSolid_ = 0,
                            Bool32 depthBounds_ = 0,
                            Bool32 wideLines_ = 0,
                            Bool32 largePoints_ = 0,
                            Bool32 alphaToOne_ = 0,
                            Bool32 multiViewport_ = 0,
                            Bool32 samplerAnisotropy_ = 0,
                            Bool32 textureCompressionETC2_ = 0,
                            Bool32 textureCompressionASTC_LDR_ = 0,
                            Bool32 textureCompressionBC_ = 0,
                            Bool32 occlusionQueryPrecise_ = 0,
                            Bool32 pipelineStatisticsQuery_ = 0,
                            Bool32 vertexPipelineStoresAndAtomics_ = 0,
                            Bool32 fragmentStoresAndAtomics_ = 0,
                            Bool32 shaderTessellationAndGeometryPointSize_ = 0,
                            Bool32 shaderImageGatherExtended_ = 0,
                            Bool32 shaderStorageImageExtendedFormats_ = 0,
                            Bool32 shaderStorageImageMultisample_ = 0,
                            Bool32 shaderStorageImageReadWithoutFormat_ = 0,
                            Bool32 shaderStorageImageWriteWithoutFormat_ = 0,
                            Bool32 shaderUniformBufferArrayDynamicIndexing_ = 0,
                            Bool32 shaderSampledImageArrayDynamicIndexing_ = 0,
                            Bool32 shaderStorageBufferArrayDynamicIndexing_ = 0,
                            Bool32 shaderStorageImageArrayDynamicIndexing_ = 0,
                            Bool32 shaderClipDistance_ = 0,
                            Bool32 shaderCullDistance_ = 0,
                            Bool32 shaderFloat64_ = 0,
                            Bool32 shaderInt64_ = 0,
                            Bool32 shaderInt16_ = 0,
                            Bool32 shaderResourceResidency_ = 0,
                            Bool32 shaderResourceMinLod_ = 0,
                            Bool32 sparseBinding_ = 0,
                            Bool32 sparseResidencyBuffer_ = 0,
                            Bool32 sparseResidencyImage2D_ = 0,
                            Bool32 sparseResidencyImage3D_ = 0,
                            Bool32 sparseResidency2Samples_ = 0,
                            Bool32 sparseResidency4Samples_ = 0,
                            Bool32 sparseResidency8Samples_ = 0,
                            Bool32 sparseResidency16Samples_ = 0,
                            Bool32 sparseResidencyAliased_ = 0,
                            Bool32 variableMultisampleRate_ = 0,
                            Bool32 inheritedQueries_ = 0 )
      : robustBufferAccess( robustBufferAccess_ )
      , fullDrawIndexUint32( fullDrawIndexUint32_ )
      , imageCubeArray( imageCubeArray_ )
      , independentBlend( independentBlend_ )
      , geometryShader( geometryShader_ )
      , tessellationShader( tessellationShader_ )
      , sampleRateShading( sampleRateShading_ )
      , dualSrcBlend( dualSrcBlend_ )
      , logicOp( logicOp_ )
      , multiDrawIndirect( multiDrawIndirect_ )
      , drawIndirectFirstInstance( drawIndirectFirstInstance_ )
      , depthClamp( depthClamp_ )
      , depthBiasClamp( depthBiasClamp_ )
      , fillModeNonSolid( fillModeNonSolid_ )
      , depthBounds( depthBounds_ )
      , wideLines( wideLines_ )
      , largePoints( largePoints_ )
      , alphaToOne( alphaToOne_ )
      , multiViewport( multiViewport_ )
      , samplerAnisotropy( samplerAnisotropy_ )
      , textureCompressionETC2( textureCompressionETC2_ )
      , textureCompressionASTC_LDR( textureCompressionASTC_LDR_ )
      , textureCompressionBC( textureCompressionBC_ )
      , occlusionQueryPrecise( occlusionQueryPrecise_ )
      , pipelineStatisticsQuery( pipelineStatisticsQuery_ )
      , vertexPipelineStoresAndAtomics( vertexPipelineStoresAndAtomics_ )
      , fragmentStoresAndAtomics( fragmentStoresAndAtomics_ )
      , shaderTessellationAndGeometryPointSize( shaderTessellationAndGeometryPointSize_ )
      , shaderImageGatherExtended( shaderImageGatherExtended_ )
      , shaderStorageImageExtendedFormats( shaderStorageImageExtendedFormats_ )
      , shaderStorageImageMultisample( shaderStorageImageMultisample_ )
      , shaderStorageImageReadWithoutFormat( shaderStorageImageReadWithoutFormat_ )
      , shaderStorageImageWriteWithoutFormat( shaderStorageImageWriteWithoutFormat_ )
      , shaderUniformBufferArrayDynamicIndexing( shaderUniformBufferArrayDynamicIndexing_ )
      , shaderSampledImageArrayDynamicIndexing( shaderSampledImageArrayDynamicIndexing_ )
      , shaderStorageBufferArrayDynamicIndexing( shaderStorageBufferArrayDynamicIndexing_ )
      , shaderStorageImageArrayDynamicIndexing( shaderStorageImageArrayDynamicIndexing_ )
      , shaderClipDistance( shaderClipDistance_ )
      , shaderCullDistance( shaderCullDistance_ )
      , shaderFloat64( shaderFloat64_ )
      , shaderInt64( shaderInt64_ )
      , shaderInt16( shaderInt16_ )
      , shaderResourceResidency( shaderResourceResidency_ )
      , shaderResourceMinLod( shaderResourceMinLod_ )
      , sparseBinding( sparseBinding_ )
      , sparseResidencyBuffer( sparseResidencyBuffer_ )
      , sparseResidencyImage2D( sparseResidencyImage2D_ )
      , sparseResidencyImage3D( sparseResidencyImage3D_ )
      , sparseResidency2Samples( sparseResidency2Samples_ )
      , sparseResidency4Samples( sparseResidency4Samples_ )
      , sparseResidency8Samples( sparseResidency8Samples_ )
      , sparseResidency16Samples( sparseResidency16Samples_ )
      , sparseResidencyAliased( sparseResidencyAliased_ )
      , variableMultisampleRate( variableMultisampleRate_ )
      , inheritedQueries( inheritedQueries_ )
    {
    }

    PhysicalDeviceFeatures( VkPhysicalDeviceFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceFeatures ) );
    }

    PhysicalDeviceFeatures& operator=( VkPhysicalDeviceFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceFeatures ) );
      return *this;
    }
    PhysicalDeviceFeatures& setRobustBufferAccess( Bool32 robustBufferAccess_ )
    {
      robustBufferAccess = robustBufferAccess_;
      return *this;
    }

    PhysicalDeviceFeatures& setFullDrawIndexUint32( Bool32 fullDrawIndexUint32_ )
    {
      fullDrawIndexUint32 = fullDrawIndexUint32_;
      return *this;
    }

    PhysicalDeviceFeatures& setImageCubeArray( Bool32 imageCubeArray_ )
    {
      imageCubeArray = imageCubeArray_;
      return *this;
    }

    PhysicalDeviceFeatures& setIndependentBlend( Bool32 independentBlend_ )
    {
      independentBlend = independentBlend_;
      return *this;
    }

    PhysicalDeviceFeatures& setGeometryShader( Bool32 geometryShader_ )
    {
      geometryShader = geometryShader_;
      return *this;
    }

    PhysicalDeviceFeatures& setTessellationShader( Bool32 tessellationShader_ )
    {
      tessellationShader = tessellationShader_;
      return *this;
    }

    PhysicalDeviceFeatures& setSampleRateShading( Bool32 sampleRateShading_ )
    {
      sampleRateShading = sampleRateShading_;
      return *this;
    }

    PhysicalDeviceFeatures& setDualSrcBlend( Bool32 dualSrcBlend_ )
    {
      dualSrcBlend = dualSrcBlend_;
      return *this;
    }

    PhysicalDeviceFeatures& setLogicOp( Bool32 logicOp_ )
    {
      logicOp = logicOp_;
      return *this;
    }

    PhysicalDeviceFeatures& setMultiDrawIndirect( Bool32 multiDrawIndirect_ )
    {
      multiDrawIndirect = multiDrawIndirect_;
      return *this;
    }

    PhysicalDeviceFeatures& setDrawIndirectFirstInstance( Bool32 drawIndirectFirstInstance_ )
    {
      drawIndirectFirstInstance = drawIndirectFirstInstance_;
      return *this;
    }

    PhysicalDeviceFeatures& setDepthClamp( Bool32 depthClamp_ )
    {
      depthClamp = depthClamp_;
      return *this;
    }

    PhysicalDeviceFeatures& setDepthBiasClamp( Bool32 depthBiasClamp_ )
    {
      depthBiasClamp = depthBiasClamp_;
      return *this;
    }

    PhysicalDeviceFeatures& setFillModeNonSolid( Bool32 fillModeNonSolid_ )
    {
      fillModeNonSolid = fillModeNonSolid_;
      return *this;
    }

    PhysicalDeviceFeatures& setDepthBounds( Bool32 depthBounds_ )
    {
      depthBounds = depthBounds_;
      return *this;
    }

    PhysicalDeviceFeatures& setWideLines( Bool32 wideLines_ )
    {
      wideLines = wideLines_;
      return *this;
    }

    PhysicalDeviceFeatures& setLargePoints( Bool32 largePoints_ )
    {
      largePoints = largePoints_;
      return *this;
    }

    PhysicalDeviceFeatures& setAlphaToOne( Bool32 alphaToOne_ )
    {
      alphaToOne = alphaToOne_;
      return *this;
    }

    PhysicalDeviceFeatures& setMultiViewport( Bool32 multiViewport_ )
    {
      multiViewport = multiViewport_;
      return *this;
    }

    PhysicalDeviceFeatures& setSamplerAnisotropy( Bool32 samplerAnisotropy_ )
    {
      samplerAnisotropy = samplerAnisotropy_;
      return *this;
    }

    PhysicalDeviceFeatures& setTextureCompressionETC2( Bool32 textureCompressionETC2_ )
    {
      textureCompressionETC2 = textureCompressionETC2_;
      return *this;
    }

    PhysicalDeviceFeatures& setTextureCompressionASTC_LDR( Bool32 textureCompressionASTC_LDR_ )
    {
      textureCompressionASTC_LDR = textureCompressionASTC_LDR_;
      return *this;
    }

    PhysicalDeviceFeatures& setTextureCompressionBC( Bool32 textureCompressionBC_ )
    {
      textureCompressionBC = textureCompressionBC_;
      return *this;
    }

    PhysicalDeviceFeatures& setOcclusionQueryPrecise( Bool32 occlusionQueryPrecise_ )
    {
      occlusionQueryPrecise = occlusionQueryPrecise_;
      return *this;
    }

    PhysicalDeviceFeatures& setPipelineStatisticsQuery( Bool32 pipelineStatisticsQuery_ )
    {
      pipelineStatisticsQuery = pipelineStatisticsQuery_;
      return *this;
    }

    PhysicalDeviceFeatures& setVertexPipelineStoresAndAtomics( Bool32 vertexPipelineStoresAndAtomics_ )
    {
      vertexPipelineStoresAndAtomics = vertexPipelineStoresAndAtomics_;
      return *this;
    }

    PhysicalDeviceFeatures& setFragmentStoresAndAtomics( Bool32 fragmentStoresAndAtomics_ )
    {
      fragmentStoresAndAtomics = fragmentStoresAndAtomics_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderTessellationAndGeometryPointSize( Bool32 shaderTessellationAndGeometryPointSize_ )
    {
      shaderTessellationAndGeometryPointSize = shaderTessellationAndGeometryPointSize_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderImageGatherExtended( Bool32 shaderImageGatherExtended_ )
    {
      shaderImageGatherExtended = shaderImageGatherExtended_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderStorageImageExtendedFormats( Bool32 shaderStorageImageExtendedFormats_ )
    {
      shaderStorageImageExtendedFormats = shaderStorageImageExtendedFormats_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderStorageImageMultisample( Bool32 shaderStorageImageMultisample_ )
    {
      shaderStorageImageMultisample = shaderStorageImageMultisample_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderStorageImageReadWithoutFormat( Bool32 shaderStorageImageReadWithoutFormat_ )
    {
      shaderStorageImageReadWithoutFormat = shaderStorageImageReadWithoutFormat_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderStorageImageWriteWithoutFormat( Bool32 shaderStorageImageWriteWithoutFormat_ )
    {
      shaderStorageImageWriteWithoutFormat = shaderStorageImageWriteWithoutFormat_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderUniformBufferArrayDynamicIndexing( Bool32 shaderUniformBufferArrayDynamicIndexing_ )
    {
      shaderUniformBufferArrayDynamicIndexing = shaderUniformBufferArrayDynamicIndexing_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderSampledImageArrayDynamicIndexing( Bool32 shaderSampledImageArrayDynamicIndexing_ )
    {
      shaderSampledImageArrayDynamicIndexing = shaderSampledImageArrayDynamicIndexing_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderStorageBufferArrayDynamicIndexing( Bool32 shaderStorageBufferArrayDynamicIndexing_ )
    {
      shaderStorageBufferArrayDynamicIndexing = shaderStorageBufferArrayDynamicIndexing_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderStorageImageArrayDynamicIndexing( Bool32 shaderStorageImageArrayDynamicIndexing_ )
    {
      shaderStorageImageArrayDynamicIndexing = shaderStorageImageArrayDynamicIndexing_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderClipDistance( Bool32 shaderClipDistance_ )
    {
      shaderClipDistance = shaderClipDistance_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderCullDistance( Bool32 shaderCullDistance_ )
    {
      shaderCullDistance = shaderCullDistance_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderFloat64( Bool32 shaderFloat64_ )
    {
      shaderFloat64 = shaderFloat64_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderInt64( Bool32 shaderInt64_ )
    {
      shaderInt64 = shaderInt64_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderInt16( Bool32 shaderInt16_ )
    {
      shaderInt16 = shaderInt16_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderResourceResidency( Bool32 shaderResourceResidency_ )
    {
      shaderResourceResidency = shaderResourceResidency_;
      return *this;
    }

    PhysicalDeviceFeatures& setShaderResourceMinLod( Bool32 shaderResourceMinLod_ )
    {
      shaderResourceMinLod = shaderResourceMinLod_;
      return *this;
    }

    PhysicalDeviceFeatures& setSparseBinding( Bool32 sparseBinding_ )
    {
      sparseBinding = sparseBinding_;
      return *this;
    }

    PhysicalDeviceFeatures& setSparseResidencyBuffer( Bool32 sparseResidencyBuffer_ )
    {
      sparseResidencyBuffer = sparseResidencyBuffer_;
      return *this;
    }

    PhysicalDeviceFeatures& setSparseResidencyImage2D( Bool32 sparseResidencyImage2D_ )
    {
      sparseResidencyImage2D = sparseResidencyImage2D_;
      return *this;
    }

    PhysicalDeviceFeatures& setSparseResidencyImage3D( Bool32 sparseResidencyImage3D_ )
    {
      sparseResidencyImage3D = sparseResidencyImage3D_;
      return *this;
    }

    PhysicalDeviceFeatures& setSparseResidency2Samples( Bool32 sparseResidency2Samples_ )
    {
      sparseResidency2Samples = sparseResidency2Samples_;
      return *this;
    }

    PhysicalDeviceFeatures& setSparseResidency4Samples( Bool32 sparseResidency4Samples_ )
    {
      sparseResidency4Samples = sparseResidency4Samples_;
      return *this;
    }

    PhysicalDeviceFeatures& setSparseResidency8Samples( Bool32 sparseResidency8Samples_ )
    {
      sparseResidency8Samples = sparseResidency8Samples_;
      return *this;
    }

    PhysicalDeviceFeatures& setSparseResidency16Samples( Bool32 sparseResidency16Samples_ )
    {
      sparseResidency16Samples = sparseResidency16Samples_;
      return *this;
    }

    PhysicalDeviceFeatures& setSparseResidencyAliased( Bool32 sparseResidencyAliased_ )
    {
      sparseResidencyAliased = sparseResidencyAliased_;
      return *this;
    }

    PhysicalDeviceFeatures& setVariableMultisampleRate( Bool32 variableMultisampleRate_ )
    {
      variableMultisampleRate = variableMultisampleRate_;
      return *this;
    }

    PhysicalDeviceFeatures& setInheritedQueries( Bool32 inheritedQueries_ )
    {
      inheritedQueries = inheritedQueries_;
      return *this;
    }

    operator VkPhysicalDeviceFeatures const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceFeatures*>(this);
    }

    operator VkPhysicalDeviceFeatures &()
    {
      return *reinterpret_cast<VkPhysicalDeviceFeatures*>(this);
    }

    bool operator==( PhysicalDeviceFeatures const& rhs ) const
    {
      return ( robustBufferAccess == rhs.robustBufferAccess )
          && ( fullDrawIndexUint32 == rhs.fullDrawIndexUint32 )
          && ( imageCubeArray == rhs.imageCubeArray )
          && ( independentBlend == rhs.independentBlend )
          && ( geometryShader == rhs.geometryShader )
          && ( tessellationShader == rhs.tessellationShader )
          && ( sampleRateShading == rhs.sampleRateShading )
          && ( dualSrcBlend == rhs.dualSrcBlend )
          && ( logicOp == rhs.logicOp )
          && ( multiDrawIndirect == rhs.multiDrawIndirect )
          && ( drawIndirectFirstInstance == rhs.drawIndirectFirstInstance )
          && ( depthClamp == rhs.depthClamp )
          && ( depthBiasClamp == rhs.depthBiasClamp )
          && ( fillModeNonSolid == rhs.fillModeNonSolid )
          && ( depthBounds == rhs.depthBounds )
          && ( wideLines == rhs.wideLines )
          && ( largePoints == rhs.largePoints )
          && ( alphaToOne == rhs.alphaToOne )
          && ( multiViewport == rhs.multiViewport )
          && ( samplerAnisotropy == rhs.samplerAnisotropy )
          && ( textureCompressionETC2 == rhs.textureCompressionETC2 )
          && ( textureCompressionASTC_LDR == rhs.textureCompressionASTC_LDR )
          && ( textureCompressionBC == rhs.textureCompressionBC )
          && ( occlusionQueryPrecise == rhs.occlusionQueryPrecise )
          && ( pipelineStatisticsQuery == rhs.pipelineStatisticsQuery )
          && ( vertexPipelineStoresAndAtomics == rhs.vertexPipelineStoresAndAtomics )
          && ( fragmentStoresAndAtomics == rhs.fragmentStoresAndAtomics )
          && ( shaderTessellationAndGeometryPointSize == rhs.shaderTessellationAndGeometryPointSize )
          && ( shaderImageGatherExtended == rhs.shaderImageGatherExtended )
          && ( shaderStorageImageExtendedFormats == rhs.shaderStorageImageExtendedFormats )
          && ( shaderStorageImageMultisample == rhs.shaderStorageImageMultisample )
          && ( shaderStorageImageReadWithoutFormat == rhs.shaderStorageImageReadWithoutFormat )
          && ( shaderStorageImageWriteWithoutFormat == rhs.shaderStorageImageWriteWithoutFormat )
          && ( shaderUniformBufferArrayDynamicIndexing == rhs.shaderUniformBufferArrayDynamicIndexing )
          && ( shaderSampledImageArrayDynamicIndexing == rhs.shaderSampledImageArrayDynamicIndexing )
          && ( shaderStorageBufferArrayDynamicIndexing == rhs.shaderStorageBufferArrayDynamicIndexing )
          && ( shaderStorageImageArrayDynamicIndexing == rhs.shaderStorageImageArrayDynamicIndexing )
          && ( shaderClipDistance == rhs.shaderClipDistance )
          && ( shaderCullDistance == rhs.shaderCullDistance )
          && ( shaderFloat64 == rhs.shaderFloat64 )
          && ( shaderInt64 == rhs.shaderInt64 )
          && ( shaderInt16 == rhs.shaderInt16 )
          && ( shaderResourceResidency == rhs.shaderResourceResidency )
          && ( shaderResourceMinLod == rhs.shaderResourceMinLod )
          && ( sparseBinding == rhs.sparseBinding )
          && ( sparseResidencyBuffer == rhs.sparseResidencyBuffer )
          && ( sparseResidencyImage2D == rhs.sparseResidencyImage2D )
          && ( sparseResidencyImage3D == rhs.sparseResidencyImage3D )
          && ( sparseResidency2Samples == rhs.sparseResidency2Samples )
          && ( sparseResidency4Samples == rhs.sparseResidency4Samples )
          && ( sparseResidency8Samples == rhs.sparseResidency8Samples )
          && ( sparseResidency16Samples == rhs.sparseResidency16Samples )
          && ( sparseResidencyAliased == rhs.sparseResidencyAliased )
          && ( variableMultisampleRate == rhs.variableMultisampleRate )
          && ( inheritedQueries == rhs.inheritedQueries );
    }

    bool operator!=( PhysicalDeviceFeatures const& rhs ) const
    {
      return !operator==( rhs );
    }

    Bool32 robustBufferAccess;
    Bool32 fullDrawIndexUint32;
    Bool32 imageCubeArray;
    Bool32 independentBlend;
    Bool32 geometryShader;
    Bool32 tessellationShader;
    Bool32 sampleRateShading;
    Bool32 dualSrcBlend;
    Bool32 logicOp;
    Bool32 multiDrawIndirect;
    Bool32 drawIndirectFirstInstance;
    Bool32 depthClamp;
    Bool32 depthBiasClamp;
    Bool32 fillModeNonSolid;
    Bool32 depthBounds;
    Bool32 wideLines;
    Bool32 largePoints;
    Bool32 alphaToOne;
    Bool32 multiViewport;
    Bool32 samplerAnisotropy;
    Bool32 textureCompressionETC2;
    Bool32 textureCompressionASTC_LDR;
    Bool32 textureCompressionBC;
    Bool32 occlusionQueryPrecise;
    Bool32 pipelineStatisticsQuery;
    Bool32 vertexPipelineStoresAndAtomics;
    Bool32 fragmentStoresAndAtomics;
    Bool32 shaderTessellationAndGeometryPointSize;
    Bool32 shaderImageGatherExtended;
    Bool32 shaderStorageImageExtendedFormats;
    Bool32 shaderStorageImageMultisample;
    Bool32 shaderStorageImageReadWithoutFormat;
    Bool32 shaderStorageImageWriteWithoutFormat;
    Bool32 shaderUniformBufferArrayDynamicIndexing;
    Bool32 shaderSampledImageArrayDynamicIndexing;
    Bool32 shaderStorageBufferArrayDynamicIndexing;
    Bool32 shaderStorageImageArrayDynamicIndexing;
    Bool32 shaderClipDistance;
    Bool32 shaderCullDistance;
    Bool32 shaderFloat64;
    Bool32 shaderInt64;
    Bool32 shaderInt16;
    Bool32 shaderResourceResidency;
    Bool32 shaderResourceMinLod;
    Bool32 sparseBinding;
    Bool32 sparseResidencyBuffer;
    Bool32 sparseResidencyImage2D;
    Bool32 sparseResidencyImage3D;
    Bool32 sparseResidency2Samples;
    Bool32 sparseResidency4Samples;
    Bool32 sparseResidency8Samples;
    Bool32 sparseResidency16Samples;
    Bool32 sparseResidencyAliased;
    Bool32 variableMultisampleRate;
    Bool32 inheritedQueries;
  };
  static_assert( sizeof( PhysicalDeviceFeatures ) == sizeof( VkPhysicalDeviceFeatures ), "struct and wrapper have different size!" );

  struct PhysicalDeviceSparseProperties
  {
    operator VkPhysicalDeviceSparseProperties const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceSparseProperties*>(this);
    }

    operator VkPhysicalDeviceSparseProperties &()
    {
      return *reinterpret_cast<VkPhysicalDeviceSparseProperties*>(this);
    }

    bool operator==( PhysicalDeviceSparseProperties const& rhs ) const
    {
      return ( residencyStandard2DBlockShape == rhs.residencyStandard2DBlockShape )
          && ( residencyStandard2DMultisampleBlockShape == rhs.residencyStandard2DMultisampleBlockShape )
          && ( residencyStandard3DBlockShape == rhs.residencyStandard3DBlockShape )
          && ( residencyAlignedMipSize == rhs.residencyAlignedMipSize )
          && ( residencyNonResidentStrict == rhs.residencyNonResidentStrict );
    }

    bool operator!=( PhysicalDeviceSparseProperties const& rhs ) const
    {
      return !operator==( rhs );
    }

    Bool32 residencyStandard2DBlockShape;
    Bool32 residencyStandard2DMultisampleBlockShape;
    Bool32 residencyStandard3DBlockShape;
    Bool32 residencyAlignedMipSize;
    Bool32 residencyNonResidentStrict;
  };
  static_assert( sizeof( PhysicalDeviceSparseProperties ) == sizeof( VkPhysicalDeviceSparseProperties ), "struct and wrapper have different size!" );

  struct DrawIndirectCommand
  {
    DrawIndirectCommand( uint32_t vertexCount_ = 0,
                         uint32_t instanceCount_ = 0,
                         uint32_t firstVertex_ = 0,
                         uint32_t firstInstance_ = 0 )
      : vertexCount( vertexCount_ )
      , instanceCount( instanceCount_ )
      , firstVertex( firstVertex_ )
      , firstInstance( firstInstance_ )
    {
    }

    DrawIndirectCommand( VkDrawIndirectCommand const & rhs )
    {
      memcpy( this, &rhs, sizeof( DrawIndirectCommand ) );
    }

    DrawIndirectCommand& operator=( VkDrawIndirectCommand const & rhs )
    {
      memcpy( this, &rhs, sizeof( DrawIndirectCommand ) );
      return *this;
    }
    DrawIndirectCommand& setVertexCount( uint32_t vertexCount_ )
    {
      vertexCount = vertexCount_;
      return *this;
    }

    DrawIndirectCommand& setInstanceCount( uint32_t instanceCount_ )
    {
      instanceCount = instanceCount_;
      return *this;
    }

    DrawIndirectCommand& setFirstVertex( uint32_t firstVertex_ )
    {
      firstVertex = firstVertex_;
      return *this;
    }

    DrawIndirectCommand& setFirstInstance( uint32_t firstInstance_ )
    {
      firstInstance = firstInstance_;
      return *this;
    }

    operator VkDrawIndirectCommand const&() const
    {
      return *reinterpret_cast<const VkDrawIndirectCommand*>(this);
    }

    operator VkDrawIndirectCommand &()
    {
      return *reinterpret_cast<VkDrawIndirectCommand*>(this);
    }

    bool operator==( DrawIndirectCommand const& rhs ) const
    {
      return ( vertexCount == rhs.vertexCount )
          && ( instanceCount == rhs.instanceCount )
          && ( firstVertex == rhs.firstVertex )
          && ( firstInstance == rhs.firstInstance );
    }

    bool operator!=( DrawIndirectCommand const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t vertexCount;
    uint32_t instanceCount;
    uint32_t firstVertex;
    uint32_t firstInstance;
  };
  static_assert( sizeof( DrawIndirectCommand ) == sizeof( VkDrawIndirectCommand ), "struct and wrapper have different size!" );

  struct DrawIndexedIndirectCommand
  {
    DrawIndexedIndirectCommand( uint32_t indexCount_ = 0,
                                uint32_t instanceCount_ = 0,
                                uint32_t firstIndex_ = 0,
                                int32_t vertexOffset_ = 0,
                                uint32_t firstInstance_ = 0 )
      : indexCount( indexCount_ )
      , instanceCount( instanceCount_ )
      , firstIndex( firstIndex_ )
      , vertexOffset( vertexOffset_ )
      , firstInstance( firstInstance_ )
    {
    }

    DrawIndexedIndirectCommand( VkDrawIndexedIndirectCommand const & rhs )
    {
      memcpy( this, &rhs, sizeof( DrawIndexedIndirectCommand ) );
    }

    DrawIndexedIndirectCommand& operator=( VkDrawIndexedIndirectCommand const & rhs )
    {
      memcpy( this, &rhs, sizeof( DrawIndexedIndirectCommand ) );
      return *this;
    }
    DrawIndexedIndirectCommand& setIndexCount( uint32_t indexCount_ )
    {
      indexCount = indexCount_;
      return *this;
    }

    DrawIndexedIndirectCommand& setInstanceCount( uint32_t instanceCount_ )
    {
      instanceCount = instanceCount_;
      return *this;
    }

    DrawIndexedIndirectCommand& setFirstIndex( uint32_t firstIndex_ )
    {
      firstIndex = firstIndex_;
      return *this;
    }

    DrawIndexedIndirectCommand& setVertexOffset( int32_t vertexOffset_ )
    {
      vertexOffset = vertexOffset_;
      return *this;
    }

    DrawIndexedIndirectCommand& setFirstInstance( uint32_t firstInstance_ )
    {
      firstInstance = firstInstance_;
      return *this;
    }

    operator VkDrawIndexedIndirectCommand const&() const
    {
      return *reinterpret_cast<const VkDrawIndexedIndirectCommand*>(this);
    }

    operator VkDrawIndexedIndirectCommand &()
    {
      return *reinterpret_cast<VkDrawIndexedIndirectCommand*>(this);
    }

    bool operator==( DrawIndexedIndirectCommand const& rhs ) const
    {
      return ( indexCount == rhs.indexCount )
          && ( instanceCount == rhs.instanceCount )
          && ( firstIndex == rhs.firstIndex )
          && ( vertexOffset == rhs.vertexOffset )
          && ( firstInstance == rhs.firstInstance );
    }

    bool operator!=( DrawIndexedIndirectCommand const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t indexCount;
    uint32_t instanceCount;
    uint32_t firstIndex;
    int32_t vertexOffset;
    uint32_t firstInstance;
  };
  static_assert( sizeof( DrawIndexedIndirectCommand ) == sizeof( VkDrawIndexedIndirectCommand ), "struct and wrapper have different size!" );

  struct DispatchIndirectCommand
  {
    DispatchIndirectCommand( uint32_t x_ = 0,
                             uint32_t y_ = 0,
                             uint32_t z_ = 0 )
      : x( x_ )
      , y( y_ )
      , z( z_ )
    {
    }

    DispatchIndirectCommand( VkDispatchIndirectCommand const & rhs )
    {
      memcpy( this, &rhs, sizeof( DispatchIndirectCommand ) );
    }

    DispatchIndirectCommand& operator=( VkDispatchIndirectCommand const & rhs )
    {
      memcpy( this, &rhs, sizeof( DispatchIndirectCommand ) );
      return *this;
    }
    DispatchIndirectCommand& setX( uint32_t x_ )
    {
      x = x_;
      return *this;
    }

    DispatchIndirectCommand& setY( uint32_t y_ )
    {
      y = y_;
      return *this;
    }

    DispatchIndirectCommand& setZ( uint32_t z_ )
    {
      z = z_;
      return *this;
    }

    operator VkDispatchIndirectCommand const&() const
    {
      return *reinterpret_cast<const VkDispatchIndirectCommand*>(this);
    }

    operator VkDispatchIndirectCommand &()
    {
      return *reinterpret_cast<VkDispatchIndirectCommand*>(this);
    }

    bool operator==( DispatchIndirectCommand const& rhs ) const
    {
      return ( x == rhs.x )
          && ( y == rhs.y )
          && ( z == rhs.z );
    }

    bool operator!=( DispatchIndirectCommand const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t x;
    uint32_t y;
    uint32_t z;
  };
  static_assert( sizeof( DispatchIndirectCommand ) == sizeof( VkDispatchIndirectCommand ), "struct and wrapper have different size!" );

  struct DisplayPlanePropertiesKHR
  {
    operator VkDisplayPlanePropertiesKHR const&() const
    {
      return *reinterpret_cast<const VkDisplayPlanePropertiesKHR*>(this);
    }

    operator VkDisplayPlanePropertiesKHR &()
    {
      return *reinterpret_cast<VkDisplayPlanePropertiesKHR*>(this);
    }

    bool operator==( DisplayPlanePropertiesKHR const& rhs ) const
    {
      return ( currentDisplay == rhs.currentDisplay )
          && ( currentStackIndex == rhs.currentStackIndex );
    }

    bool operator!=( DisplayPlanePropertiesKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

    DisplayKHR currentDisplay;
    uint32_t currentStackIndex;
  };
  static_assert( sizeof( DisplayPlanePropertiesKHR ) == sizeof( VkDisplayPlanePropertiesKHR ), "struct and wrapper have different size!" );

  struct DisplayModeParametersKHR
  {
    DisplayModeParametersKHR( Extent2D visibleRegion_ = Extent2D(),
                              uint32_t refreshRate_ = 0 )
      : visibleRegion( visibleRegion_ )
      , refreshRate( refreshRate_ )
    {
    }

    DisplayModeParametersKHR( VkDisplayModeParametersKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( DisplayModeParametersKHR ) );
    }

    DisplayModeParametersKHR& operator=( VkDisplayModeParametersKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( DisplayModeParametersKHR ) );
      return *this;
    }
    DisplayModeParametersKHR& setVisibleRegion( Extent2D visibleRegion_ )
    {
      visibleRegion = visibleRegion_;
      return *this;
    }

    DisplayModeParametersKHR& setRefreshRate( uint32_t refreshRate_ )
    {
      refreshRate = refreshRate_;
      return *this;
    }

    operator VkDisplayModeParametersKHR const&() const
    {
      return *reinterpret_cast<const VkDisplayModeParametersKHR*>(this);
    }

    operator VkDisplayModeParametersKHR &()
    {
      return *reinterpret_cast<VkDisplayModeParametersKHR*>(this);
    }

    bool operator==( DisplayModeParametersKHR const& rhs ) const
    {
      return ( visibleRegion == rhs.visibleRegion )
          && ( refreshRate == rhs.refreshRate );
    }

    bool operator!=( DisplayModeParametersKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

    Extent2D visibleRegion;
    uint32_t refreshRate;
  };
  static_assert( sizeof( DisplayModeParametersKHR ) == sizeof( VkDisplayModeParametersKHR ), "struct and wrapper have different size!" );

  struct DisplayModePropertiesKHR
  {
    operator VkDisplayModePropertiesKHR const&() const
    {
      return *reinterpret_cast<const VkDisplayModePropertiesKHR*>(this);
    }

    operator VkDisplayModePropertiesKHR &()
    {
      return *reinterpret_cast<VkDisplayModePropertiesKHR*>(this);
    }

    bool operator==( DisplayModePropertiesKHR const& rhs ) const
    {
      return ( displayMode == rhs.displayMode )
          && ( parameters == rhs.parameters );
    }

    bool operator!=( DisplayModePropertiesKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

    DisplayModeKHR displayMode;
    DisplayModeParametersKHR parameters;
  };
  static_assert( sizeof( DisplayModePropertiesKHR ) == sizeof( VkDisplayModePropertiesKHR ), "struct and wrapper have different size!" );

  struct ConformanceVersionKHR
  {
    ConformanceVersionKHR( uint8_t major_ = 0,
                           uint8_t minor_ = 0,
                           uint8_t subminor_ = 0,
                           uint8_t patch_ = 0 )
      : major( major_ )
      , minor( minor_ )
      , subminor( subminor_ )
      , patch( patch_ )
    {
    }

    ConformanceVersionKHR( VkConformanceVersionKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ConformanceVersionKHR ) );
    }

    ConformanceVersionKHR& operator=( VkConformanceVersionKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ConformanceVersionKHR ) );
      return *this;
    }
    ConformanceVersionKHR& setMajor( uint8_t major_ )
    {
      major = major_;
      return *this;
    }

    ConformanceVersionKHR& setMinor( uint8_t minor_ )
    {
      minor = minor_;
      return *this;
    }

    ConformanceVersionKHR& setSubminor( uint8_t subminor_ )
    {
      subminor = subminor_;
      return *this;
    }

    ConformanceVersionKHR& setPatch( uint8_t patch_ )
    {
      patch = patch_;
      return *this;
    }

    operator VkConformanceVersionKHR const&() const
    {
      return *reinterpret_cast<const VkConformanceVersionKHR*>(this);
    }

    operator VkConformanceVersionKHR &()
    {
      return *reinterpret_cast<VkConformanceVersionKHR*>(this);
    }

    bool operator==( ConformanceVersionKHR const& rhs ) const
    {
      return ( major == rhs.major )
          && ( minor == rhs.minor )
          && ( subminor == rhs.subminor )
          && ( patch == rhs.patch );
    }

    bool operator!=( ConformanceVersionKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint8_t major;
    uint8_t minor;
    uint8_t subminor;
    uint8_t patch;
  };
  static_assert( sizeof( ConformanceVersionKHR ) == sizeof( VkConformanceVersionKHR ), "struct and wrapper have different size!" );

  struct RectLayerKHR
  {
    RectLayerKHR( Offset2D offset_ = Offset2D(),
                  Extent2D extent_ = Extent2D(),
                  uint32_t layer_ = 0 )
      : offset( offset_ )
      , extent( extent_ )
      , layer( layer_ )
    {
    }

    explicit RectLayerKHR( Rect2D const& rect2D,
                           uint32_t layer_ = 0 )
      : offset( rect2D.offset )
      , extent( rect2D.extent )
      , layer( layer_ )
    {}

    RectLayerKHR( VkRectLayerKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( RectLayerKHR ) );
    }

    RectLayerKHR& operator=( VkRectLayerKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( RectLayerKHR ) );
      return *this;
    }
    RectLayerKHR& setOffset( Offset2D offset_ )
    {
      offset = offset_;
      return *this;
    }

    RectLayerKHR& setExtent( Extent2D extent_ )
    {
      extent = extent_;
      return *this;
    }

    RectLayerKHR& setLayer( uint32_t layer_ )
    {
      layer = layer_;
      return *this;
    }

    operator VkRectLayerKHR const&() const
    {
      return *reinterpret_cast<const VkRectLayerKHR*>(this);
    }

    operator VkRectLayerKHR &()
    {
      return *reinterpret_cast<VkRectLayerKHR*>(this);
    }

    bool operator==( RectLayerKHR const& rhs ) const
    {
      return ( offset == rhs.offset )
          && ( extent == rhs.extent )
          && ( layer == rhs.layer );
    }

    bool operator!=( RectLayerKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

    Offset2D offset;
    Extent2D extent;
    uint32_t layer;
  };
  static_assert( sizeof( RectLayerKHR ) == sizeof( VkRectLayerKHR ), "struct and wrapper have different size!" );

  struct PresentRegionKHR
  {
    PresentRegionKHR( uint32_t rectangleCount_ = 0,
                      const RectLayerKHR* pRectangles_ = nullptr )
      : rectangleCount( rectangleCount_ )
      , pRectangles( pRectangles_ )
    {
    }

    PresentRegionKHR( VkPresentRegionKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PresentRegionKHR ) );
    }

    PresentRegionKHR& operator=( VkPresentRegionKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PresentRegionKHR ) );
      return *this;
    }
    PresentRegionKHR& setRectangleCount( uint32_t rectangleCount_ )
    {
      rectangleCount = rectangleCount_;
      return *this;
    }

    PresentRegionKHR& setPRectangles( const RectLayerKHR* pRectangles_ )
    {
      pRectangles = pRectangles_;
      return *this;
    }

    operator VkPresentRegionKHR const&() const
    {
      return *reinterpret_cast<const VkPresentRegionKHR*>(this);
    }

    operator VkPresentRegionKHR &()
    {
      return *reinterpret_cast<VkPresentRegionKHR*>(this);
    }

    bool operator==( PresentRegionKHR const& rhs ) const
    {
      return ( rectangleCount == rhs.rectangleCount )
          && ( pRectangles == rhs.pRectangles );
    }

    bool operator!=( PresentRegionKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t rectangleCount;
    const RectLayerKHR* pRectangles;
  };
  static_assert( sizeof( PresentRegionKHR ) == sizeof( VkPresentRegionKHR ), "struct and wrapper have different size!" );

  struct XYColorEXT
  {
    XYColorEXT( float x_ = 0,
                float y_ = 0 )
      : x( x_ )
      , y( y_ )
    {
    }

    XYColorEXT( VkXYColorEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( XYColorEXT ) );
    }

    XYColorEXT& operator=( VkXYColorEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( XYColorEXT ) );
      return *this;
    }
    XYColorEXT& setX( float x_ )
    {
      x = x_;
      return *this;
    }

    XYColorEXT& setY( float y_ )
    {
      y = y_;
      return *this;
    }

    operator VkXYColorEXT const&() const
    {
      return *reinterpret_cast<const VkXYColorEXT*>(this);
    }

    operator VkXYColorEXT &()
    {
      return *reinterpret_cast<VkXYColorEXT*>(this);
    }

    bool operator==( XYColorEXT const& rhs ) const
    {
      return ( x == rhs.x )
          && ( y == rhs.y );
    }

    bool operator!=( XYColorEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

    float x;
    float y;
  };
  static_assert( sizeof( XYColorEXT ) == sizeof( VkXYColorEXT ), "struct and wrapper have different size!" );

  struct RefreshCycleDurationGOOGLE
  {
    operator VkRefreshCycleDurationGOOGLE const&() const
    {
      return *reinterpret_cast<const VkRefreshCycleDurationGOOGLE*>(this);
    }

    operator VkRefreshCycleDurationGOOGLE &()
    {
      return *reinterpret_cast<VkRefreshCycleDurationGOOGLE*>(this);
    }

    bool operator==( RefreshCycleDurationGOOGLE const& rhs ) const
    {
      return ( refreshDuration == rhs.refreshDuration );
    }

    bool operator!=( RefreshCycleDurationGOOGLE const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint64_t refreshDuration;
  };
  static_assert( sizeof( RefreshCycleDurationGOOGLE ) == sizeof( VkRefreshCycleDurationGOOGLE ), "struct and wrapper have different size!" );

  struct PastPresentationTimingGOOGLE
  {
    operator VkPastPresentationTimingGOOGLE const&() const
    {
      return *reinterpret_cast<const VkPastPresentationTimingGOOGLE*>(this);
    }

    operator VkPastPresentationTimingGOOGLE &()
    {
      return *reinterpret_cast<VkPastPresentationTimingGOOGLE*>(this);
    }

    bool operator==( PastPresentationTimingGOOGLE const& rhs ) const
    {
      return ( presentID == rhs.presentID )
          && ( desiredPresentTime == rhs.desiredPresentTime )
          && ( actualPresentTime == rhs.actualPresentTime )
          && ( earliestPresentTime == rhs.earliestPresentTime )
          && ( presentMargin == rhs.presentMargin );
    }

    bool operator!=( PastPresentationTimingGOOGLE const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t presentID;
    uint64_t desiredPresentTime;
    uint64_t actualPresentTime;
    uint64_t earliestPresentTime;
    uint64_t presentMargin;
  };
  static_assert( sizeof( PastPresentationTimingGOOGLE ) == sizeof( VkPastPresentationTimingGOOGLE ), "struct and wrapper have different size!" );

  struct PresentTimeGOOGLE
  {
    PresentTimeGOOGLE( uint32_t presentID_ = 0,
                       uint64_t desiredPresentTime_ = 0 )
      : presentID( presentID_ )
      , desiredPresentTime( desiredPresentTime_ )
    {
    }

    PresentTimeGOOGLE( VkPresentTimeGOOGLE const & rhs )
    {
      memcpy( this, &rhs, sizeof( PresentTimeGOOGLE ) );
    }

    PresentTimeGOOGLE& operator=( VkPresentTimeGOOGLE const & rhs )
    {
      memcpy( this, &rhs, sizeof( PresentTimeGOOGLE ) );
      return *this;
    }
    PresentTimeGOOGLE& setPresentID( uint32_t presentID_ )
    {
      presentID = presentID_;
      return *this;
    }

    PresentTimeGOOGLE& setDesiredPresentTime( uint64_t desiredPresentTime_ )
    {
      desiredPresentTime = desiredPresentTime_;
      return *this;
    }

    operator VkPresentTimeGOOGLE const&() const
    {
      return *reinterpret_cast<const VkPresentTimeGOOGLE*>(this);
    }

    operator VkPresentTimeGOOGLE &()
    {
      return *reinterpret_cast<VkPresentTimeGOOGLE*>(this);
    }

    bool operator==( PresentTimeGOOGLE const& rhs ) const
    {
      return ( presentID == rhs.presentID )
          && ( desiredPresentTime == rhs.desiredPresentTime );
    }

    bool operator!=( PresentTimeGOOGLE const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t presentID;
    uint64_t desiredPresentTime;
  };
  static_assert( sizeof( PresentTimeGOOGLE ) == sizeof( VkPresentTimeGOOGLE ), "struct and wrapper have different size!" );

  struct ViewportWScalingNV
  {
    ViewportWScalingNV( float xcoeff_ = 0,
                        float ycoeff_ = 0 )
      : xcoeff( xcoeff_ )
      , ycoeff( ycoeff_ )
    {
    }

    ViewportWScalingNV( VkViewportWScalingNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( ViewportWScalingNV ) );
    }

    ViewportWScalingNV& operator=( VkViewportWScalingNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( ViewportWScalingNV ) );
      return *this;
    }
    ViewportWScalingNV& setXcoeff( float xcoeff_ )
    {
      xcoeff = xcoeff_;
      return *this;
    }

    ViewportWScalingNV& setYcoeff( float ycoeff_ )
    {
      ycoeff = ycoeff_;
      return *this;
    }

    operator VkViewportWScalingNV const&() const
    {
      return *reinterpret_cast<const VkViewportWScalingNV*>(this);
    }

    operator VkViewportWScalingNV &()
    {
      return *reinterpret_cast<VkViewportWScalingNV*>(this);
    }

    bool operator==( ViewportWScalingNV const& rhs ) const
    {
      return ( xcoeff == rhs.xcoeff )
          && ( ycoeff == rhs.ycoeff );
    }

    bool operator!=( ViewportWScalingNV const& rhs ) const
    {
      return !operator==( rhs );
    }

    float xcoeff;
    float ycoeff;
  };
  static_assert( sizeof( ViewportWScalingNV ) == sizeof( VkViewportWScalingNV ), "struct and wrapper have different size!" );

  struct SampleLocationEXT
  {
    SampleLocationEXT( float x_ = 0,
                       float y_ = 0 )
      : x( x_ )
      , y( y_ )
    {
    }

    SampleLocationEXT( VkSampleLocationEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( SampleLocationEXT ) );
    }

    SampleLocationEXT& operator=( VkSampleLocationEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( SampleLocationEXT ) );
      return *this;
    }
    SampleLocationEXT& setX( float x_ )
    {
      x = x_;
      return *this;
    }

    SampleLocationEXT& setY( float y_ )
    {
      y = y_;
      return *this;
    }

    operator VkSampleLocationEXT const&() const
    {
      return *reinterpret_cast<const VkSampleLocationEXT*>(this);
    }

    operator VkSampleLocationEXT &()
    {
      return *reinterpret_cast<VkSampleLocationEXT*>(this);
    }

    bool operator==( SampleLocationEXT const& rhs ) const
    {
      return ( x == rhs.x )
          && ( y == rhs.y );
    }

    bool operator!=( SampleLocationEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

    float x;
    float y;
  };
  static_assert( sizeof( SampleLocationEXT ) == sizeof( VkSampleLocationEXT ), "struct and wrapper have different size!" );

  struct ShaderResourceUsageAMD
  {
    operator VkShaderResourceUsageAMD const&() const
    {
      return *reinterpret_cast<const VkShaderResourceUsageAMD*>(this);
    }

    operator VkShaderResourceUsageAMD &()
    {
      return *reinterpret_cast<VkShaderResourceUsageAMD*>(this);
    }

    bool operator==( ShaderResourceUsageAMD const& rhs ) const
    {
      return ( numUsedVgprs == rhs.numUsedVgprs )
          && ( numUsedSgprs == rhs.numUsedSgprs )
          && ( ldsSizePerLocalWorkGroup == rhs.ldsSizePerLocalWorkGroup )
          && ( ldsUsageSizeInBytes == rhs.ldsUsageSizeInBytes )
          && ( scratchMemUsageInBytes == rhs.scratchMemUsageInBytes );
    }

    bool operator!=( ShaderResourceUsageAMD const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t numUsedVgprs;
    uint32_t numUsedSgprs;
    uint32_t ldsSizePerLocalWorkGroup;
    size_t ldsUsageSizeInBytes;
    size_t scratchMemUsageInBytes;
  };
  static_assert( sizeof( ShaderResourceUsageAMD ) == sizeof( VkShaderResourceUsageAMD ), "struct and wrapper have different size!" );

  struct VertexInputBindingDivisorDescriptionEXT
  {
    VertexInputBindingDivisorDescriptionEXT( uint32_t binding_ = 0,
                                             uint32_t divisor_ = 0 )
      : binding( binding_ )
      , divisor( divisor_ )
    {
    }

    VertexInputBindingDivisorDescriptionEXT( VkVertexInputBindingDivisorDescriptionEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( VertexInputBindingDivisorDescriptionEXT ) );
    }

    VertexInputBindingDivisorDescriptionEXT& operator=( VkVertexInputBindingDivisorDescriptionEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( VertexInputBindingDivisorDescriptionEXT ) );
      return *this;
    }
    VertexInputBindingDivisorDescriptionEXT& setBinding( uint32_t binding_ )
    {
      binding = binding_;
      return *this;
    }

    VertexInputBindingDivisorDescriptionEXT& setDivisor( uint32_t divisor_ )
    {
      divisor = divisor_;
      return *this;
    }

    operator VkVertexInputBindingDivisorDescriptionEXT const&() const
    {
      return *reinterpret_cast<const VkVertexInputBindingDivisorDescriptionEXT*>(this);
    }

    operator VkVertexInputBindingDivisorDescriptionEXT &()
    {
      return *reinterpret_cast<VkVertexInputBindingDivisorDescriptionEXT*>(this);
    }

    bool operator==( VertexInputBindingDivisorDescriptionEXT const& rhs ) const
    {
      return ( binding == rhs.binding )
          && ( divisor == rhs.divisor );
    }

    bool operator!=( VertexInputBindingDivisorDescriptionEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t binding;
    uint32_t divisor;
  };
  static_assert( sizeof( VertexInputBindingDivisorDescriptionEXT ) == sizeof( VkVertexInputBindingDivisorDescriptionEXT ), "struct and wrapper have different size!" );

  struct CoarseSampleLocationNV
  {
    CoarseSampleLocationNV( uint32_t pixelX_ = 0,
                            uint32_t pixelY_ = 0,
                            uint32_t sample_ = 0 )
      : pixelX( pixelX_ )
      , pixelY( pixelY_ )
      , sample( sample_ )
    {
    }

    CoarseSampleLocationNV( VkCoarseSampleLocationNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( CoarseSampleLocationNV ) );
    }

    CoarseSampleLocationNV& operator=( VkCoarseSampleLocationNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( CoarseSampleLocationNV ) );
      return *this;
    }
    CoarseSampleLocationNV& setPixelX( uint32_t pixelX_ )
    {
      pixelX = pixelX_;
      return *this;
    }

    CoarseSampleLocationNV& setPixelY( uint32_t pixelY_ )
    {
      pixelY = pixelY_;
      return *this;
    }

    CoarseSampleLocationNV& setSample( uint32_t sample_ )
    {
      sample = sample_;
      return *this;
    }

    operator VkCoarseSampleLocationNV const&() const
    {
      return *reinterpret_cast<const VkCoarseSampleLocationNV*>(this);
    }

    operator VkCoarseSampleLocationNV &()
    {
      return *reinterpret_cast<VkCoarseSampleLocationNV*>(this);
    }

    bool operator==( CoarseSampleLocationNV const& rhs ) const
    {
      return ( pixelX == rhs.pixelX )
          && ( pixelY == rhs.pixelY )
          && ( sample == rhs.sample );
    }

    bool operator!=( CoarseSampleLocationNV const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t pixelX;
    uint32_t pixelY;
    uint32_t sample;
  };
  static_assert( sizeof( CoarseSampleLocationNV ) == sizeof( VkCoarseSampleLocationNV ), "struct and wrapper have different size!" );

  struct DrawMeshTasksIndirectCommandNV
  {
    DrawMeshTasksIndirectCommandNV( uint32_t taskCount_ = 0,
                                    uint32_t firstTask_ = 0 )
      : taskCount( taskCount_ )
      , firstTask( firstTask_ )
    {
    }

    DrawMeshTasksIndirectCommandNV( VkDrawMeshTasksIndirectCommandNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DrawMeshTasksIndirectCommandNV ) );
    }

    DrawMeshTasksIndirectCommandNV& operator=( VkDrawMeshTasksIndirectCommandNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DrawMeshTasksIndirectCommandNV ) );
      return *this;
    }
    DrawMeshTasksIndirectCommandNV& setTaskCount( uint32_t taskCount_ )
    {
      taskCount = taskCount_;
      return *this;
    }

    DrawMeshTasksIndirectCommandNV& setFirstTask( uint32_t firstTask_ )
    {
      firstTask = firstTask_;
      return *this;
    }

    operator VkDrawMeshTasksIndirectCommandNV const&() const
    {
      return *reinterpret_cast<const VkDrawMeshTasksIndirectCommandNV*>(this);
    }

    operator VkDrawMeshTasksIndirectCommandNV &()
    {
      return *reinterpret_cast<VkDrawMeshTasksIndirectCommandNV*>(this);
    }

    bool operator==( DrawMeshTasksIndirectCommandNV const& rhs ) const
    {
      return ( taskCount == rhs.taskCount )
          && ( firstTask == rhs.firstTask );
    }

    bool operator!=( DrawMeshTasksIndirectCommandNV const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t taskCount;
    uint32_t firstTask;
  };
  static_assert( sizeof( DrawMeshTasksIndirectCommandNV ) == sizeof( VkDrawMeshTasksIndirectCommandNV ), "struct and wrapper have different size!" );

  enum class ImageLayout
  {
    eUndefined = VK_IMAGE_LAYOUT_UNDEFINED,
    eGeneral = VK_IMAGE_LAYOUT_GENERAL,
    eColorAttachmentOptimal = VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL,
    eDepthStencilAttachmentOptimal = VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL,
    eDepthStencilReadOnlyOptimal = VK_IMAGE_LAYOUT_DEPTH_STENCIL_READ_ONLY_OPTIMAL,
    eShaderReadOnlyOptimal = VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL,
    eTransferSrcOptimal = VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL,
    eTransferDstOptimal = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL,
    ePreinitialized = VK_IMAGE_LAYOUT_PREINITIALIZED,
    eDepthReadOnlyStencilAttachmentOptimal = VK_IMAGE_LAYOUT_DEPTH_READ_ONLY_STENCIL_ATTACHMENT_OPTIMAL,
    eDepthReadOnlyStencilAttachmentOptimalKHR = VK_IMAGE_LAYOUT_DEPTH_READ_ONLY_STENCIL_ATTACHMENT_OPTIMAL,
    eDepthAttachmentStencilReadOnlyOptimal = VK_IMAGE_LAYOUT_DEPTH_ATTACHMENT_STENCIL_READ_ONLY_OPTIMAL,
    eDepthAttachmentStencilReadOnlyOptimalKHR = VK_IMAGE_LAYOUT_DEPTH_ATTACHMENT_STENCIL_READ_ONLY_OPTIMAL,
    ePresentSrcKHR = VK_IMAGE_LAYOUT_PRESENT_SRC_KHR,
    eSharedPresentKHR = VK_IMAGE_LAYOUT_SHARED_PRESENT_KHR,
    eShadingRateOptimalNV = VK_IMAGE_LAYOUT_SHADING_RATE_OPTIMAL_NV,
    eFragmentDensityMapOptimalEXT = VK_IMAGE_LAYOUT_FRAGMENT_DENSITY_MAP_OPTIMAL_EXT
  };

  struct DescriptorImageInfo
  {
    DescriptorImageInfo( Sampler sampler_ = Sampler(),
                         ImageView imageView_ = ImageView(),
                         ImageLayout imageLayout_ = ImageLayout::eUndefined )
      : sampler( sampler_ )
      , imageView( imageView_ )
      , imageLayout( imageLayout_ )
    {
    }

    DescriptorImageInfo( VkDescriptorImageInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorImageInfo ) );
    }

    DescriptorImageInfo& operator=( VkDescriptorImageInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorImageInfo ) );
      return *this;
    }
    DescriptorImageInfo& setSampler( Sampler sampler_ )
    {
      sampler = sampler_;
      return *this;
    }

    DescriptorImageInfo& setImageView( ImageView imageView_ )
    {
      imageView = imageView_;
      return *this;
    }

    DescriptorImageInfo& setImageLayout( ImageLayout imageLayout_ )
    {
      imageLayout = imageLayout_;
      return *this;
    }

    operator VkDescriptorImageInfo const&() const
    {
      return *reinterpret_cast<const VkDescriptorImageInfo*>(this);
    }

    operator VkDescriptorImageInfo &()
    {
      return *reinterpret_cast<VkDescriptorImageInfo*>(this);
    }

    bool operator==( DescriptorImageInfo const& rhs ) const
    {
      return ( sampler == rhs.sampler )
          && ( imageView == rhs.imageView )
          && ( imageLayout == rhs.imageLayout );
    }

    bool operator!=( DescriptorImageInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

    Sampler sampler;
    ImageView imageView;
    ImageLayout imageLayout;
  };
  static_assert( sizeof( DescriptorImageInfo ) == sizeof( VkDescriptorImageInfo ), "struct and wrapper have different size!" );

  struct AttachmentReference
  {
    AttachmentReference( uint32_t attachment_ = 0,
                         ImageLayout layout_ = ImageLayout::eUndefined )
      : attachment( attachment_ )
      , layout( layout_ )
    {
    }

    AttachmentReference( VkAttachmentReference const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentReference ) );
    }

    AttachmentReference& operator=( VkAttachmentReference const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentReference ) );
      return *this;
    }
    AttachmentReference& setAttachment( uint32_t attachment_ )
    {
      attachment = attachment_;
      return *this;
    }

    AttachmentReference& setLayout( ImageLayout layout_ )
    {
      layout = layout_;
      return *this;
    }

    operator VkAttachmentReference const&() const
    {
      return *reinterpret_cast<const VkAttachmentReference*>(this);
    }

    operator VkAttachmentReference &()
    {
      return *reinterpret_cast<VkAttachmentReference*>(this);
    }

    bool operator==( AttachmentReference const& rhs ) const
    {
      return ( attachment == rhs.attachment )
          && ( layout == rhs.layout );
    }

    bool operator!=( AttachmentReference const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t attachment;
    ImageLayout layout;
  };
  static_assert( sizeof( AttachmentReference ) == sizeof( VkAttachmentReference ), "struct and wrapper have different size!" );

  enum class AttachmentLoadOp
  {
    eLoad = VK_ATTACHMENT_LOAD_OP_LOAD,
    eClear = VK_ATTACHMENT_LOAD_OP_CLEAR,
    eDontCare = VK_ATTACHMENT_LOAD_OP_DONT_CARE
  };

  enum class AttachmentStoreOp
  {
    eStore = VK_ATTACHMENT_STORE_OP_STORE,
    eDontCare = VK_ATTACHMENT_STORE_OP_DONT_CARE
  };

  enum class ImageType
  {
    e1D = VK_IMAGE_TYPE_1D,
    e2D = VK_IMAGE_TYPE_2D,
    e3D = VK_IMAGE_TYPE_3D
  };

  enum class ImageTiling
  {
    eOptimal = VK_IMAGE_TILING_OPTIMAL,
    eLinear = VK_IMAGE_TILING_LINEAR,
    eDrmFormatModifierEXT = VK_IMAGE_TILING_DRM_FORMAT_MODIFIER_EXT
  };

  enum class ImageViewType
  {
    e1D = VK_IMAGE_VIEW_TYPE_1D,
    e2D = VK_IMAGE_VIEW_TYPE_2D,
    e3D = VK_IMAGE_VIEW_TYPE_3D,
    eCube = VK_IMAGE_VIEW_TYPE_CUBE,
    e1DArray = VK_IMAGE_VIEW_TYPE_1D_ARRAY,
    e2DArray = VK_IMAGE_VIEW_TYPE_2D_ARRAY,
    eCubeArray = VK_IMAGE_VIEW_TYPE_CUBE_ARRAY
  };

  enum class CommandBufferLevel
  {
    ePrimary = VK_COMMAND_BUFFER_LEVEL_PRIMARY,
    eSecondary = VK_COMMAND_BUFFER_LEVEL_SECONDARY
  };

  enum class ComponentSwizzle
  {
    eIdentity = VK_COMPONENT_SWIZZLE_IDENTITY,
    eZero = VK_COMPONENT_SWIZZLE_ZERO,
    eOne = VK_COMPONENT_SWIZZLE_ONE,
    eR = VK_COMPONENT_SWIZZLE_R,
    eG = VK_COMPONENT_SWIZZLE_G,
    eB = VK_COMPONENT_SWIZZLE_B,
    eA = VK_COMPONENT_SWIZZLE_A
  };

  struct ComponentMapping
  {
    ComponentMapping( ComponentSwizzle r_ = ComponentSwizzle::eIdentity,
                      ComponentSwizzle g_ = ComponentSwizzle::eIdentity,
                      ComponentSwizzle b_ = ComponentSwizzle::eIdentity,
                      ComponentSwizzle a_ = ComponentSwizzle::eIdentity )
      : r( r_ )
      , g( g_ )
      , b( b_ )
      , a( a_ )
    {
    }

    ComponentMapping( VkComponentMapping const & rhs )
    {
      memcpy( this, &rhs, sizeof( ComponentMapping ) );
    }

    ComponentMapping& operator=( VkComponentMapping const & rhs )
    {
      memcpy( this, &rhs, sizeof( ComponentMapping ) );
      return *this;
    }
    ComponentMapping& setR( ComponentSwizzle r_ )
    {
      r = r_;
      return *this;
    }

    ComponentMapping& setG( ComponentSwizzle g_ )
    {
      g = g_;
      return *this;
    }

    ComponentMapping& setB( ComponentSwizzle b_ )
    {
      b = b_;
      return *this;
    }

    ComponentMapping& setA( ComponentSwizzle a_ )
    {
      a = a_;
      return *this;
    }

    operator VkComponentMapping const&() const
    {
      return *reinterpret_cast<const VkComponentMapping*>(this);
    }

    operator VkComponentMapping &()
    {
      return *reinterpret_cast<VkComponentMapping*>(this);
    }

    bool operator==( ComponentMapping const& rhs ) const
    {
      return ( r == rhs.r )
          && ( g == rhs.g )
          && ( b == rhs.b )
          && ( a == rhs.a );
    }

    bool operator!=( ComponentMapping const& rhs ) const
    {
      return !operator==( rhs );
    }

    ComponentSwizzle r;
    ComponentSwizzle g;
    ComponentSwizzle b;
    ComponentSwizzle a;
  };
  static_assert( sizeof( ComponentMapping ) == sizeof( VkComponentMapping ), "struct and wrapper have different size!" );

  enum class DescriptorType
  {
    eSampler = VK_DESCRIPTOR_TYPE_SAMPLER,
    eCombinedImageSampler = VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER,
    eSampledImage = VK_DESCRIPTOR_TYPE_SAMPLED_IMAGE,
    eStorageImage = VK_DESCRIPTOR_TYPE_STORAGE_IMAGE,
    eUniformTexelBuffer = VK_DESCRIPTOR_TYPE_UNIFORM_TEXEL_BUFFER,
    eStorageTexelBuffer = VK_DESCRIPTOR_TYPE_STORAGE_TEXEL_BUFFER,
    eUniformBuffer = VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER,
    eStorageBuffer = VK_DESCRIPTOR_TYPE_STORAGE_BUFFER,
    eUniformBufferDynamic = VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER_DYNAMIC,
    eStorageBufferDynamic = VK_DESCRIPTOR_TYPE_STORAGE_BUFFER_DYNAMIC,
    eInputAttachment = VK_DESCRIPTOR_TYPE_INPUT_ATTACHMENT,
    eInlineUniformBlockEXT = VK_DESCRIPTOR_TYPE_INLINE_UNIFORM_BLOCK_EXT,
    eAccelerationStructureNV = VK_DESCRIPTOR_TYPE_ACCELERATION_STRUCTURE_NV
  };

  struct DescriptorPoolSize
  {
    DescriptorPoolSize( DescriptorType type_ = DescriptorType::eSampler,
                        uint32_t descriptorCount_ = 0 )
      : type( type_ )
      , descriptorCount( descriptorCount_ )
    {
    }

    DescriptorPoolSize( VkDescriptorPoolSize const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorPoolSize ) );
    }

    DescriptorPoolSize& operator=( VkDescriptorPoolSize const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorPoolSize ) );
      return *this;
    }
    DescriptorPoolSize& setType( DescriptorType type_ )
    {
      type = type_;
      return *this;
    }

    DescriptorPoolSize& setDescriptorCount( uint32_t descriptorCount_ )
    {
      descriptorCount = descriptorCount_;
      return *this;
    }

    operator VkDescriptorPoolSize const&() const
    {
      return *reinterpret_cast<const VkDescriptorPoolSize*>(this);
    }

    operator VkDescriptorPoolSize &()
    {
      return *reinterpret_cast<VkDescriptorPoolSize*>(this);
    }

    bool operator==( DescriptorPoolSize const& rhs ) const
    {
      return ( type == rhs.type )
          && ( descriptorCount == rhs.descriptorCount );
    }

    bool operator!=( DescriptorPoolSize const& rhs ) const
    {
      return !operator==( rhs );
    }

    DescriptorType type;
    uint32_t descriptorCount;
  };
  static_assert( sizeof( DescriptorPoolSize ) == sizeof( VkDescriptorPoolSize ), "struct and wrapper have different size!" );

  struct DescriptorUpdateTemplateEntry
  {
    DescriptorUpdateTemplateEntry( uint32_t dstBinding_ = 0,
                                   uint32_t dstArrayElement_ = 0,
                                   uint32_t descriptorCount_ = 0,
                                   DescriptorType descriptorType_ = DescriptorType::eSampler,
                                   size_t offset_ = 0,
                                   size_t stride_ = 0 )
      : dstBinding( dstBinding_ )
      , dstArrayElement( dstArrayElement_ )
      , descriptorCount( descriptorCount_ )
      , descriptorType( descriptorType_ )
      , offset( offset_ )
      , stride( stride_ )
    {
    }

    DescriptorUpdateTemplateEntry( VkDescriptorUpdateTemplateEntry const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorUpdateTemplateEntry ) );
    }

    DescriptorUpdateTemplateEntry& operator=( VkDescriptorUpdateTemplateEntry const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorUpdateTemplateEntry ) );
      return *this;
    }
    DescriptorUpdateTemplateEntry& setDstBinding( uint32_t dstBinding_ )
    {
      dstBinding = dstBinding_;
      return *this;
    }

    DescriptorUpdateTemplateEntry& setDstArrayElement( uint32_t dstArrayElement_ )
    {
      dstArrayElement = dstArrayElement_;
      return *this;
    }

    DescriptorUpdateTemplateEntry& setDescriptorCount( uint32_t descriptorCount_ )
    {
      descriptorCount = descriptorCount_;
      return *this;
    }

    DescriptorUpdateTemplateEntry& setDescriptorType( DescriptorType descriptorType_ )
    {
      descriptorType = descriptorType_;
      return *this;
    }

    DescriptorUpdateTemplateEntry& setOffset( size_t offset_ )
    {
      offset = offset_;
      return *this;
    }

    DescriptorUpdateTemplateEntry& setStride( size_t stride_ )
    {
      stride = stride_;
      return *this;
    }

    operator VkDescriptorUpdateTemplateEntry const&() const
    {
      return *reinterpret_cast<const VkDescriptorUpdateTemplateEntry*>(this);
    }

    operator VkDescriptorUpdateTemplateEntry &()
    {
      return *reinterpret_cast<VkDescriptorUpdateTemplateEntry*>(this);
    }

    bool operator==( DescriptorUpdateTemplateEntry const& rhs ) const
    {
      return ( dstBinding == rhs.dstBinding )
          && ( dstArrayElement == rhs.dstArrayElement )
          && ( descriptorCount == rhs.descriptorCount )
          && ( descriptorType == rhs.descriptorType )
          && ( offset == rhs.offset )
          && ( stride == rhs.stride );
    }

    bool operator!=( DescriptorUpdateTemplateEntry const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t dstBinding;
    uint32_t dstArrayElement;
    uint32_t descriptorCount;
    DescriptorType descriptorType;
    size_t offset;
    size_t stride;
  };
  static_assert( sizeof( DescriptorUpdateTemplateEntry ) == sizeof( VkDescriptorUpdateTemplateEntry ), "struct and wrapper have different size!" );

  using DescriptorUpdateTemplateEntryKHR = DescriptorUpdateTemplateEntry;

  enum class QueryType
  {
    eOcclusion = VK_QUERY_TYPE_OCCLUSION,
    ePipelineStatistics = VK_QUERY_TYPE_PIPELINE_STATISTICS,
    eTimestamp = VK_QUERY_TYPE_TIMESTAMP,
    eTransformFeedbackStreamEXT = VK_QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT,
    eAccelerationStructureCompactedSizeNV = VK_QUERY_TYPE_ACCELERATION_STRUCTURE_COMPACTED_SIZE_NV
  };

  enum class BorderColor
  {
    eFloatTransparentBlack = VK_BORDER_COLOR_FLOAT_TRANSPARENT_BLACK,
    eIntTransparentBlack = VK_BORDER_COLOR_INT_TRANSPARENT_BLACK,
    eFloatOpaqueBlack = VK_BORDER_COLOR_FLOAT_OPAQUE_BLACK,
    eIntOpaqueBlack = VK_BORDER_COLOR_INT_OPAQUE_BLACK,
    eFloatOpaqueWhite = VK_BORDER_COLOR_FLOAT_OPAQUE_WHITE,
    eIntOpaqueWhite = VK_BORDER_COLOR_INT_OPAQUE_WHITE
  };

  enum class PipelineBindPoint
  {
    eGraphics = VK_PIPELINE_BIND_POINT_GRAPHICS,
    eCompute = VK_PIPELINE_BIND_POINT_COMPUTE,
    eRayTracingNV = VK_PIPELINE_BIND_POINT_RAY_TRACING_NV
  };

  enum class PipelineCacheHeaderVersion
  {
    eOne = VK_PIPELINE_CACHE_HEADER_VERSION_ONE
  };

  enum class PrimitiveTopology
  {
    ePointList = VK_PRIMITIVE_TOPOLOGY_POINT_LIST,
    eLineList = VK_PRIMITIVE_TOPOLOGY_LINE_LIST,
    eLineStrip = VK_PRIMITIVE_TOPOLOGY_LINE_STRIP,
    eTriangleList = VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST,
    eTriangleStrip = VK_PRIMITIVE_TOPOLOGY_TRIANGLE_STRIP,
    eTriangleFan = VK_PRIMITIVE_TOPOLOGY_TRIANGLE_FAN,
    eLineListWithAdjacency = VK_PRIMITIVE_TOPOLOGY_LINE_LIST_WITH_ADJACENCY,
    eLineStripWithAdjacency = VK_PRIMITIVE_TOPOLOGY_LINE_STRIP_WITH_ADJACENCY,
    eTriangleListWithAdjacency = VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST_WITH_ADJACENCY,
    eTriangleStripWithAdjacency = VK_PRIMITIVE_TOPOLOGY_TRIANGLE_STRIP_WITH_ADJACENCY,
    ePatchList = VK_PRIMITIVE_TOPOLOGY_PATCH_LIST
  };

  enum class SharingMode
  {
    eExclusive = VK_SHARING_MODE_EXCLUSIVE,
    eConcurrent = VK_SHARING_MODE_CONCURRENT
  };

  enum class IndexType
  {
    eUint16 = VK_INDEX_TYPE_UINT16,
    eUint32 = VK_INDEX_TYPE_UINT32,
    eNoneNV = VK_INDEX_TYPE_NONE_NV
  };

  enum class Filter
  {
    eNearest = VK_FILTER_NEAREST,
    eLinear = VK_FILTER_LINEAR,
    eCubicIMG = VK_FILTER_CUBIC_IMG
  };

  enum class SamplerMipmapMode
  {
    eNearest = VK_SAMPLER_MIPMAP_MODE_NEAREST,
    eLinear = VK_SAMPLER_MIPMAP_MODE_LINEAR
  };

  enum class SamplerAddressMode
  {
    eRepeat = VK_SAMPLER_ADDRESS_MODE_REPEAT,
    eMirroredRepeat = VK_SAMPLER_ADDRESS_MODE_MIRRORED_REPEAT,
    eClampToEdge = VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE,
    eClampToBorder = VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_BORDER,
    eMirrorClampToEdge = VK_SAMPLER_ADDRESS_MODE_MIRROR_CLAMP_TO_EDGE
  };

  enum class CompareOp
  {
    eNever = VK_COMPARE_OP_NEVER,
    eLess = VK_COMPARE_OP_LESS,
    eEqual = VK_COMPARE_OP_EQUAL,
    eLessOrEqual = VK_COMPARE_OP_LESS_OR_EQUAL,
    eGreater = VK_COMPARE_OP_GREATER,
    eNotEqual = VK_COMPARE_OP_NOT_EQUAL,
    eGreaterOrEqual = VK_COMPARE_OP_GREATER_OR_EQUAL,
    eAlways = VK_COMPARE_OP_ALWAYS
  };

  enum class PolygonMode
  {
    eFill = VK_POLYGON_MODE_FILL,
    eLine = VK_POLYGON_MODE_LINE,
    ePoint = VK_POLYGON_MODE_POINT,
    eFillRectangleNV = VK_POLYGON_MODE_FILL_RECTANGLE_NV
  };

  enum class CullModeFlagBits
  {
    eNone = VK_CULL_MODE_NONE,
    eFront = VK_CULL_MODE_FRONT_BIT,
    eBack = VK_CULL_MODE_BACK_BIT,
    eFrontAndBack = VK_CULL_MODE_FRONT_AND_BACK
  };

  using CullModeFlags = Flags<CullModeFlagBits, VkCullModeFlags>;

  VULKAN_HPP_INLINE CullModeFlags operator|( CullModeFlagBits bit0, CullModeFlagBits bit1 )
  {
    return CullModeFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE CullModeFlags operator~( CullModeFlagBits bits )
  {
    return ~( CullModeFlags( bits ) );
  }

  template <> struct FlagTraits<CullModeFlagBits>
  {
    enum
    {
      allFlags = VkFlags(CullModeFlagBits::eNone) | VkFlags(CullModeFlagBits::eFront) | VkFlags(CullModeFlagBits::eBack) | VkFlags(CullModeFlagBits::eFrontAndBack)
    };
  };

  enum class FrontFace
  {
    eCounterClockwise = VK_FRONT_FACE_COUNTER_CLOCKWISE,
    eClockwise = VK_FRONT_FACE_CLOCKWISE
  };

  enum class BlendFactor
  {
    eZero = VK_BLEND_FACTOR_ZERO,
    eOne = VK_BLEND_FACTOR_ONE,
    eSrcColor = VK_BLEND_FACTOR_SRC_COLOR,
    eOneMinusSrcColor = VK_BLEND_FACTOR_ONE_MINUS_SRC_COLOR,
    eDstColor = VK_BLEND_FACTOR_DST_COLOR,
    eOneMinusDstColor = VK_BLEND_FACTOR_ONE_MINUS_DST_COLOR,
    eSrcAlpha = VK_BLEND_FACTOR_SRC_ALPHA,
    eOneMinusSrcAlpha = VK_BLEND_FACTOR_ONE_MINUS_SRC_ALPHA,
    eDstAlpha = VK_BLEND_FACTOR_DST_ALPHA,
    eOneMinusDstAlpha = VK_BLEND_FACTOR_ONE_MINUS_DST_ALPHA,
    eConstantColor = VK_BLEND_FACTOR_CONSTANT_COLOR,
    eOneMinusConstantColor = VK_BLEND_FACTOR_ONE_MINUS_CONSTANT_COLOR,
    eConstantAlpha = VK_BLEND_FACTOR_CONSTANT_ALPHA,
    eOneMinusConstantAlpha = VK_BLEND_FACTOR_ONE_MINUS_CONSTANT_ALPHA,
    eSrcAlphaSaturate = VK_BLEND_FACTOR_SRC_ALPHA_SATURATE,
    eSrc1Color = VK_BLEND_FACTOR_SRC1_COLOR,
    eOneMinusSrc1Color = VK_BLEND_FACTOR_ONE_MINUS_SRC1_COLOR,
    eSrc1Alpha = VK_BLEND_FACTOR_SRC1_ALPHA,
    eOneMinusSrc1Alpha = VK_BLEND_FACTOR_ONE_MINUS_SRC1_ALPHA
  };

  enum class BlendOp
  {
    eAdd = VK_BLEND_OP_ADD,
    eSubtract = VK_BLEND_OP_SUBTRACT,
    eReverseSubtract = VK_BLEND_OP_REVERSE_SUBTRACT,
    eMin = VK_BLEND_OP_MIN,
    eMax = VK_BLEND_OP_MAX,
    eZeroEXT = VK_BLEND_OP_ZERO_EXT,
    eSrcEXT = VK_BLEND_OP_SRC_EXT,
    eDstEXT = VK_BLEND_OP_DST_EXT,
    eSrcOverEXT = VK_BLEND_OP_SRC_OVER_EXT,
    eDstOverEXT = VK_BLEND_OP_DST_OVER_EXT,
    eSrcInEXT = VK_BLEND_OP_SRC_IN_EXT,
    eDstInEXT = VK_BLEND_OP_DST_IN_EXT,
    eSrcOutEXT = VK_BLEND_OP_SRC_OUT_EXT,
    eDstOutEXT = VK_BLEND_OP_DST_OUT_EXT,
    eSrcAtopEXT = VK_BLEND_OP_SRC_ATOP_EXT,
    eDstAtopEXT = VK_BLEND_OP_DST_ATOP_EXT,
    eXorEXT = VK_BLEND_OP_XOR_EXT,
    eMultiplyEXT = VK_BLEND_OP_MULTIPLY_EXT,
    eScreenEXT = VK_BLEND_OP_SCREEN_EXT,
    eOverlayEXT = VK_BLEND_OP_OVERLAY_EXT,
    eDarkenEXT = VK_BLEND_OP_DARKEN_EXT,
    eLightenEXT = VK_BLEND_OP_LIGHTEN_EXT,
    eColordodgeEXT = VK_BLEND_OP_COLORDODGE_EXT,
    eColorburnEXT = VK_BLEND_OP_COLORBURN_EXT,
    eHardlightEXT = VK_BLEND_OP_HARDLIGHT_EXT,
    eSoftlightEXT = VK_BLEND_OP_SOFTLIGHT_EXT,
    eDifferenceEXT = VK_BLEND_OP_DIFFERENCE_EXT,
    eExclusionEXT = VK_BLEND_OP_EXCLUSION_EXT,
    eInvertEXT = VK_BLEND_OP_INVERT_EXT,
    eInvertRgbEXT = VK_BLEND_OP_INVERT_RGB_EXT,
    eLineardodgeEXT = VK_BLEND_OP_LINEARDODGE_EXT,
    eLinearburnEXT = VK_BLEND_OP_LINEARBURN_EXT,
    eVividlightEXT = VK_BLEND_OP_VIVIDLIGHT_EXT,
    eLinearlightEXT = VK_BLEND_OP_LINEARLIGHT_EXT,
    ePinlightEXT = VK_BLEND_OP_PINLIGHT_EXT,
    eHardmixEXT = VK_BLEND_OP_HARDMIX_EXT,
    eHslHueEXT = VK_BLEND_OP_HSL_HUE_EXT,
    eHslSaturationEXT = VK_BLEND_OP_HSL_SATURATION_EXT,
    eHslColorEXT = VK_BLEND_OP_HSL_COLOR_EXT,
    eHslLuminosityEXT = VK_BLEND_OP_HSL_LUMINOSITY_EXT,
    ePlusEXT = VK_BLEND_OP_PLUS_EXT,
    ePlusClampedEXT = VK_BLEND_OP_PLUS_CLAMPED_EXT,
    ePlusClampedAlphaEXT = VK_BLEND_OP_PLUS_CLAMPED_ALPHA_EXT,
    ePlusDarkerEXT = VK_BLEND_OP_PLUS_DARKER_EXT,
    eMinusEXT = VK_BLEND_OP_MINUS_EXT,
    eMinusClampedEXT = VK_BLEND_OP_MINUS_CLAMPED_EXT,
    eContrastEXT = VK_BLEND_OP_CONTRAST_EXT,
    eInvertOvgEXT = VK_BLEND_OP_INVERT_OVG_EXT,
    eRedEXT = VK_BLEND_OP_RED_EXT,
    eGreenEXT = VK_BLEND_OP_GREEN_EXT,
    eBlueEXT = VK_BLEND_OP_BLUE_EXT
  };

  enum class StencilOp
  {
    eKeep = VK_STENCIL_OP_KEEP,
    eZero = VK_STENCIL_OP_ZERO,
    eReplace = VK_STENCIL_OP_REPLACE,
    eIncrementAndClamp = VK_STENCIL_OP_INCREMENT_AND_CLAMP,
    eDecrementAndClamp = VK_STENCIL_OP_DECREMENT_AND_CLAMP,
    eInvert = VK_STENCIL_OP_INVERT,
    eIncrementAndWrap = VK_STENCIL_OP_INCREMENT_AND_WRAP,
    eDecrementAndWrap = VK_STENCIL_OP_DECREMENT_AND_WRAP
  };

  struct StencilOpState
  {
    StencilOpState( StencilOp failOp_ = StencilOp::eKeep,
                    StencilOp passOp_ = StencilOp::eKeep,
                    StencilOp depthFailOp_ = StencilOp::eKeep,
                    CompareOp compareOp_ = CompareOp::eNever,
                    uint32_t compareMask_ = 0,
                    uint32_t writeMask_ = 0,
                    uint32_t reference_ = 0 )
      : failOp( failOp_ )
      , passOp( passOp_ )
      , depthFailOp( depthFailOp_ )
      , compareOp( compareOp_ )
      , compareMask( compareMask_ )
      , writeMask( writeMask_ )
      , reference( reference_ )
    {
    }

    StencilOpState( VkStencilOpState const & rhs )
    {
      memcpy( this, &rhs, sizeof( StencilOpState ) );
    }

    StencilOpState& operator=( VkStencilOpState const & rhs )
    {
      memcpy( this, &rhs, sizeof( StencilOpState ) );
      return *this;
    }
    StencilOpState& setFailOp( StencilOp failOp_ )
    {
      failOp = failOp_;
      return *this;
    }

    StencilOpState& setPassOp( StencilOp passOp_ )
    {
      passOp = passOp_;
      return *this;
    }

    StencilOpState& setDepthFailOp( StencilOp depthFailOp_ )
    {
      depthFailOp = depthFailOp_;
      return *this;
    }

    StencilOpState& setCompareOp( CompareOp compareOp_ )
    {
      compareOp = compareOp_;
      return *this;
    }

    StencilOpState& setCompareMask( uint32_t compareMask_ )
    {
      compareMask = compareMask_;
      return *this;
    }

    StencilOpState& setWriteMask( uint32_t writeMask_ )
    {
      writeMask = writeMask_;
      return *this;
    }

    StencilOpState& setReference( uint32_t reference_ )
    {
      reference = reference_;
      return *this;
    }

    operator VkStencilOpState const&() const
    {
      return *reinterpret_cast<const VkStencilOpState*>(this);
    }

    operator VkStencilOpState &()
    {
      return *reinterpret_cast<VkStencilOpState*>(this);
    }

    bool operator==( StencilOpState const& rhs ) const
    {
      return ( failOp == rhs.failOp )
          && ( passOp == rhs.passOp )
          && ( depthFailOp == rhs.depthFailOp )
          && ( compareOp == rhs.compareOp )
          && ( compareMask == rhs.compareMask )
          && ( writeMask == rhs.writeMask )
          && ( reference == rhs.reference );
    }

    bool operator!=( StencilOpState const& rhs ) const
    {
      return !operator==( rhs );
    }

    StencilOp failOp;
    StencilOp passOp;
    StencilOp depthFailOp;
    CompareOp compareOp;
    uint32_t compareMask;
    uint32_t writeMask;
    uint32_t reference;
  };
  static_assert( sizeof( StencilOpState ) == sizeof( VkStencilOpState ), "struct and wrapper have different size!" );

  enum class LogicOp
  {
    eClear = VK_LOGIC_OP_CLEAR,
    eAnd = VK_LOGIC_OP_AND,
    eAndReverse = VK_LOGIC_OP_AND_REVERSE,
    eCopy = VK_LOGIC_OP_COPY,
    eAndInverted = VK_LOGIC_OP_AND_INVERTED,
    eNoOp = VK_LOGIC_OP_NO_OP,
    eXor = VK_LOGIC_OP_XOR,
    eOr = VK_LOGIC_OP_OR,
    eNor = VK_LOGIC_OP_NOR,
    eEquivalent = VK_LOGIC_OP_EQUIVALENT,
    eInvert = VK_LOGIC_OP_INVERT,
    eOrReverse = VK_LOGIC_OP_OR_REVERSE,
    eCopyInverted = VK_LOGIC_OP_COPY_INVERTED,
    eOrInverted = VK_LOGIC_OP_OR_INVERTED,
    eNand = VK_LOGIC_OP_NAND,
    eSet = VK_LOGIC_OP_SET
  };

  enum class InternalAllocationType
  {
    eExecutable = VK_INTERNAL_ALLOCATION_TYPE_EXECUTABLE
  };

  enum class SystemAllocationScope
  {
    eCommand = VK_SYSTEM_ALLOCATION_SCOPE_COMMAND,
    eObject = VK_SYSTEM_ALLOCATION_SCOPE_OBJECT,
    eCache = VK_SYSTEM_ALLOCATION_SCOPE_CACHE,
    eDevice = VK_SYSTEM_ALLOCATION_SCOPE_DEVICE,
    eInstance = VK_SYSTEM_ALLOCATION_SCOPE_INSTANCE
  };

  enum class PhysicalDeviceType
  {
    eOther = VK_PHYSICAL_DEVICE_TYPE_OTHER,
    eIntegratedGpu = VK_PHYSICAL_DEVICE_TYPE_INTEGRATED_GPU,
    eDiscreteGpu = VK_PHYSICAL_DEVICE_TYPE_DISCRETE_GPU,
    eVirtualGpu = VK_PHYSICAL_DEVICE_TYPE_VIRTUAL_GPU,
    eCpu = VK_PHYSICAL_DEVICE_TYPE_CPU
  };

  enum class VertexInputRate
  {
    eVertex = VK_VERTEX_INPUT_RATE_VERTEX,
    eInstance = VK_VERTEX_INPUT_RATE_INSTANCE
  };

  struct VertexInputBindingDescription
  {
    VertexInputBindingDescription( uint32_t binding_ = 0,
                                   uint32_t stride_ = 0,
                                   VertexInputRate inputRate_ = VertexInputRate::eVertex )
      : binding( binding_ )
      , stride( stride_ )
      , inputRate( inputRate_ )
    {
    }

    VertexInputBindingDescription( VkVertexInputBindingDescription const & rhs )
    {
      memcpy( this, &rhs, sizeof( VertexInputBindingDescription ) );
    }

    VertexInputBindingDescription& operator=( VkVertexInputBindingDescription const & rhs )
    {
      memcpy( this, &rhs, sizeof( VertexInputBindingDescription ) );
      return *this;
    }
    VertexInputBindingDescription& setBinding( uint32_t binding_ )
    {
      binding = binding_;
      return *this;
    }

    VertexInputBindingDescription& setStride( uint32_t stride_ )
    {
      stride = stride_;
      return *this;
    }

    VertexInputBindingDescription& setInputRate( VertexInputRate inputRate_ )
    {
      inputRate = inputRate_;
      return *this;
    }

    operator VkVertexInputBindingDescription const&() const
    {
      return *reinterpret_cast<const VkVertexInputBindingDescription*>(this);
    }

    operator VkVertexInputBindingDescription &()
    {
      return *reinterpret_cast<VkVertexInputBindingDescription*>(this);
    }

    bool operator==( VertexInputBindingDescription const& rhs ) const
    {
      return ( binding == rhs.binding )
          && ( stride == rhs.stride )
          && ( inputRate == rhs.inputRate );
    }

    bool operator!=( VertexInputBindingDescription const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t binding;
    uint32_t stride;
    VertexInputRate inputRate;
  };
  static_assert( sizeof( VertexInputBindingDescription ) == sizeof( VkVertexInputBindingDescription ), "struct and wrapper have different size!" );

  enum class Format
  {
    eUndefined = VK_FORMAT_UNDEFINED,
    eR4G4UnormPack8 = VK_FORMAT_R4G4_UNORM_PACK8,
    eR4G4B4A4UnormPack16 = VK_FORMAT_R4G4B4A4_UNORM_PACK16,
    eB4G4R4A4UnormPack16 = VK_FORMAT_B4G4R4A4_UNORM_PACK16,
    eR5G6B5UnormPack16 = VK_FORMAT_R5G6B5_UNORM_PACK16,
    eB5G6R5UnormPack16 = VK_FORMAT_B5G6R5_UNORM_PACK16,
    eR5G5B5A1UnormPack16 = VK_FORMAT_R5G5B5A1_UNORM_PACK16,
    eB5G5R5A1UnormPack16 = VK_FORMAT_B5G5R5A1_UNORM_PACK16,
    eA1R5G5B5UnormPack16 = VK_FORMAT_A1R5G5B5_UNORM_PACK16,
    eR8Unorm = VK_FORMAT_R8_UNORM,
    eR8Snorm = VK_FORMAT_R8_SNORM,
    eR8Uscaled = VK_FORMAT_R8_USCALED,
    eR8Sscaled = VK_FORMAT_R8_SSCALED,
    eR8Uint = VK_FORMAT_R8_UINT,
    eR8Sint = VK_FORMAT_R8_SINT,
    eR8Srgb = VK_FORMAT_R8_SRGB,
    eR8G8Unorm = VK_FORMAT_R8G8_UNORM,
    eR8G8Snorm = VK_FORMAT_R8G8_SNORM,
    eR8G8Uscaled = VK_FORMAT_R8G8_USCALED,
    eR8G8Sscaled = VK_FORMAT_R8G8_SSCALED,
    eR8G8Uint = VK_FORMAT_R8G8_UINT,
    eR8G8Sint = VK_FORMAT_R8G8_SINT,
    eR8G8Srgb = VK_FORMAT_R8G8_SRGB,
    eR8G8B8Unorm = VK_FORMAT_R8G8B8_UNORM,
    eR8G8B8Snorm = VK_FORMAT_R8G8B8_SNORM,
    eR8G8B8Uscaled = VK_FORMAT_R8G8B8_USCALED,
    eR8G8B8Sscaled = VK_FORMAT_R8G8B8_SSCALED,
    eR8G8B8Uint = VK_FORMAT_R8G8B8_UINT,
    eR8G8B8Sint = VK_FORMAT_R8G8B8_SINT,
    eR8G8B8Srgb = VK_FORMAT_R8G8B8_SRGB,
    eB8G8R8Unorm = VK_FORMAT_B8G8R8_UNORM,
    eB8G8R8Snorm = VK_FORMAT_B8G8R8_SNORM,
    eB8G8R8Uscaled = VK_FORMAT_B8G8R8_USCALED,
    eB8G8R8Sscaled = VK_FORMAT_B8G8R8_SSCALED,
    eB8G8R8Uint = VK_FORMAT_B8G8R8_UINT,
    eB8G8R8Sint = VK_FORMAT_B8G8R8_SINT,
    eB8G8R8Srgb = VK_FORMAT_B8G8R8_SRGB,
    eR8G8B8A8Unorm = VK_FORMAT_R8G8B8A8_UNORM,
    eR8G8B8A8Snorm = VK_FORMAT_R8G8B8A8_SNORM,
    eR8G8B8A8Uscaled = VK_FORMAT_R8G8B8A8_USCALED,
    eR8G8B8A8Sscaled = VK_FORMAT_R8G8B8A8_SSCALED,
    eR8G8B8A8Uint = VK_FORMAT_R8G8B8A8_UINT,
    eR8G8B8A8Sint = VK_FORMAT_R8G8B8A8_SINT,
    eR8G8B8A8Srgb = VK_FORMAT_R8G8B8A8_SRGB,
    eB8G8R8A8Unorm = VK_FORMAT_B8G8R8A8_UNORM,
    eB8G8R8A8Snorm = VK_FORMAT_B8G8R8A8_SNORM,
    eB8G8R8A8Uscaled = VK_FORMAT_B8G8R8A8_USCALED,
    eB8G8R8A8Sscaled = VK_FORMAT_B8G8R8A8_SSCALED,
    eB8G8R8A8Uint = VK_FORMAT_B8G8R8A8_UINT,
    eB8G8R8A8Sint = VK_FORMAT_B8G8R8A8_SINT,
    eB8G8R8A8Srgb = VK_FORMAT_B8G8R8A8_SRGB,
    eA8B8G8R8UnormPack32 = VK_FORMAT_A8B8G8R8_UNORM_PACK32,
    eA8B8G8R8SnormPack32 = VK_FORMAT_A8B8G8R8_SNORM_PACK32,
    eA8B8G8R8UscaledPack32 = VK_FORMAT_A8B8G8R8_USCALED_PACK32,
    eA8B8G8R8SscaledPack32 = VK_FORMAT_A8B8G8R8_SSCALED_PACK32,
    eA8B8G8R8UintPack32 = VK_FORMAT_A8B8G8R8_UINT_PACK32,
    eA8B8G8R8SintPack32 = VK_FORMAT_A8B8G8R8_SINT_PACK32,
    eA8B8G8R8SrgbPack32 = VK_FORMAT_A8B8G8R8_SRGB_PACK32,
    eA2R10G10B10UnormPack32 = VK_FORMAT_A2R10G10B10_UNORM_PACK32,
    eA2R10G10B10SnormPack32 = VK_FORMAT_A2R10G10B10_SNORM_PACK32,
    eA2R10G10B10UscaledPack32 = VK_FORMAT_A2R10G10B10_USCALED_PACK32,
    eA2R10G10B10SscaledPack32 = VK_FORMAT_A2R10G10B10_SSCALED_PACK32,
    eA2R10G10B10UintPack32 = VK_FORMAT_A2R10G10B10_UINT_PACK32,
    eA2R10G10B10SintPack32 = VK_FORMAT_A2R10G10B10_SINT_PACK32,
    eA2B10G10R10UnormPack32 = VK_FORMAT_A2B10G10R10_UNORM_PACK32,
    eA2B10G10R10SnormPack32 = VK_FORMAT_A2B10G10R10_SNORM_PACK32,
    eA2B10G10R10UscaledPack32 = VK_FORMAT_A2B10G10R10_USCALED_PACK32,
    eA2B10G10R10SscaledPack32 = VK_FORMAT_A2B10G10R10_SSCALED_PACK32,
    eA2B10G10R10UintPack32 = VK_FORMAT_A2B10G10R10_UINT_PACK32,
    eA2B10G10R10SintPack32 = VK_FORMAT_A2B10G10R10_SINT_PACK32,
    eR16Unorm = VK_FORMAT_R16_UNORM,
    eR16Snorm = VK_FORMAT_R16_SNORM,
    eR16Uscaled = VK_FORMAT_R16_USCALED,
    eR16Sscaled = VK_FORMAT_R16_SSCALED,
    eR16Uint = VK_FORMAT_R16_UINT,
    eR16Sint = VK_FORMAT_R16_SINT,
    eR16Sfloat = VK_FORMAT_R16_SFLOAT,
    eR16G16Unorm = VK_FORMAT_R16G16_UNORM,
    eR16G16Snorm = VK_FORMAT_R16G16_SNORM,
    eR16G16Uscaled = VK_FORMAT_R16G16_USCALED,
    eR16G16Sscaled = VK_FORMAT_R16G16_SSCALED,
    eR16G16Uint = VK_FORMAT_R16G16_UINT,
    eR16G16Sint = VK_FORMAT_R16G16_SINT,
    eR16G16Sfloat = VK_FORMAT_R16G16_SFLOAT,
    eR16G16B16Unorm = VK_FORMAT_R16G16B16_UNORM,
    eR16G16B16Snorm = VK_FORMAT_R16G16B16_SNORM,
    eR16G16B16Uscaled = VK_FORMAT_R16G16B16_USCALED,
    eR16G16B16Sscaled = VK_FORMAT_R16G16B16_SSCALED,
    eR16G16B16Uint = VK_FORMAT_R16G16B16_UINT,
    eR16G16B16Sint = VK_FORMAT_R16G16B16_SINT,
    eR16G16B16Sfloat = VK_FORMAT_R16G16B16_SFLOAT,
    eR16G16B16A16Unorm = VK_FORMAT_R16G16B16A16_UNORM,
    eR16G16B16A16Snorm = VK_FORMAT_R16G16B16A16_SNORM,
    eR16G16B16A16Uscaled = VK_FORMAT_R16G16B16A16_USCALED,
    eR16G16B16A16Sscaled = VK_FORMAT_R16G16B16A16_SSCALED,
    eR16G16B16A16Uint = VK_FORMAT_R16G16B16A16_UINT,
    eR16G16B16A16Sint = VK_FORMAT_R16G16B16A16_SINT,
    eR16G16B16A16Sfloat = VK_FORMAT_R16G16B16A16_SFLOAT,
    eR32Uint = VK_FORMAT_R32_UINT,
    eR32Sint = VK_FORMAT_R32_SINT,
    eR32Sfloat = VK_FORMAT_R32_SFLOAT,
    eR32G32Uint = VK_FORMAT_R32G32_UINT,
    eR32G32Sint = VK_FORMAT_R32G32_SINT,
    eR32G32Sfloat = VK_FORMAT_R32G32_SFLOAT,
    eR32G32B32Uint = VK_FORMAT_R32G32B32_UINT,
    eR32G32B32Sint = VK_FORMAT_R32G32B32_SINT,
    eR32G32B32Sfloat = VK_FORMAT_R32G32B32_SFLOAT,
    eR32G32B32A32Uint = VK_FORMAT_R32G32B32A32_UINT,
    eR32G32B32A32Sint = VK_FORMAT_R32G32B32A32_SINT,
    eR32G32B32A32Sfloat = VK_FORMAT_R32G32B32A32_SFLOAT,
    eR64Uint = VK_FORMAT_R64_UINT,
    eR64Sint = VK_FORMAT_R64_SINT,
    eR64Sfloat = VK_FORMAT_R64_SFLOAT,
    eR64G64Uint = VK_FORMAT_R64G64_UINT,
    eR64G64Sint = VK_FORMAT_R64G64_SINT,
    eR64G64Sfloat = VK_FORMAT_R64G64_SFLOAT,
    eR64G64B64Uint = VK_FORMAT_R64G64B64_UINT,
    eR64G64B64Sint = VK_FORMAT_R64G64B64_SINT,
    eR64G64B64Sfloat = VK_FORMAT_R64G64B64_SFLOAT,
    eR64G64B64A64Uint = VK_FORMAT_R64G64B64A64_UINT,
    eR64G64B64A64Sint = VK_FORMAT_R64G64B64A64_SINT,
    eR64G64B64A64Sfloat = VK_FORMAT_R64G64B64A64_SFLOAT,
    eB10G11R11UfloatPack32 = VK_FORMAT_B10G11R11_UFLOAT_PACK32,
    eE5B9G9R9UfloatPack32 = VK_FORMAT_E5B9G9R9_UFLOAT_PACK32,
    eD16Unorm = VK_FORMAT_D16_UNORM,
    eX8D24UnormPack32 = VK_FORMAT_X8_D24_UNORM_PACK32,
    eD32Sfloat = VK_FORMAT_D32_SFLOAT,
    eS8Uint = VK_FORMAT_S8_UINT,
    eD16UnormS8Uint = VK_FORMAT_D16_UNORM_S8_UINT,
    eD24UnormS8Uint = VK_FORMAT_D24_UNORM_S8_UINT,
    eD32SfloatS8Uint = VK_FORMAT_D32_SFLOAT_S8_UINT,
    eBc1RgbUnormBlock = VK_FORMAT_BC1_RGB_UNORM_BLOCK,
    eBc1RgbSrgbBlock = VK_FORMAT_BC1_RGB_SRGB_BLOCK,
    eBc1RgbaUnormBlock = VK_FORMAT_BC1_RGBA_UNORM_BLOCK,
    eBc1RgbaSrgbBlock = VK_FORMAT_BC1_RGBA_SRGB_BLOCK,
    eBc2UnormBlock = VK_FORMAT_BC2_UNORM_BLOCK,
    eBc2SrgbBlock = VK_FORMAT_BC2_SRGB_BLOCK,
    eBc3UnormBlock = VK_FORMAT_BC3_UNORM_BLOCK,
    eBc3SrgbBlock = VK_FORMAT_BC3_SRGB_BLOCK,
    eBc4UnormBlock = VK_FORMAT_BC4_UNORM_BLOCK,
    eBc4SnormBlock = VK_FORMAT_BC4_SNORM_BLOCK,
    eBc5UnormBlock = VK_FORMAT_BC5_UNORM_BLOCK,
    eBc5SnormBlock = VK_FORMAT_BC5_SNORM_BLOCK,
    eBc6HUfloatBlock = VK_FORMAT_BC6H_UFLOAT_BLOCK,
    eBc6HSfloatBlock = VK_FORMAT_BC6H_SFLOAT_BLOCK,
    eBc7UnormBlock = VK_FORMAT_BC7_UNORM_BLOCK,
    eBc7SrgbBlock = VK_FORMAT_BC7_SRGB_BLOCK,
    eEtc2R8G8B8UnormBlock = VK_FORMAT_ETC2_R8G8B8_UNORM_BLOCK,
    eEtc2R8G8B8SrgbBlock = VK_FORMAT_ETC2_R8G8B8_SRGB_BLOCK,
    eEtc2R8G8B8A1UnormBlock = VK_FORMAT_ETC2_R8G8B8A1_UNORM_BLOCK,
    eEtc2R8G8B8A1SrgbBlock = VK_FORMAT_ETC2_R8G8B8A1_SRGB_BLOCK,
    eEtc2R8G8B8A8UnormBlock = VK_FORMAT_ETC2_R8G8B8A8_UNORM_BLOCK,
    eEtc2R8G8B8A8SrgbBlock = VK_FORMAT_ETC2_R8G8B8A8_SRGB_BLOCK,
    eEacR11UnormBlock = VK_FORMAT_EAC_R11_UNORM_BLOCK,
    eEacR11SnormBlock = VK_FORMAT_EAC_R11_SNORM_BLOCK,
    eEacR11G11UnormBlock = VK_FORMAT_EAC_R11G11_UNORM_BLOCK,
    eEacR11G11SnormBlock = VK_FORMAT_EAC_R11G11_SNORM_BLOCK,
    eAstc4x4UnormBlock = VK_FORMAT_ASTC_4x4_UNORM_BLOCK,
    eAstc4x4SrgbBlock = VK_FORMAT_ASTC_4x4_SRGB_BLOCK,
    eAstc5x4UnormBlock = VK_FORMAT_ASTC_5x4_UNORM_BLOCK,
    eAstc5x4SrgbBlock = VK_FORMAT_ASTC_5x4_SRGB_BLOCK,
    eAstc5x5UnormBlock = VK_FORMAT_ASTC_5x5_UNORM_BLOCK,
    eAstc5x5SrgbBlock = VK_FORMAT_ASTC_5x5_SRGB_BLOCK,
    eAstc6x5UnormBlock = VK_FORMAT_ASTC_6x5_UNORM_BLOCK,
    eAstc6x5SrgbBlock = VK_FORMAT_ASTC_6x5_SRGB_BLOCK,
    eAstc6x6UnormBlock = VK_FORMAT_ASTC_6x6_UNORM_BLOCK,
    eAstc6x6SrgbBlock = VK_FORMAT_ASTC_6x6_SRGB_BLOCK,
    eAstc8x5UnormBlock = VK_FORMAT_ASTC_8x5_UNORM_BLOCK,
    eAstc8x5SrgbBlock = VK_FORMAT_ASTC_8x5_SRGB_BLOCK,
    eAstc8x6UnormBlock = VK_FORMAT_ASTC_8x6_UNORM_BLOCK,
    eAstc8x6SrgbBlock = VK_FORMAT_ASTC_8x6_SRGB_BLOCK,
    eAstc8x8UnormBlock = VK_FORMAT_ASTC_8x8_UNORM_BLOCK,
    eAstc8x8SrgbBlock = VK_FORMAT_ASTC_8x8_SRGB_BLOCK,
    eAstc10x5UnormBlock = VK_FORMAT_ASTC_10x5_UNORM_BLOCK,
    eAstc10x5SrgbBlock = VK_FORMAT_ASTC_10x5_SRGB_BLOCK,
    eAstc10x6UnormBlock = VK_FORMAT_ASTC_10x6_UNORM_BLOCK,
    eAstc10x6SrgbBlock = VK_FORMAT_ASTC_10x6_SRGB_BLOCK,
    eAstc10x8UnormBlock = VK_FORMAT_ASTC_10x8_UNORM_BLOCK,
    eAstc10x8SrgbBlock = VK_FORMAT_ASTC_10x8_SRGB_BLOCK,
    eAstc10x10UnormBlock = VK_FORMAT_ASTC_10x10_UNORM_BLOCK,
    eAstc10x10SrgbBlock = VK_FORMAT_ASTC_10x10_SRGB_BLOCK,
    eAstc12x10UnormBlock = VK_FORMAT_ASTC_12x10_UNORM_BLOCK,
    eAstc12x10SrgbBlock = VK_FORMAT_ASTC_12x10_SRGB_BLOCK,
    eAstc12x12UnormBlock = VK_FORMAT_ASTC_12x12_UNORM_BLOCK,
    eAstc12x12SrgbBlock = VK_FORMAT_ASTC_12x12_SRGB_BLOCK,
    eG8B8G8R8422Unorm = VK_FORMAT_G8B8G8R8_422_UNORM,
    eG8B8G8R8422UnormKHR = VK_FORMAT_G8B8G8R8_422_UNORM,
    eB8G8R8G8422Unorm = VK_FORMAT_B8G8R8G8_422_UNORM,
    eB8G8R8G8422UnormKHR = VK_FORMAT_B8G8R8G8_422_UNORM,
    eG8B8R83Plane420Unorm = VK_FORMAT_G8_B8_R8_3PLANE_420_UNORM,
    eG8B8R83Plane420UnormKHR = VK_FORMAT_G8_B8_R8_3PLANE_420_UNORM,
    eG8B8R82Plane420Unorm = VK_FORMAT_G8_B8R8_2PLANE_420_UNORM,
    eG8B8R82Plane420UnormKHR = VK_FORMAT_G8_B8R8_2PLANE_420_UNORM,
    eG8B8R83Plane422Unorm = VK_FORMAT_G8_B8_R8_3PLANE_422_UNORM,
    eG8B8R83Plane422UnormKHR = VK_FORMAT_G8_B8_R8_3PLANE_422_UNORM,
    eG8B8R82Plane422Unorm = VK_FORMAT_G8_B8R8_2PLANE_422_UNORM,
    eG8B8R82Plane422UnormKHR = VK_FORMAT_G8_B8R8_2PLANE_422_UNORM,
    eG8B8R83Plane444Unorm = VK_FORMAT_G8_B8_R8_3PLANE_444_UNORM,
    eG8B8R83Plane444UnormKHR = VK_FORMAT_G8_B8_R8_3PLANE_444_UNORM,
    eR10X6UnormPack16 = VK_FORMAT_R10X6_UNORM_PACK16,
    eR10X6UnormPack16KHR = VK_FORMAT_R10X6_UNORM_PACK16,
    eR10X6G10X6Unorm2Pack16 = VK_FORMAT_R10X6G10X6_UNORM_2PACK16,
    eR10X6G10X6Unorm2Pack16KHR = VK_FORMAT_R10X6G10X6_UNORM_2PACK16,
    eR10X6G10X6B10X6A10X6Unorm4Pack16 = VK_FORMAT_R10X6G10X6B10X6A10X6_UNORM_4PACK16,
    eR10X6G10X6B10X6A10X6Unorm4Pack16KHR = VK_FORMAT_R10X6G10X6B10X6A10X6_UNORM_4PACK16,
    eG10X6B10X6G10X6R10X6422Unorm4Pack16 = VK_FORMAT_G10X6B10X6G10X6R10X6_422_UNORM_4PACK16,
    eG10X6B10X6G10X6R10X6422Unorm4Pack16KHR = VK_FORMAT_G10X6B10X6G10X6R10X6_422_UNORM_4PACK16,
    eB10X6G10X6R10X6G10X6422Unorm4Pack16 = VK_FORMAT_B10X6G10X6R10X6G10X6_422_UNORM_4PACK16,
    eB10X6G10X6R10X6G10X6422Unorm4Pack16KHR = VK_FORMAT_B10X6G10X6R10X6G10X6_422_UNORM_4PACK16,
    eG10X6B10X6R10X63Plane420Unorm3Pack16 = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_420_UNORM_3PACK16,
    eG10X6B10X6R10X63Plane420Unorm3Pack16KHR = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_420_UNORM_3PACK16,
    eG10X6B10X6R10X62Plane420Unorm3Pack16 = VK_FORMAT_G10X6_B10X6R10X6_2PLANE_420_UNORM_3PACK16,
    eG10X6B10X6R10X62Plane420Unorm3Pack16KHR = VK_FORMAT_G10X6_B10X6R10X6_2PLANE_420_UNORM_3PACK16,
    eG10X6B10X6R10X63Plane422Unorm3Pack16 = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_422_UNORM_3PACK16,
    eG10X6B10X6R10X63Plane422Unorm3Pack16KHR = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_422_UNORM_3PACK16,
    eG10X6B10X6R10X62Plane422Unorm3Pack16 = VK_FORMAT_G10X6_B10X6R10X6_2PLANE_422_UNORM_3PACK16,
    eG10X6B10X6R10X62Plane422Unorm3Pack16KHR = VK_FORMAT_G10X6_B10X6R10X6_2PLANE_422_UNORM_3PACK16,
    eG10X6B10X6R10X63Plane444Unorm3Pack16 = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_444_UNORM_3PACK16,
    eG10X6B10X6R10X63Plane444Unorm3Pack16KHR = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_444_UNORM_3PACK16,
    eR12X4UnormPack16 = VK_FORMAT_R12X4_UNORM_PACK16,
    eR12X4UnormPack16KHR = VK_FORMAT_R12X4_UNORM_PACK16,
    eR12X4G12X4Unorm2Pack16 = VK_FORMAT_R12X4G12X4_UNORM_2PACK16,
    eR12X4G12X4Unorm2Pack16KHR = VK_FORMAT_R12X4G12X4_UNORM_2PACK16,
    eR12X4G12X4B12X4A12X4Unorm4Pack16 = VK_FORMAT_R12X4G12X4B12X4A12X4_UNORM_4PACK16,
    eR12X4G12X4B12X4A12X4Unorm4Pack16KHR = VK_FORMAT_R12X4G12X4B12X4A12X4_UNORM_4PACK16,
    eG12X4B12X4G12X4R12X4422Unorm4Pack16 = VK_FORMAT_G12X4B12X4G12X4R12X4_422_UNORM_4PACK16,
    eG12X4B12X4G12X4R12X4422Unorm4Pack16KHR = VK_FORMAT_G12X4B12X4G12X4R12X4_422_UNORM_4PACK16,
    eB12X4G12X4R12X4G12X4422Unorm4Pack16 = VK_FORMAT_B12X4G12X4R12X4G12X4_422_UNORM_4PACK16,
    eB12X4G12X4R12X4G12X4422Unorm4Pack16KHR = VK_FORMAT_B12X4G12X4R12X4G12X4_422_UNORM_4PACK16,
    eG12X4B12X4R12X43Plane420Unorm3Pack16 = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_420_UNORM_3PACK16,
    eG12X4B12X4R12X43Plane420Unorm3Pack16KHR = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_420_UNORM_3PACK16,
    eG12X4B12X4R12X42Plane420Unorm3Pack16 = VK_FORMAT_G12X4_B12X4R12X4_2PLANE_420_UNORM_3PACK16,
    eG12X4B12X4R12X42Plane420Unorm3Pack16KHR = VK_FORMAT_G12X4_B12X4R12X4_2PLANE_420_UNORM_3PACK16,
    eG12X4B12X4R12X43Plane422Unorm3Pack16 = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_422_UNORM_3PACK16,
    eG12X4B12X4R12X43Plane422Unorm3Pack16KHR = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_422_UNORM_3PACK16,
    eG12X4B12X4R12X42Plane422Unorm3Pack16 = VK_FORMAT_G12X4_B12X4R12X4_2PLANE_422_UNORM_3PACK16,
    eG12X4B12X4R12X42Plane422Unorm3Pack16KHR = VK_FORMAT_G12X4_B12X4R12X4_2PLANE_422_UNORM_3PACK16,
    eG12X4B12X4R12X43Plane444Unorm3Pack16 = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_444_UNORM_3PACK16,
    eG12X4B12X4R12X43Plane444Unorm3Pack16KHR = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_444_UNORM_3PACK16,
    eG16B16G16R16422Unorm = VK_FORMAT_G16B16G16R16_422_UNORM,
    eG16B16G16R16422UnormKHR = VK_FORMAT_G16B16G16R16_422_UNORM,
    eB16G16R16G16422Unorm = VK_FORMAT_B16G16R16G16_422_UNORM,
    eB16G16R16G16422UnormKHR = VK_FORMAT_B16G16R16G16_422_UNORM,
    eG16B16R163Plane420Unorm = VK_FORMAT_G16_B16_R16_3PLANE_420_UNORM,
    eG16B16R163Plane420UnormKHR = VK_FORMAT_G16_B16_R16_3PLANE_420_UNORM,
    eG16B16R162Plane420Unorm = VK_FORMAT_G16_B16R16_2PLANE_420_UNORM,
    eG16B16R162Plane420UnormKHR = VK_FORMAT_G16_B16R16_2PLANE_420_UNORM,
    eG16B16R163Plane422Unorm = VK_FORMAT_G16_B16_R16_3PLANE_422_UNORM,
    eG16B16R163Plane422UnormKHR = VK_FORMAT_G16_B16_R16_3PLANE_422_UNORM,
    eG16B16R162Plane422Unorm = VK_FORMAT_G16_B16R16_2PLANE_422_UNORM,
    eG16B16R162Plane422UnormKHR = VK_FORMAT_G16_B16R16_2PLANE_422_UNORM,
    eG16B16R163Plane444Unorm = VK_FORMAT_G16_B16_R16_3PLANE_444_UNORM,
    eG16B16R163Plane444UnormKHR = VK_FORMAT_G16_B16_R16_3PLANE_444_UNORM,
    ePvrtc12BppUnormBlockIMG = VK_FORMAT_PVRTC1_2BPP_UNORM_BLOCK_IMG,
    ePvrtc14BppUnormBlockIMG = VK_FORMAT_PVRTC1_4BPP_UNORM_BLOCK_IMG,
    ePvrtc22BppUnormBlockIMG = VK_FORMAT_PVRTC2_2BPP_UNORM_BLOCK_IMG,
    ePvrtc24BppUnormBlockIMG = VK_FORMAT_PVRTC2_4BPP_UNORM_BLOCK_IMG,
    ePvrtc12BppSrgbBlockIMG = VK_FORMAT_PVRTC1_2BPP_SRGB_BLOCK_IMG,
    ePvrtc14BppSrgbBlockIMG = VK_FORMAT_PVRTC1_4BPP_SRGB_BLOCK_IMG,
    ePvrtc22BppSrgbBlockIMG = VK_FORMAT_PVRTC2_2BPP_SRGB_BLOCK_IMG,
    ePvrtc24BppSrgbBlockIMG = VK_FORMAT_PVRTC2_4BPP_SRGB_BLOCK_IMG
  };

  struct VertexInputAttributeDescription
  {
    VertexInputAttributeDescription( uint32_t location_ = 0,
                                     uint32_t binding_ = 0,
                                     Format format_ = Format::eUndefined,
                                     uint32_t offset_ = 0 )
      : location( location_ )
      , binding( binding_ )
      , format( format_ )
      , offset( offset_ )
    {
    }

    VertexInputAttributeDescription( VkVertexInputAttributeDescription const & rhs )
    {
      memcpy( this, &rhs, sizeof( VertexInputAttributeDescription ) );
    }

    VertexInputAttributeDescription& operator=( VkVertexInputAttributeDescription const & rhs )
    {
      memcpy( this, &rhs, sizeof( VertexInputAttributeDescription ) );
      return *this;
    }
    VertexInputAttributeDescription& setLocation( uint32_t location_ )
    {
      location = location_;
      return *this;
    }

    VertexInputAttributeDescription& setBinding( uint32_t binding_ )
    {
      binding = binding_;
      return *this;
    }

    VertexInputAttributeDescription& setFormat( Format format_ )
    {
      format = format_;
      return *this;
    }

    VertexInputAttributeDescription& setOffset( uint32_t offset_ )
    {
      offset = offset_;
      return *this;
    }

    operator VkVertexInputAttributeDescription const&() const
    {
      return *reinterpret_cast<const VkVertexInputAttributeDescription*>(this);
    }

    operator VkVertexInputAttributeDescription &()
    {
      return *reinterpret_cast<VkVertexInputAttributeDescription*>(this);
    }

    bool operator==( VertexInputAttributeDescription const& rhs ) const
    {
      return ( location == rhs.location )
          && ( binding == rhs.binding )
          && ( format == rhs.format )
          && ( offset == rhs.offset );
    }

    bool operator!=( VertexInputAttributeDescription const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t location;
    uint32_t binding;
    Format format;
    uint32_t offset;
  };
  static_assert( sizeof( VertexInputAttributeDescription ) == sizeof( VkVertexInputAttributeDescription ), "struct and wrapper have different size!" );

  enum class StructureType
  {
    eApplicationInfo = VK_STRUCTURE_TYPE_APPLICATION_INFO,
    eInstanceCreateInfo = VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO,
    eDeviceQueueCreateInfo = VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO,
    eDeviceCreateInfo = VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO,
    eSubmitInfo = VK_STRUCTURE_TYPE_SUBMIT_INFO,
    eMemoryAllocateInfo = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO,
    eMappedMemoryRange = VK_STRUCTURE_TYPE_MAPPED_MEMORY_RANGE,
    eBindSparseInfo = VK_STRUCTURE_TYPE_BIND_SPARSE_INFO,
    eFenceCreateInfo = VK_STRUCTURE_TYPE_FENCE_CREATE_INFO,
    eSemaphoreCreateInfo = VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO,
    eEventCreateInfo = VK_STRUCTURE_TYPE_EVENT_CREATE_INFO,
    eQueryPoolCreateInfo = VK_STRUCTURE_TYPE_QUERY_POOL_CREATE_INFO,
    eBufferCreateInfo = VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO,
    eBufferViewCreateInfo = VK_STRUCTURE_TYPE_BUFFER_VIEW_CREATE_INFO,
    eImageCreateInfo = VK_STRUCTURE_TYPE_IMAGE_CREATE_INFO,
    eImageViewCreateInfo = VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO,
    eShaderModuleCreateInfo = VK_STRUCTURE_TYPE_SHADER_MODULE_CREATE_INFO,
    ePipelineCacheCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_CACHE_CREATE_INFO,
    ePipelineShaderStageCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_CREATE_INFO,
    ePipelineVertexInputStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_STATE_CREATE_INFO,
    ePipelineInputAssemblyStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_INPUT_ASSEMBLY_STATE_CREATE_INFO,
    ePipelineTessellationStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_STATE_CREATE_INFO,
    ePipelineViewportStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_STATE_CREATE_INFO,
    ePipelineRasterizationStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_CREATE_INFO,
    ePipelineMultisampleStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_MULTISAMPLE_STATE_CREATE_INFO,
    ePipelineDepthStencilStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_DEPTH_STENCIL_STATE_CREATE_INFO,
    ePipelineColorBlendStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_STATE_CREATE_INFO,
    ePipelineDynamicStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_DYNAMIC_STATE_CREATE_INFO,
    eGraphicsPipelineCreateInfo = VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_CREATE_INFO,
    eComputePipelineCreateInfo = VK_STRUCTURE_TYPE_COMPUTE_PIPELINE_CREATE_INFO,
    ePipelineLayoutCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_LAYOUT_CREATE_INFO,
    eSamplerCreateInfo = VK_STRUCTURE_TYPE_SAMPLER_CREATE_INFO,
    eDescriptorSetLayoutCreateInfo = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_CREATE_INFO,
    eDescriptorPoolCreateInfo = VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_CREATE_INFO,
    eDescriptorSetAllocateInfo = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_ALLOCATE_INFO,
    eWriteDescriptorSet = VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET,
    eCopyDescriptorSet = VK_STRUCTURE_TYPE_COPY_DESCRIPTOR_SET,
    eFramebufferCreateInfo = VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO,
    eRenderPassCreateInfo = VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO,
    eCommandPoolCreateInfo = VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO,
    eCommandBufferAllocateInfo = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO,
    eCommandBufferInheritanceInfo = VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_INFO,
    eCommandBufferBeginInfo = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO,
    eRenderPassBeginInfo = VK_STRUCTURE_TYPE_RENDER_PASS_BEGIN_INFO,
    eBufferMemoryBarrier = VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER,
    eImageMemoryBarrier = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER,
    eMemoryBarrier = VK_STRUCTURE_TYPE_MEMORY_BARRIER,
    eLoaderInstanceCreateInfo = VK_STRUCTURE_TYPE_LOADER_INSTANCE_CREATE_INFO,
    eLoaderDeviceCreateInfo = VK_STRUCTURE_TYPE_LOADER_DEVICE_CREATE_INFO,
    ePhysicalDeviceSubgroupProperties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_PROPERTIES,
    eBindBufferMemoryInfo = VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_INFO,
    eBindBufferMemoryInfoKHR = VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_INFO,
    eBindImageMemoryInfo = VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_INFO,
    eBindImageMemoryInfoKHR = VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_INFO,
    ePhysicalDevice16BitStorageFeatures = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_16BIT_STORAGE_FEATURES,
    ePhysicalDevice16BitStorageFeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_16BIT_STORAGE_FEATURES,
    eMemoryDedicatedRequirements = VK_STRUCTURE_TYPE_MEMORY_DEDICATED_REQUIREMENTS,
    eMemoryDedicatedRequirementsKHR = VK_STRUCTURE_TYPE_MEMORY_DEDICATED_REQUIREMENTS,
    eMemoryDedicatedAllocateInfo = VK_STRUCTURE_TYPE_MEMORY_DEDICATED_ALLOCATE_INFO,
    eMemoryDedicatedAllocateInfoKHR = VK_STRUCTURE_TYPE_MEMORY_DEDICATED_ALLOCATE_INFO,
    eMemoryAllocateFlagsInfo = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_FLAGS_INFO,
    eMemoryAllocateFlagsInfoKHR = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_FLAGS_INFO,
    eDeviceGroupRenderPassBeginInfo = VK_STRUCTURE_TYPE_DEVICE_GROUP_RENDER_PASS_BEGIN_INFO,
    eDeviceGroupRenderPassBeginInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_RENDER_PASS_BEGIN_INFO,
    eDeviceGroupCommandBufferBeginInfo = VK_STRUCTURE_TYPE_DEVICE_GROUP_COMMAND_BUFFER_BEGIN_INFO,
    eDeviceGroupCommandBufferBeginInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_COMMAND_BUFFER_BEGIN_INFO,
    eDeviceGroupSubmitInfo = VK_STRUCTURE_TYPE_DEVICE_GROUP_SUBMIT_INFO,
    eDeviceGroupSubmitInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_SUBMIT_INFO,
    eDeviceGroupBindSparseInfo = VK_STRUCTURE_TYPE_DEVICE_GROUP_BIND_SPARSE_INFO,
    eDeviceGroupBindSparseInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_BIND_SPARSE_INFO,
    eBindBufferMemoryDeviceGroupInfo = VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_DEVICE_GROUP_INFO,
    eBindBufferMemoryDeviceGroupInfoKHR = VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_DEVICE_GROUP_INFO,
    eBindImageMemoryDeviceGroupInfo = VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_DEVICE_GROUP_INFO,
    eBindImageMemoryDeviceGroupInfoKHR = VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_DEVICE_GROUP_INFO,
    ePhysicalDeviceGroupProperties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GROUP_PROPERTIES,
    ePhysicalDeviceGroupPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GROUP_PROPERTIES,
    eDeviceGroupDeviceCreateInfo = VK_STRUCTURE_TYPE_DEVICE_GROUP_DEVICE_CREATE_INFO,
    eDeviceGroupDeviceCreateInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_DEVICE_CREATE_INFO,
    eBufferMemoryRequirementsInfo2 = VK_STRUCTURE_TYPE_BUFFER_MEMORY_REQUIREMENTS_INFO_2,
    eBufferMemoryRequirementsInfo2KHR = VK_STRUCTURE_TYPE_BUFFER_MEMORY_REQUIREMENTS_INFO_2,
    eImageMemoryRequirementsInfo2 = VK_STRUCTURE_TYPE_IMAGE_MEMORY_REQUIREMENTS_INFO_2,
    eImageMemoryRequirementsInfo2KHR = VK_STRUCTURE_TYPE_IMAGE_MEMORY_REQUIREMENTS_INFO_2,
    eImageSparseMemoryRequirementsInfo2 = VK_STRUCTURE_TYPE_IMAGE_SPARSE_MEMORY_REQUIREMENTS_INFO_2,
    eImageSparseMemoryRequirementsInfo2KHR = VK_STRUCTURE_TYPE_IMAGE_SPARSE_MEMORY_REQUIREMENTS_INFO_2,
    eMemoryRequirements2 = VK_STRUCTURE_TYPE_MEMORY_REQUIREMENTS_2,
    eMemoryRequirements2KHR = VK_STRUCTURE_TYPE_MEMORY_REQUIREMENTS_2,
    eSparseImageMemoryRequirements2 = VK_STRUCTURE_TYPE_SPARSE_IMAGE_MEMORY_REQUIREMENTS_2,
    eSparseImageMemoryRequirements2KHR = VK_STRUCTURE_TYPE_SPARSE_IMAGE_MEMORY_REQUIREMENTS_2,
    ePhysicalDeviceFeatures2 = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FEATURES_2,
    ePhysicalDeviceFeatures2KHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FEATURES_2,
    ePhysicalDeviceProperties2 = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROPERTIES_2,
    ePhysicalDeviceProperties2KHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROPERTIES_2,
    eFormatProperties2 = VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_2,
    eFormatProperties2KHR = VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_2,
    eImageFormatProperties2 = VK_STRUCTURE_TYPE_IMAGE_FORMAT_PROPERTIES_2,
    eImageFormatProperties2KHR = VK_STRUCTURE_TYPE_IMAGE_FORMAT_PROPERTIES_2,
    ePhysicalDeviceImageFormatInfo2 = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_FORMAT_INFO_2,
    ePhysicalDeviceImageFormatInfo2KHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_FORMAT_INFO_2,
    eQueueFamilyProperties2 = VK_STRUCTURE_TYPE_QUEUE_FAMILY_PROPERTIES_2,
    eQueueFamilyProperties2KHR = VK_STRUCTURE_TYPE_QUEUE_FAMILY_PROPERTIES_2,
    ePhysicalDeviceMemoryProperties2 = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PROPERTIES_2,
    ePhysicalDeviceMemoryProperties2KHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PROPERTIES_2,
    eSparseImageFormatProperties2 = VK_STRUCTURE_TYPE_SPARSE_IMAGE_FORMAT_PROPERTIES_2,
    eSparseImageFormatProperties2KHR = VK_STRUCTURE_TYPE_SPARSE_IMAGE_FORMAT_PROPERTIES_2,
    ePhysicalDeviceSparseImageFormatInfo2 = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SPARSE_IMAGE_FORMAT_INFO_2,
    ePhysicalDeviceSparseImageFormatInfo2KHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SPARSE_IMAGE_FORMAT_INFO_2,
    ePhysicalDevicePointClippingProperties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_POINT_CLIPPING_PROPERTIES,
    ePhysicalDevicePointClippingPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_POINT_CLIPPING_PROPERTIES,
    eRenderPassInputAttachmentAspectCreateInfo = VK_STRUCTURE_TYPE_RENDER_PASS_INPUT_ATTACHMENT_ASPECT_CREATE_INFO,
    eRenderPassInputAttachmentAspectCreateInfoKHR = VK_STRUCTURE_TYPE_RENDER_PASS_INPUT_ATTACHMENT_ASPECT_CREATE_INFO,
    eImageViewUsageCreateInfo = VK_STRUCTURE_TYPE_IMAGE_VIEW_USAGE_CREATE_INFO,
    eImageViewUsageCreateInfoKHR = VK_STRUCTURE_TYPE_IMAGE_VIEW_USAGE_CREATE_INFO,
    ePipelineTessellationDomainOriginStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_DOMAIN_ORIGIN_STATE_CREATE_INFO,
    ePipelineTessellationDomainOriginStateCreateInfoKHR = VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_DOMAIN_ORIGIN_STATE_CREATE_INFO,
    eRenderPassMultiviewCreateInfo = VK_STRUCTURE_TYPE_RENDER_PASS_MULTIVIEW_CREATE_INFO,
    eRenderPassMultiviewCreateInfoKHR = VK_STRUCTURE_TYPE_RENDER_PASS_MULTIVIEW_CREATE_INFO,
    ePhysicalDeviceMultiviewFeatures = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_FEATURES,
    ePhysicalDeviceMultiviewFeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_FEATURES,
    ePhysicalDeviceMultiviewProperties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PROPERTIES,
    ePhysicalDeviceMultiviewPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PROPERTIES,
    ePhysicalDeviceVariablePointerFeatures = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VARIABLE_POINTER_FEATURES,
    ePhysicalDeviceVariablePointerFeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VARIABLE_POINTER_FEATURES,
    eProtectedSubmitInfo = VK_STRUCTURE_TYPE_PROTECTED_SUBMIT_INFO,
    ePhysicalDeviceProtectedMemoryFeatures = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_FEATURES,
    ePhysicalDeviceProtectedMemoryProperties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_PROPERTIES,
    eDeviceQueueInfo2 = VK_STRUCTURE_TYPE_DEVICE_QUEUE_INFO_2,
    eSamplerYcbcrConversionCreateInfo = VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_CREATE_INFO,
    eSamplerYcbcrConversionCreateInfoKHR = VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_CREATE_INFO,
    eSamplerYcbcrConversionInfo = VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_INFO,
    eSamplerYcbcrConversionInfoKHR = VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_INFO,
    eBindImagePlaneMemoryInfo = VK_STRUCTURE_TYPE_BIND_IMAGE_PLANE_MEMORY_INFO,
    eBindImagePlaneMemoryInfoKHR = VK_STRUCTURE_TYPE_BIND_IMAGE_PLANE_MEMORY_INFO,
    eImagePlaneMemoryRequirementsInfo = VK_STRUCTURE_TYPE_IMAGE_PLANE_MEMORY_REQUIREMENTS_INFO,
    eImagePlaneMemoryRequirementsInfoKHR = VK_STRUCTURE_TYPE_IMAGE_PLANE_MEMORY_REQUIREMENTS_INFO,
    ePhysicalDeviceSamplerYcbcrConversionFeatures = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_YCBCR_CONVERSION_FEATURES,
    ePhysicalDeviceSamplerYcbcrConversionFeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_YCBCR_CONVERSION_FEATURES,
    eSamplerYcbcrConversionImageFormatProperties = VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_IMAGE_FORMAT_PROPERTIES,
    eSamplerYcbcrConversionImageFormatPropertiesKHR = VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_IMAGE_FORMAT_PROPERTIES,
    eDescriptorUpdateTemplateCreateInfo = VK_STRUCTURE_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_CREATE_INFO,
    eDescriptorUpdateTemplateCreateInfoKHR = VK_STRUCTURE_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_CREATE_INFO,
    ePhysicalDeviceExternalImageFormatInfo = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_IMAGE_FORMAT_INFO,
    ePhysicalDeviceExternalImageFormatInfoKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_IMAGE_FORMAT_INFO,
    eExternalImageFormatProperties = VK_STRUCTURE_TYPE_EXTERNAL_IMAGE_FORMAT_PROPERTIES,
    eExternalImageFormatPropertiesKHR = VK_STRUCTURE_TYPE_EXTERNAL_IMAGE_FORMAT_PROPERTIES,
    ePhysicalDeviceExternalBufferInfo = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_BUFFER_INFO,
    ePhysicalDeviceExternalBufferInfoKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_BUFFER_INFO,
    eExternalBufferProperties = VK_STRUCTURE_TYPE_EXTERNAL_BUFFER_PROPERTIES,
    eExternalBufferPropertiesKHR = VK_STRUCTURE_TYPE_EXTERNAL_BUFFER_PROPERTIES,
    ePhysicalDeviceIdProperties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ID_PROPERTIES,
    ePhysicalDeviceIdPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ID_PROPERTIES,
    eExternalMemoryBufferCreateInfo = VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_BUFFER_CREATE_INFO,
    eExternalMemoryBufferCreateInfoKHR = VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_BUFFER_CREATE_INFO,
    eExternalMemoryImageCreateInfo = VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO,
    eExternalMemoryImageCreateInfoKHR = VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO,
    eExportMemoryAllocateInfo = VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO,
    eExportMemoryAllocateInfoKHR = VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO,
    ePhysicalDeviceExternalFenceInfo = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FENCE_INFO,
    ePhysicalDeviceExternalFenceInfoKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FENCE_INFO,
    eExternalFenceProperties = VK_STRUCTURE_TYPE_EXTERNAL_FENCE_PROPERTIES,
    eExternalFencePropertiesKHR = VK_STRUCTURE_TYPE_EXTERNAL_FENCE_PROPERTIES,
    eExportFenceCreateInfo = VK_STRUCTURE_TYPE_EXPORT_FENCE_CREATE_INFO,
    eExportFenceCreateInfoKHR = VK_STRUCTURE_TYPE_EXPORT_FENCE_CREATE_INFO,
    eExportSemaphoreCreateInfo = VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_CREATE_INFO,
    eExportSemaphoreCreateInfoKHR = VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_CREATE_INFO,
    ePhysicalDeviceExternalSemaphoreInfo = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SEMAPHORE_INFO,
    ePhysicalDeviceExternalSemaphoreInfoKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SEMAPHORE_INFO,
    eExternalSemaphoreProperties = VK_STRUCTURE_TYPE_EXTERNAL_SEMAPHORE_PROPERTIES,
    eExternalSemaphorePropertiesKHR = VK_STRUCTURE_TYPE_EXTERNAL_SEMAPHORE_PROPERTIES,
    ePhysicalDeviceMaintenance3Properties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_3_PROPERTIES,
    ePhysicalDeviceMaintenance3PropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_3_PROPERTIES,
    eDescriptorSetLayoutSupport = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_SUPPORT,
    eDescriptorSetLayoutSupportKHR = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_SUPPORT,
    ePhysicalDeviceShaderDrawParameterFeatures = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_DRAW_PARAMETER_FEATURES,
    eSwapchainCreateInfoKHR = VK_STRUCTURE_TYPE_SWAPCHAIN_CREATE_INFO_KHR,
    ePresentInfoKHR = VK_STRUCTURE_TYPE_PRESENT_INFO_KHR,
    eDeviceGroupPresentCapabilitiesKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_CAPABILITIES_KHR,
    eImageSwapchainCreateInfoKHR = VK_STRUCTURE_TYPE_IMAGE_SWAPCHAIN_CREATE_INFO_KHR,
    eBindImageMemorySwapchainInfoKHR = VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_SWAPCHAIN_INFO_KHR,
    eAcquireNextImageInfoKHR = VK_STRUCTURE_TYPE_ACQUIRE_NEXT_IMAGE_INFO_KHR,
    eDeviceGroupPresentInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_INFO_KHR,
    eDeviceGroupSwapchainCreateInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_SWAPCHAIN_CREATE_INFO_KHR,
    eDisplayModeCreateInfoKHR = VK_STRUCTURE_TYPE_DISPLAY_MODE_CREATE_INFO_KHR,
    eDisplaySurfaceCreateInfoKHR = VK_STRUCTURE_TYPE_DISPLAY_SURFACE_CREATE_INFO_KHR,
    eDisplayPresentInfoKHR = VK_STRUCTURE_TYPE_DISPLAY_PRESENT_INFO_KHR,
    eXlibSurfaceCreateInfoKHR = VK_STRUCTURE_TYPE_XLIB_SURFACE_CREATE_INFO_KHR,
    eXcbSurfaceCreateInfoKHR = VK_STRUCTURE_TYPE_XCB_SURFACE_CREATE_INFO_KHR,
    eWaylandSurfaceCreateInfoKHR = VK_STRUCTURE_TYPE_WAYLAND_SURFACE_CREATE_INFO_KHR,
    eAndroidSurfaceCreateInfoKHR = VK_STRUCTURE_TYPE_ANDROID_SURFACE_CREATE_INFO_KHR,
    eWin32SurfaceCreateInfoKHR = VK_STRUCTURE_TYPE_WIN32_SURFACE_CREATE_INFO_KHR,
    eDebugReportCallbackCreateInfoEXT = VK_STRUCTURE_TYPE_DEBUG_REPORT_CALLBACK_CREATE_INFO_EXT,
    eDebugReportCreateInfoEXT = VK_STRUCTURE_TYPE_DEBUG_REPORT_CALLBACK_CREATE_INFO_EXT,
    ePipelineRasterizationStateRasterizationOrderAMD = VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_RASTERIZATION_ORDER_AMD,
    eDebugMarkerObjectNameInfoEXT = VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_NAME_INFO_EXT,
    eDebugMarkerObjectTagInfoEXT = VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_TAG_INFO_EXT,
    eDebugMarkerMarkerInfoEXT = VK_STRUCTURE_TYPE_DEBUG_MARKER_MARKER_INFO_EXT,
    eDedicatedAllocationImageCreateInfoNV = VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_IMAGE_CREATE_INFO_NV,
    eDedicatedAllocationBufferCreateInfoNV = VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_BUFFER_CREATE_INFO_NV,
    eDedicatedAllocationMemoryAllocateInfoNV = VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_MEMORY_ALLOCATE_INFO_NV,
    ePhysicalDeviceTransformFeedbackFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_FEATURES_EXT,
    ePhysicalDeviceTransformFeedbackPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_PROPERTIES_EXT,
    ePipelineRasterizationStateStreamCreateInfoEXT = VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_STREAM_CREATE_INFO_EXT,
    eTextureLodGatherFormatPropertiesAMD = VK_STRUCTURE_TYPE_TEXTURE_LOD_GATHER_FORMAT_PROPERTIES_AMD,
    ePhysicalDeviceCornerSampledImageFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CORNER_SAMPLED_IMAGE_FEATURES_NV,
    eExternalMemoryImageCreateInfoNV = VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO_NV,
    eExportMemoryAllocateInfoNV = VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO_NV,
    eImportMemoryWin32HandleInfoNV = VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_NV,
    eExportMemoryWin32HandleInfoNV = VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_NV,
    eWin32KeyedMutexAcquireReleaseInfoNV = VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_NV,
    eValidationFlagsEXT = VK_STRUCTURE_TYPE_VALIDATION_FLAGS_EXT,
    eViSurfaceCreateInfoNN = VK_STRUCTURE_TYPE_VI_SURFACE_CREATE_INFO_NN,
    eImageViewAstcDecodeModeEXT = VK_STRUCTURE_TYPE_IMAGE_VIEW_ASTC_DECODE_MODE_EXT,
    ePhysicalDeviceAstcDecodeFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ASTC_DECODE_FEATURES_EXT,
    eImportMemoryWin32HandleInfoKHR = VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_KHR,
    eExportMemoryWin32HandleInfoKHR = VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_KHR,
    eMemoryWin32HandlePropertiesKHR = VK_STRUCTURE_TYPE_MEMORY_WIN32_HANDLE_PROPERTIES_KHR,
    eMemoryGetWin32HandleInfoKHR = VK_STRUCTURE_TYPE_MEMORY_GET_WIN32_HANDLE_INFO_KHR,
    eImportMemoryFdInfoKHR = VK_STRUCTURE_TYPE_IMPORT_MEMORY_FD_INFO_KHR,
    eMemoryFdPropertiesKHR = VK_STRUCTURE_TYPE_MEMORY_FD_PROPERTIES_KHR,
    eMemoryGetFdInfoKHR = VK_STRUCTURE_TYPE_MEMORY_GET_FD_INFO_KHR,
    eWin32KeyedMutexAcquireReleaseInfoKHR = VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_KHR,
    eImportSemaphoreWin32HandleInfoKHR = VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR,
    eExportSemaphoreWin32HandleInfoKHR = VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR,
    eD3D12FenceSubmitInfoKHR = VK_STRUCTURE_TYPE_D3D12_FENCE_SUBMIT_INFO_KHR,
    eSemaphoreGetWin32HandleInfoKHR = VK_STRUCTURE_TYPE_SEMAPHORE_GET_WIN32_HANDLE_INFO_KHR,
    eImportSemaphoreFdInfoKHR = VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_FD_INFO_KHR,
    eSemaphoreGetFdInfoKHR = VK_STRUCTURE_TYPE_SEMAPHORE_GET_FD_INFO_KHR,
    ePhysicalDevicePushDescriptorPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PUSH_DESCRIPTOR_PROPERTIES_KHR,
    eCommandBufferInheritanceConditionalRenderingInfoEXT = VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_CONDITIONAL_RENDERING_INFO_EXT,
    ePhysicalDeviceConditionalRenderingFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONDITIONAL_RENDERING_FEATURES_EXT,
    eConditionalRenderingBeginInfoEXT = VK_STRUCTURE_TYPE_CONDITIONAL_RENDERING_BEGIN_INFO_EXT,
    ePhysicalDeviceFloat16Int8FeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FLOAT16_INT8_FEATURES_KHR,
    ePresentRegionsKHR = VK_STRUCTURE_TYPE_PRESENT_REGIONS_KHR,
    eObjectTableCreateInfoNVX = VK_STRUCTURE_TYPE_OBJECT_TABLE_CREATE_INFO_NVX,
    eIndirectCommandsLayoutCreateInfoNVX = VK_STRUCTURE_TYPE_INDIRECT_COMMANDS_LAYOUT_CREATE_INFO_NVX,
    eCmdProcessCommandsInfoNVX = VK_STRUCTURE_TYPE_CMD_PROCESS_COMMANDS_INFO_NVX,
    eCmdReserveSpaceForCommandsInfoNVX = VK_STRUCTURE_TYPE_CMD_RESERVE_SPACE_FOR_COMMANDS_INFO_NVX,
    eDeviceGeneratedCommandsLimitsNVX = VK_STRUCTURE_TYPE_DEVICE_GENERATED_COMMANDS_LIMITS_NVX,
    eDeviceGeneratedCommandsFeaturesNVX = VK_STRUCTURE_TYPE_DEVICE_GENERATED_COMMANDS_FEATURES_NVX,
    ePipelineViewportWScalingStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_W_SCALING_STATE_CREATE_INFO_NV,
    eSurfaceCapabilities2EXT = VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_EXT,
    eDisplayPowerInfoEXT = VK_STRUCTURE_TYPE_DISPLAY_POWER_INFO_EXT,
    eDeviceEventInfoEXT = VK_STRUCTURE_TYPE_DEVICE_EVENT_INFO_EXT,
    eDisplayEventInfoEXT = VK_STRUCTURE_TYPE_DISPLAY_EVENT_INFO_EXT,
    eSwapchainCounterCreateInfoEXT = VK_STRUCTURE_TYPE_SWAPCHAIN_COUNTER_CREATE_INFO_EXT,
    ePresentTimesInfoGOOGLE = VK_STRUCTURE_TYPE_PRESENT_TIMES_INFO_GOOGLE,
    ePhysicalDeviceMultiviewPerViewAttributesPropertiesNVX = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PER_VIEW_ATTRIBUTES_PROPERTIES_NVX,
    ePipelineViewportSwizzleStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SWIZZLE_STATE_CREATE_INFO_NV,
    ePhysicalDeviceDiscardRectanglePropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DISCARD_RECTANGLE_PROPERTIES_EXT,
    ePipelineDiscardRectangleStateCreateInfoEXT = VK_STRUCTURE_TYPE_PIPELINE_DISCARD_RECTANGLE_STATE_CREATE_INFO_EXT,
    ePhysicalDeviceConservativeRasterizationPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONSERVATIVE_RASTERIZATION_PROPERTIES_EXT,
    ePipelineRasterizationConservativeStateCreateInfoEXT = VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_CONSERVATIVE_STATE_CREATE_INFO_EXT,
    eHdrMetadataEXT = VK_STRUCTURE_TYPE_HDR_METADATA_EXT,
    eAttachmentDescription2KHR = VK_STRUCTURE_TYPE_ATTACHMENT_DESCRIPTION_2_KHR,
    eAttachmentReference2KHR = VK_STRUCTURE_TYPE_ATTACHMENT_REFERENCE_2_KHR,
    eSubpassDescription2KHR = VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_2_KHR,
    eSubpassDependency2KHR = VK_STRUCTURE_TYPE_SUBPASS_DEPENDENCY_2_KHR,
    eRenderPassCreateInfo2KHR = VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO_2_KHR,
    eSubpassBeginInfoKHR = VK_STRUCTURE_TYPE_SUBPASS_BEGIN_INFO_KHR,
    eSubpassEndInfoKHR = VK_STRUCTURE_TYPE_SUBPASS_END_INFO_KHR,
    eSharedPresentSurfaceCapabilitiesKHR = VK_STRUCTURE_TYPE_SHARED_PRESENT_SURFACE_CAPABILITIES_KHR,
    eImportFenceWin32HandleInfoKHR = VK_STRUCTURE_TYPE_IMPORT_FENCE_WIN32_HANDLE_INFO_KHR,
    eExportFenceWin32HandleInfoKHR = VK_STRUCTURE_TYPE_EXPORT_FENCE_WIN32_HANDLE_INFO_KHR,
    eFenceGetWin32HandleInfoKHR = VK_STRUCTURE_TYPE_FENCE_GET_WIN32_HANDLE_INFO_KHR,
    eImportFenceFdInfoKHR = VK_STRUCTURE_TYPE_IMPORT_FENCE_FD_INFO_KHR,
    eFenceGetFdInfoKHR = VK_STRUCTURE_TYPE_FENCE_GET_FD_INFO_KHR,
    ePhysicalDeviceSurfaceInfo2KHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SURFACE_INFO_2_KHR,
    eSurfaceCapabilities2KHR = VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_KHR,
    eSurfaceFormat2KHR = VK_STRUCTURE_TYPE_SURFACE_FORMAT_2_KHR,
    eDisplayProperties2KHR = VK_STRUCTURE_TYPE_DISPLAY_PROPERTIES_2_KHR,
    eDisplayPlaneProperties2KHR = VK_STRUCTURE_TYPE_DISPLAY_PLANE_PROPERTIES_2_KHR,
    eDisplayModeProperties2KHR = VK_STRUCTURE_TYPE_DISPLAY_MODE_PROPERTIES_2_KHR,
    eDisplayPlaneInfo2KHR = VK_STRUCTURE_TYPE_DISPLAY_PLANE_INFO_2_KHR,
    eDisplayPlaneCapabilities2KHR = VK_STRUCTURE_TYPE_DISPLAY_PLANE_CAPABILITIES_2_KHR,
    eIosSurfaceCreateInfoMVK = VK_STRUCTURE_TYPE_IOS_SURFACE_CREATE_INFO_MVK,
    eMacosSurfaceCreateInfoMVK = VK_STRUCTURE_TYPE_MACOS_SURFACE_CREATE_INFO_MVK,
    eDebugUtilsObjectNameInfoEXT = VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_NAME_INFO_EXT,
    eDebugUtilsObjectTagInfoEXT = VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_TAG_INFO_EXT,
    eDebugUtilsLabelEXT = VK_STRUCTURE_TYPE_DEBUG_UTILS_LABEL_EXT,
    eDebugUtilsMessengerCallbackDataEXT = VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CALLBACK_DATA_EXT,
    eDebugUtilsMessengerCreateInfoEXT = VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CREATE_INFO_EXT,
    eAndroidHardwareBufferUsageANDROID = VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_USAGE_ANDROID,
    eAndroidHardwareBufferPropertiesANDROID = VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_PROPERTIES_ANDROID,
    eAndroidHardwareBufferFormatPropertiesANDROID = VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_FORMAT_PROPERTIES_ANDROID,
    eImportAndroidHardwareBufferInfoANDROID = VK_STRUCTURE_TYPE_IMPORT_ANDROID_HARDWARE_BUFFER_INFO_ANDROID,
    eMemoryGetAndroidHardwareBufferInfoANDROID = VK_STRUCTURE_TYPE_MEMORY_GET_ANDROID_HARDWARE_BUFFER_INFO_ANDROID,
    eExternalFormatANDROID = VK_STRUCTURE_TYPE_EXTERNAL_FORMAT_ANDROID,
    ePhysicalDeviceSamplerFilterMinmaxPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_FILTER_MINMAX_PROPERTIES_EXT,
    eSamplerReductionModeCreateInfoEXT = VK_STRUCTURE_TYPE_SAMPLER_REDUCTION_MODE_CREATE_INFO_EXT,
    ePhysicalDeviceInlineUniformBlockFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_FEATURES_EXT,
    ePhysicalDeviceInlineUniformBlockPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_PROPERTIES_EXT,
    eWriteDescriptorSetInlineUniformBlockEXT = VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_INLINE_UNIFORM_BLOCK_EXT,
    eDescriptorPoolInlineUniformBlockCreateInfoEXT = VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_INLINE_UNIFORM_BLOCK_CREATE_INFO_EXT,
    eSampleLocationsInfoEXT = VK_STRUCTURE_TYPE_SAMPLE_LOCATIONS_INFO_EXT,
    eRenderPassSampleLocationsBeginInfoEXT = VK_STRUCTURE_TYPE_RENDER_PASS_SAMPLE_LOCATIONS_BEGIN_INFO_EXT,
    ePipelineSampleLocationsStateCreateInfoEXT = VK_STRUCTURE_TYPE_PIPELINE_SAMPLE_LOCATIONS_STATE_CREATE_INFO_EXT,
    ePhysicalDeviceSampleLocationsPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLE_LOCATIONS_PROPERTIES_EXT,
    eMultisamplePropertiesEXT = VK_STRUCTURE_TYPE_MULTISAMPLE_PROPERTIES_EXT,
    eImageFormatListCreateInfoKHR = VK_STRUCTURE_TYPE_IMAGE_FORMAT_LIST_CREATE_INFO_KHR,
    ePhysicalDeviceBlendOperationAdvancedFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_FEATURES_EXT,
    ePhysicalDeviceBlendOperationAdvancedPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_PROPERTIES_EXT,
    ePipelineColorBlendAdvancedStateCreateInfoEXT = VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_ADVANCED_STATE_CREATE_INFO_EXT,
    ePipelineCoverageToColorStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_TO_COLOR_STATE_CREATE_INFO_NV,
    ePipelineCoverageModulationStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_MODULATION_STATE_CREATE_INFO_NV,
    eDrmFormatModifierPropertiesListEXT = VK_STRUCTURE_TYPE_DRM_FORMAT_MODIFIER_PROPERTIES_LIST_EXT,
    eDrmFormatModifierPropertiesEXT = VK_STRUCTURE_TYPE_DRM_FORMAT_MODIFIER_PROPERTIES_EXT,
    ePhysicalDeviceImageDrmFormatModifierInfoEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_DRM_FORMAT_MODIFIER_INFO_EXT,
    eImageDrmFormatModifierListCreateInfoEXT = VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_LIST_CREATE_INFO_EXT,
    eImageDrmFormatModifierExplicitCreateInfoEXT = VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_EXPLICIT_CREATE_INFO_EXT,
    eImageDrmFormatModifierPropertiesEXT = VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_PROPERTIES_EXT,
    eValidationCacheCreateInfoEXT = VK_STRUCTURE_TYPE_VALIDATION_CACHE_CREATE_INFO_EXT,
    eShaderModuleValidationCacheCreateInfoEXT = VK_STRUCTURE_TYPE_SHADER_MODULE_VALIDATION_CACHE_CREATE_INFO_EXT,
    eDescriptorSetLayoutBindingFlagsCreateInfoEXT = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_BINDING_FLAGS_CREATE_INFO_EXT,
    ePhysicalDeviceDescriptorIndexingFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_FEATURES_EXT,
    ePhysicalDeviceDescriptorIndexingPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_PROPERTIES_EXT,
    eDescriptorSetVariableDescriptorCountAllocateInfoEXT = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_ALLOCATE_INFO_EXT,
    eDescriptorSetVariableDescriptorCountLayoutSupportEXT = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_LAYOUT_SUPPORT_EXT,
    ePipelineViewportShadingRateImageStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SHADING_RATE_IMAGE_STATE_CREATE_INFO_NV,
    ePhysicalDeviceShadingRateImageFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_FEATURES_NV,
    ePhysicalDeviceShadingRateImagePropertiesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_PROPERTIES_NV,
    ePipelineViewportCoarseSampleOrderStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_COARSE_SAMPLE_ORDER_STATE_CREATE_INFO_NV,
    eRayTracingPipelineCreateInfoNV = VK_STRUCTURE_TYPE_RAY_TRACING_PIPELINE_CREATE_INFO_NV,
    eAccelerationStructureCreateInfoNV = VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_CREATE_INFO_NV,
    eGeometryNV = VK_STRUCTURE_TYPE_GEOMETRY_NV,
    eGeometryTrianglesNV = VK_STRUCTURE_TYPE_GEOMETRY_TRIANGLES_NV,
    eGeometryAabbNV = VK_STRUCTURE_TYPE_GEOMETRY_AABB_NV,
    eBindAccelerationStructureMemoryInfoNV = VK_STRUCTURE_TYPE_BIND_ACCELERATION_STRUCTURE_MEMORY_INFO_NV,
    eWriteDescriptorSetAccelerationStructureNV = VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_ACCELERATION_STRUCTURE_NV,
    eAccelerationStructureMemoryRequirementsInfoNV = VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_INFO_NV,
    ePhysicalDeviceRayTracingPropertiesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_PROPERTIES_NV,
    eRayTracingShaderGroupCreateInfoNV = VK_STRUCTURE_TYPE_RAY_TRACING_SHADER_GROUP_CREATE_INFO_NV,
    eAccelerationStructureInfoNV = VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_INFO_NV,
    ePhysicalDeviceRepresentativeFragmentTestFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_REPRESENTATIVE_FRAGMENT_TEST_FEATURES_NV,
    ePipelineRepresentativeFragmentTestStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_REPRESENTATIVE_FRAGMENT_TEST_STATE_CREATE_INFO_NV,
    eDeviceQueueGlobalPriorityCreateInfoEXT = VK_STRUCTURE_TYPE_DEVICE_QUEUE_GLOBAL_PRIORITY_CREATE_INFO_EXT,
    ePhysicalDevice8BitStorageFeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_8BIT_STORAGE_FEATURES_KHR,
    eImportMemoryHostPointerInfoEXT = VK_STRUCTURE_TYPE_IMPORT_MEMORY_HOST_POINTER_INFO_EXT,
    eMemoryHostPointerPropertiesEXT = VK_STRUCTURE_TYPE_MEMORY_HOST_POINTER_PROPERTIES_EXT,
    ePhysicalDeviceExternalMemoryHostPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_HOST_PROPERTIES_EXT,
    ePhysicalDeviceShaderAtomicInt64FeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ATOMIC_INT64_FEATURES_KHR,
    eCalibratedTimestampInfoEXT = VK_STRUCTURE_TYPE_CALIBRATED_TIMESTAMP_INFO_EXT,
    ePhysicalDeviceShaderCorePropertiesAMD = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_PROPERTIES_AMD,
    eDeviceMemoryOverallocationCreateInfoAMD = VK_STRUCTURE_TYPE_DEVICE_MEMORY_OVERALLOCATION_CREATE_INFO_AMD,
    ePhysicalDeviceVertexAttributeDivisorPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_PROPERTIES_EXT,
    ePipelineVertexInputDivisorStateCreateInfoEXT = VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_DIVISOR_STATE_CREATE_INFO_EXT,
    ePhysicalDeviceVertexAttributeDivisorFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_FEATURES_EXT,
    ePhysicalDeviceDriverPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DRIVER_PROPERTIES_KHR,
    ePhysicalDeviceFloatControlsPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FLOAT_CONTROLS_PROPERTIES_KHR,
    ePhysicalDeviceDepthStencilResolvePropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_STENCIL_RESOLVE_PROPERTIES_KHR,
    eSubpassDescriptionDepthStencilResolveKHR = VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_DEPTH_STENCIL_RESOLVE_KHR,
    ePhysicalDeviceComputeShaderDerivativesFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COMPUTE_SHADER_DERIVATIVES_FEATURES_NV,
    ePhysicalDeviceMeshShaderFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_FEATURES_NV,
    ePhysicalDeviceMeshShaderPropertiesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_PROPERTIES_NV,
    ePhysicalDeviceFragmentShaderBarycentricFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADER_BARYCENTRIC_FEATURES_NV,
    ePhysicalDeviceShaderImageFootprintFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_IMAGE_FOOTPRINT_FEATURES_NV,
    ePipelineViewportExclusiveScissorStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_EXCLUSIVE_SCISSOR_STATE_CREATE_INFO_NV,
    ePhysicalDeviceExclusiveScissorFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXCLUSIVE_SCISSOR_FEATURES_NV,
    eCheckpointDataNV = VK_STRUCTURE_TYPE_CHECKPOINT_DATA_NV,
    eQueueFamilyCheckpointPropertiesNV = VK_STRUCTURE_TYPE_QUEUE_FAMILY_CHECKPOINT_PROPERTIES_NV,
    ePhysicalDeviceVulkanMemoryModelFeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_MEMORY_MODEL_FEATURES_KHR,
    ePhysicalDevicePciBusInfoPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PCI_BUS_INFO_PROPERTIES_EXT,
    eImagepipeSurfaceCreateInfoFUCHSIA = VK_STRUCTURE_TYPE_IMAGEPIPE_SURFACE_CREATE_INFO_FUCHSIA,
    ePhysicalDeviceFragmentDensityMapFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_FEATURES_EXT,
    ePhysicalDeviceFragmentDensityMapPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_PROPERTIES_EXT,
    eRenderPassFragmentDensityMapCreateInfoEXT = VK_STRUCTURE_TYPE_RENDER_PASS_FRAGMENT_DENSITY_MAP_CREATE_INFO_EXT,
    ePhysicalDeviceScalarBlockLayoutFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SCALAR_BLOCK_LAYOUT_FEATURES_EXT,
    ePhysicalDeviceMemoryBudgetPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_BUDGET_PROPERTIES_EXT,
    ePhysicalDeviceMemoryPriorityFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PRIORITY_FEATURES_EXT,
    eMemoryPriorityAllocateInfoEXT = VK_STRUCTURE_TYPE_MEMORY_PRIORITY_ALLOCATE_INFO_EXT,
    ePhysicalDeviceBufferAddressFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BUFFER_ADDRESS_FEATURES_EXT,
    eBufferDeviceAddressInfoEXT = VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_INFO_EXT,
    eBufferDeviceAddressCreateInfoEXT = VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_CREATE_INFO_EXT,
    eImageStencilUsageCreateInfoEXT = VK_STRUCTURE_TYPE_IMAGE_STENCIL_USAGE_CREATE_INFO_EXT,
    eValidationFeaturesEXT = VK_STRUCTURE_TYPE_VALIDATION_FEATURES_EXT
  };

  struct ApplicationInfo
  {
    ApplicationInfo( const char* pApplicationName_ = nullptr,
                     uint32_t applicationVersion_ = 0,
                     const char* pEngineName_ = nullptr,
                     uint32_t engineVersion_ = 0,
                     uint32_t apiVersion_ = 0 )
      : pApplicationName( pApplicationName_ )
      , applicationVersion( applicationVersion_ )
      , pEngineName( pEngineName_ )
      , engineVersion( engineVersion_ )
      , apiVersion( apiVersion_ )
    {
    }

    ApplicationInfo( VkApplicationInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ApplicationInfo ) );
    }

    ApplicationInfo& operator=( VkApplicationInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ApplicationInfo ) );
      return *this;
    }
    ApplicationInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ApplicationInfo& setPApplicationName( const char* pApplicationName_ )
    {
      pApplicationName = pApplicationName_;
      return *this;
    }

    ApplicationInfo& setApplicationVersion( uint32_t applicationVersion_ )
    {
      applicationVersion = applicationVersion_;
      return *this;
    }

    ApplicationInfo& setPEngineName( const char* pEngineName_ )
    {
      pEngineName = pEngineName_;
      return *this;
    }

    ApplicationInfo& setEngineVersion( uint32_t engineVersion_ )
    {
      engineVersion = engineVersion_;
      return *this;
    }

    ApplicationInfo& setApiVersion( uint32_t apiVersion_ )
    {
      apiVersion = apiVersion_;
      return *this;
    }

    operator VkApplicationInfo const&() const
    {
      return *reinterpret_cast<const VkApplicationInfo*>(this);
    }

    operator VkApplicationInfo &()
    {
      return *reinterpret_cast<VkApplicationInfo*>(this);
    }

    bool operator==( ApplicationInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( pApplicationName == rhs.pApplicationName )
          && ( applicationVersion == rhs.applicationVersion )
          && ( pEngineName == rhs.pEngineName )
          && ( engineVersion == rhs.engineVersion )
          && ( apiVersion == rhs.apiVersion );
    }

    bool operator!=( ApplicationInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eApplicationInfo;

  public:
    const void* pNext = nullptr;
    const char* pApplicationName;
    uint32_t applicationVersion;
    const char* pEngineName;
    uint32_t engineVersion;
    uint32_t apiVersion;
  };
  static_assert( sizeof( ApplicationInfo ) == sizeof( VkApplicationInfo ), "struct and wrapper have different size!" );

  struct InstanceCreateInfo
  {
    InstanceCreateInfo( InstanceCreateFlags flags_ = InstanceCreateFlags(),
                        const ApplicationInfo* pApplicationInfo_ = nullptr,
                        uint32_t enabledLayerCount_ = 0,
                        const char* const* ppEnabledLayerNames_ = nullptr,
                        uint32_t enabledExtensionCount_ = 0,
                        const char* const* ppEnabledExtensionNames_ = nullptr )
      : flags( flags_ )
      , pApplicationInfo( pApplicationInfo_ )
      , enabledLayerCount( enabledLayerCount_ )
      , ppEnabledLayerNames( ppEnabledLayerNames_ )
      , enabledExtensionCount( enabledExtensionCount_ )
      , ppEnabledExtensionNames( ppEnabledExtensionNames_ )
    {
    }

    InstanceCreateInfo( VkInstanceCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( InstanceCreateInfo ) );
    }

    InstanceCreateInfo& operator=( VkInstanceCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( InstanceCreateInfo ) );
      return *this;
    }
    InstanceCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    InstanceCreateInfo& setFlags( InstanceCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    InstanceCreateInfo& setPApplicationInfo( const ApplicationInfo* pApplicationInfo_ )
    {
      pApplicationInfo = pApplicationInfo_;
      return *this;
    }

    InstanceCreateInfo& setEnabledLayerCount( uint32_t enabledLayerCount_ )
    {
      enabledLayerCount = enabledLayerCount_;
      return *this;
    }

    InstanceCreateInfo& setPpEnabledLayerNames( const char* const* ppEnabledLayerNames_ )
    {
      ppEnabledLayerNames = ppEnabledLayerNames_;
      return *this;
    }

    InstanceCreateInfo& setEnabledExtensionCount( uint32_t enabledExtensionCount_ )
    {
      enabledExtensionCount = enabledExtensionCount_;
      return *this;
    }

    InstanceCreateInfo& setPpEnabledExtensionNames( const char* const* ppEnabledExtensionNames_ )
    {
      ppEnabledExtensionNames = ppEnabledExtensionNames_;
      return *this;
    }

    operator VkInstanceCreateInfo const&() const
    {
      return *reinterpret_cast<const VkInstanceCreateInfo*>(this);
    }

    operator VkInstanceCreateInfo &()
    {
      return *reinterpret_cast<VkInstanceCreateInfo*>(this);
    }

    bool operator==( InstanceCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( pApplicationInfo == rhs.pApplicationInfo )
          && ( enabledLayerCount == rhs.enabledLayerCount )
          && ( ppEnabledLayerNames == rhs.ppEnabledLayerNames )
          && ( enabledExtensionCount == rhs.enabledExtensionCount )
          && ( ppEnabledExtensionNames == rhs.ppEnabledExtensionNames );
    }

    bool operator!=( InstanceCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eInstanceCreateInfo;

  public:
    const void* pNext = nullptr;
    InstanceCreateFlags flags;
    const ApplicationInfo* pApplicationInfo;
    uint32_t enabledLayerCount;
    const char* const* ppEnabledLayerNames;
    uint32_t enabledExtensionCount;
    const char* const* ppEnabledExtensionNames;
  };
  static_assert( sizeof( InstanceCreateInfo ) == sizeof( VkInstanceCreateInfo ), "struct and wrapper have different size!" );

  struct MemoryAllocateInfo
  {
    MemoryAllocateInfo( DeviceSize allocationSize_ = 0,
                        uint32_t memoryTypeIndex_ = 0 )
      : allocationSize( allocationSize_ )
      , memoryTypeIndex( memoryTypeIndex_ )
    {
    }

    MemoryAllocateInfo( VkMemoryAllocateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( MemoryAllocateInfo ) );
    }

    MemoryAllocateInfo& operator=( VkMemoryAllocateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( MemoryAllocateInfo ) );
      return *this;
    }
    MemoryAllocateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    MemoryAllocateInfo& setAllocationSize( DeviceSize allocationSize_ )
    {
      allocationSize = allocationSize_;
      return *this;
    }

    MemoryAllocateInfo& setMemoryTypeIndex( uint32_t memoryTypeIndex_ )
    {
      memoryTypeIndex = memoryTypeIndex_;
      return *this;
    }

    operator VkMemoryAllocateInfo const&() const
    {
      return *reinterpret_cast<const VkMemoryAllocateInfo*>(this);
    }

    operator VkMemoryAllocateInfo &()
    {
      return *reinterpret_cast<VkMemoryAllocateInfo*>(this);
    }

    bool operator==( MemoryAllocateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( allocationSize == rhs.allocationSize )
          && ( memoryTypeIndex == rhs.memoryTypeIndex );
    }

    bool operator!=( MemoryAllocateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eMemoryAllocateInfo;

  public:
    const void* pNext = nullptr;
    DeviceSize allocationSize;
    uint32_t memoryTypeIndex;
  };
  static_assert( sizeof( MemoryAllocateInfo ) == sizeof( VkMemoryAllocateInfo ), "struct and wrapper have different size!" );

  struct MappedMemoryRange
  {
    MappedMemoryRange( DeviceMemory memory_ = DeviceMemory(),
                       DeviceSize offset_ = 0,
                       DeviceSize size_ = 0 )
      : memory( memory_ )
      , offset( offset_ )
      , size( size_ )
    {
    }

    MappedMemoryRange( VkMappedMemoryRange const & rhs )
    {
      memcpy( this, &rhs, sizeof( MappedMemoryRange ) );
    }

    MappedMemoryRange& operator=( VkMappedMemoryRange const & rhs )
    {
      memcpy( this, &rhs, sizeof( MappedMemoryRange ) );
      return *this;
    }
    MappedMemoryRange& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    MappedMemoryRange& setMemory( DeviceMemory memory_ )
    {
      memory = memory_;
      return *this;
    }

    MappedMemoryRange& setOffset( DeviceSize offset_ )
    {
      offset = offset_;
      return *this;
    }

    MappedMemoryRange& setSize( DeviceSize size_ )
    {
      size = size_;
      return *this;
    }

    operator VkMappedMemoryRange const&() const
    {
      return *reinterpret_cast<const VkMappedMemoryRange*>(this);
    }

    operator VkMappedMemoryRange &()
    {
      return *reinterpret_cast<VkMappedMemoryRange*>(this);
    }

    bool operator==( MappedMemoryRange const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( memory == rhs.memory )
          && ( offset == rhs.offset )
          && ( size == rhs.size );
    }

    bool operator!=( MappedMemoryRange const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eMappedMemoryRange;

  public:
    const void* pNext = nullptr;
    DeviceMemory memory;
    DeviceSize offset;
    DeviceSize size;
  };
  static_assert( sizeof( MappedMemoryRange ) == sizeof( VkMappedMemoryRange ), "struct and wrapper have different size!" );

  struct WriteDescriptorSet
  {
    WriteDescriptorSet( DescriptorSet dstSet_ = DescriptorSet(),
                        uint32_t dstBinding_ = 0,
                        uint32_t dstArrayElement_ = 0,
                        uint32_t descriptorCount_ = 0,
                        DescriptorType descriptorType_ = DescriptorType::eSampler,
                        const DescriptorImageInfo* pImageInfo_ = nullptr,
                        const DescriptorBufferInfo* pBufferInfo_ = nullptr,
                        const BufferView* pTexelBufferView_ = nullptr )
      : dstSet( dstSet_ )
      , dstBinding( dstBinding_ )
      , dstArrayElement( dstArrayElement_ )
      , descriptorCount( descriptorCount_ )
      , descriptorType( descriptorType_ )
      , pImageInfo( pImageInfo_ )
      , pBufferInfo( pBufferInfo_ )
      , pTexelBufferView( pTexelBufferView_ )
    {
    }

    WriteDescriptorSet( VkWriteDescriptorSet const & rhs )
    {
      memcpy( this, &rhs, sizeof( WriteDescriptorSet ) );
    }

    WriteDescriptorSet& operator=( VkWriteDescriptorSet const & rhs )
    {
      memcpy( this, &rhs, sizeof( WriteDescriptorSet ) );
      return *this;
    }
    WriteDescriptorSet& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    WriteDescriptorSet& setDstSet( DescriptorSet dstSet_ )
    {
      dstSet = dstSet_;
      return *this;
    }

    WriteDescriptorSet& setDstBinding( uint32_t dstBinding_ )
    {
      dstBinding = dstBinding_;
      return *this;
    }

    WriteDescriptorSet& setDstArrayElement( uint32_t dstArrayElement_ )
    {
      dstArrayElement = dstArrayElement_;
      return *this;
    }

    WriteDescriptorSet& setDescriptorCount( uint32_t descriptorCount_ )
    {
      descriptorCount = descriptorCount_;
      return *this;
    }

    WriteDescriptorSet& setDescriptorType( DescriptorType descriptorType_ )
    {
      descriptorType = descriptorType_;
      return *this;
    }

    WriteDescriptorSet& setPImageInfo( const DescriptorImageInfo* pImageInfo_ )
    {
      pImageInfo = pImageInfo_;
      return *this;
    }

    WriteDescriptorSet& setPBufferInfo( const DescriptorBufferInfo* pBufferInfo_ )
    {
      pBufferInfo = pBufferInfo_;
      return *this;
    }

    WriteDescriptorSet& setPTexelBufferView( const BufferView* pTexelBufferView_ )
    {
      pTexelBufferView = pTexelBufferView_;
      return *this;
    }

    operator VkWriteDescriptorSet const&() const
    {
      return *reinterpret_cast<const VkWriteDescriptorSet*>(this);
    }

    operator VkWriteDescriptorSet &()
    {
      return *reinterpret_cast<VkWriteDescriptorSet*>(this);
    }

    bool operator==( WriteDescriptorSet const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( dstSet == rhs.dstSet )
          && ( dstBinding == rhs.dstBinding )
          && ( dstArrayElement == rhs.dstArrayElement )
          && ( descriptorCount == rhs.descriptorCount )
          && ( descriptorType == rhs.descriptorType )
          && ( pImageInfo == rhs.pImageInfo )
          && ( pBufferInfo == rhs.pBufferInfo )
          && ( pTexelBufferView == rhs.pTexelBufferView );
    }

    bool operator!=( WriteDescriptorSet const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eWriteDescriptorSet;

  public:
    const void* pNext = nullptr;
    DescriptorSet dstSet;
    uint32_t dstBinding;
    uint32_t dstArrayElement;
    uint32_t descriptorCount;
    DescriptorType descriptorType;
    const DescriptorImageInfo* pImageInfo;
    const DescriptorBufferInfo* pBufferInfo;
    const BufferView* pTexelBufferView;
  };
  static_assert( sizeof( WriteDescriptorSet ) == sizeof( VkWriteDescriptorSet ), "struct and wrapper have different size!" );

  struct CopyDescriptorSet
  {
    CopyDescriptorSet( DescriptorSet srcSet_ = DescriptorSet(),
                       uint32_t srcBinding_ = 0,
                       uint32_t srcArrayElement_ = 0,
                       DescriptorSet dstSet_ = DescriptorSet(),
                       uint32_t dstBinding_ = 0,
                       uint32_t dstArrayElement_ = 0,
                       uint32_t descriptorCount_ = 0 )
      : srcSet( srcSet_ )
      , srcBinding( srcBinding_ )
      , srcArrayElement( srcArrayElement_ )
      , dstSet( dstSet_ )
      , dstBinding( dstBinding_ )
      , dstArrayElement( dstArrayElement_ )
      , descriptorCount( descriptorCount_ )
    {
    }

    CopyDescriptorSet( VkCopyDescriptorSet const & rhs )
    {
      memcpy( this, &rhs, sizeof( CopyDescriptorSet ) );
    }

    CopyDescriptorSet& operator=( VkCopyDescriptorSet const & rhs )
    {
      memcpy( this, &rhs, sizeof( CopyDescriptorSet ) );
      return *this;
    }
    CopyDescriptorSet& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CopyDescriptorSet& setSrcSet( DescriptorSet srcSet_ )
    {
      srcSet = srcSet_;
      return *this;
    }

    CopyDescriptorSet& setSrcBinding( uint32_t srcBinding_ )
    {
      srcBinding = srcBinding_;
      return *this;
    }

    CopyDescriptorSet& setSrcArrayElement( uint32_t srcArrayElement_ )
    {
      srcArrayElement = srcArrayElement_;
      return *this;
    }

    CopyDescriptorSet& setDstSet( DescriptorSet dstSet_ )
    {
      dstSet = dstSet_;
      return *this;
    }

    CopyDescriptorSet& setDstBinding( uint32_t dstBinding_ )
    {
      dstBinding = dstBinding_;
      return *this;
    }

    CopyDescriptorSet& setDstArrayElement( uint32_t dstArrayElement_ )
    {
      dstArrayElement = dstArrayElement_;
      return *this;
    }

    CopyDescriptorSet& setDescriptorCount( uint32_t descriptorCount_ )
    {
      descriptorCount = descriptorCount_;
      return *this;
    }

    operator VkCopyDescriptorSet const&() const
    {
      return *reinterpret_cast<const VkCopyDescriptorSet*>(this);
    }

    operator VkCopyDescriptorSet &()
    {
      return *reinterpret_cast<VkCopyDescriptorSet*>(this);
    }

    bool operator==( CopyDescriptorSet const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( srcSet == rhs.srcSet )
          && ( srcBinding == rhs.srcBinding )
          && ( srcArrayElement == rhs.srcArrayElement )
          && ( dstSet == rhs.dstSet )
          && ( dstBinding == rhs.dstBinding )
          && ( dstArrayElement == rhs.dstArrayElement )
          && ( descriptorCount == rhs.descriptorCount );
    }

    bool operator!=( CopyDescriptorSet const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCopyDescriptorSet;

  public:
    const void* pNext = nullptr;
    DescriptorSet srcSet;
    uint32_t srcBinding;
    uint32_t srcArrayElement;
    DescriptorSet dstSet;
    uint32_t dstBinding;
    uint32_t dstArrayElement;
    uint32_t descriptorCount;
  };
  static_assert( sizeof( CopyDescriptorSet ) == sizeof( VkCopyDescriptorSet ), "struct and wrapper have different size!" );

  struct BufferViewCreateInfo
  {
    BufferViewCreateInfo( BufferViewCreateFlags flags_ = BufferViewCreateFlags(),
                          Buffer buffer_ = Buffer(),
                          Format format_ = Format::eUndefined,
                          DeviceSize offset_ = 0,
                          DeviceSize range_ = 0 )
      : flags( flags_ )
      , buffer( buffer_ )
      , format( format_ )
      , offset( offset_ )
      , range( range_ )
    {
    }

    BufferViewCreateInfo( VkBufferViewCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferViewCreateInfo ) );
    }

    BufferViewCreateInfo& operator=( VkBufferViewCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferViewCreateInfo ) );
      return *this;
    }
    BufferViewCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BufferViewCreateInfo& setFlags( BufferViewCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    BufferViewCreateInfo& setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    BufferViewCreateInfo& setFormat( Format format_ )
    {
      format = format_;
      return *this;
    }

    BufferViewCreateInfo& setOffset( DeviceSize offset_ )
    {
      offset = offset_;
      return *this;
    }

    BufferViewCreateInfo& setRange( DeviceSize range_ )
    {
      range = range_;
      return *this;
    }

    operator VkBufferViewCreateInfo const&() const
    {
      return *reinterpret_cast<const VkBufferViewCreateInfo*>(this);
    }

    operator VkBufferViewCreateInfo &()
    {
      return *reinterpret_cast<VkBufferViewCreateInfo*>(this);
    }

    bool operator==( BufferViewCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( buffer == rhs.buffer )
          && ( format == rhs.format )
          && ( offset == rhs.offset )
          && ( range == rhs.range );
    }

    bool operator!=( BufferViewCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBufferViewCreateInfo;

  public:
    const void* pNext = nullptr;
    BufferViewCreateFlags flags;
    Buffer buffer;
    Format format;
    DeviceSize offset;
    DeviceSize range;
  };
  static_assert( sizeof( BufferViewCreateInfo ) == sizeof( VkBufferViewCreateInfo ), "struct and wrapper have different size!" );

  struct ShaderModuleCreateInfo
  {
    ShaderModuleCreateInfo( ShaderModuleCreateFlags flags_ = ShaderModuleCreateFlags(),
                            size_t codeSize_ = 0,
                            const uint32_t* pCode_ = nullptr )
      : flags( flags_ )
      , codeSize( codeSize_ )
      , pCode( pCode_ )
    {
    }

    ShaderModuleCreateInfo( VkShaderModuleCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ShaderModuleCreateInfo ) );
    }

    ShaderModuleCreateInfo& operator=( VkShaderModuleCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ShaderModuleCreateInfo ) );
      return *this;
    }
    ShaderModuleCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ShaderModuleCreateInfo& setFlags( ShaderModuleCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    ShaderModuleCreateInfo& setCodeSize( size_t codeSize_ )
    {
      codeSize = codeSize_;
      return *this;
    }

    ShaderModuleCreateInfo& setPCode( const uint32_t* pCode_ )
    {
      pCode = pCode_;
      return *this;
    }

    operator VkShaderModuleCreateInfo const&() const
    {
      return *reinterpret_cast<const VkShaderModuleCreateInfo*>(this);
    }

    operator VkShaderModuleCreateInfo &()
    {
      return *reinterpret_cast<VkShaderModuleCreateInfo*>(this);
    }

    bool operator==( ShaderModuleCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( codeSize == rhs.codeSize )
          && ( pCode == rhs.pCode );
    }

    bool operator!=( ShaderModuleCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eShaderModuleCreateInfo;

  public:
    const void* pNext = nullptr;
    ShaderModuleCreateFlags flags;
    size_t codeSize;
    const uint32_t* pCode;
  };
  static_assert( sizeof( ShaderModuleCreateInfo ) == sizeof( VkShaderModuleCreateInfo ), "struct and wrapper have different size!" );

  struct DescriptorSetAllocateInfo
  {
    DescriptorSetAllocateInfo( DescriptorPool descriptorPool_ = DescriptorPool(),
                               uint32_t descriptorSetCount_ = 0,
                               const DescriptorSetLayout* pSetLayouts_ = nullptr )
      : descriptorPool( descriptorPool_ )
      , descriptorSetCount( descriptorSetCount_ )
      , pSetLayouts( pSetLayouts_ )
    {
    }

    DescriptorSetAllocateInfo( VkDescriptorSetAllocateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetAllocateInfo ) );
    }

    DescriptorSetAllocateInfo& operator=( VkDescriptorSetAllocateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetAllocateInfo ) );
      return *this;
    }
    DescriptorSetAllocateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DescriptorSetAllocateInfo& setDescriptorPool( DescriptorPool descriptorPool_ )
    {
      descriptorPool = descriptorPool_;
      return *this;
    }

    DescriptorSetAllocateInfo& setDescriptorSetCount( uint32_t descriptorSetCount_ )
    {
      descriptorSetCount = descriptorSetCount_;
      return *this;
    }

    DescriptorSetAllocateInfo& setPSetLayouts( const DescriptorSetLayout* pSetLayouts_ )
    {
      pSetLayouts = pSetLayouts_;
      return *this;
    }

    operator VkDescriptorSetAllocateInfo const&() const
    {
      return *reinterpret_cast<const VkDescriptorSetAllocateInfo*>(this);
    }

    operator VkDescriptorSetAllocateInfo &()
    {
      return *reinterpret_cast<VkDescriptorSetAllocateInfo*>(this);
    }

    bool operator==( DescriptorSetAllocateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( descriptorPool == rhs.descriptorPool )
          && ( descriptorSetCount == rhs.descriptorSetCount )
          && ( pSetLayouts == rhs.pSetLayouts );
    }

    bool operator!=( DescriptorSetAllocateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDescriptorSetAllocateInfo;

  public:
    const void* pNext = nullptr;
    DescriptorPool descriptorPool;
    uint32_t descriptorSetCount;
    const DescriptorSetLayout* pSetLayouts;
  };
  static_assert( sizeof( DescriptorSetAllocateInfo ) == sizeof( VkDescriptorSetAllocateInfo ), "struct and wrapper have different size!" );

  struct PipelineVertexInputStateCreateInfo
  {
    PipelineVertexInputStateCreateInfo( PipelineVertexInputStateCreateFlags flags_ = PipelineVertexInputStateCreateFlags(),
                                        uint32_t vertexBindingDescriptionCount_ = 0,
                                        const VertexInputBindingDescription* pVertexBindingDescriptions_ = nullptr,
                                        uint32_t vertexAttributeDescriptionCount_ = 0,
                                        const VertexInputAttributeDescription* pVertexAttributeDescriptions_ = nullptr )
      : flags( flags_ )
      , vertexBindingDescriptionCount( vertexBindingDescriptionCount_ )
      , pVertexBindingDescriptions( pVertexBindingDescriptions_ )
      , vertexAttributeDescriptionCount( vertexAttributeDescriptionCount_ )
      , pVertexAttributeDescriptions( pVertexAttributeDescriptions_ )
    {
    }

    PipelineVertexInputStateCreateInfo( VkPipelineVertexInputStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineVertexInputStateCreateInfo ) );
    }

    PipelineVertexInputStateCreateInfo& operator=( VkPipelineVertexInputStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineVertexInputStateCreateInfo ) );
      return *this;
    }
    PipelineVertexInputStateCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineVertexInputStateCreateInfo& setFlags( PipelineVertexInputStateCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineVertexInputStateCreateInfo& setVertexBindingDescriptionCount( uint32_t vertexBindingDescriptionCount_ )
    {
      vertexBindingDescriptionCount = vertexBindingDescriptionCount_;
      return *this;
    }

    PipelineVertexInputStateCreateInfo& setPVertexBindingDescriptions( const VertexInputBindingDescription* pVertexBindingDescriptions_ )
    {
      pVertexBindingDescriptions = pVertexBindingDescriptions_;
      return *this;
    }

    PipelineVertexInputStateCreateInfo& setVertexAttributeDescriptionCount( uint32_t vertexAttributeDescriptionCount_ )
    {
      vertexAttributeDescriptionCount = vertexAttributeDescriptionCount_;
      return *this;
    }

    PipelineVertexInputStateCreateInfo& setPVertexAttributeDescriptions( const VertexInputAttributeDescription* pVertexAttributeDescriptions_ )
    {
      pVertexAttributeDescriptions = pVertexAttributeDescriptions_;
      return *this;
    }

    operator VkPipelineVertexInputStateCreateInfo const&() const
    {
      return *reinterpret_cast<const VkPipelineVertexInputStateCreateInfo*>(this);
    }

    operator VkPipelineVertexInputStateCreateInfo &()
    {
      return *reinterpret_cast<VkPipelineVertexInputStateCreateInfo*>(this);
    }

    bool operator==( PipelineVertexInputStateCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( vertexBindingDescriptionCount == rhs.vertexBindingDescriptionCount )
          && ( pVertexBindingDescriptions == rhs.pVertexBindingDescriptions )
          && ( vertexAttributeDescriptionCount == rhs.vertexAttributeDescriptionCount )
          && ( pVertexAttributeDescriptions == rhs.pVertexAttributeDescriptions );
    }

    bool operator!=( PipelineVertexInputStateCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineVertexInputStateCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineVertexInputStateCreateFlags flags;
    uint32_t vertexBindingDescriptionCount;
    const VertexInputBindingDescription* pVertexBindingDescriptions;
    uint32_t vertexAttributeDescriptionCount;
    const VertexInputAttributeDescription* pVertexAttributeDescriptions;
  };
  static_assert( sizeof( PipelineVertexInputStateCreateInfo ) == sizeof( VkPipelineVertexInputStateCreateInfo ), "struct and wrapper have different size!" );

  struct PipelineInputAssemblyStateCreateInfo
  {
    PipelineInputAssemblyStateCreateInfo( PipelineInputAssemblyStateCreateFlags flags_ = PipelineInputAssemblyStateCreateFlags(),
                                          PrimitiveTopology topology_ = PrimitiveTopology::ePointList,
                                          Bool32 primitiveRestartEnable_ = 0 )
      : flags( flags_ )
      , topology( topology_ )
      , primitiveRestartEnable( primitiveRestartEnable_ )
    {
    }

    PipelineInputAssemblyStateCreateInfo( VkPipelineInputAssemblyStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineInputAssemblyStateCreateInfo ) );
    }

    PipelineInputAssemblyStateCreateInfo& operator=( VkPipelineInputAssemblyStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineInputAssemblyStateCreateInfo ) );
      return *this;
    }
    PipelineInputAssemblyStateCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineInputAssemblyStateCreateInfo& setFlags( PipelineInputAssemblyStateCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineInputAssemblyStateCreateInfo& setTopology( PrimitiveTopology topology_ )
    {
      topology = topology_;
      return *this;
    }

    PipelineInputAssemblyStateCreateInfo& setPrimitiveRestartEnable( Bool32 primitiveRestartEnable_ )
    {
      primitiveRestartEnable = primitiveRestartEnable_;
      return *this;
    }

    operator VkPipelineInputAssemblyStateCreateInfo const&() const
    {
      return *reinterpret_cast<const VkPipelineInputAssemblyStateCreateInfo*>(this);
    }

    operator VkPipelineInputAssemblyStateCreateInfo &()
    {
      return *reinterpret_cast<VkPipelineInputAssemblyStateCreateInfo*>(this);
    }

    bool operator==( PipelineInputAssemblyStateCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( topology == rhs.topology )
          && ( primitiveRestartEnable == rhs.primitiveRestartEnable );
    }

    bool operator!=( PipelineInputAssemblyStateCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineInputAssemblyStateCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineInputAssemblyStateCreateFlags flags;
    PrimitiveTopology topology;
    Bool32 primitiveRestartEnable;
  };
  static_assert( sizeof( PipelineInputAssemblyStateCreateInfo ) == sizeof( VkPipelineInputAssemblyStateCreateInfo ), "struct and wrapper have different size!" );

  struct PipelineTessellationStateCreateInfo
  {
    PipelineTessellationStateCreateInfo( PipelineTessellationStateCreateFlags flags_ = PipelineTessellationStateCreateFlags(),
                                         uint32_t patchControlPoints_ = 0 )
      : flags( flags_ )
      , patchControlPoints( patchControlPoints_ )
    {
    }

    PipelineTessellationStateCreateInfo( VkPipelineTessellationStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineTessellationStateCreateInfo ) );
    }

    PipelineTessellationStateCreateInfo& operator=( VkPipelineTessellationStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineTessellationStateCreateInfo ) );
      return *this;
    }
    PipelineTessellationStateCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineTessellationStateCreateInfo& setFlags( PipelineTessellationStateCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineTessellationStateCreateInfo& setPatchControlPoints( uint32_t patchControlPoints_ )
    {
      patchControlPoints = patchControlPoints_;
      return *this;
    }

    operator VkPipelineTessellationStateCreateInfo const&() const
    {
      return *reinterpret_cast<const VkPipelineTessellationStateCreateInfo*>(this);
    }

    operator VkPipelineTessellationStateCreateInfo &()
    {
      return *reinterpret_cast<VkPipelineTessellationStateCreateInfo*>(this);
    }

    bool operator==( PipelineTessellationStateCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( patchControlPoints == rhs.patchControlPoints );
    }

    bool operator!=( PipelineTessellationStateCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineTessellationStateCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineTessellationStateCreateFlags flags;
    uint32_t patchControlPoints;
  };
  static_assert( sizeof( PipelineTessellationStateCreateInfo ) == sizeof( VkPipelineTessellationStateCreateInfo ), "struct and wrapper have different size!" );

  struct PipelineViewportStateCreateInfo
  {
    PipelineViewportStateCreateInfo( PipelineViewportStateCreateFlags flags_ = PipelineViewportStateCreateFlags(),
                                     uint32_t viewportCount_ = 0,
                                     const Viewport* pViewports_ = nullptr,
                                     uint32_t scissorCount_ = 0,
                                     const Rect2D* pScissors_ = nullptr )
      : flags( flags_ )
      , viewportCount( viewportCount_ )
      , pViewports( pViewports_ )
      , scissorCount( scissorCount_ )
      , pScissors( pScissors_ )
    {
    }

    PipelineViewportStateCreateInfo( VkPipelineViewportStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineViewportStateCreateInfo ) );
    }

    PipelineViewportStateCreateInfo& operator=( VkPipelineViewportStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineViewportStateCreateInfo ) );
      return *this;
    }
    PipelineViewportStateCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineViewportStateCreateInfo& setFlags( PipelineViewportStateCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineViewportStateCreateInfo& setViewportCount( uint32_t viewportCount_ )
    {
      viewportCount = viewportCount_;
      return *this;
    }

    PipelineViewportStateCreateInfo& setPViewports( const Viewport* pViewports_ )
    {
      pViewports = pViewports_;
      return *this;
    }

    PipelineViewportStateCreateInfo& setScissorCount( uint32_t scissorCount_ )
    {
      scissorCount = scissorCount_;
      return *this;
    }

    PipelineViewportStateCreateInfo& setPScissors( const Rect2D* pScissors_ )
    {
      pScissors = pScissors_;
      return *this;
    }

    operator VkPipelineViewportStateCreateInfo const&() const
    {
      return *reinterpret_cast<const VkPipelineViewportStateCreateInfo*>(this);
    }

    operator VkPipelineViewportStateCreateInfo &()
    {
      return *reinterpret_cast<VkPipelineViewportStateCreateInfo*>(this);
    }

    bool operator==( PipelineViewportStateCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( viewportCount == rhs.viewportCount )
          && ( pViewports == rhs.pViewports )
          && ( scissorCount == rhs.scissorCount )
          && ( pScissors == rhs.pScissors );
    }

    bool operator!=( PipelineViewportStateCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineViewportStateCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineViewportStateCreateFlags flags;
    uint32_t viewportCount;
    const Viewport* pViewports;
    uint32_t scissorCount;
    const Rect2D* pScissors;
  };
  static_assert( sizeof( PipelineViewportStateCreateInfo ) == sizeof( VkPipelineViewportStateCreateInfo ), "struct and wrapper have different size!" );

  struct PipelineRasterizationStateCreateInfo
  {
    PipelineRasterizationStateCreateInfo( PipelineRasterizationStateCreateFlags flags_ = PipelineRasterizationStateCreateFlags(),
                                          Bool32 depthClampEnable_ = 0,
                                          Bool32 rasterizerDiscardEnable_ = 0,
                                          PolygonMode polygonMode_ = PolygonMode::eFill,
                                          CullModeFlags cullMode_ = CullModeFlags(),
                                          FrontFace frontFace_ = FrontFace::eCounterClockwise,
                                          Bool32 depthBiasEnable_ = 0,
                                          float depthBiasConstantFactor_ = 0,
                                          float depthBiasClamp_ = 0,
                                          float depthBiasSlopeFactor_ = 0,
                                          float lineWidth_ = 0 )
      : flags( flags_ )
      , depthClampEnable( depthClampEnable_ )
      , rasterizerDiscardEnable( rasterizerDiscardEnable_ )
      , polygonMode( polygonMode_ )
      , cullMode( cullMode_ )
      , frontFace( frontFace_ )
      , depthBiasEnable( depthBiasEnable_ )
      , depthBiasConstantFactor( depthBiasConstantFactor_ )
      , depthBiasClamp( depthBiasClamp_ )
      , depthBiasSlopeFactor( depthBiasSlopeFactor_ )
      , lineWidth( lineWidth_ )
    {
    }

    PipelineRasterizationStateCreateInfo( VkPipelineRasterizationStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineRasterizationStateCreateInfo ) );
    }

    PipelineRasterizationStateCreateInfo& operator=( VkPipelineRasterizationStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineRasterizationStateCreateInfo ) );
      return *this;
    }
    PipelineRasterizationStateCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineRasterizationStateCreateInfo& setFlags( PipelineRasterizationStateCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineRasterizationStateCreateInfo& setDepthClampEnable( Bool32 depthClampEnable_ )
    {
      depthClampEnable = depthClampEnable_;
      return *this;
    }

    PipelineRasterizationStateCreateInfo& setRasterizerDiscardEnable( Bool32 rasterizerDiscardEnable_ )
    {
      rasterizerDiscardEnable = rasterizerDiscardEnable_;
      return *this;
    }

    PipelineRasterizationStateCreateInfo& setPolygonMode( PolygonMode polygonMode_ )
    {
      polygonMode = polygonMode_;
      return *this;
    }

    PipelineRasterizationStateCreateInfo& setCullMode( CullModeFlags cullMode_ )
    {
      cullMode = cullMode_;
      return *this;
    }

    PipelineRasterizationStateCreateInfo& setFrontFace( FrontFace frontFace_ )
    {
      frontFace = frontFace_;
      return *this;
    }

    PipelineRasterizationStateCreateInfo& setDepthBiasEnable( Bool32 depthBiasEnable_ )
    {
      depthBiasEnable = depthBiasEnable_;
      return *this;
    }

    PipelineRasterizationStateCreateInfo& setDepthBiasConstantFactor( float depthBiasConstantFactor_ )
    {
      depthBiasConstantFactor = depthBiasConstantFactor_;
      return *this;
    }

    PipelineRasterizationStateCreateInfo& setDepthBiasClamp( float depthBiasClamp_ )
    {
      depthBiasClamp = depthBiasClamp_;
      return *this;
    }

    PipelineRasterizationStateCreateInfo& setDepthBiasSlopeFactor( float depthBiasSlopeFactor_ )
    {
      depthBiasSlopeFactor = depthBiasSlopeFactor_;
      return *this;
    }

    PipelineRasterizationStateCreateInfo& setLineWidth( float lineWidth_ )
    {
      lineWidth = lineWidth_;
      return *this;
    }

    operator VkPipelineRasterizationStateCreateInfo const&() const
    {
      return *reinterpret_cast<const VkPipelineRasterizationStateCreateInfo*>(this);
    }

    operator VkPipelineRasterizationStateCreateInfo &()
    {
      return *reinterpret_cast<VkPipelineRasterizationStateCreateInfo*>(this);
    }

    bool operator==( PipelineRasterizationStateCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( depthClampEnable == rhs.depthClampEnable )
          && ( rasterizerDiscardEnable == rhs.rasterizerDiscardEnable )
          && ( polygonMode == rhs.polygonMode )
          && ( cullMode == rhs.cullMode )
          && ( frontFace == rhs.frontFace )
          && ( depthBiasEnable == rhs.depthBiasEnable )
          && ( depthBiasConstantFactor == rhs.depthBiasConstantFactor )
          && ( depthBiasClamp == rhs.depthBiasClamp )
          && ( depthBiasSlopeFactor == rhs.depthBiasSlopeFactor )
          && ( lineWidth == rhs.lineWidth );
    }

    bool operator!=( PipelineRasterizationStateCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineRasterizationStateCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineRasterizationStateCreateFlags flags;
    Bool32 depthClampEnable;
    Bool32 rasterizerDiscardEnable;
    PolygonMode polygonMode;
    CullModeFlags cullMode;
    FrontFace frontFace;
    Bool32 depthBiasEnable;
    float depthBiasConstantFactor;
    float depthBiasClamp;
    float depthBiasSlopeFactor;
    float lineWidth;
  };
  static_assert( sizeof( PipelineRasterizationStateCreateInfo ) == sizeof( VkPipelineRasterizationStateCreateInfo ), "struct and wrapper have different size!" );

  struct PipelineDepthStencilStateCreateInfo
  {
    PipelineDepthStencilStateCreateInfo( PipelineDepthStencilStateCreateFlags flags_ = PipelineDepthStencilStateCreateFlags(),
                                         Bool32 depthTestEnable_ = 0,
                                         Bool32 depthWriteEnable_ = 0,
                                         CompareOp depthCompareOp_ = CompareOp::eNever,
                                         Bool32 depthBoundsTestEnable_ = 0,
                                         Bool32 stencilTestEnable_ = 0,
                                         StencilOpState front_ = StencilOpState(),
                                         StencilOpState back_ = StencilOpState(),
                                         float minDepthBounds_ = 0,
                                         float maxDepthBounds_ = 0 )
      : flags( flags_ )
      , depthTestEnable( depthTestEnable_ )
      , depthWriteEnable( depthWriteEnable_ )
      , depthCompareOp( depthCompareOp_ )
      , depthBoundsTestEnable( depthBoundsTestEnable_ )
      , stencilTestEnable( stencilTestEnable_ )
      , front( front_ )
      , back( back_ )
      , minDepthBounds( minDepthBounds_ )
      , maxDepthBounds( maxDepthBounds_ )
    {
    }

    PipelineDepthStencilStateCreateInfo( VkPipelineDepthStencilStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineDepthStencilStateCreateInfo ) );
    }

    PipelineDepthStencilStateCreateInfo& operator=( VkPipelineDepthStencilStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineDepthStencilStateCreateInfo ) );
      return *this;
    }
    PipelineDepthStencilStateCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineDepthStencilStateCreateInfo& setFlags( PipelineDepthStencilStateCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineDepthStencilStateCreateInfo& setDepthTestEnable( Bool32 depthTestEnable_ )
    {
      depthTestEnable = depthTestEnable_;
      return *this;
    }

    PipelineDepthStencilStateCreateInfo& setDepthWriteEnable( Bool32 depthWriteEnable_ )
    {
      depthWriteEnable = depthWriteEnable_;
      return *this;
    }

    PipelineDepthStencilStateCreateInfo& setDepthCompareOp( CompareOp depthCompareOp_ )
    {
      depthCompareOp = depthCompareOp_;
      return *this;
    }

    PipelineDepthStencilStateCreateInfo& setDepthBoundsTestEnable( Bool32 depthBoundsTestEnable_ )
    {
      depthBoundsTestEnable = depthBoundsTestEnable_;
      return *this;
    }

    PipelineDepthStencilStateCreateInfo& setStencilTestEnable( Bool32 stencilTestEnable_ )
    {
      stencilTestEnable = stencilTestEnable_;
      return *this;
    }

    PipelineDepthStencilStateCreateInfo& setFront( StencilOpState front_ )
    {
      front = front_;
      return *this;
    }

    PipelineDepthStencilStateCreateInfo& setBack( StencilOpState back_ )
    {
      back = back_;
      return *this;
    }

    PipelineDepthStencilStateCreateInfo& setMinDepthBounds( float minDepthBounds_ )
    {
      minDepthBounds = minDepthBounds_;
      return *this;
    }

    PipelineDepthStencilStateCreateInfo& setMaxDepthBounds( float maxDepthBounds_ )
    {
      maxDepthBounds = maxDepthBounds_;
      return *this;
    }

    operator VkPipelineDepthStencilStateCreateInfo const&() const
    {
      return *reinterpret_cast<const VkPipelineDepthStencilStateCreateInfo*>(this);
    }

    operator VkPipelineDepthStencilStateCreateInfo &()
    {
      return *reinterpret_cast<VkPipelineDepthStencilStateCreateInfo*>(this);
    }

    bool operator==( PipelineDepthStencilStateCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( depthTestEnable == rhs.depthTestEnable )
          && ( depthWriteEnable == rhs.depthWriteEnable )
          && ( depthCompareOp == rhs.depthCompareOp )
          && ( depthBoundsTestEnable == rhs.depthBoundsTestEnable )
          && ( stencilTestEnable == rhs.stencilTestEnable )
          && ( front == rhs.front )
          && ( back == rhs.back )
          && ( minDepthBounds == rhs.minDepthBounds )
          && ( maxDepthBounds == rhs.maxDepthBounds );
    }

    bool operator!=( PipelineDepthStencilStateCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineDepthStencilStateCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineDepthStencilStateCreateFlags flags;
    Bool32 depthTestEnable;
    Bool32 depthWriteEnable;
    CompareOp depthCompareOp;
    Bool32 depthBoundsTestEnable;
    Bool32 stencilTestEnable;
    StencilOpState front;
    StencilOpState back;
    float minDepthBounds;
    float maxDepthBounds;
  };
  static_assert( sizeof( PipelineDepthStencilStateCreateInfo ) == sizeof( VkPipelineDepthStencilStateCreateInfo ), "struct and wrapper have different size!" );

  struct PipelineCacheCreateInfo
  {
    PipelineCacheCreateInfo( PipelineCacheCreateFlags flags_ = PipelineCacheCreateFlags(),
                             size_t initialDataSize_ = 0,
                             const void* pInitialData_ = nullptr )
      : flags( flags_ )
      , initialDataSize( initialDataSize_ )
      , pInitialData( pInitialData_ )
    {
    }

    PipelineCacheCreateInfo( VkPipelineCacheCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineCacheCreateInfo ) );
    }

    PipelineCacheCreateInfo& operator=( VkPipelineCacheCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineCacheCreateInfo ) );
      return *this;
    }
    PipelineCacheCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineCacheCreateInfo& setFlags( PipelineCacheCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineCacheCreateInfo& setInitialDataSize( size_t initialDataSize_ )
    {
      initialDataSize = initialDataSize_;
      return *this;
    }

    PipelineCacheCreateInfo& setPInitialData( const void* pInitialData_ )
    {
      pInitialData = pInitialData_;
      return *this;
    }

    operator VkPipelineCacheCreateInfo const&() const
    {
      return *reinterpret_cast<const VkPipelineCacheCreateInfo*>(this);
    }

    operator VkPipelineCacheCreateInfo &()
    {
      return *reinterpret_cast<VkPipelineCacheCreateInfo*>(this);
    }

    bool operator==( PipelineCacheCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( initialDataSize == rhs.initialDataSize )
          && ( pInitialData == rhs.pInitialData );
    }

    bool operator!=( PipelineCacheCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineCacheCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineCacheCreateFlags flags;
    size_t initialDataSize;
    const void* pInitialData;
  };
  static_assert( sizeof( PipelineCacheCreateInfo ) == sizeof( VkPipelineCacheCreateInfo ), "struct and wrapper have different size!" );

  struct CommandBufferAllocateInfo
  {
    CommandBufferAllocateInfo( CommandPool commandPool_ = CommandPool(),
                               CommandBufferLevel level_ = CommandBufferLevel::ePrimary,
                               uint32_t commandBufferCount_ = 0 )
      : commandPool( commandPool_ )
      , level( level_ )
      , commandBufferCount( commandBufferCount_ )
    {
    }

    CommandBufferAllocateInfo( VkCommandBufferAllocateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferAllocateInfo ) );
    }

    CommandBufferAllocateInfo& operator=( VkCommandBufferAllocateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferAllocateInfo ) );
      return *this;
    }
    CommandBufferAllocateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CommandBufferAllocateInfo& setCommandPool( CommandPool commandPool_ )
    {
      commandPool = commandPool_;
      return *this;
    }

    CommandBufferAllocateInfo& setLevel( CommandBufferLevel level_ )
    {
      level = level_;
      return *this;
    }

    CommandBufferAllocateInfo& setCommandBufferCount( uint32_t commandBufferCount_ )
    {
      commandBufferCount = commandBufferCount_;
      return *this;
    }

    operator VkCommandBufferAllocateInfo const&() const
    {
      return *reinterpret_cast<const VkCommandBufferAllocateInfo*>(this);
    }

    operator VkCommandBufferAllocateInfo &()
    {
      return *reinterpret_cast<VkCommandBufferAllocateInfo*>(this);
    }

    bool operator==( CommandBufferAllocateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( commandPool == rhs.commandPool )
          && ( level == rhs.level )
          && ( commandBufferCount == rhs.commandBufferCount );
    }

    bool operator!=( CommandBufferAllocateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCommandBufferAllocateInfo;

  public:
    const void* pNext = nullptr;
    CommandPool commandPool;
    CommandBufferLevel level;
    uint32_t commandBufferCount;
  };
  static_assert( sizeof( CommandBufferAllocateInfo ) == sizeof( VkCommandBufferAllocateInfo ), "struct and wrapper have different size!" );

  struct RenderPassBeginInfo
  {
    RenderPassBeginInfo( RenderPass renderPass_ = RenderPass(),
                         Framebuffer framebuffer_ = Framebuffer(),
                         Rect2D renderArea_ = Rect2D(),
                         uint32_t clearValueCount_ = 0,
                         const ClearValue* pClearValues_ = nullptr )
      : renderPass( renderPass_ )
      , framebuffer( framebuffer_ )
      , renderArea( renderArea_ )
      , clearValueCount( clearValueCount_ )
      , pClearValues( pClearValues_ )
    {
    }

    RenderPassBeginInfo( VkRenderPassBeginInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( RenderPassBeginInfo ) );
    }

    RenderPassBeginInfo& operator=( VkRenderPassBeginInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( RenderPassBeginInfo ) );
      return *this;
    }
    RenderPassBeginInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    RenderPassBeginInfo& setRenderPass( RenderPass renderPass_ )
    {
      renderPass = renderPass_;
      return *this;
    }

    RenderPassBeginInfo& setFramebuffer( Framebuffer framebuffer_ )
    {
      framebuffer = framebuffer_;
      return *this;
    }

    RenderPassBeginInfo& setRenderArea( Rect2D renderArea_ )
    {
      renderArea = renderArea_;
      return *this;
    }

    RenderPassBeginInfo& setClearValueCount( uint32_t clearValueCount_ )
    {
      clearValueCount = clearValueCount_;
      return *this;
    }

    RenderPassBeginInfo& setPClearValues( const ClearValue* pClearValues_ )
    {
      pClearValues = pClearValues_;
      return *this;
    }

    operator VkRenderPassBeginInfo const&() const
    {
      return *reinterpret_cast<const VkRenderPassBeginInfo*>(this);
    }

    operator VkRenderPassBeginInfo &()
    {
      return *reinterpret_cast<VkRenderPassBeginInfo*>(this);
    }

    bool operator==( RenderPassBeginInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( renderPass == rhs.renderPass )
          && ( framebuffer == rhs.framebuffer )
          && ( renderArea == rhs.renderArea )
          && ( clearValueCount == rhs.clearValueCount )
          && ( pClearValues == rhs.pClearValues );
    }

    bool operator!=( RenderPassBeginInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eRenderPassBeginInfo;

  public:
    const void* pNext = nullptr;
    RenderPass renderPass;
    Framebuffer framebuffer;
    Rect2D renderArea;
    uint32_t clearValueCount;
    const ClearValue* pClearValues;
  };
  static_assert( sizeof( RenderPassBeginInfo ) == sizeof( VkRenderPassBeginInfo ), "struct and wrapper have different size!" );

  struct EventCreateInfo
  {
    EventCreateInfo( EventCreateFlags flags_ = EventCreateFlags() )
      : flags( flags_ )
    {
    }

    EventCreateInfo( VkEventCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( EventCreateInfo ) );
    }

    EventCreateInfo& operator=( VkEventCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( EventCreateInfo ) );
      return *this;
    }
    EventCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    EventCreateInfo& setFlags( EventCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    operator VkEventCreateInfo const&() const
    {
      return *reinterpret_cast<const VkEventCreateInfo*>(this);
    }

    operator VkEventCreateInfo &()
    {
      return *reinterpret_cast<VkEventCreateInfo*>(this);
    }

    bool operator==( EventCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags );
    }

    bool operator!=( EventCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eEventCreateInfo;

  public:
    const void* pNext = nullptr;
    EventCreateFlags flags;
  };
  static_assert( sizeof( EventCreateInfo ) == sizeof( VkEventCreateInfo ), "struct and wrapper have different size!" );

  struct SemaphoreCreateInfo
  {
    SemaphoreCreateInfo( SemaphoreCreateFlags flags_ = SemaphoreCreateFlags() )
      : flags( flags_ )
    {
    }

    SemaphoreCreateInfo( VkSemaphoreCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SemaphoreCreateInfo ) );
    }

    SemaphoreCreateInfo& operator=( VkSemaphoreCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SemaphoreCreateInfo ) );
      return *this;
    }
    SemaphoreCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    SemaphoreCreateInfo& setFlags( SemaphoreCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    operator VkSemaphoreCreateInfo const&() const
    {
      return *reinterpret_cast<const VkSemaphoreCreateInfo*>(this);
    }

    operator VkSemaphoreCreateInfo &()
    {
      return *reinterpret_cast<VkSemaphoreCreateInfo*>(this);
    }

    bool operator==( SemaphoreCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags );
    }

    bool operator!=( SemaphoreCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eSemaphoreCreateInfo;

  public:
    const void* pNext = nullptr;
    SemaphoreCreateFlags flags;
  };
  static_assert( sizeof( SemaphoreCreateInfo ) == sizeof( VkSemaphoreCreateInfo ), "struct and wrapper have different size!" );

  struct FramebufferCreateInfo
  {
    FramebufferCreateInfo( FramebufferCreateFlags flags_ = FramebufferCreateFlags(),
                           RenderPass renderPass_ = RenderPass(),
                           uint32_t attachmentCount_ = 0,
                           const ImageView* pAttachments_ = nullptr,
                           uint32_t width_ = 0,
                           uint32_t height_ = 0,
                           uint32_t layers_ = 0 )
      : flags( flags_ )
      , renderPass( renderPass_ )
      , attachmentCount( attachmentCount_ )
      , pAttachments( pAttachments_ )
      , width( width_ )
      , height( height_ )
      , layers( layers_ )
    {
    }

    FramebufferCreateInfo( VkFramebufferCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( FramebufferCreateInfo ) );
    }

    FramebufferCreateInfo& operator=( VkFramebufferCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( FramebufferCreateInfo ) );
      return *this;
    }
    FramebufferCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    FramebufferCreateInfo& setFlags( FramebufferCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    FramebufferCreateInfo& setRenderPass( RenderPass renderPass_ )
    {
      renderPass = renderPass_;
      return *this;
    }

    FramebufferCreateInfo& setAttachmentCount( uint32_t attachmentCount_ )
    {
      attachmentCount = attachmentCount_;
      return *this;
    }

    FramebufferCreateInfo& setPAttachments( const ImageView* pAttachments_ )
    {
      pAttachments = pAttachments_;
      return *this;
    }

    FramebufferCreateInfo& setWidth( uint32_t width_ )
    {
      width = width_;
      return *this;
    }

    FramebufferCreateInfo& setHeight( uint32_t height_ )
    {
      height = height_;
      return *this;
    }

    FramebufferCreateInfo& setLayers( uint32_t layers_ )
    {
      layers = layers_;
      return *this;
    }

    operator VkFramebufferCreateInfo const&() const
    {
      return *reinterpret_cast<const VkFramebufferCreateInfo*>(this);
    }

    operator VkFramebufferCreateInfo &()
    {
      return *reinterpret_cast<VkFramebufferCreateInfo*>(this);
    }

    bool operator==( FramebufferCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( renderPass == rhs.renderPass )
          && ( attachmentCount == rhs.attachmentCount )
          && ( pAttachments == rhs.pAttachments )
          && ( width == rhs.width )
          && ( height == rhs.height )
          && ( layers == rhs.layers );
    }

    bool operator!=( FramebufferCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eFramebufferCreateInfo;

  public:
    const void* pNext = nullptr;
    FramebufferCreateFlags flags;
    RenderPass renderPass;
    uint32_t attachmentCount;
    const ImageView* pAttachments;
    uint32_t width;
    uint32_t height;
    uint32_t layers;
  };
  static_assert( sizeof( FramebufferCreateInfo ) == sizeof( VkFramebufferCreateInfo ), "struct and wrapper have different size!" );

  struct DisplayModeCreateInfoKHR
  {
    DisplayModeCreateInfoKHR( DisplayModeCreateFlagsKHR flags_ = DisplayModeCreateFlagsKHR(),
                              DisplayModeParametersKHR parameters_ = DisplayModeParametersKHR() )
      : flags( flags_ )
      , parameters( parameters_ )
    {
    }

    DisplayModeCreateInfoKHR( VkDisplayModeCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( DisplayModeCreateInfoKHR ) );
    }

    DisplayModeCreateInfoKHR& operator=( VkDisplayModeCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( DisplayModeCreateInfoKHR ) );
      return *this;
    }
    DisplayModeCreateInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DisplayModeCreateInfoKHR& setFlags( DisplayModeCreateFlagsKHR flags_ )
    {
      flags = flags_;
      return *this;
    }

    DisplayModeCreateInfoKHR& setParameters( DisplayModeParametersKHR parameters_ )
    {
      parameters = parameters_;
      return *this;
    }

    operator VkDisplayModeCreateInfoKHR const&() const
    {
      return *reinterpret_cast<const VkDisplayModeCreateInfoKHR*>(this);
    }

    operator VkDisplayModeCreateInfoKHR &()
    {
      return *reinterpret_cast<VkDisplayModeCreateInfoKHR*>(this);
    }

    bool operator==( DisplayModeCreateInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( parameters == rhs.parameters );
    }

    bool operator!=( DisplayModeCreateInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDisplayModeCreateInfoKHR;

  public:
    const void* pNext = nullptr;
    DisplayModeCreateFlagsKHR flags;
    DisplayModeParametersKHR parameters;
  };
  static_assert( sizeof( DisplayModeCreateInfoKHR ) == sizeof( VkDisplayModeCreateInfoKHR ), "struct and wrapper have different size!" );

  struct DisplayPresentInfoKHR
  {
    DisplayPresentInfoKHR( Rect2D srcRect_ = Rect2D(),
                           Rect2D dstRect_ = Rect2D(),
                           Bool32 persistent_ = 0 )
      : srcRect( srcRect_ )
      , dstRect( dstRect_ )
      , persistent( persistent_ )
    {
    }

    DisplayPresentInfoKHR( VkDisplayPresentInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( DisplayPresentInfoKHR ) );
    }

    DisplayPresentInfoKHR& operator=( VkDisplayPresentInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( DisplayPresentInfoKHR ) );
      return *this;
    }
    DisplayPresentInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DisplayPresentInfoKHR& setSrcRect( Rect2D srcRect_ )
    {
      srcRect = srcRect_;
      return *this;
    }

    DisplayPresentInfoKHR& setDstRect( Rect2D dstRect_ )
    {
      dstRect = dstRect_;
      return *this;
    }

    DisplayPresentInfoKHR& setPersistent( Bool32 persistent_ )
    {
      persistent = persistent_;
      return *this;
    }

    operator VkDisplayPresentInfoKHR const&() const
    {
      return *reinterpret_cast<const VkDisplayPresentInfoKHR*>(this);
    }

    operator VkDisplayPresentInfoKHR &()
    {
      return *reinterpret_cast<VkDisplayPresentInfoKHR*>(this);
    }

    bool operator==( DisplayPresentInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( srcRect == rhs.srcRect )
          && ( dstRect == rhs.dstRect )
          && ( persistent == rhs.persistent );
    }

    bool operator!=( DisplayPresentInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDisplayPresentInfoKHR;

  public:
    const void* pNext = nullptr;
    Rect2D srcRect;
    Rect2D dstRect;
    Bool32 persistent;
  };
  static_assert( sizeof( DisplayPresentInfoKHR ) == sizeof( VkDisplayPresentInfoKHR ), "struct and wrapper have different size!" );

#ifdef VK_USE_PLATFORM_ANDROID_KHR
  struct AndroidSurfaceCreateInfoKHR
  {
    AndroidSurfaceCreateInfoKHR( AndroidSurfaceCreateFlagsKHR flags_ = AndroidSurfaceCreateFlagsKHR(),
                                 struct ANativeWindow* window_ = nullptr )
      : flags( flags_ )
      , window( window_ )
    {
    }

    AndroidSurfaceCreateInfoKHR( VkAndroidSurfaceCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AndroidSurfaceCreateInfoKHR ) );
    }

    AndroidSurfaceCreateInfoKHR& operator=( VkAndroidSurfaceCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AndroidSurfaceCreateInfoKHR ) );
      return *this;
    }
    AndroidSurfaceCreateInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    AndroidSurfaceCreateInfoKHR& setFlags( AndroidSurfaceCreateFlagsKHR flags_ )
    {
      flags = flags_;
      return *this;
    }

    AndroidSurfaceCreateInfoKHR& setWindow( struct ANativeWindow* window_ )
    {
      window = window_;
      return *this;
    }

    operator VkAndroidSurfaceCreateInfoKHR const&() const
    {
      return *reinterpret_cast<const VkAndroidSurfaceCreateInfoKHR*>(this);
    }

    operator VkAndroidSurfaceCreateInfoKHR &()
    {
      return *reinterpret_cast<VkAndroidSurfaceCreateInfoKHR*>(this);
    }

    bool operator==( AndroidSurfaceCreateInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( window == rhs.window );
    }

    bool operator!=( AndroidSurfaceCreateInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAndroidSurfaceCreateInfoKHR;

  public:
    const void* pNext = nullptr;
    AndroidSurfaceCreateFlagsKHR flags;
    struct ANativeWindow* window;
  };
  static_assert( sizeof( AndroidSurfaceCreateInfoKHR ) == sizeof( VkAndroidSurfaceCreateInfoKHR ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

#ifdef VK_USE_PLATFORM_VI_NN
  struct ViSurfaceCreateInfoNN
  {
    ViSurfaceCreateInfoNN( ViSurfaceCreateFlagsNN flags_ = ViSurfaceCreateFlagsNN(),
                           void* window_ = nullptr )
      : flags( flags_ )
      , window( window_ )
    {
    }

    ViSurfaceCreateInfoNN( VkViSurfaceCreateInfoNN const & rhs )
    {
      memcpy( this, &rhs, sizeof( ViSurfaceCreateInfoNN ) );
    }

    ViSurfaceCreateInfoNN& operator=( VkViSurfaceCreateInfoNN const & rhs )
    {
      memcpy( this, &rhs, sizeof( ViSurfaceCreateInfoNN ) );
      return *this;
    }
    ViSurfaceCreateInfoNN& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ViSurfaceCreateInfoNN& setFlags( ViSurfaceCreateFlagsNN flags_ )
    {
      flags = flags_;
      return *this;
    }

    ViSurfaceCreateInfoNN& setWindow( void* window_ )
    {
      window = window_;
      return *this;
    }

    operator VkViSurfaceCreateInfoNN const&() const
    {
      return *reinterpret_cast<const VkViSurfaceCreateInfoNN*>(this);
    }

    operator VkViSurfaceCreateInfoNN &()
    {
      return *reinterpret_cast<VkViSurfaceCreateInfoNN*>(this);
    }

    bool operator==( ViSurfaceCreateInfoNN const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( window == rhs.window );
    }

    bool operator!=( ViSurfaceCreateInfoNN const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eViSurfaceCreateInfoNN;

  public:
    const void* pNext = nullptr;
    ViSurfaceCreateFlagsNN flags;
    void* window;
  };
  static_assert( sizeof( ViSurfaceCreateInfoNN ) == sizeof( VkViSurfaceCreateInfoNN ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_VI_NN*/

#ifdef VK_USE_PLATFORM_WAYLAND_KHR
  struct WaylandSurfaceCreateInfoKHR
  {
    WaylandSurfaceCreateInfoKHR( WaylandSurfaceCreateFlagsKHR flags_ = WaylandSurfaceCreateFlagsKHR(),
                                 struct wl_display* display_ = nullptr,
                                 struct wl_surface* surface_ = nullptr )
      : flags( flags_ )
      , display( display_ )
      , surface( surface_ )
    {
    }

    WaylandSurfaceCreateInfoKHR( VkWaylandSurfaceCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( WaylandSurfaceCreateInfoKHR ) );
    }

    WaylandSurfaceCreateInfoKHR& operator=( VkWaylandSurfaceCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( WaylandSurfaceCreateInfoKHR ) );
      return *this;
    }
    WaylandSurfaceCreateInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    WaylandSurfaceCreateInfoKHR& setFlags( WaylandSurfaceCreateFlagsKHR flags_ )
    {
      flags = flags_;
      return *this;
    }

    WaylandSurfaceCreateInfoKHR& setDisplay( struct wl_display* display_ )
    {
      display = display_;
      return *this;
    }

    WaylandSurfaceCreateInfoKHR& setSurface( struct wl_surface* surface_ )
    {
      surface = surface_;
      return *this;
    }

    operator VkWaylandSurfaceCreateInfoKHR const&() const
    {
      return *reinterpret_cast<const VkWaylandSurfaceCreateInfoKHR*>(this);
    }

    operator VkWaylandSurfaceCreateInfoKHR &()
    {
      return *reinterpret_cast<VkWaylandSurfaceCreateInfoKHR*>(this);
    }

    bool operator==( WaylandSurfaceCreateInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( display == rhs.display )
          && ( surface == rhs.surface );
    }

    bool operator!=( WaylandSurfaceCreateInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eWaylandSurfaceCreateInfoKHR;

  public:
    const void* pNext = nullptr;
    WaylandSurfaceCreateFlagsKHR flags;
    struct wl_display* display;
    struct wl_surface* surface;
  };
  static_assert( sizeof( WaylandSurfaceCreateInfoKHR ) == sizeof( VkWaylandSurfaceCreateInfoKHR ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_WAYLAND_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct Win32SurfaceCreateInfoKHR
  {
    Win32SurfaceCreateInfoKHR( Win32SurfaceCreateFlagsKHR flags_ = Win32SurfaceCreateFlagsKHR(),
                               HINSTANCE hinstance_ = 0,
                               HWND hwnd_ = 0 )
      : flags( flags_ )
      , hinstance( hinstance_ )
      , hwnd( hwnd_ )
    {
    }

    Win32SurfaceCreateInfoKHR( VkWin32SurfaceCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( Win32SurfaceCreateInfoKHR ) );
    }

    Win32SurfaceCreateInfoKHR& operator=( VkWin32SurfaceCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( Win32SurfaceCreateInfoKHR ) );
      return *this;
    }
    Win32SurfaceCreateInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    Win32SurfaceCreateInfoKHR& setFlags( Win32SurfaceCreateFlagsKHR flags_ )
    {
      flags = flags_;
      return *this;
    }

    Win32SurfaceCreateInfoKHR& setHinstance( HINSTANCE hinstance_ )
    {
      hinstance = hinstance_;
      return *this;
    }

    Win32SurfaceCreateInfoKHR& setHwnd( HWND hwnd_ )
    {
      hwnd = hwnd_;
      return *this;
    }

    operator VkWin32SurfaceCreateInfoKHR const&() const
    {
      return *reinterpret_cast<const VkWin32SurfaceCreateInfoKHR*>(this);
    }

    operator VkWin32SurfaceCreateInfoKHR &()
    {
      return *reinterpret_cast<VkWin32SurfaceCreateInfoKHR*>(this);
    }

    bool operator==( Win32SurfaceCreateInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( hinstance == rhs.hinstance )
          && ( hwnd == rhs.hwnd );
    }

    bool operator!=( Win32SurfaceCreateInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eWin32SurfaceCreateInfoKHR;

  public:
    const void* pNext = nullptr;
    Win32SurfaceCreateFlagsKHR flags;
    HINSTANCE hinstance;
    HWND hwnd;
  };
  static_assert( sizeof( Win32SurfaceCreateInfoKHR ) == sizeof( VkWin32SurfaceCreateInfoKHR ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_XLIB_KHR
  struct XlibSurfaceCreateInfoKHR
  {
    XlibSurfaceCreateInfoKHR( XlibSurfaceCreateFlagsKHR flags_ = XlibSurfaceCreateFlagsKHR(),
                              Display* dpy_ = nullptr,
                              Window window_ = 0 )
      : flags( flags_ )
      , dpy( dpy_ )
      , window( window_ )
    {
    }

    XlibSurfaceCreateInfoKHR( VkXlibSurfaceCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( XlibSurfaceCreateInfoKHR ) );
    }

    XlibSurfaceCreateInfoKHR& operator=( VkXlibSurfaceCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( XlibSurfaceCreateInfoKHR ) );
      return *this;
    }
    XlibSurfaceCreateInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    XlibSurfaceCreateInfoKHR& setFlags( XlibSurfaceCreateFlagsKHR flags_ )
    {
      flags = flags_;
      return *this;
    }

    XlibSurfaceCreateInfoKHR& setDpy( Display* dpy_ )
    {
      dpy = dpy_;
      return *this;
    }

    XlibSurfaceCreateInfoKHR& setWindow( Window window_ )
    {
      window = window_;
      return *this;
    }

    operator VkXlibSurfaceCreateInfoKHR const&() const
    {
      return *reinterpret_cast<const VkXlibSurfaceCreateInfoKHR*>(this);
    }

    operator VkXlibSurfaceCreateInfoKHR &()
    {
      return *reinterpret_cast<VkXlibSurfaceCreateInfoKHR*>(this);
    }

    bool operator==( XlibSurfaceCreateInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( dpy == rhs.dpy )
          && ( window == rhs.window );
    }

    bool operator!=( XlibSurfaceCreateInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eXlibSurfaceCreateInfoKHR;

  public:
    const void* pNext = nullptr;
    XlibSurfaceCreateFlagsKHR flags;
    Display* dpy;
    Window window;
  };
  static_assert( sizeof( XlibSurfaceCreateInfoKHR ) == sizeof( VkXlibSurfaceCreateInfoKHR ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_XLIB_KHR*/

#ifdef VK_USE_PLATFORM_XCB_KHR
  struct XcbSurfaceCreateInfoKHR
  {
    XcbSurfaceCreateInfoKHR( XcbSurfaceCreateFlagsKHR flags_ = XcbSurfaceCreateFlagsKHR(),
                             xcb_connection_t* connection_ = nullptr,
                             xcb_window_t window_ = 0 )
      : flags( flags_ )
      , connection( connection_ )
      , window( window_ )
    {
    }

    XcbSurfaceCreateInfoKHR( VkXcbSurfaceCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( XcbSurfaceCreateInfoKHR ) );
    }

    XcbSurfaceCreateInfoKHR& operator=( VkXcbSurfaceCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( XcbSurfaceCreateInfoKHR ) );
      return *this;
    }
    XcbSurfaceCreateInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    XcbSurfaceCreateInfoKHR& setFlags( XcbSurfaceCreateFlagsKHR flags_ )
    {
      flags = flags_;
      return *this;
    }

    XcbSurfaceCreateInfoKHR& setConnection( xcb_connection_t* connection_ )
    {
      connection = connection_;
      return *this;
    }

    XcbSurfaceCreateInfoKHR& setWindow( xcb_window_t window_ )
    {
      window = window_;
      return *this;
    }

    operator VkXcbSurfaceCreateInfoKHR const&() const
    {
      return *reinterpret_cast<const VkXcbSurfaceCreateInfoKHR*>(this);
    }

    operator VkXcbSurfaceCreateInfoKHR &()
    {
      return *reinterpret_cast<VkXcbSurfaceCreateInfoKHR*>(this);
    }

    bool operator==( XcbSurfaceCreateInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( connection == rhs.connection )
          && ( window == rhs.window );
    }

    bool operator!=( XcbSurfaceCreateInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eXcbSurfaceCreateInfoKHR;

  public:
    const void* pNext = nullptr;
    XcbSurfaceCreateFlagsKHR flags;
    xcb_connection_t* connection;
    xcb_window_t window;
  };
  static_assert( sizeof( XcbSurfaceCreateInfoKHR ) == sizeof( VkXcbSurfaceCreateInfoKHR ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_XCB_KHR*/

#ifdef VK_USE_PLATFORM_FUCHSIA
  struct ImagePipeSurfaceCreateInfoFUCHSIA
  {
    ImagePipeSurfaceCreateInfoFUCHSIA( ImagePipeSurfaceCreateFlagsFUCHSIA flags_ = ImagePipeSurfaceCreateFlagsFUCHSIA(),
                                       zx_handle_t imagePipeHandle_ = 0 )
      : flags( flags_ )
      , imagePipeHandle( imagePipeHandle_ )
    {
    }

    ImagePipeSurfaceCreateInfoFUCHSIA( VkImagePipeSurfaceCreateInfoFUCHSIA const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImagePipeSurfaceCreateInfoFUCHSIA ) );
    }

    ImagePipeSurfaceCreateInfoFUCHSIA& operator=( VkImagePipeSurfaceCreateInfoFUCHSIA const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImagePipeSurfaceCreateInfoFUCHSIA ) );
      return *this;
    }
    ImagePipeSurfaceCreateInfoFUCHSIA& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImagePipeSurfaceCreateInfoFUCHSIA& setFlags( ImagePipeSurfaceCreateFlagsFUCHSIA flags_ )
    {
      flags = flags_;
      return *this;
    }

    ImagePipeSurfaceCreateInfoFUCHSIA& setImagePipeHandle( zx_handle_t imagePipeHandle_ )
    {
      imagePipeHandle = imagePipeHandle_;
      return *this;
    }

    operator VkImagePipeSurfaceCreateInfoFUCHSIA const&() const
    {
      return *reinterpret_cast<const VkImagePipeSurfaceCreateInfoFUCHSIA*>(this);
    }

    operator VkImagePipeSurfaceCreateInfoFUCHSIA &()
    {
      return *reinterpret_cast<VkImagePipeSurfaceCreateInfoFUCHSIA*>(this);
    }

    bool operator==( ImagePipeSurfaceCreateInfoFUCHSIA const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( imagePipeHandle == rhs.imagePipeHandle );
    }

    bool operator!=( ImagePipeSurfaceCreateInfoFUCHSIA const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImagepipeSurfaceCreateInfoFUCHSIA;

  public:
    const void* pNext = nullptr;
    ImagePipeSurfaceCreateFlagsFUCHSIA flags;
    zx_handle_t imagePipeHandle;
  };
  static_assert( sizeof( ImagePipeSurfaceCreateInfoFUCHSIA ) == sizeof( VkImagePipeSurfaceCreateInfoFUCHSIA ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_FUCHSIA*/

  struct DebugMarkerMarkerInfoEXT
  {
    DebugMarkerMarkerInfoEXT( const char* pMarkerName_ = nullptr,
                              std::array<float,4> const& color_ = { { 0, 0, 0, 0 } } )
      : pMarkerName( pMarkerName_ )
    {
      memcpy( &color, color_.data(), 4 * sizeof( float ) );
    }

    DebugMarkerMarkerInfoEXT( VkDebugMarkerMarkerInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugMarkerMarkerInfoEXT ) );
    }

    DebugMarkerMarkerInfoEXT& operator=( VkDebugMarkerMarkerInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugMarkerMarkerInfoEXT ) );
      return *this;
    }
    DebugMarkerMarkerInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugMarkerMarkerInfoEXT& setPMarkerName( const char* pMarkerName_ )
    {
      pMarkerName = pMarkerName_;
      return *this;
    }

    DebugMarkerMarkerInfoEXT& setColor( std::array<float,4> color_ )
    {
      memcpy( &color, color_.data(), 4 * sizeof( float ) );
      return *this;
    }

    operator VkDebugMarkerMarkerInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDebugMarkerMarkerInfoEXT*>(this);
    }

    operator VkDebugMarkerMarkerInfoEXT &()
    {
      return *reinterpret_cast<VkDebugMarkerMarkerInfoEXT*>(this);
    }

    bool operator==( DebugMarkerMarkerInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( pMarkerName == rhs.pMarkerName )
          && ( memcmp( color, rhs.color, 4 * sizeof( float ) ) == 0 );
    }

    bool operator!=( DebugMarkerMarkerInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugMarkerMarkerInfoEXT;

  public:
    const void* pNext = nullptr;
    const char* pMarkerName;
    float color[4];
  };
  static_assert( sizeof( DebugMarkerMarkerInfoEXT ) == sizeof( VkDebugMarkerMarkerInfoEXT ), "struct and wrapper have different size!" );

  struct DedicatedAllocationImageCreateInfoNV
  {
    DedicatedAllocationImageCreateInfoNV( Bool32 dedicatedAllocation_ = 0 )
      : dedicatedAllocation( dedicatedAllocation_ )
    {
    }

    DedicatedAllocationImageCreateInfoNV( VkDedicatedAllocationImageCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DedicatedAllocationImageCreateInfoNV ) );
    }

    DedicatedAllocationImageCreateInfoNV& operator=( VkDedicatedAllocationImageCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DedicatedAllocationImageCreateInfoNV ) );
      return *this;
    }
    DedicatedAllocationImageCreateInfoNV& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DedicatedAllocationImageCreateInfoNV& setDedicatedAllocation( Bool32 dedicatedAllocation_ )
    {
      dedicatedAllocation = dedicatedAllocation_;
      return *this;
    }

    operator VkDedicatedAllocationImageCreateInfoNV const&() const
    {
      return *reinterpret_cast<const VkDedicatedAllocationImageCreateInfoNV*>(this);
    }

    operator VkDedicatedAllocationImageCreateInfoNV &()
    {
      return *reinterpret_cast<VkDedicatedAllocationImageCreateInfoNV*>(this);
    }

    bool operator==( DedicatedAllocationImageCreateInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( dedicatedAllocation == rhs.dedicatedAllocation );
    }

    bool operator!=( DedicatedAllocationImageCreateInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDedicatedAllocationImageCreateInfoNV;

  public:
    const void* pNext = nullptr;
    Bool32 dedicatedAllocation;
  };
  static_assert( sizeof( DedicatedAllocationImageCreateInfoNV ) == sizeof( VkDedicatedAllocationImageCreateInfoNV ), "struct and wrapper have different size!" );

  struct DedicatedAllocationBufferCreateInfoNV
  {
    DedicatedAllocationBufferCreateInfoNV( Bool32 dedicatedAllocation_ = 0 )
      : dedicatedAllocation( dedicatedAllocation_ )
    {
    }

    DedicatedAllocationBufferCreateInfoNV( VkDedicatedAllocationBufferCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DedicatedAllocationBufferCreateInfoNV ) );
    }

    DedicatedAllocationBufferCreateInfoNV& operator=( VkDedicatedAllocationBufferCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DedicatedAllocationBufferCreateInfoNV ) );
      return *this;
    }
    DedicatedAllocationBufferCreateInfoNV& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DedicatedAllocationBufferCreateInfoNV& setDedicatedAllocation( Bool32 dedicatedAllocation_ )
    {
      dedicatedAllocation = dedicatedAllocation_;
      return *this;
    }

    operator VkDedicatedAllocationBufferCreateInfoNV const&() const
    {
      return *reinterpret_cast<const VkDedicatedAllocationBufferCreateInfoNV*>(this);
    }

    operator VkDedicatedAllocationBufferCreateInfoNV &()
    {
      return *reinterpret_cast<VkDedicatedAllocationBufferCreateInfoNV*>(this);
    }

    bool operator==( DedicatedAllocationBufferCreateInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( dedicatedAllocation == rhs.dedicatedAllocation );
    }

    bool operator!=( DedicatedAllocationBufferCreateInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDedicatedAllocationBufferCreateInfoNV;

  public:
    const void* pNext = nullptr;
    Bool32 dedicatedAllocation;
  };
  static_assert( sizeof( DedicatedAllocationBufferCreateInfoNV ) == sizeof( VkDedicatedAllocationBufferCreateInfoNV ), "struct and wrapper have different size!" );

  struct DedicatedAllocationMemoryAllocateInfoNV
  {
    DedicatedAllocationMemoryAllocateInfoNV( Image image_ = Image(),
                                             Buffer buffer_ = Buffer() )
      : image( image_ )
      , buffer( buffer_ )
    {
    }

    DedicatedAllocationMemoryAllocateInfoNV( VkDedicatedAllocationMemoryAllocateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DedicatedAllocationMemoryAllocateInfoNV ) );
    }

    DedicatedAllocationMemoryAllocateInfoNV& operator=( VkDedicatedAllocationMemoryAllocateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DedicatedAllocationMemoryAllocateInfoNV ) );
      return *this;
    }
    DedicatedAllocationMemoryAllocateInfoNV& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DedicatedAllocationMemoryAllocateInfoNV& setImage( Image image_ )
    {
      image = image_;
      return *this;
    }

    DedicatedAllocationMemoryAllocateInfoNV& setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    operator VkDedicatedAllocationMemoryAllocateInfoNV const&() const
    {
      return *reinterpret_cast<const VkDedicatedAllocationMemoryAllocateInfoNV*>(this);
    }

    operator VkDedicatedAllocationMemoryAllocateInfoNV &()
    {
      return *reinterpret_cast<VkDedicatedAllocationMemoryAllocateInfoNV*>(this);
    }

    bool operator==( DedicatedAllocationMemoryAllocateInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( image == rhs.image )
          && ( buffer == rhs.buffer );
    }

    bool operator!=( DedicatedAllocationMemoryAllocateInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDedicatedAllocationMemoryAllocateInfoNV;

  public:
    const void* pNext = nullptr;
    Image image;
    Buffer buffer;
  };
  static_assert( sizeof( DedicatedAllocationMemoryAllocateInfoNV ) == sizeof( VkDedicatedAllocationMemoryAllocateInfoNV ), "struct and wrapper have different size!" );

#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct ExportMemoryWin32HandleInfoNV
  {
    ExportMemoryWin32HandleInfoNV( const SECURITY_ATTRIBUTES* pAttributes_ = nullptr,
                                   DWORD dwAccess_ = 0 )
      : pAttributes( pAttributes_ )
      , dwAccess( dwAccess_ )
    {
    }

    ExportMemoryWin32HandleInfoNV( VkExportMemoryWin32HandleInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( ExportMemoryWin32HandleInfoNV ) );
    }

    ExportMemoryWin32HandleInfoNV& operator=( VkExportMemoryWin32HandleInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( ExportMemoryWin32HandleInfoNV ) );
      return *this;
    }
    ExportMemoryWin32HandleInfoNV& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ExportMemoryWin32HandleInfoNV& setPAttributes( const SECURITY_ATTRIBUTES* pAttributes_ )
    {
      pAttributes = pAttributes_;
      return *this;
    }

    ExportMemoryWin32HandleInfoNV& setDwAccess( DWORD dwAccess_ )
    {
      dwAccess = dwAccess_;
      return *this;
    }

    operator VkExportMemoryWin32HandleInfoNV const&() const
    {
      return *reinterpret_cast<const VkExportMemoryWin32HandleInfoNV*>(this);
    }

    operator VkExportMemoryWin32HandleInfoNV &()
    {
      return *reinterpret_cast<VkExportMemoryWin32HandleInfoNV*>(this);
    }

    bool operator==( ExportMemoryWin32HandleInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( pAttributes == rhs.pAttributes )
          && ( dwAccess == rhs.dwAccess );
    }

    bool operator!=( ExportMemoryWin32HandleInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eExportMemoryWin32HandleInfoNV;

  public:
    const void* pNext = nullptr;
    const SECURITY_ATTRIBUTES* pAttributes;
    DWORD dwAccess;
  };
  static_assert( sizeof( ExportMemoryWin32HandleInfoNV ) == sizeof( VkExportMemoryWin32HandleInfoNV ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct Win32KeyedMutexAcquireReleaseInfoNV
  {
    Win32KeyedMutexAcquireReleaseInfoNV( uint32_t acquireCount_ = 0,
                                         const DeviceMemory* pAcquireSyncs_ = nullptr,
                                         const uint64_t* pAcquireKeys_ = nullptr,
                                         const uint32_t* pAcquireTimeoutMilliseconds_ = nullptr,
                                         uint32_t releaseCount_ = 0,
                                         const DeviceMemory* pReleaseSyncs_ = nullptr,
                                         const uint64_t* pReleaseKeys_ = nullptr )
      : acquireCount( acquireCount_ )
      , pAcquireSyncs( pAcquireSyncs_ )
      , pAcquireKeys( pAcquireKeys_ )
      , pAcquireTimeoutMilliseconds( pAcquireTimeoutMilliseconds_ )
      , releaseCount( releaseCount_ )
      , pReleaseSyncs( pReleaseSyncs_ )
      , pReleaseKeys( pReleaseKeys_ )
    {
    }

    Win32KeyedMutexAcquireReleaseInfoNV( VkWin32KeyedMutexAcquireReleaseInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( Win32KeyedMutexAcquireReleaseInfoNV ) );
    }

    Win32KeyedMutexAcquireReleaseInfoNV& operator=( VkWin32KeyedMutexAcquireReleaseInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( Win32KeyedMutexAcquireReleaseInfoNV ) );
      return *this;
    }
    Win32KeyedMutexAcquireReleaseInfoNV& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoNV& setAcquireCount( uint32_t acquireCount_ )
    {
      acquireCount = acquireCount_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoNV& setPAcquireSyncs( const DeviceMemory* pAcquireSyncs_ )
    {
      pAcquireSyncs = pAcquireSyncs_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoNV& setPAcquireKeys( const uint64_t* pAcquireKeys_ )
    {
      pAcquireKeys = pAcquireKeys_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoNV& setPAcquireTimeoutMilliseconds( const uint32_t* pAcquireTimeoutMilliseconds_ )
    {
      pAcquireTimeoutMilliseconds = pAcquireTimeoutMilliseconds_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoNV& setReleaseCount( uint32_t releaseCount_ )
    {
      releaseCount = releaseCount_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoNV& setPReleaseSyncs( const DeviceMemory* pReleaseSyncs_ )
    {
      pReleaseSyncs = pReleaseSyncs_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoNV& setPReleaseKeys( const uint64_t* pReleaseKeys_ )
    {
      pReleaseKeys = pReleaseKeys_;
      return *this;
    }

    operator VkWin32KeyedMutexAcquireReleaseInfoNV const&() const
    {
      return *reinterpret_cast<const VkWin32KeyedMutexAcquireReleaseInfoNV*>(this);
    }

    operator VkWin32KeyedMutexAcquireReleaseInfoNV &()
    {
      return *reinterpret_cast<VkWin32KeyedMutexAcquireReleaseInfoNV*>(this);
    }

    bool operator==( Win32KeyedMutexAcquireReleaseInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( acquireCount == rhs.acquireCount )
          && ( pAcquireSyncs == rhs.pAcquireSyncs )
          && ( pAcquireKeys == rhs.pAcquireKeys )
          && ( pAcquireTimeoutMilliseconds == rhs.pAcquireTimeoutMilliseconds )
          && ( releaseCount == rhs.releaseCount )
          && ( pReleaseSyncs == rhs.pReleaseSyncs )
          && ( pReleaseKeys == rhs.pReleaseKeys );
    }

    bool operator!=( Win32KeyedMutexAcquireReleaseInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eWin32KeyedMutexAcquireReleaseInfoNV;

  public:
    const void* pNext = nullptr;
    uint32_t acquireCount;
    const DeviceMemory* pAcquireSyncs;
    const uint64_t* pAcquireKeys;
    const uint32_t* pAcquireTimeoutMilliseconds;
    uint32_t releaseCount;
    const DeviceMemory* pReleaseSyncs;
    const uint64_t* pReleaseKeys;
  };
  static_assert( sizeof( Win32KeyedMutexAcquireReleaseInfoNV ) == sizeof( VkWin32KeyedMutexAcquireReleaseInfoNV ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

  struct DeviceGeneratedCommandsFeaturesNVX
  {
    DeviceGeneratedCommandsFeaturesNVX( Bool32 computeBindingPointSupport_ = 0 )
      : computeBindingPointSupport( computeBindingPointSupport_ )
    {
    }

    DeviceGeneratedCommandsFeaturesNVX( VkDeviceGeneratedCommandsFeaturesNVX const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceGeneratedCommandsFeaturesNVX ) );
    }

    DeviceGeneratedCommandsFeaturesNVX& operator=( VkDeviceGeneratedCommandsFeaturesNVX const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceGeneratedCommandsFeaturesNVX ) );
      return *this;
    }
    DeviceGeneratedCommandsFeaturesNVX& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DeviceGeneratedCommandsFeaturesNVX& setComputeBindingPointSupport( Bool32 computeBindingPointSupport_ )
    {
      computeBindingPointSupport = computeBindingPointSupport_;
      return *this;
    }

    operator VkDeviceGeneratedCommandsFeaturesNVX const&() const
    {
      return *reinterpret_cast<const VkDeviceGeneratedCommandsFeaturesNVX*>(this);
    }

    operator VkDeviceGeneratedCommandsFeaturesNVX &()
    {
      return *reinterpret_cast<VkDeviceGeneratedCommandsFeaturesNVX*>(this);
    }

    bool operator==( DeviceGeneratedCommandsFeaturesNVX const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( computeBindingPointSupport == rhs.computeBindingPointSupport );
    }

    bool operator!=( DeviceGeneratedCommandsFeaturesNVX const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDeviceGeneratedCommandsFeaturesNVX;

  public:
    const void* pNext = nullptr;
    Bool32 computeBindingPointSupport;
  };
  static_assert( sizeof( DeviceGeneratedCommandsFeaturesNVX ) == sizeof( VkDeviceGeneratedCommandsFeaturesNVX ), "struct and wrapper have different size!" );

  struct DeviceGeneratedCommandsLimitsNVX
  {
    DeviceGeneratedCommandsLimitsNVX( uint32_t maxIndirectCommandsLayoutTokenCount_ = 0,
                                      uint32_t maxObjectEntryCounts_ = 0,
                                      uint32_t minSequenceCountBufferOffsetAlignment_ = 0,
                                      uint32_t minSequenceIndexBufferOffsetAlignment_ = 0,
                                      uint32_t minCommandsTokenBufferOffsetAlignment_ = 0 )
      : maxIndirectCommandsLayoutTokenCount( maxIndirectCommandsLayoutTokenCount_ )
      , maxObjectEntryCounts( maxObjectEntryCounts_ )
      , minSequenceCountBufferOffsetAlignment( minSequenceCountBufferOffsetAlignment_ )
      , minSequenceIndexBufferOffsetAlignment( minSequenceIndexBufferOffsetAlignment_ )
      , minCommandsTokenBufferOffsetAlignment( minCommandsTokenBufferOffsetAlignment_ )
    {
    }

    DeviceGeneratedCommandsLimitsNVX( VkDeviceGeneratedCommandsLimitsNVX const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceGeneratedCommandsLimitsNVX ) );
    }

    DeviceGeneratedCommandsLimitsNVX& operator=( VkDeviceGeneratedCommandsLimitsNVX const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceGeneratedCommandsLimitsNVX ) );
      return *this;
    }
    DeviceGeneratedCommandsLimitsNVX& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DeviceGeneratedCommandsLimitsNVX& setMaxIndirectCommandsLayoutTokenCount( uint32_t maxIndirectCommandsLayoutTokenCount_ )
    {
      maxIndirectCommandsLayoutTokenCount = maxIndirectCommandsLayoutTokenCount_;
      return *this;
    }

    DeviceGeneratedCommandsLimitsNVX& setMaxObjectEntryCounts( uint32_t maxObjectEntryCounts_ )
    {
      maxObjectEntryCounts = maxObjectEntryCounts_;
      return *this;
    }

    DeviceGeneratedCommandsLimitsNVX& setMinSequenceCountBufferOffsetAlignment( uint32_t minSequenceCountBufferOffsetAlignment_ )
    {
      minSequenceCountBufferOffsetAlignment = minSequenceCountBufferOffsetAlignment_;
      return *this;
    }

    DeviceGeneratedCommandsLimitsNVX& setMinSequenceIndexBufferOffsetAlignment( uint32_t minSequenceIndexBufferOffsetAlignment_ )
    {
      minSequenceIndexBufferOffsetAlignment = minSequenceIndexBufferOffsetAlignment_;
      return *this;
    }

    DeviceGeneratedCommandsLimitsNVX& setMinCommandsTokenBufferOffsetAlignment( uint32_t minCommandsTokenBufferOffsetAlignment_ )
    {
      minCommandsTokenBufferOffsetAlignment = minCommandsTokenBufferOffsetAlignment_;
      return *this;
    }

    operator VkDeviceGeneratedCommandsLimitsNVX const&() const
    {
      return *reinterpret_cast<const VkDeviceGeneratedCommandsLimitsNVX*>(this);
    }

    operator VkDeviceGeneratedCommandsLimitsNVX &()
    {
      return *reinterpret_cast<VkDeviceGeneratedCommandsLimitsNVX*>(this);
    }

    bool operator==( DeviceGeneratedCommandsLimitsNVX const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxIndirectCommandsLayoutTokenCount == rhs.maxIndirectCommandsLayoutTokenCount )
          && ( maxObjectEntryCounts == rhs.maxObjectEntryCounts )
          && ( minSequenceCountBufferOffsetAlignment == rhs.minSequenceCountBufferOffsetAlignment )
          && ( minSequenceIndexBufferOffsetAlignment == rhs.minSequenceIndexBufferOffsetAlignment )
          && ( minCommandsTokenBufferOffsetAlignment == rhs.minCommandsTokenBufferOffsetAlignment );
    }

    bool operator!=( DeviceGeneratedCommandsLimitsNVX const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDeviceGeneratedCommandsLimitsNVX;

  public:
    const void* pNext = nullptr;
    uint32_t maxIndirectCommandsLayoutTokenCount;
    uint32_t maxObjectEntryCounts;
    uint32_t minSequenceCountBufferOffsetAlignment;
    uint32_t minSequenceIndexBufferOffsetAlignment;
    uint32_t minCommandsTokenBufferOffsetAlignment;
  };
  static_assert( sizeof( DeviceGeneratedCommandsLimitsNVX ) == sizeof( VkDeviceGeneratedCommandsLimitsNVX ), "struct and wrapper have different size!" );

  struct CmdReserveSpaceForCommandsInfoNVX
  {
    CmdReserveSpaceForCommandsInfoNVX( ObjectTableNVX objectTable_ = ObjectTableNVX(),
                                       IndirectCommandsLayoutNVX indirectCommandsLayout_ = IndirectCommandsLayoutNVX(),
                                       uint32_t maxSequencesCount_ = 0 )
      : objectTable( objectTable_ )
      , indirectCommandsLayout( indirectCommandsLayout_ )
      , maxSequencesCount( maxSequencesCount_ )
    {
    }

    CmdReserveSpaceForCommandsInfoNVX( VkCmdReserveSpaceForCommandsInfoNVX const & rhs )
    {
      memcpy( this, &rhs, sizeof( CmdReserveSpaceForCommandsInfoNVX ) );
    }

    CmdReserveSpaceForCommandsInfoNVX& operator=( VkCmdReserveSpaceForCommandsInfoNVX const & rhs )
    {
      memcpy( this, &rhs, sizeof( CmdReserveSpaceForCommandsInfoNVX ) );
      return *this;
    }
    CmdReserveSpaceForCommandsInfoNVX& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CmdReserveSpaceForCommandsInfoNVX& setObjectTable( ObjectTableNVX objectTable_ )
    {
      objectTable = objectTable_;
      return *this;
    }

    CmdReserveSpaceForCommandsInfoNVX& setIndirectCommandsLayout( IndirectCommandsLayoutNVX indirectCommandsLayout_ )
    {
      indirectCommandsLayout = indirectCommandsLayout_;
      return *this;
    }

    CmdReserveSpaceForCommandsInfoNVX& setMaxSequencesCount( uint32_t maxSequencesCount_ )
    {
      maxSequencesCount = maxSequencesCount_;
      return *this;
    }

    operator VkCmdReserveSpaceForCommandsInfoNVX const&() const
    {
      return *reinterpret_cast<const VkCmdReserveSpaceForCommandsInfoNVX*>(this);
    }

    operator VkCmdReserveSpaceForCommandsInfoNVX &()
    {
      return *reinterpret_cast<VkCmdReserveSpaceForCommandsInfoNVX*>(this);
    }

    bool operator==( CmdReserveSpaceForCommandsInfoNVX const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( objectTable == rhs.objectTable )
          && ( indirectCommandsLayout == rhs.indirectCommandsLayout )
          && ( maxSequencesCount == rhs.maxSequencesCount );
    }

    bool operator!=( CmdReserveSpaceForCommandsInfoNVX const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCmdReserveSpaceForCommandsInfoNVX;

  public:
    const void* pNext = nullptr;
    ObjectTableNVX objectTable;
    IndirectCommandsLayoutNVX indirectCommandsLayout;
    uint32_t maxSequencesCount;
  };
  static_assert( sizeof( CmdReserveSpaceForCommandsInfoNVX ) == sizeof( VkCmdReserveSpaceForCommandsInfoNVX ), "struct and wrapper have different size!" );

  struct PhysicalDeviceFeatures2
  {
    PhysicalDeviceFeatures2( PhysicalDeviceFeatures features_ = PhysicalDeviceFeatures() )
      : features( features_ )
    {
    }

    PhysicalDeviceFeatures2( VkPhysicalDeviceFeatures2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceFeatures2 ) );
    }

    PhysicalDeviceFeatures2& operator=( VkPhysicalDeviceFeatures2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceFeatures2 ) );
      return *this;
    }
    PhysicalDeviceFeatures2& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceFeatures2& setFeatures( PhysicalDeviceFeatures features_ )
    {
      features = features_;
      return *this;
    }

    operator VkPhysicalDeviceFeatures2 const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceFeatures2*>(this);
    }

    operator VkPhysicalDeviceFeatures2 &()
    {
      return *reinterpret_cast<VkPhysicalDeviceFeatures2*>(this);
    }

    bool operator==( PhysicalDeviceFeatures2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( features == rhs.features );
    }

    bool operator!=( PhysicalDeviceFeatures2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceFeatures2;

  public:
    void* pNext = nullptr;
    PhysicalDeviceFeatures features;
  };
  static_assert( sizeof( PhysicalDeviceFeatures2 ) == sizeof( VkPhysicalDeviceFeatures2 ), "struct and wrapper have different size!" );

  using PhysicalDeviceFeatures2KHR = PhysicalDeviceFeatures2;

  struct PhysicalDevicePushDescriptorPropertiesKHR
  {
    PhysicalDevicePushDescriptorPropertiesKHR( uint32_t maxPushDescriptors_ = 0 )
      : maxPushDescriptors( maxPushDescriptors_ )
    {
    }

    PhysicalDevicePushDescriptorPropertiesKHR( VkPhysicalDevicePushDescriptorPropertiesKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDevicePushDescriptorPropertiesKHR ) );
    }

    PhysicalDevicePushDescriptorPropertiesKHR& operator=( VkPhysicalDevicePushDescriptorPropertiesKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDevicePushDescriptorPropertiesKHR ) );
      return *this;
    }
    PhysicalDevicePushDescriptorPropertiesKHR& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDevicePushDescriptorPropertiesKHR& setMaxPushDescriptors( uint32_t maxPushDescriptors_ )
    {
      maxPushDescriptors = maxPushDescriptors_;
      return *this;
    }

    operator VkPhysicalDevicePushDescriptorPropertiesKHR const&() const
    {
      return *reinterpret_cast<const VkPhysicalDevicePushDescriptorPropertiesKHR*>(this);
    }

    operator VkPhysicalDevicePushDescriptorPropertiesKHR &()
    {
      return *reinterpret_cast<VkPhysicalDevicePushDescriptorPropertiesKHR*>(this);
    }

    bool operator==( PhysicalDevicePushDescriptorPropertiesKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxPushDescriptors == rhs.maxPushDescriptors );
    }

    bool operator!=( PhysicalDevicePushDescriptorPropertiesKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDevicePushDescriptorPropertiesKHR;

  public:
    void* pNext = nullptr;
    uint32_t maxPushDescriptors;
  };
  static_assert( sizeof( PhysicalDevicePushDescriptorPropertiesKHR ) == sizeof( VkPhysicalDevicePushDescriptorPropertiesKHR ), "struct and wrapper have different size!" );

  struct PresentRegionsKHR
  {
    PresentRegionsKHR( uint32_t swapchainCount_ = 0,
                       const PresentRegionKHR* pRegions_ = nullptr )
      : swapchainCount( swapchainCount_ )
      , pRegions( pRegions_ )
    {
    }

    PresentRegionsKHR( VkPresentRegionsKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PresentRegionsKHR ) );
    }

    PresentRegionsKHR& operator=( VkPresentRegionsKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PresentRegionsKHR ) );
      return *this;
    }
    PresentRegionsKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PresentRegionsKHR& setSwapchainCount( uint32_t swapchainCount_ )
    {
      swapchainCount = swapchainCount_;
      return *this;
    }

    PresentRegionsKHR& setPRegions( const PresentRegionKHR* pRegions_ )
    {
      pRegions = pRegions_;
      return *this;
    }

    operator VkPresentRegionsKHR const&() const
    {
      return *reinterpret_cast<const VkPresentRegionsKHR*>(this);
    }

    operator VkPresentRegionsKHR &()
    {
      return *reinterpret_cast<VkPresentRegionsKHR*>(this);
    }

    bool operator==( PresentRegionsKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( swapchainCount == rhs.swapchainCount )
          && ( pRegions == rhs.pRegions );
    }

    bool operator!=( PresentRegionsKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePresentRegionsKHR;

  public:
    const void* pNext = nullptr;
    uint32_t swapchainCount;
    const PresentRegionKHR* pRegions;
  };
  static_assert( sizeof( PresentRegionsKHR ) == sizeof( VkPresentRegionsKHR ), "struct and wrapper have different size!" );

  struct PhysicalDeviceVariablePointerFeatures
  {
    PhysicalDeviceVariablePointerFeatures( Bool32 variablePointersStorageBuffer_ = 0,
                                           Bool32 variablePointers_ = 0 )
      : variablePointersStorageBuffer( variablePointersStorageBuffer_ )
      , variablePointers( variablePointers_ )
    {
    }

    PhysicalDeviceVariablePointerFeatures( VkPhysicalDeviceVariablePointerFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceVariablePointerFeatures ) );
    }

    PhysicalDeviceVariablePointerFeatures& operator=( VkPhysicalDeviceVariablePointerFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceVariablePointerFeatures ) );
      return *this;
    }
    PhysicalDeviceVariablePointerFeatures& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceVariablePointerFeatures& setVariablePointersStorageBuffer( Bool32 variablePointersStorageBuffer_ )
    {
      variablePointersStorageBuffer = variablePointersStorageBuffer_;
      return *this;
    }

    PhysicalDeviceVariablePointerFeatures& setVariablePointers( Bool32 variablePointers_ )
    {
      variablePointers = variablePointers_;
      return *this;
    }

    operator VkPhysicalDeviceVariablePointerFeatures const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceVariablePointerFeatures*>(this);
    }

    operator VkPhysicalDeviceVariablePointerFeatures &()
    {
      return *reinterpret_cast<VkPhysicalDeviceVariablePointerFeatures*>(this);
    }

    bool operator==( PhysicalDeviceVariablePointerFeatures const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( variablePointersStorageBuffer == rhs.variablePointersStorageBuffer )
          && ( variablePointers == rhs.variablePointers );
    }

    bool operator!=( PhysicalDeviceVariablePointerFeatures const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceVariablePointerFeatures;

  public:
    void* pNext = nullptr;
    Bool32 variablePointersStorageBuffer;
    Bool32 variablePointers;
  };
  static_assert( sizeof( PhysicalDeviceVariablePointerFeatures ) == sizeof( VkPhysicalDeviceVariablePointerFeatures ), "struct and wrapper have different size!" );

  using PhysicalDeviceVariablePointerFeaturesKHR = PhysicalDeviceVariablePointerFeatures;

  struct PhysicalDeviceIDProperties
  {
    operator VkPhysicalDeviceIDProperties const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceIDProperties*>(this);
    }

    operator VkPhysicalDeviceIDProperties &()
    {
      return *reinterpret_cast<VkPhysicalDeviceIDProperties*>(this);
    }

    bool operator==( PhysicalDeviceIDProperties const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( memcmp( deviceUUID, rhs.deviceUUID, VK_UUID_SIZE * sizeof( uint8_t ) ) == 0 )
          && ( memcmp( driverUUID, rhs.driverUUID, VK_UUID_SIZE * sizeof( uint8_t ) ) == 0 )
          && ( memcmp( deviceLUID, rhs.deviceLUID, VK_LUID_SIZE * sizeof( uint8_t ) ) == 0 )
          && ( deviceNodeMask == rhs.deviceNodeMask )
          && ( deviceLUIDValid == rhs.deviceLUIDValid );
    }

    bool operator!=( PhysicalDeviceIDProperties const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceIdProperties;

  public:
    void* pNext = nullptr;
    uint8_t deviceUUID[VK_UUID_SIZE];
    uint8_t driverUUID[VK_UUID_SIZE];
    uint8_t deviceLUID[VK_LUID_SIZE];
    uint32_t deviceNodeMask;
    Bool32 deviceLUIDValid;
  };
  static_assert( sizeof( PhysicalDeviceIDProperties ) == sizeof( VkPhysicalDeviceIDProperties ), "struct and wrapper have different size!" );

  using PhysicalDeviceIDPropertiesKHR = PhysicalDeviceIDProperties;

#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct ExportMemoryWin32HandleInfoKHR
  {
    ExportMemoryWin32HandleInfoKHR( const SECURITY_ATTRIBUTES* pAttributes_ = nullptr,
                                    DWORD dwAccess_ = 0,
                                    LPCWSTR name_ = 0 )
      : pAttributes( pAttributes_ )
      , dwAccess( dwAccess_ )
      , name( name_ )
    {
    }

    ExportMemoryWin32HandleInfoKHR( VkExportMemoryWin32HandleInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ExportMemoryWin32HandleInfoKHR ) );
    }

    ExportMemoryWin32HandleInfoKHR& operator=( VkExportMemoryWin32HandleInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ExportMemoryWin32HandleInfoKHR ) );
      return *this;
    }
    ExportMemoryWin32HandleInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ExportMemoryWin32HandleInfoKHR& setPAttributes( const SECURITY_ATTRIBUTES* pAttributes_ )
    {
      pAttributes = pAttributes_;
      return *this;
    }

    ExportMemoryWin32HandleInfoKHR& setDwAccess( DWORD dwAccess_ )
    {
      dwAccess = dwAccess_;
      return *this;
    }

    ExportMemoryWin32HandleInfoKHR& setName( LPCWSTR name_ )
    {
      name = name_;
      return *this;
    }

    operator VkExportMemoryWin32HandleInfoKHR const&() const
    {
      return *reinterpret_cast<const VkExportMemoryWin32HandleInfoKHR*>(this);
    }

    operator VkExportMemoryWin32HandleInfoKHR &()
    {
      return *reinterpret_cast<VkExportMemoryWin32HandleInfoKHR*>(this);
    }

    bool operator==( ExportMemoryWin32HandleInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( pAttributes == rhs.pAttributes )
          && ( dwAccess == rhs.dwAccess )
          && ( name == rhs.name );
    }

    bool operator!=( ExportMemoryWin32HandleInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eExportMemoryWin32HandleInfoKHR;

  public:
    const void* pNext = nullptr;
    const SECURITY_ATTRIBUTES* pAttributes;
    DWORD dwAccess;
    LPCWSTR name;
  };
  static_assert( sizeof( ExportMemoryWin32HandleInfoKHR ) == sizeof( VkExportMemoryWin32HandleInfoKHR ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct MemoryWin32HandlePropertiesKHR
  {
    operator VkMemoryWin32HandlePropertiesKHR const&() const
    {
      return *reinterpret_cast<const VkMemoryWin32HandlePropertiesKHR*>(this);
    }

    operator VkMemoryWin32HandlePropertiesKHR &()
    {
      return *reinterpret_cast<VkMemoryWin32HandlePropertiesKHR*>(this);
    }

    bool operator==( MemoryWin32HandlePropertiesKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( memoryTypeBits == rhs.memoryTypeBits );
    }

    bool operator!=( MemoryWin32HandlePropertiesKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eMemoryWin32HandlePropertiesKHR;

  public:
    void* pNext = nullptr;
    uint32_t memoryTypeBits;
  };
  static_assert( sizeof( MemoryWin32HandlePropertiesKHR ) == sizeof( VkMemoryWin32HandlePropertiesKHR ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

  struct MemoryFdPropertiesKHR
  {
    operator VkMemoryFdPropertiesKHR const&() const
    {
      return *reinterpret_cast<const VkMemoryFdPropertiesKHR*>(this);
    }

    operator VkMemoryFdPropertiesKHR &()
    {
      return *reinterpret_cast<VkMemoryFdPropertiesKHR*>(this);
    }

    bool operator==( MemoryFdPropertiesKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( memoryTypeBits == rhs.memoryTypeBits );
    }

    bool operator!=( MemoryFdPropertiesKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eMemoryFdPropertiesKHR;

  public:
    void* pNext = nullptr;
    uint32_t memoryTypeBits;
  };
  static_assert( sizeof( MemoryFdPropertiesKHR ) == sizeof( VkMemoryFdPropertiesKHR ), "struct and wrapper have different size!" );

#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct Win32KeyedMutexAcquireReleaseInfoKHR
  {
    Win32KeyedMutexAcquireReleaseInfoKHR( uint32_t acquireCount_ = 0,
                                          const DeviceMemory* pAcquireSyncs_ = nullptr,
                                          const uint64_t* pAcquireKeys_ = nullptr,
                                          const uint32_t* pAcquireTimeouts_ = nullptr,
                                          uint32_t releaseCount_ = 0,
                                          const DeviceMemory* pReleaseSyncs_ = nullptr,
                                          const uint64_t* pReleaseKeys_ = nullptr )
      : acquireCount( acquireCount_ )
      , pAcquireSyncs( pAcquireSyncs_ )
      , pAcquireKeys( pAcquireKeys_ )
      , pAcquireTimeouts( pAcquireTimeouts_ )
      , releaseCount( releaseCount_ )
      , pReleaseSyncs( pReleaseSyncs_ )
      , pReleaseKeys( pReleaseKeys_ )
    {
    }

    Win32KeyedMutexAcquireReleaseInfoKHR( VkWin32KeyedMutexAcquireReleaseInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( Win32KeyedMutexAcquireReleaseInfoKHR ) );
    }

    Win32KeyedMutexAcquireReleaseInfoKHR& operator=( VkWin32KeyedMutexAcquireReleaseInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( Win32KeyedMutexAcquireReleaseInfoKHR ) );
      return *this;
    }
    Win32KeyedMutexAcquireReleaseInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoKHR& setAcquireCount( uint32_t acquireCount_ )
    {
      acquireCount = acquireCount_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoKHR& setPAcquireSyncs( const DeviceMemory* pAcquireSyncs_ )
    {
      pAcquireSyncs = pAcquireSyncs_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoKHR& setPAcquireKeys( const uint64_t* pAcquireKeys_ )
    {
      pAcquireKeys = pAcquireKeys_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoKHR& setPAcquireTimeouts( const uint32_t* pAcquireTimeouts_ )
    {
      pAcquireTimeouts = pAcquireTimeouts_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoKHR& setReleaseCount( uint32_t releaseCount_ )
    {
      releaseCount = releaseCount_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoKHR& setPReleaseSyncs( const DeviceMemory* pReleaseSyncs_ )
    {
      pReleaseSyncs = pReleaseSyncs_;
      return *this;
    }

    Win32KeyedMutexAcquireReleaseInfoKHR& setPReleaseKeys( const uint64_t* pReleaseKeys_ )
    {
      pReleaseKeys = pReleaseKeys_;
      return *this;
    }

    operator VkWin32KeyedMutexAcquireReleaseInfoKHR const&() const
    {
      return *reinterpret_cast<const VkWin32KeyedMutexAcquireReleaseInfoKHR*>(this);
    }

    operator VkWin32KeyedMutexAcquireReleaseInfoKHR &()
    {
      return *reinterpret_cast<VkWin32KeyedMutexAcquireReleaseInfoKHR*>(this);
    }

    bool operator==( Win32KeyedMutexAcquireReleaseInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( acquireCount == rhs.acquireCount )
          && ( pAcquireSyncs == rhs.pAcquireSyncs )
          && ( pAcquireKeys == rhs.pAcquireKeys )
          && ( pAcquireTimeouts == rhs.pAcquireTimeouts )
          && ( releaseCount == rhs.releaseCount )
          && ( pReleaseSyncs == rhs.pReleaseSyncs )
          && ( pReleaseKeys == rhs.pReleaseKeys );
    }

    bool operator!=( Win32KeyedMutexAcquireReleaseInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eWin32KeyedMutexAcquireReleaseInfoKHR;

  public:
    const void* pNext = nullptr;
    uint32_t acquireCount;
    const DeviceMemory* pAcquireSyncs;
    const uint64_t* pAcquireKeys;
    const uint32_t* pAcquireTimeouts;
    uint32_t releaseCount;
    const DeviceMemory* pReleaseSyncs;
    const uint64_t* pReleaseKeys;
  };
  static_assert( sizeof( Win32KeyedMutexAcquireReleaseInfoKHR ) == sizeof( VkWin32KeyedMutexAcquireReleaseInfoKHR ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct ExportSemaphoreWin32HandleInfoKHR
  {
    ExportSemaphoreWin32HandleInfoKHR( const SECURITY_ATTRIBUTES* pAttributes_ = nullptr,
                                       DWORD dwAccess_ = 0,
                                       LPCWSTR name_ = 0 )
      : pAttributes( pAttributes_ )
      , dwAccess( dwAccess_ )
      , name( name_ )
    {
    }

    ExportSemaphoreWin32HandleInfoKHR( VkExportSemaphoreWin32HandleInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ExportSemaphoreWin32HandleInfoKHR ) );
    }

    ExportSemaphoreWin32HandleInfoKHR& operator=( VkExportSemaphoreWin32HandleInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ExportSemaphoreWin32HandleInfoKHR ) );
      return *this;
    }
    ExportSemaphoreWin32HandleInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ExportSemaphoreWin32HandleInfoKHR& setPAttributes( const SECURITY_ATTRIBUTES* pAttributes_ )
    {
      pAttributes = pAttributes_;
      return *this;
    }

    ExportSemaphoreWin32HandleInfoKHR& setDwAccess( DWORD dwAccess_ )
    {
      dwAccess = dwAccess_;
      return *this;
    }

    ExportSemaphoreWin32HandleInfoKHR& setName( LPCWSTR name_ )
    {
      name = name_;
      return *this;
    }

    operator VkExportSemaphoreWin32HandleInfoKHR const&() const
    {
      return *reinterpret_cast<const VkExportSemaphoreWin32HandleInfoKHR*>(this);
    }

    operator VkExportSemaphoreWin32HandleInfoKHR &()
    {
      return *reinterpret_cast<VkExportSemaphoreWin32HandleInfoKHR*>(this);
    }

    bool operator==( ExportSemaphoreWin32HandleInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( pAttributes == rhs.pAttributes )
          && ( dwAccess == rhs.dwAccess )
          && ( name == rhs.name );
    }

    bool operator!=( ExportSemaphoreWin32HandleInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eExportSemaphoreWin32HandleInfoKHR;

  public:
    const void* pNext = nullptr;
    const SECURITY_ATTRIBUTES* pAttributes;
    DWORD dwAccess;
    LPCWSTR name;
  };
  static_assert( sizeof( ExportSemaphoreWin32HandleInfoKHR ) == sizeof( VkExportSemaphoreWin32HandleInfoKHR ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct D3D12FenceSubmitInfoKHR
  {
    D3D12FenceSubmitInfoKHR( uint32_t waitSemaphoreValuesCount_ = 0,
                             const uint64_t* pWaitSemaphoreValues_ = nullptr,
                             uint32_t signalSemaphoreValuesCount_ = 0,
                             const uint64_t* pSignalSemaphoreValues_ = nullptr )
      : waitSemaphoreValuesCount( waitSemaphoreValuesCount_ )
      , pWaitSemaphoreValues( pWaitSemaphoreValues_ )
      , signalSemaphoreValuesCount( signalSemaphoreValuesCount_ )
      , pSignalSemaphoreValues( pSignalSemaphoreValues_ )
    {
    }

    D3D12FenceSubmitInfoKHR( VkD3D12FenceSubmitInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( D3D12FenceSubmitInfoKHR ) );
    }

    D3D12FenceSubmitInfoKHR& operator=( VkD3D12FenceSubmitInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( D3D12FenceSubmitInfoKHR ) );
      return *this;
    }
    D3D12FenceSubmitInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    D3D12FenceSubmitInfoKHR& setWaitSemaphoreValuesCount( uint32_t waitSemaphoreValuesCount_ )
    {
      waitSemaphoreValuesCount = waitSemaphoreValuesCount_;
      return *this;
    }

    D3D12FenceSubmitInfoKHR& setPWaitSemaphoreValues( const uint64_t* pWaitSemaphoreValues_ )
    {
      pWaitSemaphoreValues = pWaitSemaphoreValues_;
      return *this;
    }

    D3D12FenceSubmitInfoKHR& setSignalSemaphoreValuesCount( uint32_t signalSemaphoreValuesCount_ )
    {
      signalSemaphoreValuesCount = signalSemaphoreValuesCount_;
      return *this;
    }

    D3D12FenceSubmitInfoKHR& setPSignalSemaphoreValues( const uint64_t* pSignalSemaphoreValues_ )
    {
      pSignalSemaphoreValues = pSignalSemaphoreValues_;
      return *this;
    }

    operator VkD3D12FenceSubmitInfoKHR const&() const
    {
      return *reinterpret_cast<const VkD3D12FenceSubmitInfoKHR*>(this);
    }

    operator VkD3D12FenceSubmitInfoKHR &()
    {
      return *reinterpret_cast<VkD3D12FenceSubmitInfoKHR*>(this);
    }

    bool operator==( D3D12FenceSubmitInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( waitSemaphoreValuesCount == rhs.waitSemaphoreValuesCount )
          && ( pWaitSemaphoreValues == rhs.pWaitSemaphoreValues )
          && ( signalSemaphoreValuesCount == rhs.signalSemaphoreValuesCount )
          && ( pSignalSemaphoreValues == rhs.pSignalSemaphoreValues );
    }

    bool operator!=( D3D12FenceSubmitInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eD3D12FenceSubmitInfoKHR;

  public:
    const void* pNext = nullptr;
    uint32_t waitSemaphoreValuesCount;
    const uint64_t* pWaitSemaphoreValues;
    uint32_t signalSemaphoreValuesCount;
    const uint64_t* pSignalSemaphoreValues;
  };
  static_assert( sizeof( D3D12FenceSubmitInfoKHR ) == sizeof( VkD3D12FenceSubmitInfoKHR ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct ExportFenceWin32HandleInfoKHR
  {
    ExportFenceWin32HandleInfoKHR( const SECURITY_ATTRIBUTES* pAttributes_ = nullptr,
                                   DWORD dwAccess_ = 0,
                                   LPCWSTR name_ = 0 )
      : pAttributes( pAttributes_ )
      , dwAccess( dwAccess_ )
      , name( name_ )
    {
    }

    ExportFenceWin32HandleInfoKHR( VkExportFenceWin32HandleInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ExportFenceWin32HandleInfoKHR ) );
    }

    ExportFenceWin32HandleInfoKHR& operator=( VkExportFenceWin32HandleInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ExportFenceWin32HandleInfoKHR ) );
      return *this;
    }
    ExportFenceWin32HandleInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ExportFenceWin32HandleInfoKHR& setPAttributes( const SECURITY_ATTRIBUTES* pAttributes_ )
    {
      pAttributes = pAttributes_;
      return *this;
    }

    ExportFenceWin32HandleInfoKHR& setDwAccess( DWORD dwAccess_ )
    {
      dwAccess = dwAccess_;
      return *this;
    }

    ExportFenceWin32HandleInfoKHR& setName( LPCWSTR name_ )
    {
      name = name_;
      return *this;
    }

    operator VkExportFenceWin32HandleInfoKHR const&() const
    {
      return *reinterpret_cast<const VkExportFenceWin32HandleInfoKHR*>(this);
    }

    operator VkExportFenceWin32HandleInfoKHR &()
    {
      return *reinterpret_cast<VkExportFenceWin32HandleInfoKHR*>(this);
    }

    bool operator==( ExportFenceWin32HandleInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( pAttributes == rhs.pAttributes )
          && ( dwAccess == rhs.dwAccess )
          && ( name == rhs.name );
    }

    bool operator!=( ExportFenceWin32HandleInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eExportFenceWin32HandleInfoKHR;

  public:
    const void* pNext = nullptr;
    const SECURITY_ATTRIBUTES* pAttributes;
    DWORD dwAccess;
    LPCWSTR name;
  };
  static_assert( sizeof( ExportFenceWin32HandleInfoKHR ) == sizeof( VkExportFenceWin32HandleInfoKHR ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

  struct PhysicalDeviceMultiviewFeatures
  {
    PhysicalDeviceMultiviewFeatures( Bool32 multiview_ = 0,
                                     Bool32 multiviewGeometryShader_ = 0,
                                     Bool32 multiviewTessellationShader_ = 0 )
      : multiview( multiview_ )
      , multiviewGeometryShader( multiviewGeometryShader_ )
      , multiviewTessellationShader( multiviewTessellationShader_ )
    {
    }

    PhysicalDeviceMultiviewFeatures( VkPhysicalDeviceMultiviewFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceMultiviewFeatures ) );
    }

    PhysicalDeviceMultiviewFeatures& operator=( VkPhysicalDeviceMultiviewFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceMultiviewFeatures ) );
      return *this;
    }
    PhysicalDeviceMultiviewFeatures& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceMultiviewFeatures& setMultiview( Bool32 multiview_ )
    {
      multiview = multiview_;
      return *this;
    }

    PhysicalDeviceMultiviewFeatures& setMultiviewGeometryShader( Bool32 multiviewGeometryShader_ )
    {
      multiviewGeometryShader = multiviewGeometryShader_;
      return *this;
    }

    PhysicalDeviceMultiviewFeatures& setMultiviewTessellationShader( Bool32 multiviewTessellationShader_ )
    {
      multiviewTessellationShader = multiviewTessellationShader_;
      return *this;
    }

    operator VkPhysicalDeviceMultiviewFeatures const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceMultiviewFeatures*>(this);
    }

    operator VkPhysicalDeviceMultiviewFeatures &()
    {
      return *reinterpret_cast<VkPhysicalDeviceMultiviewFeatures*>(this);
    }

    bool operator==( PhysicalDeviceMultiviewFeatures const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( multiview == rhs.multiview )
          && ( multiviewGeometryShader == rhs.multiviewGeometryShader )
          && ( multiviewTessellationShader == rhs.multiviewTessellationShader );
    }

    bool operator!=( PhysicalDeviceMultiviewFeatures const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceMultiviewFeatures;

  public:
    void* pNext = nullptr;
    Bool32 multiview;
    Bool32 multiviewGeometryShader;
    Bool32 multiviewTessellationShader;
  };
  static_assert( sizeof( PhysicalDeviceMultiviewFeatures ) == sizeof( VkPhysicalDeviceMultiviewFeatures ), "struct and wrapper have different size!" );

  using PhysicalDeviceMultiviewFeaturesKHR = PhysicalDeviceMultiviewFeatures;

  struct PhysicalDeviceMultiviewProperties
  {
    operator VkPhysicalDeviceMultiviewProperties const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceMultiviewProperties*>(this);
    }

    operator VkPhysicalDeviceMultiviewProperties &()
    {
      return *reinterpret_cast<VkPhysicalDeviceMultiviewProperties*>(this);
    }

    bool operator==( PhysicalDeviceMultiviewProperties const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxMultiviewViewCount == rhs.maxMultiviewViewCount )
          && ( maxMultiviewInstanceIndex == rhs.maxMultiviewInstanceIndex );
    }

    bool operator!=( PhysicalDeviceMultiviewProperties const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceMultiviewProperties;

  public:
    void* pNext = nullptr;
    uint32_t maxMultiviewViewCount;
    uint32_t maxMultiviewInstanceIndex;
  };
  static_assert( sizeof( PhysicalDeviceMultiviewProperties ) == sizeof( VkPhysicalDeviceMultiviewProperties ), "struct and wrapper have different size!" );

  using PhysicalDeviceMultiviewPropertiesKHR = PhysicalDeviceMultiviewProperties;

  struct RenderPassMultiviewCreateInfo
  {
    RenderPassMultiviewCreateInfo( uint32_t subpassCount_ = 0,
                                   const uint32_t* pViewMasks_ = nullptr,
                                   uint32_t dependencyCount_ = 0,
                                   const int32_t* pViewOffsets_ = nullptr,
                                   uint32_t correlationMaskCount_ = 0,
                                   const uint32_t* pCorrelationMasks_ = nullptr )
      : subpassCount( subpassCount_ )
      , pViewMasks( pViewMasks_ )
      , dependencyCount( dependencyCount_ )
      , pViewOffsets( pViewOffsets_ )
      , correlationMaskCount( correlationMaskCount_ )
      , pCorrelationMasks( pCorrelationMasks_ )
    {
    }

    RenderPassMultiviewCreateInfo( VkRenderPassMultiviewCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( RenderPassMultiviewCreateInfo ) );
    }

    RenderPassMultiviewCreateInfo& operator=( VkRenderPassMultiviewCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( RenderPassMultiviewCreateInfo ) );
      return *this;
    }
    RenderPassMultiviewCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    RenderPassMultiviewCreateInfo& setSubpassCount( uint32_t subpassCount_ )
    {
      subpassCount = subpassCount_;
      return *this;
    }

    RenderPassMultiviewCreateInfo& setPViewMasks( const uint32_t* pViewMasks_ )
    {
      pViewMasks = pViewMasks_;
      return *this;
    }

    RenderPassMultiviewCreateInfo& setDependencyCount( uint32_t dependencyCount_ )
    {
      dependencyCount = dependencyCount_;
      return *this;
    }

    RenderPassMultiviewCreateInfo& setPViewOffsets( const int32_t* pViewOffsets_ )
    {
      pViewOffsets = pViewOffsets_;
      return *this;
    }

    RenderPassMultiviewCreateInfo& setCorrelationMaskCount( uint32_t correlationMaskCount_ )
    {
      correlationMaskCount = correlationMaskCount_;
      return *this;
    }

    RenderPassMultiviewCreateInfo& setPCorrelationMasks( const uint32_t* pCorrelationMasks_ )
    {
      pCorrelationMasks = pCorrelationMasks_;
      return *this;
    }

    operator VkRenderPassMultiviewCreateInfo const&() const
    {
      return *reinterpret_cast<const VkRenderPassMultiviewCreateInfo*>(this);
    }

    operator VkRenderPassMultiviewCreateInfo &()
    {
      return *reinterpret_cast<VkRenderPassMultiviewCreateInfo*>(this);
    }

    bool operator==( RenderPassMultiviewCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( subpassCount == rhs.subpassCount )
          && ( pViewMasks == rhs.pViewMasks )
          && ( dependencyCount == rhs.dependencyCount )
          && ( pViewOffsets == rhs.pViewOffsets )
          && ( correlationMaskCount == rhs.correlationMaskCount )
          && ( pCorrelationMasks == rhs.pCorrelationMasks );
    }

    bool operator!=( RenderPassMultiviewCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eRenderPassMultiviewCreateInfo;

  public:
    const void* pNext = nullptr;
    uint32_t subpassCount;
    const uint32_t* pViewMasks;
    uint32_t dependencyCount;
    const int32_t* pViewOffsets;
    uint32_t correlationMaskCount;
    const uint32_t* pCorrelationMasks;
  };
  static_assert( sizeof( RenderPassMultiviewCreateInfo ) == sizeof( VkRenderPassMultiviewCreateInfo ), "struct and wrapper have different size!" );

  using RenderPassMultiviewCreateInfoKHR = RenderPassMultiviewCreateInfo;

  struct BindBufferMemoryInfo
  {
    BindBufferMemoryInfo( Buffer buffer_ = Buffer(),
                          DeviceMemory memory_ = DeviceMemory(),
                          DeviceSize memoryOffset_ = 0 )
      : buffer( buffer_ )
      , memory( memory_ )
      , memoryOffset( memoryOffset_ )
    {
    }

    BindBufferMemoryInfo( VkBindBufferMemoryInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindBufferMemoryInfo ) );
    }

    BindBufferMemoryInfo& operator=( VkBindBufferMemoryInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindBufferMemoryInfo ) );
      return *this;
    }
    BindBufferMemoryInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindBufferMemoryInfo& setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    BindBufferMemoryInfo& setMemory( DeviceMemory memory_ )
    {
      memory = memory_;
      return *this;
    }

    BindBufferMemoryInfo& setMemoryOffset( DeviceSize memoryOffset_ )
    {
      memoryOffset = memoryOffset_;
      return *this;
    }

    operator VkBindBufferMemoryInfo const&() const
    {
      return *reinterpret_cast<const VkBindBufferMemoryInfo*>(this);
    }

    operator VkBindBufferMemoryInfo &()
    {
      return *reinterpret_cast<VkBindBufferMemoryInfo*>(this);
    }

    bool operator==( BindBufferMemoryInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( buffer == rhs.buffer )
          && ( memory == rhs.memory )
          && ( memoryOffset == rhs.memoryOffset );
    }

    bool operator!=( BindBufferMemoryInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindBufferMemoryInfo;

  public:
    const void* pNext = nullptr;
    Buffer buffer;
    DeviceMemory memory;
    DeviceSize memoryOffset;
  };
  static_assert( sizeof( BindBufferMemoryInfo ) == sizeof( VkBindBufferMemoryInfo ), "struct and wrapper have different size!" );

  using BindBufferMemoryInfoKHR = BindBufferMemoryInfo;

  struct BindBufferMemoryDeviceGroupInfo
  {
    BindBufferMemoryDeviceGroupInfo( uint32_t deviceIndexCount_ = 0,
                                     const uint32_t* pDeviceIndices_ = nullptr )
      : deviceIndexCount( deviceIndexCount_ )
      , pDeviceIndices( pDeviceIndices_ )
    {
    }

    BindBufferMemoryDeviceGroupInfo( VkBindBufferMemoryDeviceGroupInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindBufferMemoryDeviceGroupInfo ) );
    }

    BindBufferMemoryDeviceGroupInfo& operator=( VkBindBufferMemoryDeviceGroupInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindBufferMemoryDeviceGroupInfo ) );
      return *this;
    }
    BindBufferMemoryDeviceGroupInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindBufferMemoryDeviceGroupInfo& setDeviceIndexCount( uint32_t deviceIndexCount_ )
    {
      deviceIndexCount = deviceIndexCount_;
      return *this;
    }

    BindBufferMemoryDeviceGroupInfo& setPDeviceIndices( const uint32_t* pDeviceIndices_ )
    {
      pDeviceIndices = pDeviceIndices_;
      return *this;
    }

    operator VkBindBufferMemoryDeviceGroupInfo const&() const
    {
      return *reinterpret_cast<const VkBindBufferMemoryDeviceGroupInfo*>(this);
    }

    operator VkBindBufferMemoryDeviceGroupInfo &()
    {
      return *reinterpret_cast<VkBindBufferMemoryDeviceGroupInfo*>(this);
    }

    bool operator==( BindBufferMemoryDeviceGroupInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( deviceIndexCount == rhs.deviceIndexCount )
          && ( pDeviceIndices == rhs.pDeviceIndices );
    }

    bool operator!=( BindBufferMemoryDeviceGroupInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindBufferMemoryDeviceGroupInfo;

  public:
    const void* pNext = nullptr;
    uint32_t deviceIndexCount;
    const uint32_t* pDeviceIndices;
  };
  static_assert( sizeof( BindBufferMemoryDeviceGroupInfo ) == sizeof( VkBindBufferMemoryDeviceGroupInfo ), "struct and wrapper have different size!" );

  using BindBufferMemoryDeviceGroupInfoKHR = BindBufferMemoryDeviceGroupInfo;

  struct BindImageMemoryInfo
  {
    BindImageMemoryInfo( Image image_ = Image(),
                         DeviceMemory memory_ = DeviceMemory(),
                         DeviceSize memoryOffset_ = 0 )
      : image( image_ )
      , memory( memory_ )
      , memoryOffset( memoryOffset_ )
    {
    }

    BindImageMemoryInfo( VkBindImageMemoryInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImageMemoryInfo ) );
    }

    BindImageMemoryInfo& operator=( VkBindImageMemoryInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImageMemoryInfo ) );
      return *this;
    }
    BindImageMemoryInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindImageMemoryInfo& setImage( Image image_ )
    {
      image = image_;
      return *this;
    }

    BindImageMemoryInfo& setMemory( DeviceMemory memory_ )
    {
      memory = memory_;
      return *this;
    }

    BindImageMemoryInfo& setMemoryOffset( DeviceSize memoryOffset_ )
    {
      memoryOffset = memoryOffset_;
      return *this;
    }

    operator VkBindImageMemoryInfo const&() const
    {
      return *reinterpret_cast<const VkBindImageMemoryInfo*>(this);
    }

    operator VkBindImageMemoryInfo &()
    {
      return *reinterpret_cast<VkBindImageMemoryInfo*>(this);
    }

    bool operator==( BindImageMemoryInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( image == rhs.image )
          && ( memory == rhs.memory )
          && ( memoryOffset == rhs.memoryOffset );
    }

    bool operator!=( BindImageMemoryInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindImageMemoryInfo;

  public:
    const void* pNext = nullptr;
    Image image;
    DeviceMemory memory;
    DeviceSize memoryOffset;
  };
  static_assert( sizeof( BindImageMemoryInfo ) == sizeof( VkBindImageMemoryInfo ), "struct and wrapper have different size!" );

  using BindImageMemoryInfoKHR = BindImageMemoryInfo;

  struct BindImageMemoryDeviceGroupInfo
  {
    BindImageMemoryDeviceGroupInfo( uint32_t deviceIndexCount_ = 0,
                                    const uint32_t* pDeviceIndices_ = nullptr,
                                    uint32_t splitInstanceBindRegionCount_ = 0,
                                    const Rect2D* pSplitInstanceBindRegions_ = nullptr )
      : deviceIndexCount( deviceIndexCount_ )
      , pDeviceIndices( pDeviceIndices_ )
      , splitInstanceBindRegionCount( splitInstanceBindRegionCount_ )
      , pSplitInstanceBindRegions( pSplitInstanceBindRegions_ )
    {
    }

    BindImageMemoryDeviceGroupInfo( VkBindImageMemoryDeviceGroupInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImageMemoryDeviceGroupInfo ) );
    }

    BindImageMemoryDeviceGroupInfo& operator=( VkBindImageMemoryDeviceGroupInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImageMemoryDeviceGroupInfo ) );
      return *this;
    }
    BindImageMemoryDeviceGroupInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindImageMemoryDeviceGroupInfo& setDeviceIndexCount( uint32_t deviceIndexCount_ )
    {
      deviceIndexCount = deviceIndexCount_;
      return *this;
    }

    BindImageMemoryDeviceGroupInfo& setPDeviceIndices( const uint32_t* pDeviceIndices_ )
    {
      pDeviceIndices = pDeviceIndices_;
      return *this;
    }

    BindImageMemoryDeviceGroupInfo& setSplitInstanceBindRegionCount( uint32_t splitInstanceBindRegionCount_ )
    {
      splitInstanceBindRegionCount = splitInstanceBindRegionCount_;
      return *this;
    }

    BindImageMemoryDeviceGroupInfo& setPSplitInstanceBindRegions( const Rect2D* pSplitInstanceBindRegions_ )
    {
      pSplitInstanceBindRegions = pSplitInstanceBindRegions_;
      return *this;
    }

    operator VkBindImageMemoryDeviceGroupInfo const&() const
    {
      return *reinterpret_cast<const VkBindImageMemoryDeviceGroupInfo*>(this);
    }

    operator VkBindImageMemoryDeviceGroupInfo &()
    {
      return *reinterpret_cast<VkBindImageMemoryDeviceGroupInfo*>(this);
    }

    bool operator==( BindImageMemoryDeviceGroupInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( deviceIndexCount == rhs.deviceIndexCount )
          && ( pDeviceIndices == rhs.pDeviceIndices )
          && ( splitInstanceBindRegionCount == rhs.splitInstanceBindRegionCount )
          && ( pSplitInstanceBindRegions == rhs.pSplitInstanceBindRegions );
    }

    bool operator!=( BindImageMemoryDeviceGroupInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindImageMemoryDeviceGroupInfo;

  public:
    const void* pNext = nullptr;
    uint32_t deviceIndexCount;
    const uint32_t* pDeviceIndices;
    uint32_t splitInstanceBindRegionCount;
    const Rect2D* pSplitInstanceBindRegions;
  };
  static_assert( sizeof( BindImageMemoryDeviceGroupInfo ) == sizeof( VkBindImageMemoryDeviceGroupInfo ), "struct and wrapper have different size!" );

  using BindImageMemoryDeviceGroupInfoKHR = BindImageMemoryDeviceGroupInfo;

  struct DeviceGroupRenderPassBeginInfo
  {
    DeviceGroupRenderPassBeginInfo( uint32_t deviceMask_ = 0,
                                    uint32_t deviceRenderAreaCount_ = 0,
                                    const Rect2D* pDeviceRenderAreas_ = nullptr )
      : deviceMask( deviceMask_ )
      , deviceRenderAreaCount( deviceRenderAreaCount_ )
      , pDeviceRenderAreas( pDeviceRenderAreas_ )
    {
    }

    DeviceGroupRenderPassBeginInfo( VkDeviceGroupRenderPassBeginInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceGroupRenderPassBeginInfo ) );
    }

    DeviceGroupRenderPassBeginInfo& operator=( VkDeviceGroupRenderPassBeginInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceGroupRenderPassBeginInfo ) );
      return *this;
    }
    DeviceGroupRenderPassBeginInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DeviceGroupRenderPassBeginInfo& setDeviceMask( uint32_t deviceMask_ )
    {
      deviceMask = deviceMask_;
      return *this;
    }

    DeviceGroupRenderPassBeginInfo& setDeviceRenderAreaCount( uint32_t deviceRenderAreaCount_ )
    {
      deviceRenderAreaCount = deviceRenderAreaCount_;
      return *this;
    }

    DeviceGroupRenderPassBeginInfo& setPDeviceRenderAreas( const Rect2D* pDeviceRenderAreas_ )
    {
      pDeviceRenderAreas = pDeviceRenderAreas_;
      return *this;
    }

    operator VkDeviceGroupRenderPassBeginInfo const&() const
    {
      return *reinterpret_cast<const VkDeviceGroupRenderPassBeginInfo*>(this);
    }

    operator VkDeviceGroupRenderPassBeginInfo &()
    {
      return *reinterpret_cast<VkDeviceGroupRenderPassBeginInfo*>(this);
    }

    bool operator==( DeviceGroupRenderPassBeginInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( deviceMask == rhs.deviceMask )
          && ( deviceRenderAreaCount == rhs.deviceRenderAreaCount )
          && ( pDeviceRenderAreas == rhs.pDeviceRenderAreas );
    }

    bool operator!=( DeviceGroupRenderPassBeginInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDeviceGroupRenderPassBeginInfo;

  public:
    const void* pNext = nullptr;
    uint32_t deviceMask;
    uint32_t deviceRenderAreaCount;
    const Rect2D* pDeviceRenderAreas;
  };
  static_assert( sizeof( DeviceGroupRenderPassBeginInfo ) == sizeof( VkDeviceGroupRenderPassBeginInfo ), "struct and wrapper have different size!" );

  using DeviceGroupRenderPassBeginInfoKHR = DeviceGroupRenderPassBeginInfo;

  struct DeviceGroupCommandBufferBeginInfo
  {
    DeviceGroupCommandBufferBeginInfo( uint32_t deviceMask_ = 0 )
      : deviceMask( deviceMask_ )
    {
    }

    DeviceGroupCommandBufferBeginInfo( VkDeviceGroupCommandBufferBeginInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceGroupCommandBufferBeginInfo ) );
    }

    DeviceGroupCommandBufferBeginInfo& operator=( VkDeviceGroupCommandBufferBeginInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceGroupCommandBufferBeginInfo ) );
      return *this;
    }
    DeviceGroupCommandBufferBeginInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DeviceGroupCommandBufferBeginInfo& setDeviceMask( uint32_t deviceMask_ )
    {
      deviceMask = deviceMask_;
      return *this;
    }

    operator VkDeviceGroupCommandBufferBeginInfo const&() const
    {
      return *reinterpret_cast<const VkDeviceGroupCommandBufferBeginInfo*>(this);
    }

    operator VkDeviceGroupCommandBufferBeginInfo &()
    {
      return *reinterpret_cast<VkDeviceGroupCommandBufferBeginInfo*>(this);
    }

    bool operator==( DeviceGroupCommandBufferBeginInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( deviceMask == rhs.deviceMask );
    }

    bool operator!=( DeviceGroupCommandBufferBeginInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDeviceGroupCommandBufferBeginInfo;

  public:
    const void* pNext = nullptr;
    uint32_t deviceMask;
  };
  static_assert( sizeof( DeviceGroupCommandBufferBeginInfo ) == sizeof( VkDeviceGroupCommandBufferBeginInfo ), "struct and wrapper have different size!" );

  using DeviceGroupCommandBufferBeginInfoKHR = DeviceGroupCommandBufferBeginInfo;

  struct DeviceGroupSubmitInfo
  {
    DeviceGroupSubmitInfo( uint32_t waitSemaphoreCount_ = 0,
                           const uint32_t* pWaitSemaphoreDeviceIndices_ = nullptr,
                           uint32_t commandBufferCount_ = 0,
                           const uint32_t* pCommandBufferDeviceMasks_ = nullptr,
                           uint32_t signalSemaphoreCount_ = 0,
                           const uint32_t* pSignalSemaphoreDeviceIndices_ = nullptr )
      : waitSemaphoreCount( waitSemaphoreCount_ )
      , pWaitSemaphoreDeviceIndices( pWaitSemaphoreDeviceIndices_ )
      , commandBufferCount( commandBufferCount_ )
      , pCommandBufferDeviceMasks( pCommandBufferDeviceMasks_ )
      , signalSemaphoreCount( signalSemaphoreCount_ )
      , pSignalSemaphoreDeviceIndices( pSignalSemaphoreDeviceIndices_ )
    {
    }

    DeviceGroupSubmitInfo( VkDeviceGroupSubmitInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceGroupSubmitInfo ) );
    }

    DeviceGroupSubmitInfo& operator=( VkDeviceGroupSubmitInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceGroupSubmitInfo ) );
      return *this;
    }
    DeviceGroupSubmitInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DeviceGroupSubmitInfo& setWaitSemaphoreCount( uint32_t waitSemaphoreCount_ )
    {
      waitSemaphoreCount = waitSemaphoreCount_;
      return *this;
    }

    DeviceGroupSubmitInfo& setPWaitSemaphoreDeviceIndices( const uint32_t* pWaitSemaphoreDeviceIndices_ )
    {
      pWaitSemaphoreDeviceIndices = pWaitSemaphoreDeviceIndices_;
      return *this;
    }

    DeviceGroupSubmitInfo& setCommandBufferCount( uint32_t commandBufferCount_ )
    {
      commandBufferCount = commandBufferCount_;
      return *this;
    }

    DeviceGroupSubmitInfo& setPCommandBufferDeviceMasks( const uint32_t* pCommandBufferDeviceMasks_ )
    {
      pCommandBufferDeviceMasks = pCommandBufferDeviceMasks_;
      return *this;
    }

    DeviceGroupSubmitInfo& setSignalSemaphoreCount( uint32_t signalSemaphoreCount_ )
    {
      signalSemaphoreCount = signalSemaphoreCount_;
      return *this;
    }

    DeviceGroupSubmitInfo& setPSignalSemaphoreDeviceIndices( const uint32_t* pSignalSemaphoreDeviceIndices_ )
    {
      pSignalSemaphoreDeviceIndices = pSignalSemaphoreDeviceIndices_;
      return *this;
    }

    operator VkDeviceGroupSubmitInfo const&() const
    {
      return *reinterpret_cast<const VkDeviceGroupSubmitInfo*>(this);
    }

    operator VkDeviceGroupSubmitInfo &()
    {
      return *reinterpret_cast<VkDeviceGroupSubmitInfo*>(this);
    }

    bool operator==( DeviceGroupSubmitInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( waitSemaphoreCount == rhs.waitSemaphoreCount )
          && ( pWaitSemaphoreDeviceIndices == rhs.pWaitSemaphoreDeviceIndices )
          && ( commandBufferCount == rhs.commandBufferCount )
          && ( pCommandBufferDeviceMasks == rhs.pCommandBufferDeviceMasks )
          && ( signalSemaphoreCount == rhs.signalSemaphoreCount )
          && ( pSignalSemaphoreDeviceIndices == rhs.pSignalSemaphoreDeviceIndices );
    }

    bool operator!=( DeviceGroupSubmitInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDeviceGroupSubmitInfo;

  public:
    const void* pNext = nullptr;
    uint32_t waitSemaphoreCount;
    const uint32_t* pWaitSemaphoreDeviceIndices;
    uint32_t commandBufferCount;
    const uint32_t* pCommandBufferDeviceMasks;
    uint32_t signalSemaphoreCount;
    const uint32_t* pSignalSemaphoreDeviceIndices;
  };
  static_assert( sizeof( DeviceGroupSubmitInfo ) == sizeof( VkDeviceGroupSubmitInfo ), "struct and wrapper have different size!" );

  using DeviceGroupSubmitInfoKHR = DeviceGroupSubmitInfo;

  struct DeviceGroupBindSparseInfo
  {
    DeviceGroupBindSparseInfo( uint32_t resourceDeviceIndex_ = 0,
                               uint32_t memoryDeviceIndex_ = 0 )
      : resourceDeviceIndex( resourceDeviceIndex_ )
      , memoryDeviceIndex( memoryDeviceIndex_ )
    {
    }

    DeviceGroupBindSparseInfo( VkDeviceGroupBindSparseInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceGroupBindSparseInfo ) );
    }

    DeviceGroupBindSparseInfo& operator=( VkDeviceGroupBindSparseInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceGroupBindSparseInfo ) );
      return *this;
    }
    DeviceGroupBindSparseInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DeviceGroupBindSparseInfo& setResourceDeviceIndex( uint32_t resourceDeviceIndex_ )
    {
      resourceDeviceIndex = resourceDeviceIndex_;
      return *this;
    }

    DeviceGroupBindSparseInfo& setMemoryDeviceIndex( uint32_t memoryDeviceIndex_ )
    {
      memoryDeviceIndex = memoryDeviceIndex_;
      return *this;
    }

    operator VkDeviceGroupBindSparseInfo const&() const
    {
      return *reinterpret_cast<const VkDeviceGroupBindSparseInfo*>(this);
    }

    operator VkDeviceGroupBindSparseInfo &()
    {
      return *reinterpret_cast<VkDeviceGroupBindSparseInfo*>(this);
    }

    bool operator==( DeviceGroupBindSparseInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( resourceDeviceIndex == rhs.resourceDeviceIndex )
          && ( memoryDeviceIndex == rhs.memoryDeviceIndex );
    }

    bool operator!=( DeviceGroupBindSparseInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDeviceGroupBindSparseInfo;

  public:
    const void* pNext = nullptr;
    uint32_t resourceDeviceIndex;
    uint32_t memoryDeviceIndex;
  };
  static_assert( sizeof( DeviceGroupBindSparseInfo ) == sizeof( VkDeviceGroupBindSparseInfo ), "struct and wrapper have different size!" );

  using DeviceGroupBindSparseInfoKHR = DeviceGroupBindSparseInfo;

  struct ImageSwapchainCreateInfoKHR
  {
    ImageSwapchainCreateInfoKHR( SwapchainKHR swapchain_ = SwapchainKHR() )
      : swapchain( swapchain_ )
    {
    }

    ImageSwapchainCreateInfoKHR( VkImageSwapchainCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSwapchainCreateInfoKHR ) );
    }

    ImageSwapchainCreateInfoKHR& operator=( VkImageSwapchainCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSwapchainCreateInfoKHR ) );
      return *this;
    }
    ImageSwapchainCreateInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImageSwapchainCreateInfoKHR& setSwapchain( SwapchainKHR swapchain_ )
    {
      swapchain = swapchain_;
      return *this;
    }

    operator VkImageSwapchainCreateInfoKHR const&() const
    {
      return *reinterpret_cast<const VkImageSwapchainCreateInfoKHR*>(this);
    }

    operator VkImageSwapchainCreateInfoKHR &()
    {
      return *reinterpret_cast<VkImageSwapchainCreateInfoKHR*>(this);
    }

    bool operator==( ImageSwapchainCreateInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( swapchain == rhs.swapchain );
    }

    bool operator!=( ImageSwapchainCreateInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageSwapchainCreateInfoKHR;

  public:
    const void* pNext = nullptr;
    SwapchainKHR swapchain;
  };
  static_assert( sizeof( ImageSwapchainCreateInfoKHR ) == sizeof( VkImageSwapchainCreateInfoKHR ), "struct and wrapper have different size!" );

  struct BindImageMemorySwapchainInfoKHR
  {
    BindImageMemorySwapchainInfoKHR( SwapchainKHR swapchain_ = SwapchainKHR(),
                                     uint32_t imageIndex_ = 0 )
      : swapchain( swapchain_ )
      , imageIndex( imageIndex_ )
    {
    }

    BindImageMemorySwapchainInfoKHR( VkBindImageMemorySwapchainInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImageMemorySwapchainInfoKHR ) );
    }

    BindImageMemorySwapchainInfoKHR& operator=( VkBindImageMemorySwapchainInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImageMemorySwapchainInfoKHR ) );
      return *this;
    }
    BindImageMemorySwapchainInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindImageMemorySwapchainInfoKHR& setSwapchain( SwapchainKHR swapchain_ )
    {
      swapchain = swapchain_;
      return *this;
    }

    BindImageMemorySwapchainInfoKHR& setImageIndex( uint32_t imageIndex_ )
    {
      imageIndex = imageIndex_;
      return *this;
    }

    operator VkBindImageMemorySwapchainInfoKHR const&() const
    {
      return *reinterpret_cast<const VkBindImageMemorySwapchainInfoKHR*>(this);
    }

    operator VkBindImageMemorySwapchainInfoKHR &()
    {
      return *reinterpret_cast<VkBindImageMemorySwapchainInfoKHR*>(this);
    }

    bool operator==( BindImageMemorySwapchainInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( swapchain == rhs.swapchain )
          && ( imageIndex == rhs.imageIndex );
    }

    bool operator!=( BindImageMemorySwapchainInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindImageMemorySwapchainInfoKHR;

  public:
    const void* pNext = nullptr;
    SwapchainKHR swapchain;
    uint32_t imageIndex;
  };
  static_assert( sizeof( BindImageMemorySwapchainInfoKHR ) == sizeof( VkBindImageMemorySwapchainInfoKHR ), "struct and wrapper have different size!" );

  struct AcquireNextImageInfoKHR
  {
    AcquireNextImageInfoKHR( SwapchainKHR swapchain_ = SwapchainKHR(),
                             uint64_t timeout_ = 0,
                             Semaphore semaphore_ = Semaphore(),
                             Fence fence_ = Fence(),
                             uint32_t deviceMask_ = 0 )
      : swapchain( swapchain_ )
      , timeout( timeout_ )
      , semaphore( semaphore_ )
      , fence( fence_ )
      , deviceMask( deviceMask_ )
    {
    }

    AcquireNextImageInfoKHR( VkAcquireNextImageInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AcquireNextImageInfoKHR ) );
    }

    AcquireNextImageInfoKHR& operator=( VkAcquireNextImageInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AcquireNextImageInfoKHR ) );
      return *this;
    }
    AcquireNextImageInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    AcquireNextImageInfoKHR& setSwapchain( SwapchainKHR swapchain_ )
    {
      swapchain = swapchain_;
      return *this;
    }

    AcquireNextImageInfoKHR& setTimeout( uint64_t timeout_ )
    {
      timeout = timeout_;
      return *this;
    }

    AcquireNextImageInfoKHR& setSemaphore( Semaphore semaphore_ )
    {
      semaphore = semaphore_;
      return *this;
    }

    AcquireNextImageInfoKHR& setFence( Fence fence_ )
    {
      fence = fence_;
      return *this;
    }

    AcquireNextImageInfoKHR& setDeviceMask( uint32_t deviceMask_ )
    {
      deviceMask = deviceMask_;
      return *this;
    }

    operator VkAcquireNextImageInfoKHR const&() const
    {
      return *reinterpret_cast<const VkAcquireNextImageInfoKHR*>(this);
    }

    operator VkAcquireNextImageInfoKHR &()
    {
      return *reinterpret_cast<VkAcquireNextImageInfoKHR*>(this);
    }

    bool operator==( AcquireNextImageInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( swapchain == rhs.swapchain )
          && ( timeout == rhs.timeout )
          && ( semaphore == rhs.semaphore )
          && ( fence == rhs.fence )
          && ( deviceMask == rhs.deviceMask );
    }

    bool operator!=( AcquireNextImageInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAcquireNextImageInfoKHR;

  public:
    const void* pNext = nullptr;
    SwapchainKHR swapchain;
    uint64_t timeout;
    Semaphore semaphore;
    Fence fence;
    uint32_t deviceMask;
  };
  static_assert( sizeof( AcquireNextImageInfoKHR ) == sizeof( VkAcquireNextImageInfoKHR ), "struct and wrapper have different size!" );

  struct HdrMetadataEXT
  {
    HdrMetadataEXT( XYColorEXT displayPrimaryRed_ = XYColorEXT(),
                    XYColorEXT displayPrimaryGreen_ = XYColorEXT(),
                    XYColorEXT displayPrimaryBlue_ = XYColorEXT(),
                    XYColorEXT whitePoint_ = XYColorEXT(),
                    float maxLuminance_ = 0,
                    float minLuminance_ = 0,
                    float maxContentLightLevel_ = 0,
                    float maxFrameAverageLightLevel_ = 0 )
      : displayPrimaryRed( displayPrimaryRed_ )
      , displayPrimaryGreen( displayPrimaryGreen_ )
      , displayPrimaryBlue( displayPrimaryBlue_ )
      , whitePoint( whitePoint_ )
      , maxLuminance( maxLuminance_ )
      , minLuminance( minLuminance_ )
      , maxContentLightLevel( maxContentLightLevel_ )
      , maxFrameAverageLightLevel( maxFrameAverageLightLevel_ )
    {
    }

    HdrMetadataEXT( VkHdrMetadataEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( HdrMetadataEXT ) );
    }

    HdrMetadataEXT& operator=( VkHdrMetadataEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( HdrMetadataEXT ) );
      return *this;
    }
    HdrMetadataEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    HdrMetadataEXT& setDisplayPrimaryRed( XYColorEXT displayPrimaryRed_ )
    {
      displayPrimaryRed = displayPrimaryRed_;
      return *this;
    }

    HdrMetadataEXT& setDisplayPrimaryGreen( XYColorEXT displayPrimaryGreen_ )
    {
      displayPrimaryGreen = displayPrimaryGreen_;
      return *this;
    }

    HdrMetadataEXT& setDisplayPrimaryBlue( XYColorEXT displayPrimaryBlue_ )
    {
      displayPrimaryBlue = displayPrimaryBlue_;
      return *this;
    }

    HdrMetadataEXT& setWhitePoint( XYColorEXT whitePoint_ )
    {
      whitePoint = whitePoint_;
      return *this;
    }

    HdrMetadataEXT& setMaxLuminance( float maxLuminance_ )
    {
      maxLuminance = maxLuminance_;
      return *this;
    }

    HdrMetadataEXT& setMinLuminance( float minLuminance_ )
    {
      minLuminance = minLuminance_;
      return *this;
    }

    HdrMetadataEXT& setMaxContentLightLevel( float maxContentLightLevel_ )
    {
      maxContentLightLevel = maxContentLightLevel_;
      return *this;
    }

    HdrMetadataEXT& setMaxFrameAverageLightLevel( float maxFrameAverageLightLevel_ )
    {
      maxFrameAverageLightLevel = maxFrameAverageLightLevel_;
      return *this;
    }

    operator VkHdrMetadataEXT const&() const
    {
      return *reinterpret_cast<const VkHdrMetadataEXT*>(this);
    }

    operator VkHdrMetadataEXT &()
    {
      return *reinterpret_cast<VkHdrMetadataEXT*>(this);
    }

    bool operator==( HdrMetadataEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( displayPrimaryRed == rhs.displayPrimaryRed )
          && ( displayPrimaryGreen == rhs.displayPrimaryGreen )
          && ( displayPrimaryBlue == rhs.displayPrimaryBlue )
          && ( whitePoint == rhs.whitePoint )
          && ( maxLuminance == rhs.maxLuminance )
          && ( minLuminance == rhs.minLuminance )
          && ( maxContentLightLevel == rhs.maxContentLightLevel )
          && ( maxFrameAverageLightLevel == rhs.maxFrameAverageLightLevel );
    }

    bool operator!=( HdrMetadataEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eHdrMetadataEXT;

  public:
    const void* pNext = nullptr;
    XYColorEXT displayPrimaryRed;
    XYColorEXT displayPrimaryGreen;
    XYColorEXT displayPrimaryBlue;
    XYColorEXT whitePoint;
    float maxLuminance;
    float minLuminance;
    float maxContentLightLevel;
    float maxFrameAverageLightLevel;
  };
  static_assert( sizeof( HdrMetadataEXT ) == sizeof( VkHdrMetadataEXT ), "struct and wrapper have different size!" );

  struct PresentTimesInfoGOOGLE
  {
    PresentTimesInfoGOOGLE( uint32_t swapchainCount_ = 0,
                            const PresentTimeGOOGLE* pTimes_ = nullptr )
      : swapchainCount( swapchainCount_ )
      , pTimes( pTimes_ )
    {
    }

    PresentTimesInfoGOOGLE( VkPresentTimesInfoGOOGLE const & rhs )
    {
      memcpy( this, &rhs, sizeof( PresentTimesInfoGOOGLE ) );
    }

    PresentTimesInfoGOOGLE& operator=( VkPresentTimesInfoGOOGLE const & rhs )
    {
      memcpy( this, &rhs, sizeof( PresentTimesInfoGOOGLE ) );
      return *this;
    }
    PresentTimesInfoGOOGLE& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PresentTimesInfoGOOGLE& setSwapchainCount( uint32_t swapchainCount_ )
    {
      swapchainCount = swapchainCount_;
      return *this;
    }

    PresentTimesInfoGOOGLE& setPTimes( const PresentTimeGOOGLE* pTimes_ )
    {
      pTimes = pTimes_;
      return *this;
    }

    operator VkPresentTimesInfoGOOGLE const&() const
    {
      return *reinterpret_cast<const VkPresentTimesInfoGOOGLE*>(this);
    }

    operator VkPresentTimesInfoGOOGLE &()
    {
      return *reinterpret_cast<VkPresentTimesInfoGOOGLE*>(this);
    }

    bool operator==( PresentTimesInfoGOOGLE const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( swapchainCount == rhs.swapchainCount )
          && ( pTimes == rhs.pTimes );
    }

    bool operator!=( PresentTimesInfoGOOGLE const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePresentTimesInfoGOOGLE;

  public:
    const void* pNext = nullptr;
    uint32_t swapchainCount;
    const PresentTimeGOOGLE* pTimes;
  };
  static_assert( sizeof( PresentTimesInfoGOOGLE ) == sizeof( VkPresentTimesInfoGOOGLE ), "struct and wrapper have different size!" );

#ifdef VK_USE_PLATFORM_IOS_MVK
  struct IOSSurfaceCreateInfoMVK
  {
    IOSSurfaceCreateInfoMVK( IOSSurfaceCreateFlagsMVK flags_ = IOSSurfaceCreateFlagsMVK(),
                             const void* pView_ = nullptr )
      : flags( flags_ )
      , pView( pView_ )
    {
    }

    IOSSurfaceCreateInfoMVK( VkIOSSurfaceCreateInfoMVK const & rhs )
    {
      memcpy( this, &rhs, sizeof( IOSSurfaceCreateInfoMVK ) );
    }

    IOSSurfaceCreateInfoMVK& operator=( VkIOSSurfaceCreateInfoMVK const & rhs )
    {
      memcpy( this, &rhs, sizeof( IOSSurfaceCreateInfoMVK ) );
      return *this;
    }
    IOSSurfaceCreateInfoMVK& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    IOSSurfaceCreateInfoMVK& setFlags( IOSSurfaceCreateFlagsMVK flags_ )
    {
      flags = flags_;
      return *this;
    }

    IOSSurfaceCreateInfoMVK& setPView( const void* pView_ )
    {
      pView = pView_;
      return *this;
    }

    operator VkIOSSurfaceCreateInfoMVK const&() const
    {
      return *reinterpret_cast<const VkIOSSurfaceCreateInfoMVK*>(this);
    }

    operator VkIOSSurfaceCreateInfoMVK &()
    {
      return *reinterpret_cast<VkIOSSurfaceCreateInfoMVK*>(this);
    }

    bool operator==( IOSSurfaceCreateInfoMVK const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( pView == rhs.pView );
    }

    bool operator!=( IOSSurfaceCreateInfoMVK const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eIosSurfaceCreateInfoMVK;

  public:
    const void* pNext = nullptr;
    IOSSurfaceCreateFlagsMVK flags;
    const void* pView;
  };
  static_assert( sizeof( IOSSurfaceCreateInfoMVK ) == sizeof( VkIOSSurfaceCreateInfoMVK ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_IOS_MVK*/

#ifdef VK_USE_PLATFORM_MACOS_MVK
  struct MacOSSurfaceCreateInfoMVK
  {
    MacOSSurfaceCreateInfoMVK( MacOSSurfaceCreateFlagsMVK flags_ = MacOSSurfaceCreateFlagsMVK(),
                               const void* pView_ = nullptr )
      : flags( flags_ )
      , pView( pView_ )
    {
    }

    MacOSSurfaceCreateInfoMVK( VkMacOSSurfaceCreateInfoMVK const & rhs )
    {
      memcpy( this, &rhs, sizeof( MacOSSurfaceCreateInfoMVK ) );
    }

    MacOSSurfaceCreateInfoMVK& operator=( VkMacOSSurfaceCreateInfoMVK const & rhs )
    {
      memcpy( this, &rhs, sizeof( MacOSSurfaceCreateInfoMVK ) );
      return *this;
    }
    MacOSSurfaceCreateInfoMVK& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    MacOSSurfaceCreateInfoMVK& setFlags( MacOSSurfaceCreateFlagsMVK flags_ )
    {
      flags = flags_;
      return *this;
    }

    MacOSSurfaceCreateInfoMVK& setPView( const void* pView_ )
    {
      pView = pView_;
      return *this;
    }

    operator VkMacOSSurfaceCreateInfoMVK const&() const
    {
      return *reinterpret_cast<const VkMacOSSurfaceCreateInfoMVK*>(this);
    }

    operator VkMacOSSurfaceCreateInfoMVK &()
    {
      return *reinterpret_cast<VkMacOSSurfaceCreateInfoMVK*>(this);
    }

    bool operator==( MacOSSurfaceCreateInfoMVK const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( pView == rhs.pView );
    }

    bool operator!=( MacOSSurfaceCreateInfoMVK const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eMacosSurfaceCreateInfoMVK;

  public:
    const void* pNext = nullptr;
    MacOSSurfaceCreateFlagsMVK flags;
    const void* pView;
  };
  static_assert( sizeof( MacOSSurfaceCreateInfoMVK ) == sizeof( VkMacOSSurfaceCreateInfoMVK ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_MACOS_MVK*/

  struct PipelineViewportWScalingStateCreateInfoNV
  {
    PipelineViewportWScalingStateCreateInfoNV( Bool32 viewportWScalingEnable_ = 0,
                                               uint32_t viewportCount_ = 0,
                                               const ViewportWScalingNV* pViewportWScalings_ = nullptr )
      : viewportWScalingEnable( viewportWScalingEnable_ )
      , viewportCount( viewportCount_ )
      , pViewportWScalings( pViewportWScalings_ )
    {
    }

    PipelineViewportWScalingStateCreateInfoNV( VkPipelineViewportWScalingStateCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineViewportWScalingStateCreateInfoNV ) );
    }

    PipelineViewportWScalingStateCreateInfoNV& operator=( VkPipelineViewportWScalingStateCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineViewportWScalingStateCreateInfoNV ) );
      return *this;
    }
    PipelineViewportWScalingStateCreateInfoNV& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineViewportWScalingStateCreateInfoNV& setViewportWScalingEnable( Bool32 viewportWScalingEnable_ )
    {
      viewportWScalingEnable = viewportWScalingEnable_;
      return *this;
    }

    PipelineViewportWScalingStateCreateInfoNV& setViewportCount( uint32_t viewportCount_ )
    {
      viewportCount = viewportCount_;
      return *this;
    }

    PipelineViewportWScalingStateCreateInfoNV& setPViewportWScalings( const ViewportWScalingNV* pViewportWScalings_ )
    {
      pViewportWScalings = pViewportWScalings_;
      return *this;
    }

    operator VkPipelineViewportWScalingStateCreateInfoNV const&() const
    {
      return *reinterpret_cast<const VkPipelineViewportWScalingStateCreateInfoNV*>(this);
    }

    operator VkPipelineViewportWScalingStateCreateInfoNV &()
    {
      return *reinterpret_cast<VkPipelineViewportWScalingStateCreateInfoNV*>(this);
    }

    bool operator==( PipelineViewportWScalingStateCreateInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( viewportWScalingEnable == rhs.viewportWScalingEnable )
          && ( viewportCount == rhs.viewportCount )
          && ( pViewportWScalings == rhs.pViewportWScalings );
    }

    bool operator!=( PipelineViewportWScalingStateCreateInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineViewportWScalingStateCreateInfoNV;

  public:
    const void* pNext = nullptr;
    Bool32 viewportWScalingEnable;
    uint32_t viewportCount;
    const ViewportWScalingNV* pViewportWScalings;
  };
  static_assert( sizeof( PipelineViewportWScalingStateCreateInfoNV ) == sizeof( VkPipelineViewportWScalingStateCreateInfoNV ), "struct and wrapper have different size!" );

  struct PhysicalDeviceDiscardRectanglePropertiesEXT
  {
    PhysicalDeviceDiscardRectanglePropertiesEXT( uint32_t maxDiscardRectangles_ = 0 )
      : maxDiscardRectangles( maxDiscardRectangles_ )
    {
    }

    PhysicalDeviceDiscardRectanglePropertiesEXT( VkPhysicalDeviceDiscardRectanglePropertiesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceDiscardRectanglePropertiesEXT ) );
    }

    PhysicalDeviceDiscardRectanglePropertiesEXT& operator=( VkPhysicalDeviceDiscardRectanglePropertiesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceDiscardRectanglePropertiesEXT ) );
      return *this;
    }
    PhysicalDeviceDiscardRectanglePropertiesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceDiscardRectanglePropertiesEXT& setMaxDiscardRectangles( uint32_t maxDiscardRectangles_ )
    {
      maxDiscardRectangles = maxDiscardRectangles_;
      return *this;
    }

    operator VkPhysicalDeviceDiscardRectanglePropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceDiscardRectanglePropertiesEXT*>(this);
    }

    operator VkPhysicalDeviceDiscardRectanglePropertiesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceDiscardRectanglePropertiesEXT*>(this);
    }

    bool operator==( PhysicalDeviceDiscardRectanglePropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxDiscardRectangles == rhs.maxDiscardRectangles );
    }

    bool operator!=( PhysicalDeviceDiscardRectanglePropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceDiscardRectanglePropertiesEXT;

  public:
    void* pNext = nullptr;
    uint32_t maxDiscardRectangles;
  };
  static_assert( sizeof( PhysicalDeviceDiscardRectanglePropertiesEXT ) == sizeof( VkPhysicalDeviceDiscardRectanglePropertiesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceMultiviewPerViewAttributesPropertiesNVX
  {
    operator VkPhysicalDeviceMultiviewPerViewAttributesPropertiesNVX const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceMultiviewPerViewAttributesPropertiesNVX*>(this);
    }

    operator VkPhysicalDeviceMultiviewPerViewAttributesPropertiesNVX &()
    {
      return *reinterpret_cast<VkPhysicalDeviceMultiviewPerViewAttributesPropertiesNVX*>(this);
    }

    bool operator==( PhysicalDeviceMultiviewPerViewAttributesPropertiesNVX const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( perViewPositionAllComponents == rhs.perViewPositionAllComponents );
    }

    bool operator!=( PhysicalDeviceMultiviewPerViewAttributesPropertiesNVX const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceMultiviewPerViewAttributesPropertiesNVX;

  public:
    void* pNext = nullptr;
    Bool32 perViewPositionAllComponents;
  };
  static_assert( sizeof( PhysicalDeviceMultiviewPerViewAttributesPropertiesNVX ) == sizeof( VkPhysicalDeviceMultiviewPerViewAttributesPropertiesNVX ), "struct and wrapper have different size!" );

  struct PhysicalDeviceSurfaceInfo2KHR
  {
    PhysicalDeviceSurfaceInfo2KHR( SurfaceKHR surface_ = SurfaceKHR() )
      : surface( surface_ )
    {
    }

    PhysicalDeviceSurfaceInfo2KHR( VkPhysicalDeviceSurfaceInfo2KHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceSurfaceInfo2KHR ) );
    }

    PhysicalDeviceSurfaceInfo2KHR& operator=( VkPhysicalDeviceSurfaceInfo2KHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceSurfaceInfo2KHR ) );
      return *this;
    }
    PhysicalDeviceSurfaceInfo2KHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceSurfaceInfo2KHR& setSurface( SurfaceKHR surface_ )
    {
      surface = surface_;
      return *this;
    }

    operator VkPhysicalDeviceSurfaceInfo2KHR const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceSurfaceInfo2KHR*>(this);
    }

    operator VkPhysicalDeviceSurfaceInfo2KHR &()
    {
      return *reinterpret_cast<VkPhysicalDeviceSurfaceInfo2KHR*>(this);
    }

    bool operator==( PhysicalDeviceSurfaceInfo2KHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( surface == rhs.surface );
    }

    bool operator!=( PhysicalDeviceSurfaceInfo2KHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceSurfaceInfo2KHR;

  public:
    const void* pNext = nullptr;
    SurfaceKHR surface;
  };
  static_assert( sizeof( PhysicalDeviceSurfaceInfo2KHR ) == sizeof( VkPhysicalDeviceSurfaceInfo2KHR ), "struct and wrapper have different size!" );

  struct DisplayPlaneProperties2KHR
  {
    operator VkDisplayPlaneProperties2KHR const&() const
    {
      return *reinterpret_cast<const VkDisplayPlaneProperties2KHR*>(this);
    }

    operator VkDisplayPlaneProperties2KHR &()
    {
      return *reinterpret_cast<VkDisplayPlaneProperties2KHR*>(this);
    }

    bool operator==( DisplayPlaneProperties2KHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( displayPlaneProperties == rhs.displayPlaneProperties );
    }

    bool operator!=( DisplayPlaneProperties2KHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDisplayPlaneProperties2KHR;

  public:
    void* pNext = nullptr;
    DisplayPlanePropertiesKHR displayPlaneProperties;
  };
  static_assert( sizeof( DisplayPlaneProperties2KHR ) == sizeof( VkDisplayPlaneProperties2KHR ), "struct and wrapper have different size!" );

  struct DisplayModeProperties2KHR
  {
    operator VkDisplayModeProperties2KHR const&() const
    {
      return *reinterpret_cast<const VkDisplayModeProperties2KHR*>(this);
    }

    operator VkDisplayModeProperties2KHR &()
    {
      return *reinterpret_cast<VkDisplayModeProperties2KHR*>(this);
    }

    bool operator==( DisplayModeProperties2KHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( displayModeProperties == rhs.displayModeProperties );
    }

    bool operator!=( DisplayModeProperties2KHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDisplayModeProperties2KHR;

  public:
    void* pNext = nullptr;
    DisplayModePropertiesKHR displayModeProperties;
  };
  static_assert( sizeof( DisplayModeProperties2KHR ) == sizeof( VkDisplayModeProperties2KHR ), "struct and wrapper have different size!" );

  struct DisplayPlaneInfo2KHR
  {
    DisplayPlaneInfo2KHR( DisplayModeKHR mode_ = DisplayModeKHR(),
                          uint32_t planeIndex_ = 0 )
      : mode( mode_ )
      , planeIndex( planeIndex_ )
    {
    }

    DisplayPlaneInfo2KHR( VkDisplayPlaneInfo2KHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( DisplayPlaneInfo2KHR ) );
    }

    DisplayPlaneInfo2KHR& operator=( VkDisplayPlaneInfo2KHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( DisplayPlaneInfo2KHR ) );
      return *this;
    }
    DisplayPlaneInfo2KHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DisplayPlaneInfo2KHR& setMode( DisplayModeKHR mode_ )
    {
      mode = mode_;
      return *this;
    }

    DisplayPlaneInfo2KHR& setPlaneIndex( uint32_t planeIndex_ )
    {
      planeIndex = planeIndex_;
      return *this;
    }

    operator VkDisplayPlaneInfo2KHR const&() const
    {
      return *reinterpret_cast<const VkDisplayPlaneInfo2KHR*>(this);
    }

    operator VkDisplayPlaneInfo2KHR &()
    {
      return *reinterpret_cast<VkDisplayPlaneInfo2KHR*>(this);
    }

    bool operator==( DisplayPlaneInfo2KHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( mode == rhs.mode )
          && ( planeIndex == rhs.planeIndex );
    }

    bool operator!=( DisplayPlaneInfo2KHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDisplayPlaneInfo2KHR;

  public:
    const void* pNext = nullptr;
    DisplayModeKHR mode;
    uint32_t planeIndex;
  };
  static_assert( sizeof( DisplayPlaneInfo2KHR ) == sizeof( VkDisplayPlaneInfo2KHR ), "struct and wrapper have different size!" );

  struct PhysicalDevice16BitStorageFeatures
  {
    PhysicalDevice16BitStorageFeatures( Bool32 storageBuffer16BitAccess_ = 0,
                                        Bool32 uniformAndStorageBuffer16BitAccess_ = 0,
                                        Bool32 storagePushConstant16_ = 0,
                                        Bool32 storageInputOutput16_ = 0 )
      : storageBuffer16BitAccess( storageBuffer16BitAccess_ )
      , uniformAndStorageBuffer16BitAccess( uniformAndStorageBuffer16BitAccess_ )
      , storagePushConstant16( storagePushConstant16_ )
      , storageInputOutput16( storageInputOutput16_ )
    {
    }

    PhysicalDevice16BitStorageFeatures( VkPhysicalDevice16BitStorageFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDevice16BitStorageFeatures ) );
    }

    PhysicalDevice16BitStorageFeatures& operator=( VkPhysicalDevice16BitStorageFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDevice16BitStorageFeatures ) );
      return *this;
    }
    PhysicalDevice16BitStorageFeatures& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDevice16BitStorageFeatures& setStorageBuffer16BitAccess( Bool32 storageBuffer16BitAccess_ )
    {
      storageBuffer16BitAccess = storageBuffer16BitAccess_;
      return *this;
    }

    PhysicalDevice16BitStorageFeatures& setUniformAndStorageBuffer16BitAccess( Bool32 uniformAndStorageBuffer16BitAccess_ )
    {
      uniformAndStorageBuffer16BitAccess = uniformAndStorageBuffer16BitAccess_;
      return *this;
    }

    PhysicalDevice16BitStorageFeatures& setStoragePushConstant16( Bool32 storagePushConstant16_ )
    {
      storagePushConstant16 = storagePushConstant16_;
      return *this;
    }

    PhysicalDevice16BitStorageFeatures& setStorageInputOutput16( Bool32 storageInputOutput16_ )
    {
      storageInputOutput16 = storageInputOutput16_;
      return *this;
    }

    operator VkPhysicalDevice16BitStorageFeatures const&() const
    {
      return *reinterpret_cast<const VkPhysicalDevice16BitStorageFeatures*>(this);
    }

    operator VkPhysicalDevice16BitStorageFeatures &()
    {
      return *reinterpret_cast<VkPhysicalDevice16BitStorageFeatures*>(this);
    }

    bool operator==( PhysicalDevice16BitStorageFeatures const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( storageBuffer16BitAccess == rhs.storageBuffer16BitAccess )
          && ( uniformAndStorageBuffer16BitAccess == rhs.uniformAndStorageBuffer16BitAccess )
          && ( storagePushConstant16 == rhs.storagePushConstant16 )
          && ( storageInputOutput16 == rhs.storageInputOutput16 );
    }

    bool operator!=( PhysicalDevice16BitStorageFeatures const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDevice16BitStorageFeatures;

  public:
    void* pNext = nullptr;
    Bool32 storageBuffer16BitAccess;
    Bool32 uniformAndStorageBuffer16BitAccess;
    Bool32 storagePushConstant16;
    Bool32 storageInputOutput16;
  };
  static_assert( sizeof( PhysicalDevice16BitStorageFeatures ) == sizeof( VkPhysicalDevice16BitStorageFeatures ), "struct and wrapper have different size!" );

  using PhysicalDevice16BitStorageFeaturesKHR = PhysicalDevice16BitStorageFeatures;

  struct BufferMemoryRequirementsInfo2
  {
    BufferMemoryRequirementsInfo2( Buffer buffer_ = Buffer() )
      : buffer( buffer_ )
    {
    }

    BufferMemoryRequirementsInfo2( VkBufferMemoryRequirementsInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferMemoryRequirementsInfo2 ) );
    }

    BufferMemoryRequirementsInfo2& operator=( VkBufferMemoryRequirementsInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferMemoryRequirementsInfo2 ) );
      return *this;
    }
    BufferMemoryRequirementsInfo2& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BufferMemoryRequirementsInfo2& setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    operator VkBufferMemoryRequirementsInfo2 const&() const
    {
      return *reinterpret_cast<const VkBufferMemoryRequirementsInfo2*>(this);
    }

    operator VkBufferMemoryRequirementsInfo2 &()
    {
      return *reinterpret_cast<VkBufferMemoryRequirementsInfo2*>(this);
    }

    bool operator==( BufferMemoryRequirementsInfo2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( buffer == rhs.buffer );
    }

    bool operator!=( BufferMemoryRequirementsInfo2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBufferMemoryRequirementsInfo2;

  public:
    const void* pNext = nullptr;
    Buffer buffer;
  };
  static_assert( sizeof( BufferMemoryRequirementsInfo2 ) == sizeof( VkBufferMemoryRequirementsInfo2 ), "struct and wrapper have different size!" );

  using BufferMemoryRequirementsInfo2KHR = BufferMemoryRequirementsInfo2;

  struct ImageMemoryRequirementsInfo2
  {
    ImageMemoryRequirementsInfo2( Image image_ = Image() )
      : image( image_ )
    {
    }

    ImageMemoryRequirementsInfo2( VkImageMemoryRequirementsInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageMemoryRequirementsInfo2 ) );
    }

    ImageMemoryRequirementsInfo2& operator=( VkImageMemoryRequirementsInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageMemoryRequirementsInfo2 ) );
      return *this;
    }
    ImageMemoryRequirementsInfo2& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImageMemoryRequirementsInfo2& setImage( Image image_ )
    {
      image = image_;
      return *this;
    }

    operator VkImageMemoryRequirementsInfo2 const&() const
    {
      return *reinterpret_cast<const VkImageMemoryRequirementsInfo2*>(this);
    }

    operator VkImageMemoryRequirementsInfo2 &()
    {
      return *reinterpret_cast<VkImageMemoryRequirementsInfo2*>(this);
    }

    bool operator==( ImageMemoryRequirementsInfo2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( image == rhs.image );
    }

    bool operator!=( ImageMemoryRequirementsInfo2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageMemoryRequirementsInfo2;

  public:
    const void* pNext = nullptr;
    Image image;
  };
  static_assert( sizeof( ImageMemoryRequirementsInfo2 ) == sizeof( VkImageMemoryRequirementsInfo2 ), "struct and wrapper have different size!" );

  using ImageMemoryRequirementsInfo2KHR = ImageMemoryRequirementsInfo2;

  struct ImageSparseMemoryRequirementsInfo2
  {
    ImageSparseMemoryRequirementsInfo2( Image image_ = Image() )
      : image( image_ )
    {
    }

    ImageSparseMemoryRequirementsInfo2( VkImageSparseMemoryRequirementsInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSparseMemoryRequirementsInfo2 ) );
    }

    ImageSparseMemoryRequirementsInfo2& operator=( VkImageSparseMemoryRequirementsInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSparseMemoryRequirementsInfo2 ) );
      return *this;
    }
    ImageSparseMemoryRequirementsInfo2& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImageSparseMemoryRequirementsInfo2& setImage( Image image_ )
    {
      image = image_;
      return *this;
    }

    operator VkImageSparseMemoryRequirementsInfo2 const&() const
    {
      return *reinterpret_cast<const VkImageSparseMemoryRequirementsInfo2*>(this);
    }

    operator VkImageSparseMemoryRequirementsInfo2 &()
    {
      return *reinterpret_cast<VkImageSparseMemoryRequirementsInfo2*>(this);
    }

    bool operator==( ImageSparseMemoryRequirementsInfo2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( image == rhs.image );
    }

    bool operator!=( ImageSparseMemoryRequirementsInfo2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageSparseMemoryRequirementsInfo2;

  public:
    const void* pNext = nullptr;
    Image image;
  };
  static_assert( sizeof( ImageSparseMemoryRequirementsInfo2 ) == sizeof( VkImageSparseMemoryRequirementsInfo2 ), "struct and wrapper have different size!" );

  using ImageSparseMemoryRequirementsInfo2KHR = ImageSparseMemoryRequirementsInfo2;

  struct MemoryRequirements2
  {
    operator VkMemoryRequirements2 const&() const
    {
      return *reinterpret_cast<const VkMemoryRequirements2*>(this);
    }

    operator VkMemoryRequirements2 &()
    {
      return *reinterpret_cast<VkMemoryRequirements2*>(this);
    }

    bool operator==( MemoryRequirements2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( memoryRequirements == rhs.memoryRequirements );
    }

    bool operator!=( MemoryRequirements2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eMemoryRequirements2;

  public:
    void* pNext = nullptr;
    MemoryRequirements memoryRequirements;
  };
  static_assert( sizeof( MemoryRequirements2 ) == sizeof( VkMemoryRequirements2 ), "struct and wrapper have different size!" );

  using MemoryRequirements2KHR = MemoryRequirements2;

  struct MemoryDedicatedRequirements
  {
    operator VkMemoryDedicatedRequirements const&() const
    {
      return *reinterpret_cast<const VkMemoryDedicatedRequirements*>(this);
    }

    operator VkMemoryDedicatedRequirements &()
    {
      return *reinterpret_cast<VkMemoryDedicatedRequirements*>(this);
    }

    bool operator==( MemoryDedicatedRequirements const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( prefersDedicatedAllocation == rhs.prefersDedicatedAllocation )
          && ( requiresDedicatedAllocation == rhs.requiresDedicatedAllocation );
    }

    bool operator!=( MemoryDedicatedRequirements const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eMemoryDedicatedRequirements;

  public:
    void* pNext = nullptr;
    Bool32 prefersDedicatedAllocation;
    Bool32 requiresDedicatedAllocation;
  };
  static_assert( sizeof( MemoryDedicatedRequirements ) == sizeof( VkMemoryDedicatedRequirements ), "struct and wrapper have different size!" );

  using MemoryDedicatedRequirementsKHR = MemoryDedicatedRequirements;

  struct MemoryDedicatedAllocateInfo
  {
    MemoryDedicatedAllocateInfo( Image image_ = Image(),
                                 Buffer buffer_ = Buffer() )
      : image( image_ )
      , buffer( buffer_ )
    {
    }

    MemoryDedicatedAllocateInfo( VkMemoryDedicatedAllocateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( MemoryDedicatedAllocateInfo ) );
    }

    MemoryDedicatedAllocateInfo& operator=( VkMemoryDedicatedAllocateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( MemoryDedicatedAllocateInfo ) );
      return *this;
    }
    MemoryDedicatedAllocateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    MemoryDedicatedAllocateInfo& setImage( Image image_ )
    {
      image = image_;
      return *this;
    }

    MemoryDedicatedAllocateInfo& setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    operator VkMemoryDedicatedAllocateInfo const&() const
    {
      return *reinterpret_cast<const VkMemoryDedicatedAllocateInfo*>(this);
    }

    operator VkMemoryDedicatedAllocateInfo &()
    {
      return *reinterpret_cast<VkMemoryDedicatedAllocateInfo*>(this);
    }

    bool operator==( MemoryDedicatedAllocateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( image == rhs.image )
          && ( buffer == rhs.buffer );
    }

    bool operator!=( MemoryDedicatedAllocateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eMemoryDedicatedAllocateInfo;

  public:
    const void* pNext = nullptr;
    Image image;
    Buffer buffer;
  };
  static_assert( sizeof( MemoryDedicatedAllocateInfo ) == sizeof( VkMemoryDedicatedAllocateInfo ), "struct and wrapper have different size!" );

  using MemoryDedicatedAllocateInfoKHR = MemoryDedicatedAllocateInfo;

  struct SamplerYcbcrConversionInfo
  {
    SamplerYcbcrConversionInfo( SamplerYcbcrConversion conversion_ = SamplerYcbcrConversion() )
      : conversion( conversion_ )
    {
    }

    SamplerYcbcrConversionInfo( VkSamplerYcbcrConversionInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SamplerYcbcrConversionInfo ) );
    }

    SamplerYcbcrConversionInfo& operator=( VkSamplerYcbcrConversionInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SamplerYcbcrConversionInfo ) );
      return *this;
    }
    SamplerYcbcrConversionInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    SamplerYcbcrConversionInfo& setConversion( SamplerYcbcrConversion conversion_ )
    {
      conversion = conversion_;
      return *this;
    }

    operator VkSamplerYcbcrConversionInfo const&() const
    {
      return *reinterpret_cast<const VkSamplerYcbcrConversionInfo*>(this);
    }

    operator VkSamplerYcbcrConversionInfo &()
    {
      return *reinterpret_cast<VkSamplerYcbcrConversionInfo*>(this);
    }

    bool operator==( SamplerYcbcrConversionInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( conversion == rhs.conversion );
    }

    bool operator!=( SamplerYcbcrConversionInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eSamplerYcbcrConversionInfo;

  public:
    const void* pNext = nullptr;
    SamplerYcbcrConversion conversion;
  };
  static_assert( sizeof( SamplerYcbcrConversionInfo ) == sizeof( VkSamplerYcbcrConversionInfo ), "struct and wrapper have different size!" );

  using SamplerYcbcrConversionInfoKHR = SamplerYcbcrConversionInfo;

  struct PhysicalDeviceSamplerYcbcrConversionFeatures
  {
    PhysicalDeviceSamplerYcbcrConversionFeatures( Bool32 samplerYcbcrConversion_ = 0 )
      : samplerYcbcrConversion( samplerYcbcrConversion_ )
    {
    }

    PhysicalDeviceSamplerYcbcrConversionFeatures( VkPhysicalDeviceSamplerYcbcrConversionFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceSamplerYcbcrConversionFeatures ) );
    }

    PhysicalDeviceSamplerYcbcrConversionFeatures& operator=( VkPhysicalDeviceSamplerYcbcrConversionFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceSamplerYcbcrConversionFeatures ) );
      return *this;
    }
    PhysicalDeviceSamplerYcbcrConversionFeatures& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceSamplerYcbcrConversionFeatures& setSamplerYcbcrConversion( Bool32 samplerYcbcrConversion_ )
    {
      samplerYcbcrConversion = samplerYcbcrConversion_;
      return *this;
    }

    operator VkPhysicalDeviceSamplerYcbcrConversionFeatures const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceSamplerYcbcrConversionFeatures*>(this);
    }

    operator VkPhysicalDeviceSamplerYcbcrConversionFeatures &()
    {
      return *reinterpret_cast<VkPhysicalDeviceSamplerYcbcrConversionFeatures*>(this);
    }

    bool operator==( PhysicalDeviceSamplerYcbcrConversionFeatures const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( samplerYcbcrConversion == rhs.samplerYcbcrConversion );
    }

    bool operator!=( PhysicalDeviceSamplerYcbcrConversionFeatures const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceSamplerYcbcrConversionFeatures;

  public:
    void* pNext = nullptr;
    Bool32 samplerYcbcrConversion;
  };
  static_assert( sizeof( PhysicalDeviceSamplerYcbcrConversionFeatures ) == sizeof( VkPhysicalDeviceSamplerYcbcrConversionFeatures ), "struct and wrapper have different size!" );

  using PhysicalDeviceSamplerYcbcrConversionFeaturesKHR = PhysicalDeviceSamplerYcbcrConversionFeatures;

  struct SamplerYcbcrConversionImageFormatProperties
  {
    operator VkSamplerYcbcrConversionImageFormatProperties const&() const
    {
      return *reinterpret_cast<const VkSamplerYcbcrConversionImageFormatProperties*>(this);
    }

    operator VkSamplerYcbcrConversionImageFormatProperties &()
    {
      return *reinterpret_cast<VkSamplerYcbcrConversionImageFormatProperties*>(this);
    }

    bool operator==( SamplerYcbcrConversionImageFormatProperties const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( combinedImageSamplerDescriptorCount == rhs.combinedImageSamplerDescriptorCount );
    }

    bool operator!=( SamplerYcbcrConversionImageFormatProperties const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eSamplerYcbcrConversionImageFormatProperties;

  public:
    void* pNext = nullptr;
    uint32_t combinedImageSamplerDescriptorCount;
  };
  static_assert( sizeof( SamplerYcbcrConversionImageFormatProperties ) == sizeof( VkSamplerYcbcrConversionImageFormatProperties ), "struct and wrapper have different size!" );

  using SamplerYcbcrConversionImageFormatPropertiesKHR = SamplerYcbcrConversionImageFormatProperties;

  struct TextureLODGatherFormatPropertiesAMD
  {
    operator VkTextureLODGatherFormatPropertiesAMD const&() const
    {
      return *reinterpret_cast<const VkTextureLODGatherFormatPropertiesAMD*>(this);
    }

    operator VkTextureLODGatherFormatPropertiesAMD &()
    {
      return *reinterpret_cast<VkTextureLODGatherFormatPropertiesAMD*>(this);
    }

    bool operator==( TextureLODGatherFormatPropertiesAMD const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( supportsTextureGatherLODBiasAMD == rhs.supportsTextureGatherLODBiasAMD );
    }

    bool operator!=( TextureLODGatherFormatPropertiesAMD const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eTextureLodGatherFormatPropertiesAMD;

  public:
    void* pNext = nullptr;
    Bool32 supportsTextureGatherLODBiasAMD;
  };
  static_assert( sizeof( TextureLODGatherFormatPropertiesAMD ) == sizeof( VkTextureLODGatherFormatPropertiesAMD ), "struct and wrapper have different size!" );

  struct ProtectedSubmitInfo
  {
    ProtectedSubmitInfo( Bool32 protectedSubmit_ = 0 )
      : protectedSubmit( protectedSubmit_ )
    {
    }

    ProtectedSubmitInfo( VkProtectedSubmitInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ProtectedSubmitInfo ) );
    }

    ProtectedSubmitInfo& operator=( VkProtectedSubmitInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ProtectedSubmitInfo ) );
      return *this;
    }
    ProtectedSubmitInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ProtectedSubmitInfo& setProtectedSubmit( Bool32 protectedSubmit_ )
    {
      protectedSubmit = protectedSubmit_;
      return *this;
    }

    operator VkProtectedSubmitInfo const&() const
    {
      return *reinterpret_cast<const VkProtectedSubmitInfo*>(this);
    }

    operator VkProtectedSubmitInfo &()
    {
      return *reinterpret_cast<VkProtectedSubmitInfo*>(this);
    }

    bool operator==( ProtectedSubmitInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( protectedSubmit == rhs.protectedSubmit );
    }

    bool operator!=( ProtectedSubmitInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eProtectedSubmitInfo;

  public:
    const void* pNext = nullptr;
    Bool32 protectedSubmit;
  };
  static_assert( sizeof( ProtectedSubmitInfo ) == sizeof( VkProtectedSubmitInfo ), "struct and wrapper have different size!" );

  struct PhysicalDeviceProtectedMemoryFeatures
  {
    PhysicalDeviceProtectedMemoryFeatures( Bool32 protectedMemory_ = 0 )
      : protectedMemory( protectedMemory_ )
    {
    }

    PhysicalDeviceProtectedMemoryFeatures( VkPhysicalDeviceProtectedMemoryFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceProtectedMemoryFeatures ) );
    }

    PhysicalDeviceProtectedMemoryFeatures& operator=( VkPhysicalDeviceProtectedMemoryFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceProtectedMemoryFeatures ) );
      return *this;
    }
    PhysicalDeviceProtectedMemoryFeatures& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceProtectedMemoryFeatures& setProtectedMemory( Bool32 protectedMemory_ )
    {
      protectedMemory = protectedMemory_;
      return *this;
    }

    operator VkPhysicalDeviceProtectedMemoryFeatures const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceProtectedMemoryFeatures*>(this);
    }

    operator VkPhysicalDeviceProtectedMemoryFeatures &()
    {
      return *reinterpret_cast<VkPhysicalDeviceProtectedMemoryFeatures*>(this);
    }

    bool operator==( PhysicalDeviceProtectedMemoryFeatures const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( protectedMemory == rhs.protectedMemory );
    }

    bool operator!=( PhysicalDeviceProtectedMemoryFeatures const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceProtectedMemoryFeatures;

  public:
    void* pNext = nullptr;
    Bool32 protectedMemory;
  };
  static_assert( sizeof( PhysicalDeviceProtectedMemoryFeatures ) == sizeof( VkPhysicalDeviceProtectedMemoryFeatures ), "struct and wrapper have different size!" );

  struct PhysicalDeviceProtectedMemoryProperties
  {
    PhysicalDeviceProtectedMemoryProperties( Bool32 protectedNoFault_ = 0 )
      : protectedNoFault( protectedNoFault_ )
    {
    }

    PhysicalDeviceProtectedMemoryProperties( VkPhysicalDeviceProtectedMemoryProperties const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceProtectedMemoryProperties ) );
    }

    PhysicalDeviceProtectedMemoryProperties& operator=( VkPhysicalDeviceProtectedMemoryProperties const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceProtectedMemoryProperties ) );
      return *this;
    }
    PhysicalDeviceProtectedMemoryProperties& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceProtectedMemoryProperties& setProtectedNoFault( Bool32 protectedNoFault_ )
    {
      protectedNoFault = protectedNoFault_;
      return *this;
    }

    operator VkPhysicalDeviceProtectedMemoryProperties const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceProtectedMemoryProperties*>(this);
    }

    operator VkPhysicalDeviceProtectedMemoryProperties &()
    {
      return *reinterpret_cast<VkPhysicalDeviceProtectedMemoryProperties*>(this);
    }

    bool operator==( PhysicalDeviceProtectedMemoryProperties const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( protectedNoFault == rhs.protectedNoFault );
    }

    bool operator!=( PhysicalDeviceProtectedMemoryProperties const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceProtectedMemoryProperties;

  public:
    void* pNext = nullptr;
    Bool32 protectedNoFault;
  };
  static_assert( sizeof( PhysicalDeviceProtectedMemoryProperties ) == sizeof( VkPhysicalDeviceProtectedMemoryProperties ), "struct and wrapper have different size!" );

  struct PipelineCoverageToColorStateCreateInfoNV
  {
    PipelineCoverageToColorStateCreateInfoNV( PipelineCoverageToColorStateCreateFlagsNV flags_ = PipelineCoverageToColorStateCreateFlagsNV(),
                                              Bool32 coverageToColorEnable_ = 0,
                                              uint32_t coverageToColorLocation_ = 0 )
      : flags( flags_ )
      , coverageToColorEnable( coverageToColorEnable_ )
      , coverageToColorLocation( coverageToColorLocation_ )
    {
    }

    PipelineCoverageToColorStateCreateInfoNV( VkPipelineCoverageToColorStateCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineCoverageToColorStateCreateInfoNV ) );
    }

    PipelineCoverageToColorStateCreateInfoNV& operator=( VkPipelineCoverageToColorStateCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineCoverageToColorStateCreateInfoNV ) );
      return *this;
    }
    PipelineCoverageToColorStateCreateInfoNV& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineCoverageToColorStateCreateInfoNV& setFlags( PipelineCoverageToColorStateCreateFlagsNV flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineCoverageToColorStateCreateInfoNV& setCoverageToColorEnable( Bool32 coverageToColorEnable_ )
    {
      coverageToColorEnable = coverageToColorEnable_;
      return *this;
    }

    PipelineCoverageToColorStateCreateInfoNV& setCoverageToColorLocation( uint32_t coverageToColorLocation_ )
    {
      coverageToColorLocation = coverageToColorLocation_;
      return *this;
    }

    operator VkPipelineCoverageToColorStateCreateInfoNV const&() const
    {
      return *reinterpret_cast<const VkPipelineCoverageToColorStateCreateInfoNV*>(this);
    }

    operator VkPipelineCoverageToColorStateCreateInfoNV &()
    {
      return *reinterpret_cast<VkPipelineCoverageToColorStateCreateInfoNV*>(this);
    }

    bool operator==( PipelineCoverageToColorStateCreateInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( coverageToColorEnable == rhs.coverageToColorEnable )
          && ( coverageToColorLocation == rhs.coverageToColorLocation );
    }

    bool operator!=( PipelineCoverageToColorStateCreateInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineCoverageToColorStateCreateInfoNV;

  public:
    const void* pNext = nullptr;
    PipelineCoverageToColorStateCreateFlagsNV flags;
    Bool32 coverageToColorEnable;
    uint32_t coverageToColorLocation;
  };
  static_assert( sizeof( PipelineCoverageToColorStateCreateInfoNV ) == sizeof( VkPipelineCoverageToColorStateCreateInfoNV ), "struct and wrapper have different size!" );

  struct PhysicalDeviceSamplerFilterMinmaxPropertiesEXT
  {
    operator VkPhysicalDeviceSamplerFilterMinmaxPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceSamplerFilterMinmaxPropertiesEXT*>(this);
    }

    operator VkPhysicalDeviceSamplerFilterMinmaxPropertiesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceSamplerFilterMinmaxPropertiesEXT*>(this);
    }

    bool operator==( PhysicalDeviceSamplerFilterMinmaxPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( filterMinmaxSingleComponentFormats == rhs.filterMinmaxSingleComponentFormats )
          && ( filterMinmaxImageComponentMapping == rhs.filterMinmaxImageComponentMapping );
    }

    bool operator!=( PhysicalDeviceSamplerFilterMinmaxPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceSamplerFilterMinmaxPropertiesEXT;

  public:
    void* pNext = nullptr;
    Bool32 filterMinmaxSingleComponentFormats;
    Bool32 filterMinmaxImageComponentMapping;
  };
  static_assert( sizeof( PhysicalDeviceSamplerFilterMinmaxPropertiesEXT ) == sizeof( VkPhysicalDeviceSamplerFilterMinmaxPropertiesEXT ), "struct and wrapper have different size!" );

  struct MultisamplePropertiesEXT
  {
    operator VkMultisamplePropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkMultisamplePropertiesEXT*>(this);
    }

    operator VkMultisamplePropertiesEXT &()
    {
      return *reinterpret_cast<VkMultisamplePropertiesEXT*>(this);
    }

    bool operator==( MultisamplePropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxSampleLocationGridSize == rhs.maxSampleLocationGridSize );
    }

    bool operator!=( MultisamplePropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eMultisamplePropertiesEXT;

  public:
    void* pNext = nullptr;
    Extent2D maxSampleLocationGridSize;
  };
  static_assert( sizeof( MultisamplePropertiesEXT ) == sizeof( VkMultisamplePropertiesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceBlendOperationAdvancedFeaturesEXT
  {
    PhysicalDeviceBlendOperationAdvancedFeaturesEXT( Bool32 advancedBlendCoherentOperations_ = 0 )
      : advancedBlendCoherentOperations( advancedBlendCoherentOperations_ )
    {
    }

    PhysicalDeviceBlendOperationAdvancedFeaturesEXT( VkPhysicalDeviceBlendOperationAdvancedFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceBlendOperationAdvancedFeaturesEXT ) );
    }

    PhysicalDeviceBlendOperationAdvancedFeaturesEXT& operator=( VkPhysicalDeviceBlendOperationAdvancedFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceBlendOperationAdvancedFeaturesEXT ) );
      return *this;
    }
    PhysicalDeviceBlendOperationAdvancedFeaturesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceBlendOperationAdvancedFeaturesEXT& setAdvancedBlendCoherentOperations( Bool32 advancedBlendCoherentOperations_ )
    {
      advancedBlendCoherentOperations = advancedBlendCoherentOperations_;
      return *this;
    }

    operator VkPhysicalDeviceBlendOperationAdvancedFeaturesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceBlendOperationAdvancedFeaturesEXT*>(this);
    }

    operator VkPhysicalDeviceBlendOperationAdvancedFeaturesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceBlendOperationAdvancedFeaturesEXT*>(this);
    }

    bool operator==( PhysicalDeviceBlendOperationAdvancedFeaturesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( advancedBlendCoherentOperations == rhs.advancedBlendCoherentOperations );
    }

    bool operator!=( PhysicalDeviceBlendOperationAdvancedFeaturesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceBlendOperationAdvancedFeaturesEXT;

  public:
    void* pNext = nullptr;
    Bool32 advancedBlendCoherentOperations;
  };
  static_assert( sizeof( PhysicalDeviceBlendOperationAdvancedFeaturesEXT ) == sizeof( VkPhysicalDeviceBlendOperationAdvancedFeaturesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceBlendOperationAdvancedPropertiesEXT
  {
    operator VkPhysicalDeviceBlendOperationAdvancedPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceBlendOperationAdvancedPropertiesEXT*>(this);
    }

    operator VkPhysicalDeviceBlendOperationAdvancedPropertiesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceBlendOperationAdvancedPropertiesEXT*>(this);
    }

    bool operator==( PhysicalDeviceBlendOperationAdvancedPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( advancedBlendMaxColorAttachments == rhs.advancedBlendMaxColorAttachments )
          && ( advancedBlendIndependentBlend == rhs.advancedBlendIndependentBlend )
          && ( advancedBlendNonPremultipliedSrcColor == rhs.advancedBlendNonPremultipliedSrcColor )
          && ( advancedBlendNonPremultipliedDstColor == rhs.advancedBlendNonPremultipliedDstColor )
          && ( advancedBlendCorrelatedOverlap == rhs.advancedBlendCorrelatedOverlap )
          && ( advancedBlendAllOperations == rhs.advancedBlendAllOperations );
    }

    bool operator!=( PhysicalDeviceBlendOperationAdvancedPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceBlendOperationAdvancedPropertiesEXT;

  public:
    void* pNext = nullptr;
    uint32_t advancedBlendMaxColorAttachments;
    Bool32 advancedBlendIndependentBlend;
    Bool32 advancedBlendNonPremultipliedSrcColor;
    Bool32 advancedBlendNonPremultipliedDstColor;
    Bool32 advancedBlendCorrelatedOverlap;
    Bool32 advancedBlendAllOperations;
  };
  static_assert( sizeof( PhysicalDeviceBlendOperationAdvancedPropertiesEXT ) == sizeof( VkPhysicalDeviceBlendOperationAdvancedPropertiesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceInlineUniformBlockFeaturesEXT
  {
    PhysicalDeviceInlineUniformBlockFeaturesEXT( Bool32 inlineUniformBlock_ = 0,
                                                 Bool32 descriptorBindingInlineUniformBlockUpdateAfterBind_ = 0 )
      : inlineUniformBlock( inlineUniformBlock_ )
      , descriptorBindingInlineUniformBlockUpdateAfterBind( descriptorBindingInlineUniformBlockUpdateAfterBind_ )
    {
    }

    PhysicalDeviceInlineUniformBlockFeaturesEXT( VkPhysicalDeviceInlineUniformBlockFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceInlineUniformBlockFeaturesEXT ) );
    }

    PhysicalDeviceInlineUniformBlockFeaturesEXT& operator=( VkPhysicalDeviceInlineUniformBlockFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceInlineUniformBlockFeaturesEXT ) );
      return *this;
    }
    PhysicalDeviceInlineUniformBlockFeaturesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceInlineUniformBlockFeaturesEXT& setInlineUniformBlock( Bool32 inlineUniformBlock_ )
    {
      inlineUniformBlock = inlineUniformBlock_;
      return *this;
    }

    PhysicalDeviceInlineUniformBlockFeaturesEXT& setDescriptorBindingInlineUniformBlockUpdateAfterBind( Bool32 descriptorBindingInlineUniformBlockUpdateAfterBind_ )
    {
      descriptorBindingInlineUniformBlockUpdateAfterBind = descriptorBindingInlineUniformBlockUpdateAfterBind_;
      return *this;
    }

    operator VkPhysicalDeviceInlineUniformBlockFeaturesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceInlineUniformBlockFeaturesEXT*>(this);
    }

    operator VkPhysicalDeviceInlineUniformBlockFeaturesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceInlineUniformBlockFeaturesEXT*>(this);
    }

    bool operator==( PhysicalDeviceInlineUniformBlockFeaturesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( inlineUniformBlock == rhs.inlineUniformBlock )
          && ( descriptorBindingInlineUniformBlockUpdateAfterBind == rhs.descriptorBindingInlineUniformBlockUpdateAfterBind );
    }

    bool operator!=( PhysicalDeviceInlineUniformBlockFeaturesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceInlineUniformBlockFeaturesEXT;

  public:
    void* pNext = nullptr;
    Bool32 inlineUniformBlock;
    Bool32 descriptorBindingInlineUniformBlockUpdateAfterBind;
  };
  static_assert( sizeof( PhysicalDeviceInlineUniformBlockFeaturesEXT ) == sizeof( VkPhysicalDeviceInlineUniformBlockFeaturesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceInlineUniformBlockPropertiesEXT
  {
    operator VkPhysicalDeviceInlineUniformBlockPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceInlineUniformBlockPropertiesEXT*>(this);
    }

    operator VkPhysicalDeviceInlineUniformBlockPropertiesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceInlineUniformBlockPropertiesEXT*>(this);
    }

    bool operator==( PhysicalDeviceInlineUniformBlockPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxInlineUniformBlockSize == rhs.maxInlineUniformBlockSize )
          && ( maxPerStageDescriptorInlineUniformBlocks == rhs.maxPerStageDescriptorInlineUniformBlocks )
          && ( maxPerStageDescriptorUpdateAfterBindInlineUniformBlocks == rhs.maxPerStageDescriptorUpdateAfterBindInlineUniformBlocks )
          && ( maxDescriptorSetInlineUniformBlocks == rhs.maxDescriptorSetInlineUniformBlocks )
          && ( maxDescriptorSetUpdateAfterBindInlineUniformBlocks == rhs.maxDescriptorSetUpdateAfterBindInlineUniformBlocks );
    }

    bool operator!=( PhysicalDeviceInlineUniformBlockPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceInlineUniformBlockPropertiesEXT;

  public:
    void* pNext = nullptr;
    uint32_t maxInlineUniformBlockSize;
    uint32_t maxPerStageDescriptorInlineUniformBlocks;
    uint32_t maxPerStageDescriptorUpdateAfterBindInlineUniformBlocks;
    uint32_t maxDescriptorSetInlineUniformBlocks;
    uint32_t maxDescriptorSetUpdateAfterBindInlineUniformBlocks;
  };
  static_assert( sizeof( PhysicalDeviceInlineUniformBlockPropertiesEXT ) == sizeof( VkPhysicalDeviceInlineUniformBlockPropertiesEXT ), "struct and wrapper have different size!" );

  struct WriteDescriptorSetInlineUniformBlockEXT
  {
    WriteDescriptorSetInlineUniformBlockEXT( uint32_t dataSize_ = 0,
                                             const void* pData_ = nullptr )
      : dataSize( dataSize_ )
      , pData( pData_ )
    {
    }

    WriteDescriptorSetInlineUniformBlockEXT( VkWriteDescriptorSetInlineUniformBlockEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( WriteDescriptorSetInlineUniformBlockEXT ) );
    }

    WriteDescriptorSetInlineUniformBlockEXT& operator=( VkWriteDescriptorSetInlineUniformBlockEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( WriteDescriptorSetInlineUniformBlockEXT ) );
      return *this;
    }
    WriteDescriptorSetInlineUniformBlockEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    WriteDescriptorSetInlineUniformBlockEXT& setDataSize( uint32_t dataSize_ )
    {
      dataSize = dataSize_;
      return *this;
    }

    WriteDescriptorSetInlineUniformBlockEXT& setPData( const void* pData_ )
    {
      pData = pData_;
      return *this;
    }

    operator VkWriteDescriptorSetInlineUniformBlockEXT const&() const
    {
      return *reinterpret_cast<const VkWriteDescriptorSetInlineUniformBlockEXT*>(this);
    }

    operator VkWriteDescriptorSetInlineUniformBlockEXT &()
    {
      return *reinterpret_cast<VkWriteDescriptorSetInlineUniformBlockEXT*>(this);
    }

    bool operator==( WriteDescriptorSetInlineUniformBlockEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( dataSize == rhs.dataSize )
          && ( pData == rhs.pData );
    }

    bool operator!=( WriteDescriptorSetInlineUniformBlockEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eWriteDescriptorSetInlineUniformBlockEXT;

  public:
    const void* pNext = nullptr;
    uint32_t dataSize;
    const void* pData;
  };
  static_assert( sizeof( WriteDescriptorSetInlineUniformBlockEXT ) == sizeof( VkWriteDescriptorSetInlineUniformBlockEXT ), "struct and wrapper have different size!" );

  struct DescriptorPoolInlineUniformBlockCreateInfoEXT
  {
    DescriptorPoolInlineUniformBlockCreateInfoEXT( uint32_t maxInlineUniformBlockBindings_ = 0 )
      : maxInlineUniformBlockBindings( maxInlineUniformBlockBindings_ )
    {
    }

    DescriptorPoolInlineUniformBlockCreateInfoEXT( VkDescriptorPoolInlineUniformBlockCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorPoolInlineUniformBlockCreateInfoEXT ) );
    }

    DescriptorPoolInlineUniformBlockCreateInfoEXT& operator=( VkDescriptorPoolInlineUniformBlockCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorPoolInlineUniformBlockCreateInfoEXT ) );
      return *this;
    }
    DescriptorPoolInlineUniformBlockCreateInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DescriptorPoolInlineUniformBlockCreateInfoEXT& setMaxInlineUniformBlockBindings( uint32_t maxInlineUniformBlockBindings_ )
    {
      maxInlineUniformBlockBindings = maxInlineUniformBlockBindings_;
      return *this;
    }

    operator VkDescriptorPoolInlineUniformBlockCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDescriptorPoolInlineUniformBlockCreateInfoEXT*>(this);
    }

    operator VkDescriptorPoolInlineUniformBlockCreateInfoEXT &()
    {
      return *reinterpret_cast<VkDescriptorPoolInlineUniformBlockCreateInfoEXT*>(this);
    }

    bool operator==( DescriptorPoolInlineUniformBlockCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxInlineUniformBlockBindings == rhs.maxInlineUniformBlockBindings );
    }

    bool operator!=( DescriptorPoolInlineUniformBlockCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDescriptorPoolInlineUniformBlockCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    uint32_t maxInlineUniformBlockBindings;
  };
  static_assert( sizeof( DescriptorPoolInlineUniformBlockCreateInfoEXT ) == sizeof( VkDescriptorPoolInlineUniformBlockCreateInfoEXT ), "struct and wrapper have different size!" );

  struct ImageFormatListCreateInfoKHR
  {
    ImageFormatListCreateInfoKHR( uint32_t viewFormatCount_ = 0,
                                  const Format* pViewFormats_ = nullptr )
      : viewFormatCount( viewFormatCount_ )
      , pViewFormats( pViewFormats_ )
    {
    }

    ImageFormatListCreateInfoKHR( VkImageFormatListCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageFormatListCreateInfoKHR ) );
    }

    ImageFormatListCreateInfoKHR& operator=( VkImageFormatListCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageFormatListCreateInfoKHR ) );
      return *this;
    }
    ImageFormatListCreateInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImageFormatListCreateInfoKHR& setViewFormatCount( uint32_t viewFormatCount_ )
    {
      viewFormatCount = viewFormatCount_;
      return *this;
    }

    ImageFormatListCreateInfoKHR& setPViewFormats( const Format* pViewFormats_ )
    {
      pViewFormats = pViewFormats_;
      return *this;
    }

    operator VkImageFormatListCreateInfoKHR const&() const
    {
      return *reinterpret_cast<const VkImageFormatListCreateInfoKHR*>(this);
    }

    operator VkImageFormatListCreateInfoKHR &()
    {
      return *reinterpret_cast<VkImageFormatListCreateInfoKHR*>(this);
    }

    bool operator==( ImageFormatListCreateInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( viewFormatCount == rhs.viewFormatCount )
          && ( pViewFormats == rhs.pViewFormats );
    }

    bool operator!=( ImageFormatListCreateInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageFormatListCreateInfoKHR;

  public:
    const void* pNext = nullptr;
    uint32_t viewFormatCount;
    const Format* pViewFormats;
  };
  static_assert( sizeof( ImageFormatListCreateInfoKHR ) == sizeof( VkImageFormatListCreateInfoKHR ), "struct and wrapper have different size!" );

  struct ValidationCacheCreateInfoEXT
  {
    ValidationCacheCreateInfoEXT( ValidationCacheCreateFlagsEXT flags_ = ValidationCacheCreateFlagsEXT(),
                                  size_t initialDataSize_ = 0,
                                  const void* pInitialData_ = nullptr )
      : flags( flags_ )
      , initialDataSize( initialDataSize_ )
      , pInitialData( pInitialData_ )
    {
    }

    ValidationCacheCreateInfoEXT( VkValidationCacheCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ValidationCacheCreateInfoEXT ) );
    }

    ValidationCacheCreateInfoEXT& operator=( VkValidationCacheCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ValidationCacheCreateInfoEXT ) );
      return *this;
    }
    ValidationCacheCreateInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ValidationCacheCreateInfoEXT& setFlags( ValidationCacheCreateFlagsEXT flags_ )
    {
      flags = flags_;
      return *this;
    }

    ValidationCacheCreateInfoEXT& setInitialDataSize( size_t initialDataSize_ )
    {
      initialDataSize = initialDataSize_;
      return *this;
    }

    ValidationCacheCreateInfoEXT& setPInitialData( const void* pInitialData_ )
    {
      pInitialData = pInitialData_;
      return *this;
    }

    operator VkValidationCacheCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkValidationCacheCreateInfoEXT*>(this);
    }

    operator VkValidationCacheCreateInfoEXT &()
    {
      return *reinterpret_cast<VkValidationCacheCreateInfoEXT*>(this);
    }

    bool operator==( ValidationCacheCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( initialDataSize == rhs.initialDataSize )
          && ( pInitialData == rhs.pInitialData );
    }

    bool operator!=( ValidationCacheCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eValidationCacheCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    ValidationCacheCreateFlagsEXT flags;
    size_t initialDataSize;
    const void* pInitialData;
  };
  static_assert( sizeof( ValidationCacheCreateInfoEXT ) == sizeof( VkValidationCacheCreateInfoEXT ), "struct and wrapper have different size!" );

  struct ShaderModuleValidationCacheCreateInfoEXT
  {
    ShaderModuleValidationCacheCreateInfoEXT( ValidationCacheEXT validationCache_ = ValidationCacheEXT() )
      : validationCache( validationCache_ )
    {
    }

    ShaderModuleValidationCacheCreateInfoEXT( VkShaderModuleValidationCacheCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ShaderModuleValidationCacheCreateInfoEXT ) );
    }

    ShaderModuleValidationCacheCreateInfoEXT& operator=( VkShaderModuleValidationCacheCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ShaderModuleValidationCacheCreateInfoEXT ) );
      return *this;
    }
    ShaderModuleValidationCacheCreateInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ShaderModuleValidationCacheCreateInfoEXT& setValidationCache( ValidationCacheEXT validationCache_ )
    {
      validationCache = validationCache_;
      return *this;
    }

    operator VkShaderModuleValidationCacheCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkShaderModuleValidationCacheCreateInfoEXT*>(this);
    }

    operator VkShaderModuleValidationCacheCreateInfoEXT &()
    {
      return *reinterpret_cast<VkShaderModuleValidationCacheCreateInfoEXT*>(this);
    }

    bool operator==( ShaderModuleValidationCacheCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( validationCache == rhs.validationCache );
    }

    bool operator!=( ShaderModuleValidationCacheCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eShaderModuleValidationCacheCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    ValidationCacheEXT validationCache;
  };
  static_assert( sizeof( ShaderModuleValidationCacheCreateInfoEXT ) == sizeof( VkShaderModuleValidationCacheCreateInfoEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceMaintenance3Properties
  {
    operator VkPhysicalDeviceMaintenance3Properties const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceMaintenance3Properties*>(this);
    }

    operator VkPhysicalDeviceMaintenance3Properties &()
    {
      return *reinterpret_cast<VkPhysicalDeviceMaintenance3Properties*>(this);
    }

    bool operator==( PhysicalDeviceMaintenance3Properties const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxPerSetDescriptors == rhs.maxPerSetDescriptors )
          && ( maxMemoryAllocationSize == rhs.maxMemoryAllocationSize );
    }

    bool operator!=( PhysicalDeviceMaintenance3Properties const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceMaintenance3Properties;

  public:
    void* pNext = nullptr;
    uint32_t maxPerSetDescriptors;
    DeviceSize maxMemoryAllocationSize;
  };
  static_assert( sizeof( PhysicalDeviceMaintenance3Properties ) == sizeof( VkPhysicalDeviceMaintenance3Properties ), "struct and wrapper have different size!" );

  using PhysicalDeviceMaintenance3PropertiesKHR = PhysicalDeviceMaintenance3Properties;

  struct DescriptorSetLayoutSupport
  {
    operator VkDescriptorSetLayoutSupport const&() const
    {
      return *reinterpret_cast<const VkDescriptorSetLayoutSupport*>(this);
    }

    operator VkDescriptorSetLayoutSupport &()
    {
      return *reinterpret_cast<VkDescriptorSetLayoutSupport*>(this);
    }

    bool operator==( DescriptorSetLayoutSupport const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( supported == rhs.supported );
    }

    bool operator!=( DescriptorSetLayoutSupport const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDescriptorSetLayoutSupport;

  public:
    void* pNext = nullptr;
    Bool32 supported;
  };
  static_assert( sizeof( DescriptorSetLayoutSupport ) == sizeof( VkDescriptorSetLayoutSupport ), "struct and wrapper have different size!" );

  using DescriptorSetLayoutSupportKHR = DescriptorSetLayoutSupport;

  struct PhysicalDeviceShaderDrawParameterFeatures
  {
    PhysicalDeviceShaderDrawParameterFeatures( Bool32 shaderDrawParameters_ = 0 )
      : shaderDrawParameters( shaderDrawParameters_ )
    {
    }

    PhysicalDeviceShaderDrawParameterFeatures( VkPhysicalDeviceShaderDrawParameterFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceShaderDrawParameterFeatures ) );
    }

    PhysicalDeviceShaderDrawParameterFeatures& operator=( VkPhysicalDeviceShaderDrawParameterFeatures const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceShaderDrawParameterFeatures ) );
      return *this;
    }
    PhysicalDeviceShaderDrawParameterFeatures& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceShaderDrawParameterFeatures& setShaderDrawParameters( Bool32 shaderDrawParameters_ )
    {
      shaderDrawParameters = shaderDrawParameters_;
      return *this;
    }

    operator VkPhysicalDeviceShaderDrawParameterFeatures const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceShaderDrawParameterFeatures*>(this);
    }

    operator VkPhysicalDeviceShaderDrawParameterFeatures &()
    {
      return *reinterpret_cast<VkPhysicalDeviceShaderDrawParameterFeatures*>(this);
    }

    bool operator==( PhysicalDeviceShaderDrawParameterFeatures const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( shaderDrawParameters == rhs.shaderDrawParameters );
    }

    bool operator!=( PhysicalDeviceShaderDrawParameterFeatures const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceShaderDrawParameterFeatures;

  public:
    void* pNext = nullptr;
    Bool32 shaderDrawParameters;
  };
  static_assert( sizeof( PhysicalDeviceShaderDrawParameterFeatures ) == sizeof( VkPhysicalDeviceShaderDrawParameterFeatures ), "struct and wrapper have different size!" );

  struct PhysicalDeviceFloat16Int8FeaturesKHR
  {
    PhysicalDeviceFloat16Int8FeaturesKHR( Bool32 shaderFloat16_ = 0,
                                          Bool32 shaderInt8_ = 0 )
      : shaderFloat16( shaderFloat16_ )
      , shaderInt8( shaderInt8_ )
    {
    }

    PhysicalDeviceFloat16Int8FeaturesKHR( VkPhysicalDeviceFloat16Int8FeaturesKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceFloat16Int8FeaturesKHR ) );
    }

    PhysicalDeviceFloat16Int8FeaturesKHR& operator=( VkPhysicalDeviceFloat16Int8FeaturesKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceFloat16Int8FeaturesKHR ) );
      return *this;
    }
    PhysicalDeviceFloat16Int8FeaturesKHR& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceFloat16Int8FeaturesKHR& setShaderFloat16( Bool32 shaderFloat16_ )
    {
      shaderFloat16 = shaderFloat16_;
      return *this;
    }

    PhysicalDeviceFloat16Int8FeaturesKHR& setShaderInt8( Bool32 shaderInt8_ )
    {
      shaderInt8 = shaderInt8_;
      return *this;
    }

    operator VkPhysicalDeviceFloat16Int8FeaturesKHR const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceFloat16Int8FeaturesKHR*>(this);
    }

    operator VkPhysicalDeviceFloat16Int8FeaturesKHR &()
    {
      return *reinterpret_cast<VkPhysicalDeviceFloat16Int8FeaturesKHR*>(this);
    }

    bool operator==( PhysicalDeviceFloat16Int8FeaturesKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( shaderFloat16 == rhs.shaderFloat16 )
          && ( shaderInt8 == rhs.shaderInt8 );
    }

    bool operator!=( PhysicalDeviceFloat16Int8FeaturesKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceFloat16Int8FeaturesKHR;

  public:
    void* pNext = nullptr;
    Bool32 shaderFloat16;
    Bool32 shaderInt8;
  };
  static_assert( sizeof( PhysicalDeviceFloat16Int8FeaturesKHR ) == sizeof( VkPhysicalDeviceFloat16Int8FeaturesKHR ), "struct and wrapper have different size!" );

  struct PhysicalDeviceFloatControlsPropertiesKHR
  {
    PhysicalDeviceFloatControlsPropertiesKHR( Bool32 separateDenormSettings_ = 0,
                                              Bool32 separateRoundingModeSettings_ = 0,
                                              Bool32 shaderSignedZeroInfNanPreserveFloat16_ = 0,
                                              Bool32 shaderSignedZeroInfNanPreserveFloat32_ = 0,
                                              Bool32 shaderSignedZeroInfNanPreserveFloat64_ = 0,
                                              Bool32 shaderDenormPreserveFloat16_ = 0,
                                              Bool32 shaderDenormPreserveFloat32_ = 0,
                                              Bool32 shaderDenormPreserveFloat64_ = 0,
                                              Bool32 shaderDenormFlushToZeroFloat16_ = 0,
                                              Bool32 shaderDenormFlushToZeroFloat32_ = 0,
                                              Bool32 shaderDenormFlushToZeroFloat64_ = 0,
                                              Bool32 shaderRoundingModeRTEFloat16_ = 0,
                                              Bool32 shaderRoundingModeRTEFloat32_ = 0,
                                              Bool32 shaderRoundingModeRTEFloat64_ = 0,
                                              Bool32 shaderRoundingModeRTZFloat16_ = 0,
                                              Bool32 shaderRoundingModeRTZFloat32_ = 0,
                                              Bool32 shaderRoundingModeRTZFloat64_ = 0 )
      : separateDenormSettings( separateDenormSettings_ )
      , separateRoundingModeSettings( separateRoundingModeSettings_ )
      , shaderSignedZeroInfNanPreserveFloat16( shaderSignedZeroInfNanPreserveFloat16_ )
      , shaderSignedZeroInfNanPreserveFloat32( shaderSignedZeroInfNanPreserveFloat32_ )
      , shaderSignedZeroInfNanPreserveFloat64( shaderSignedZeroInfNanPreserveFloat64_ )
      , shaderDenormPreserveFloat16( shaderDenormPreserveFloat16_ )
      , shaderDenormPreserveFloat32( shaderDenormPreserveFloat32_ )
      , shaderDenormPreserveFloat64( shaderDenormPreserveFloat64_ )
      , shaderDenormFlushToZeroFloat16( shaderDenormFlushToZeroFloat16_ )
      , shaderDenormFlushToZeroFloat32( shaderDenormFlushToZeroFloat32_ )
      , shaderDenormFlushToZeroFloat64( shaderDenormFlushToZeroFloat64_ )
      , shaderRoundingModeRTEFloat16( shaderRoundingModeRTEFloat16_ )
      , shaderRoundingModeRTEFloat32( shaderRoundingModeRTEFloat32_ )
      , shaderRoundingModeRTEFloat64( shaderRoundingModeRTEFloat64_ )
      , shaderRoundingModeRTZFloat16( shaderRoundingModeRTZFloat16_ )
      , shaderRoundingModeRTZFloat32( shaderRoundingModeRTZFloat32_ )
      , shaderRoundingModeRTZFloat64( shaderRoundingModeRTZFloat64_ )
    {
    }

    PhysicalDeviceFloatControlsPropertiesKHR( VkPhysicalDeviceFloatControlsPropertiesKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceFloatControlsPropertiesKHR ) );
    }

    PhysicalDeviceFloatControlsPropertiesKHR& operator=( VkPhysicalDeviceFloatControlsPropertiesKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceFloatControlsPropertiesKHR ) );
      return *this;
    }
    PhysicalDeviceFloatControlsPropertiesKHR& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setSeparateDenormSettings( Bool32 separateDenormSettings_ )
    {
      separateDenormSettings = separateDenormSettings_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setSeparateRoundingModeSettings( Bool32 separateRoundingModeSettings_ )
    {
      separateRoundingModeSettings = separateRoundingModeSettings_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderSignedZeroInfNanPreserveFloat16( Bool32 shaderSignedZeroInfNanPreserveFloat16_ )
    {
      shaderSignedZeroInfNanPreserveFloat16 = shaderSignedZeroInfNanPreserveFloat16_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderSignedZeroInfNanPreserveFloat32( Bool32 shaderSignedZeroInfNanPreserveFloat32_ )
    {
      shaderSignedZeroInfNanPreserveFloat32 = shaderSignedZeroInfNanPreserveFloat32_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderSignedZeroInfNanPreserveFloat64( Bool32 shaderSignedZeroInfNanPreserveFloat64_ )
    {
      shaderSignedZeroInfNanPreserveFloat64 = shaderSignedZeroInfNanPreserveFloat64_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderDenormPreserveFloat16( Bool32 shaderDenormPreserveFloat16_ )
    {
      shaderDenormPreserveFloat16 = shaderDenormPreserveFloat16_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderDenormPreserveFloat32( Bool32 shaderDenormPreserveFloat32_ )
    {
      shaderDenormPreserveFloat32 = shaderDenormPreserveFloat32_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderDenormPreserveFloat64( Bool32 shaderDenormPreserveFloat64_ )
    {
      shaderDenormPreserveFloat64 = shaderDenormPreserveFloat64_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderDenormFlushToZeroFloat16( Bool32 shaderDenormFlushToZeroFloat16_ )
    {
      shaderDenormFlushToZeroFloat16 = shaderDenormFlushToZeroFloat16_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderDenormFlushToZeroFloat32( Bool32 shaderDenormFlushToZeroFloat32_ )
    {
      shaderDenormFlushToZeroFloat32 = shaderDenormFlushToZeroFloat32_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderDenormFlushToZeroFloat64( Bool32 shaderDenormFlushToZeroFloat64_ )
    {
      shaderDenormFlushToZeroFloat64 = shaderDenormFlushToZeroFloat64_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderRoundingModeRTEFloat16( Bool32 shaderRoundingModeRTEFloat16_ )
    {
      shaderRoundingModeRTEFloat16 = shaderRoundingModeRTEFloat16_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderRoundingModeRTEFloat32( Bool32 shaderRoundingModeRTEFloat32_ )
    {
      shaderRoundingModeRTEFloat32 = shaderRoundingModeRTEFloat32_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderRoundingModeRTEFloat64( Bool32 shaderRoundingModeRTEFloat64_ )
    {
      shaderRoundingModeRTEFloat64 = shaderRoundingModeRTEFloat64_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderRoundingModeRTZFloat16( Bool32 shaderRoundingModeRTZFloat16_ )
    {
      shaderRoundingModeRTZFloat16 = shaderRoundingModeRTZFloat16_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderRoundingModeRTZFloat32( Bool32 shaderRoundingModeRTZFloat32_ )
    {
      shaderRoundingModeRTZFloat32 = shaderRoundingModeRTZFloat32_;
      return *this;
    }

    PhysicalDeviceFloatControlsPropertiesKHR& setShaderRoundingModeRTZFloat64( Bool32 shaderRoundingModeRTZFloat64_ )
    {
      shaderRoundingModeRTZFloat64 = shaderRoundingModeRTZFloat64_;
      return *this;
    }

    operator VkPhysicalDeviceFloatControlsPropertiesKHR const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceFloatControlsPropertiesKHR*>(this);
    }

    operator VkPhysicalDeviceFloatControlsPropertiesKHR &()
    {
      return *reinterpret_cast<VkPhysicalDeviceFloatControlsPropertiesKHR*>(this);
    }

    bool operator==( PhysicalDeviceFloatControlsPropertiesKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( separateDenormSettings == rhs.separateDenormSettings )
          && ( separateRoundingModeSettings == rhs.separateRoundingModeSettings )
          && ( shaderSignedZeroInfNanPreserveFloat16 == rhs.shaderSignedZeroInfNanPreserveFloat16 )
          && ( shaderSignedZeroInfNanPreserveFloat32 == rhs.shaderSignedZeroInfNanPreserveFloat32 )
          && ( shaderSignedZeroInfNanPreserveFloat64 == rhs.shaderSignedZeroInfNanPreserveFloat64 )
          && ( shaderDenormPreserveFloat16 == rhs.shaderDenormPreserveFloat16 )
          && ( shaderDenormPreserveFloat32 == rhs.shaderDenormPreserveFloat32 )
          && ( shaderDenormPreserveFloat64 == rhs.shaderDenormPreserveFloat64 )
          && ( shaderDenormFlushToZeroFloat16 == rhs.shaderDenormFlushToZeroFloat16 )
          && ( shaderDenormFlushToZeroFloat32 == rhs.shaderDenormFlushToZeroFloat32 )
          && ( shaderDenormFlushToZeroFloat64 == rhs.shaderDenormFlushToZeroFloat64 )
          && ( shaderRoundingModeRTEFloat16 == rhs.shaderRoundingModeRTEFloat16 )
          && ( shaderRoundingModeRTEFloat32 == rhs.shaderRoundingModeRTEFloat32 )
          && ( shaderRoundingModeRTEFloat64 == rhs.shaderRoundingModeRTEFloat64 )
          && ( shaderRoundingModeRTZFloat16 == rhs.shaderRoundingModeRTZFloat16 )
          && ( shaderRoundingModeRTZFloat32 == rhs.shaderRoundingModeRTZFloat32 )
          && ( shaderRoundingModeRTZFloat64 == rhs.shaderRoundingModeRTZFloat64 );
    }

    bool operator!=( PhysicalDeviceFloatControlsPropertiesKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceFloatControlsPropertiesKHR;

  public:
    void* pNext = nullptr;
    Bool32 separateDenormSettings;
    Bool32 separateRoundingModeSettings;
    Bool32 shaderSignedZeroInfNanPreserveFloat16;
    Bool32 shaderSignedZeroInfNanPreserveFloat32;
    Bool32 shaderSignedZeroInfNanPreserveFloat64;
    Bool32 shaderDenormPreserveFloat16;
    Bool32 shaderDenormPreserveFloat32;
    Bool32 shaderDenormPreserveFloat64;
    Bool32 shaderDenormFlushToZeroFloat16;
    Bool32 shaderDenormFlushToZeroFloat32;
    Bool32 shaderDenormFlushToZeroFloat64;
    Bool32 shaderRoundingModeRTEFloat16;
    Bool32 shaderRoundingModeRTEFloat32;
    Bool32 shaderRoundingModeRTEFloat64;
    Bool32 shaderRoundingModeRTZFloat16;
    Bool32 shaderRoundingModeRTZFloat32;
    Bool32 shaderRoundingModeRTZFloat64;
  };
  static_assert( sizeof( PhysicalDeviceFloatControlsPropertiesKHR ) == sizeof( VkPhysicalDeviceFloatControlsPropertiesKHR ), "struct and wrapper have different size!" );

  struct DebugUtilsLabelEXT
  {
    DebugUtilsLabelEXT( const char* pLabelName_ = nullptr,
                        std::array<float,4> const& color_ = { { 0, 0, 0, 0 } } )
      : pLabelName( pLabelName_ )
    {
      memcpy( &color, color_.data(), 4 * sizeof( float ) );
    }

    DebugUtilsLabelEXT( VkDebugUtilsLabelEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsLabelEXT ) );
    }

    DebugUtilsLabelEXT& operator=( VkDebugUtilsLabelEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsLabelEXT ) );
      return *this;
    }
    DebugUtilsLabelEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugUtilsLabelEXT& setPLabelName( const char* pLabelName_ )
    {
      pLabelName = pLabelName_;
      return *this;
    }

    DebugUtilsLabelEXT& setColor( std::array<float,4> color_ )
    {
      memcpy( &color, color_.data(), 4 * sizeof( float ) );
      return *this;
    }

    operator VkDebugUtilsLabelEXT const&() const
    {
      return *reinterpret_cast<const VkDebugUtilsLabelEXT*>(this);
    }

    operator VkDebugUtilsLabelEXT &()
    {
      return *reinterpret_cast<VkDebugUtilsLabelEXT*>(this);
    }

    bool operator==( DebugUtilsLabelEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( pLabelName == rhs.pLabelName )
          && ( memcmp( color, rhs.color, 4 * sizeof( float ) ) == 0 );
    }

    bool operator!=( DebugUtilsLabelEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugUtilsLabelEXT;

  public:
    const void* pNext = nullptr;
    const char* pLabelName;
    float color[4];
  };
  static_assert( sizeof( DebugUtilsLabelEXT ) == sizeof( VkDebugUtilsLabelEXT ), "struct and wrapper have different size!" );

  struct MemoryHostPointerPropertiesEXT
  {
    MemoryHostPointerPropertiesEXT( uint32_t memoryTypeBits_ = 0 )
      : memoryTypeBits( memoryTypeBits_ )
    {
    }

    MemoryHostPointerPropertiesEXT( VkMemoryHostPointerPropertiesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( MemoryHostPointerPropertiesEXT ) );
    }

    MemoryHostPointerPropertiesEXT& operator=( VkMemoryHostPointerPropertiesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( MemoryHostPointerPropertiesEXT ) );
      return *this;
    }
    MemoryHostPointerPropertiesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    MemoryHostPointerPropertiesEXT& setMemoryTypeBits( uint32_t memoryTypeBits_ )
    {
      memoryTypeBits = memoryTypeBits_;
      return *this;
    }

    operator VkMemoryHostPointerPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkMemoryHostPointerPropertiesEXT*>(this);
    }

    operator VkMemoryHostPointerPropertiesEXT &()
    {
      return *reinterpret_cast<VkMemoryHostPointerPropertiesEXT*>(this);
    }

    bool operator==( MemoryHostPointerPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( memoryTypeBits == rhs.memoryTypeBits );
    }

    bool operator!=( MemoryHostPointerPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eMemoryHostPointerPropertiesEXT;

  public:
    void* pNext = nullptr;
    uint32_t memoryTypeBits;
  };
  static_assert( sizeof( MemoryHostPointerPropertiesEXT ) == sizeof( VkMemoryHostPointerPropertiesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceExternalMemoryHostPropertiesEXT
  {
    PhysicalDeviceExternalMemoryHostPropertiesEXT( DeviceSize minImportedHostPointerAlignment_ = 0 )
      : minImportedHostPointerAlignment( minImportedHostPointerAlignment_ )
    {
    }

    PhysicalDeviceExternalMemoryHostPropertiesEXT( VkPhysicalDeviceExternalMemoryHostPropertiesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceExternalMemoryHostPropertiesEXT ) );
    }

    PhysicalDeviceExternalMemoryHostPropertiesEXT& operator=( VkPhysicalDeviceExternalMemoryHostPropertiesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceExternalMemoryHostPropertiesEXT ) );
      return *this;
    }
    PhysicalDeviceExternalMemoryHostPropertiesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceExternalMemoryHostPropertiesEXT& setMinImportedHostPointerAlignment( DeviceSize minImportedHostPointerAlignment_ )
    {
      minImportedHostPointerAlignment = minImportedHostPointerAlignment_;
      return *this;
    }

    operator VkPhysicalDeviceExternalMemoryHostPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceExternalMemoryHostPropertiesEXT*>(this);
    }

    operator VkPhysicalDeviceExternalMemoryHostPropertiesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceExternalMemoryHostPropertiesEXT*>(this);
    }

    bool operator==( PhysicalDeviceExternalMemoryHostPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( minImportedHostPointerAlignment == rhs.minImportedHostPointerAlignment );
    }

    bool operator!=( PhysicalDeviceExternalMemoryHostPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceExternalMemoryHostPropertiesEXT;

  public:
    void* pNext = nullptr;
    DeviceSize minImportedHostPointerAlignment;
  };
  static_assert( sizeof( PhysicalDeviceExternalMemoryHostPropertiesEXT ) == sizeof( VkPhysicalDeviceExternalMemoryHostPropertiesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceConservativeRasterizationPropertiesEXT
  {
    PhysicalDeviceConservativeRasterizationPropertiesEXT( float primitiveOverestimationSize_ = 0,
                                                          float maxExtraPrimitiveOverestimationSize_ = 0,
                                                          float extraPrimitiveOverestimationSizeGranularity_ = 0,
                                                          Bool32 primitiveUnderestimation_ = 0,
                                                          Bool32 conservativePointAndLineRasterization_ = 0,
                                                          Bool32 degenerateTrianglesRasterized_ = 0,
                                                          Bool32 degenerateLinesRasterized_ = 0,
                                                          Bool32 fullyCoveredFragmentShaderInputVariable_ = 0,
                                                          Bool32 conservativeRasterizationPostDepthCoverage_ = 0 )
      : primitiveOverestimationSize( primitiveOverestimationSize_ )
      , maxExtraPrimitiveOverestimationSize( maxExtraPrimitiveOverestimationSize_ )
      , extraPrimitiveOverestimationSizeGranularity( extraPrimitiveOverestimationSizeGranularity_ )
      , primitiveUnderestimation( primitiveUnderestimation_ )
      , conservativePointAndLineRasterization( conservativePointAndLineRasterization_ )
      , degenerateTrianglesRasterized( degenerateTrianglesRasterized_ )
      , degenerateLinesRasterized( degenerateLinesRasterized_ )
      , fullyCoveredFragmentShaderInputVariable( fullyCoveredFragmentShaderInputVariable_ )
      , conservativeRasterizationPostDepthCoverage( conservativeRasterizationPostDepthCoverage_ )
    {
    }

    PhysicalDeviceConservativeRasterizationPropertiesEXT( VkPhysicalDeviceConservativeRasterizationPropertiesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceConservativeRasterizationPropertiesEXT ) );
    }

    PhysicalDeviceConservativeRasterizationPropertiesEXT& operator=( VkPhysicalDeviceConservativeRasterizationPropertiesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceConservativeRasterizationPropertiesEXT ) );
      return *this;
    }
    PhysicalDeviceConservativeRasterizationPropertiesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceConservativeRasterizationPropertiesEXT& setPrimitiveOverestimationSize( float primitiveOverestimationSize_ )
    {
      primitiveOverestimationSize = primitiveOverestimationSize_;
      return *this;
    }

    PhysicalDeviceConservativeRasterizationPropertiesEXT& setMaxExtraPrimitiveOverestimationSize( float maxExtraPrimitiveOverestimationSize_ )
    {
      maxExtraPrimitiveOverestimationSize = maxExtraPrimitiveOverestimationSize_;
      return *this;
    }

    PhysicalDeviceConservativeRasterizationPropertiesEXT& setExtraPrimitiveOverestimationSizeGranularity( float extraPrimitiveOverestimationSizeGranularity_ )
    {
      extraPrimitiveOverestimationSizeGranularity = extraPrimitiveOverestimationSizeGranularity_;
      return *this;
    }

    PhysicalDeviceConservativeRasterizationPropertiesEXT& setPrimitiveUnderestimation( Bool32 primitiveUnderestimation_ )
    {
      primitiveUnderestimation = primitiveUnderestimation_;
      return *this;
    }

    PhysicalDeviceConservativeRasterizationPropertiesEXT& setConservativePointAndLineRasterization( Bool32 conservativePointAndLineRasterization_ )
    {
      conservativePointAndLineRasterization = conservativePointAndLineRasterization_;
      return *this;
    }

    PhysicalDeviceConservativeRasterizationPropertiesEXT& setDegenerateTrianglesRasterized( Bool32 degenerateTrianglesRasterized_ )
    {
      degenerateTrianglesRasterized = degenerateTrianglesRasterized_;
      return *this;
    }

    PhysicalDeviceConservativeRasterizationPropertiesEXT& setDegenerateLinesRasterized( Bool32 degenerateLinesRasterized_ )
    {
      degenerateLinesRasterized = degenerateLinesRasterized_;
      return *this;
    }

    PhysicalDeviceConservativeRasterizationPropertiesEXT& setFullyCoveredFragmentShaderInputVariable( Bool32 fullyCoveredFragmentShaderInputVariable_ )
    {
      fullyCoveredFragmentShaderInputVariable = fullyCoveredFragmentShaderInputVariable_;
      return *this;
    }

    PhysicalDeviceConservativeRasterizationPropertiesEXT& setConservativeRasterizationPostDepthCoverage( Bool32 conservativeRasterizationPostDepthCoverage_ )
    {
      conservativeRasterizationPostDepthCoverage = conservativeRasterizationPostDepthCoverage_;
      return *this;
    }

    operator VkPhysicalDeviceConservativeRasterizationPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceConservativeRasterizationPropertiesEXT*>(this);
    }

    operator VkPhysicalDeviceConservativeRasterizationPropertiesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceConservativeRasterizationPropertiesEXT*>(this);
    }

    bool operator==( PhysicalDeviceConservativeRasterizationPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( primitiveOverestimationSize == rhs.primitiveOverestimationSize )
          && ( maxExtraPrimitiveOverestimationSize == rhs.maxExtraPrimitiveOverestimationSize )
          && ( extraPrimitiveOverestimationSizeGranularity == rhs.extraPrimitiveOverestimationSizeGranularity )
          && ( primitiveUnderestimation == rhs.primitiveUnderestimation )
          && ( conservativePointAndLineRasterization == rhs.conservativePointAndLineRasterization )
          && ( degenerateTrianglesRasterized == rhs.degenerateTrianglesRasterized )
          && ( degenerateLinesRasterized == rhs.degenerateLinesRasterized )
          && ( fullyCoveredFragmentShaderInputVariable == rhs.fullyCoveredFragmentShaderInputVariable )
          && ( conservativeRasterizationPostDepthCoverage == rhs.conservativeRasterizationPostDepthCoverage );
    }

    bool operator!=( PhysicalDeviceConservativeRasterizationPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceConservativeRasterizationPropertiesEXT;

  public:
    void* pNext = nullptr;
    float primitiveOverestimationSize;
    float maxExtraPrimitiveOverestimationSize;
    float extraPrimitiveOverestimationSizeGranularity;
    Bool32 primitiveUnderestimation;
    Bool32 conservativePointAndLineRasterization;
    Bool32 degenerateTrianglesRasterized;
    Bool32 degenerateLinesRasterized;
    Bool32 fullyCoveredFragmentShaderInputVariable;
    Bool32 conservativeRasterizationPostDepthCoverage;
  };
  static_assert( sizeof( PhysicalDeviceConservativeRasterizationPropertiesEXT ) == sizeof( VkPhysicalDeviceConservativeRasterizationPropertiesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceShaderCorePropertiesAMD
  {
    operator VkPhysicalDeviceShaderCorePropertiesAMD const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceShaderCorePropertiesAMD*>(this);
    }

    operator VkPhysicalDeviceShaderCorePropertiesAMD &()
    {
      return *reinterpret_cast<VkPhysicalDeviceShaderCorePropertiesAMD*>(this);
    }

    bool operator==( PhysicalDeviceShaderCorePropertiesAMD const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( shaderEngineCount == rhs.shaderEngineCount )
          && ( shaderArraysPerEngineCount == rhs.shaderArraysPerEngineCount )
          && ( computeUnitsPerShaderArray == rhs.computeUnitsPerShaderArray )
          && ( simdPerComputeUnit == rhs.simdPerComputeUnit )
          && ( wavefrontsPerSimd == rhs.wavefrontsPerSimd )
          && ( wavefrontSize == rhs.wavefrontSize )
          && ( sgprsPerSimd == rhs.sgprsPerSimd )
          && ( minSgprAllocation == rhs.minSgprAllocation )
          && ( maxSgprAllocation == rhs.maxSgprAllocation )
          && ( sgprAllocationGranularity == rhs.sgprAllocationGranularity )
          && ( vgprsPerSimd == rhs.vgprsPerSimd )
          && ( minVgprAllocation == rhs.minVgprAllocation )
          && ( maxVgprAllocation == rhs.maxVgprAllocation )
          && ( vgprAllocationGranularity == rhs.vgprAllocationGranularity );
    }

    bool operator!=( PhysicalDeviceShaderCorePropertiesAMD const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceShaderCorePropertiesAMD;

  public:
    void* pNext = nullptr;
    uint32_t shaderEngineCount;
    uint32_t shaderArraysPerEngineCount;
    uint32_t computeUnitsPerShaderArray;
    uint32_t simdPerComputeUnit;
    uint32_t wavefrontsPerSimd;
    uint32_t wavefrontSize;
    uint32_t sgprsPerSimd;
    uint32_t minSgprAllocation;
    uint32_t maxSgprAllocation;
    uint32_t sgprAllocationGranularity;
    uint32_t vgprsPerSimd;
    uint32_t minVgprAllocation;
    uint32_t maxVgprAllocation;
    uint32_t vgprAllocationGranularity;
  };
  static_assert( sizeof( PhysicalDeviceShaderCorePropertiesAMD ) == sizeof( VkPhysicalDeviceShaderCorePropertiesAMD ), "struct and wrapper have different size!" );

  struct PhysicalDeviceDescriptorIndexingFeaturesEXT
  {
    PhysicalDeviceDescriptorIndexingFeaturesEXT( Bool32 shaderInputAttachmentArrayDynamicIndexing_ = 0,
                                                 Bool32 shaderUniformTexelBufferArrayDynamicIndexing_ = 0,
                                                 Bool32 shaderStorageTexelBufferArrayDynamicIndexing_ = 0,
                                                 Bool32 shaderUniformBufferArrayNonUniformIndexing_ = 0,
                                                 Bool32 shaderSampledImageArrayNonUniformIndexing_ = 0,
                                                 Bool32 shaderStorageBufferArrayNonUniformIndexing_ = 0,
                                                 Bool32 shaderStorageImageArrayNonUniformIndexing_ = 0,
                                                 Bool32 shaderInputAttachmentArrayNonUniformIndexing_ = 0,
                                                 Bool32 shaderUniformTexelBufferArrayNonUniformIndexing_ = 0,
                                                 Bool32 shaderStorageTexelBufferArrayNonUniformIndexing_ = 0,
                                                 Bool32 descriptorBindingUniformBufferUpdateAfterBind_ = 0,
                                                 Bool32 descriptorBindingSampledImageUpdateAfterBind_ = 0,
                                                 Bool32 descriptorBindingStorageImageUpdateAfterBind_ = 0,
                                                 Bool32 descriptorBindingStorageBufferUpdateAfterBind_ = 0,
                                                 Bool32 descriptorBindingUniformTexelBufferUpdateAfterBind_ = 0,
                                                 Bool32 descriptorBindingStorageTexelBufferUpdateAfterBind_ = 0,
                                                 Bool32 descriptorBindingUpdateUnusedWhilePending_ = 0,
                                                 Bool32 descriptorBindingPartiallyBound_ = 0,
                                                 Bool32 descriptorBindingVariableDescriptorCount_ = 0,
                                                 Bool32 runtimeDescriptorArray_ = 0 )
      : shaderInputAttachmentArrayDynamicIndexing( shaderInputAttachmentArrayDynamicIndexing_ )
      , shaderUniformTexelBufferArrayDynamicIndexing( shaderUniformTexelBufferArrayDynamicIndexing_ )
      , shaderStorageTexelBufferArrayDynamicIndexing( shaderStorageTexelBufferArrayDynamicIndexing_ )
      , shaderUniformBufferArrayNonUniformIndexing( shaderUniformBufferArrayNonUniformIndexing_ )
      , shaderSampledImageArrayNonUniformIndexing( shaderSampledImageArrayNonUniformIndexing_ )
      , shaderStorageBufferArrayNonUniformIndexing( shaderStorageBufferArrayNonUniformIndexing_ )
      , shaderStorageImageArrayNonUniformIndexing( shaderStorageImageArrayNonUniformIndexing_ )
      , shaderInputAttachmentArrayNonUniformIndexing( shaderInputAttachmentArrayNonUniformIndexing_ )
      , shaderUniformTexelBufferArrayNonUniformIndexing( shaderUniformTexelBufferArrayNonUniformIndexing_ )
      , shaderStorageTexelBufferArrayNonUniformIndexing( shaderStorageTexelBufferArrayNonUniformIndexing_ )
      , descriptorBindingUniformBufferUpdateAfterBind( descriptorBindingUniformBufferUpdateAfterBind_ )
      , descriptorBindingSampledImageUpdateAfterBind( descriptorBindingSampledImageUpdateAfterBind_ )
      , descriptorBindingStorageImageUpdateAfterBind( descriptorBindingStorageImageUpdateAfterBind_ )
      , descriptorBindingStorageBufferUpdateAfterBind( descriptorBindingStorageBufferUpdateAfterBind_ )
      , descriptorBindingUniformTexelBufferUpdateAfterBind( descriptorBindingUniformTexelBufferUpdateAfterBind_ )
      , descriptorBindingStorageTexelBufferUpdateAfterBind( descriptorBindingStorageTexelBufferUpdateAfterBind_ )
      , descriptorBindingUpdateUnusedWhilePending( descriptorBindingUpdateUnusedWhilePending_ )
      , descriptorBindingPartiallyBound( descriptorBindingPartiallyBound_ )
      , descriptorBindingVariableDescriptorCount( descriptorBindingVariableDescriptorCount_ )
      , runtimeDescriptorArray( runtimeDescriptorArray_ )
    {
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT( VkPhysicalDeviceDescriptorIndexingFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceDescriptorIndexingFeaturesEXT ) );
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& operator=( VkPhysicalDeviceDescriptorIndexingFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceDescriptorIndexingFeaturesEXT ) );
      return *this;
    }
    PhysicalDeviceDescriptorIndexingFeaturesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setShaderInputAttachmentArrayDynamicIndexing( Bool32 shaderInputAttachmentArrayDynamicIndexing_ )
    {
      shaderInputAttachmentArrayDynamicIndexing = shaderInputAttachmentArrayDynamicIndexing_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setShaderUniformTexelBufferArrayDynamicIndexing( Bool32 shaderUniformTexelBufferArrayDynamicIndexing_ )
    {
      shaderUniformTexelBufferArrayDynamicIndexing = shaderUniformTexelBufferArrayDynamicIndexing_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setShaderStorageTexelBufferArrayDynamicIndexing( Bool32 shaderStorageTexelBufferArrayDynamicIndexing_ )
    {
      shaderStorageTexelBufferArrayDynamicIndexing = shaderStorageTexelBufferArrayDynamicIndexing_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setShaderUniformBufferArrayNonUniformIndexing( Bool32 shaderUniformBufferArrayNonUniformIndexing_ )
    {
      shaderUniformBufferArrayNonUniformIndexing = shaderUniformBufferArrayNonUniformIndexing_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setShaderSampledImageArrayNonUniformIndexing( Bool32 shaderSampledImageArrayNonUniformIndexing_ )
    {
      shaderSampledImageArrayNonUniformIndexing = shaderSampledImageArrayNonUniformIndexing_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setShaderStorageBufferArrayNonUniformIndexing( Bool32 shaderStorageBufferArrayNonUniformIndexing_ )
    {
      shaderStorageBufferArrayNonUniformIndexing = shaderStorageBufferArrayNonUniformIndexing_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setShaderStorageImageArrayNonUniformIndexing( Bool32 shaderStorageImageArrayNonUniformIndexing_ )
    {
      shaderStorageImageArrayNonUniformIndexing = shaderStorageImageArrayNonUniformIndexing_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setShaderInputAttachmentArrayNonUniformIndexing( Bool32 shaderInputAttachmentArrayNonUniformIndexing_ )
    {
      shaderInputAttachmentArrayNonUniformIndexing = shaderInputAttachmentArrayNonUniformIndexing_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setShaderUniformTexelBufferArrayNonUniformIndexing( Bool32 shaderUniformTexelBufferArrayNonUniformIndexing_ )
    {
      shaderUniformTexelBufferArrayNonUniformIndexing = shaderUniformTexelBufferArrayNonUniformIndexing_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setShaderStorageTexelBufferArrayNonUniformIndexing( Bool32 shaderStorageTexelBufferArrayNonUniformIndexing_ )
    {
      shaderStorageTexelBufferArrayNonUniformIndexing = shaderStorageTexelBufferArrayNonUniformIndexing_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setDescriptorBindingUniformBufferUpdateAfterBind( Bool32 descriptorBindingUniformBufferUpdateAfterBind_ )
    {
      descriptorBindingUniformBufferUpdateAfterBind = descriptorBindingUniformBufferUpdateAfterBind_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setDescriptorBindingSampledImageUpdateAfterBind( Bool32 descriptorBindingSampledImageUpdateAfterBind_ )
    {
      descriptorBindingSampledImageUpdateAfterBind = descriptorBindingSampledImageUpdateAfterBind_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setDescriptorBindingStorageImageUpdateAfterBind( Bool32 descriptorBindingStorageImageUpdateAfterBind_ )
    {
      descriptorBindingStorageImageUpdateAfterBind = descriptorBindingStorageImageUpdateAfterBind_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setDescriptorBindingStorageBufferUpdateAfterBind( Bool32 descriptorBindingStorageBufferUpdateAfterBind_ )
    {
      descriptorBindingStorageBufferUpdateAfterBind = descriptorBindingStorageBufferUpdateAfterBind_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setDescriptorBindingUniformTexelBufferUpdateAfterBind( Bool32 descriptorBindingUniformTexelBufferUpdateAfterBind_ )
    {
      descriptorBindingUniformTexelBufferUpdateAfterBind = descriptorBindingUniformTexelBufferUpdateAfterBind_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setDescriptorBindingStorageTexelBufferUpdateAfterBind( Bool32 descriptorBindingStorageTexelBufferUpdateAfterBind_ )
    {
      descriptorBindingStorageTexelBufferUpdateAfterBind = descriptorBindingStorageTexelBufferUpdateAfterBind_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setDescriptorBindingUpdateUnusedWhilePending( Bool32 descriptorBindingUpdateUnusedWhilePending_ )
    {
      descriptorBindingUpdateUnusedWhilePending = descriptorBindingUpdateUnusedWhilePending_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setDescriptorBindingPartiallyBound( Bool32 descriptorBindingPartiallyBound_ )
    {
      descriptorBindingPartiallyBound = descriptorBindingPartiallyBound_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setDescriptorBindingVariableDescriptorCount( Bool32 descriptorBindingVariableDescriptorCount_ )
    {
      descriptorBindingVariableDescriptorCount = descriptorBindingVariableDescriptorCount_;
      return *this;
    }

    PhysicalDeviceDescriptorIndexingFeaturesEXT& setRuntimeDescriptorArray( Bool32 runtimeDescriptorArray_ )
    {
      runtimeDescriptorArray = runtimeDescriptorArray_;
      return *this;
    }

    operator VkPhysicalDeviceDescriptorIndexingFeaturesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceDescriptorIndexingFeaturesEXT*>(this);
    }

    operator VkPhysicalDeviceDescriptorIndexingFeaturesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceDescriptorIndexingFeaturesEXT*>(this);
    }

    bool operator==( PhysicalDeviceDescriptorIndexingFeaturesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( shaderInputAttachmentArrayDynamicIndexing == rhs.shaderInputAttachmentArrayDynamicIndexing )
          && ( shaderUniformTexelBufferArrayDynamicIndexing == rhs.shaderUniformTexelBufferArrayDynamicIndexing )
          && ( shaderStorageTexelBufferArrayDynamicIndexing == rhs.shaderStorageTexelBufferArrayDynamicIndexing )
          && ( shaderUniformBufferArrayNonUniformIndexing == rhs.shaderUniformBufferArrayNonUniformIndexing )
          && ( shaderSampledImageArrayNonUniformIndexing == rhs.shaderSampledImageArrayNonUniformIndexing )
          && ( shaderStorageBufferArrayNonUniformIndexing == rhs.shaderStorageBufferArrayNonUniformIndexing )
          && ( shaderStorageImageArrayNonUniformIndexing == rhs.shaderStorageImageArrayNonUniformIndexing )
          && ( shaderInputAttachmentArrayNonUniformIndexing == rhs.shaderInputAttachmentArrayNonUniformIndexing )
          && ( shaderUniformTexelBufferArrayNonUniformIndexing == rhs.shaderUniformTexelBufferArrayNonUniformIndexing )
          && ( shaderStorageTexelBufferArrayNonUniformIndexing == rhs.shaderStorageTexelBufferArrayNonUniformIndexing )
          && ( descriptorBindingUniformBufferUpdateAfterBind == rhs.descriptorBindingUniformBufferUpdateAfterBind )
          && ( descriptorBindingSampledImageUpdateAfterBind == rhs.descriptorBindingSampledImageUpdateAfterBind )
          && ( descriptorBindingStorageImageUpdateAfterBind == rhs.descriptorBindingStorageImageUpdateAfterBind )
          && ( descriptorBindingStorageBufferUpdateAfterBind == rhs.descriptorBindingStorageBufferUpdateAfterBind )
          && ( descriptorBindingUniformTexelBufferUpdateAfterBind == rhs.descriptorBindingUniformTexelBufferUpdateAfterBind )
          && ( descriptorBindingStorageTexelBufferUpdateAfterBind == rhs.descriptorBindingStorageTexelBufferUpdateAfterBind )
          && ( descriptorBindingUpdateUnusedWhilePending == rhs.descriptorBindingUpdateUnusedWhilePending )
          && ( descriptorBindingPartiallyBound == rhs.descriptorBindingPartiallyBound )
          && ( descriptorBindingVariableDescriptorCount == rhs.descriptorBindingVariableDescriptorCount )
          && ( runtimeDescriptorArray == rhs.runtimeDescriptorArray );
    }

    bool operator!=( PhysicalDeviceDescriptorIndexingFeaturesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceDescriptorIndexingFeaturesEXT;

  public:
    void* pNext = nullptr;
    Bool32 shaderInputAttachmentArrayDynamicIndexing;
    Bool32 shaderUniformTexelBufferArrayDynamicIndexing;
    Bool32 shaderStorageTexelBufferArrayDynamicIndexing;
    Bool32 shaderUniformBufferArrayNonUniformIndexing;
    Bool32 shaderSampledImageArrayNonUniformIndexing;
    Bool32 shaderStorageBufferArrayNonUniformIndexing;
    Bool32 shaderStorageImageArrayNonUniformIndexing;
    Bool32 shaderInputAttachmentArrayNonUniformIndexing;
    Bool32 shaderUniformTexelBufferArrayNonUniformIndexing;
    Bool32 shaderStorageTexelBufferArrayNonUniformIndexing;
    Bool32 descriptorBindingUniformBufferUpdateAfterBind;
    Bool32 descriptorBindingSampledImageUpdateAfterBind;
    Bool32 descriptorBindingStorageImageUpdateAfterBind;
    Bool32 descriptorBindingStorageBufferUpdateAfterBind;
    Bool32 descriptorBindingUniformTexelBufferUpdateAfterBind;
    Bool32 descriptorBindingStorageTexelBufferUpdateAfterBind;
    Bool32 descriptorBindingUpdateUnusedWhilePending;
    Bool32 descriptorBindingPartiallyBound;
    Bool32 descriptorBindingVariableDescriptorCount;
    Bool32 runtimeDescriptorArray;
  };
  static_assert( sizeof( PhysicalDeviceDescriptorIndexingFeaturesEXT ) == sizeof( VkPhysicalDeviceDescriptorIndexingFeaturesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceDescriptorIndexingPropertiesEXT
  {
    operator VkPhysicalDeviceDescriptorIndexingPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceDescriptorIndexingPropertiesEXT*>(this);
    }

    operator VkPhysicalDeviceDescriptorIndexingPropertiesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceDescriptorIndexingPropertiesEXT*>(this);
    }

    bool operator==( PhysicalDeviceDescriptorIndexingPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxUpdateAfterBindDescriptorsInAllPools == rhs.maxUpdateAfterBindDescriptorsInAllPools )
          && ( shaderUniformBufferArrayNonUniformIndexingNative == rhs.shaderUniformBufferArrayNonUniformIndexingNative )
          && ( shaderSampledImageArrayNonUniformIndexingNative == rhs.shaderSampledImageArrayNonUniformIndexingNative )
          && ( shaderStorageBufferArrayNonUniformIndexingNative == rhs.shaderStorageBufferArrayNonUniformIndexingNative )
          && ( shaderStorageImageArrayNonUniformIndexingNative == rhs.shaderStorageImageArrayNonUniformIndexingNative )
          && ( shaderInputAttachmentArrayNonUniformIndexingNative == rhs.shaderInputAttachmentArrayNonUniformIndexingNative )
          && ( robustBufferAccessUpdateAfterBind == rhs.robustBufferAccessUpdateAfterBind )
          && ( quadDivergentImplicitLod == rhs.quadDivergentImplicitLod )
          && ( maxPerStageDescriptorUpdateAfterBindSamplers == rhs.maxPerStageDescriptorUpdateAfterBindSamplers )
          && ( maxPerStageDescriptorUpdateAfterBindUniformBuffers == rhs.maxPerStageDescriptorUpdateAfterBindUniformBuffers )
          && ( maxPerStageDescriptorUpdateAfterBindStorageBuffers == rhs.maxPerStageDescriptorUpdateAfterBindStorageBuffers )
          && ( maxPerStageDescriptorUpdateAfterBindSampledImages == rhs.maxPerStageDescriptorUpdateAfterBindSampledImages )
          && ( maxPerStageDescriptorUpdateAfterBindStorageImages == rhs.maxPerStageDescriptorUpdateAfterBindStorageImages )
          && ( maxPerStageDescriptorUpdateAfterBindInputAttachments == rhs.maxPerStageDescriptorUpdateAfterBindInputAttachments )
          && ( maxPerStageUpdateAfterBindResources == rhs.maxPerStageUpdateAfterBindResources )
          && ( maxDescriptorSetUpdateAfterBindSamplers == rhs.maxDescriptorSetUpdateAfterBindSamplers )
          && ( maxDescriptorSetUpdateAfterBindUniformBuffers == rhs.maxDescriptorSetUpdateAfterBindUniformBuffers )
          && ( maxDescriptorSetUpdateAfterBindUniformBuffersDynamic == rhs.maxDescriptorSetUpdateAfterBindUniformBuffersDynamic )
          && ( maxDescriptorSetUpdateAfterBindStorageBuffers == rhs.maxDescriptorSetUpdateAfterBindStorageBuffers )
          && ( maxDescriptorSetUpdateAfterBindStorageBuffersDynamic == rhs.maxDescriptorSetUpdateAfterBindStorageBuffersDynamic )
          && ( maxDescriptorSetUpdateAfterBindSampledImages == rhs.maxDescriptorSetUpdateAfterBindSampledImages )
          && ( maxDescriptorSetUpdateAfterBindStorageImages == rhs.maxDescriptorSetUpdateAfterBindStorageImages )
          && ( maxDescriptorSetUpdateAfterBindInputAttachments == rhs.maxDescriptorSetUpdateAfterBindInputAttachments );
    }

    bool operator!=( PhysicalDeviceDescriptorIndexingPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceDescriptorIndexingPropertiesEXT;

  public:
    void* pNext = nullptr;
    uint32_t maxUpdateAfterBindDescriptorsInAllPools;
    Bool32 shaderUniformBufferArrayNonUniformIndexingNative;
    Bool32 shaderSampledImageArrayNonUniformIndexingNative;
    Bool32 shaderStorageBufferArrayNonUniformIndexingNative;
    Bool32 shaderStorageImageArrayNonUniformIndexingNative;
    Bool32 shaderInputAttachmentArrayNonUniformIndexingNative;
    Bool32 robustBufferAccessUpdateAfterBind;
    Bool32 quadDivergentImplicitLod;
    uint32_t maxPerStageDescriptorUpdateAfterBindSamplers;
    uint32_t maxPerStageDescriptorUpdateAfterBindUniformBuffers;
    uint32_t maxPerStageDescriptorUpdateAfterBindStorageBuffers;
    uint32_t maxPerStageDescriptorUpdateAfterBindSampledImages;
    uint32_t maxPerStageDescriptorUpdateAfterBindStorageImages;
    uint32_t maxPerStageDescriptorUpdateAfterBindInputAttachments;
    uint32_t maxPerStageUpdateAfterBindResources;
    uint32_t maxDescriptorSetUpdateAfterBindSamplers;
    uint32_t maxDescriptorSetUpdateAfterBindUniformBuffers;
    uint32_t maxDescriptorSetUpdateAfterBindUniformBuffersDynamic;
    uint32_t maxDescriptorSetUpdateAfterBindStorageBuffers;
    uint32_t maxDescriptorSetUpdateAfterBindStorageBuffersDynamic;
    uint32_t maxDescriptorSetUpdateAfterBindSampledImages;
    uint32_t maxDescriptorSetUpdateAfterBindStorageImages;
    uint32_t maxDescriptorSetUpdateAfterBindInputAttachments;
  };
  static_assert( sizeof( PhysicalDeviceDescriptorIndexingPropertiesEXT ) == sizeof( VkPhysicalDeviceDescriptorIndexingPropertiesEXT ), "struct and wrapper have different size!" );

  struct DescriptorSetVariableDescriptorCountAllocateInfoEXT
  {
    DescriptorSetVariableDescriptorCountAllocateInfoEXT( uint32_t descriptorSetCount_ = 0,
                                                         const uint32_t* pDescriptorCounts_ = nullptr )
      : descriptorSetCount( descriptorSetCount_ )
      , pDescriptorCounts( pDescriptorCounts_ )
    {
    }

    DescriptorSetVariableDescriptorCountAllocateInfoEXT( VkDescriptorSetVariableDescriptorCountAllocateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetVariableDescriptorCountAllocateInfoEXT ) );
    }

    DescriptorSetVariableDescriptorCountAllocateInfoEXT& operator=( VkDescriptorSetVariableDescriptorCountAllocateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetVariableDescriptorCountAllocateInfoEXT ) );
      return *this;
    }
    DescriptorSetVariableDescriptorCountAllocateInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DescriptorSetVariableDescriptorCountAllocateInfoEXT& setDescriptorSetCount( uint32_t descriptorSetCount_ )
    {
      descriptorSetCount = descriptorSetCount_;
      return *this;
    }

    DescriptorSetVariableDescriptorCountAllocateInfoEXT& setPDescriptorCounts( const uint32_t* pDescriptorCounts_ )
    {
      pDescriptorCounts = pDescriptorCounts_;
      return *this;
    }

    operator VkDescriptorSetVariableDescriptorCountAllocateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDescriptorSetVariableDescriptorCountAllocateInfoEXT*>(this);
    }

    operator VkDescriptorSetVariableDescriptorCountAllocateInfoEXT &()
    {
      return *reinterpret_cast<VkDescriptorSetVariableDescriptorCountAllocateInfoEXT*>(this);
    }

    bool operator==( DescriptorSetVariableDescriptorCountAllocateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( descriptorSetCount == rhs.descriptorSetCount )
          && ( pDescriptorCounts == rhs.pDescriptorCounts );
    }

    bool operator!=( DescriptorSetVariableDescriptorCountAllocateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDescriptorSetVariableDescriptorCountAllocateInfoEXT;

  public:
    const void* pNext = nullptr;
    uint32_t descriptorSetCount;
    const uint32_t* pDescriptorCounts;
  };
  static_assert( sizeof( DescriptorSetVariableDescriptorCountAllocateInfoEXT ) == sizeof( VkDescriptorSetVariableDescriptorCountAllocateInfoEXT ), "struct and wrapper have different size!" );

  struct DescriptorSetVariableDescriptorCountLayoutSupportEXT
  {
    operator VkDescriptorSetVariableDescriptorCountLayoutSupportEXT const&() const
    {
      return *reinterpret_cast<const VkDescriptorSetVariableDescriptorCountLayoutSupportEXT*>(this);
    }

    operator VkDescriptorSetVariableDescriptorCountLayoutSupportEXT &()
    {
      return *reinterpret_cast<VkDescriptorSetVariableDescriptorCountLayoutSupportEXT*>(this);
    }

    bool operator==( DescriptorSetVariableDescriptorCountLayoutSupportEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxVariableDescriptorCount == rhs.maxVariableDescriptorCount );
    }

    bool operator!=( DescriptorSetVariableDescriptorCountLayoutSupportEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDescriptorSetVariableDescriptorCountLayoutSupportEXT;

  public:
    void* pNext = nullptr;
    uint32_t maxVariableDescriptorCount;
  };
  static_assert( sizeof( DescriptorSetVariableDescriptorCountLayoutSupportEXT ) == sizeof( VkDescriptorSetVariableDescriptorCountLayoutSupportEXT ), "struct and wrapper have different size!" );

  struct SubpassEndInfoKHR
  {
    SubpassEndInfoKHR(  )
    {
    }

    SubpassEndInfoKHR( VkSubpassEndInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( SubpassEndInfoKHR ) );
    }

    SubpassEndInfoKHR& operator=( VkSubpassEndInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( SubpassEndInfoKHR ) );
      return *this;
    }
    SubpassEndInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    operator VkSubpassEndInfoKHR const&() const
    {
      return *reinterpret_cast<const VkSubpassEndInfoKHR*>(this);
    }

    operator VkSubpassEndInfoKHR &()
    {
      return *reinterpret_cast<VkSubpassEndInfoKHR*>(this);
    }

    bool operator==( SubpassEndInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext );
    }

    bool operator!=( SubpassEndInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eSubpassEndInfoKHR;

  public:
    const void* pNext = nullptr;
  };
  static_assert( sizeof( SubpassEndInfoKHR ) == sizeof( VkSubpassEndInfoKHR ), "struct and wrapper have different size!" );

  struct PipelineVertexInputDivisorStateCreateInfoEXT
  {
    PipelineVertexInputDivisorStateCreateInfoEXT( uint32_t vertexBindingDivisorCount_ = 0,
                                                  const VertexInputBindingDivisorDescriptionEXT* pVertexBindingDivisors_ = nullptr )
      : vertexBindingDivisorCount( vertexBindingDivisorCount_ )
      , pVertexBindingDivisors( pVertexBindingDivisors_ )
    {
    }

    PipelineVertexInputDivisorStateCreateInfoEXT( VkPipelineVertexInputDivisorStateCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineVertexInputDivisorStateCreateInfoEXT ) );
    }

    PipelineVertexInputDivisorStateCreateInfoEXT& operator=( VkPipelineVertexInputDivisorStateCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineVertexInputDivisorStateCreateInfoEXT ) );
      return *this;
    }
    PipelineVertexInputDivisorStateCreateInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineVertexInputDivisorStateCreateInfoEXT& setVertexBindingDivisorCount( uint32_t vertexBindingDivisorCount_ )
    {
      vertexBindingDivisorCount = vertexBindingDivisorCount_;
      return *this;
    }

    PipelineVertexInputDivisorStateCreateInfoEXT& setPVertexBindingDivisors( const VertexInputBindingDivisorDescriptionEXT* pVertexBindingDivisors_ )
    {
      pVertexBindingDivisors = pVertexBindingDivisors_;
      return *this;
    }

    operator VkPipelineVertexInputDivisorStateCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkPipelineVertexInputDivisorStateCreateInfoEXT*>(this);
    }

    operator VkPipelineVertexInputDivisorStateCreateInfoEXT &()
    {
      return *reinterpret_cast<VkPipelineVertexInputDivisorStateCreateInfoEXT*>(this);
    }

    bool operator==( PipelineVertexInputDivisorStateCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( vertexBindingDivisorCount == rhs.vertexBindingDivisorCount )
          && ( pVertexBindingDivisors == rhs.pVertexBindingDivisors );
    }

    bool operator!=( PipelineVertexInputDivisorStateCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineVertexInputDivisorStateCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    uint32_t vertexBindingDivisorCount;
    const VertexInputBindingDivisorDescriptionEXT* pVertexBindingDivisors;
  };
  static_assert( sizeof( PipelineVertexInputDivisorStateCreateInfoEXT ) == sizeof( VkPipelineVertexInputDivisorStateCreateInfoEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceVertexAttributeDivisorPropertiesEXT
  {
    PhysicalDeviceVertexAttributeDivisorPropertiesEXT( uint32_t maxVertexAttribDivisor_ = 0 )
      : maxVertexAttribDivisor( maxVertexAttribDivisor_ )
    {
    }

    PhysicalDeviceVertexAttributeDivisorPropertiesEXT( VkPhysicalDeviceVertexAttributeDivisorPropertiesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceVertexAttributeDivisorPropertiesEXT ) );
    }

    PhysicalDeviceVertexAttributeDivisorPropertiesEXT& operator=( VkPhysicalDeviceVertexAttributeDivisorPropertiesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceVertexAttributeDivisorPropertiesEXT ) );
      return *this;
    }
    PhysicalDeviceVertexAttributeDivisorPropertiesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceVertexAttributeDivisorPropertiesEXT& setMaxVertexAttribDivisor( uint32_t maxVertexAttribDivisor_ )
    {
      maxVertexAttribDivisor = maxVertexAttribDivisor_;
      return *this;
    }

    operator VkPhysicalDeviceVertexAttributeDivisorPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceVertexAttributeDivisorPropertiesEXT*>(this);
    }

    operator VkPhysicalDeviceVertexAttributeDivisorPropertiesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceVertexAttributeDivisorPropertiesEXT*>(this);
    }

    bool operator==( PhysicalDeviceVertexAttributeDivisorPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxVertexAttribDivisor == rhs.maxVertexAttribDivisor );
    }

    bool operator!=( PhysicalDeviceVertexAttributeDivisorPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceVertexAttributeDivisorPropertiesEXT;

  public:
    void* pNext = nullptr;
    uint32_t maxVertexAttribDivisor;
  };
  static_assert( sizeof( PhysicalDeviceVertexAttributeDivisorPropertiesEXT ) == sizeof( VkPhysicalDeviceVertexAttributeDivisorPropertiesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDevicePCIBusInfoPropertiesEXT
  {
    operator VkPhysicalDevicePCIBusInfoPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDevicePCIBusInfoPropertiesEXT*>(this);
    }

    operator VkPhysicalDevicePCIBusInfoPropertiesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDevicePCIBusInfoPropertiesEXT*>(this);
    }

    bool operator==( PhysicalDevicePCIBusInfoPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( pciDomain == rhs.pciDomain )
          && ( pciBus == rhs.pciBus )
          && ( pciDevice == rhs.pciDevice )
          && ( pciFunction == rhs.pciFunction );
    }

    bool operator!=( PhysicalDevicePCIBusInfoPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDevicePciBusInfoPropertiesEXT;

  public:
    void* pNext = nullptr;
    uint32_t pciDomain;
    uint32_t pciBus;
    uint32_t pciDevice;
    uint32_t pciFunction;
  };
  static_assert( sizeof( PhysicalDevicePCIBusInfoPropertiesEXT ) == sizeof( VkPhysicalDevicePCIBusInfoPropertiesEXT ), "struct and wrapper have different size!" );

#ifdef VK_USE_PLATFORM_ANDROID_KHR
  struct ImportAndroidHardwareBufferInfoANDROID
  {
    ImportAndroidHardwareBufferInfoANDROID( struct AHardwareBuffer* buffer_ = nullptr )
      : buffer( buffer_ )
    {
    }

    ImportAndroidHardwareBufferInfoANDROID( VkImportAndroidHardwareBufferInfoANDROID const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImportAndroidHardwareBufferInfoANDROID ) );
    }

    ImportAndroidHardwareBufferInfoANDROID& operator=( VkImportAndroidHardwareBufferInfoANDROID const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImportAndroidHardwareBufferInfoANDROID ) );
      return *this;
    }
    ImportAndroidHardwareBufferInfoANDROID& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImportAndroidHardwareBufferInfoANDROID& setBuffer( struct AHardwareBuffer* buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    operator VkImportAndroidHardwareBufferInfoANDROID const&() const
    {
      return *reinterpret_cast<const VkImportAndroidHardwareBufferInfoANDROID*>(this);
    }

    operator VkImportAndroidHardwareBufferInfoANDROID &()
    {
      return *reinterpret_cast<VkImportAndroidHardwareBufferInfoANDROID*>(this);
    }

    bool operator==( ImportAndroidHardwareBufferInfoANDROID const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( buffer == rhs.buffer );
    }

    bool operator!=( ImportAndroidHardwareBufferInfoANDROID const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImportAndroidHardwareBufferInfoANDROID;

  public:
    const void* pNext = nullptr;
    struct AHardwareBuffer* buffer;
  };
  static_assert( sizeof( ImportAndroidHardwareBufferInfoANDROID ) == sizeof( VkImportAndroidHardwareBufferInfoANDROID ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

#ifdef VK_USE_PLATFORM_ANDROID_KHR
  struct AndroidHardwareBufferUsageANDROID
  {
    operator VkAndroidHardwareBufferUsageANDROID const&() const
    {
      return *reinterpret_cast<const VkAndroidHardwareBufferUsageANDROID*>(this);
    }

    operator VkAndroidHardwareBufferUsageANDROID &()
    {
      return *reinterpret_cast<VkAndroidHardwareBufferUsageANDROID*>(this);
    }

    bool operator==( AndroidHardwareBufferUsageANDROID const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( androidHardwareBufferUsage == rhs.androidHardwareBufferUsage );
    }

    bool operator!=( AndroidHardwareBufferUsageANDROID const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAndroidHardwareBufferUsageANDROID;

  public:
    void* pNext = nullptr;
    uint64_t androidHardwareBufferUsage;
  };
  static_assert( sizeof( AndroidHardwareBufferUsageANDROID ) == sizeof( VkAndroidHardwareBufferUsageANDROID ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

#ifdef VK_USE_PLATFORM_ANDROID_KHR
  struct AndroidHardwareBufferPropertiesANDROID
  {
    operator VkAndroidHardwareBufferPropertiesANDROID const&() const
    {
      return *reinterpret_cast<const VkAndroidHardwareBufferPropertiesANDROID*>(this);
    }

    operator VkAndroidHardwareBufferPropertiesANDROID &()
    {
      return *reinterpret_cast<VkAndroidHardwareBufferPropertiesANDROID*>(this);
    }

    bool operator==( AndroidHardwareBufferPropertiesANDROID const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( allocationSize == rhs.allocationSize )
          && ( memoryTypeBits == rhs.memoryTypeBits );
    }

    bool operator!=( AndroidHardwareBufferPropertiesANDROID const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAndroidHardwareBufferPropertiesANDROID;

  public:
    void* pNext = nullptr;
    DeviceSize allocationSize;
    uint32_t memoryTypeBits;
  };
  static_assert( sizeof( AndroidHardwareBufferPropertiesANDROID ) == sizeof( VkAndroidHardwareBufferPropertiesANDROID ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

#ifdef VK_USE_PLATFORM_ANDROID_KHR
  struct MemoryGetAndroidHardwareBufferInfoANDROID
  {
    MemoryGetAndroidHardwareBufferInfoANDROID( DeviceMemory memory_ = DeviceMemory() )
      : memory( memory_ )
    {
    }

    MemoryGetAndroidHardwareBufferInfoANDROID( VkMemoryGetAndroidHardwareBufferInfoANDROID const & rhs )
    {
      memcpy( this, &rhs, sizeof( MemoryGetAndroidHardwareBufferInfoANDROID ) );
    }

    MemoryGetAndroidHardwareBufferInfoANDROID& operator=( VkMemoryGetAndroidHardwareBufferInfoANDROID const & rhs )
    {
      memcpy( this, &rhs, sizeof( MemoryGetAndroidHardwareBufferInfoANDROID ) );
      return *this;
    }
    MemoryGetAndroidHardwareBufferInfoANDROID& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    MemoryGetAndroidHardwareBufferInfoANDROID& setMemory( DeviceMemory memory_ )
    {
      memory = memory_;
      return *this;
    }

    operator VkMemoryGetAndroidHardwareBufferInfoANDROID const&() const
    {
      return *reinterpret_cast<const VkMemoryGetAndroidHardwareBufferInfoANDROID*>(this);
    }

    operator VkMemoryGetAndroidHardwareBufferInfoANDROID &()
    {
      return *reinterpret_cast<VkMemoryGetAndroidHardwareBufferInfoANDROID*>(this);
    }

    bool operator==( MemoryGetAndroidHardwareBufferInfoANDROID const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( memory == rhs.memory );
    }

    bool operator!=( MemoryGetAndroidHardwareBufferInfoANDROID const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eMemoryGetAndroidHardwareBufferInfoANDROID;

  public:
    const void* pNext = nullptr;
    DeviceMemory memory;
  };
  static_assert( sizeof( MemoryGetAndroidHardwareBufferInfoANDROID ) == sizeof( VkMemoryGetAndroidHardwareBufferInfoANDROID ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

  struct CommandBufferInheritanceConditionalRenderingInfoEXT
  {
    CommandBufferInheritanceConditionalRenderingInfoEXT( Bool32 conditionalRenderingEnable_ = 0 )
      : conditionalRenderingEnable( conditionalRenderingEnable_ )
    {
    }

    CommandBufferInheritanceConditionalRenderingInfoEXT( VkCommandBufferInheritanceConditionalRenderingInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferInheritanceConditionalRenderingInfoEXT ) );
    }

    CommandBufferInheritanceConditionalRenderingInfoEXT& operator=( VkCommandBufferInheritanceConditionalRenderingInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferInheritanceConditionalRenderingInfoEXT ) );
      return *this;
    }
    CommandBufferInheritanceConditionalRenderingInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CommandBufferInheritanceConditionalRenderingInfoEXT& setConditionalRenderingEnable( Bool32 conditionalRenderingEnable_ )
    {
      conditionalRenderingEnable = conditionalRenderingEnable_;
      return *this;
    }

    operator VkCommandBufferInheritanceConditionalRenderingInfoEXT const&() const
    {
      return *reinterpret_cast<const VkCommandBufferInheritanceConditionalRenderingInfoEXT*>(this);
    }

    operator VkCommandBufferInheritanceConditionalRenderingInfoEXT &()
    {
      return *reinterpret_cast<VkCommandBufferInheritanceConditionalRenderingInfoEXT*>(this);
    }

    bool operator==( CommandBufferInheritanceConditionalRenderingInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( conditionalRenderingEnable == rhs.conditionalRenderingEnable );
    }

    bool operator!=( CommandBufferInheritanceConditionalRenderingInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCommandBufferInheritanceConditionalRenderingInfoEXT;

  public:
    const void* pNext = nullptr;
    Bool32 conditionalRenderingEnable;
  };
  static_assert( sizeof( CommandBufferInheritanceConditionalRenderingInfoEXT ) == sizeof( VkCommandBufferInheritanceConditionalRenderingInfoEXT ), "struct and wrapper have different size!" );

#ifdef VK_USE_PLATFORM_ANDROID_KHR
  struct ExternalFormatANDROID
  {
    ExternalFormatANDROID( uint64_t externalFormat_ = 0 )
      : externalFormat( externalFormat_ )
    {
    }

    ExternalFormatANDROID( VkExternalFormatANDROID const & rhs )
    {
      memcpy( this, &rhs, sizeof( ExternalFormatANDROID ) );
    }

    ExternalFormatANDROID& operator=( VkExternalFormatANDROID const & rhs )
    {
      memcpy( this, &rhs, sizeof( ExternalFormatANDROID ) );
      return *this;
    }
    ExternalFormatANDROID& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ExternalFormatANDROID& setExternalFormat( uint64_t externalFormat_ )
    {
      externalFormat = externalFormat_;
      return *this;
    }

    operator VkExternalFormatANDROID const&() const
    {
      return *reinterpret_cast<const VkExternalFormatANDROID*>(this);
    }

    operator VkExternalFormatANDROID &()
    {
      return *reinterpret_cast<VkExternalFormatANDROID*>(this);
    }

    bool operator==( ExternalFormatANDROID const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( externalFormat == rhs.externalFormat );
    }

    bool operator!=( ExternalFormatANDROID const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eExternalFormatANDROID;

  public:
    void* pNext = nullptr;
    uint64_t externalFormat;
  };
  static_assert( sizeof( ExternalFormatANDROID ) == sizeof( VkExternalFormatANDROID ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

  struct PhysicalDevice8BitStorageFeaturesKHR
  {
    PhysicalDevice8BitStorageFeaturesKHR( Bool32 storageBuffer8BitAccess_ = 0,
                                          Bool32 uniformAndStorageBuffer8BitAccess_ = 0,
                                          Bool32 storagePushConstant8_ = 0 )
      : storageBuffer8BitAccess( storageBuffer8BitAccess_ )
      , uniformAndStorageBuffer8BitAccess( uniformAndStorageBuffer8BitAccess_ )
      , storagePushConstant8( storagePushConstant8_ )
    {
    }

    PhysicalDevice8BitStorageFeaturesKHR( VkPhysicalDevice8BitStorageFeaturesKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDevice8BitStorageFeaturesKHR ) );
    }

    PhysicalDevice8BitStorageFeaturesKHR& operator=( VkPhysicalDevice8BitStorageFeaturesKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDevice8BitStorageFeaturesKHR ) );
      return *this;
    }
    PhysicalDevice8BitStorageFeaturesKHR& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDevice8BitStorageFeaturesKHR& setStorageBuffer8BitAccess( Bool32 storageBuffer8BitAccess_ )
    {
      storageBuffer8BitAccess = storageBuffer8BitAccess_;
      return *this;
    }

    PhysicalDevice8BitStorageFeaturesKHR& setUniformAndStorageBuffer8BitAccess( Bool32 uniformAndStorageBuffer8BitAccess_ )
    {
      uniformAndStorageBuffer8BitAccess = uniformAndStorageBuffer8BitAccess_;
      return *this;
    }

    PhysicalDevice8BitStorageFeaturesKHR& setStoragePushConstant8( Bool32 storagePushConstant8_ )
    {
      storagePushConstant8 = storagePushConstant8_;
      return *this;
    }

    operator VkPhysicalDevice8BitStorageFeaturesKHR const&() const
    {
      return *reinterpret_cast<const VkPhysicalDevice8BitStorageFeaturesKHR*>(this);
    }

    operator VkPhysicalDevice8BitStorageFeaturesKHR &()
    {
      return *reinterpret_cast<VkPhysicalDevice8BitStorageFeaturesKHR*>(this);
    }

    bool operator==( PhysicalDevice8BitStorageFeaturesKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( storageBuffer8BitAccess == rhs.storageBuffer8BitAccess )
          && ( uniformAndStorageBuffer8BitAccess == rhs.uniformAndStorageBuffer8BitAccess )
          && ( storagePushConstant8 == rhs.storagePushConstant8 );
    }

    bool operator!=( PhysicalDevice8BitStorageFeaturesKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDevice8BitStorageFeaturesKHR;

  public:
    void* pNext = nullptr;
    Bool32 storageBuffer8BitAccess;
    Bool32 uniformAndStorageBuffer8BitAccess;
    Bool32 storagePushConstant8;
  };
  static_assert( sizeof( PhysicalDevice8BitStorageFeaturesKHR ) == sizeof( VkPhysicalDevice8BitStorageFeaturesKHR ), "struct and wrapper have different size!" );

  struct PhysicalDeviceConditionalRenderingFeaturesEXT
  {
    PhysicalDeviceConditionalRenderingFeaturesEXT( Bool32 conditionalRendering_ = 0,
                                                   Bool32 inheritedConditionalRendering_ = 0 )
      : conditionalRendering( conditionalRendering_ )
      , inheritedConditionalRendering( inheritedConditionalRendering_ )
    {
    }

    PhysicalDeviceConditionalRenderingFeaturesEXT( VkPhysicalDeviceConditionalRenderingFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceConditionalRenderingFeaturesEXT ) );
    }

    PhysicalDeviceConditionalRenderingFeaturesEXT& operator=( VkPhysicalDeviceConditionalRenderingFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceConditionalRenderingFeaturesEXT ) );
      return *this;
    }
    PhysicalDeviceConditionalRenderingFeaturesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceConditionalRenderingFeaturesEXT& setConditionalRendering( Bool32 conditionalRendering_ )
    {
      conditionalRendering = conditionalRendering_;
      return *this;
    }

    PhysicalDeviceConditionalRenderingFeaturesEXT& setInheritedConditionalRendering( Bool32 inheritedConditionalRendering_ )
    {
      inheritedConditionalRendering = inheritedConditionalRendering_;
      return *this;
    }

    operator VkPhysicalDeviceConditionalRenderingFeaturesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceConditionalRenderingFeaturesEXT*>(this);
    }

    operator VkPhysicalDeviceConditionalRenderingFeaturesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceConditionalRenderingFeaturesEXT*>(this);
    }

    bool operator==( PhysicalDeviceConditionalRenderingFeaturesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( conditionalRendering == rhs.conditionalRendering )
          && ( inheritedConditionalRendering == rhs.inheritedConditionalRendering );
    }

    bool operator!=( PhysicalDeviceConditionalRenderingFeaturesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceConditionalRenderingFeaturesEXT;

  public:
    void* pNext = nullptr;
    Bool32 conditionalRendering;
    Bool32 inheritedConditionalRendering;
  };
  static_assert( sizeof( PhysicalDeviceConditionalRenderingFeaturesEXT ) == sizeof( VkPhysicalDeviceConditionalRenderingFeaturesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceVulkanMemoryModelFeaturesKHR
  {
    PhysicalDeviceVulkanMemoryModelFeaturesKHR( Bool32 vulkanMemoryModel_ = 0,
                                                Bool32 vulkanMemoryModelDeviceScope_ = 0 )
      : vulkanMemoryModel( vulkanMemoryModel_ )
      , vulkanMemoryModelDeviceScope( vulkanMemoryModelDeviceScope_ )
    {
    }

    PhysicalDeviceVulkanMemoryModelFeaturesKHR( VkPhysicalDeviceVulkanMemoryModelFeaturesKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceVulkanMemoryModelFeaturesKHR ) );
    }

    PhysicalDeviceVulkanMemoryModelFeaturesKHR& operator=( VkPhysicalDeviceVulkanMemoryModelFeaturesKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceVulkanMemoryModelFeaturesKHR ) );
      return *this;
    }
    PhysicalDeviceVulkanMemoryModelFeaturesKHR& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceVulkanMemoryModelFeaturesKHR& setVulkanMemoryModel( Bool32 vulkanMemoryModel_ )
    {
      vulkanMemoryModel = vulkanMemoryModel_;
      return *this;
    }

    PhysicalDeviceVulkanMemoryModelFeaturesKHR& setVulkanMemoryModelDeviceScope( Bool32 vulkanMemoryModelDeviceScope_ )
    {
      vulkanMemoryModelDeviceScope = vulkanMemoryModelDeviceScope_;
      return *this;
    }

    operator VkPhysicalDeviceVulkanMemoryModelFeaturesKHR const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceVulkanMemoryModelFeaturesKHR*>(this);
    }

    operator VkPhysicalDeviceVulkanMemoryModelFeaturesKHR &()
    {
      return *reinterpret_cast<VkPhysicalDeviceVulkanMemoryModelFeaturesKHR*>(this);
    }

    bool operator==( PhysicalDeviceVulkanMemoryModelFeaturesKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( vulkanMemoryModel == rhs.vulkanMemoryModel )
          && ( vulkanMemoryModelDeviceScope == rhs.vulkanMemoryModelDeviceScope );
    }

    bool operator!=( PhysicalDeviceVulkanMemoryModelFeaturesKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceVulkanMemoryModelFeaturesKHR;

  public:
    void* pNext = nullptr;
    Bool32 vulkanMemoryModel;
    Bool32 vulkanMemoryModelDeviceScope;
  };
  static_assert( sizeof( PhysicalDeviceVulkanMemoryModelFeaturesKHR ) == sizeof( VkPhysicalDeviceVulkanMemoryModelFeaturesKHR ), "struct and wrapper have different size!" );

  struct PhysicalDeviceShaderAtomicInt64FeaturesKHR
  {
    PhysicalDeviceShaderAtomicInt64FeaturesKHR( Bool32 shaderBufferInt64Atomics_ = 0,
                                                Bool32 shaderSharedInt64Atomics_ = 0 )
      : shaderBufferInt64Atomics( shaderBufferInt64Atomics_ )
      , shaderSharedInt64Atomics( shaderSharedInt64Atomics_ )
    {
    }

    PhysicalDeviceShaderAtomicInt64FeaturesKHR( VkPhysicalDeviceShaderAtomicInt64FeaturesKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceShaderAtomicInt64FeaturesKHR ) );
    }

    PhysicalDeviceShaderAtomicInt64FeaturesKHR& operator=( VkPhysicalDeviceShaderAtomicInt64FeaturesKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceShaderAtomicInt64FeaturesKHR ) );
      return *this;
    }
    PhysicalDeviceShaderAtomicInt64FeaturesKHR& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceShaderAtomicInt64FeaturesKHR& setShaderBufferInt64Atomics( Bool32 shaderBufferInt64Atomics_ )
    {
      shaderBufferInt64Atomics = shaderBufferInt64Atomics_;
      return *this;
    }

    PhysicalDeviceShaderAtomicInt64FeaturesKHR& setShaderSharedInt64Atomics( Bool32 shaderSharedInt64Atomics_ )
    {
      shaderSharedInt64Atomics = shaderSharedInt64Atomics_;
      return *this;
    }

    operator VkPhysicalDeviceShaderAtomicInt64FeaturesKHR const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceShaderAtomicInt64FeaturesKHR*>(this);
    }

    operator VkPhysicalDeviceShaderAtomicInt64FeaturesKHR &()
    {
      return *reinterpret_cast<VkPhysicalDeviceShaderAtomicInt64FeaturesKHR*>(this);
    }

    bool operator==( PhysicalDeviceShaderAtomicInt64FeaturesKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( shaderBufferInt64Atomics == rhs.shaderBufferInt64Atomics )
          && ( shaderSharedInt64Atomics == rhs.shaderSharedInt64Atomics );
    }

    bool operator!=( PhysicalDeviceShaderAtomicInt64FeaturesKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceShaderAtomicInt64FeaturesKHR;

  public:
    void* pNext = nullptr;
    Bool32 shaderBufferInt64Atomics;
    Bool32 shaderSharedInt64Atomics;
  };
  static_assert( sizeof( PhysicalDeviceShaderAtomicInt64FeaturesKHR ) == sizeof( VkPhysicalDeviceShaderAtomicInt64FeaturesKHR ), "struct and wrapper have different size!" );

  struct PhysicalDeviceVertexAttributeDivisorFeaturesEXT
  {
    PhysicalDeviceVertexAttributeDivisorFeaturesEXT( Bool32 vertexAttributeInstanceRateDivisor_ = 0,
                                                     Bool32 vertexAttributeInstanceRateZeroDivisor_ = 0 )
      : vertexAttributeInstanceRateDivisor( vertexAttributeInstanceRateDivisor_ )
      , vertexAttributeInstanceRateZeroDivisor( vertexAttributeInstanceRateZeroDivisor_ )
    {
    }

    PhysicalDeviceVertexAttributeDivisorFeaturesEXT( VkPhysicalDeviceVertexAttributeDivisorFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceVertexAttributeDivisorFeaturesEXT ) );
    }

    PhysicalDeviceVertexAttributeDivisorFeaturesEXT& operator=( VkPhysicalDeviceVertexAttributeDivisorFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceVertexAttributeDivisorFeaturesEXT ) );
      return *this;
    }
    PhysicalDeviceVertexAttributeDivisorFeaturesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceVertexAttributeDivisorFeaturesEXT& setVertexAttributeInstanceRateDivisor( Bool32 vertexAttributeInstanceRateDivisor_ )
    {
      vertexAttributeInstanceRateDivisor = vertexAttributeInstanceRateDivisor_;
      return *this;
    }

    PhysicalDeviceVertexAttributeDivisorFeaturesEXT& setVertexAttributeInstanceRateZeroDivisor( Bool32 vertexAttributeInstanceRateZeroDivisor_ )
    {
      vertexAttributeInstanceRateZeroDivisor = vertexAttributeInstanceRateZeroDivisor_;
      return *this;
    }

    operator VkPhysicalDeviceVertexAttributeDivisorFeaturesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceVertexAttributeDivisorFeaturesEXT*>(this);
    }

    operator VkPhysicalDeviceVertexAttributeDivisorFeaturesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceVertexAttributeDivisorFeaturesEXT*>(this);
    }

    bool operator==( PhysicalDeviceVertexAttributeDivisorFeaturesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( vertexAttributeInstanceRateDivisor == rhs.vertexAttributeInstanceRateDivisor )
          && ( vertexAttributeInstanceRateZeroDivisor == rhs.vertexAttributeInstanceRateZeroDivisor );
    }

    bool operator!=( PhysicalDeviceVertexAttributeDivisorFeaturesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceVertexAttributeDivisorFeaturesEXT;

  public:
    void* pNext = nullptr;
    Bool32 vertexAttributeInstanceRateDivisor;
    Bool32 vertexAttributeInstanceRateZeroDivisor;
  };
  static_assert( sizeof( PhysicalDeviceVertexAttributeDivisorFeaturesEXT ) == sizeof( VkPhysicalDeviceVertexAttributeDivisorFeaturesEXT ), "struct and wrapper have different size!" );

  struct ImageViewASTCDecodeModeEXT
  {
    ImageViewASTCDecodeModeEXT( Format decodeMode_ = Format::eUndefined )
      : decodeMode( decodeMode_ )
    {
    }

    ImageViewASTCDecodeModeEXT( VkImageViewASTCDecodeModeEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageViewASTCDecodeModeEXT ) );
    }

    ImageViewASTCDecodeModeEXT& operator=( VkImageViewASTCDecodeModeEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageViewASTCDecodeModeEXT ) );
      return *this;
    }
    ImageViewASTCDecodeModeEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImageViewASTCDecodeModeEXT& setDecodeMode( Format decodeMode_ )
    {
      decodeMode = decodeMode_;
      return *this;
    }

    operator VkImageViewASTCDecodeModeEXT const&() const
    {
      return *reinterpret_cast<const VkImageViewASTCDecodeModeEXT*>(this);
    }

    operator VkImageViewASTCDecodeModeEXT &()
    {
      return *reinterpret_cast<VkImageViewASTCDecodeModeEXT*>(this);
    }

    bool operator==( ImageViewASTCDecodeModeEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( decodeMode == rhs.decodeMode );
    }

    bool operator!=( ImageViewASTCDecodeModeEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageViewAstcDecodeModeEXT;

  public:
    const void* pNext = nullptr;
    Format decodeMode;
  };
  static_assert( sizeof( ImageViewASTCDecodeModeEXT ) == sizeof( VkImageViewASTCDecodeModeEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceASTCDecodeFeaturesEXT
  {
    PhysicalDeviceASTCDecodeFeaturesEXT( Bool32 decodeModeSharedExponent_ = 0 )
      : decodeModeSharedExponent( decodeModeSharedExponent_ )
    {
    }

    PhysicalDeviceASTCDecodeFeaturesEXT( VkPhysicalDeviceASTCDecodeFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceASTCDecodeFeaturesEXT ) );
    }

    PhysicalDeviceASTCDecodeFeaturesEXT& operator=( VkPhysicalDeviceASTCDecodeFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceASTCDecodeFeaturesEXT ) );
      return *this;
    }
    PhysicalDeviceASTCDecodeFeaturesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceASTCDecodeFeaturesEXT& setDecodeModeSharedExponent( Bool32 decodeModeSharedExponent_ )
    {
      decodeModeSharedExponent = decodeModeSharedExponent_;
      return *this;
    }

    operator VkPhysicalDeviceASTCDecodeFeaturesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceASTCDecodeFeaturesEXT*>(this);
    }

    operator VkPhysicalDeviceASTCDecodeFeaturesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceASTCDecodeFeaturesEXT*>(this);
    }

    bool operator==( PhysicalDeviceASTCDecodeFeaturesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( decodeModeSharedExponent == rhs.decodeModeSharedExponent );
    }

    bool operator!=( PhysicalDeviceASTCDecodeFeaturesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceAstcDecodeFeaturesEXT;

  public:
    void* pNext = nullptr;
    Bool32 decodeModeSharedExponent;
  };
  static_assert( sizeof( PhysicalDeviceASTCDecodeFeaturesEXT ) == sizeof( VkPhysicalDeviceASTCDecodeFeaturesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceTransformFeedbackFeaturesEXT
  {
    PhysicalDeviceTransformFeedbackFeaturesEXT( Bool32 transformFeedback_ = 0,
                                                Bool32 geometryStreams_ = 0 )
      : transformFeedback( transformFeedback_ )
      , geometryStreams( geometryStreams_ )
    {
    }

    PhysicalDeviceTransformFeedbackFeaturesEXT( VkPhysicalDeviceTransformFeedbackFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceTransformFeedbackFeaturesEXT ) );
    }

    PhysicalDeviceTransformFeedbackFeaturesEXT& operator=( VkPhysicalDeviceTransformFeedbackFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceTransformFeedbackFeaturesEXT ) );
      return *this;
    }
    PhysicalDeviceTransformFeedbackFeaturesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceTransformFeedbackFeaturesEXT& setTransformFeedback( Bool32 transformFeedback_ )
    {
      transformFeedback = transformFeedback_;
      return *this;
    }

    PhysicalDeviceTransformFeedbackFeaturesEXT& setGeometryStreams( Bool32 geometryStreams_ )
    {
      geometryStreams = geometryStreams_;
      return *this;
    }

    operator VkPhysicalDeviceTransformFeedbackFeaturesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceTransformFeedbackFeaturesEXT*>(this);
    }

    operator VkPhysicalDeviceTransformFeedbackFeaturesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceTransformFeedbackFeaturesEXT*>(this);
    }

    bool operator==( PhysicalDeviceTransformFeedbackFeaturesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( transformFeedback == rhs.transformFeedback )
          && ( geometryStreams == rhs.geometryStreams );
    }

    bool operator!=( PhysicalDeviceTransformFeedbackFeaturesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceTransformFeedbackFeaturesEXT;

  public:
    void* pNext = nullptr;
    Bool32 transformFeedback;
    Bool32 geometryStreams;
  };
  static_assert( sizeof( PhysicalDeviceTransformFeedbackFeaturesEXT ) == sizeof( VkPhysicalDeviceTransformFeedbackFeaturesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceTransformFeedbackPropertiesEXT
  {
    operator VkPhysicalDeviceTransformFeedbackPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceTransformFeedbackPropertiesEXT*>(this);
    }

    operator VkPhysicalDeviceTransformFeedbackPropertiesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceTransformFeedbackPropertiesEXT*>(this);
    }

    bool operator==( PhysicalDeviceTransformFeedbackPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxTransformFeedbackStreams == rhs.maxTransformFeedbackStreams )
          && ( maxTransformFeedbackBuffers == rhs.maxTransformFeedbackBuffers )
          && ( maxTransformFeedbackBufferSize == rhs.maxTransformFeedbackBufferSize )
          && ( maxTransformFeedbackStreamDataSize == rhs.maxTransformFeedbackStreamDataSize )
          && ( maxTransformFeedbackBufferDataSize == rhs.maxTransformFeedbackBufferDataSize )
          && ( maxTransformFeedbackBufferDataStride == rhs.maxTransformFeedbackBufferDataStride )
          && ( transformFeedbackQueries == rhs.transformFeedbackQueries )
          && ( transformFeedbackStreamsLinesTriangles == rhs.transformFeedbackStreamsLinesTriangles )
          && ( transformFeedbackRasterizationStreamSelect == rhs.transformFeedbackRasterizationStreamSelect )
          && ( transformFeedbackDraw == rhs.transformFeedbackDraw );
    }

    bool operator!=( PhysicalDeviceTransformFeedbackPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceTransformFeedbackPropertiesEXT;

  public:
    void* pNext = nullptr;
    uint32_t maxTransformFeedbackStreams;
    uint32_t maxTransformFeedbackBuffers;
    DeviceSize maxTransformFeedbackBufferSize;
    uint32_t maxTransformFeedbackStreamDataSize;
    uint32_t maxTransformFeedbackBufferDataSize;
    uint32_t maxTransformFeedbackBufferDataStride;
    Bool32 transformFeedbackQueries;
    Bool32 transformFeedbackStreamsLinesTriangles;
    Bool32 transformFeedbackRasterizationStreamSelect;
    Bool32 transformFeedbackDraw;
  };
  static_assert( sizeof( PhysicalDeviceTransformFeedbackPropertiesEXT ) == sizeof( VkPhysicalDeviceTransformFeedbackPropertiesEXT ), "struct and wrapper have different size!" );

  struct PipelineRasterizationStateStreamCreateInfoEXT
  {
    PipelineRasterizationStateStreamCreateInfoEXT( PipelineRasterizationStateStreamCreateFlagsEXT flags_ = PipelineRasterizationStateStreamCreateFlagsEXT(),
                                                   uint32_t rasterizationStream_ = 0 )
      : flags( flags_ )
      , rasterizationStream( rasterizationStream_ )
    {
    }

    PipelineRasterizationStateStreamCreateInfoEXT( VkPipelineRasterizationStateStreamCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineRasterizationStateStreamCreateInfoEXT ) );
    }

    PipelineRasterizationStateStreamCreateInfoEXT& operator=( VkPipelineRasterizationStateStreamCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineRasterizationStateStreamCreateInfoEXT ) );
      return *this;
    }
    PipelineRasterizationStateStreamCreateInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineRasterizationStateStreamCreateInfoEXT& setFlags( PipelineRasterizationStateStreamCreateFlagsEXT flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineRasterizationStateStreamCreateInfoEXT& setRasterizationStream( uint32_t rasterizationStream_ )
    {
      rasterizationStream = rasterizationStream_;
      return *this;
    }

    operator VkPipelineRasterizationStateStreamCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkPipelineRasterizationStateStreamCreateInfoEXT*>(this);
    }

    operator VkPipelineRasterizationStateStreamCreateInfoEXT &()
    {
      return *reinterpret_cast<VkPipelineRasterizationStateStreamCreateInfoEXT*>(this);
    }

    bool operator==( PipelineRasterizationStateStreamCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( rasterizationStream == rhs.rasterizationStream );
    }

    bool operator!=( PipelineRasterizationStateStreamCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineRasterizationStateStreamCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    PipelineRasterizationStateStreamCreateFlagsEXT flags;
    uint32_t rasterizationStream;
  };
  static_assert( sizeof( PipelineRasterizationStateStreamCreateInfoEXT ) == sizeof( VkPipelineRasterizationStateStreamCreateInfoEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceRepresentativeFragmentTestFeaturesNV
  {
    PhysicalDeviceRepresentativeFragmentTestFeaturesNV( Bool32 representativeFragmentTest_ = 0 )
      : representativeFragmentTest( representativeFragmentTest_ )
    {
    }

    PhysicalDeviceRepresentativeFragmentTestFeaturesNV( VkPhysicalDeviceRepresentativeFragmentTestFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceRepresentativeFragmentTestFeaturesNV ) );
    }

    PhysicalDeviceRepresentativeFragmentTestFeaturesNV& operator=( VkPhysicalDeviceRepresentativeFragmentTestFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceRepresentativeFragmentTestFeaturesNV ) );
      return *this;
    }
    PhysicalDeviceRepresentativeFragmentTestFeaturesNV& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceRepresentativeFragmentTestFeaturesNV& setRepresentativeFragmentTest( Bool32 representativeFragmentTest_ )
    {
      representativeFragmentTest = representativeFragmentTest_;
      return *this;
    }

    operator VkPhysicalDeviceRepresentativeFragmentTestFeaturesNV const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceRepresentativeFragmentTestFeaturesNV*>(this);
    }

    operator VkPhysicalDeviceRepresentativeFragmentTestFeaturesNV &()
    {
      return *reinterpret_cast<VkPhysicalDeviceRepresentativeFragmentTestFeaturesNV*>(this);
    }

    bool operator==( PhysicalDeviceRepresentativeFragmentTestFeaturesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( representativeFragmentTest == rhs.representativeFragmentTest );
    }

    bool operator!=( PhysicalDeviceRepresentativeFragmentTestFeaturesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceRepresentativeFragmentTestFeaturesNV;

  public:
    void* pNext = nullptr;
    Bool32 representativeFragmentTest;
  };
  static_assert( sizeof( PhysicalDeviceRepresentativeFragmentTestFeaturesNV ) == sizeof( VkPhysicalDeviceRepresentativeFragmentTestFeaturesNV ), "struct and wrapper have different size!" );

  struct PipelineRepresentativeFragmentTestStateCreateInfoNV
  {
    PipelineRepresentativeFragmentTestStateCreateInfoNV( Bool32 representativeFragmentTestEnable_ = 0 )
      : representativeFragmentTestEnable( representativeFragmentTestEnable_ )
    {
    }

    PipelineRepresentativeFragmentTestStateCreateInfoNV( VkPipelineRepresentativeFragmentTestStateCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineRepresentativeFragmentTestStateCreateInfoNV ) );
    }

    PipelineRepresentativeFragmentTestStateCreateInfoNV& operator=( VkPipelineRepresentativeFragmentTestStateCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineRepresentativeFragmentTestStateCreateInfoNV ) );
      return *this;
    }
    PipelineRepresentativeFragmentTestStateCreateInfoNV& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineRepresentativeFragmentTestStateCreateInfoNV& setRepresentativeFragmentTestEnable( Bool32 representativeFragmentTestEnable_ )
    {
      representativeFragmentTestEnable = representativeFragmentTestEnable_;
      return *this;
    }

    operator VkPipelineRepresentativeFragmentTestStateCreateInfoNV const&() const
    {
      return *reinterpret_cast<const VkPipelineRepresentativeFragmentTestStateCreateInfoNV*>(this);
    }

    operator VkPipelineRepresentativeFragmentTestStateCreateInfoNV &()
    {
      return *reinterpret_cast<VkPipelineRepresentativeFragmentTestStateCreateInfoNV*>(this);
    }

    bool operator==( PipelineRepresentativeFragmentTestStateCreateInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( representativeFragmentTestEnable == rhs.representativeFragmentTestEnable );
    }

    bool operator!=( PipelineRepresentativeFragmentTestStateCreateInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineRepresentativeFragmentTestStateCreateInfoNV;

  public:
    const void* pNext = nullptr;
    Bool32 representativeFragmentTestEnable;
  };
  static_assert( sizeof( PipelineRepresentativeFragmentTestStateCreateInfoNV ) == sizeof( VkPipelineRepresentativeFragmentTestStateCreateInfoNV ), "struct and wrapper have different size!" );

  struct PhysicalDeviceExclusiveScissorFeaturesNV
  {
    PhysicalDeviceExclusiveScissorFeaturesNV( Bool32 exclusiveScissor_ = 0 )
      : exclusiveScissor( exclusiveScissor_ )
    {
    }

    PhysicalDeviceExclusiveScissorFeaturesNV( VkPhysicalDeviceExclusiveScissorFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceExclusiveScissorFeaturesNV ) );
    }

    PhysicalDeviceExclusiveScissorFeaturesNV& operator=( VkPhysicalDeviceExclusiveScissorFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceExclusiveScissorFeaturesNV ) );
      return *this;
    }
    PhysicalDeviceExclusiveScissorFeaturesNV& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceExclusiveScissorFeaturesNV& setExclusiveScissor( Bool32 exclusiveScissor_ )
    {
      exclusiveScissor = exclusiveScissor_;
      return *this;
    }

    operator VkPhysicalDeviceExclusiveScissorFeaturesNV const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceExclusiveScissorFeaturesNV*>(this);
    }

    operator VkPhysicalDeviceExclusiveScissorFeaturesNV &()
    {
      return *reinterpret_cast<VkPhysicalDeviceExclusiveScissorFeaturesNV*>(this);
    }

    bool operator==( PhysicalDeviceExclusiveScissorFeaturesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( exclusiveScissor == rhs.exclusiveScissor );
    }

    bool operator!=( PhysicalDeviceExclusiveScissorFeaturesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceExclusiveScissorFeaturesNV;

  public:
    void* pNext = nullptr;
    Bool32 exclusiveScissor;
  };
  static_assert( sizeof( PhysicalDeviceExclusiveScissorFeaturesNV ) == sizeof( VkPhysicalDeviceExclusiveScissorFeaturesNV ), "struct and wrapper have different size!" );

  struct PipelineViewportExclusiveScissorStateCreateInfoNV
  {
    PipelineViewportExclusiveScissorStateCreateInfoNV( uint32_t exclusiveScissorCount_ = 0,
                                                       const Rect2D* pExclusiveScissors_ = nullptr )
      : exclusiveScissorCount( exclusiveScissorCount_ )
      , pExclusiveScissors( pExclusiveScissors_ )
    {
    }

    PipelineViewportExclusiveScissorStateCreateInfoNV( VkPipelineViewportExclusiveScissorStateCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineViewportExclusiveScissorStateCreateInfoNV ) );
    }

    PipelineViewportExclusiveScissorStateCreateInfoNV& operator=( VkPipelineViewportExclusiveScissorStateCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineViewportExclusiveScissorStateCreateInfoNV ) );
      return *this;
    }
    PipelineViewportExclusiveScissorStateCreateInfoNV& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineViewportExclusiveScissorStateCreateInfoNV& setExclusiveScissorCount( uint32_t exclusiveScissorCount_ )
    {
      exclusiveScissorCount = exclusiveScissorCount_;
      return *this;
    }

    PipelineViewportExclusiveScissorStateCreateInfoNV& setPExclusiveScissors( const Rect2D* pExclusiveScissors_ )
    {
      pExclusiveScissors = pExclusiveScissors_;
      return *this;
    }

    operator VkPipelineViewportExclusiveScissorStateCreateInfoNV const&() const
    {
      return *reinterpret_cast<const VkPipelineViewportExclusiveScissorStateCreateInfoNV*>(this);
    }

    operator VkPipelineViewportExclusiveScissorStateCreateInfoNV &()
    {
      return *reinterpret_cast<VkPipelineViewportExclusiveScissorStateCreateInfoNV*>(this);
    }

    bool operator==( PipelineViewportExclusiveScissorStateCreateInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( exclusiveScissorCount == rhs.exclusiveScissorCount )
          && ( pExclusiveScissors == rhs.pExclusiveScissors );
    }

    bool operator!=( PipelineViewportExclusiveScissorStateCreateInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineViewportExclusiveScissorStateCreateInfoNV;

  public:
    const void* pNext = nullptr;
    uint32_t exclusiveScissorCount;
    const Rect2D* pExclusiveScissors;
  };
  static_assert( sizeof( PipelineViewportExclusiveScissorStateCreateInfoNV ) == sizeof( VkPipelineViewportExclusiveScissorStateCreateInfoNV ), "struct and wrapper have different size!" );

  struct PhysicalDeviceCornerSampledImageFeaturesNV
  {
    PhysicalDeviceCornerSampledImageFeaturesNV( Bool32 cornerSampledImage_ = 0 )
      : cornerSampledImage( cornerSampledImage_ )
    {
    }

    PhysicalDeviceCornerSampledImageFeaturesNV( VkPhysicalDeviceCornerSampledImageFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceCornerSampledImageFeaturesNV ) );
    }

    PhysicalDeviceCornerSampledImageFeaturesNV& operator=( VkPhysicalDeviceCornerSampledImageFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceCornerSampledImageFeaturesNV ) );
      return *this;
    }
    PhysicalDeviceCornerSampledImageFeaturesNV& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceCornerSampledImageFeaturesNV& setCornerSampledImage( Bool32 cornerSampledImage_ )
    {
      cornerSampledImage = cornerSampledImage_;
      return *this;
    }

    operator VkPhysicalDeviceCornerSampledImageFeaturesNV const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceCornerSampledImageFeaturesNV*>(this);
    }

    operator VkPhysicalDeviceCornerSampledImageFeaturesNV &()
    {
      return *reinterpret_cast<VkPhysicalDeviceCornerSampledImageFeaturesNV*>(this);
    }

    bool operator==( PhysicalDeviceCornerSampledImageFeaturesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( cornerSampledImage == rhs.cornerSampledImage );
    }

    bool operator!=( PhysicalDeviceCornerSampledImageFeaturesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceCornerSampledImageFeaturesNV;

  public:
    void* pNext = nullptr;
    Bool32 cornerSampledImage;
  };
  static_assert( sizeof( PhysicalDeviceCornerSampledImageFeaturesNV ) == sizeof( VkPhysicalDeviceCornerSampledImageFeaturesNV ), "struct and wrapper have different size!" );

  struct PhysicalDeviceComputeShaderDerivativesFeaturesNV
  {
    PhysicalDeviceComputeShaderDerivativesFeaturesNV( Bool32 computeDerivativeGroupQuads_ = 0,
                                                      Bool32 computeDerivativeGroupLinear_ = 0 )
      : computeDerivativeGroupQuads( computeDerivativeGroupQuads_ )
      , computeDerivativeGroupLinear( computeDerivativeGroupLinear_ )
    {
    }

    PhysicalDeviceComputeShaderDerivativesFeaturesNV( VkPhysicalDeviceComputeShaderDerivativesFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceComputeShaderDerivativesFeaturesNV ) );
    }

    PhysicalDeviceComputeShaderDerivativesFeaturesNV& operator=( VkPhysicalDeviceComputeShaderDerivativesFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceComputeShaderDerivativesFeaturesNV ) );
      return *this;
    }
    PhysicalDeviceComputeShaderDerivativesFeaturesNV& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceComputeShaderDerivativesFeaturesNV& setComputeDerivativeGroupQuads( Bool32 computeDerivativeGroupQuads_ )
    {
      computeDerivativeGroupQuads = computeDerivativeGroupQuads_;
      return *this;
    }

    PhysicalDeviceComputeShaderDerivativesFeaturesNV& setComputeDerivativeGroupLinear( Bool32 computeDerivativeGroupLinear_ )
    {
      computeDerivativeGroupLinear = computeDerivativeGroupLinear_;
      return *this;
    }

    operator VkPhysicalDeviceComputeShaderDerivativesFeaturesNV const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceComputeShaderDerivativesFeaturesNV*>(this);
    }

    operator VkPhysicalDeviceComputeShaderDerivativesFeaturesNV &()
    {
      return *reinterpret_cast<VkPhysicalDeviceComputeShaderDerivativesFeaturesNV*>(this);
    }

    bool operator==( PhysicalDeviceComputeShaderDerivativesFeaturesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( computeDerivativeGroupQuads == rhs.computeDerivativeGroupQuads )
          && ( computeDerivativeGroupLinear == rhs.computeDerivativeGroupLinear );
    }

    bool operator!=( PhysicalDeviceComputeShaderDerivativesFeaturesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceComputeShaderDerivativesFeaturesNV;

  public:
    void* pNext = nullptr;
    Bool32 computeDerivativeGroupQuads;
    Bool32 computeDerivativeGroupLinear;
  };
  static_assert( sizeof( PhysicalDeviceComputeShaderDerivativesFeaturesNV ) == sizeof( VkPhysicalDeviceComputeShaderDerivativesFeaturesNV ), "struct and wrapper have different size!" );

  struct PhysicalDeviceFragmentShaderBarycentricFeaturesNV
  {
    PhysicalDeviceFragmentShaderBarycentricFeaturesNV( Bool32 fragmentShaderBarycentric_ = 0 )
      : fragmentShaderBarycentric( fragmentShaderBarycentric_ )
    {
    }

    PhysicalDeviceFragmentShaderBarycentricFeaturesNV( VkPhysicalDeviceFragmentShaderBarycentricFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceFragmentShaderBarycentricFeaturesNV ) );
    }

    PhysicalDeviceFragmentShaderBarycentricFeaturesNV& operator=( VkPhysicalDeviceFragmentShaderBarycentricFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceFragmentShaderBarycentricFeaturesNV ) );
      return *this;
    }
    PhysicalDeviceFragmentShaderBarycentricFeaturesNV& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceFragmentShaderBarycentricFeaturesNV& setFragmentShaderBarycentric( Bool32 fragmentShaderBarycentric_ )
    {
      fragmentShaderBarycentric = fragmentShaderBarycentric_;
      return *this;
    }

    operator VkPhysicalDeviceFragmentShaderBarycentricFeaturesNV const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceFragmentShaderBarycentricFeaturesNV*>(this);
    }

    operator VkPhysicalDeviceFragmentShaderBarycentricFeaturesNV &()
    {
      return *reinterpret_cast<VkPhysicalDeviceFragmentShaderBarycentricFeaturesNV*>(this);
    }

    bool operator==( PhysicalDeviceFragmentShaderBarycentricFeaturesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( fragmentShaderBarycentric == rhs.fragmentShaderBarycentric );
    }

    bool operator!=( PhysicalDeviceFragmentShaderBarycentricFeaturesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceFragmentShaderBarycentricFeaturesNV;

  public:
    void* pNext = nullptr;
    Bool32 fragmentShaderBarycentric;
  };
  static_assert( sizeof( PhysicalDeviceFragmentShaderBarycentricFeaturesNV ) == sizeof( VkPhysicalDeviceFragmentShaderBarycentricFeaturesNV ), "struct and wrapper have different size!" );

  struct PhysicalDeviceShaderImageFootprintFeaturesNV
  {
    PhysicalDeviceShaderImageFootprintFeaturesNV( Bool32 imageFootprint_ = 0 )
      : imageFootprint( imageFootprint_ )
    {
    }

    PhysicalDeviceShaderImageFootprintFeaturesNV( VkPhysicalDeviceShaderImageFootprintFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceShaderImageFootprintFeaturesNV ) );
    }

    PhysicalDeviceShaderImageFootprintFeaturesNV& operator=( VkPhysicalDeviceShaderImageFootprintFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceShaderImageFootprintFeaturesNV ) );
      return *this;
    }
    PhysicalDeviceShaderImageFootprintFeaturesNV& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceShaderImageFootprintFeaturesNV& setImageFootprint( Bool32 imageFootprint_ )
    {
      imageFootprint = imageFootprint_;
      return *this;
    }

    operator VkPhysicalDeviceShaderImageFootprintFeaturesNV const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceShaderImageFootprintFeaturesNV*>(this);
    }

    operator VkPhysicalDeviceShaderImageFootprintFeaturesNV &()
    {
      return *reinterpret_cast<VkPhysicalDeviceShaderImageFootprintFeaturesNV*>(this);
    }

    bool operator==( PhysicalDeviceShaderImageFootprintFeaturesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( imageFootprint == rhs.imageFootprint );
    }

    bool operator!=( PhysicalDeviceShaderImageFootprintFeaturesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceShaderImageFootprintFeaturesNV;

  public:
    void* pNext = nullptr;
    Bool32 imageFootprint;
  };
  static_assert( sizeof( PhysicalDeviceShaderImageFootprintFeaturesNV ) == sizeof( VkPhysicalDeviceShaderImageFootprintFeaturesNV ), "struct and wrapper have different size!" );

  struct PhysicalDeviceShadingRateImageFeaturesNV
  {
    PhysicalDeviceShadingRateImageFeaturesNV( Bool32 shadingRateImage_ = 0,
                                              Bool32 shadingRateCoarseSampleOrder_ = 0 )
      : shadingRateImage( shadingRateImage_ )
      , shadingRateCoarseSampleOrder( shadingRateCoarseSampleOrder_ )
    {
    }

    PhysicalDeviceShadingRateImageFeaturesNV( VkPhysicalDeviceShadingRateImageFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceShadingRateImageFeaturesNV ) );
    }

    PhysicalDeviceShadingRateImageFeaturesNV& operator=( VkPhysicalDeviceShadingRateImageFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceShadingRateImageFeaturesNV ) );
      return *this;
    }
    PhysicalDeviceShadingRateImageFeaturesNV& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceShadingRateImageFeaturesNV& setShadingRateImage( Bool32 shadingRateImage_ )
    {
      shadingRateImage = shadingRateImage_;
      return *this;
    }

    PhysicalDeviceShadingRateImageFeaturesNV& setShadingRateCoarseSampleOrder( Bool32 shadingRateCoarseSampleOrder_ )
    {
      shadingRateCoarseSampleOrder = shadingRateCoarseSampleOrder_;
      return *this;
    }

    operator VkPhysicalDeviceShadingRateImageFeaturesNV const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceShadingRateImageFeaturesNV*>(this);
    }

    operator VkPhysicalDeviceShadingRateImageFeaturesNV &()
    {
      return *reinterpret_cast<VkPhysicalDeviceShadingRateImageFeaturesNV*>(this);
    }

    bool operator==( PhysicalDeviceShadingRateImageFeaturesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( shadingRateImage == rhs.shadingRateImage )
          && ( shadingRateCoarseSampleOrder == rhs.shadingRateCoarseSampleOrder );
    }

    bool operator!=( PhysicalDeviceShadingRateImageFeaturesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceShadingRateImageFeaturesNV;

  public:
    void* pNext = nullptr;
    Bool32 shadingRateImage;
    Bool32 shadingRateCoarseSampleOrder;
  };
  static_assert( sizeof( PhysicalDeviceShadingRateImageFeaturesNV ) == sizeof( VkPhysicalDeviceShadingRateImageFeaturesNV ), "struct and wrapper have different size!" );

  struct PhysicalDeviceShadingRateImagePropertiesNV
  {
    operator VkPhysicalDeviceShadingRateImagePropertiesNV const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceShadingRateImagePropertiesNV*>(this);
    }

    operator VkPhysicalDeviceShadingRateImagePropertiesNV &()
    {
      return *reinterpret_cast<VkPhysicalDeviceShadingRateImagePropertiesNV*>(this);
    }

    bool operator==( PhysicalDeviceShadingRateImagePropertiesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( shadingRateTexelSize == rhs.shadingRateTexelSize )
          && ( shadingRatePaletteSize == rhs.shadingRatePaletteSize )
          && ( shadingRateMaxCoarseSamples == rhs.shadingRateMaxCoarseSamples );
    }

    bool operator!=( PhysicalDeviceShadingRateImagePropertiesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceShadingRateImagePropertiesNV;

  public:
    void* pNext = nullptr;
    Extent2D shadingRateTexelSize;
    uint32_t shadingRatePaletteSize;
    uint32_t shadingRateMaxCoarseSamples;
  };
  static_assert( sizeof( PhysicalDeviceShadingRateImagePropertiesNV ) == sizeof( VkPhysicalDeviceShadingRateImagePropertiesNV ), "struct and wrapper have different size!" );

  struct PhysicalDeviceMeshShaderFeaturesNV
  {
    PhysicalDeviceMeshShaderFeaturesNV( Bool32 taskShader_ = 0,
                                        Bool32 meshShader_ = 0 )
      : taskShader( taskShader_ )
      , meshShader( meshShader_ )
    {
    }

    PhysicalDeviceMeshShaderFeaturesNV( VkPhysicalDeviceMeshShaderFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceMeshShaderFeaturesNV ) );
    }

    PhysicalDeviceMeshShaderFeaturesNV& operator=( VkPhysicalDeviceMeshShaderFeaturesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceMeshShaderFeaturesNV ) );
      return *this;
    }
    PhysicalDeviceMeshShaderFeaturesNV& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceMeshShaderFeaturesNV& setTaskShader( Bool32 taskShader_ )
    {
      taskShader = taskShader_;
      return *this;
    }

    PhysicalDeviceMeshShaderFeaturesNV& setMeshShader( Bool32 meshShader_ )
    {
      meshShader = meshShader_;
      return *this;
    }

    operator VkPhysicalDeviceMeshShaderFeaturesNV const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceMeshShaderFeaturesNV*>(this);
    }

    operator VkPhysicalDeviceMeshShaderFeaturesNV &()
    {
      return *reinterpret_cast<VkPhysicalDeviceMeshShaderFeaturesNV*>(this);
    }

    bool operator==( PhysicalDeviceMeshShaderFeaturesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( taskShader == rhs.taskShader )
          && ( meshShader == rhs.meshShader );
    }

    bool operator!=( PhysicalDeviceMeshShaderFeaturesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceMeshShaderFeaturesNV;

  public:
    void* pNext = nullptr;
    Bool32 taskShader;
    Bool32 meshShader;
  };
  static_assert( sizeof( PhysicalDeviceMeshShaderFeaturesNV ) == sizeof( VkPhysicalDeviceMeshShaderFeaturesNV ), "struct and wrapper have different size!" );

  struct PhysicalDeviceMeshShaderPropertiesNV
  {
    PhysicalDeviceMeshShaderPropertiesNV( uint32_t maxDrawMeshTasksCount_ = 0,
                                          uint32_t maxTaskWorkGroupInvocations_ = 0,
                                          std::array<uint32_t,3> const& maxTaskWorkGroupSize_ = { { 0, 0, 0 } },
                                          uint32_t maxTaskTotalMemorySize_ = 0,
                                          uint32_t maxTaskOutputCount_ = 0,
                                          uint32_t maxMeshWorkGroupInvocations_ = 0,
                                          std::array<uint32_t,3> const& maxMeshWorkGroupSize_ = { { 0, 0, 0 } },
                                          uint32_t maxMeshTotalMemorySize_ = 0,
                                          uint32_t maxMeshOutputVertices_ = 0,
                                          uint32_t maxMeshOutputPrimitives_ = 0,
                                          uint32_t maxMeshMultiviewViewCount_ = 0,
                                          uint32_t meshOutputPerVertexGranularity_ = 0,
                                          uint32_t meshOutputPerPrimitiveGranularity_ = 0 )
      : maxDrawMeshTasksCount( maxDrawMeshTasksCount_ )
      , maxTaskWorkGroupInvocations( maxTaskWorkGroupInvocations_ )
      , maxTaskTotalMemorySize( maxTaskTotalMemorySize_ )
      , maxTaskOutputCount( maxTaskOutputCount_ )
      , maxMeshWorkGroupInvocations( maxMeshWorkGroupInvocations_ )
      , maxMeshTotalMemorySize( maxMeshTotalMemorySize_ )
      , maxMeshOutputVertices( maxMeshOutputVertices_ )
      , maxMeshOutputPrimitives( maxMeshOutputPrimitives_ )
      , maxMeshMultiviewViewCount( maxMeshMultiviewViewCount_ )
      , meshOutputPerVertexGranularity( meshOutputPerVertexGranularity_ )
      , meshOutputPerPrimitiveGranularity( meshOutputPerPrimitiveGranularity_ )
    {
      memcpy( &maxTaskWorkGroupSize, maxTaskWorkGroupSize_.data(), 3 * sizeof( uint32_t ) );
      memcpy( &maxMeshWorkGroupSize, maxMeshWorkGroupSize_.data(), 3 * sizeof( uint32_t ) );
    }

    PhysicalDeviceMeshShaderPropertiesNV( VkPhysicalDeviceMeshShaderPropertiesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceMeshShaderPropertiesNV ) );
    }

    PhysicalDeviceMeshShaderPropertiesNV& operator=( VkPhysicalDeviceMeshShaderPropertiesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceMeshShaderPropertiesNV ) );
      return *this;
    }
    PhysicalDeviceMeshShaderPropertiesNV& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceMeshShaderPropertiesNV& setMaxDrawMeshTasksCount( uint32_t maxDrawMeshTasksCount_ )
    {
      maxDrawMeshTasksCount = maxDrawMeshTasksCount_;
      return *this;
    }

    PhysicalDeviceMeshShaderPropertiesNV& setMaxTaskWorkGroupInvocations( uint32_t maxTaskWorkGroupInvocations_ )
    {
      maxTaskWorkGroupInvocations = maxTaskWorkGroupInvocations_;
      return *this;
    }

    PhysicalDeviceMeshShaderPropertiesNV& setMaxTaskWorkGroupSize( std::array<uint32_t,3> maxTaskWorkGroupSize_ )
    {
      memcpy( &maxTaskWorkGroupSize, maxTaskWorkGroupSize_.data(), 3 * sizeof( uint32_t ) );
      return *this;
    }

    PhysicalDeviceMeshShaderPropertiesNV& setMaxTaskTotalMemorySize( uint32_t maxTaskTotalMemorySize_ )
    {
      maxTaskTotalMemorySize = maxTaskTotalMemorySize_;
      return *this;
    }

    PhysicalDeviceMeshShaderPropertiesNV& setMaxTaskOutputCount( uint32_t maxTaskOutputCount_ )
    {
      maxTaskOutputCount = maxTaskOutputCount_;
      return *this;
    }

    PhysicalDeviceMeshShaderPropertiesNV& setMaxMeshWorkGroupInvocations( uint32_t maxMeshWorkGroupInvocations_ )
    {
      maxMeshWorkGroupInvocations = maxMeshWorkGroupInvocations_;
      return *this;
    }

    PhysicalDeviceMeshShaderPropertiesNV& setMaxMeshWorkGroupSize( std::array<uint32_t,3> maxMeshWorkGroupSize_ )
    {
      memcpy( &maxMeshWorkGroupSize, maxMeshWorkGroupSize_.data(), 3 * sizeof( uint32_t ) );
      return *this;
    }

    PhysicalDeviceMeshShaderPropertiesNV& setMaxMeshTotalMemorySize( uint32_t maxMeshTotalMemorySize_ )
    {
      maxMeshTotalMemorySize = maxMeshTotalMemorySize_;
      return *this;
    }

    PhysicalDeviceMeshShaderPropertiesNV& setMaxMeshOutputVertices( uint32_t maxMeshOutputVertices_ )
    {
      maxMeshOutputVertices = maxMeshOutputVertices_;
      return *this;
    }

    PhysicalDeviceMeshShaderPropertiesNV& setMaxMeshOutputPrimitives( uint32_t maxMeshOutputPrimitives_ )
    {
      maxMeshOutputPrimitives = maxMeshOutputPrimitives_;
      return *this;
    }

    PhysicalDeviceMeshShaderPropertiesNV& setMaxMeshMultiviewViewCount( uint32_t maxMeshMultiviewViewCount_ )
    {
      maxMeshMultiviewViewCount = maxMeshMultiviewViewCount_;
      return *this;
    }

    PhysicalDeviceMeshShaderPropertiesNV& setMeshOutputPerVertexGranularity( uint32_t meshOutputPerVertexGranularity_ )
    {
      meshOutputPerVertexGranularity = meshOutputPerVertexGranularity_;
      return *this;
    }

    PhysicalDeviceMeshShaderPropertiesNV& setMeshOutputPerPrimitiveGranularity( uint32_t meshOutputPerPrimitiveGranularity_ )
    {
      meshOutputPerPrimitiveGranularity = meshOutputPerPrimitiveGranularity_;
      return *this;
    }

    operator VkPhysicalDeviceMeshShaderPropertiesNV const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceMeshShaderPropertiesNV*>(this);
    }

    operator VkPhysicalDeviceMeshShaderPropertiesNV &()
    {
      return *reinterpret_cast<VkPhysicalDeviceMeshShaderPropertiesNV*>(this);
    }

    bool operator==( PhysicalDeviceMeshShaderPropertiesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxDrawMeshTasksCount == rhs.maxDrawMeshTasksCount )
          && ( maxTaskWorkGroupInvocations == rhs.maxTaskWorkGroupInvocations )
          && ( memcmp( maxTaskWorkGroupSize, rhs.maxTaskWorkGroupSize, 3 * sizeof( uint32_t ) ) == 0 )
          && ( maxTaskTotalMemorySize == rhs.maxTaskTotalMemorySize )
          && ( maxTaskOutputCount == rhs.maxTaskOutputCount )
          && ( maxMeshWorkGroupInvocations == rhs.maxMeshWorkGroupInvocations )
          && ( memcmp( maxMeshWorkGroupSize, rhs.maxMeshWorkGroupSize, 3 * sizeof( uint32_t ) ) == 0 )
          && ( maxMeshTotalMemorySize == rhs.maxMeshTotalMemorySize )
          && ( maxMeshOutputVertices == rhs.maxMeshOutputVertices )
          && ( maxMeshOutputPrimitives == rhs.maxMeshOutputPrimitives )
          && ( maxMeshMultiviewViewCount == rhs.maxMeshMultiviewViewCount )
          && ( meshOutputPerVertexGranularity == rhs.meshOutputPerVertexGranularity )
          && ( meshOutputPerPrimitiveGranularity == rhs.meshOutputPerPrimitiveGranularity );
    }

    bool operator!=( PhysicalDeviceMeshShaderPropertiesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceMeshShaderPropertiesNV;

  public:
    void* pNext = nullptr;
    uint32_t maxDrawMeshTasksCount;
    uint32_t maxTaskWorkGroupInvocations;
    uint32_t maxTaskWorkGroupSize[3];
    uint32_t maxTaskTotalMemorySize;
    uint32_t maxTaskOutputCount;
    uint32_t maxMeshWorkGroupInvocations;
    uint32_t maxMeshWorkGroupSize[3];
    uint32_t maxMeshTotalMemorySize;
    uint32_t maxMeshOutputVertices;
    uint32_t maxMeshOutputPrimitives;
    uint32_t maxMeshMultiviewViewCount;
    uint32_t meshOutputPerVertexGranularity;
    uint32_t meshOutputPerPrimitiveGranularity;
  };
  static_assert( sizeof( PhysicalDeviceMeshShaderPropertiesNV ) == sizeof( VkPhysicalDeviceMeshShaderPropertiesNV ), "struct and wrapper have different size!" );

  struct GeometryTrianglesNV
  {
    GeometryTrianglesNV( Buffer vertexData_ = Buffer(),
                         DeviceSize vertexOffset_ = 0,
                         uint32_t vertexCount_ = 0,
                         DeviceSize vertexStride_ = 0,
                         Format vertexFormat_ = Format::eUndefined,
                         Buffer indexData_ = Buffer(),
                         DeviceSize indexOffset_ = 0,
                         uint32_t indexCount_ = 0,
                         IndexType indexType_ = IndexType::eUint16,
                         Buffer transformData_ = Buffer(),
                         DeviceSize transformOffset_ = 0 )
      : vertexData( vertexData_ )
      , vertexOffset( vertexOffset_ )
      , vertexCount( vertexCount_ )
      , vertexStride( vertexStride_ )
      , vertexFormat( vertexFormat_ )
      , indexData( indexData_ )
      , indexOffset( indexOffset_ )
      , indexCount( indexCount_ )
      , indexType( indexType_ )
      , transformData( transformData_ )
      , transformOffset( transformOffset_ )
    {
    }

    GeometryTrianglesNV( VkGeometryTrianglesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryTrianglesNV ) );
    }

    GeometryTrianglesNV& operator=( VkGeometryTrianglesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryTrianglesNV ) );
      return *this;
    }
    GeometryTrianglesNV& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    GeometryTrianglesNV& setVertexData( Buffer vertexData_ )
    {
      vertexData = vertexData_;
      return *this;
    }

    GeometryTrianglesNV& setVertexOffset( DeviceSize vertexOffset_ )
    {
      vertexOffset = vertexOffset_;
      return *this;
    }

    GeometryTrianglesNV& setVertexCount( uint32_t vertexCount_ )
    {
      vertexCount = vertexCount_;
      return *this;
    }

    GeometryTrianglesNV& setVertexStride( DeviceSize vertexStride_ )
    {
      vertexStride = vertexStride_;
      return *this;
    }

    GeometryTrianglesNV& setVertexFormat( Format vertexFormat_ )
    {
      vertexFormat = vertexFormat_;
      return *this;
    }

    GeometryTrianglesNV& setIndexData( Buffer indexData_ )
    {
      indexData = indexData_;
      return *this;
    }

    GeometryTrianglesNV& setIndexOffset( DeviceSize indexOffset_ )
    {
      indexOffset = indexOffset_;
      return *this;
    }

    GeometryTrianglesNV& setIndexCount( uint32_t indexCount_ )
    {
      indexCount = indexCount_;
      return *this;
    }

    GeometryTrianglesNV& setIndexType( IndexType indexType_ )
    {
      indexType = indexType_;
      return *this;
    }

    GeometryTrianglesNV& setTransformData( Buffer transformData_ )
    {
      transformData = transformData_;
      return *this;
    }

    GeometryTrianglesNV& setTransformOffset( DeviceSize transformOffset_ )
    {
      transformOffset = transformOffset_;
      return *this;
    }

    operator VkGeometryTrianglesNV const&() const
    {
      return *reinterpret_cast<const VkGeometryTrianglesNV*>(this);
    }

    operator VkGeometryTrianglesNV &()
    {
      return *reinterpret_cast<VkGeometryTrianglesNV*>(this);
    }

    bool operator==( GeometryTrianglesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( vertexData == rhs.vertexData )
          && ( vertexOffset == rhs.vertexOffset )
          && ( vertexCount == rhs.vertexCount )
          && ( vertexStride == rhs.vertexStride )
          && ( vertexFormat == rhs.vertexFormat )
          && ( indexData == rhs.indexData )
          && ( indexOffset == rhs.indexOffset )
          && ( indexCount == rhs.indexCount )
          && ( indexType == rhs.indexType )
          && ( transformData == rhs.transformData )
          && ( transformOffset == rhs.transformOffset );
    }

    bool operator!=( GeometryTrianglesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eGeometryTrianglesNV;

  public:
    const void* pNext = nullptr;
    Buffer vertexData;
    DeviceSize vertexOffset;
    uint32_t vertexCount;
    DeviceSize vertexStride;
    Format vertexFormat;
    Buffer indexData;
    DeviceSize indexOffset;
    uint32_t indexCount;
    IndexType indexType;
    Buffer transformData;
    DeviceSize transformOffset;
  };
  static_assert( sizeof( GeometryTrianglesNV ) == sizeof( VkGeometryTrianglesNV ), "struct and wrapper have different size!" );

  struct GeometryAABBNV
  {
    GeometryAABBNV( Buffer aabbData_ = Buffer(),
                    uint32_t numAABBs_ = 0,
                    uint32_t stride_ = 0,
                    DeviceSize offset_ = 0 )
      : aabbData( aabbData_ )
      , numAABBs( numAABBs_ )
      , stride( stride_ )
      , offset( offset_ )
    {
    }

    GeometryAABBNV( VkGeometryAABBNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryAABBNV ) );
    }

    GeometryAABBNV& operator=( VkGeometryAABBNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryAABBNV ) );
      return *this;
    }
    GeometryAABBNV& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    GeometryAABBNV& setAabbData( Buffer aabbData_ )
    {
      aabbData = aabbData_;
      return *this;
    }

    GeometryAABBNV& setNumAABBs( uint32_t numAABBs_ )
    {
      numAABBs = numAABBs_;
      return *this;
    }

    GeometryAABBNV& setStride( uint32_t stride_ )
    {
      stride = stride_;
      return *this;
    }

    GeometryAABBNV& setOffset( DeviceSize offset_ )
    {
      offset = offset_;
      return *this;
    }

    operator VkGeometryAABBNV const&() const
    {
      return *reinterpret_cast<const VkGeometryAABBNV*>(this);
    }

    operator VkGeometryAABBNV &()
    {
      return *reinterpret_cast<VkGeometryAABBNV*>(this);
    }

    bool operator==( GeometryAABBNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( aabbData == rhs.aabbData )
          && ( numAABBs == rhs.numAABBs )
          && ( stride == rhs.stride )
          && ( offset == rhs.offset );
    }

    bool operator!=( GeometryAABBNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eGeometryAabbNV;

  public:
    const void* pNext = nullptr;
    Buffer aabbData;
    uint32_t numAABBs;
    uint32_t stride;
    DeviceSize offset;
  };
  static_assert( sizeof( GeometryAABBNV ) == sizeof( VkGeometryAABBNV ), "struct and wrapper have different size!" );

  struct GeometryDataNV
  {
    GeometryDataNV( GeometryTrianglesNV triangles_ = GeometryTrianglesNV(),
                    GeometryAABBNV aabbs_ = GeometryAABBNV() )
      : triangles( triangles_ )
      , aabbs( aabbs_ )
    {
    }

    GeometryDataNV( VkGeometryDataNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryDataNV ) );
    }

    GeometryDataNV& operator=( VkGeometryDataNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryDataNV ) );
      return *this;
    }
    GeometryDataNV& setTriangles( GeometryTrianglesNV triangles_ )
    {
      triangles = triangles_;
      return *this;
    }

    GeometryDataNV& setAabbs( GeometryAABBNV aabbs_ )
    {
      aabbs = aabbs_;
      return *this;
    }

    operator VkGeometryDataNV const&() const
    {
      return *reinterpret_cast<const VkGeometryDataNV*>(this);
    }

    operator VkGeometryDataNV &()
    {
      return *reinterpret_cast<VkGeometryDataNV*>(this);
    }

    bool operator==( GeometryDataNV const& rhs ) const
    {
      return ( triangles == rhs.triangles )
          && ( aabbs == rhs.aabbs );
    }

    bool operator!=( GeometryDataNV const& rhs ) const
    {
      return !operator==( rhs );
    }

    GeometryTrianglesNV triangles;
    GeometryAABBNV aabbs;
  };
  static_assert( sizeof( GeometryDataNV ) == sizeof( VkGeometryDataNV ), "struct and wrapper have different size!" );

  struct BindAccelerationStructureMemoryInfoNV
  {
    BindAccelerationStructureMemoryInfoNV( AccelerationStructureNV accelerationStructure_ = AccelerationStructureNV(),
                                           DeviceMemory memory_ = DeviceMemory(),
                                           DeviceSize memoryOffset_ = 0,
                                           uint32_t deviceIndexCount_ = 0,
                                           const uint32_t* pDeviceIndices_ = nullptr )
      : accelerationStructure( accelerationStructure_ )
      , memory( memory_ )
      , memoryOffset( memoryOffset_ )
      , deviceIndexCount( deviceIndexCount_ )
      , pDeviceIndices( pDeviceIndices_ )
    {
    }

    BindAccelerationStructureMemoryInfoNV( VkBindAccelerationStructureMemoryInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindAccelerationStructureMemoryInfoNV ) );
    }

    BindAccelerationStructureMemoryInfoNV& operator=( VkBindAccelerationStructureMemoryInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindAccelerationStructureMemoryInfoNV ) );
      return *this;
    }
    BindAccelerationStructureMemoryInfoNV& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindAccelerationStructureMemoryInfoNV& setAccelerationStructure( AccelerationStructureNV accelerationStructure_ )
    {
      accelerationStructure = accelerationStructure_;
      return *this;
    }

    BindAccelerationStructureMemoryInfoNV& setMemory( DeviceMemory memory_ )
    {
      memory = memory_;
      return *this;
    }

    BindAccelerationStructureMemoryInfoNV& setMemoryOffset( DeviceSize memoryOffset_ )
    {
      memoryOffset = memoryOffset_;
      return *this;
    }

    BindAccelerationStructureMemoryInfoNV& setDeviceIndexCount( uint32_t deviceIndexCount_ )
    {
      deviceIndexCount = deviceIndexCount_;
      return *this;
    }

    BindAccelerationStructureMemoryInfoNV& setPDeviceIndices( const uint32_t* pDeviceIndices_ )
    {
      pDeviceIndices = pDeviceIndices_;
      return *this;
    }

    operator VkBindAccelerationStructureMemoryInfoNV const&() const
    {
      return *reinterpret_cast<const VkBindAccelerationStructureMemoryInfoNV*>(this);
    }

    operator VkBindAccelerationStructureMemoryInfoNV &()
    {
      return *reinterpret_cast<VkBindAccelerationStructureMemoryInfoNV*>(this);
    }

    bool operator==( BindAccelerationStructureMemoryInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( accelerationStructure == rhs.accelerationStructure )
          && ( memory == rhs.memory )
          && ( memoryOffset == rhs.memoryOffset )
          && ( deviceIndexCount == rhs.deviceIndexCount )
          && ( pDeviceIndices == rhs.pDeviceIndices );
    }

    bool operator!=( BindAccelerationStructureMemoryInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindAccelerationStructureMemoryInfoNV;

  public:
    const void* pNext = nullptr;
    AccelerationStructureNV accelerationStructure;
    DeviceMemory memory;
    DeviceSize memoryOffset;
    uint32_t deviceIndexCount;
    const uint32_t* pDeviceIndices;
  };
  static_assert( sizeof( BindAccelerationStructureMemoryInfoNV ) == sizeof( VkBindAccelerationStructureMemoryInfoNV ), "struct and wrapper have different size!" );

  struct WriteDescriptorSetAccelerationStructureNV
  {
    WriteDescriptorSetAccelerationStructureNV( uint32_t accelerationStructureCount_ = 0,
                                               const AccelerationStructureNV* pAccelerationStructures_ = nullptr )
      : accelerationStructureCount( accelerationStructureCount_ )
      , pAccelerationStructures( pAccelerationStructures_ )
    {
    }

    WriteDescriptorSetAccelerationStructureNV( VkWriteDescriptorSetAccelerationStructureNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( WriteDescriptorSetAccelerationStructureNV ) );
    }

    WriteDescriptorSetAccelerationStructureNV& operator=( VkWriteDescriptorSetAccelerationStructureNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( WriteDescriptorSetAccelerationStructureNV ) );
      return *this;
    }
    WriteDescriptorSetAccelerationStructureNV& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    WriteDescriptorSetAccelerationStructureNV& setAccelerationStructureCount( uint32_t accelerationStructureCount_ )
    {
      accelerationStructureCount = accelerationStructureCount_;
      return *this;
    }

    WriteDescriptorSetAccelerationStructureNV& setPAccelerationStructures( const AccelerationStructureNV* pAccelerationStructures_ )
    {
      pAccelerationStructures = pAccelerationStructures_;
      return *this;
    }

    operator VkWriteDescriptorSetAccelerationStructureNV const&() const
    {
      return *reinterpret_cast<const VkWriteDescriptorSetAccelerationStructureNV*>(this);
    }

    operator VkWriteDescriptorSetAccelerationStructureNV &()
    {
      return *reinterpret_cast<VkWriteDescriptorSetAccelerationStructureNV*>(this);
    }

    bool operator==( WriteDescriptorSetAccelerationStructureNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( accelerationStructureCount == rhs.accelerationStructureCount )
          && ( pAccelerationStructures == rhs.pAccelerationStructures );
    }

    bool operator!=( WriteDescriptorSetAccelerationStructureNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eWriteDescriptorSetAccelerationStructureNV;

  public:
    const void* pNext = nullptr;
    uint32_t accelerationStructureCount;
    const AccelerationStructureNV* pAccelerationStructures;
  };
  static_assert( sizeof( WriteDescriptorSetAccelerationStructureNV ) == sizeof( VkWriteDescriptorSetAccelerationStructureNV ), "struct and wrapper have different size!" );

  struct PhysicalDeviceRayTracingPropertiesNV
  {
    PhysicalDeviceRayTracingPropertiesNV( uint32_t shaderGroupHandleSize_ = 0,
                                          uint32_t maxRecursionDepth_ = 0,
                                          uint32_t maxShaderGroupStride_ = 0,
                                          uint32_t shaderGroupBaseAlignment_ = 0,
                                          uint64_t maxGeometryCount_ = 0,
                                          uint64_t maxInstanceCount_ = 0,
                                          uint64_t maxTriangleCount_ = 0,
                                          uint32_t maxDescriptorSetAccelerationStructures_ = 0 )
      : shaderGroupHandleSize( shaderGroupHandleSize_ )
      , maxRecursionDepth( maxRecursionDepth_ )
      , maxShaderGroupStride( maxShaderGroupStride_ )
      , shaderGroupBaseAlignment( shaderGroupBaseAlignment_ )
      , maxGeometryCount( maxGeometryCount_ )
      , maxInstanceCount( maxInstanceCount_ )
      , maxTriangleCount( maxTriangleCount_ )
      , maxDescriptorSetAccelerationStructures( maxDescriptorSetAccelerationStructures_ )
    {
    }

    PhysicalDeviceRayTracingPropertiesNV( VkPhysicalDeviceRayTracingPropertiesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceRayTracingPropertiesNV ) );
    }

    PhysicalDeviceRayTracingPropertiesNV& operator=( VkPhysicalDeviceRayTracingPropertiesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceRayTracingPropertiesNV ) );
      return *this;
    }
    PhysicalDeviceRayTracingPropertiesNV& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceRayTracingPropertiesNV& setShaderGroupHandleSize( uint32_t shaderGroupHandleSize_ )
    {
      shaderGroupHandleSize = shaderGroupHandleSize_;
      return *this;
    }

    PhysicalDeviceRayTracingPropertiesNV& setMaxRecursionDepth( uint32_t maxRecursionDepth_ )
    {
      maxRecursionDepth = maxRecursionDepth_;
      return *this;
    }

    PhysicalDeviceRayTracingPropertiesNV& setMaxShaderGroupStride( uint32_t maxShaderGroupStride_ )
    {
      maxShaderGroupStride = maxShaderGroupStride_;
      return *this;
    }

    PhysicalDeviceRayTracingPropertiesNV& setShaderGroupBaseAlignment( uint32_t shaderGroupBaseAlignment_ )
    {
      shaderGroupBaseAlignment = shaderGroupBaseAlignment_;
      return *this;
    }

    PhysicalDeviceRayTracingPropertiesNV& setMaxGeometryCount( uint64_t maxGeometryCount_ )
    {
      maxGeometryCount = maxGeometryCount_;
      return *this;
    }

    PhysicalDeviceRayTracingPropertiesNV& setMaxInstanceCount( uint64_t maxInstanceCount_ )
    {
      maxInstanceCount = maxInstanceCount_;
      return *this;
    }

    PhysicalDeviceRayTracingPropertiesNV& setMaxTriangleCount( uint64_t maxTriangleCount_ )
    {
      maxTriangleCount = maxTriangleCount_;
      return *this;
    }

    PhysicalDeviceRayTracingPropertiesNV& setMaxDescriptorSetAccelerationStructures( uint32_t maxDescriptorSetAccelerationStructures_ )
    {
      maxDescriptorSetAccelerationStructures = maxDescriptorSetAccelerationStructures_;
      return *this;
    }

    operator VkPhysicalDeviceRayTracingPropertiesNV const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceRayTracingPropertiesNV*>(this);
    }

    operator VkPhysicalDeviceRayTracingPropertiesNV &()
    {
      return *reinterpret_cast<VkPhysicalDeviceRayTracingPropertiesNV*>(this);
    }

    bool operator==( PhysicalDeviceRayTracingPropertiesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( shaderGroupHandleSize == rhs.shaderGroupHandleSize )
          && ( maxRecursionDepth == rhs.maxRecursionDepth )
          && ( maxShaderGroupStride == rhs.maxShaderGroupStride )
          && ( shaderGroupBaseAlignment == rhs.shaderGroupBaseAlignment )
          && ( maxGeometryCount == rhs.maxGeometryCount )
          && ( maxInstanceCount == rhs.maxInstanceCount )
          && ( maxTriangleCount == rhs.maxTriangleCount )
          && ( maxDescriptorSetAccelerationStructures == rhs.maxDescriptorSetAccelerationStructures );
    }

    bool operator!=( PhysicalDeviceRayTracingPropertiesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceRayTracingPropertiesNV;

  public:
    void* pNext = nullptr;
    uint32_t shaderGroupHandleSize;
    uint32_t maxRecursionDepth;
    uint32_t maxShaderGroupStride;
    uint32_t shaderGroupBaseAlignment;
    uint64_t maxGeometryCount;
    uint64_t maxInstanceCount;
    uint64_t maxTriangleCount;
    uint32_t maxDescriptorSetAccelerationStructures;
  };
  static_assert( sizeof( PhysicalDeviceRayTracingPropertiesNV ) == sizeof( VkPhysicalDeviceRayTracingPropertiesNV ), "struct and wrapper have different size!" );

  struct PhysicalDeviceImageDrmFormatModifierInfoEXT
  {
    PhysicalDeviceImageDrmFormatModifierInfoEXT( uint64_t drmFormatModifier_ = 0,
                                                 SharingMode sharingMode_ = SharingMode::eExclusive,
                                                 uint32_t queueFamilyIndexCount_ = 0,
                                                 const uint32_t* pQueueFamilyIndices_ = nullptr )
      : drmFormatModifier( drmFormatModifier_ )
      , sharingMode( sharingMode_ )
      , queueFamilyIndexCount( queueFamilyIndexCount_ )
      , pQueueFamilyIndices( pQueueFamilyIndices_ )
    {
    }

    PhysicalDeviceImageDrmFormatModifierInfoEXT( VkPhysicalDeviceImageDrmFormatModifierInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceImageDrmFormatModifierInfoEXT ) );
    }

    PhysicalDeviceImageDrmFormatModifierInfoEXT& operator=( VkPhysicalDeviceImageDrmFormatModifierInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceImageDrmFormatModifierInfoEXT ) );
      return *this;
    }
    PhysicalDeviceImageDrmFormatModifierInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceImageDrmFormatModifierInfoEXT& setDrmFormatModifier( uint64_t drmFormatModifier_ )
    {
      drmFormatModifier = drmFormatModifier_;
      return *this;
    }

    PhysicalDeviceImageDrmFormatModifierInfoEXT& setSharingMode( SharingMode sharingMode_ )
    {
      sharingMode = sharingMode_;
      return *this;
    }

    PhysicalDeviceImageDrmFormatModifierInfoEXT& setQueueFamilyIndexCount( uint32_t queueFamilyIndexCount_ )
    {
      queueFamilyIndexCount = queueFamilyIndexCount_;
      return *this;
    }

    PhysicalDeviceImageDrmFormatModifierInfoEXT& setPQueueFamilyIndices( const uint32_t* pQueueFamilyIndices_ )
    {
      pQueueFamilyIndices = pQueueFamilyIndices_;
      return *this;
    }

    operator VkPhysicalDeviceImageDrmFormatModifierInfoEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceImageDrmFormatModifierInfoEXT*>(this);
    }

    operator VkPhysicalDeviceImageDrmFormatModifierInfoEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceImageDrmFormatModifierInfoEXT*>(this);
    }

    bool operator==( PhysicalDeviceImageDrmFormatModifierInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( drmFormatModifier == rhs.drmFormatModifier )
          && ( sharingMode == rhs.sharingMode )
          && ( queueFamilyIndexCount == rhs.queueFamilyIndexCount )
          && ( pQueueFamilyIndices == rhs.pQueueFamilyIndices );
    }

    bool operator!=( PhysicalDeviceImageDrmFormatModifierInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceImageDrmFormatModifierInfoEXT;

  public:
    const void* pNext = nullptr;
    uint64_t drmFormatModifier;
    SharingMode sharingMode;
    uint32_t queueFamilyIndexCount;
    const uint32_t* pQueueFamilyIndices;
  };
  static_assert( sizeof( PhysicalDeviceImageDrmFormatModifierInfoEXT ) == sizeof( VkPhysicalDeviceImageDrmFormatModifierInfoEXT ), "struct and wrapper have different size!" );

  struct ImageDrmFormatModifierListCreateInfoEXT
  {
    ImageDrmFormatModifierListCreateInfoEXT( uint32_t drmFormatModifierCount_ = 0,
                                             const uint64_t* pDrmFormatModifiers_ = nullptr )
      : drmFormatModifierCount( drmFormatModifierCount_ )
      , pDrmFormatModifiers( pDrmFormatModifiers_ )
    {
    }

    ImageDrmFormatModifierListCreateInfoEXT( VkImageDrmFormatModifierListCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageDrmFormatModifierListCreateInfoEXT ) );
    }

    ImageDrmFormatModifierListCreateInfoEXT& operator=( VkImageDrmFormatModifierListCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageDrmFormatModifierListCreateInfoEXT ) );
      return *this;
    }
    ImageDrmFormatModifierListCreateInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImageDrmFormatModifierListCreateInfoEXT& setDrmFormatModifierCount( uint32_t drmFormatModifierCount_ )
    {
      drmFormatModifierCount = drmFormatModifierCount_;
      return *this;
    }

    ImageDrmFormatModifierListCreateInfoEXT& setPDrmFormatModifiers( const uint64_t* pDrmFormatModifiers_ )
    {
      pDrmFormatModifiers = pDrmFormatModifiers_;
      return *this;
    }

    operator VkImageDrmFormatModifierListCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkImageDrmFormatModifierListCreateInfoEXT*>(this);
    }

    operator VkImageDrmFormatModifierListCreateInfoEXT &()
    {
      return *reinterpret_cast<VkImageDrmFormatModifierListCreateInfoEXT*>(this);
    }

    bool operator==( ImageDrmFormatModifierListCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( drmFormatModifierCount == rhs.drmFormatModifierCount )
          && ( pDrmFormatModifiers == rhs.pDrmFormatModifiers );
    }

    bool operator!=( ImageDrmFormatModifierListCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageDrmFormatModifierListCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    uint32_t drmFormatModifierCount;
    const uint64_t* pDrmFormatModifiers;
  };
  static_assert( sizeof( ImageDrmFormatModifierListCreateInfoEXT ) == sizeof( VkImageDrmFormatModifierListCreateInfoEXT ), "struct and wrapper have different size!" );

  struct ImageDrmFormatModifierExplicitCreateInfoEXT
  {
    ImageDrmFormatModifierExplicitCreateInfoEXT( uint64_t drmFormatModifier_ = 0,
                                                 uint32_t drmFormatModifierPlaneCount_ = 0,
                                                 const SubresourceLayout* pPlaneLayouts_ = nullptr )
      : drmFormatModifier( drmFormatModifier_ )
      , drmFormatModifierPlaneCount( drmFormatModifierPlaneCount_ )
      , pPlaneLayouts( pPlaneLayouts_ )
    {
    }

    ImageDrmFormatModifierExplicitCreateInfoEXT( VkImageDrmFormatModifierExplicitCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageDrmFormatModifierExplicitCreateInfoEXT ) );
    }

    ImageDrmFormatModifierExplicitCreateInfoEXT& operator=( VkImageDrmFormatModifierExplicitCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageDrmFormatModifierExplicitCreateInfoEXT ) );
      return *this;
    }
    ImageDrmFormatModifierExplicitCreateInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImageDrmFormatModifierExplicitCreateInfoEXT& setDrmFormatModifier( uint64_t drmFormatModifier_ )
    {
      drmFormatModifier = drmFormatModifier_;
      return *this;
    }

    ImageDrmFormatModifierExplicitCreateInfoEXT& setDrmFormatModifierPlaneCount( uint32_t drmFormatModifierPlaneCount_ )
    {
      drmFormatModifierPlaneCount = drmFormatModifierPlaneCount_;
      return *this;
    }

    ImageDrmFormatModifierExplicitCreateInfoEXT& setPPlaneLayouts( const SubresourceLayout* pPlaneLayouts_ )
    {
      pPlaneLayouts = pPlaneLayouts_;
      return *this;
    }

    operator VkImageDrmFormatModifierExplicitCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkImageDrmFormatModifierExplicitCreateInfoEXT*>(this);
    }

    operator VkImageDrmFormatModifierExplicitCreateInfoEXT &()
    {
      return *reinterpret_cast<VkImageDrmFormatModifierExplicitCreateInfoEXT*>(this);
    }

    bool operator==( ImageDrmFormatModifierExplicitCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( drmFormatModifier == rhs.drmFormatModifier )
          && ( drmFormatModifierPlaneCount == rhs.drmFormatModifierPlaneCount )
          && ( pPlaneLayouts == rhs.pPlaneLayouts );
    }

    bool operator!=( ImageDrmFormatModifierExplicitCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageDrmFormatModifierExplicitCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    uint64_t drmFormatModifier;
    uint32_t drmFormatModifierPlaneCount;
    const SubresourceLayout* pPlaneLayouts;
  };
  static_assert( sizeof( ImageDrmFormatModifierExplicitCreateInfoEXT ) == sizeof( VkImageDrmFormatModifierExplicitCreateInfoEXT ), "struct and wrapper have different size!" );

  struct ImageDrmFormatModifierPropertiesEXT
  {
    operator VkImageDrmFormatModifierPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkImageDrmFormatModifierPropertiesEXT*>(this);
    }

    operator VkImageDrmFormatModifierPropertiesEXT &()
    {
      return *reinterpret_cast<VkImageDrmFormatModifierPropertiesEXT*>(this);
    }

    bool operator==( ImageDrmFormatModifierPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( drmFormatModifier == rhs.drmFormatModifier );
    }

    bool operator!=( ImageDrmFormatModifierPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageDrmFormatModifierPropertiesEXT;

  public:
    void* pNext = nullptr;
    uint64_t drmFormatModifier;
  };
  static_assert( sizeof( ImageDrmFormatModifierPropertiesEXT ) == sizeof( VkImageDrmFormatModifierPropertiesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceFragmentDensityMapFeaturesEXT
  {
    operator VkPhysicalDeviceFragmentDensityMapFeaturesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceFragmentDensityMapFeaturesEXT*>(this);
    }

    operator VkPhysicalDeviceFragmentDensityMapFeaturesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceFragmentDensityMapFeaturesEXT*>(this);
    }

    bool operator==( PhysicalDeviceFragmentDensityMapFeaturesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( fragmentDensityMap == rhs.fragmentDensityMap )
          && ( fragmentDensityMapDynamic == rhs.fragmentDensityMapDynamic )
          && ( fragmentDensityMapNonSubsampledImages == rhs.fragmentDensityMapNonSubsampledImages );
    }

    bool operator!=( PhysicalDeviceFragmentDensityMapFeaturesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceFragmentDensityMapFeaturesEXT;

  public:
    void* pNext = nullptr;
    Bool32 fragmentDensityMap;
    Bool32 fragmentDensityMapDynamic;
    Bool32 fragmentDensityMapNonSubsampledImages;
  };
  static_assert( sizeof( PhysicalDeviceFragmentDensityMapFeaturesEXT ) == sizeof( VkPhysicalDeviceFragmentDensityMapFeaturesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceFragmentDensityMapPropertiesEXT
  {
    operator VkPhysicalDeviceFragmentDensityMapPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceFragmentDensityMapPropertiesEXT*>(this);
    }

    operator VkPhysicalDeviceFragmentDensityMapPropertiesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceFragmentDensityMapPropertiesEXT*>(this);
    }

    bool operator==( PhysicalDeviceFragmentDensityMapPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( minFragmentDensityTexelSize == rhs.minFragmentDensityTexelSize )
          && ( maxFragmentDensityTexelSize == rhs.maxFragmentDensityTexelSize )
          && ( fragmentDensityInvocations == rhs.fragmentDensityInvocations );
    }

    bool operator!=( PhysicalDeviceFragmentDensityMapPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceFragmentDensityMapPropertiesEXT;

  public:
    void* pNext = nullptr;
    Extent2D minFragmentDensityTexelSize;
    Extent2D maxFragmentDensityTexelSize;
    Bool32 fragmentDensityInvocations;
  };
  static_assert( sizeof( PhysicalDeviceFragmentDensityMapPropertiesEXT ) == sizeof( VkPhysicalDeviceFragmentDensityMapPropertiesEXT ), "struct and wrapper have different size!" );

  struct RenderPassFragmentDensityMapCreateInfoEXT
  {
    RenderPassFragmentDensityMapCreateInfoEXT( AttachmentReference fragmentDensityMapAttachment_ = AttachmentReference() )
      : fragmentDensityMapAttachment( fragmentDensityMapAttachment_ )
    {
    }

    RenderPassFragmentDensityMapCreateInfoEXT( VkRenderPassFragmentDensityMapCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( RenderPassFragmentDensityMapCreateInfoEXT ) );
    }

    RenderPassFragmentDensityMapCreateInfoEXT& operator=( VkRenderPassFragmentDensityMapCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( RenderPassFragmentDensityMapCreateInfoEXT ) );
      return *this;
    }
    RenderPassFragmentDensityMapCreateInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    RenderPassFragmentDensityMapCreateInfoEXT& setFragmentDensityMapAttachment( AttachmentReference fragmentDensityMapAttachment_ )
    {
      fragmentDensityMapAttachment = fragmentDensityMapAttachment_;
      return *this;
    }

    operator VkRenderPassFragmentDensityMapCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkRenderPassFragmentDensityMapCreateInfoEXT*>(this);
    }

    operator VkRenderPassFragmentDensityMapCreateInfoEXT &()
    {
      return *reinterpret_cast<VkRenderPassFragmentDensityMapCreateInfoEXT*>(this);
    }

    bool operator==( RenderPassFragmentDensityMapCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( fragmentDensityMapAttachment == rhs.fragmentDensityMapAttachment );
    }

    bool operator!=( RenderPassFragmentDensityMapCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eRenderPassFragmentDensityMapCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    AttachmentReference fragmentDensityMapAttachment;
  };
  static_assert( sizeof( RenderPassFragmentDensityMapCreateInfoEXT ) == sizeof( VkRenderPassFragmentDensityMapCreateInfoEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceScalarBlockLayoutFeaturesEXT
  {
    PhysicalDeviceScalarBlockLayoutFeaturesEXT( Bool32 scalarBlockLayout_ = 0 )
      : scalarBlockLayout( scalarBlockLayout_ )
    {
    }

    PhysicalDeviceScalarBlockLayoutFeaturesEXT( VkPhysicalDeviceScalarBlockLayoutFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceScalarBlockLayoutFeaturesEXT ) );
    }

    PhysicalDeviceScalarBlockLayoutFeaturesEXT& operator=( VkPhysicalDeviceScalarBlockLayoutFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceScalarBlockLayoutFeaturesEXT ) );
      return *this;
    }
    PhysicalDeviceScalarBlockLayoutFeaturesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceScalarBlockLayoutFeaturesEXT& setScalarBlockLayout( Bool32 scalarBlockLayout_ )
    {
      scalarBlockLayout = scalarBlockLayout_;
      return *this;
    }

    operator VkPhysicalDeviceScalarBlockLayoutFeaturesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceScalarBlockLayoutFeaturesEXT*>(this);
    }

    operator VkPhysicalDeviceScalarBlockLayoutFeaturesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceScalarBlockLayoutFeaturesEXT*>(this);
    }

    bool operator==( PhysicalDeviceScalarBlockLayoutFeaturesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( scalarBlockLayout == rhs.scalarBlockLayout );
    }

    bool operator!=( PhysicalDeviceScalarBlockLayoutFeaturesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceScalarBlockLayoutFeaturesEXT;

  public:
    void* pNext = nullptr;
    Bool32 scalarBlockLayout;
  };
  static_assert( sizeof( PhysicalDeviceScalarBlockLayoutFeaturesEXT ) == sizeof( VkPhysicalDeviceScalarBlockLayoutFeaturesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceMemoryBudgetPropertiesEXT
  {
    operator VkPhysicalDeviceMemoryBudgetPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceMemoryBudgetPropertiesEXT*>(this);
    }

    operator VkPhysicalDeviceMemoryBudgetPropertiesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceMemoryBudgetPropertiesEXT*>(this);
    }

    bool operator==( PhysicalDeviceMemoryBudgetPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( memcmp( heapBudget, rhs.heapBudget, VK_MAX_MEMORY_HEAPS * sizeof( DeviceSize ) ) == 0 )
          && ( memcmp( heapUsage, rhs.heapUsage, VK_MAX_MEMORY_HEAPS * sizeof( DeviceSize ) ) == 0 );
    }

    bool operator!=( PhysicalDeviceMemoryBudgetPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceMemoryBudgetPropertiesEXT;

  public:
    void* pNext = nullptr;
    DeviceSize heapBudget[VK_MAX_MEMORY_HEAPS];
    DeviceSize heapUsage[VK_MAX_MEMORY_HEAPS];
  };
  static_assert( sizeof( PhysicalDeviceMemoryBudgetPropertiesEXT ) == sizeof( VkPhysicalDeviceMemoryBudgetPropertiesEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceMemoryPriorityFeaturesEXT
  {
    PhysicalDeviceMemoryPriorityFeaturesEXT( Bool32 memoryPriority_ = 0 )
      : memoryPriority( memoryPriority_ )
    {
    }

    PhysicalDeviceMemoryPriorityFeaturesEXT( VkPhysicalDeviceMemoryPriorityFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceMemoryPriorityFeaturesEXT ) );
    }

    PhysicalDeviceMemoryPriorityFeaturesEXT& operator=( VkPhysicalDeviceMemoryPriorityFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceMemoryPriorityFeaturesEXT ) );
      return *this;
    }
    PhysicalDeviceMemoryPriorityFeaturesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceMemoryPriorityFeaturesEXT& setMemoryPriority( Bool32 memoryPriority_ )
    {
      memoryPriority = memoryPriority_;
      return *this;
    }

    operator VkPhysicalDeviceMemoryPriorityFeaturesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceMemoryPriorityFeaturesEXT*>(this);
    }

    operator VkPhysicalDeviceMemoryPriorityFeaturesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceMemoryPriorityFeaturesEXT*>(this);
    }

    bool operator==( PhysicalDeviceMemoryPriorityFeaturesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( memoryPriority == rhs.memoryPriority );
    }

    bool operator!=( PhysicalDeviceMemoryPriorityFeaturesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceMemoryPriorityFeaturesEXT;

  public:
    void* pNext = nullptr;
    Bool32 memoryPriority;
  };
  static_assert( sizeof( PhysicalDeviceMemoryPriorityFeaturesEXT ) == sizeof( VkPhysicalDeviceMemoryPriorityFeaturesEXT ), "struct and wrapper have different size!" );

  struct MemoryPriorityAllocateInfoEXT
  {
    MemoryPriorityAllocateInfoEXT( float priority_ = 0 )
      : priority( priority_ )
    {
    }

    MemoryPriorityAllocateInfoEXT( VkMemoryPriorityAllocateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( MemoryPriorityAllocateInfoEXT ) );
    }

    MemoryPriorityAllocateInfoEXT& operator=( VkMemoryPriorityAllocateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( MemoryPriorityAllocateInfoEXT ) );
      return *this;
    }
    MemoryPriorityAllocateInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    MemoryPriorityAllocateInfoEXT& setPriority( float priority_ )
    {
      priority = priority_;
      return *this;
    }

    operator VkMemoryPriorityAllocateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkMemoryPriorityAllocateInfoEXT*>(this);
    }

    operator VkMemoryPriorityAllocateInfoEXT &()
    {
      return *reinterpret_cast<VkMemoryPriorityAllocateInfoEXT*>(this);
    }

    bool operator==( MemoryPriorityAllocateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( priority == rhs.priority );
    }

    bool operator!=( MemoryPriorityAllocateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eMemoryPriorityAllocateInfoEXT;

  public:
    const void* pNext = nullptr;
    float priority;
  };
  static_assert( sizeof( MemoryPriorityAllocateInfoEXT ) == sizeof( VkMemoryPriorityAllocateInfoEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceBufferAddressFeaturesEXT
  {
    PhysicalDeviceBufferAddressFeaturesEXT( Bool32 bufferDeviceAddress_ = 0,
                                            Bool32 bufferDeviceAddressCaptureReplay_ = 0,
                                            Bool32 bufferDeviceAddressMultiDevice_ = 0 )
      : bufferDeviceAddress( bufferDeviceAddress_ )
      , bufferDeviceAddressCaptureReplay( bufferDeviceAddressCaptureReplay_ )
      , bufferDeviceAddressMultiDevice( bufferDeviceAddressMultiDevice_ )
    {
    }

    PhysicalDeviceBufferAddressFeaturesEXT( VkPhysicalDeviceBufferAddressFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceBufferAddressFeaturesEXT ) );
    }

    PhysicalDeviceBufferAddressFeaturesEXT& operator=( VkPhysicalDeviceBufferAddressFeaturesEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceBufferAddressFeaturesEXT ) );
      return *this;
    }
    PhysicalDeviceBufferAddressFeaturesEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceBufferAddressFeaturesEXT& setBufferDeviceAddress( Bool32 bufferDeviceAddress_ )
    {
      bufferDeviceAddress = bufferDeviceAddress_;
      return *this;
    }

    PhysicalDeviceBufferAddressFeaturesEXT& setBufferDeviceAddressCaptureReplay( Bool32 bufferDeviceAddressCaptureReplay_ )
    {
      bufferDeviceAddressCaptureReplay = bufferDeviceAddressCaptureReplay_;
      return *this;
    }

    PhysicalDeviceBufferAddressFeaturesEXT& setBufferDeviceAddressMultiDevice( Bool32 bufferDeviceAddressMultiDevice_ )
    {
      bufferDeviceAddressMultiDevice = bufferDeviceAddressMultiDevice_;
      return *this;
    }

    operator VkPhysicalDeviceBufferAddressFeaturesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceBufferAddressFeaturesEXT*>(this);
    }

    operator VkPhysicalDeviceBufferAddressFeaturesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceBufferAddressFeaturesEXT*>(this);
    }

    bool operator==( PhysicalDeviceBufferAddressFeaturesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( bufferDeviceAddress == rhs.bufferDeviceAddress )
          && ( bufferDeviceAddressCaptureReplay == rhs.bufferDeviceAddressCaptureReplay )
          && ( bufferDeviceAddressMultiDevice == rhs.bufferDeviceAddressMultiDevice );
    }

    bool operator!=( PhysicalDeviceBufferAddressFeaturesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceBufferAddressFeaturesEXT;

  public:
    void* pNext = nullptr;
    Bool32 bufferDeviceAddress;
    Bool32 bufferDeviceAddressCaptureReplay;
    Bool32 bufferDeviceAddressMultiDevice;
  };
  static_assert( sizeof( PhysicalDeviceBufferAddressFeaturesEXT ) == sizeof( VkPhysicalDeviceBufferAddressFeaturesEXT ), "struct and wrapper have different size!" );

  struct BufferDeviceAddressInfoEXT
  {
    BufferDeviceAddressInfoEXT( Buffer buffer_ = Buffer() )
      : buffer( buffer_ )
    {
    }

    BufferDeviceAddressInfoEXT( VkBufferDeviceAddressInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferDeviceAddressInfoEXT ) );
    }

    BufferDeviceAddressInfoEXT& operator=( VkBufferDeviceAddressInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferDeviceAddressInfoEXT ) );
      return *this;
    }
    BufferDeviceAddressInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BufferDeviceAddressInfoEXT& setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    operator VkBufferDeviceAddressInfoEXT const&() const
    {
      return *reinterpret_cast<const VkBufferDeviceAddressInfoEXT*>(this);
    }

    operator VkBufferDeviceAddressInfoEXT &()
    {
      return *reinterpret_cast<VkBufferDeviceAddressInfoEXT*>(this);
    }

    bool operator==( BufferDeviceAddressInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( buffer == rhs.buffer );
    }

    bool operator!=( BufferDeviceAddressInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBufferDeviceAddressInfoEXT;

  public:
    const void* pNext = nullptr;
    Buffer buffer;
  };
  static_assert( sizeof( BufferDeviceAddressInfoEXT ) == sizeof( VkBufferDeviceAddressInfoEXT ), "struct and wrapper have different size!" );

  struct BufferDeviceAddressCreateInfoEXT
  {
    BufferDeviceAddressCreateInfoEXT( DeviceSize deviceAddress_ = 0 )
      : deviceAddress( deviceAddress_ )
    {
    }

    BufferDeviceAddressCreateInfoEXT( VkBufferDeviceAddressCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferDeviceAddressCreateInfoEXT ) );
    }

    BufferDeviceAddressCreateInfoEXT& operator=( VkBufferDeviceAddressCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferDeviceAddressCreateInfoEXT ) );
      return *this;
    }
    BufferDeviceAddressCreateInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BufferDeviceAddressCreateInfoEXT& setDeviceAddress( DeviceSize deviceAddress_ )
    {
      deviceAddress = deviceAddress_;
      return *this;
    }

    operator VkBufferDeviceAddressCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkBufferDeviceAddressCreateInfoEXT*>(this);
    }

    operator VkBufferDeviceAddressCreateInfoEXT &()
    {
      return *reinterpret_cast<VkBufferDeviceAddressCreateInfoEXT*>(this);
    }

    bool operator==( BufferDeviceAddressCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( deviceAddress == rhs.deviceAddress );
    }

    bool operator!=( BufferDeviceAddressCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBufferDeviceAddressCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    DeviceSize deviceAddress;
  };
  static_assert( sizeof( BufferDeviceAddressCreateInfoEXT ) == sizeof( VkBufferDeviceAddressCreateInfoEXT ), "struct and wrapper have different size!" );

  enum class SubpassContents
  {
    eInline = VK_SUBPASS_CONTENTS_INLINE,
    eSecondaryCommandBuffers = VK_SUBPASS_CONTENTS_SECONDARY_COMMAND_BUFFERS
  };

  struct SubpassBeginInfoKHR
  {
    SubpassBeginInfoKHR( SubpassContents contents_ = SubpassContents::eInline )
      : contents( contents_ )
    {
    }

    SubpassBeginInfoKHR( VkSubpassBeginInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( SubpassBeginInfoKHR ) );
    }

    SubpassBeginInfoKHR& operator=( VkSubpassBeginInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( SubpassBeginInfoKHR ) );
      return *this;
    }
    SubpassBeginInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    SubpassBeginInfoKHR& setContents( SubpassContents contents_ )
    {
      contents = contents_;
      return *this;
    }

    operator VkSubpassBeginInfoKHR const&() const
    {
      return *reinterpret_cast<const VkSubpassBeginInfoKHR*>(this);
    }

    operator VkSubpassBeginInfoKHR &()
    {
      return *reinterpret_cast<VkSubpassBeginInfoKHR*>(this);
    }

    bool operator==( SubpassBeginInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( contents == rhs.contents );
    }

    bool operator!=( SubpassBeginInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eSubpassBeginInfoKHR;

  public:
    const void* pNext = nullptr;
    SubpassContents contents;
  };
  static_assert( sizeof( SubpassBeginInfoKHR ) == sizeof( VkSubpassBeginInfoKHR ), "struct and wrapper have different size!" );

  struct PresentInfoKHR
  {
    PresentInfoKHR( uint32_t waitSemaphoreCount_ = 0,
                    const Semaphore* pWaitSemaphores_ = nullptr,
                    uint32_t swapchainCount_ = 0,
                    const SwapchainKHR* pSwapchains_ = nullptr,
                    const uint32_t* pImageIndices_ = nullptr,
                    Result* pResults_ = nullptr )
      : waitSemaphoreCount( waitSemaphoreCount_ )
      , pWaitSemaphores( pWaitSemaphores_ )
      , swapchainCount( swapchainCount_ )
      , pSwapchains( pSwapchains_ )
      , pImageIndices( pImageIndices_ )
      , pResults( pResults_ )
    {
    }

    PresentInfoKHR( VkPresentInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PresentInfoKHR ) );
    }

    PresentInfoKHR& operator=( VkPresentInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( PresentInfoKHR ) );
      return *this;
    }
    PresentInfoKHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PresentInfoKHR& setWaitSemaphoreCount( uint32_t waitSemaphoreCount_ )
    {
      waitSemaphoreCount = waitSemaphoreCount_;
      return *this;
    }

    PresentInfoKHR& setPWaitSemaphores( const Semaphore* pWaitSemaphores_ )
    {
      pWaitSemaphores = pWaitSemaphores_;
      return *this;
    }

    PresentInfoKHR& setSwapchainCount( uint32_t swapchainCount_ )
    {
      swapchainCount = swapchainCount_;
      return *this;
    }

    PresentInfoKHR& setPSwapchains( const SwapchainKHR* pSwapchains_ )
    {
      pSwapchains = pSwapchains_;
      return *this;
    }

    PresentInfoKHR& setPImageIndices( const uint32_t* pImageIndices_ )
    {
      pImageIndices = pImageIndices_;
      return *this;
    }

    PresentInfoKHR& setPResults( Result* pResults_ )
    {
      pResults = pResults_;
      return *this;
    }

    operator VkPresentInfoKHR const&() const
    {
      return *reinterpret_cast<const VkPresentInfoKHR*>(this);
    }

    operator VkPresentInfoKHR &()
    {
      return *reinterpret_cast<VkPresentInfoKHR*>(this);
    }

    bool operator==( PresentInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( waitSemaphoreCount == rhs.waitSemaphoreCount )
          && ( pWaitSemaphores == rhs.pWaitSemaphores )
          && ( swapchainCount == rhs.swapchainCount )
          && ( pSwapchains == rhs.pSwapchains )
          && ( pImageIndices == rhs.pImageIndices )
          && ( pResults == rhs.pResults );
    }

    bool operator!=( PresentInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePresentInfoKHR;

  public:
    const void* pNext = nullptr;
    uint32_t waitSemaphoreCount;
    const Semaphore* pWaitSemaphores;
    uint32_t swapchainCount;
    const SwapchainKHR* pSwapchains;
    const uint32_t* pImageIndices;
    Result* pResults;
  };
  static_assert( sizeof( PresentInfoKHR ) == sizeof( VkPresentInfoKHR ), "struct and wrapper have different size!" );

  enum class DynamicState
  {
    eViewport = VK_DYNAMIC_STATE_VIEWPORT,
    eScissor = VK_DYNAMIC_STATE_SCISSOR,
    eLineWidth = VK_DYNAMIC_STATE_LINE_WIDTH,
    eDepthBias = VK_DYNAMIC_STATE_DEPTH_BIAS,
    eBlendConstants = VK_DYNAMIC_STATE_BLEND_CONSTANTS,
    eDepthBounds = VK_DYNAMIC_STATE_DEPTH_BOUNDS,
    eStencilCompareMask = VK_DYNAMIC_STATE_STENCIL_COMPARE_MASK,
    eStencilWriteMask = VK_DYNAMIC_STATE_STENCIL_WRITE_MASK,
    eStencilReference = VK_DYNAMIC_STATE_STENCIL_REFERENCE,
    eViewportWScalingNV = VK_DYNAMIC_STATE_VIEWPORT_W_SCALING_NV,
    eDiscardRectangleEXT = VK_DYNAMIC_STATE_DISCARD_RECTANGLE_EXT,
    eSampleLocationsEXT = VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT,
    eViewportShadingRatePaletteNV = VK_DYNAMIC_STATE_VIEWPORT_SHADING_RATE_PALETTE_NV,
    eViewportCoarseSampleOrderNV = VK_DYNAMIC_STATE_VIEWPORT_COARSE_SAMPLE_ORDER_NV,
    eExclusiveScissorNV = VK_DYNAMIC_STATE_EXCLUSIVE_SCISSOR_NV
  };

  struct PipelineDynamicStateCreateInfo
  {
    PipelineDynamicStateCreateInfo( PipelineDynamicStateCreateFlags flags_ = PipelineDynamicStateCreateFlags(),
                                    uint32_t dynamicStateCount_ = 0,
                                    const DynamicState* pDynamicStates_ = nullptr )
      : flags( flags_ )
      , dynamicStateCount( dynamicStateCount_ )
      , pDynamicStates( pDynamicStates_ )
    {
    }

    PipelineDynamicStateCreateInfo( VkPipelineDynamicStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineDynamicStateCreateInfo ) );
    }

    PipelineDynamicStateCreateInfo& operator=( VkPipelineDynamicStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineDynamicStateCreateInfo ) );
      return *this;
    }
    PipelineDynamicStateCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineDynamicStateCreateInfo& setFlags( PipelineDynamicStateCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineDynamicStateCreateInfo& setDynamicStateCount( uint32_t dynamicStateCount_ )
    {
      dynamicStateCount = dynamicStateCount_;
      return *this;
    }

    PipelineDynamicStateCreateInfo& setPDynamicStates( const DynamicState* pDynamicStates_ )
    {
      pDynamicStates = pDynamicStates_;
      return *this;
    }

    operator VkPipelineDynamicStateCreateInfo const&() const
    {
      return *reinterpret_cast<const VkPipelineDynamicStateCreateInfo*>(this);
    }

    operator VkPipelineDynamicStateCreateInfo &()
    {
      return *reinterpret_cast<VkPipelineDynamicStateCreateInfo*>(this);
    }

    bool operator==( PipelineDynamicStateCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( dynamicStateCount == rhs.dynamicStateCount )
          && ( pDynamicStates == rhs.pDynamicStates );
    }

    bool operator!=( PipelineDynamicStateCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineDynamicStateCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineDynamicStateCreateFlags flags;
    uint32_t dynamicStateCount;
    const DynamicState* pDynamicStates;
  };
  static_assert( sizeof( PipelineDynamicStateCreateInfo ) == sizeof( VkPipelineDynamicStateCreateInfo ), "struct and wrapper have different size!" );

  enum class DescriptorUpdateTemplateType
  {
    eDescriptorSet = VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE_DESCRIPTOR_SET,
    eDescriptorSetKHR = VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE_DESCRIPTOR_SET,
    ePushDescriptorsKHR = VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE_PUSH_DESCRIPTORS_KHR
  };

  struct DescriptorUpdateTemplateCreateInfo
  {
    DescriptorUpdateTemplateCreateInfo( DescriptorUpdateTemplateCreateFlags flags_ = DescriptorUpdateTemplateCreateFlags(),
                                        uint32_t descriptorUpdateEntryCount_ = 0,
                                        const DescriptorUpdateTemplateEntry* pDescriptorUpdateEntries_ = nullptr,
                                        DescriptorUpdateTemplateType templateType_ = DescriptorUpdateTemplateType::eDescriptorSet,
                                        DescriptorSetLayout descriptorSetLayout_ = DescriptorSetLayout(),
                                        PipelineBindPoint pipelineBindPoint_ = PipelineBindPoint::eGraphics,
                                        PipelineLayout pipelineLayout_ = PipelineLayout(),
                                        uint32_t set_ = 0 )
      : flags( flags_ )
      , descriptorUpdateEntryCount( descriptorUpdateEntryCount_ )
      , pDescriptorUpdateEntries( pDescriptorUpdateEntries_ )
      , templateType( templateType_ )
      , descriptorSetLayout( descriptorSetLayout_ )
      , pipelineBindPoint( pipelineBindPoint_ )
      , pipelineLayout( pipelineLayout_ )
      , set( set_ )
    {
    }

    DescriptorUpdateTemplateCreateInfo( VkDescriptorUpdateTemplateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorUpdateTemplateCreateInfo ) );
    }

    DescriptorUpdateTemplateCreateInfo& operator=( VkDescriptorUpdateTemplateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorUpdateTemplateCreateInfo ) );
      return *this;
    }
    DescriptorUpdateTemplateCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DescriptorUpdateTemplateCreateInfo& setFlags( DescriptorUpdateTemplateCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    DescriptorUpdateTemplateCreateInfo& setDescriptorUpdateEntryCount( uint32_t descriptorUpdateEntryCount_ )
    {
      descriptorUpdateEntryCount = descriptorUpdateEntryCount_;
      return *this;
    }

    DescriptorUpdateTemplateCreateInfo& setPDescriptorUpdateEntries( const DescriptorUpdateTemplateEntry* pDescriptorUpdateEntries_ )
    {
      pDescriptorUpdateEntries = pDescriptorUpdateEntries_;
      return *this;
    }

    DescriptorUpdateTemplateCreateInfo& setTemplateType( DescriptorUpdateTemplateType templateType_ )
    {
      templateType = templateType_;
      return *this;
    }

    DescriptorUpdateTemplateCreateInfo& setDescriptorSetLayout( DescriptorSetLayout descriptorSetLayout_ )
    {
      descriptorSetLayout = descriptorSetLayout_;
      return *this;
    }

    DescriptorUpdateTemplateCreateInfo& setPipelineBindPoint( PipelineBindPoint pipelineBindPoint_ )
    {
      pipelineBindPoint = pipelineBindPoint_;
      return *this;
    }

    DescriptorUpdateTemplateCreateInfo& setPipelineLayout( PipelineLayout pipelineLayout_ )
    {
      pipelineLayout = pipelineLayout_;
      return *this;
    }

    DescriptorUpdateTemplateCreateInfo& setSet( uint32_t set_ )
    {
      set = set_;
      return *this;
    }

    operator VkDescriptorUpdateTemplateCreateInfo const&() const
    {
      return *reinterpret_cast<const VkDescriptorUpdateTemplateCreateInfo*>(this);
    }

    operator VkDescriptorUpdateTemplateCreateInfo &()
    {
      return *reinterpret_cast<VkDescriptorUpdateTemplateCreateInfo*>(this);
    }

    bool operator==( DescriptorUpdateTemplateCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( descriptorUpdateEntryCount == rhs.descriptorUpdateEntryCount )
          && ( pDescriptorUpdateEntries == rhs.pDescriptorUpdateEntries )
          && ( templateType == rhs.templateType )
          && ( descriptorSetLayout == rhs.descriptorSetLayout )
          && ( pipelineBindPoint == rhs.pipelineBindPoint )
          && ( pipelineLayout == rhs.pipelineLayout )
          && ( set == rhs.set );
    }

    bool operator!=( DescriptorUpdateTemplateCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDescriptorUpdateTemplateCreateInfo;

  public:
    const void* pNext = nullptr;
    DescriptorUpdateTemplateCreateFlags flags;
    uint32_t descriptorUpdateEntryCount;
    const DescriptorUpdateTemplateEntry* pDescriptorUpdateEntries;
    DescriptorUpdateTemplateType templateType;
    DescriptorSetLayout descriptorSetLayout;
    PipelineBindPoint pipelineBindPoint;
    PipelineLayout pipelineLayout;
    uint32_t set;
  };
  static_assert( sizeof( DescriptorUpdateTemplateCreateInfo ) == sizeof( VkDescriptorUpdateTemplateCreateInfo ), "struct and wrapper have different size!" );

  using DescriptorUpdateTemplateCreateInfoKHR = DescriptorUpdateTemplateCreateInfo;

  enum class ObjectType
  {
    eUnknown = VK_OBJECT_TYPE_UNKNOWN,
    eInstance = VK_OBJECT_TYPE_INSTANCE,
    ePhysicalDevice = VK_OBJECT_TYPE_PHYSICAL_DEVICE,
    eDevice = VK_OBJECT_TYPE_DEVICE,
    eQueue = VK_OBJECT_TYPE_QUEUE,
    eSemaphore = VK_OBJECT_TYPE_SEMAPHORE,
    eCommandBuffer = VK_OBJECT_TYPE_COMMAND_BUFFER,
    eFence = VK_OBJECT_TYPE_FENCE,
    eDeviceMemory = VK_OBJECT_TYPE_DEVICE_MEMORY,
    eBuffer = VK_OBJECT_TYPE_BUFFER,
    eImage = VK_OBJECT_TYPE_IMAGE,
    eEvent = VK_OBJECT_TYPE_EVENT,
    eQueryPool = VK_OBJECT_TYPE_QUERY_POOL,
    eBufferView = VK_OBJECT_TYPE_BUFFER_VIEW,
    eImageView = VK_OBJECT_TYPE_IMAGE_VIEW,
    eShaderModule = VK_OBJECT_TYPE_SHADER_MODULE,
    ePipelineCache = VK_OBJECT_TYPE_PIPELINE_CACHE,
    ePipelineLayout = VK_OBJECT_TYPE_PIPELINE_LAYOUT,
    eRenderPass = VK_OBJECT_TYPE_RENDER_PASS,
    ePipeline = VK_OBJECT_TYPE_PIPELINE,
    eDescriptorSetLayout = VK_OBJECT_TYPE_DESCRIPTOR_SET_LAYOUT,
    eSampler = VK_OBJECT_TYPE_SAMPLER,
    eDescriptorPool = VK_OBJECT_TYPE_DESCRIPTOR_POOL,
    eDescriptorSet = VK_OBJECT_TYPE_DESCRIPTOR_SET,
    eFramebuffer = VK_OBJECT_TYPE_FRAMEBUFFER,
    eCommandPool = VK_OBJECT_TYPE_COMMAND_POOL,
    eSamplerYcbcrConversion = VK_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION,
    eSamplerYcbcrConversionKHR = VK_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION,
    eDescriptorUpdateTemplate = VK_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE,
    eDescriptorUpdateTemplateKHR = VK_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE,
    eSurfaceKHR = VK_OBJECT_TYPE_SURFACE_KHR,
    eSwapchainKHR = VK_OBJECT_TYPE_SWAPCHAIN_KHR,
    eDisplayKHR = VK_OBJECT_TYPE_DISPLAY_KHR,
    eDisplayModeKHR = VK_OBJECT_TYPE_DISPLAY_MODE_KHR,
    eDebugReportCallbackEXT = VK_OBJECT_TYPE_DEBUG_REPORT_CALLBACK_EXT,
    eObjectTableNVX = VK_OBJECT_TYPE_OBJECT_TABLE_NVX,
    eIndirectCommandsLayoutNVX = VK_OBJECT_TYPE_INDIRECT_COMMANDS_LAYOUT_NVX,
    eDebugUtilsMessengerEXT = VK_OBJECT_TYPE_DEBUG_UTILS_MESSENGER_EXT,
    eValidationCacheEXT = VK_OBJECT_TYPE_VALIDATION_CACHE_EXT,
    eAccelerationStructureNV = VK_OBJECT_TYPE_ACCELERATION_STRUCTURE_NV
  };

  struct DebugUtilsObjectNameInfoEXT
  {
    DebugUtilsObjectNameInfoEXT( ObjectType objectType_ = ObjectType::eUnknown,
                                 uint64_t objectHandle_ = 0,
                                 const char* pObjectName_ = nullptr )
      : objectType( objectType_ )
      , objectHandle( objectHandle_ )
      , pObjectName( pObjectName_ )
    {
    }

    DebugUtilsObjectNameInfoEXT( VkDebugUtilsObjectNameInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsObjectNameInfoEXT ) );
    }

    DebugUtilsObjectNameInfoEXT& operator=( VkDebugUtilsObjectNameInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsObjectNameInfoEXT ) );
      return *this;
    }
    DebugUtilsObjectNameInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugUtilsObjectNameInfoEXT& setObjectType( ObjectType objectType_ )
    {
      objectType = objectType_;
      return *this;
    }

    DebugUtilsObjectNameInfoEXT& setObjectHandle( uint64_t objectHandle_ )
    {
      objectHandle = objectHandle_;
      return *this;
    }

    DebugUtilsObjectNameInfoEXT& setPObjectName( const char* pObjectName_ )
    {
      pObjectName = pObjectName_;
      return *this;
    }

    operator VkDebugUtilsObjectNameInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDebugUtilsObjectNameInfoEXT*>(this);
    }

    operator VkDebugUtilsObjectNameInfoEXT &()
    {
      return *reinterpret_cast<VkDebugUtilsObjectNameInfoEXT*>(this);
    }

    bool operator==( DebugUtilsObjectNameInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( objectType == rhs.objectType )
          && ( objectHandle == rhs.objectHandle )
          && ( pObjectName == rhs.pObjectName );
    }

    bool operator!=( DebugUtilsObjectNameInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugUtilsObjectNameInfoEXT;

  public:
    const void* pNext = nullptr;
    ObjectType objectType;
    uint64_t objectHandle;
    const char* pObjectName;
  };
  static_assert( sizeof( DebugUtilsObjectNameInfoEXT ) == sizeof( VkDebugUtilsObjectNameInfoEXT ), "struct and wrapper have different size!" );

  struct DebugUtilsObjectTagInfoEXT
  {
    DebugUtilsObjectTagInfoEXT( ObjectType objectType_ = ObjectType::eUnknown,
                                uint64_t objectHandle_ = 0,
                                uint64_t tagName_ = 0,
                                size_t tagSize_ = 0,
                                const void* pTag_ = nullptr )
      : objectType( objectType_ )
      , objectHandle( objectHandle_ )
      , tagName( tagName_ )
      , tagSize( tagSize_ )
      , pTag( pTag_ )
    {
    }

    DebugUtilsObjectTagInfoEXT( VkDebugUtilsObjectTagInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsObjectTagInfoEXT ) );
    }

    DebugUtilsObjectTagInfoEXT& operator=( VkDebugUtilsObjectTagInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsObjectTagInfoEXT ) );
      return *this;
    }
    DebugUtilsObjectTagInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugUtilsObjectTagInfoEXT& setObjectType( ObjectType objectType_ )
    {
      objectType = objectType_;
      return *this;
    }

    DebugUtilsObjectTagInfoEXT& setObjectHandle( uint64_t objectHandle_ )
    {
      objectHandle = objectHandle_;
      return *this;
    }

    DebugUtilsObjectTagInfoEXT& setTagName( uint64_t tagName_ )
    {
      tagName = tagName_;
      return *this;
    }

    DebugUtilsObjectTagInfoEXT& setTagSize( size_t tagSize_ )
    {
      tagSize = tagSize_;
      return *this;
    }

    DebugUtilsObjectTagInfoEXT& setPTag( const void* pTag_ )
    {
      pTag = pTag_;
      return *this;
    }

    operator VkDebugUtilsObjectTagInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDebugUtilsObjectTagInfoEXT*>(this);
    }

    operator VkDebugUtilsObjectTagInfoEXT &()
    {
      return *reinterpret_cast<VkDebugUtilsObjectTagInfoEXT*>(this);
    }

    bool operator==( DebugUtilsObjectTagInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( objectType == rhs.objectType )
          && ( objectHandle == rhs.objectHandle )
          && ( tagName == rhs.tagName )
          && ( tagSize == rhs.tagSize )
          && ( pTag == rhs.pTag );
    }

    bool operator!=( DebugUtilsObjectTagInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugUtilsObjectTagInfoEXT;

  public:
    const void* pNext = nullptr;
    ObjectType objectType;
    uint64_t objectHandle;
    uint64_t tagName;
    size_t tagSize;
    const void* pTag;
  };
  static_assert( sizeof( DebugUtilsObjectTagInfoEXT ) == sizeof( VkDebugUtilsObjectTagInfoEXT ), "struct and wrapper have different size!" );

  struct DebugUtilsMessengerCallbackDataEXT
  {
    DebugUtilsMessengerCallbackDataEXT( DebugUtilsMessengerCallbackDataFlagsEXT flags_ = DebugUtilsMessengerCallbackDataFlagsEXT(),
                                        const char* pMessageIdName_ = nullptr,
                                        int32_t messageIdNumber_ = 0,
                                        const char* pMessage_ = nullptr,
                                        uint32_t queueLabelCount_ = 0,
                                        const DebugUtilsLabelEXT* pQueueLabels_ = nullptr,
                                        uint32_t cmdBufLabelCount_ = 0,
                                        const DebugUtilsLabelEXT* pCmdBufLabels_ = nullptr,
                                        uint32_t objectCount_ = 0,
                                        const DebugUtilsObjectNameInfoEXT* pObjects_ = nullptr )
      : flags( flags_ )
      , pMessageIdName( pMessageIdName_ )
      , messageIdNumber( messageIdNumber_ )
      , pMessage( pMessage_ )
      , queueLabelCount( queueLabelCount_ )
      , pQueueLabels( pQueueLabels_ )
      , cmdBufLabelCount( cmdBufLabelCount_ )
      , pCmdBufLabels( pCmdBufLabels_ )
      , objectCount( objectCount_ )
      , pObjects( pObjects_ )
    {
    }

    DebugUtilsMessengerCallbackDataEXT( VkDebugUtilsMessengerCallbackDataEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsMessengerCallbackDataEXT ) );
    }

    DebugUtilsMessengerCallbackDataEXT& operator=( VkDebugUtilsMessengerCallbackDataEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsMessengerCallbackDataEXT ) );
      return *this;
    }
    DebugUtilsMessengerCallbackDataEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT& setFlags( DebugUtilsMessengerCallbackDataFlagsEXT flags_ )
    {
      flags = flags_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT& setPMessageIdName( const char* pMessageIdName_ )
    {
      pMessageIdName = pMessageIdName_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT& setMessageIdNumber( int32_t messageIdNumber_ )
    {
      messageIdNumber = messageIdNumber_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT& setPMessage( const char* pMessage_ )
    {
      pMessage = pMessage_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT& setQueueLabelCount( uint32_t queueLabelCount_ )
    {
      queueLabelCount = queueLabelCount_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT& setPQueueLabels( const DebugUtilsLabelEXT* pQueueLabels_ )
    {
      pQueueLabels = pQueueLabels_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT& setCmdBufLabelCount( uint32_t cmdBufLabelCount_ )
    {
      cmdBufLabelCount = cmdBufLabelCount_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT& setPCmdBufLabels( const DebugUtilsLabelEXT* pCmdBufLabels_ )
    {
      pCmdBufLabels = pCmdBufLabels_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT& setObjectCount( uint32_t objectCount_ )
    {
      objectCount = objectCount_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT& setPObjects( const DebugUtilsObjectNameInfoEXT* pObjects_ )
    {
      pObjects = pObjects_;
      return *this;
    }

    operator VkDebugUtilsMessengerCallbackDataEXT const&() const
    {
      return *reinterpret_cast<const VkDebugUtilsMessengerCallbackDataEXT*>(this);
    }

    operator VkDebugUtilsMessengerCallbackDataEXT &()
    {
      return *reinterpret_cast<VkDebugUtilsMessengerCallbackDataEXT*>(this);
    }

    bool operator==( DebugUtilsMessengerCallbackDataEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( pMessageIdName == rhs.pMessageIdName )
          && ( messageIdNumber == rhs.messageIdNumber )
          && ( pMessage == rhs.pMessage )
          && ( queueLabelCount == rhs.queueLabelCount )
          && ( pQueueLabels == rhs.pQueueLabels )
          && ( cmdBufLabelCount == rhs.cmdBufLabelCount )
          && ( pCmdBufLabels == rhs.pCmdBufLabels )
          && ( objectCount == rhs.objectCount )
          && ( pObjects == rhs.pObjects );
    }

    bool operator!=( DebugUtilsMessengerCallbackDataEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugUtilsMessengerCallbackDataEXT;

  public:
    const void* pNext = nullptr;
    DebugUtilsMessengerCallbackDataFlagsEXT flags;
    const char* pMessageIdName;
    int32_t messageIdNumber;
    const char* pMessage;
    uint32_t queueLabelCount;
    const DebugUtilsLabelEXT* pQueueLabels;
    uint32_t cmdBufLabelCount;
    const DebugUtilsLabelEXT* pCmdBufLabels;
    uint32_t objectCount;
    const DebugUtilsObjectNameInfoEXT* pObjects;
  };
  static_assert( sizeof( DebugUtilsMessengerCallbackDataEXT ) == sizeof( VkDebugUtilsMessengerCallbackDataEXT ), "struct and wrapper have different size!" );

  enum class QueueFlagBits
  {
    eGraphics = VK_QUEUE_GRAPHICS_BIT,
    eCompute = VK_QUEUE_COMPUTE_BIT,
    eTransfer = VK_QUEUE_TRANSFER_BIT,
    eSparseBinding = VK_QUEUE_SPARSE_BINDING_BIT,
    eProtected = VK_QUEUE_PROTECTED_BIT
  };

  using QueueFlags = Flags<QueueFlagBits, VkQueueFlags>;

  VULKAN_HPP_INLINE QueueFlags operator|( QueueFlagBits bit0, QueueFlagBits bit1 )
  {
    return QueueFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE QueueFlags operator~( QueueFlagBits bits )
  {
    return ~( QueueFlags( bits ) );
  }

  template <> struct FlagTraits<QueueFlagBits>
  {
    enum
    {
      allFlags = VkFlags(QueueFlagBits::eGraphics) | VkFlags(QueueFlagBits::eCompute) | VkFlags(QueueFlagBits::eTransfer) | VkFlags(QueueFlagBits::eSparseBinding) | VkFlags(QueueFlagBits::eProtected)
    };
  };

  struct QueueFamilyProperties
  {
    operator VkQueueFamilyProperties const&() const
    {
      return *reinterpret_cast<const VkQueueFamilyProperties*>(this);
    }

    operator VkQueueFamilyProperties &()
    {
      return *reinterpret_cast<VkQueueFamilyProperties*>(this);
    }

    bool operator==( QueueFamilyProperties const& rhs ) const
    {
      return ( queueFlags == rhs.queueFlags )
          && ( queueCount == rhs.queueCount )
          && ( timestampValidBits == rhs.timestampValidBits )
          && ( minImageTransferGranularity == rhs.minImageTransferGranularity );
    }

    bool operator!=( QueueFamilyProperties const& rhs ) const
    {
      return !operator==( rhs );
    }

    QueueFlags queueFlags;
    uint32_t queueCount;
    uint32_t timestampValidBits;
    Extent3D minImageTransferGranularity;
  };
  static_assert( sizeof( QueueFamilyProperties ) == sizeof( VkQueueFamilyProperties ), "struct and wrapper have different size!" );

  struct QueueFamilyProperties2
  {
    operator VkQueueFamilyProperties2 const&() const
    {
      return *reinterpret_cast<const VkQueueFamilyProperties2*>(this);
    }

    operator VkQueueFamilyProperties2 &()
    {
      return *reinterpret_cast<VkQueueFamilyProperties2*>(this);
    }

    bool operator==( QueueFamilyProperties2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( queueFamilyProperties == rhs.queueFamilyProperties );
    }

    bool operator!=( QueueFamilyProperties2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eQueueFamilyProperties2;

  public:
    void* pNext = nullptr;
    QueueFamilyProperties queueFamilyProperties;
  };
  static_assert( sizeof( QueueFamilyProperties2 ) == sizeof( VkQueueFamilyProperties2 ), "struct and wrapper have different size!" );

  using QueueFamilyProperties2KHR = QueueFamilyProperties2;

  enum class DeviceQueueCreateFlagBits
  {
    eProtected = VK_DEVICE_QUEUE_CREATE_PROTECTED_BIT
  };

  using DeviceQueueCreateFlags = Flags<DeviceQueueCreateFlagBits, VkDeviceQueueCreateFlags>;

  VULKAN_HPP_INLINE DeviceQueueCreateFlags operator|( DeviceQueueCreateFlagBits bit0, DeviceQueueCreateFlagBits bit1 )
  {
    return DeviceQueueCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE DeviceQueueCreateFlags operator~( DeviceQueueCreateFlagBits bits )
  {
    return ~( DeviceQueueCreateFlags( bits ) );
  }

  template <> struct FlagTraits<DeviceQueueCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(DeviceQueueCreateFlagBits::eProtected)
    };
  };

  struct DeviceQueueCreateInfo
  {
    DeviceQueueCreateInfo( DeviceQueueCreateFlags flags_ = DeviceQueueCreateFlags(),
                           uint32_t queueFamilyIndex_ = 0,
                           uint32_t queueCount_ = 0,
                           const float* pQueuePriorities_ = nullptr )
      : flags( flags_ )
      , queueFamilyIndex( queueFamilyIndex_ )
      , queueCount( queueCount_ )
      , pQueuePriorities( pQueuePriorities_ )
    {
    }

    DeviceQueueCreateInfo( VkDeviceQueueCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceQueueCreateInfo ) );
    }

    DeviceQueueCreateInfo& operator=( VkDeviceQueueCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceQueueCreateInfo ) );
      return *this;
    }
    DeviceQueueCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DeviceQueueCreateInfo& setFlags( DeviceQueueCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    DeviceQueueCreateInfo& setQueueFamilyIndex( uint32_t queueFamilyIndex_ )
    {
      queueFamilyIndex = queueFamilyIndex_;
      return *this;
    }

    DeviceQueueCreateInfo& setQueueCount( uint32_t queueCount_ )
    {
      queueCount = queueCount_;
      return *this;
    }

    DeviceQueueCreateInfo& setPQueuePriorities( const float* pQueuePriorities_ )
    {
      pQueuePriorities = pQueuePriorities_;
      return *this;
    }

    operator VkDeviceQueueCreateInfo const&() const
    {
      return *reinterpret_cast<const VkDeviceQueueCreateInfo*>(this);
    }

    operator VkDeviceQueueCreateInfo &()
    {
      return *reinterpret_cast<VkDeviceQueueCreateInfo*>(this);
    }

    bool operator==( DeviceQueueCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( queueFamilyIndex == rhs.queueFamilyIndex )
          && ( queueCount == rhs.queueCount )
          && ( pQueuePriorities == rhs.pQueuePriorities );
    }

    bool operator!=( DeviceQueueCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDeviceQueueCreateInfo;

  public:
    const void* pNext = nullptr;
    DeviceQueueCreateFlags flags;
    uint32_t queueFamilyIndex;
    uint32_t queueCount;
    const float* pQueuePriorities;
  };
  static_assert( sizeof( DeviceQueueCreateInfo ) == sizeof( VkDeviceQueueCreateInfo ), "struct and wrapper have different size!" );

  struct DeviceCreateInfo
  {
    DeviceCreateInfo( DeviceCreateFlags flags_ = DeviceCreateFlags(),
                      uint32_t queueCreateInfoCount_ = 0,
                      const DeviceQueueCreateInfo* pQueueCreateInfos_ = nullptr,
                      uint32_t enabledLayerCount_ = 0,
                      const char* const* ppEnabledLayerNames_ = nullptr,
                      uint32_t enabledExtensionCount_ = 0,
                      const char* const* ppEnabledExtensionNames_ = nullptr,
                      const PhysicalDeviceFeatures* pEnabledFeatures_ = nullptr )
      : flags( flags_ )
      , queueCreateInfoCount( queueCreateInfoCount_ )
      , pQueueCreateInfos( pQueueCreateInfos_ )
      , enabledLayerCount( enabledLayerCount_ )
      , ppEnabledLayerNames( ppEnabledLayerNames_ )
      , enabledExtensionCount( enabledExtensionCount_ )
      , ppEnabledExtensionNames( ppEnabledExtensionNames_ )
      , pEnabledFeatures( pEnabledFeatures_ )
    {
    }

    DeviceCreateInfo( VkDeviceCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceCreateInfo ) );
    }

    DeviceCreateInfo& operator=( VkDeviceCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceCreateInfo ) );
      return *this;
    }
    DeviceCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DeviceCreateInfo& setFlags( DeviceCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    DeviceCreateInfo& setQueueCreateInfoCount( uint32_t queueCreateInfoCount_ )
    {
      queueCreateInfoCount = queueCreateInfoCount_;
      return *this;
    }

    DeviceCreateInfo& setPQueueCreateInfos( const DeviceQueueCreateInfo* pQueueCreateInfos_ )
    {
      pQueueCreateInfos = pQueueCreateInfos_;
      return *this;
    }

    DeviceCreateInfo& setEnabledLayerCount( uint32_t enabledLayerCount_ )
    {
      enabledLayerCount = enabledLayerCount_;
      return *this;
    }

    DeviceCreateInfo& setPpEnabledLayerNames( const char* const* ppEnabledLayerNames_ )
    {
      ppEnabledLayerNames = ppEnabledLayerNames_;
      return *this;
    }

    DeviceCreateInfo& setEnabledExtensionCount( uint32_t enabledExtensionCount_ )
    {
      enabledExtensionCount = enabledExtensionCount_;
      return *this;
    }

    DeviceCreateInfo& setPpEnabledExtensionNames( const char* const* ppEnabledExtensionNames_ )
    {
      ppEnabledExtensionNames = ppEnabledExtensionNames_;
      return *this;
    }

    DeviceCreateInfo& setPEnabledFeatures( const PhysicalDeviceFeatures* pEnabledFeatures_ )
    {
      pEnabledFeatures = pEnabledFeatures_;
      return *this;
    }

    operator VkDeviceCreateInfo const&() const
    {
      return *reinterpret_cast<const VkDeviceCreateInfo*>(this);
    }

    operator VkDeviceCreateInfo &()
    {
      return *reinterpret_cast<VkDeviceCreateInfo*>(this);
    }

    bool operator==( DeviceCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( queueCreateInfoCount == rhs.queueCreateInfoCount )
          && ( pQueueCreateInfos == rhs.pQueueCreateInfos )
          && ( enabledLayerCount == rhs.enabledLayerCount )
          && ( ppEnabledLayerNames == rhs.ppEnabledLayerNames )
          && ( enabledExtensionCount == rhs.enabledExtensionCount )
          && ( ppEnabledExtensionNames == rhs.ppEnabledExtensionNames )
          && ( pEnabledFeatures == rhs.pEnabledFeatures );
    }

    bool operator!=( DeviceCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDeviceCreateInfo;

  public:
    const void* pNext = nullptr;
    DeviceCreateFlags flags;
    uint32_t queueCreateInfoCount;
    const DeviceQueueCreateInfo* pQueueCreateInfos;
    uint32_t enabledLayerCount;
    const char* const* ppEnabledLayerNames;
    uint32_t enabledExtensionCount;
    const char* const* ppEnabledExtensionNames;
    const PhysicalDeviceFeatures* pEnabledFeatures;
  };
  static_assert( sizeof( DeviceCreateInfo ) == sizeof( VkDeviceCreateInfo ), "struct and wrapper have different size!" );

  struct DeviceQueueInfo2
  {
    DeviceQueueInfo2( DeviceQueueCreateFlags flags_ = DeviceQueueCreateFlags(),
                      uint32_t queueFamilyIndex_ = 0,
                      uint32_t queueIndex_ = 0 )
      : flags( flags_ )
      , queueFamilyIndex( queueFamilyIndex_ )
      , queueIndex( queueIndex_ )
    {
    }

    DeviceQueueInfo2( VkDeviceQueueInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceQueueInfo2 ) );
    }

    DeviceQueueInfo2& operator=( VkDeviceQueueInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( DeviceQueueInfo2 ) );
      return *this;
    }
    DeviceQueueInfo2& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DeviceQueueInfo2& setFlags( DeviceQueueCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    DeviceQueueInfo2& setQueueFamilyIndex( uint32_t queueFamilyIndex_ )
    {
      queueFamilyIndex = queueFamilyIndex_;
      return *this;
    }

    DeviceQueueInfo2& setQueueIndex( uint32_t queueIndex_ )
    {
      queueIndex = queueIndex_;
      return *this;
    }

    operator VkDeviceQueueInfo2 const&() const
    {
      return *reinterpret_cast<const VkDeviceQueueInfo2*>(this);
    }

    operator VkDeviceQueueInfo2 &()
    {
      return *reinterpret_cast<VkDeviceQueueInfo2*>(this);
    }

    bool operator==( DeviceQueueInfo2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( queueFamilyIndex == rhs.queueFamilyIndex )
          && ( queueIndex == rhs.queueIndex );
    }

    bool operator!=( DeviceQueueInfo2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDeviceQueueInfo2;

  public:
    const void* pNext = nullptr;
    DeviceQueueCreateFlags flags;
    uint32_t queueFamilyIndex;
    uint32_t queueIndex;
  };
  static_assert( sizeof( DeviceQueueInfo2 ) == sizeof( VkDeviceQueueInfo2 ), "struct and wrapper have different size!" );

  enum class MemoryPropertyFlagBits
  {
    eDeviceLocal = VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT,
    eHostVisible = VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT,
    eHostCoherent = VK_MEMORY_PROPERTY_HOST_COHERENT_BIT,
    eHostCached = VK_MEMORY_PROPERTY_HOST_CACHED_BIT,
    eLazilyAllocated = VK_MEMORY_PROPERTY_LAZILY_ALLOCATED_BIT,
    eProtected = VK_MEMORY_PROPERTY_PROTECTED_BIT
  };

  using MemoryPropertyFlags = Flags<MemoryPropertyFlagBits, VkMemoryPropertyFlags>;

  VULKAN_HPP_INLINE MemoryPropertyFlags operator|( MemoryPropertyFlagBits bit0, MemoryPropertyFlagBits bit1 )
  {
    return MemoryPropertyFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE MemoryPropertyFlags operator~( MemoryPropertyFlagBits bits )
  {
    return ~( MemoryPropertyFlags( bits ) );
  }

  template <> struct FlagTraits<MemoryPropertyFlagBits>
  {
    enum
    {
      allFlags = VkFlags(MemoryPropertyFlagBits::eDeviceLocal) | VkFlags(MemoryPropertyFlagBits::eHostVisible) | VkFlags(MemoryPropertyFlagBits::eHostCoherent) | VkFlags(MemoryPropertyFlagBits::eHostCached) | VkFlags(MemoryPropertyFlagBits::eLazilyAllocated) | VkFlags(MemoryPropertyFlagBits::eProtected)
    };
  };

  struct MemoryType
  {
    operator VkMemoryType const&() const
    {
      return *reinterpret_cast<const VkMemoryType*>(this);
    }

    operator VkMemoryType &()
    {
      return *reinterpret_cast<VkMemoryType*>(this);
    }

    bool operator==( MemoryType const& rhs ) const
    {
      return ( propertyFlags == rhs.propertyFlags )
          && ( heapIndex == rhs.heapIndex );
    }

    bool operator!=( MemoryType const& rhs ) const
    {
      return !operator==( rhs );
    }

    MemoryPropertyFlags propertyFlags;
    uint32_t heapIndex;
  };
  static_assert( sizeof( MemoryType ) == sizeof( VkMemoryType ), "struct and wrapper have different size!" );

  enum class MemoryHeapFlagBits
  {
    eDeviceLocal = VK_MEMORY_HEAP_DEVICE_LOCAL_BIT,
    eMultiInstance = VK_MEMORY_HEAP_MULTI_INSTANCE_BIT,
    eMultiInstanceKHR = VK_MEMORY_HEAP_MULTI_INSTANCE_BIT
  };

  using MemoryHeapFlags = Flags<MemoryHeapFlagBits, VkMemoryHeapFlags>;

  VULKAN_HPP_INLINE MemoryHeapFlags operator|( MemoryHeapFlagBits bit0, MemoryHeapFlagBits bit1 )
  {
    return MemoryHeapFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE MemoryHeapFlags operator~( MemoryHeapFlagBits bits )
  {
    return ~( MemoryHeapFlags( bits ) );
  }

  template <> struct FlagTraits<MemoryHeapFlagBits>
  {
    enum
    {
      allFlags = VkFlags(MemoryHeapFlagBits::eDeviceLocal) | VkFlags(MemoryHeapFlagBits::eMultiInstance)
    };
  };

  struct MemoryHeap
  {
    operator VkMemoryHeap const&() const
    {
      return *reinterpret_cast<const VkMemoryHeap*>(this);
    }

    operator VkMemoryHeap &()
    {
      return *reinterpret_cast<VkMemoryHeap*>(this);
    }

    bool operator==( MemoryHeap const& rhs ) const
    {
      return ( size == rhs.size )
          && ( flags == rhs.flags );
    }

    bool operator!=( MemoryHeap const& rhs ) const
    {
      return !operator==( rhs );
    }

    DeviceSize size;
    MemoryHeapFlags flags;
  };
  static_assert( sizeof( MemoryHeap ) == sizeof( VkMemoryHeap ), "struct and wrapper have different size!" );

  struct PhysicalDeviceMemoryProperties
  {
    operator VkPhysicalDeviceMemoryProperties const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceMemoryProperties*>(this);
    }

    operator VkPhysicalDeviceMemoryProperties &()
    {
      return *reinterpret_cast<VkPhysicalDeviceMemoryProperties*>(this);
    }

    bool operator==( PhysicalDeviceMemoryProperties const& rhs ) const
    {
      return ( memoryTypeCount == rhs.memoryTypeCount )
          && ( memcmp( memoryTypes, rhs.memoryTypes, VK_MAX_MEMORY_TYPES * sizeof( MemoryType ) ) == 0 )
          && ( memoryHeapCount == rhs.memoryHeapCount )
          && ( memcmp( memoryHeaps, rhs.memoryHeaps, VK_MAX_MEMORY_HEAPS * sizeof( MemoryHeap ) ) == 0 );
    }

    bool operator!=( PhysicalDeviceMemoryProperties const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t memoryTypeCount;
    MemoryType memoryTypes[VK_MAX_MEMORY_TYPES];
    uint32_t memoryHeapCount;
    MemoryHeap memoryHeaps[VK_MAX_MEMORY_HEAPS];
  };
  static_assert( sizeof( PhysicalDeviceMemoryProperties ) == sizeof( VkPhysicalDeviceMemoryProperties ), "struct and wrapper have different size!" );

  struct PhysicalDeviceMemoryProperties2
  {
    operator VkPhysicalDeviceMemoryProperties2 const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceMemoryProperties2*>(this);
    }

    operator VkPhysicalDeviceMemoryProperties2 &()
    {
      return *reinterpret_cast<VkPhysicalDeviceMemoryProperties2*>(this);
    }

    bool operator==( PhysicalDeviceMemoryProperties2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( memoryProperties == rhs.memoryProperties );
    }

    bool operator!=( PhysicalDeviceMemoryProperties2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceMemoryProperties2;

  public:
    void* pNext = nullptr;
    PhysicalDeviceMemoryProperties memoryProperties;
  };
  static_assert( sizeof( PhysicalDeviceMemoryProperties2 ) == sizeof( VkPhysicalDeviceMemoryProperties2 ), "struct and wrapper have different size!" );

  using PhysicalDeviceMemoryProperties2KHR = PhysicalDeviceMemoryProperties2;

  enum class AccessFlagBits
  {
    eIndirectCommandRead = VK_ACCESS_INDIRECT_COMMAND_READ_BIT,
    eIndexRead = VK_ACCESS_INDEX_READ_BIT,
    eVertexAttributeRead = VK_ACCESS_VERTEX_ATTRIBUTE_READ_BIT,
    eUniformRead = VK_ACCESS_UNIFORM_READ_BIT,
    eInputAttachmentRead = VK_ACCESS_INPUT_ATTACHMENT_READ_BIT,
    eShaderRead = VK_ACCESS_SHADER_READ_BIT,
    eShaderWrite = VK_ACCESS_SHADER_WRITE_BIT,
    eColorAttachmentRead = VK_ACCESS_COLOR_ATTACHMENT_READ_BIT,
    eColorAttachmentWrite = VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT,
    eDepthStencilAttachmentRead = VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_READ_BIT,
    eDepthStencilAttachmentWrite = VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT,
    eTransferRead = VK_ACCESS_TRANSFER_READ_BIT,
    eTransferWrite = VK_ACCESS_TRANSFER_WRITE_BIT,
    eHostRead = VK_ACCESS_HOST_READ_BIT,
    eHostWrite = VK_ACCESS_HOST_WRITE_BIT,
    eMemoryRead = VK_ACCESS_MEMORY_READ_BIT,
    eMemoryWrite = VK_ACCESS_MEMORY_WRITE_BIT,
    eTransformFeedbackWriteEXT = VK_ACCESS_TRANSFORM_FEEDBACK_WRITE_BIT_EXT,
    eTransformFeedbackCounterReadEXT = VK_ACCESS_TRANSFORM_FEEDBACK_COUNTER_READ_BIT_EXT,
    eTransformFeedbackCounterWriteEXT = VK_ACCESS_TRANSFORM_FEEDBACK_COUNTER_WRITE_BIT_EXT,
    eConditionalRenderingReadEXT = VK_ACCESS_CONDITIONAL_RENDERING_READ_BIT_EXT,
    eCommandProcessReadNVX = VK_ACCESS_COMMAND_PROCESS_READ_BIT_NVX,
    eCommandProcessWriteNVX = VK_ACCESS_COMMAND_PROCESS_WRITE_BIT_NVX,
    eColorAttachmentReadNoncoherentEXT = VK_ACCESS_COLOR_ATTACHMENT_READ_NONCOHERENT_BIT_EXT,
    eShadingRateImageReadNV = VK_ACCESS_SHADING_RATE_IMAGE_READ_BIT_NV,
    eAccelerationStructureReadNV = VK_ACCESS_ACCELERATION_STRUCTURE_READ_BIT_NV,
    eAccelerationStructureWriteNV = VK_ACCESS_ACCELERATION_STRUCTURE_WRITE_BIT_NV,
    eFragmentDensityMapReadEXT = VK_ACCESS_FRAGMENT_DENSITY_MAP_READ_BIT_EXT
  };

  using AccessFlags = Flags<AccessFlagBits, VkAccessFlags>;

  VULKAN_HPP_INLINE AccessFlags operator|( AccessFlagBits bit0, AccessFlagBits bit1 )
  {
    return AccessFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE AccessFlags operator~( AccessFlagBits bits )
  {
    return ~( AccessFlags( bits ) );
  }

  template <> struct FlagTraits<AccessFlagBits>
  {
    enum
    {
      allFlags = VkFlags(AccessFlagBits::eIndirectCommandRead) | VkFlags(AccessFlagBits::eIndexRead) | VkFlags(AccessFlagBits::eVertexAttributeRead) | VkFlags(AccessFlagBits::eUniformRead) | VkFlags(AccessFlagBits::eInputAttachmentRead) | VkFlags(AccessFlagBits::eShaderRead) | VkFlags(AccessFlagBits::eShaderWrite) | VkFlags(AccessFlagBits::eColorAttachmentRead) | VkFlags(AccessFlagBits::eColorAttachmentWrite) | VkFlags(AccessFlagBits::eDepthStencilAttachmentRead) | VkFlags(AccessFlagBits::eDepthStencilAttachmentWrite) | VkFlags(AccessFlagBits::eTransferRead) | VkFlags(AccessFlagBits::eTransferWrite) | VkFlags(AccessFlagBits::eHostRead) | VkFlags(AccessFlagBits::eHostWrite) | VkFlags(AccessFlagBits::eMemoryRead) | VkFlags(AccessFlagBits::eMemoryWrite) | VkFlags(AccessFlagBits::eTransformFeedbackWriteEXT) | VkFlags(AccessFlagBits::eTransformFeedbackCounterReadEXT) | VkFlags(AccessFlagBits::eTransformFeedbackCounterWriteEXT) | VkFlags(AccessFlagBits::eConditionalRenderingReadEXT) | VkFlags(AccessFlagBits::eCommandProcessReadNVX) | VkFlags(AccessFlagBits::eCommandProcessWriteNVX) | VkFlags(AccessFlagBits::eColorAttachmentReadNoncoherentEXT) | VkFlags(AccessFlagBits::eShadingRateImageReadNV) | VkFlags(AccessFlagBits::eAccelerationStructureReadNV) | VkFlags(AccessFlagBits::eAccelerationStructureWriteNV) | VkFlags(AccessFlagBits::eFragmentDensityMapReadEXT)
    };
  };

  struct MemoryBarrier
  {
    MemoryBarrier( AccessFlags srcAccessMask_ = AccessFlags(),
                   AccessFlags dstAccessMask_ = AccessFlags() )
      : srcAccessMask( srcAccessMask_ )
      , dstAccessMask( dstAccessMask_ )
    {
    }

    MemoryBarrier( VkMemoryBarrier const & rhs )
    {
      memcpy( this, &rhs, sizeof( MemoryBarrier ) );
    }

    MemoryBarrier& operator=( VkMemoryBarrier const & rhs )
    {
      memcpy( this, &rhs, sizeof( MemoryBarrier ) );
      return *this;
    }
    MemoryBarrier& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    MemoryBarrier& setSrcAccessMask( AccessFlags srcAccessMask_ )
    {
      srcAccessMask = srcAccessMask_;
      return *this;
    }

    MemoryBarrier& setDstAccessMask( AccessFlags dstAccessMask_ )
    {
      dstAccessMask = dstAccessMask_;
      return *this;
    }

    operator VkMemoryBarrier const&() const
    {
      return *reinterpret_cast<const VkMemoryBarrier*>(this);
    }

    operator VkMemoryBarrier &()
    {
      return *reinterpret_cast<VkMemoryBarrier*>(this);
    }

    bool operator==( MemoryBarrier const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( srcAccessMask == rhs.srcAccessMask )
          && ( dstAccessMask == rhs.dstAccessMask );
    }

    bool operator!=( MemoryBarrier const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eMemoryBarrier;

  public:
    const void* pNext = nullptr;
    AccessFlags srcAccessMask;
    AccessFlags dstAccessMask;
  };
  static_assert( sizeof( MemoryBarrier ) == sizeof( VkMemoryBarrier ), "struct and wrapper have different size!" );

  struct BufferMemoryBarrier
  {
    BufferMemoryBarrier( AccessFlags srcAccessMask_ = AccessFlags(),
                         AccessFlags dstAccessMask_ = AccessFlags(),
                         uint32_t srcQueueFamilyIndex_ = 0,
                         uint32_t dstQueueFamilyIndex_ = 0,
                         Buffer buffer_ = Buffer(),
                         DeviceSize offset_ = 0,
                         DeviceSize size_ = 0 )
      : srcAccessMask( srcAccessMask_ )
      , dstAccessMask( dstAccessMask_ )
      , srcQueueFamilyIndex( srcQueueFamilyIndex_ )
      , dstQueueFamilyIndex( dstQueueFamilyIndex_ )
      , buffer( buffer_ )
      , offset( offset_ )
      , size( size_ )
    {
    }

    BufferMemoryBarrier( VkBufferMemoryBarrier const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferMemoryBarrier ) );
    }

    BufferMemoryBarrier& operator=( VkBufferMemoryBarrier const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferMemoryBarrier ) );
      return *this;
    }
    BufferMemoryBarrier& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BufferMemoryBarrier& setSrcAccessMask( AccessFlags srcAccessMask_ )
    {
      srcAccessMask = srcAccessMask_;
      return *this;
    }

    BufferMemoryBarrier& setDstAccessMask( AccessFlags dstAccessMask_ )
    {
      dstAccessMask = dstAccessMask_;
      return *this;
    }

    BufferMemoryBarrier& setSrcQueueFamilyIndex( uint32_t srcQueueFamilyIndex_ )
    {
      srcQueueFamilyIndex = srcQueueFamilyIndex_;
      return *this;
    }

    BufferMemoryBarrier& setDstQueueFamilyIndex( uint32_t dstQueueFamilyIndex_ )
    {
      dstQueueFamilyIndex = dstQueueFamilyIndex_;
      return *this;
    }

    BufferMemoryBarrier& setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    BufferMemoryBarrier& setOffset( DeviceSize offset_ )
    {
      offset = offset_;
      return *this;
    }

    BufferMemoryBarrier& setSize( DeviceSize size_ )
    {
      size = size_;
      return *this;
    }

    operator VkBufferMemoryBarrier const&() const
    {
      return *reinterpret_cast<const VkBufferMemoryBarrier*>(this);
    }

    operator VkBufferMemoryBarrier &()
    {
      return *reinterpret_cast<VkBufferMemoryBarrier*>(this);
    }

    bool operator==( BufferMemoryBarrier const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( srcAccessMask == rhs.srcAccessMask )
          && ( dstAccessMask == rhs.dstAccessMask )
          && ( srcQueueFamilyIndex == rhs.srcQueueFamilyIndex )
          && ( dstQueueFamilyIndex == rhs.dstQueueFamilyIndex )
          && ( buffer == rhs.buffer )
          && ( offset == rhs.offset )
          && ( size == rhs.size );
    }

    bool operator!=( BufferMemoryBarrier const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBufferMemoryBarrier;

  public:
    const void* pNext = nullptr;
    AccessFlags srcAccessMask;
    AccessFlags dstAccessMask;
    uint32_t srcQueueFamilyIndex;
    uint32_t dstQueueFamilyIndex;
    Buffer buffer;
    DeviceSize offset;
    DeviceSize size;
  };
  static_assert( sizeof( BufferMemoryBarrier ) == sizeof( VkBufferMemoryBarrier ), "struct and wrapper have different size!" );

  enum class BufferUsageFlagBits
  {
    eTransferSrc = VK_BUFFER_USAGE_TRANSFER_SRC_BIT,
    eTransferDst = VK_BUFFER_USAGE_TRANSFER_DST_BIT,
    eUniformTexelBuffer = VK_BUFFER_USAGE_UNIFORM_TEXEL_BUFFER_BIT,
    eStorageTexelBuffer = VK_BUFFER_USAGE_STORAGE_TEXEL_BUFFER_BIT,
    eUniformBuffer = VK_BUFFER_USAGE_UNIFORM_BUFFER_BIT,
    eStorageBuffer = VK_BUFFER_USAGE_STORAGE_BUFFER_BIT,
    eIndexBuffer = VK_BUFFER_USAGE_INDEX_BUFFER_BIT,
    eVertexBuffer = VK_BUFFER_USAGE_VERTEX_BUFFER_BIT,
    eIndirectBuffer = VK_BUFFER_USAGE_INDIRECT_BUFFER_BIT,
    eTransformFeedbackBufferEXT = VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_BUFFER_BIT_EXT,
    eTransformFeedbackCounterBufferEXT = VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_COUNTER_BUFFER_BIT_EXT,
    eConditionalRenderingEXT = VK_BUFFER_USAGE_CONDITIONAL_RENDERING_BIT_EXT,
    eRayTracingNV = VK_BUFFER_USAGE_RAY_TRACING_BIT_NV,
    eShaderDeviceAddressEXT = VK_BUFFER_USAGE_SHADER_DEVICE_ADDRESS_BIT_EXT
  };

  using BufferUsageFlags = Flags<BufferUsageFlagBits, VkBufferUsageFlags>;

  VULKAN_HPP_INLINE BufferUsageFlags operator|( BufferUsageFlagBits bit0, BufferUsageFlagBits bit1 )
  {
    return BufferUsageFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE BufferUsageFlags operator~( BufferUsageFlagBits bits )
  {
    return ~( BufferUsageFlags( bits ) );
  }

  template <> struct FlagTraits<BufferUsageFlagBits>
  {
    enum
    {
      allFlags = VkFlags(BufferUsageFlagBits::eTransferSrc) | VkFlags(BufferUsageFlagBits::eTransferDst) | VkFlags(BufferUsageFlagBits::eUniformTexelBuffer) | VkFlags(BufferUsageFlagBits::eStorageTexelBuffer) | VkFlags(BufferUsageFlagBits::eUniformBuffer) | VkFlags(BufferUsageFlagBits::eStorageBuffer) | VkFlags(BufferUsageFlagBits::eIndexBuffer) | VkFlags(BufferUsageFlagBits::eVertexBuffer) | VkFlags(BufferUsageFlagBits::eIndirectBuffer) | VkFlags(BufferUsageFlagBits::eTransformFeedbackBufferEXT) | VkFlags(BufferUsageFlagBits::eTransformFeedbackCounterBufferEXT) | VkFlags(BufferUsageFlagBits::eConditionalRenderingEXT) | VkFlags(BufferUsageFlagBits::eRayTracingNV) | VkFlags(BufferUsageFlagBits::eShaderDeviceAddressEXT)
    };
  };

  enum class BufferCreateFlagBits
  {
    eSparseBinding = VK_BUFFER_CREATE_SPARSE_BINDING_BIT,
    eSparseResidency = VK_BUFFER_CREATE_SPARSE_RESIDENCY_BIT,
    eSparseAliased = VK_BUFFER_CREATE_SPARSE_ALIASED_BIT,
    eProtected = VK_BUFFER_CREATE_PROTECTED_BIT,
    eDeviceAddressCaptureReplayEXT = VK_BUFFER_CREATE_DEVICE_ADDRESS_CAPTURE_REPLAY_BIT_EXT
  };

  using BufferCreateFlags = Flags<BufferCreateFlagBits, VkBufferCreateFlags>;

  VULKAN_HPP_INLINE BufferCreateFlags operator|( BufferCreateFlagBits bit0, BufferCreateFlagBits bit1 )
  {
    return BufferCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE BufferCreateFlags operator~( BufferCreateFlagBits bits )
  {
    return ~( BufferCreateFlags( bits ) );
  }

  template <> struct FlagTraits<BufferCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(BufferCreateFlagBits::eSparseBinding) | VkFlags(BufferCreateFlagBits::eSparseResidency) | VkFlags(BufferCreateFlagBits::eSparseAliased) | VkFlags(BufferCreateFlagBits::eProtected) | VkFlags(BufferCreateFlagBits::eDeviceAddressCaptureReplayEXT)
    };
  };

  struct BufferCreateInfo
  {
    BufferCreateInfo( BufferCreateFlags flags_ = BufferCreateFlags(),
                      DeviceSize size_ = 0,
                      BufferUsageFlags usage_ = BufferUsageFlags(),
                      SharingMode sharingMode_ = SharingMode::eExclusive,
                      uint32_t queueFamilyIndexCount_ = 0,
                      const uint32_t* pQueueFamilyIndices_ = nullptr )
      : flags( flags_ )
      , size( size_ )
      , usage( usage_ )
      , sharingMode( sharingMode_ )
      , queueFamilyIndexCount( queueFamilyIndexCount_ )
      , pQueueFamilyIndices( pQueueFamilyIndices_ )
    {
    }

    BufferCreateInfo( VkBufferCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferCreateInfo ) );
    }

    BufferCreateInfo& operator=( VkBufferCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferCreateInfo ) );
      return *this;
    }
    BufferCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BufferCreateInfo& setFlags( BufferCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    BufferCreateInfo& setSize( DeviceSize size_ )
    {
      size = size_;
      return *this;
    }

    BufferCreateInfo& setUsage( BufferUsageFlags usage_ )
    {
      usage = usage_;
      return *this;
    }

    BufferCreateInfo& setSharingMode( SharingMode sharingMode_ )
    {
      sharingMode = sharingMode_;
      return *this;
    }

    BufferCreateInfo& setQueueFamilyIndexCount( uint32_t queueFamilyIndexCount_ )
    {
      queueFamilyIndexCount = queueFamilyIndexCount_;
      return *this;
    }

    BufferCreateInfo& setPQueueFamilyIndices( const uint32_t* pQueueFamilyIndices_ )
    {
      pQueueFamilyIndices = pQueueFamilyIndices_;
      return *this;
    }

    operator VkBufferCreateInfo const&() const
    {
      return *reinterpret_cast<const VkBufferCreateInfo*>(this);
    }

    operator VkBufferCreateInfo &()
    {
      return *reinterpret_cast<VkBufferCreateInfo*>(this);
    }

    bool operator==( BufferCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( size == rhs.size )
          && ( usage == rhs.usage )
          && ( sharingMode == rhs.sharingMode )
          && ( queueFamilyIndexCount == rhs.queueFamilyIndexCount )
          && ( pQueueFamilyIndices == rhs.pQueueFamilyIndices );
    }

    bool operator!=( BufferCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBufferCreateInfo;

  public:
    const void* pNext = nullptr;
    BufferCreateFlags flags;
    DeviceSize size;
    BufferUsageFlags usage;
    SharingMode sharingMode;
    uint32_t queueFamilyIndexCount;
    const uint32_t* pQueueFamilyIndices;
  };
  static_assert( sizeof( BufferCreateInfo ) == sizeof( VkBufferCreateInfo ), "struct and wrapper have different size!" );

  enum class ShaderStageFlagBits
  {
    eVertex = VK_SHADER_STAGE_VERTEX_BIT,
    eTessellationControl = VK_SHADER_STAGE_TESSELLATION_CONTROL_BIT,
    eTessellationEvaluation = VK_SHADER_STAGE_TESSELLATION_EVALUATION_BIT,
    eGeometry = VK_SHADER_STAGE_GEOMETRY_BIT,
    eFragment = VK_SHADER_STAGE_FRAGMENT_BIT,
    eCompute = VK_SHADER_STAGE_COMPUTE_BIT,
    eAllGraphics = VK_SHADER_STAGE_ALL_GRAPHICS,
    eAll = VK_SHADER_STAGE_ALL,
    eRaygenNV = VK_SHADER_STAGE_RAYGEN_BIT_NV,
    eAnyHitNV = VK_SHADER_STAGE_ANY_HIT_BIT_NV,
    eClosestHitNV = VK_SHADER_STAGE_CLOSEST_HIT_BIT_NV,
    eMissNV = VK_SHADER_STAGE_MISS_BIT_NV,
    eIntersectionNV = VK_SHADER_STAGE_INTERSECTION_BIT_NV,
    eCallableNV = VK_SHADER_STAGE_CALLABLE_BIT_NV,
    eTaskNV = VK_SHADER_STAGE_TASK_BIT_NV,
    eMeshNV = VK_SHADER_STAGE_MESH_BIT_NV
  };

  using ShaderStageFlags = Flags<ShaderStageFlagBits, VkShaderStageFlags>;

  VULKAN_HPP_INLINE ShaderStageFlags operator|( ShaderStageFlagBits bit0, ShaderStageFlagBits bit1 )
  {
    return ShaderStageFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ShaderStageFlags operator~( ShaderStageFlagBits bits )
  {
    return ~( ShaderStageFlags( bits ) );
  }

  template <> struct FlagTraits<ShaderStageFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ShaderStageFlagBits::eVertex) | VkFlags(ShaderStageFlagBits::eTessellationControl) | VkFlags(ShaderStageFlagBits::eTessellationEvaluation) | VkFlags(ShaderStageFlagBits::eGeometry) | VkFlags(ShaderStageFlagBits::eFragment) | VkFlags(ShaderStageFlagBits::eCompute) | VkFlags(ShaderStageFlagBits::eAllGraphics) | VkFlags(ShaderStageFlagBits::eAll) | VkFlags(ShaderStageFlagBits::eRaygenNV) | VkFlags(ShaderStageFlagBits::eAnyHitNV) | VkFlags(ShaderStageFlagBits::eClosestHitNV) | VkFlags(ShaderStageFlagBits::eMissNV) | VkFlags(ShaderStageFlagBits::eIntersectionNV) | VkFlags(ShaderStageFlagBits::eCallableNV) | VkFlags(ShaderStageFlagBits::eTaskNV) | VkFlags(ShaderStageFlagBits::eMeshNV)
    };
  };

  struct DescriptorSetLayoutBinding
  {
    DescriptorSetLayoutBinding( uint32_t binding_ = 0,
                                DescriptorType descriptorType_ = DescriptorType::eSampler,
                                uint32_t descriptorCount_ = 0,
                                ShaderStageFlags stageFlags_ = ShaderStageFlags(),
                                const Sampler* pImmutableSamplers_ = nullptr )
      : binding( binding_ )
      , descriptorType( descriptorType_ )
      , descriptorCount( descriptorCount_ )
      , stageFlags( stageFlags_ )
      , pImmutableSamplers( pImmutableSamplers_ )
    {
    }

    DescriptorSetLayoutBinding( VkDescriptorSetLayoutBinding const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetLayoutBinding ) );
    }

    DescriptorSetLayoutBinding& operator=( VkDescriptorSetLayoutBinding const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetLayoutBinding ) );
      return *this;
    }
    DescriptorSetLayoutBinding& setBinding( uint32_t binding_ )
    {
      binding = binding_;
      return *this;
    }

    DescriptorSetLayoutBinding& setDescriptorType( DescriptorType descriptorType_ )
    {
      descriptorType = descriptorType_;
      return *this;
    }

    DescriptorSetLayoutBinding& setDescriptorCount( uint32_t descriptorCount_ )
    {
      descriptorCount = descriptorCount_;
      return *this;
    }

    DescriptorSetLayoutBinding& setStageFlags( ShaderStageFlags stageFlags_ )
    {
      stageFlags = stageFlags_;
      return *this;
    }

    DescriptorSetLayoutBinding& setPImmutableSamplers( const Sampler* pImmutableSamplers_ )
    {
      pImmutableSamplers = pImmutableSamplers_;
      return *this;
    }

    operator VkDescriptorSetLayoutBinding const&() const
    {
      return *reinterpret_cast<const VkDescriptorSetLayoutBinding*>(this);
    }

    operator VkDescriptorSetLayoutBinding &()
    {
      return *reinterpret_cast<VkDescriptorSetLayoutBinding*>(this);
    }

    bool operator==( DescriptorSetLayoutBinding const& rhs ) const
    {
      return ( binding == rhs.binding )
          && ( descriptorType == rhs.descriptorType )
          && ( descriptorCount == rhs.descriptorCount )
          && ( stageFlags == rhs.stageFlags )
          && ( pImmutableSamplers == rhs.pImmutableSamplers );
    }

    bool operator!=( DescriptorSetLayoutBinding const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t binding;
    DescriptorType descriptorType;
    uint32_t descriptorCount;
    ShaderStageFlags stageFlags;
    const Sampler* pImmutableSamplers;
  };
  static_assert( sizeof( DescriptorSetLayoutBinding ) == sizeof( VkDescriptorSetLayoutBinding ), "struct and wrapper have different size!" );

  struct PipelineShaderStageCreateInfo
  {
    PipelineShaderStageCreateInfo( PipelineShaderStageCreateFlags flags_ = PipelineShaderStageCreateFlags(),
                                   ShaderStageFlagBits stage_ = ShaderStageFlagBits::eVertex,
                                   ShaderModule module_ = ShaderModule(),
                                   const char* pName_ = nullptr,
                                   const SpecializationInfo* pSpecializationInfo_ = nullptr )
      : flags( flags_ )
      , stage( stage_ )
      , module( module_ )
      , pName( pName_ )
      , pSpecializationInfo( pSpecializationInfo_ )
    {
    }

    PipelineShaderStageCreateInfo( VkPipelineShaderStageCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineShaderStageCreateInfo ) );
    }

    PipelineShaderStageCreateInfo& operator=( VkPipelineShaderStageCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineShaderStageCreateInfo ) );
      return *this;
    }
    PipelineShaderStageCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineShaderStageCreateInfo& setFlags( PipelineShaderStageCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineShaderStageCreateInfo& setStage( ShaderStageFlagBits stage_ )
    {
      stage = stage_;
      return *this;
    }

    PipelineShaderStageCreateInfo& setModule( ShaderModule module_ )
    {
      module = module_;
      return *this;
    }

    PipelineShaderStageCreateInfo& setPName( const char* pName_ )
    {
      pName = pName_;
      return *this;
    }

    PipelineShaderStageCreateInfo& setPSpecializationInfo( const SpecializationInfo* pSpecializationInfo_ )
    {
      pSpecializationInfo = pSpecializationInfo_;
      return *this;
    }

    operator VkPipelineShaderStageCreateInfo const&() const
    {
      return *reinterpret_cast<const VkPipelineShaderStageCreateInfo*>(this);
    }

    operator VkPipelineShaderStageCreateInfo &()
    {
      return *reinterpret_cast<VkPipelineShaderStageCreateInfo*>(this);
    }

    bool operator==( PipelineShaderStageCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( stage == rhs.stage )
          && ( module == rhs.module )
          && ( pName == rhs.pName )
          && ( pSpecializationInfo == rhs.pSpecializationInfo );
    }

    bool operator!=( PipelineShaderStageCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineShaderStageCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineShaderStageCreateFlags flags;
    ShaderStageFlagBits stage;
    ShaderModule module;
    const char* pName;
    const SpecializationInfo* pSpecializationInfo;
  };
  static_assert( sizeof( PipelineShaderStageCreateInfo ) == sizeof( VkPipelineShaderStageCreateInfo ), "struct and wrapper have different size!" );

  struct PushConstantRange
  {
    PushConstantRange( ShaderStageFlags stageFlags_ = ShaderStageFlags(),
                       uint32_t offset_ = 0,
                       uint32_t size_ = 0 )
      : stageFlags( stageFlags_ )
      , offset( offset_ )
      , size( size_ )
    {
    }

    PushConstantRange( VkPushConstantRange const & rhs )
    {
      memcpy( this, &rhs, sizeof( PushConstantRange ) );
    }

    PushConstantRange& operator=( VkPushConstantRange const & rhs )
    {
      memcpy( this, &rhs, sizeof( PushConstantRange ) );
      return *this;
    }
    PushConstantRange& setStageFlags( ShaderStageFlags stageFlags_ )
    {
      stageFlags = stageFlags_;
      return *this;
    }

    PushConstantRange& setOffset( uint32_t offset_ )
    {
      offset = offset_;
      return *this;
    }

    PushConstantRange& setSize( uint32_t size_ )
    {
      size = size_;
      return *this;
    }

    operator VkPushConstantRange const&() const
    {
      return *reinterpret_cast<const VkPushConstantRange*>(this);
    }

    operator VkPushConstantRange &()
    {
      return *reinterpret_cast<VkPushConstantRange*>(this);
    }

    bool operator==( PushConstantRange const& rhs ) const
    {
      return ( stageFlags == rhs.stageFlags )
          && ( offset == rhs.offset )
          && ( size == rhs.size );
    }

    bool operator!=( PushConstantRange const& rhs ) const
    {
      return !operator==( rhs );
    }

    ShaderStageFlags stageFlags;
    uint32_t offset;
    uint32_t size;
  };
  static_assert( sizeof( PushConstantRange ) == sizeof( VkPushConstantRange ), "struct and wrapper have different size!" );

  struct PipelineLayoutCreateInfo
  {
    PipelineLayoutCreateInfo( PipelineLayoutCreateFlags flags_ = PipelineLayoutCreateFlags(),
                              uint32_t setLayoutCount_ = 0,
                              const DescriptorSetLayout* pSetLayouts_ = nullptr,
                              uint32_t pushConstantRangeCount_ = 0,
                              const PushConstantRange* pPushConstantRanges_ = nullptr )
      : flags( flags_ )
      , setLayoutCount( setLayoutCount_ )
      , pSetLayouts( pSetLayouts_ )
      , pushConstantRangeCount( pushConstantRangeCount_ )
      , pPushConstantRanges( pPushConstantRanges_ )
    {
    }

    PipelineLayoutCreateInfo( VkPipelineLayoutCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineLayoutCreateInfo ) );
    }

    PipelineLayoutCreateInfo& operator=( VkPipelineLayoutCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineLayoutCreateInfo ) );
      return *this;
    }
    PipelineLayoutCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineLayoutCreateInfo& setFlags( PipelineLayoutCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineLayoutCreateInfo& setSetLayoutCount( uint32_t setLayoutCount_ )
    {
      setLayoutCount = setLayoutCount_;
      return *this;
    }

    PipelineLayoutCreateInfo& setPSetLayouts( const DescriptorSetLayout* pSetLayouts_ )
    {
      pSetLayouts = pSetLayouts_;
      return *this;
    }

    PipelineLayoutCreateInfo& setPushConstantRangeCount( uint32_t pushConstantRangeCount_ )
    {
      pushConstantRangeCount = pushConstantRangeCount_;
      return *this;
    }

    PipelineLayoutCreateInfo& setPPushConstantRanges( const PushConstantRange* pPushConstantRanges_ )
    {
      pPushConstantRanges = pPushConstantRanges_;
      return *this;
    }

    operator VkPipelineLayoutCreateInfo const&() const
    {
      return *reinterpret_cast<const VkPipelineLayoutCreateInfo*>(this);
    }

    operator VkPipelineLayoutCreateInfo &()
    {
      return *reinterpret_cast<VkPipelineLayoutCreateInfo*>(this);
    }

    bool operator==( PipelineLayoutCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( setLayoutCount == rhs.setLayoutCount )
          && ( pSetLayouts == rhs.pSetLayouts )
          && ( pushConstantRangeCount == rhs.pushConstantRangeCount )
          && ( pPushConstantRanges == rhs.pPushConstantRanges );
    }

    bool operator!=( PipelineLayoutCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineLayoutCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineLayoutCreateFlags flags;
    uint32_t setLayoutCount;
    const DescriptorSetLayout* pSetLayouts;
    uint32_t pushConstantRangeCount;
    const PushConstantRange* pPushConstantRanges;
  };
  static_assert( sizeof( PipelineLayoutCreateInfo ) == sizeof( VkPipelineLayoutCreateInfo ), "struct and wrapper have different size!" );

  struct ShaderStatisticsInfoAMD
  {
    operator VkShaderStatisticsInfoAMD const&() const
    {
      return *reinterpret_cast<const VkShaderStatisticsInfoAMD*>(this);
    }

    operator VkShaderStatisticsInfoAMD &()
    {
      return *reinterpret_cast<VkShaderStatisticsInfoAMD*>(this);
    }

    bool operator==( ShaderStatisticsInfoAMD const& rhs ) const
    {
      return ( shaderStageMask == rhs.shaderStageMask )
          && ( resourceUsage == rhs.resourceUsage )
          && ( numPhysicalVgprs == rhs.numPhysicalVgprs )
          && ( numPhysicalSgprs == rhs.numPhysicalSgprs )
          && ( numAvailableVgprs == rhs.numAvailableVgprs )
          && ( numAvailableSgprs == rhs.numAvailableSgprs )
          && ( memcmp( computeWorkGroupSize, rhs.computeWorkGroupSize, 3 * sizeof( uint32_t ) ) == 0 );
    }

    bool operator!=( ShaderStatisticsInfoAMD const& rhs ) const
    {
      return !operator==( rhs );
    }

    ShaderStageFlags shaderStageMask;
    ShaderResourceUsageAMD resourceUsage;
    uint32_t numPhysicalVgprs;
    uint32_t numPhysicalSgprs;
    uint32_t numAvailableVgprs;
    uint32_t numAvailableSgprs;
    uint32_t computeWorkGroupSize[3];
  };
  static_assert( sizeof( ShaderStatisticsInfoAMD ) == sizeof( VkShaderStatisticsInfoAMD ), "struct and wrapper have different size!" );

  enum class ImageUsageFlagBits
  {
    eTransferSrc = VK_IMAGE_USAGE_TRANSFER_SRC_BIT,
    eTransferDst = VK_IMAGE_USAGE_TRANSFER_DST_BIT,
    eSampled = VK_IMAGE_USAGE_SAMPLED_BIT,
    eStorage = VK_IMAGE_USAGE_STORAGE_BIT,
    eColorAttachment = VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT,
    eDepthStencilAttachment = VK_IMAGE_USAGE_DEPTH_STENCIL_ATTACHMENT_BIT,
    eTransientAttachment = VK_IMAGE_USAGE_TRANSIENT_ATTACHMENT_BIT,
    eInputAttachment = VK_IMAGE_USAGE_INPUT_ATTACHMENT_BIT,
    eShadingRateImageNV = VK_IMAGE_USAGE_SHADING_RATE_IMAGE_BIT_NV,
    eFragmentDensityMapEXT = VK_IMAGE_USAGE_FRAGMENT_DENSITY_MAP_BIT_EXT
  };

  using ImageUsageFlags = Flags<ImageUsageFlagBits, VkImageUsageFlags>;

  VULKAN_HPP_INLINE ImageUsageFlags operator|( ImageUsageFlagBits bit0, ImageUsageFlagBits bit1 )
  {
    return ImageUsageFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ImageUsageFlags operator~( ImageUsageFlagBits bits )
  {
    return ~( ImageUsageFlags( bits ) );
  }

  template <> struct FlagTraits<ImageUsageFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ImageUsageFlagBits::eTransferSrc) | VkFlags(ImageUsageFlagBits::eTransferDst) | VkFlags(ImageUsageFlagBits::eSampled) | VkFlags(ImageUsageFlagBits::eStorage) | VkFlags(ImageUsageFlagBits::eColorAttachment) | VkFlags(ImageUsageFlagBits::eDepthStencilAttachment) | VkFlags(ImageUsageFlagBits::eTransientAttachment) | VkFlags(ImageUsageFlagBits::eInputAttachment) | VkFlags(ImageUsageFlagBits::eShadingRateImageNV) | VkFlags(ImageUsageFlagBits::eFragmentDensityMapEXT)
    };
  };

  struct SharedPresentSurfaceCapabilitiesKHR
  {
    operator VkSharedPresentSurfaceCapabilitiesKHR const&() const
    {
      return *reinterpret_cast<const VkSharedPresentSurfaceCapabilitiesKHR*>(this);
    }

    operator VkSharedPresentSurfaceCapabilitiesKHR &()
    {
      return *reinterpret_cast<VkSharedPresentSurfaceCapabilitiesKHR*>(this);
    }

    bool operator==( SharedPresentSurfaceCapabilitiesKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( sharedPresentSupportedUsageFlags == rhs.sharedPresentSupportedUsageFlags );
    }

    bool operator!=( SharedPresentSurfaceCapabilitiesKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eSharedPresentSurfaceCapabilitiesKHR;

  public:
    void* pNext = nullptr;
    ImageUsageFlags sharedPresentSupportedUsageFlags;
  };
  static_assert( sizeof( SharedPresentSurfaceCapabilitiesKHR ) == sizeof( VkSharedPresentSurfaceCapabilitiesKHR ), "struct and wrapper have different size!" );

  struct ImageViewUsageCreateInfo
  {
    ImageViewUsageCreateInfo( ImageUsageFlags usage_ = ImageUsageFlags() )
      : usage( usage_ )
    {
    }

    ImageViewUsageCreateInfo( VkImageViewUsageCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageViewUsageCreateInfo ) );
    }

    ImageViewUsageCreateInfo& operator=( VkImageViewUsageCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageViewUsageCreateInfo ) );
      return *this;
    }
    ImageViewUsageCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImageViewUsageCreateInfo& setUsage( ImageUsageFlags usage_ )
    {
      usage = usage_;
      return *this;
    }

    operator VkImageViewUsageCreateInfo const&() const
    {
      return *reinterpret_cast<const VkImageViewUsageCreateInfo*>(this);
    }

    operator VkImageViewUsageCreateInfo &()
    {
      return *reinterpret_cast<VkImageViewUsageCreateInfo*>(this);
    }

    bool operator==( ImageViewUsageCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( usage == rhs.usage );
    }

    bool operator!=( ImageViewUsageCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageViewUsageCreateInfo;

  public:
    const void* pNext = nullptr;
    ImageUsageFlags usage;
  };
  static_assert( sizeof( ImageViewUsageCreateInfo ) == sizeof( VkImageViewUsageCreateInfo ), "struct and wrapper have different size!" );

  using ImageViewUsageCreateInfoKHR = ImageViewUsageCreateInfo;

  struct ImageStencilUsageCreateInfoEXT
  {
    ImageStencilUsageCreateInfoEXT( ImageUsageFlags stencilUsage_ = ImageUsageFlags() )
      : stencilUsage( stencilUsage_ )
    {
    }

    ImageStencilUsageCreateInfoEXT( VkImageStencilUsageCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageStencilUsageCreateInfoEXT ) );
    }

    ImageStencilUsageCreateInfoEXT& operator=( VkImageStencilUsageCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageStencilUsageCreateInfoEXT ) );
      return *this;
    }
    ImageStencilUsageCreateInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImageStencilUsageCreateInfoEXT& setStencilUsage( ImageUsageFlags stencilUsage_ )
    {
      stencilUsage = stencilUsage_;
      return *this;
    }

    operator VkImageStencilUsageCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkImageStencilUsageCreateInfoEXT*>(this);
    }

    operator VkImageStencilUsageCreateInfoEXT &()
    {
      return *reinterpret_cast<VkImageStencilUsageCreateInfoEXT*>(this);
    }

    bool operator==( ImageStencilUsageCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( stencilUsage == rhs.stencilUsage );
    }

    bool operator!=( ImageStencilUsageCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageStencilUsageCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    ImageUsageFlags stencilUsage;
  };
  static_assert( sizeof( ImageStencilUsageCreateInfoEXT ) == sizeof( VkImageStencilUsageCreateInfoEXT ), "struct and wrapper have different size!" );

  enum class ImageCreateFlagBits
  {
    eSparseBinding = VK_IMAGE_CREATE_SPARSE_BINDING_BIT,
    eSparseResidency = VK_IMAGE_CREATE_SPARSE_RESIDENCY_BIT,
    eSparseAliased = VK_IMAGE_CREATE_SPARSE_ALIASED_BIT,
    eMutableFormat = VK_IMAGE_CREATE_MUTABLE_FORMAT_BIT,
    eCubeCompatible = VK_IMAGE_CREATE_CUBE_COMPATIBLE_BIT,
    eAlias = VK_IMAGE_CREATE_ALIAS_BIT,
    eAliasKHR = VK_IMAGE_CREATE_ALIAS_BIT,
    eSplitInstanceBindRegions = VK_IMAGE_CREATE_SPLIT_INSTANCE_BIND_REGIONS_BIT,
    eSplitInstanceBindRegionsKHR = VK_IMAGE_CREATE_SPLIT_INSTANCE_BIND_REGIONS_BIT,
    e2DArrayCompatible = VK_IMAGE_CREATE_2D_ARRAY_COMPATIBLE_BIT,
    e2DArrayCompatibleKHR = VK_IMAGE_CREATE_2D_ARRAY_COMPATIBLE_BIT,
    eBlockTexelViewCompatible = VK_IMAGE_CREATE_BLOCK_TEXEL_VIEW_COMPATIBLE_BIT,
    eBlockTexelViewCompatibleKHR = VK_IMAGE_CREATE_BLOCK_TEXEL_VIEW_COMPATIBLE_BIT,
    eExtendedUsage = VK_IMAGE_CREATE_EXTENDED_USAGE_BIT,
    eExtendedUsageKHR = VK_IMAGE_CREATE_EXTENDED_USAGE_BIT,
    eProtected = VK_IMAGE_CREATE_PROTECTED_BIT,
    eDisjoint = VK_IMAGE_CREATE_DISJOINT_BIT,
    eDisjointKHR = VK_IMAGE_CREATE_DISJOINT_BIT,
    eCornerSampledNV = VK_IMAGE_CREATE_CORNER_SAMPLED_BIT_NV,
    eSampleLocationsCompatibleDepthEXT = VK_IMAGE_CREATE_SAMPLE_LOCATIONS_COMPATIBLE_DEPTH_BIT_EXT,
    eSubsampledEXT = VK_IMAGE_CREATE_SUBSAMPLED_BIT_EXT
  };

  using ImageCreateFlags = Flags<ImageCreateFlagBits, VkImageCreateFlags>;

  VULKAN_HPP_INLINE ImageCreateFlags operator|( ImageCreateFlagBits bit0, ImageCreateFlagBits bit1 )
  {
    return ImageCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ImageCreateFlags operator~( ImageCreateFlagBits bits )
  {
    return ~( ImageCreateFlags( bits ) );
  }

  template <> struct FlagTraits<ImageCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ImageCreateFlagBits::eSparseBinding) | VkFlags(ImageCreateFlagBits::eSparseResidency) | VkFlags(ImageCreateFlagBits::eSparseAliased) | VkFlags(ImageCreateFlagBits::eMutableFormat) | VkFlags(ImageCreateFlagBits::eCubeCompatible) | VkFlags(ImageCreateFlagBits::eAlias) | VkFlags(ImageCreateFlagBits::eSplitInstanceBindRegions) | VkFlags(ImageCreateFlagBits::e2DArrayCompatible) | VkFlags(ImageCreateFlagBits::eBlockTexelViewCompatible) | VkFlags(ImageCreateFlagBits::eExtendedUsage) | VkFlags(ImageCreateFlagBits::eProtected) | VkFlags(ImageCreateFlagBits::eDisjoint) | VkFlags(ImageCreateFlagBits::eCornerSampledNV) | VkFlags(ImageCreateFlagBits::eSampleLocationsCompatibleDepthEXT) | VkFlags(ImageCreateFlagBits::eSubsampledEXT)
    };
  };

  struct PhysicalDeviceImageFormatInfo2
  {
    PhysicalDeviceImageFormatInfo2( Format format_ = Format::eUndefined,
                                    ImageType type_ = ImageType::e1D,
                                    ImageTiling tiling_ = ImageTiling::eOptimal,
                                    ImageUsageFlags usage_ = ImageUsageFlags(),
                                    ImageCreateFlags flags_ = ImageCreateFlags() )
      : format( format_ )
      , type( type_ )
      , tiling( tiling_ )
      , usage( usage_ )
      , flags( flags_ )
    {
    }

    PhysicalDeviceImageFormatInfo2( VkPhysicalDeviceImageFormatInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceImageFormatInfo2 ) );
    }

    PhysicalDeviceImageFormatInfo2& operator=( VkPhysicalDeviceImageFormatInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceImageFormatInfo2 ) );
      return *this;
    }
    PhysicalDeviceImageFormatInfo2& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceImageFormatInfo2& setFormat( Format format_ )
    {
      format = format_;
      return *this;
    }

    PhysicalDeviceImageFormatInfo2& setType( ImageType type_ )
    {
      type = type_;
      return *this;
    }

    PhysicalDeviceImageFormatInfo2& setTiling( ImageTiling tiling_ )
    {
      tiling = tiling_;
      return *this;
    }

    PhysicalDeviceImageFormatInfo2& setUsage( ImageUsageFlags usage_ )
    {
      usage = usage_;
      return *this;
    }

    PhysicalDeviceImageFormatInfo2& setFlags( ImageCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    operator VkPhysicalDeviceImageFormatInfo2 const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceImageFormatInfo2*>(this);
    }

    operator VkPhysicalDeviceImageFormatInfo2 &()
    {
      return *reinterpret_cast<VkPhysicalDeviceImageFormatInfo2*>(this);
    }

    bool operator==( PhysicalDeviceImageFormatInfo2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( format == rhs.format )
          && ( type == rhs.type )
          && ( tiling == rhs.tiling )
          && ( usage == rhs.usage )
          && ( flags == rhs.flags );
    }

    bool operator!=( PhysicalDeviceImageFormatInfo2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceImageFormatInfo2;

  public:
    const void* pNext = nullptr;
    Format format;
    ImageType type;
    ImageTiling tiling;
    ImageUsageFlags usage;
    ImageCreateFlags flags;
  };
  static_assert( sizeof( PhysicalDeviceImageFormatInfo2 ) == sizeof( VkPhysicalDeviceImageFormatInfo2 ), "struct and wrapper have different size!" );

  using PhysicalDeviceImageFormatInfo2KHR = PhysicalDeviceImageFormatInfo2;

  enum class ImageViewCreateFlagBits
  {
    eFragmentDensityMapDynamicEXT = VK_IMAGE_VIEW_CREATE_FRAGMENT_DENSITY_MAP_DYNAMIC_BIT_EXT
  };

  using ImageViewCreateFlags = Flags<ImageViewCreateFlagBits, VkImageViewCreateFlags>;

  VULKAN_HPP_INLINE ImageViewCreateFlags operator|( ImageViewCreateFlagBits bit0, ImageViewCreateFlagBits bit1 )
  {
    return ImageViewCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ImageViewCreateFlags operator~( ImageViewCreateFlagBits bits )
  {
    return ~( ImageViewCreateFlags( bits ) );
  }

  template <> struct FlagTraits<ImageViewCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ImageViewCreateFlagBits::eFragmentDensityMapDynamicEXT)
    };
  };

  enum class SamplerCreateFlagBits
  {
    eSubsampledEXT = VK_SAMPLER_CREATE_SUBSAMPLED_BIT_EXT,
    eSubsampledCoarseReconstructionEXT = VK_SAMPLER_CREATE_SUBSAMPLED_COARSE_RECONSTRUCTION_BIT_EXT
  };

  using SamplerCreateFlags = Flags<SamplerCreateFlagBits, VkSamplerCreateFlags>;

  VULKAN_HPP_INLINE SamplerCreateFlags operator|( SamplerCreateFlagBits bit0, SamplerCreateFlagBits bit1 )
  {
    return SamplerCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SamplerCreateFlags operator~( SamplerCreateFlagBits bits )
  {
    return ~( SamplerCreateFlags( bits ) );
  }

  template <> struct FlagTraits<SamplerCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(SamplerCreateFlagBits::eSubsampledEXT) | VkFlags(SamplerCreateFlagBits::eSubsampledCoarseReconstructionEXT)
    };
  };

  struct SamplerCreateInfo
  {
    SamplerCreateInfo( SamplerCreateFlags flags_ = SamplerCreateFlags(),
                       Filter magFilter_ = Filter::eNearest,
                       Filter minFilter_ = Filter::eNearest,
                       SamplerMipmapMode mipmapMode_ = SamplerMipmapMode::eNearest,
                       SamplerAddressMode addressModeU_ = SamplerAddressMode::eRepeat,
                       SamplerAddressMode addressModeV_ = SamplerAddressMode::eRepeat,
                       SamplerAddressMode addressModeW_ = SamplerAddressMode::eRepeat,
                       float mipLodBias_ = 0,
                       Bool32 anisotropyEnable_ = 0,
                       float maxAnisotropy_ = 0,
                       Bool32 compareEnable_ = 0,
                       CompareOp compareOp_ = CompareOp::eNever,
                       float minLod_ = 0,
                       float maxLod_ = 0,
                       BorderColor borderColor_ = BorderColor::eFloatTransparentBlack,
                       Bool32 unnormalizedCoordinates_ = 0 )
      : flags( flags_ )
      , magFilter( magFilter_ )
      , minFilter( minFilter_ )
      , mipmapMode( mipmapMode_ )
      , addressModeU( addressModeU_ )
      , addressModeV( addressModeV_ )
      , addressModeW( addressModeW_ )
      , mipLodBias( mipLodBias_ )
      , anisotropyEnable( anisotropyEnable_ )
      , maxAnisotropy( maxAnisotropy_ )
      , compareEnable( compareEnable_ )
      , compareOp( compareOp_ )
      , minLod( minLod_ )
      , maxLod( maxLod_ )
      , borderColor( borderColor_ )
      , unnormalizedCoordinates( unnormalizedCoordinates_ )
    {
    }

    SamplerCreateInfo( VkSamplerCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SamplerCreateInfo ) );
    }

    SamplerCreateInfo& operator=( VkSamplerCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SamplerCreateInfo ) );
      return *this;
    }
    SamplerCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    SamplerCreateInfo& setFlags( SamplerCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    SamplerCreateInfo& setMagFilter( Filter magFilter_ )
    {
      magFilter = magFilter_;
      return *this;
    }

    SamplerCreateInfo& setMinFilter( Filter minFilter_ )
    {
      minFilter = minFilter_;
      return *this;
    }

    SamplerCreateInfo& setMipmapMode( SamplerMipmapMode mipmapMode_ )
    {
      mipmapMode = mipmapMode_;
      return *this;
    }

    SamplerCreateInfo& setAddressModeU( SamplerAddressMode addressModeU_ )
    {
      addressModeU = addressModeU_;
      return *this;
    }

    SamplerCreateInfo& setAddressModeV( SamplerAddressMode addressModeV_ )
    {
      addressModeV = addressModeV_;
      return *this;
    }

    SamplerCreateInfo& setAddressModeW( SamplerAddressMode addressModeW_ )
    {
      addressModeW = addressModeW_;
      return *this;
    }

    SamplerCreateInfo& setMipLodBias( float mipLodBias_ )
    {
      mipLodBias = mipLodBias_;
      return *this;
    }

    SamplerCreateInfo& setAnisotropyEnable( Bool32 anisotropyEnable_ )
    {
      anisotropyEnable = anisotropyEnable_;
      return *this;
    }

    SamplerCreateInfo& setMaxAnisotropy( float maxAnisotropy_ )
    {
      maxAnisotropy = maxAnisotropy_;
      return *this;
    }

    SamplerCreateInfo& setCompareEnable( Bool32 compareEnable_ )
    {
      compareEnable = compareEnable_;
      return *this;
    }

    SamplerCreateInfo& setCompareOp( CompareOp compareOp_ )
    {
      compareOp = compareOp_;
      return *this;
    }

    SamplerCreateInfo& setMinLod( float minLod_ )
    {
      minLod = minLod_;
      return *this;
    }

    SamplerCreateInfo& setMaxLod( float maxLod_ )
    {
      maxLod = maxLod_;
      return *this;
    }

    SamplerCreateInfo& setBorderColor( BorderColor borderColor_ )
    {
      borderColor = borderColor_;
      return *this;
    }

    SamplerCreateInfo& setUnnormalizedCoordinates( Bool32 unnormalizedCoordinates_ )
    {
      unnormalizedCoordinates = unnormalizedCoordinates_;
      return *this;
    }

    operator VkSamplerCreateInfo const&() const
    {
      return *reinterpret_cast<const VkSamplerCreateInfo*>(this);
    }

    operator VkSamplerCreateInfo &()
    {
      return *reinterpret_cast<VkSamplerCreateInfo*>(this);
    }

    bool operator==( SamplerCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( magFilter == rhs.magFilter )
          && ( minFilter == rhs.minFilter )
          && ( mipmapMode == rhs.mipmapMode )
          && ( addressModeU == rhs.addressModeU )
          && ( addressModeV == rhs.addressModeV )
          && ( addressModeW == rhs.addressModeW )
          && ( mipLodBias == rhs.mipLodBias )
          && ( anisotropyEnable == rhs.anisotropyEnable )
          && ( maxAnisotropy == rhs.maxAnisotropy )
          && ( compareEnable == rhs.compareEnable )
          && ( compareOp == rhs.compareOp )
          && ( minLod == rhs.minLod )
          && ( maxLod == rhs.maxLod )
          && ( borderColor == rhs.borderColor )
          && ( unnormalizedCoordinates == rhs.unnormalizedCoordinates );
    }

    bool operator!=( SamplerCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eSamplerCreateInfo;

  public:
    const void* pNext = nullptr;
    SamplerCreateFlags flags;
    Filter magFilter;
    Filter minFilter;
    SamplerMipmapMode mipmapMode;
    SamplerAddressMode addressModeU;
    SamplerAddressMode addressModeV;
    SamplerAddressMode addressModeW;
    float mipLodBias;
    Bool32 anisotropyEnable;
    float maxAnisotropy;
    Bool32 compareEnable;
    CompareOp compareOp;
    float minLod;
    float maxLod;
    BorderColor borderColor;
    Bool32 unnormalizedCoordinates;
  };
  static_assert( sizeof( SamplerCreateInfo ) == sizeof( VkSamplerCreateInfo ), "struct and wrapper have different size!" );

  enum class PipelineCreateFlagBits
  {
    eDisableOptimization = VK_PIPELINE_CREATE_DISABLE_OPTIMIZATION_BIT,
    eAllowDerivatives = VK_PIPELINE_CREATE_ALLOW_DERIVATIVES_BIT,
    eDerivative = VK_PIPELINE_CREATE_DERIVATIVE_BIT,
    eViewIndexFromDeviceIndex = VK_PIPELINE_CREATE_VIEW_INDEX_FROM_DEVICE_INDEX_BIT,
    eViewIndexFromDeviceIndexKHR = VK_PIPELINE_CREATE_VIEW_INDEX_FROM_DEVICE_INDEX_BIT,
    eDispatchBase = VK_PIPELINE_CREATE_DISPATCH_BASE,
    eDispatchBaseKHR = VK_PIPELINE_CREATE_DISPATCH_BASE,
    eDeferCompileNV = VK_PIPELINE_CREATE_DEFER_COMPILE_BIT_NV
  };

  using PipelineCreateFlags = Flags<PipelineCreateFlagBits, VkPipelineCreateFlags>;

  VULKAN_HPP_INLINE PipelineCreateFlags operator|( PipelineCreateFlagBits bit0, PipelineCreateFlagBits bit1 )
  {
    return PipelineCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE PipelineCreateFlags operator~( PipelineCreateFlagBits bits )
  {
    return ~( PipelineCreateFlags( bits ) );
  }

  template <> struct FlagTraits<PipelineCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(PipelineCreateFlagBits::eDisableOptimization) | VkFlags(PipelineCreateFlagBits::eAllowDerivatives) | VkFlags(PipelineCreateFlagBits::eDerivative) | VkFlags(PipelineCreateFlagBits::eViewIndexFromDeviceIndex) | VkFlags(PipelineCreateFlagBits::eDispatchBase) | VkFlags(PipelineCreateFlagBits::eDeferCompileNV)
    };
  };

  struct ComputePipelineCreateInfo
  {
    ComputePipelineCreateInfo( PipelineCreateFlags flags_ = PipelineCreateFlags(),
                               PipelineShaderStageCreateInfo stage_ = PipelineShaderStageCreateInfo(),
                               PipelineLayout layout_ = PipelineLayout(),
                               Pipeline basePipelineHandle_ = Pipeline(),
                               int32_t basePipelineIndex_ = 0 )
      : flags( flags_ )
      , stage( stage_ )
      , layout( layout_ )
      , basePipelineHandle( basePipelineHandle_ )
      , basePipelineIndex( basePipelineIndex_ )
    {
    }

    ComputePipelineCreateInfo( VkComputePipelineCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ComputePipelineCreateInfo ) );
    }

    ComputePipelineCreateInfo& operator=( VkComputePipelineCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ComputePipelineCreateInfo ) );
      return *this;
    }
    ComputePipelineCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ComputePipelineCreateInfo& setFlags( PipelineCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    ComputePipelineCreateInfo& setStage( PipelineShaderStageCreateInfo stage_ )
    {
      stage = stage_;
      return *this;
    }

    ComputePipelineCreateInfo& setLayout( PipelineLayout layout_ )
    {
      layout = layout_;
      return *this;
    }

    ComputePipelineCreateInfo& setBasePipelineHandle( Pipeline basePipelineHandle_ )
    {
      basePipelineHandle = basePipelineHandle_;
      return *this;
    }

    ComputePipelineCreateInfo& setBasePipelineIndex( int32_t basePipelineIndex_ )
    {
      basePipelineIndex = basePipelineIndex_;
      return *this;
    }

    operator VkComputePipelineCreateInfo const&() const
    {
      return *reinterpret_cast<const VkComputePipelineCreateInfo*>(this);
    }

    operator VkComputePipelineCreateInfo &()
    {
      return *reinterpret_cast<VkComputePipelineCreateInfo*>(this);
    }

    bool operator==( ComputePipelineCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( stage == rhs.stage )
          && ( layout == rhs.layout )
          && ( basePipelineHandle == rhs.basePipelineHandle )
          && ( basePipelineIndex == rhs.basePipelineIndex );
    }

    bool operator!=( ComputePipelineCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eComputePipelineCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineCreateFlags flags;
    PipelineShaderStageCreateInfo stage;
    PipelineLayout layout;
    Pipeline basePipelineHandle;
    int32_t basePipelineIndex;
  };
  static_assert( sizeof( ComputePipelineCreateInfo ) == sizeof( VkComputePipelineCreateInfo ), "struct and wrapper have different size!" );

  enum class ColorComponentFlagBits
  {
    eR = VK_COLOR_COMPONENT_R_BIT,
    eG = VK_COLOR_COMPONENT_G_BIT,
    eB = VK_COLOR_COMPONENT_B_BIT,
    eA = VK_COLOR_COMPONENT_A_BIT
  };

  using ColorComponentFlags = Flags<ColorComponentFlagBits, VkColorComponentFlags>;

  VULKAN_HPP_INLINE ColorComponentFlags operator|( ColorComponentFlagBits bit0, ColorComponentFlagBits bit1 )
  {
    return ColorComponentFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ColorComponentFlags operator~( ColorComponentFlagBits bits )
  {
    return ~( ColorComponentFlags( bits ) );
  }

  template <> struct FlagTraits<ColorComponentFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ColorComponentFlagBits::eR) | VkFlags(ColorComponentFlagBits::eG) | VkFlags(ColorComponentFlagBits::eB) | VkFlags(ColorComponentFlagBits::eA)
    };
  };

  struct PipelineColorBlendAttachmentState
  {
    PipelineColorBlendAttachmentState( Bool32 blendEnable_ = 0,
                                       BlendFactor srcColorBlendFactor_ = BlendFactor::eZero,
                                       BlendFactor dstColorBlendFactor_ = BlendFactor::eZero,
                                       BlendOp colorBlendOp_ = BlendOp::eAdd,
                                       BlendFactor srcAlphaBlendFactor_ = BlendFactor::eZero,
                                       BlendFactor dstAlphaBlendFactor_ = BlendFactor::eZero,
                                       BlendOp alphaBlendOp_ = BlendOp::eAdd,
                                       ColorComponentFlags colorWriteMask_ = ColorComponentFlags() )
      : blendEnable( blendEnable_ )
      , srcColorBlendFactor( srcColorBlendFactor_ )
      , dstColorBlendFactor( dstColorBlendFactor_ )
      , colorBlendOp( colorBlendOp_ )
      , srcAlphaBlendFactor( srcAlphaBlendFactor_ )
      , dstAlphaBlendFactor( dstAlphaBlendFactor_ )
      , alphaBlendOp( alphaBlendOp_ )
      , colorWriteMask( colorWriteMask_ )
    {
    }

    PipelineColorBlendAttachmentState( VkPipelineColorBlendAttachmentState const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineColorBlendAttachmentState ) );
    }

    PipelineColorBlendAttachmentState& operator=( VkPipelineColorBlendAttachmentState const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineColorBlendAttachmentState ) );
      return *this;
    }
    PipelineColorBlendAttachmentState& setBlendEnable( Bool32 blendEnable_ )
    {
      blendEnable = blendEnable_;
      return *this;
    }

    PipelineColorBlendAttachmentState& setSrcColorBlendFactor( BlendFactor srcColorBlendFactor_ )
    {
      srcColorBlendFactor = srcColorBlendFactor_;
      return *this;
    }

    PipelineColorBlendAttachmentState& setDstColorBlendFactor( BlendFactor dstColorBlendFactor_ )
    {
      dstColorBlendFactor = dstColorBlendFactor_;
      return *this;
    }

    PipelineColorBlendAttachmentState& setColorBlendOp( BlendOp colorBlendOp_ )
    {
      colorBlendOp = colorBlendOp_;
      return *this;
    }

    PipelineColorBlendAttachmentState& setSrcAlphaBlendFactor( BlendFactor srcAlphaBlendFactor_ )
    {
      srcAlphaBlendFactor = srcAlphaBlendFactor_;
      return *this;
    }

    PipelineColorBlendAttachmentState& setDstAlphaBlendFactor( BlendFactor dstAlphaBlendFactor_ )
    {
      dstAlphaBlendFactor = dstAlphaBlendFactor_;
      return *this;
    }

    PipelineColorBlendAttachmentState& setAlphaBlendOp( BlendOp alphaBlendOp_ )
    {
      alphaBlendOp = alphaBlendOp_;
      return *this;
    }

    PipelineColorBlendAttachmentState& setColorWriteMask( ColorComponentFlags colorWriteMask_ )
    {
      colorWriteMask = colorWriteMask_;
      return *this;
    }

    operator VkPipelineColorBlendAttachmentState const&() const
    {
      return *reinterpret_cast<const VkPipelineColorBlendAttachmentState*>(this);
    }

    operator VkPipelineColorBlendAttachmentState &()
    {
      return *reinterpret_cast<VkPipelineColorBlendAttachmentState*>(this);
    }

    bool operator==( PipelineColorBlendAttachmentState const& rhs ) const
    {
      return ( blendEnable == rhs.blendEnable )
          && ( srcColorBlendFactor == rhs.srcColorBlendFactor )
          && ( dstColorBlendFactor == rhs.dstColorBlendFactor )
          && ( colorBlendOp == rhs.colorBlendOp )
          && ( srcAlphaBlendFactor == rhs.srcAlphaBlendFactor )
          && ( dstAlphaBlendFactor == rhs.dstAlphaBlendFactor )
          && ( alphaBlendOp == rhs.alphaBlendOp )
          && ( colorWriteMask == rhs.colorWriteMask );
    }

    bool operator!=( PipelineColorBlendAttachmentState const& rhs ) const
    {
      return !operator==( rhs );
    }

    Bool32 blendEnable;
    BlendFactor srcColorBlendFactor;
    BlendFactor dstColorBlendFactor;
    BlendOp colorBlendOp;
    BlendFactor srcAlphaBlendFactor;
    BlendFactor dstAlphaBlendFactor;
    BlendOp alphaBlendOp;
    ColorComponentFlags colorWriteMask;
  };
  static_assert( sizeof( PipelineColorBlendAttachmentState ) == sizeof( VkPipelineColorBlendAttachmentState ), "struct and wrapper have different size!" );

  struct PipelineColorBlendStateCreateInfo
  {
    PipelineColorBlendStateCreateInfo( PipelineColorBlendStateCreateFlags flags_ = PipelineColorBlendStateCreateFlags(),
                                       Bool32 logicOpEnable_ = 0,
                                       LogicOp logicOp_ = LogicOp::eClear,
                                       uint32_t attachmentCount_ = 0,
                                       const PipelineColorBlendAttachmentState* pAttachments_ = nullptr,
                                       std::array<float,4> const& blendConstants_ = { { 0, 0, 0, 0 } } )
      : flags( flags_ )
      , logicOpEnable( logicOpEnable_ )
      , logicOp( logicOp_ )
      , attachmentCount( attachmentCount_ )
      , pAttachments( pAttachments_ )
    {
      memcpy( &blendConstants, blendConstants_.data(), 4 * sizeof( float ) );
    }

    PipelineColorBlendStateCreateInfo( VkPipelineColorBlendStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineColorBlendStateCreateInfo ) );
    }

    PipelineColorBlendStateCreateInfo& operator=( VkPipelineColorBlendStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineColorBlendStateCreateInfo ) );
      return *this;
    }
    PipelineColorBlendStateCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineColorBlendStateCreateInfo& setFlags( PipelineColorBlendStateCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineColorBlendStateCreateInfo& setLogicOpEnable( Bool32 logicOpEnable_ )
    {
      logicOpEnable = logicOpEnable_;
      return *this;
    }

    PipelineColorBlendStateCreateInfo& setLogicOp( LogicOp logicOp_ )
    {
      logicOp = logicOp_;
      return *this;
    }

    PipelineColorBlendStateCreateInfo& setAttachmentCount( uint32_t attachmentCount_ )
    {
      attachmentCount = attachmentCount_;
      return *this;
    }

    PipelineColorBlendStateCreateInfo& setPAttachments( const PipelineColorBlendAttachmentState* pAttachments_ )
    {
      pAttachments = pAttachments_;
      return *this;
    }

    PipelineColorBlendStateCreateInfo& setBlendConstants( std::array<float,4> blendConstants_ )
    {
      memcpy( &blendConstants, blendConstants_.data(), 4 * sizeof( float ) );
      return *this;
    }

    operator VkPipelineColorBlendStateCreateInfo const&() const
    {
      return *reinterpret_cast<const VkPipelineColorBlendStateCreateInfo*>(this);
    }

    operator VkPipelineColorBlendStateCreateInfo &()
    {
      return *reinterpret_cast<VkPipelineColorBlendStateCreateInfo*>(this);
    }

    bool operator==( PipelineColorBlendStateCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( logicOpEnable == rhs.logicOpEnable )
          && ( logicOp == rhs.logicOp )
          && ( attachmentCount == rhs.attachmentCount )
          && ( pAttachments == rhs.pAttachments )
          && ( memcmp( blendConstants, rhs.blendConstants, 4 * sizeof( float ) ) == 0 );
    }

    bool operator!=( PipelineColorBlendStateCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineColorBlendStateCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineColorBlendStateCreateFlags flags;
    Bool32 logicOpEnable;
    LogicOp logicOp;
    uint32_t attachmentCount;
    const PipelineColorBlendAttachmentState* pAttachments;
    float blendConstants[4];
  };
  static_assert( sizeof( PipelineColorBlendStateCreateInfo ) == sizeof( VkPipelineColorBlendStateCreateInfo ), "struct and wrapper have different size!" );

  enum class FenceCreateFlagBits
  {
    eSignaled = VK_FENCE_CREATE_SIGNALED_BIT
  };

  using FenceCreateFlags = Flags<FenceCreateFlagBits, VkFenceCreateFlags>;

  VULKAN_HPP_INLINE FenceCreateFlags operator|( FenceCreateFlagBits bit0, FenceCreateFlagBits bit1 )
  {
    return FenceCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE FenceCreateFlags operator~( FenceCreateFlagBits bits )
  {
    return ~( FenceCreateFlags( bits ) );
  }

  template <> struct FlagTraits<FenceCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(FenceCreateFlagBits::eSignaled)
    };
  };

  struct FenceCreateInfo
  {
    FenceCreateInfo( FenceCreateFlags flags_ = FenceCreateFlags() )
      : flags( flags_ )
    {
    }

    FenceCreateInfo( VkFenceCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( FenceCreateInfo ) );
    }

    FenceCreateInfo& operator=( VkFenceCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( FenceCreateInfo ) );
      return *this;
    }
    FenceCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    FenceCreateInfo& setFlags( FenceCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    operator VkFenceCreateInfo const&() const
    {
      return *reinterpret_cast<const VkFenceCreateInfo*>(this);
    }

    operator VkFenceCreateInfo &()
    {
      return *reinterpret_cast<VkFenceCreateInfo*>(this);
    }

    bool operator==( FenceCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags );
    }

    bool operator!=( FenceCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eFenceCreateInfo;

  public:
    const void* pNext = nullptr;
    FenceCreateFlags flags;
  };
  static_assert( sizeof( FenceCreateInfo ) == sizeof( VkFenceCreateInfo ), "struct and wrapper have different size!" );

  enum class FormatFeatureFlagBits
  {
    eSampledImage = VK_FORMAT_FEATURE_SAMPLED_IMAGE_BIT,
    eStorageImage = VK_FORMAT_FEATURE_STORAGE_IMAGE_BIT,
    eStorageImageAtomic = VK_FORMAT_FEATURE_STORAGE_IMAGE_ATOMIC_BIT,
    eUniformTexelBuffer = VK_FORMAT_FEATURE_UNIFORM_TEXEL_BUFFER_BIT,
    eStorageTexelBuffer = VK_FORMAT_FEATURE_STORAGE_TEXEL_BUFFER_BIT,
    eStorageTexelBufferAtomic = VK_FORMAT_FEATURE_STORAGE_TEXEL_BUFFER_ATOMIC_BIT,
    eVertexBuffer = VK_FORMAT_FEATURE_VERTEX_BUFFER_BIT,
    eColorAttachment = VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BIT,
    eColorAttachmentBlend = VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BLEND_BIT,
    eDepthStencilAttachment = VK_FORMAT_FEATURE_DEPTH_STENCIL_ATTACHMENT_BIT,
    eBlitSrc = VK_FORMAT_FEATURE_BLIT_SRC_BIT,
    eBlitDst = VK_FORMAT_FEATURE_BLIT_DST_BIT,
    eSampledImageFilterLinear = VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_LINEAR_BIT,
    eTransferSrc = VK_FORMAT_FEATURE_TRANSFER_SRC_BIT,
    eTransferSrcKHR = VK_FORMAT_FEATURE_TRANSFER_SRC_BIT,
    eTransferDst = VK_FORMAT_FEATURE_TRANSFER_DST_BIT,
    eTransferDstKHR = VK_FORMAT_FEATURE_TRANSFER_DST_BIT,
    eMidpointChromaSamples = VK_FORMAT_FEATURE_MIDPOINT_CHROMA_SAMPLES_BIT,
    eMidpointChromaSamplesKHR = VK_FORMAT_FEATURE_MIDPOINT_CHROMA_SAMPLES_BIT,
    eSampledImageYcbcrConversionLinearFilter = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_LINEAR_FILTER_BIT,
    eSampledImageYcbcrConversionLinearFilterKHR = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_LINEAR_FILTER_BIT,
    eSampledImageYcbcrConversionSeparateReconstructionFilter = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_SEPARATE_RECONSTRUCTION_FILTER_BIT,
    eSampledImageYcbcrConversionSeparateReconstructionFilterKHR = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_SEPARATE_RECONSTRUCTION_FILTER_BIT,
    eSampledImageYcbcrConversionChromaReconstructionExplicit = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_BIT,
    eSampledImageYcbcrConversionChromaReconstructionExplicitKHR = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_BIT,
    eSampledImageYcbcrConversionChromaReconstructionExplicitForceable = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_FORCEABLE_BIT,
    eSampledImageYcbcrConversionChromaReconstructionExplicitForceableKHR = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_FORCEABLE_BIT,
    eDisjoint = VK_FORMAT_FEATURE_DISJOINT_BIT,
    eDisjointKHR = VK_FORMAT_FEATURE_DISJOINT_BIT,
    eCositedChromaSamples = VK_FORMAT_FEATURE_COSITED_CHROMA_SAMPLES_BIT,
    eCositedChromaSamplesKHR = VK_FORMAT_FEATURE_COSITED_CHROMA_SAMPLES_BIT,
    eSampledImageFilterCubicIMG = VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_CUBIC_BIT_IMG,
    eSampledImageFilterMinmaxEXT = VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_MINMAX_BIT_EXT,
    eFragmentDensityMapEXT = VK_FORMAT_FEATURE_FRAGMENT_DENSITY_MAP_BIT_EXT
  };

  using FormatFeatureFlags = Flags<FormatFeatureFlagBits, VkFormatFeatureFlags>;

  VULKAN_HPP_INLINE FormatFeatureFlags operator|( FormatFeatureFlagBits bit0, FormatFeatureFlagBits bit1 )
  {
    return FormatFeatureFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE FormatFeatureFlags operator~( FormatFeatureFlagBits bits )
  {
    return ~( FormatFeatureFlags( bits ) );
  }

  template <> struct FlagTraits<FormatFeatureFlagBits>
  {
    enum
    {
      allFlags = VkFlags(FormatFeatureFlagBits::eSampledImage) | VkFlags(FormatFeatureFlagBits::eStorageImage) | VkFlags(FormatFeatureFlagBits::eStorageImageAtomic) | VkFlags(FormatFeatureFlagBits::eUniformTexelBuffer) | VkFlags(FormatFeatureFlagBits::eStorageTexelBuffer) | VkFlags(FormatFeatureFlagBits::eStorageTexelBufferAtomic) | VkFlags(FormatFeatureFlagBits::eVertexBuffer) | VkFlags(FormatFeatureFlagBits::eColorAttachment) | VkFlags(FormatFeatureFlagBits::eColorAttachmentBlend) | VkFlags(FormatFeatureFlagBits::eDepthStencilAttachment) | VkFlags(FormatFeatureFlagBits::eBlitSrc) | VkFlags(FormatFeatureFlagBits::eBlitDst) | VkFlags(FormatFeatureFlagBits::eSampledImageFilterLinear) | VkFlags(FormatFeatureFlagBits::eTransferSrc) | VkFlags(FormatFeatureFlagBits::eTransferDst) | VkFlags(FormatFeatureFlagBits::eMidpointChromaSamples) | VkFlags(FormatFeatureFlagBits::eSampledImageYcbcrConversionLinearFilter) | VkFlags(FormatFeatureFlagBits::eSampledImageYcbcrConversionSeparateReconstructionFilter) | VkFlags(FormatFeatureFlagBits::eSampledImageYcbcrConversionChromaReconstructionExplicit) | VkFlags(FormatFeatureFlagBits::eSampledImageYcbcrConversionChromaReconstructionExplicitForceable) | VkFlags(FormatFeatureFlagBits::eDisjoint) | VkFlags(FormatFeatureFlagBits::eCositedChromaSamples) | VkFlags(FormatFeatureFlagBits::eSampledImageFilterCubicIMG) | VkFlags(FormatFeatureFlagBits::eSampledImageFilterMinmaxEXT) | VkFlags(FormatFeatureFlagBits::eFragmentDensityMapEXT)
    };
  };

  struct FormatProperties
  {
    operator VkFormatProperties const&() const
    {
      return *reinterpret_cast<const VkFormatProperties*>(this);
    }

    operator VkFormatProperties &()
    {
      return *reinterpret_cast<VkFormatProperties*>(this);
    }

    bool operator==( FormatProperties const& rhs ) const
    {
      return ( linearTilingFeatures == rhs.linearTilingFeatures )
          && ( optimalTilingFeatures == rhs.optimalTilingFeatures )
          && ( bufferFeatures == rhs.bufferFeatures );
    }

    bool operator!=( FormatProperties const& rhs ) const
    {
      return !operator==( rhs );
    }

    FormatFeatureFlags linearTilingFeatures;
    FormatFeatureFlags optimalTilingFeatures;
    FormatFeatureFlags bufferFeatures;
  };
  static_assert( sizeof( FormatProperties ) == sizeof( VkFormatProperties ), "struct and wrapper have different size!" );

  struct FormatProperties2
  {
    operator VkFormatProperties2 const&() const
    {
      return *reinterpret_cast<const VkFormatProperties2*>(this);
    }

    operator VkFormatProperties2 &()
    {
      return *reinterpret_cast<VkFormatProperties2*>(this);
    }

    bool operator==( FormatProperties2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( formatProperties == rhs.formatProperties );
    }

    bool operator!=( FormatProperties2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eFormatProperties2;

  public:
    void* pNext = nullptr;
    FormatProperties formatProperties;
  };
  static_assert( sizeof( FormatProperties2 ) == sizeof( VkFormatProperties2 ), "struct and wrapper have different size!" );

  using FormatProperties2KHR = FormatProperties2;

  struct DrmFormatModifierPropertiesEXT
  {
    operator VkDrmFormatModifierPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkDrmFormatModifierPropertiesEXT*>(this);
    }

    operator VkDrmFormatModifierPropertiesEXT &()
    {
      return *reinterpret_cast<VkDrmFormatModifierPropertiesEXT*>(this);
    }

    bool operator==( DrmFormatModifierPropertiesEXT const& rhs ) const
    {
      return ( drmFormatModifier == rhs.drmFormatModifier )
          && ( drmFormatModifierPlaneCount == rhs.drmFormatModifierPlaneCount )
          && ( drmFormatModifierTilingFeatures == rhs.drmFormatModifierTilingFeatures );
    }

    bool operator!=( DrmFormatModifierPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint64_t drmFormatModifier;
    uint32_t drmFormatModifierPlaneCount;
    FormatFeatureFlags drmFormatModifierTilingFeatures;
  };
  static_assert( sizeof( DrmFormatModifierPropertiesEXT ) == sizeof( VkDrmFormatModifierPropertiesEXT ), "struct and wrapper have different size!" );

  struct DrmFormatModifierPropertiesListEXT
  {
    DrmFormatModifierPropertiesListEXT( uint32_t drmFormatModifierCount_ = 0,
                                        DrmFormatModifierPropertiesEXT* pDrmFormatModifierProperties_ = nullptr )
      : drmFormatModifierCount( drmFormatModifierCount_ )
      , pDrmFormatModifierProperties( pDrmFormatModifierProperties_ )
    {
    }

    DrmFormatModifierPropertiesListEXT( VkDrmFormatModifierPropertiesListEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DrmFormatModifierPropertiesListEXT ) );
    }

    DrmFormatModifierPropertiesListEXT& operator=( VkDrmFormatModifierPropertiesListEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DrmFormatModifierPropertiesListEXT ) );
      return *this;
    }
    DrmFormatModifierPropertiesListEXT& setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DrmFormatModifierPropertiesListEXT& setDrmFormatModifierCount( uint32_t drmFormatModifierCount_ )
    {
      drmFormatModifierCount = drmFormatModifierCount_;
      return *this;
    }

    DrmFormatModifierPropertiesListEXT& setPDrmFormatModifierProperties( DrmFormatModifierPropertiesEXT* pDrmFormatModifierProperties_ )
    {
      pDrmFormatModifierProperties = pDrmFormatModifierProperties_;
      return *this;
    }

    operator VkDrmFormatModifierPropertiesListEXT const&() const
    {
      return *reinterpret_cast<const VkDrmFormatModifierPropertiesListEXT*>(this);
    }

    operator VkDrmFormatModifierPropertiesListEXT &()
    {
      return *reinterpret_cast<VkDrmFormatModifierPropertiesListEXT*>(this);
    }

    bool operator==( DrmFormatModifierPropertiesListEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( drmFormatModifierCount == rhs.drmFormatModifierCount )
          && ( pDrmFormatModifierProperties == rhs.pDrmFormatModifierProperties );
    }

    bool operator!=( DrmFormatModifierPropertiesListEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDrmFormatModifierPropertiesListEXT;

  public:
    void* pNext = nullptr;
    uint32_t drmFormatModifierCount;
    DrmFormatModifierPropertiesEXT* pDrmFormatModifierProperties;
  };
  static_assert( sizeof( DrmFormatModifierPropertiesListEXT ) == sizeof( VkDrmFormatModifierPropertiesListEXT ), "struct and wrapper have different size!" );

  enum class QueryControlFlagBits
  {
    ePrecise = VK_QUERY_CONTROL_PRECISE_BIT
  };

  using QueryControlFlags = Flags<QueryControlFlagBits, VkQueryControlFlags>;

  VULKAN_HPP_INLINE QueryControlFlags operator|( QueryControlFlagBits bit0, QueryControlFlagBits bit1 )
  {
    return QueryControlFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE QueryControlFlags operator~( QueryControlFlagBits bits )
  {
    return ~( QueryControlFlags( bits ) );
  }

  template <> struct FlagTraits<QueryControlFlagBits>
  {
    enum
    {
      allFlags = VkFlags(QueryControlFlagBits::ePrecise)
    };
  };

  enum class QueryResultFlagBits
  {
    e64 = VK_QUERY_RESULT_64_BIT,
    eWait = VK_QUERY_RESULT_WAIT_BIT,
    eWithAvailability = VK_QUERY_RESULT_WITH_AVAILABILITY_BIT,
    ePartial = VK_QUERY_RESULT_PARTIAL_BIT
  };

  using QueryResultFlags = Flags<QueryResultFlagBits, VkQueryResultFlags>;

  VULKAN_HPP_INLINE QueryResultFlags operator|( QueryResultFlagBits bit0, QueryResultFlagBits bit1 )
  {
    return QueryResultFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE QueryResultFlags operator~( QueryResultFlagBits bits )
  {
    return ~( QueryResultFlags( bits ) );
  }

  template <> struct FlagTraits<QueryResultFlagBits>
  {
    enum
    {
      allFlags = VkFlags(QueryResultFlagBits::e64) | VkFlags(QueryResultFlagBits::eWait) | VkFlags(QueryResultFlagBits::eWithAvailability) | VkFlags(QueryResultFlagBits::ePartial)
    };
  };

  enum class CommandBufferUsageFlagBits
  {
    eOneTimeSubmit = VK_COMMAND_BUFFER_USAGE_ONE_TIME_SUBMIT_BIT,
    eRenderPassContinue = VK_COMMAND_BUFFER_USAGE_RENDER_PASS_CONTINUE_BIT,
    eSimultaneousUse = VK_COMMAND_BUFFER_USAGE_SIMULTANEOUS_USE_BIT
  };

  using CommandBufferUsageFlags = Flags<CommandBufferUsageFlagBits, VkCommandBufferUsageFlags>;

  VULKAN_HPP_INLINE CommandBufferUsageFlags operator|( CommandBufferUsageFlagBits bit0, CommandBufferUsageFlagBits bit1 )
  {
    return CommandBufferUsageFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE CommandBufferUsageFlags operator~( CommandBufferUsageFlagBits bits )
  {
    return ~( CommandBufferUsageFlags( bits ) );
  }

  template <> struct FlagTraits<CommandBufferUsageFlagBits>
  {
    enum
    {
      allFlags = VkFlags(CommandBufferUsageFlagBits::eOneTimeSubmit) | VkFlags(CommandBufferUsageFlagBits::eRenderPassContinue) | VkFlags(CommandBufferUsageFlagBits::eSimultaneousUse)
    };
  };

  enum class QueryPipelineStatisticFlagBits
  {
    eInputAssemblyVertices = VK_QUERY_PIPELINE_STATISTIC_INPUT_ASSEMBLY_VERTICES_BIT,
    eInputAssemblyPrimitives = VK_QUERY_PIPELINE_STATISTIC_INPUT_ASSEMBLY_PRIMITIVES_BIT,
    eVertexShaderInvocations = VK_QUERY_PIPELINE_STATISTIC_VERTEX_SHADER_INVOCATIONS_BIT,
    eGeometryShaderInvocations = VK_QUERY_PIPELINE_STATISTIC_GEOMETRY_SHADER_INVOCATIONS_BIT,
    eGeometryShaderPrimitives = VK_QUERY_PIPELINE_STATISTIC_GEOMETRY_SHADER_PRIMITIVES_BIT,
    eClippingInvocations = VK_QUERY_PIPELINE_STATISTIC_CLIPPING_INVOCATIONS_BIT,
    eClippingPrimitives = VK_QUERY_PIPELINE_STATISTIC_CLIPPING_PRIMITIVES_BIT,
    eFragmentShaderInvocations = VK_QUERY_PIPELINE_STATISTIC_FRAGMENT_SHADER_INVOCATIONS_BIT,
    eTessellationControlShaderPatches = VK_QUERY_PIPELINE_STATISTIC_TESSELLATION_CONTROL_SHADER_PATCHES_BIT,
    eTessellationEvaluationShaderInvocations = VK_QUERY_PIPELINE_STATISTIC_TESSELLATION_EVALUATION_SHADER_INVOCATIONS_BIT,
    eComputeShaderInvocations = VK_QUERY_PIPELINE_STATISTIC_COMPUTE_SHADER_INVOCATIONS_BIT
  };

  using QueryPipelineStatisticFlags = Flags<QueryPipelineStatisticFlagBits, VkQueryPipelineStatisticFlags>;

  VULKAN_HPP_INLINE QueryPipelineStatisticFlags operator|( QueryPipelineStatisticFlagBits bit0, QueryPipelineStatisticFlagBits bit1 )
  {
    return QueryPipelineStatisticFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE QueryPipelineStatisticFlags operator~( QueryPipelineStatisticFlagBits bits )
  {
    return ~( QueryPipelineStatisticFlags( bits ) );
  }

  template <> struct FlagTraits<QueryPipelineStatisticFlagBits>
  {
    enum
    {
      allFlags = VkFlags(QueryPipelineStatisticFlagBits::eInputAssemblyVertices) | VkFlags(QueryPipelineStatisticFlagBits::eInputAssemblyPrimitives) | VkFlags(QueryPipelineStatisticFlagBits::eVertexShaderInvocations) | VkFlags(QueryPipelineStatisticFlagBits::eGeometryShaderInvocations) | VkFlags(QueryPipelineStatisticFlagBits::eGeometryShaderPrimitives) | VkFlags(QueryPipelineStatisticFlagBits::eClippingInvocations) | VkFlags(QueryPipelineStatisticFlagBits::eClippingPrimitives) | VkFlags(QueryPipelineStatisticFlagBits::eFragmentShaderInvocations) | VkFlags(QueryPipelineStatisticFlagBits::eTessellationControlShaderPatches) | VkFlags(QueryPipelineStatisticFlagBits::eTessellationEvaluationShaderInvocations) | VkFlags(QueryPipelineStatisticFlagBits::eComputeShaderInvocations)
    };
  };

  struct CommandBufferInheritanceInfo
  {
    CommandBufferInheritanceInfo( RenderPass renderPass_ = RenderPass(),
                                  uint32_t subpass_ = 0,
                                  Framebuffer framebuffer_ = Framebuffer(),
                                  Bool32 occlusionQueryEnable_ = 0,
                                  QueryControlFlags queryFlags_ = QueryControlFlags(),
                                  QueryPipelineStatisticFlags pipelineStatistics_ = QueryPipelineStatisticFlags() )
      : renderPass( renderPass_ )
      , subpass( subpass_ )
      , framebuffer( framebuffer_ )
      , occlusionQueryEnable( occlusionQueryEnable_ )
      , queryFlags( queryFlags_ )
      , pipelineStatistics( pipelineStatistics_ )
    {
    }

    CommandBufferInheritanceInfo( VkCommandBufferInheritanceInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferInheritanceInfo ) );
    }

    CommandBufferInheritanceInfo& operator=( VkCommandBufferInheritanceInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferInheritanceInfo ) );
      return *this;
    }
    CommandBufferInheritanceInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CommandBufferInheritanceInfo& setRenderPass( RenderPass renderPass_ )
    {
      renderPass = renderPass_;
      return *this;
    }

    CommandBufferInheritanceInfo& setSubpass( uint32_t subpass_ )
    {
      subpass = subpass_;
      return *this;
    }

    CommandBufferInheritanceInfo& setFramebuffer( Framebuffer framebuffer_ )
    {
      framebuffer = framebuffer_;
      return *this;
    }

    CommandBufferInheritanceInfo& setOcclusionQueryEnable( Bool32 occlusionQueryEnable_ )
    {
      occlusionQueryEnable = occlusionQueryEnable_;
      return *this;
    }

    CommandBufferInheritanceInfo& setQueryFlags( QueryControlFlags queryFlags_ )
    {
      queryFlags = queryFlags_;
      return *this;
    }

    CommandBufferInheritanceInfo& setPipelineStatistics( QueryPipelineStatisticFlags pipelineStatistics_ )
    {
      pipelineStatistics = pipelineStatistics_;
      return *this;
    }

    operator VkCommandBufferInheritanceInfo const&() const
    {
      return *reinterpret_cast<const VkCommandBufferInheritanceInfo*>(this);
    }

    operator VkCommandBufferInheritanceInfo &()
    {
      return *reinterpret_cast<VkCommandBufferInheritanceInfo*>(this);
    }

    bool operator==( CommandBufferInheritanceInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( renderPass == rhs.renderPass )
          && ( subpass == rhs.subpass )
          && ( framebuffer == rhs.framebuffer )
          && ( occlusionQueryEnable == rhs.occlusionQueryEnable )
          && ( queryFlags == rhs.queryFlags )
          && ( pipelineStatistics == rhs.pipelineStatistics );
    }

    bool operator!=( CommandBufferInheritanceInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCommandBufferInheritanceInfo;

  public:
    const void* pNext = nullptr;
    RenderPass renderPass;
    uint32_t subpass;
    Framebuffer framebuffer;
    Bool32 occlusionQueryEnable;
    QueryControlFlags queryFlags;
    QueryPipelineStatisticFlags pipelineStatistics;
  };
  static_assert( sizeof( CommandBufferInheritanceInfo ) == sizeof( VkCommandBufferInheritanceInfo ), "struct and wrapper have different size!" );

  struct CommandBufferBeginInfo
  {
    CommandBufferBeginInfo( CommandBufferUsageFlags flags_ = CommandBufferUsageFlags(),
                            const CommandBufferInheritanceInfo* pInheritanceInfo_ = nullptr )
      : flags( flags_ )
      , pInheritanceInfo( pInheritanceInfo_ )
    {
    }

    CommandBufferBeginInfo( VkCommandBufferBeginInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferBeginInfo ) );
    }

    CommandBufferBeginInfo& operator=( VkCommandBufferBeginInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferBeginInfo ) );
      return *this;
    }
    CommandBufferBeginInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CommandBufferBeginInfo& setFlags( CommandBufferUsageFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    CommandBufferBeginInfo& setPInheritanceInfo( const CommandBufferInheritanceInfo* pInheritanceInfo_ )
    {
      pInheritanceInfo = pInheritanceInfo_;
      return *this;
    }

    operator VkCommandBufferBeginInfo const&() const
    {
      return *reinterpret_cast<const VkCommandBufferBeginInfo*>(this);
    }

    operator VkCommandBufferBeginInfo &()
    {
      return *reinterpret_cast<VkCommandBufferBeginInfo*>(this);
    }

    bool operator==( CommandBufferBeginInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( pInheritanceInfo == rhs.pInheritanceInfo );
    }

    bool operator!=( CommandBufferBeginInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCommandBufferBeginInfo;

  public:
    const void* pNext = nullptr;
    CommandBufferUsageFlags flags;
    const CommandBufferInheritanceInfo* pInheritanceInfo;
  };
  static_assert( sizeof( CommandBufferBeginInfo ) == sizeof( VkCommandBufferBeginInfo ), "struct and wrapper have different size!" );

  struct QueryPoolCreateInfo
  {
    QueryPoolCreateInfo( QueryPoolCreateFlags flags_ = QueryPoolCreateFlags(),
                         QueryType queryType_ = QueryType::eOcclusion,
                         uint32_t queryCount_ = 0,
                         QueryPipelineStatisticFlags pipelineStatistics_ = QueryPipelineStatisticFlags() )
      : flags( flags_ )
      , queryType( queryType_ )
      , queryCount( queryCount_ )
      , pipelineStatistics( pipelineStatistics_ )
    {
    }

    QueryPoolCreateInfo( VkQueryPoolCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( QueryPoolCreateInfo ) );
    }

    QueryPoolCreateInfo& operator=( VkQueryPoolCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( QueryPoolCreateInfo ) );
      return *this;
    }
    QueryPoolCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    QueryPoolCreateInfo& setFlags( QueryPoolCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    QueryPoolCreateInfo& setQueryType( QueryType queryType_ )
    {
      queryType = queryType_;
      return *this;
    }

    QueryPoolCreateInfo& setQueryCount( uint32_t queryCount_ )
    {
      queryCount = queryCount_;
      return *this;
    }

    QueryPoolCreateInfo& setPipelineStatistics( QueryPipelineStatisticFlags pipelineStatistics_ )
    {
      pipelineStatistics = pipelineStatistics_;
      return *this;
    }

    operator VkQueryPoolCreateInfo const&() const
    {
      return *reinterpret_cast<const VkQueryPoolCreateInfo*>(this);
    }

    operator VkQueryPoolCreateInfo &()
    {
      return *reinterpret_cast<VkQueryPoolCreateInfo*>(this);
    }

    bool operator==( QueryPoolCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( queryType == rhs.queryType )
          && ( queryCount == rhs.queryCount )
          && ( pipelineStatistics == rhs.pipelineStatistics );
    }

    bool operator!=( QueryPoolCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eQueryPoolCreateInfo;

  public:
    const void* pNext = nullptr;
    QueryPoolCreateFlags flags;
    QueryType queryType;
    uint32_t queryCount;
    QueryPipelineStatisticFlags pipelineStatistics;
  };
  static_assert( sizeof( QueryPoolCreateInfo ) == sizeof( VkQueryPoolCreateInfo ), "struct and wrapper have different size!" );

  enum class ImageAspectFlagBits
  {
    eColor = VK_IMAGE_ASPECT_COLOR_BIT,
    eDepth = VK_IMAGE_ASPECT_DEPTH_BIT,
    eStencil = VK_IMAGE_ASPECT_STENCIL_BIT,
    eMetadata = VK_IMAGE_ASPECT_METADATA_BIT,
    ePlane0 = VK_IMAGE_ASPECT_PLANE_0_BIT,
    ePlane0KHR = VK_IMAGE_ASPECT_PLANE_0_BIT,
    ePlane1 = VK_IMAGE_ASPECT_PLANE_1_BIT,
    ePlane1KHR = VK_IMAGE_ASPECT_PLANE_1_BIT,
    ePlane2 = VK_IMAGE_ASPECT_PLANE_2_BIT,
    ePlane2KHR = VK_IMAGE_ASPECT_PLANE_2_BIT,
    eMemoryPlane0EXT = VK_IMAGE_ASPECT_MEMORY_PLANE_0_BIT_EXT,
    eMemoryPlane1EXT = VK_IMAGE_ASPECT_MEMORY_PLANE_1_BIT_EXT,
    eMemoryPlane2EXT = VK_IMAGE_ASPECT_MEMORY_PLANE_2_BIT_EXT,
    eMemoryPlane3EXT = VK_IMAGE_ASPECT_MEMORY_PLANE_3_BIT_EXT
  };

  using ImageAspectFlags = Flags<ImageAspectFlagBits, VkImageAspectFlags>;

  VULKAN_HPP_INLINE ImageAspectFlags operator|( ImageAspectFlagBits bit0, ImageAspectFlagBits bit1 )
  {
    return ImageAspectFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ImageAspectFlags operator~( ImageAspectFlagBits bits )
  {
    return ~( ImageAspectFlags( bits ) );
  }

  template <> struct FlagTraits<ImageAspectFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ImageAspectFlagBits::eColor) | VkFlags(ImageAspectFlagBits::eDepth) | VkFlags(ImageAspectFlagBits::eStencil) | VkFlags(ImageAspectFlagBits::eMetadata) | VkFlags(ImageAspectFlagBits::ePlane0) | VkFlags(ImageAspectFlagBits::ePlane1) | VkFlags(ImageAspectFlagBits::ePlane2) | VkFlags(ImageAspectFlagBits::eMemoryPlane0EXT) | VkFlags(ImageAspectFlagBits::eMemoryPlane1EXT) | VkFlags(ImageAspectFlagBits::eMemoryPlane2EXT) | VkFlags(ImageAspectFlagBits::eMemoryPlane3EXT)
    };
  };

  struct ImageSubresource
  {
    ImageSubresource( ImageAspectFlags aspectMask_ = ImageAspectFlags(),
                      uint32_t mipLevel_ = 0,
                      uint32_t arrayLayer_ = 0 )
      : aspectMask( aspectMask_ )
      , mipLevel( mipLevel_ )
      , arrayLayer( arrayLayer_ )
    {
    }

    ImageSubresource( VkImageSubresource const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSubresource ) );
    }

    ImageSubresource& operator=( VkImageSubresource const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSubresource ) );
      return *this;
    }
    ImageSubresource& setAspectMask( ImageAspectFlags aspectMask_ )
    {
      aspectMask = aspectMask_;
      return *this;
    }

    ImageSubresource& setMipLevel( uint32_t mipLevel_ )
    {
      mipLevel = mipLevel_;
      return *this;
    }

    ImageSubresource& setArrayLayer( uint32_t arrayLayer_ )
    {
      arrayLayer = arrayLayer_;
      return *this;
    }

    operator VkImageSubresource const&() const
    {
      return *reinterpret_cast<const VkImageSubresource*>(this);
    }

    operator VkImageSubresource &()
    {
      return *reinterpret_cast<VkImageSubresource*>(this);
    }

    bool operator==( ImageSubresource const& rhs ) const
    {
      return ( aspectMask == rhs.aspectMask )
          && ( mipLevel == rhs.mipLevel )
          && ( arrayLayer == rhs.arrayLayer );
    }

    bool operator!=( ImageSubresource const& rhs ) const
    {
      return !operator==( rhs );
    }

    ImageAspectFlags aspectMask;
    uint32_t mipLevel;
    uint32_t arrayLayer;
  };
  static_assert( sizeof( ImageSubresource ) == sizeof( VkImageSubresource ), "struct and wrapper have different size!" );

  struct ImageSubresourceLayers
  {
    ImageSubresourceLayers( ImageAspectFlags aspectMask_ = ImageAspectFlags(),
                            uint32_t mipLevel_ = 0,
                            uint32_t baseArrayLayer_ = 0,
                            uint32_t layerCount_ = 0 )
      : aspectMask( aspectMask_ )
      , mipLevel( mipLevel_ )
      , baseArrayLayer( baseArrayLayer_ )
      , layerCount( layerCount_ )
    {
    }

    ImageSubresourceLayers( VkImageSubresourceLayers const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSubresourceLayers ) );
    }

    ImageSubresourceLayers& operator=( VkImageSubresourceLayers const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSubresourceLayers ) );
      return *this;
    }
    ImageSubresourceLayers& setAspectMask( ImageAspectFlags aspectMask_ )
    {
      aspectMask = aspectMask_;
      return *this;
    }

    ImageSubresourceLayers& setMipLevel( uint32_t mipLevel_ )
    {
      mipLevel = mipLevel_;
      return *this;
    }

    ImageSubresourceLayers& setBaseArrayLayer( uint32_t baseArrayLayer_ )
    {
      baseArrayLayer = baseArrayLayer_;
      return *this;
    }

    ImageSubresourceLayers& setLayerCount( uint32_t layerCount_ )
    {
      layerCount = layerCount_;
      return *this;
    }

    operator VkImageSubresourceLayers const&() const
    {
      return *reinterpret_cast<const VkImageSubresourceLayers*>(this);
    }

    operator VkImageSubresourceLayers &()
    {
      return *reinterpret_cast<VkImageSubresourceLayers*>(this);
    }

    bool operator==( ImageSubresourceLayers const& rhs ) const
    {
      return ( aspectMask == rhs.aspectMask )
          && ( mipLevel == rhs.mipLevel )
          && ( baseArrayLayer == rhs.baseArrayLayer )
          && ( layerCount == rhs.layerCount );
    }

    bool operator!=( ImageSubresourceLayers const& rhs ) const
    {
      return !operator==( rhs );
    }

    ImageAspectFlags aspectMask;
    uint32_t mipLevel;
    uint32_t baseArrayLayer;
    uint32_t layerCount;
  };
  static_assert( sizeof( ImageSubresourceLayers ) == sizeof( VkImageSubresourceLayers ), "struct and wrapper have different size!" );

  struct ImageSubresourceRange
  {
    ImageSubresourceRange( ImageAspectFlags aspectMask_ = ImageAspectFlags(),
                           uint32_t baseMipLevel_ = 0,
                           uint32_t levelCount_ = 0,
                           uint32_t baseArrayLayer_ = 0,
                           uint32_t layerCount_ = 0 )
      : aspectMask( aspectMask_ )
      , baseMipLevel( baseMipLevel_ )
      , levelCount( levelCount_ )
      , baseArrayLayer( baseArrayLayer_ )
      , layerCount( layerCount_ )
    {
    }

    ImageSubresourceRange( VkImageSubresourceRange const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSubresourceRange ) );
    }

    ImageSubresourceRange& operator=( VkImageSubresourceRange const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSubresourceRange ) );
      return *this;
    }
    ImageSubresourceRange& setAspectMask( ImageAspectFlags aspectMask_ )
    {
      aspectMask = aspectMask_;
      return *this;
    }

    ImageSubresourceRange& setBaseMipLevel( uint32_t baseMipLevel_ )
    {
      baseMipLevel = baseMipLevel_;
      return *this;
    }

    ImageSubresourceRange& setLevelCount( uint32_t levelCount_ )
    {
      levelCount = levelCount_;
      return *this;
    }

    ImageSubresourceRange& setBaseArrayLayer( uint32_t baseArrayLayer_ )
    {
      baseArrayLayer = baseArrayLayer_;
      return *this;
    }

    ImageSubresourceRange& setLayerCount( uint32_t layerCount_ )
    {
      layerCount = layerCount_;
      return *this;
    }

    operator VkImageSubresourceRange const&() const
    {
      return *reinterpret_cast<const VkImageSubresourceRange*>(this);
    }

    operator VkImageSubresourceRange &()
    {
      return *reinterpret_cast<VkImageSubresourceRange*>(this);
    }

    bool operator==( ImageSubresourceRange const& rhs ) const
    {
      return ( aspectMask == rhs.aspectMask )
          && ( baseMipLevel == rhs.baseMipLevel )
          && ( levelCount == rhs.levelCount )
          && ( baseArrayLayer == rhs.baseArrayLayer )
          && ( layerCount == rhs.layerCount );
    }

    bool operator!=( ImageSubresourceRange const& rhs ) const
    {
      return !operator==( rhs );
    }

    ImageAspectFlags aspectMask;
    uint32_t baseMipLevel;
    uint32_t levelCount;
    uint32_t baseArrayLayer;
    uint32_t layerCount;
  };
  static_assert( sizeof( ImageSubresourceRange ) == sizeof( VkImageSubresourceRange ), "struct and wrapper have different size!" );

  struct ImageMemoryBarrier
  {
    ImageMemoryBarrier( AccessFlags srcAccessMask_ = AccessFlags(),
                        AccessFlags dstAccessMask_ = AccessFlags(),
                        ImageLayout oldLayout_ = ImageLayout::eUndefined,
                        ImageLayout newLayout_ = ImageLayout::eUndefined,
                        uint32_t srcQueueFamilyIndex_ = 0,
                        uint32_t dstQueueFamilyIndex_ = 0,
                        Image image_ = Image(),
                        ImageSubresourceRange subresourceRange_ = ImageSubresourceRange() )
      : srcAccessMask( srcAccessMask_ )
      , dstAccessMask( dstAccessMask_ )
      , oldLayout( oldLayout_ )
      , newLayout( newLayout_ )
      , srcQueueFamilyIndex( srcQueueFamilyIndex_ )
      , dstQueueFamilyIndex( dstQueueFamilyIndex_ )
      , image( image_ )
      , subresourceRange( subresourceRange_ )
    {
    }

    ImageMemoryBarrier( VkImageMemoryBarrier const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageMemoryBarrier ) );
    }

    ImageMemoryBarrier& operator=( VkImageMemoryBarrier const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageMemoryBarrier ) );
      return *this;
    }
    ImageMemoryBarrier& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImageMemoryBarrier& setSrcAccessMask( AccessFlags srcAccessMask_ )
    {
      srcAccessMask = srcAccessMask_;
      return *this;
    }

    ImageMemoryBarrier& setDstAccessMask( AccessFlags dstAccessMask_ )
    {
      dstAccessMask = dstAccessMask_;
      return *this;
    }

    ImageMemoryBarrier& setOldLayout( ImageLayout oldLayout_ )
    {
      oldLayout = oldLayout_;
      return *this;
    }

    ImageMemoryBarrier& setNewLayout( ImageLayout newLayout_ )
    {
      newLayout = newLayout_;
      return *this;
    }

    ImageMemoryBarrier& setSrcQueueFamilyIndex( uint32_t srcQueueFamilyIndex_ )
    {
      srcQueueFamilyIndex = srcQueueFamilyIndex_;
      return *this;
    }

    ImageMemoryBarrier& setDstQueueFamilyIndex( uint32_t dstQueueFamilyIndex_ )
    {
      dstQueueFamilyIndex = dstQueueFamilyIndex_;
      return *this;
    }

    ImageMemoryBarrier& setImage( Image image_ )
    {
      image = image_;
      return *this;
    }

    ImageMemoryBarrier& setSubresourceRange( ImageSubresourceRange subresourceRange_ )
    {
      subresourceRange = subresourceRange_;
      return *this;
    }

    operator VkImageMemoryBarrier const&() const
    {
      return *reinterpret_cast<const VkImageMemoryBarrier*>(this);
    }

    operator VkImageMemoryBarrier &()
    {
      return *reinterpret_cast<VkImageMemoryBarrier*>(this);
    }

    bool operator==( ImageMemoryBarrier const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( srcAccessMask == rhs.srcAccessMask )
          && ( dstAccessMask == rhs.dstAccessMask )
          && ( oldLayout == rhs.oldLayout )
          && ( newLayout == rhs.newLayout )
          && ( srcQueueFamilyIndex == rhs.srcQueueFamilyIndex )
          && ( dstQueueFamilyIndex == rhs.dstQueueFamilyIndex )
          && ( image == rhs.image )
          && ( subresourceRange == rhs.subresourceRange );
    }

    bool operator!=( ImageMemoryBarrier const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageMemoryBarrier;

  public:
    const void* pNext = nullptr;
    AccessFlags srcAccessMask;
    AccessFlags dstAccessMask;
    ImageLayout oldLayout;
    ImageLayout newLayout;
    uint32_t srcQueueFamilyIndex;
    uint32_t dstQueueFamilyIndex;
    Image image;
    ImageSubresourceRange subresourceRange;
  };
  static_assert( sizeof( ImageMemoryBarrier ) == sizeof( VkImageMemoryBarrier ), "struct and wrapper have different size!" );

  struct ImageViewCreateInfo
  {
    ImageViewCreateInfo( ImageViewCreateFlags flags_ = ImageViewCreateFlags(),
                         Image image_ = Image(),
                         ImageViewType viewType_ = ImageViewType::e1D,
                         Format format_ = Format::eUndefined,
                         ComponentMapping components_ = ComponentMapping(),
                         ImageSubresourceRange subresourceRange_ = ImageSubresourceRange() )
      : flags( flags_ )
      , image( image_ )
      , viewType( viewType_ )
      , format( format_ )
      , components( components_ )
      , subresourceRange( subresourceRange_ )
    {
    }

    ImageViewCreateInfo( VkImageViewCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageViewCreateInfo ) );
    }

    ImageViewCreateInfo& operator=( VkImageViewCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageViewCreateInfo ) );
      return *this;
    }
    ImageViewCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImageViewCreateInfo& setFlags( ImageViewCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    ImageViewCreateInfo& setImage( Image image_ )
    {
      image = image_;
      return *this;
    }

    ImageViewCreateInfo& setViewType( ImageViewType viewType_ )
    {
      viewType = viewType_;
      return *this;
    }

    ImageViewCreateInfo& setFormat( Format format_ )
    {
      format = format_;
      return *this;
    }

    ImageViewCreateInfo& setComponents( ComponentMapping components_ )
    {
      components = components_;
      return *this;
    }

    ImageViewCreateInfo& setSubresourceRange( ImageSubresourceRange subresourceRange_ )
    {
      subresourceRange = subresourceRange_;
      return *this;
    }

    operator VkImageViewCreateInfo const&() const
    {
      return *reinterpret_cast<const VkImageViewCreateInfo*>(this);
    }

    operator VkImageViewCreateInfo &()
    {
      return *reinterpret_cast<VkImageViewCreateInfo*>(this);
    }

    bool operator==( ImageViewCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( image == rhs.image )
          && ( viewType == rhs.viewType )
          && ( format == rhs.format )
          && ( components == rhs.components )
          && ( subresourceRange == rhs.subresourceRange );
    }

    bool operator!=( ImageViewCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageViewCreateInfo;

  public:
    const void* pNext = nullptr;
    ImageViewCreateFlags flags;
    Image image;
    ImageViewType viewType;
    Format format;
    ComponentMapping components;
    ImageSubresourceRange subresourceRange;
  };
  static_assert( sizeof( ImageViewCreateInfo ) == sizeof( VkImageViewCreateInfo ), "struct and wrapper have different size!" );

  struct ImageCopy
  {
    ImageCopy( ImageSubresourceLayers srcSubresource_ = ImageSubresourceLayers(),
               Offset3D srcOffset_ = Offset3D(),
               ImageSubresourceLayers dstSubresource_ = ImageSubresourceLayers(),
               Offset3D dstOffset_ = Offset3D(),
               Extent3D extent_ = Extent3D() )
      : srcSubresource( srcSubresource_ )
      , srcOffset( srcOffset_ )
      , dstSubresource( dstSubresource_ )
      , dstOffset( dstOffset_ )
      , extent( extent_ )
    {
    }

    ImageCopy( VkImageCopy const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageCopy ) );
    }

    ImageCopy& operator=( VkImageCopy const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageCopy ) );
      return *this;
    }
    ImageCopy& setSrcSubresource( ImageSubresourceLayers srcSubresource_ )
    {
      srcSubresource = srcSubresource_;
      return *this;
    }

    ImageCopy& setSrcOffset( Offset3D srcOffset_ )
    {
      srcOffset = srcOffset_;
      return *this;
    }

    ImageCopy& setDstSubresource( ImageSubresourceLayers dstSubresource_ )
    {
      dstSubresource = dstSubresource_;
      return *this;
    }

    ImageCopy& setDstOffset( Offset3D dstOffset_ )
    {
      dstOffset = dstOffset_;
      return *this;
    }

    ImageCopy& setExtent( Extent3D extent_ )
    {
      extent = extent_;
      return *this;
    }

    operator VkImageCopy const&() const
    {
      return *reinterpret_cast<const VkImageCopy*>(this);
    }

    operator VkImageCopy &()
    {
      return *reinterpret_cast<VkImageCopy*>(this);
    }

    bool operator==( ImageCopy const& rhs ) const
    {
      return ( srcSubresource == rhs.srcSubresource )
          && ( srcOffset == rhs.srcOffset )
          && ( dstSubresource == rhs.dstSubresource )
          && ( dstOffset == rhs.dstOffset )
          && ( extent == rhs.extent );
    }

    bool operator!=( ImageCopy const& rhs ) const
    {
      return !operator==( rhs );
    }

    ImageSubresourceLayers srcSubresource;
    Offset3D srcOffset;
    ImageSubresourceLayers dstSubresource;
    Offset3D dstOffset;
    Extent3D extent;
  };
  static_assert( sizeof( ImageCopy ) == sizeof( VkImageCopy ), "struct and wrapper have different size!" );

  struct ImageBlit
  {
    ImageBlit( ImageSubresourceLayers srcSubresource_ = ImageSubresourceLayers(),
               std::array<Offset3D,2> const& srcOffsets_ = { { Offset3D(), Offset3D() } },
               ImageSubresourceLayers dstSubresource_ = ImageSubresourceLayers(),
               std::array<Offset3D,2> const& dstOffsets_ = { { Offset3D(), Offset3D() } } )
      : srcSubresource( srcSubresource_ )
      , dstSubresource( dstSubresource_ )
    {
      memcpy( &srcOffsets, srcOffsets_.data(), 2 * sizeof( Offset3D ) );
      memcpy( &dstOffsets, dstOffsets_.data(), 2 * sizeof( Offset3D ) );
    }

    ImageBlit( VkImageBlit const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageBlit ) );
    }

    ImageBlit& operator=( VkImageBlit const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageBlit ) );
      return *this;
    }
    ImageBlit& setSrcSubresource( ImageSubresourceLayers srcSubresource_ )
    {
      srcSubresource = srcSubresource_;
      return *this;
    }

    ImageBlit& setSrcOffsets( std::array<Offset3D,2> srcOffsets_ )
    {
      memcpy( &srcOffsets, srcOffsets_.data(), 2 * sizeof( Offset3D ) );
      return *this;
    }

    ImageBlit& setDstSubresource( ImageSubresourceLayers dstSubresource_ )
    {
      dstSubresource = dstSubresource_;
      return *this;
    }

    ImageBlit& setDstOffsets( std::array<Offset3D,2> dstOffsets_ )
    {
      memcpy( &dstOffsets, dstOffsets_.data(), 2 * sizeof( Offset3D ) );
      return *this;
    }

    operator VkImageBlit const&() const
    {
      return *reinterpret_cast<const VkImageBlit*>(this);
    }

    operator VkImageBlit &()
    {
      return *reinterpret_cast<VkImageBlit*>(this);
    }

    bool operator==( ImageBlit const& rhs ) const
    {
      return ( srcSubresource == rhs.srcSubresource )
          && ( memcmp( srcOffsets, rhs.srcOffsets, 2 * sizeof( Offset3D ) ) == 0 )
          && ( dstSubresource == rhs.dstSubresource )
          && ( memcmp( dstOffsets, rhs.dstOffsets, 2 * sizeof( Offset3D ) ) == 0 );
    }

    bool operator!=( ImageBlit const& rhs ) const
    {
      return !operator==( rhs );
    }

    ImageSubresourceLayers srcSubresource;
    Offset3D srcOffsets[2];
    ImageSubresourceLayers dstSubresource;
    Offset3D dstOffsets[2];
  };
  static_assert( sizeof( ImageBlit ) == sizeof( VkImageBlit ), "struct and wrapper have different size!" );

  struct BufferImageCopy
  {
    BufferImageCopy( DeviceSize bufferOffset_ = 0,
                     uint32_t bufferRowLength_ = 0,
                     uint32_t bufferImageHeight_ = 0,
                     ImageSubresourceLayers imageSubresource_ = ImageSubresourceLayers(),
                     Offset3D imageOffset_ = Offset3D(),
                     Extent3D imageExtent_ = Extent3D() )
      : bufferOffset( bufferOffset_ )
      , bufferRowLength( bufferRowLength_ )
      , bufferImageHeight( bufferImageHeight_ )
      , imageSubresource( imageSubresource_ )
      , imageOffset( imageOffset_ )
      , imageExtent( imageExtent_ )
    {
    }

    BufferImageCopy( VkBufferImageCopy const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferImageCopy ) );
    }

    BufferImageCopy& operator=( VkBufferImageCopy const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferImageCopy ) );
      return *this;
    }
    BufferImageCopy& setBufferOffset( DeviceSize bufferOffset_ )
    {
      bufferOffset = bufferOffset_;
      return *this;
    }

    BufferImageCopy& setBufferRowLength( uint32_t bufferRowLength_ )
    {
      bufferRowLength = bufferRowLength_;
      return *this;
    }

    BufferImageCopy& setBufferImageHeight( uint32_t bufferImageHeight_ )
    {
      bufferImageHeight = bufferImageHeight_;
      return *this;
    }

    BufferImageCopy& setImageSubresource( ImageSubresourceLayers imageSubresource_ )
    {
      imageSubresource = imageSubresource_;
      return *this;
    }

    BufferImageCopy& setImageOffset( Offset3D imageOffset_ )
    {
      imageOffset = imageOffset_;
      return *this;
    }

    BufferImageCopy& setImageExtent( Extent3D imageExtent_ )
    {
      imageExtent = imageExtent_;
      return *this;
    }

    operator VkBufferImageCopy const&() const
    {
      return *reinterpret_cast<const VkBufferImageCopy*>(this);
    }

    operator VkBufferImageCopy &()
    {
      return *reinterpret_cast<VkBufferImageCopy*>(this);
    }

    bool operator==( BufferImageCopy const& rhs ) const
    {
      return ( bufferOffset == rhs.bufferOffset )
          && ( bufferRowLength == rhs.bufferRowLength )
          && ( bufferImageHeight == rhs.bufferImageHeight )
          && ( imageSubresource == rhs.imageSubresource )
          && ( imageOffset == rhs.imageOffset )
          && ( imageExtent == rhs.imageExtent );
    }

    bool operator!=( BufferImageCopy const& rhs ) const
    {
      return !operator==( rhs );
    }

    DeviceSize bufferOffset;
    uint32_t bufferRowLength;
    uint32_t bufferImageHeight;
    ImageSubresourceLayers imageSubresource;
    Offset3D imageOffset;
    Extent3D imageExtent;
  };
  static_assert( sizeof( BufferImageCopy ) == sizeof( VkBufferImageCopy ), "struct and wrapper have different size!" );

  struct ImageResolve
  {
    ImageResolve( ImageSubresourceLayers srcSubresource_ = ImageSubresourceLayers(),
                  Offset3D srcOffset_ = Offset3D(),
                  ImageSubresourceLayers dstSubresource_ = ImageSubresourceLayers(),
                  Offset3D dstOffset_ = Offset3D(),
                  Extent3D extent_ = Extent3D() )
      : srcSubresource( srcSubresource_ )
      , srcOffset( srcOffset_ )
      , dstSubresource( dstSubresource_ )
      , dstOffset( dstOffset_ )
      , extent( extent_ )
    {
    }

    ImageResolve( VkImageResolve const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageResolve ) );
    }

    ImageResolve& operator=( VkImageResolve const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageResolve ) );
      return *this;
    }
    ImageResolve& setSrcSubresource( ImageSubresourceLayers srcSubresource_ )
    {
      srcSubresource = srcSubresource_;
      return *this;
    }

    ImageResolve& setSrcOffset( Offset3D srcOffset_ )
    {
      srcOffset = srcOffset_;
      return *this;
    }

    ImageResolve& setDstSubresource( ImageSubresourceLayers dstSubresource_ )
    {
      dstSubresource = dstSubresource_;
      return *this;
    }

    ImageResolve& setDstOffset( Offset3D dstOffset_ )
    {
      dstOffset = dstOffset_;
      return *this;
    }

    ImageResolve& setExtent( Extent3D extent_ )
    {
      extent = extent_;
      return *this;
    }

    operator VkImageResolve const&() const
    {
      return *reinterpret_cast<const VkImageResolve*>(this);
    }

    operator VkImageResolve &()
    {
      return *reinterpret_cast<VkImageResolve*>(this);
    }

    bool operator==( ImageResolve const& rhs ) const
    {
      return ( srcSubresource == rhs.srcSubresource )
          && ( srcOffset == rhs.srcOffset )
          && ( dstSubresource == rhs.dstSubresource )
          && ( dstOffset == rhs.dstOffset )
          && ( extent == rhs.extent );
    }

    bool operator!=( ImageResolve const& rhs ) const
    {
      return !operator==( rhs );
    }

    ImageSubresourceLayers srcSubresource;
    Offset3D srcOffset;
    ImageSubresourceLayers dstSubresource;
    Offset3D dstOffset;
    Extent3D extent;
  };
  static_assert( sizeof( ImageResolve ) == sizeof( VkImageResolve ), "struct and wrapper have different size!" );

  struct ClearAttachment
  {
    ClearAttachment( ImageAspectFlags aspectMask_ = ImageAspectFlags(),
                     uint32_t colorAttachment_ = 0,
                     ClearValue clearValue_ = ClearValue() )
      : aspectMask( aspectMask_ )
      , colorAttachment( colorAttachment_ )
      , clearValue( clearValue_ )
    {
    }

    ClearAttachment( VkClearAttachment const & rhs )
    {
      memcpy( this, &rhs, sizeof( ClearAttachment ) );
    }

    ClearAttachment& operator=( VkClearAttachment const & rhs )
    {
      memcpy( this, &rhs, sizeof( ClearAttachment ) );
      return *this;
    }
    ClearAttachment& setAspectMask( ImageAspectFlags aspectMask_ )
    {
      aspectMask = aspectMask_;
      return *this;
    }

    ClearAttachment& setColorAttachment( uint32_t colorAttachment_ )
    {
      colorAttachment = colorAttachment_;
      return *this;
    }

    ClearAttachment& setClearValue( ClearValue clearValue_ )
    {
      clearValue = clearValue_;
      return *this;
    }

    operator VkClearAttachment const&() const
    {
      return *reinterpret_cast<const VkClearAttachment*>(this);
    }

    operator VkClearAttachment &()
    {
      return *reinterpret_cast<VkClearAttachment*>(this);
    }

    ImageAspectFlags aspectMask;
    uint32_t colorAttachment;
    ClearValue clearValue;
  };
  static_assert( sizeof( ClearAttachment ) == sizeof( VkClearAttachment ), "struct and wrapper have different size!" );

  struct InputAttachmentAspectReference
  {
    InputAttachmentAspectReference( uint32_t subpass_ = 0,
                                    uint32_t inputAttachmentIndex_ = 0,
                                    ImageAspectFlags aspectMask_ = ImageAspectFlags() )
      : subpass( subpass_ )
      , inputAttachmentIndex( inputAttachmentIndex_ )
      , aspectMask( aspectMask_ )
    {
    }

    InputAttachmentAspectReference( VkInputAttachmentAspectReference const & rhs )
    {
      memcpy( this, &rhs, sizeof( InputAttachmentAspectReference ) );
    }

    InputAttachmentAspectReference& operator=( VkInputAttachmentAspectReference const & rhs )
    {
      memcpy( this, &rhs, sizeof( InputAttachmentAspectReference ) );
      return *this;
    }
    InputAttachmentAspectReference& setSubpass( uint32_t subpass_ )
    {
      subpass = subpass_;
      return *this;
    }

    InputAttachmentAspectReference& setInputAttachmentIndex( uint32_t inputAttachmentIndex_ )
    {
      inputAttachmentIndex = inputAttachmentIndex_;
      return *this;
    }

    InputAttachmentAspectReference& setAspectMask( ImageAspectFlags aspectMask_ )
    {
      aspectMask = aspectMask_;
      return *this;
    }

    operator VkInputAttachmentAspectReference const&() const
    {
      return *reinterpret_cast<const VkInputAttachmentAspectReference*>(this);
    }

    operator VkInputAttachmentAspectReference &()
    {
      return *reinterpret_cast<VkInputAttachmentAspectReference*>(this);
    }

    bool operator==( InputAttachmentAspectReference const& rhs ) const
    {
      return ( subpass == rhs.subpass )
          && ( inputAttachmentIndex == rhs.inputAttachmentIndex )
          && ( aspectMask == rhs.aspectMask );
    }

    bool operator!=( InputAttachmentAspectReference const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t subpass;
    uint32_t inputAttachmentIndex;
    ImageAspectFlags aspectMask;
  };
  static_assert( sizeof( InputAttachmentAspectReference ) == sizeof( VkInputAttachmentAspectReference ), "struct and wrapper have different size!" );

  using InputAttachmentAspectReferenceKHR = InputAttachmentAspectReference;

  struct RenderPassInputAttachmentAspectCreateInfo
  {
    RenderPassInputAttachmentAspectCreateInfo( uint32_t aspectReferenceCount_ = 0,
                                               const InputAttachmentAspectReference* pAspectReferences_ = nullptr )
      : aspectReferenceCount( aspectReferenceCount_ )
      , pAspectReferences( pAspectReferences_ )
    {
    }

    RenderPassInputAttachmentAspectCreateInfo( VkRenderPassInputAttachmentAspectCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( RenderPassInputAttachmentAspectCreateInfo ) );
    }

    RenderPassInputAttachmentAspectCreateInfo& operator=( VkRenderPassInputAttachmentAspectCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( RenderPassInputAttachmentAspectCreateInfo ) );
      return *this;
    }
    RenderPassInputAttachmentAspectCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    RenderPassInputAttachmentAspectCreateInfo& setAspectReferenceCount( uint32_t aspectReferenceCount_ )
    {
      aspectReferenceCount = aspectReferenceCount_;
      return *this;
    }

    RenderPassInputAttachmentAspectCreateInfo& setPAspectReferences( const InputAttachmentAspectReference* pAspectReferences_ )
    {
      pAspectReferences = pAspectReferences_;
      return *this;
    }

    operator VkRenderPassInputAttachmentAspectCreateInfo const&() const
    {
      return *reinterpret_cast<const VkRenderPassInputAttachmentAspectCreateInfo*>(this);
    }

    operator VkRenderPassInputAttachmentAspectCreateInfo &()
    {
      return *reinterpret_cast<VkRenderPassInputAttachmentAspectCreateInfo*>(this);
    }

    bool operator==( RenderPassInputAttachmentAspectCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( aspectReferenceCount == rhs.aspectReferenceCount )
          && ( pAspectReferences == rhs.pAspectReferences );
    }

    bool operator!=( RenderPassInputAttachmentAspectCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eRenderPassInputAttachmentAspectCreateInfo;

  public:
    const void* pNext = nullptr;
    uint32_t aspectReferenceCount;
    const InputAttachmentAspectReference* pAspectReferences;
  };
  static_assert( sizeof( RenderPassInputAttachmentAspectCreateInfo ) == sizeof( VkRenderPassInputAttachmentAspectCreateInfo ), "struct and wrapper have different size!" );

  using RenderPassInputAttachmentAspectCreateInfoKHR = RenderPassInputAttachmentAspectCreateInfo;

  struct BindImagePlaneMemoryInfo
  {
    BindImagePlaneMemoryInfo( ImageAspectFlagBits planeAspect_ = ImageAspectFlagBits::eColor )
      : planeAspect( planeAspect_ )
    {
    }

    BindImagePlaneMemoryInfo( VkBindImagePlaneMemoryInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImagePlaneMemoryInfo ) );
    }

    BindImagePlaneMemoryInfo& operator=( VkBindImagePlaneMemoryInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImagePlaneMemoryInfo ) );
      return *this;
    }
    BindImagePlaneMemoryInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindImagePlaneMemoryInfo& setPlaneAspect( ImageAspectFlagBits planeAspect_ )
    {
      planeAspect = planeAspect_;
      return *this;
    }

    operator VkBindImagePlaneMemoryInfo const&() const
    {
      return *reinterpret_cast<const VkBindImagePlaneMemoryInfo*>(this);
    }

    operator VkBindImagePlaneMemoryInfo &()
    {
      return *reinterpret_cast<VkBindImagePlaneMemoryInfo*>(this);
    }

    bool operator==( BindImagePlaneMemoryInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( planeAspect == rhs.planeAspect );
    }

    bool operator!=( BindImagePlaneMemoryInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindImagePlaneMemoryInfo;

  public:
    const void* pNext = nullptr;
    ImageAspectFlagBits planeAspect;
  };
  static_assert( sizeof( BindImagePlaneMemoryInfo ) == sizeof( VkBindImagePlaneMemoryInfo ), "struct and wrapper have different size!" );

  using BindImagePlaneMemoryInfoKHR = BindImagePlaneMemoryInfo;

  struct ImagePlaneMemoryRequirementsInfo
  {
    ImagePlaneMemoryRequirementsInfo( ImageAspectFlagBits planeAspect_ = ImageAspectFlagBits::eColor )
      : planeAspect( planeAspect_ )
    {
    }

    ImagePlaneMemoryRequirementsInfo( VkImagePlaneMemoryRequirementsInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImagePlaneMemoryRequirementsInfo ) );
    }

    ImagePlaneMemoryRequirementsInfo& operator=( VkImagePlaneMemoryRequirementsInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImagePlaneMemoryRequirementsInfo ) );
      return *this;
    }
    ImagePlaneMemoryRequirementsInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImagePlaneMemoryRequirementsInfo& setPlaneAspect( ImageAspectFlagBits planeAspect_ )
    {
      planeAspect = planeAspect_;
      return *this;
    }

    operator VkImagePlaneMemoryRequirementsInfo const&() const
    {
      return *reinterpret_cast<const VkImagePlaneMemoryRequirementsInfo*>(this);
    }

    operator VkImagePlaneMemoryRequirementsInfo &()
    {
      return *reinterpret_cast<VkImagePlaneMemoryRequirementsInfo*>(this);
    }

    bool operator==( ImagePlaneMemoryRequirementsInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( planeAspect == rhs.planeAspect );
    }

    bool operator!=( ImagePlaneMemoryRequirementsInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImagePlaneMemoryRequirementsInfo;

  public:
    const void* pNext = nullptr;
    ImageAspectFlagBits planeAspect;
  };
  static_assert( sizeof( ImagePlaneMemoryRequirementsInfo ) == sizeof( VkImagePlaneMemoryRequirementsInfo ), "struct and wrapper have different size!" );

  using ImagePlaneMemoryRequirementsInfoKHR = ImagePlaneMemoryRequirementsInfo;

  struct AttachmentReference2KHR
  {
    AttachmentReference2KHR( uint32_t attachment_ = 0,
                             ImageLayout layout_ = ImageLayout::eUndefined,
                             ImageAspectFlags aspectMask_ = ImageAspectFlags() )
      : attachment( attachment_ )
      , layout( layout_ )
      , aspectMask( aspectMask_ )
    {
    }

    AttachmentReference2KHR( VkAttachmentReference2KHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentReference2KHR ) );
    }

    AttachmentReference2KHR& operator=( VkAttachmentReference2KHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentReference2KHR ) );
      return *this;
    }
    AttachmentReference2KHR& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    AttachmentReference2KHR& setAttachment( uint32_t attachment_ )
    {
      attachment = attachment_;
      return *this;
    }

    AttachmentReference2KHR& setLayout( ImageLayout layout_ )
    {
      layout = layout_;
      return *this;
    }

    AttachmentReference2KHR& setAspectMask( ImageAspectFlags aspectMask_ )
    {
      aspectMask = aspectMask_;
      return *this;
    }

    operator VkAttachmentReference2KHR const&() const
    {
      return *reinterpret_cast<const VkAttachmentReference2KHR*>(this);
    }

    operator VkAttachmentReference2KHR &()
    {
      return *reinterpret_cast<VkAttachmentReference2KHR*>(this);
    }

    bool operator==( AttachmentReference2KHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( attachment == rhs.attachment )
          && ( layout == rhs.layout )
          && ( aspectMask == rhs.aspectMask );
    }

    bool operator!=( AttachmentReference2KHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAttachmentReference2KHR;

  public:
    const void* pNext = nullptr;
    uint32_t attachment;
    ImageLayout layout;
    ImageAspectFlags aspectMask;
  };
  static_assert( sizeof( AttachmentReference2KHR ) == sizeof( VkAttachmentReference2KHR ), "struct and wrapper have different size!" );

  enum class SparseImageFormatFlagBits
  {
    eSingleMiptail = VK_SPARSE_IMAGE_FORMAT_SINGLE_MIPTAIL_BIT,
    eAlignedMipSize = VK_SPARSE_IMAGE_FORMAT_ALIGNED_MIP_SIZE_BIT,
    eNonstandardBlockSize = VK_SPARSE_IMAGE_FORMAT_NONSTANDARD_BLOCK_SIZE_BIT
  };

  using SparseImageFormatFlags = Flags<SparseImageFormatFlagBits, VkSparseImageFormatFlags>;

  VULKAN_HPP_INLINE SparseImageFormatFlags operator|( SparseImageFormatFlagBits bit0, SparseImageFormatFlagBits bit1 )
  {
    return SparseImageFormatFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SparseImageFormatFlags operator~( SparseImageFormatFlagBits bits )
  {
    return ~( SparseImageFormatFlags( bits ) );
  }

  template <> struct FlagTraits<SparseImageFormatFlagBits>
  {
    enum
    {
      allFlags = VkFlags(SparseImageFormatFlagBits::eSingleMiptail) | VkFlags(SparseImageFormatFlagBits::eAlignedMipSize) | VkFlags(SparseImageFormatFlagBits::eNonstandardBlockSize)
    };
  };

  struct SparseImageFormatProperties
  {
    operator VkSparseImageFormatProperties const&() const
    {
      return *reinterpret_cast<const VkSparseImageFormatProperties*>(this);
    }

    operator VkSparseImageFormatProperties &()
    {
      return *reinterpret_cast<VkSparseImageFormatProperties*>(this);
    }

    bool operator==( SparseImageFormatProperties const& rhs ) const
    {
      return ( aspectMask == rhs.aspectMask )
          && ( imageGranularity == rhs.imageGranularity )
          && ( flags == rhs.flags );
    }

    bool operator!=( SparseImageFormatProperties const& rhs ) const
    {
      return !operator==( rhs );
    }

    ImageAspectFlags aspectMask;
    Extent3D imageGranularity;
    SparseImageFormatFlags flags;
  };
  static_assert( sizeof( SparseImageFormatProperties ) == sizeof( VkSparseImageFormatProperties ), "struct and wrapper have different size!" );

  struct SparseImageMemoryRequirements
  {
    operator VkSparseImageMemoryRequirements const&() const
    {
      return *reinterpret_cast<const VkSparseImageMemoryRequirements*>(this);
    }

    operator VkSparseImageMemoryRequirements &()
    {
      return *reinterpret_cast<VkSparseImageMemoryRequirements*>(this);
    }

    bool operator==( SparseImageMemoryRequirements const& rhs ) const
    {
      return ( formatProperties == rhs.formatProperties )
          && ( imageMipTailFirstLod == rhs.imageMipTailFirstLod )
          && ( imageMipTailSize == rhs.imageMipTailSize )
          && ( imageMipTailOffset == rhs.imageMipTailOffset )
          && ( imageMipTailStride == rhs.imageMipTailStride );
    }

    bool operator!=( SparseImageMemoryRequirements const& rhs ) const
    {
      return !operator==( rhs );
    }

    SparseImageFormatProperties formatProperties;
    uint32_t imageMipTailFirstLod;
    DeviceSize imageMipTailSize;
    DeviceSize imageMipTailOffset;
    DeviceSize imageMipTailStride;
  };
  static_assert( sizeof( SparseImageMemoryRequirements ) == sizeof( VkSparseImageMemoryRequirements ), "struct and wrapper have different size!" );

  struct SparseImageFormatProperties2
  {
    operator VkSparseImageFormatProperties2 const&() const
    {
      return *reinterpret_cast<const VkSparseImageFormatProperties2*>(this);
    }

    operator VkSparseImageFormatProperties2 &()
    {
      return *reinterpret_cast<VkSparseImageFormatProperties2*>(this);
    }

    bool operator==( SparseImageFormatProperties2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( properties == rhs.properties );
    }

    bool operator!=( SparseImageFormatProperties2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eSparseImageFormatProperties2;

  public:
    void* pNext = nullptr;
    SparseImageFormatProperties properties;
  };
  static_assert( sizeof( SparseImageFormatProperties2 ) == sizeof( VkSparseImageFormatProperties2 ), "struct and wrapper have different size!" );

  using SparseImageFormatProperties2KHR = SparseImageFormatProperties2;

  struct SparseImageMemoryRequirements2
  {
    operator VkSparseImageMemoryRequirements2 const&() const
    {
      return *reinterpret_cast<const VkSparseImageMemoryRequirements2*>(this);
    }

    operator VkSparseImageMemoryRequirements2 &()
    {
      return *reinterpret_cast<VkSparseImageMemoryRequirements2*>(this);
    }

    bool operator==( SparseImageMemoryRequirements2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( memoryRequirements == rhs.memoryRequirements );
    }

    bool operator!=( SparseImageMemoryRequirements2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eSparseImageMemoryRequirements2;

  public:
    void* pNext = nullptr;
    SparseImageMemoryRequirements memoryRequirements;
  };
  static_assert( sizeof( SparseImageMemoryRequirements2 ) == sizeof( VkSparseImageMemoryRequirements2 ), "struct and wrapper have different size!" );

  using SparseImageMemoryRequirements2KHR = SparseImageMemoryRequirements2;

  enum class SparseMemoryBindFlagBits
  {
    eMetadata = VK_SPARSE_MEMORY_BIND_METADATA_BIT
  };

  using SparseMemoryBindFlags = Flags<SparseMemoryBindFlagBits, VkSparseMemoryBindFlags>;

  VULKAN_HPP_INLINE SparseMemoryBindFlags operator|( SparseMemoryBindFlagBits bit0, SparseMemoryBindFlagBits bit1 )
  {
    return SparseMemoryBindFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SparseMemoryBindFlags operator~( SparseMemoryBindFlagBits bits )
  {
    return ~( SparseMemoryBindFlags( bits ) );
  }

  template <> struct FlagTraits<SparseMemoryBindFlagBits>
  {
    enum
    {
      allFlags = VkFlags(SparseMemoryBindFlagBits::eMetadata)
    };
  };

  struct SparseMemoryBind
  {
    SparseMemoryBind( DeviceSize resourceOffset_ = 0,
                      DeviceSize size_ = 0,
                      DeviceMemory memory_ = DeviceMemory(),
                      DeviceSize memoryOffset_ = 0,
                      SparseMemoryBindFlags flags_ = SparseMemoryBindFlags() )
      : resourceOffset( resourceOffset_ )
      , size( size_ )
      , memory( memory_ )
      , memoryOffset( memoryOffset_ )
      , flags( flags_ )
    {
    }

    SparseMemoryBind( VkSparseMemoryBind const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseMemoryBind ) );
    }

    SparseMemoryBind& operator=( VkSparseMemoryBind const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseMemoryBind ) );
      return *this;
    }
    SparseMemoryBind& setResourceOffset( DeviceSize resourceOffset_ )
    {
      resourceOffset = resourceOffset_;
      return *this;
    }

    SparseMemoryBind& setSize( DeviceSize size_ )
    {
      size = size_;
      return *this;
    }

    SparseMemoryBind& setMemory( DeviceMemory memory_ )
    {
      memory = memory_;
      return *this;
    }

    SparseMemoryBind& setMemoryOffset( DeviceSize memoryOffset_ )
    {
      memoryOffset = memoryOffset_;
      return *this;
    }

    SparseMemoryBind& setFlags( SparseMemoryBindFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    operator VkSparseMemoryBind const&() const
    {
      return *reinterpret_cast<const VkSparseMemoryBind*>(this);
    }

    operator VkSparseMemoryBind &()
    {
      return *reinterpret_cast<VkSparseMemoryBind*>(this);
    }

    bool operator==( SparseMemoryBind const& rhs ) const
    {
      return ( resourceOffset == rhs.resourceOffset )
          && ( size == rhs.size )
          && ( memory == rhs.memory )
          && ( memoryOffset == rhs.memoryOffset )
          && ( flags == rhs.flags );
    }

    bool operator!=( SparseMemoryBind const& rhs ) const
    {
      return !operator==( rhs );
    }

    DeviceSize resourceOffset;
    DeviceSize size;
    DeviceMemory memory;
    DeviceSize memoryOffset;
    SparseMemoryBindFlags flags;
  };
  static_assert( sizeof( SparseMemoryBind ) == sizeof( VkSparseMemoryBind ), "struct and wrapper have different size!" );

  struct SparseImageMemoryBind
  {
    SparseImageMemoryBind( ImageSubresource subresource_ = ImageSubresource(),
                           Offset3D offset_ = Offset3D(),
                           Extent3D extent_ = Extent3D(),
                           DeviceMemory memory_ = DeviceMemory(),
                           DeviceSize memoryOffset_ = 0,
                           SparseMemoryBindFlags flags_ = SparseMemoryBindFlags() )
      : subresource( subresource_ )
      , offset( offset_ )
      , extent( extent_ )
      , memory( memory_ )
      , memoryOffset( memoryOffset_ )
      , flags( flags_ )
    {
    }

    SparseImageMemoryBind( VkSparseImageMemoryBind const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseImageMemoryBind ) );
    }

    SparseImageMemoryBind& operator=( VkSparseImageMemoryBind const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseImageMemoryBind ) );
      return *this;
    }
    SparseImageMemoryBind& setSubresource( ImageSubresource subresource_ )
    {
      subresource = subresource_;
      return *this;
    }

    SparseImageMemoryBind& setOffset( Offset3D offset_ )
    {
      offset = offset_;
      return *this;
    }

    SparseImageMemoryBind& setExtent( Extent3D extent_ )
    {
      extent = extent_;
      return *this;
    }

    SparseImageMemoryBind& setMemory( DeviceMemory memory_ )
    {
      memory = memory_;
      return *this;
    }

    SparseImageMemoryBind& setMemoryOffset( DeviceSize memoryOffset_ )
    {
      memoryOffset = memoryOffset_;
      return *this;
    }

    SparseImageMemoryBind& setFlags( SparseMemoryBindFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    operator VkSparseImageMemoryBind const&() const
    {
      return *reinterpret_cast<const VkSparseImageMemoryBind*>(this);
    }

    operator VkSparseImageMemoryBind &()
    {
      return *reinterpret_cast<VkSparseImageMemoryBind*>(this);
    }

    bool operator==( SparseImageMemoryBind const& rhs ) const
    {
      return ( subresource == rhs.subresource )
          && ( offset == rhs.offset )
          && ( extent == rhs.extent )
          && ( memory == rhs.memory )
          && ( memoryOffset == rhs.memoryOffset )
          && ( flags == rhs.flags );
    }

    bool operator!=( SparseImageMemoryBind const& rhs ) const
    {
      return !operator==( rhs );
    }

    ImageSubresource subresource;
    Offset3D offset;
    Extent3D extent;
    DeviceMemory memory;
    DeviceSize memoryOffset;
    SparseMemoryBindFlags flags;
  };
  static_assert( sizeof( SparseImageMemoryBind ) == sizeof( VkSparseImageMemoryBind ), "struct and wrapper have different size!" );

  struct SparseBufferMemoryBindInfo
  {
    SparseBufferMemoryBindInfo( Buffer buffer_ = Buffer(),
                                uint32_t bindCount_ = 0,
                                const SparseMemoryBind* pBinds_ = nullptr )
      : buffer( buffer_ )
      , bindCount( bindCount_ )
      , pBinds( pBinds_ )
    {
    }

    SparseBufferMemoryBindInfo( VkSparseBufferMemoryBindInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseBufferMemoryBindInfo ) );
    }

    SparseBufferMemoryBindInfo& operator=( VkSparseBufferMemoryBindInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseBufferMemoryBindInfo ) );
      return *this;
    }
    SparseBufferMemoryBindInfo& setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    SparseBufferMemoryBindInfo& setBindCount( uint32_t bindCount_ )
    {
      bindCount = bindCount_;
      return *this;
    }

    SparseBufferMemoryBindInfo& setPBinds( const SparseMemoryBind* pBinds_ )
    {
      pBinds = pBinds_;
      return *this;
    }

    operator VkSparseBufferMemoryBindInfo const&() const
    {
      return *reinterpret_cast<const VkSparseBufferMemoryBindInfo*>(this);
    }

    operator VkSparseBufferMemoryBindInfo &()
    {
      return *reinterpret_cast<VkSparseBufferMemoryBindInfo*>(this);
    }

    bool operator==( SparseBufferMemoryBindInfo const& rhs ) const
    {
      return ( buffer == rhs.buffer )
          && ( bindCount == rhs.bindCount )
          && ( pBinds == rhs.pBinds );
    }

    bool operator!=( SparseBufferMemoryBindInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

    Buffer buffer;
    uint32_t bindCount;
    const SparseMemoryBind* pBinds;
  };
  static_assert( sizeof( SparseBufferMemoryBindInfo ) == sizeof( VkSparseBufferMemoryBindInfo ), "struct and wrapper have different size!" );

  struct SparseImageOpaqueMemoryBindInfo
  {
    SparseImageOpaqueMemoryBindInfo( Image image_ = Image(),
                                     uint32_t bindCount_ = 0,
                                     const SparseMemoryBind* pBinds_ = nullptr )
      : image( image_ )
      , bindCount( bindCount_ )
      , pBinds( pBinds_ )
    {
    }

    SparseImageOpaqueMemoryBindInfo( VkSparseImageOpaqueMemoryBindInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseImageOpaqueMemoryBindInfo ) );
    }

    SparseImageOpaqueMemoryBindInfo& operator=( VkSparseImageOpaqueMemoryBindInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseImageOpaqueMemoryBindInfo ) );
      return *this;
    }
    SparseImageOpaqueMemoryBindInfo& setImage( Image image_ )
    {
      image = image_;
      return *this;
    }

    SparseImageOpaqueMemoryBindInfo& setBindCount( uint32_t bindCount_ )
    {
      bindCount = bindCount_;
      return *this;
    }

    SparseImageOpaqueMemoryBindInfo& setPBinds( const SparseMemoryBind* pBinds_ )
    {
      pBinds = pBinds_;
      return *this;
    }

    operator VkSparseImageOpaqueMemoryBindInfo const&() const
    {
      return *reinterpret_cast<const VkSparseImageOpaqueMemoryBindInfo*>(this);
    }

    operator VkSparseImageOpaqueMemoryBindInfo &()
    {
      return *reinterpret_cast<VkSparseImageOpaqueMemoryBindInfo*>(this);
    }

    bool operator==( SparseImageOpaqueMemoryBindInfo const& rhs ) const
    {
      return ( image == rhs.image )
          && ( bindCount == rhs.bindCount )
          && ( pBinds == rhs.pBinds );
    }

    bool operator!=( SparseImageOpaqueMemoryBindInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

    Image image;
    uint32_t bindCount;
    const SparseMemoryBind* pBinds;
  };
  static_assert( sizeof( SparseImageOpaqueMemoryBindInfo ) == sizeof( VkSparseImageOpaqueMemoryBindInfo ), "struct and wrapper have different size!" );

  struct SparseImageMemoryBindInfo
  {
    SparseImageMemoryBindInfo( Image image_ = Image(),
                               uint32_t bindCount_ = 0,
                               const SparseImageMemoryBind* pBinds_ = nullptr )
      : image( image_ )
      , bindCount( bindCount_ )
      , pBinds( pBinds_ )
    {
    }

    SparseImageMemoryBindInfo( VkSparseImageMemoryBindInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseImageMemoryBindInfo ) );
    }

    SparseImageMemoryBindInfo& operator=( VkSparseImageMemoryBindInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseImageMemoryBindInfo ) );
      return *this;
    }
    SparseImageMemoryBindInfo& setImage( Image image_ )
    {
      image = image_;
      return *this;
    }

    SparseImageMemoryBindInfo& setBindCount( uint32_t bindCount_ )
    {
      bindCount = bindCount_;
      return *this;
    }

    SparseImageMemoryBindInfo& setPBinds( const SparseImageMemoryBind* pBinds_ )
    {
      pBinds = pBinds_;
      return *this;
    }

    operator VkSparseImageMemoryBindInfo const&() const
    {
      return *reinterpret_cast<const VkSparseImageMemoryBindInfo*>(this);
    }

    operator VkSparseImageMemoryBindInfo &()
    {
      return *reinterpret_cast<VkSparseImageMemoryBindInfo*>(this);
    }

    bool operator==( SparseImageMemoryBindInfo const& rhs ) const
    {
      return ( image == rhs.image )
          && ( bindCount == rhs.bindCount )
          && ( pBinds == rhs.pBinds );
    }

    bool operator!=( SparseImageMemoryBindInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

    Image image;
    uint32_t bindCount;
    const SparseImageMemoryBind* pBinds;
  };
  static_assert( sizeof( SparseImageMemoryBindInfo ) == sizeof( VkSparseImageMemoryBindInfo ), "struct and wrapper have different size!" );

  struct BindSparseInfo
  {
    BindSparseInfo( uint32_t waitSemaphoreCount_ = 0,
                    const Semaphore* pWaitSemaphores_ = nullptr,
                    uint32_t bufferBindCount_ = 0,
                    const SparseBufferMemoryBindInfo* pBufferBinds_ = nullptr,
                    uint32_t imageOpaqueBindCount_ = 0,
                    const SparseImageOpaqueMemoryBindInfo* pImageOpaqueBinds_ = nullptr,
                    uint32_t imageBindCount_ = 0,
                    const SparseImageMemoryBindInfo* pImageBinds_ = nullptr,
                    uint32_t signalSemaphoreCount_ = 0,
                    const Semaphore* pSignalSemaphores_ = nullptr )
      : waitSemaphoreCount( waitSemaphoreCount_ )
      , pWaitSemaphores( pWaitSemaphores_ )
      , bufferBindCount( bufferBindCount_ )
      , pBufferBinds( pBufferBinds_ )
      , imageOpaqueBindCount( imageOpaqueBindCount_ )
      , pImageOpaqueBinds( pImageOpaqueBinds_ )
      , imageBindCount( imageBindCount_ )
      , pImageBinds( pImageBinds_ )
      , signalSemaphoreCount( signalSemaphoreCount_ )
      , pSignalSemaphores( pSignalSemaphores_ )
    {
    }

    BindSparseInfo( VkBindSparseInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindSparseInfo ) );
    }

    BindSparseInfo& operator=( VkBindSparseInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindSparseInfo ) );
      return *this;
    }
    BindSparseInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindSparseInfo& setWaitSemaphoreCount( uint32_t waitSemaphoreCount_ )
    {
      waitSemaphoreCount = waitSemaphoreCount_;
      return *this;
    }

    BindSparseInfo& setPWaitSemaphores( const Semaphore* pWaitSemaphores_ )
    {
      pWaitSemaphores = pWaitSemaphores_;
      return *this;
    }

    BindSparseInfo& setBufferBindCount( uint32_t bufferBindCount_ )
    {
      bufferBindCount = bufferBindCount_;
      return *this;
    }

    BindSparseInfo& setPBufferBinds( const SparseBufferMemoryBindInfo* pBufferBinds_ )
    {
      pBufferBinds = pBufferBinds_;
      return *this;
    }

    BindSparseInfo& setImageOpaqueBindCount( uint32_t imageOpaqueBindCount_ )
    {
      imageOpaqueBindCount = imageOpaqueBindCount_;
      return *this;
    }

    BindSparseInfo& setPImageOpaqueBinds( const SparseImageOpaqueMemoryBindInfo* pImageOpaqueBinds_ )
    {
      pImageOpaqueBinds = pImageOpaqueBinds_;
      return *this;
    }

    BindSparseInfo& setImageBindCount( uint32_t imageBindCount_ )
    {
      imageBindCount = imageBindCount_;
      return *this;
    }

    BindSparseInfo& setPImageBinds( const SparseImageMemoryBindInfo* pImageBinds_ )
    {
      pImageBinds = pImageBinds_;
      return *this;
    }

    BindSparseInfo& setSignalSemaphoreCount( uint32_t signalSemaphoreCount_ )
    {
      signalSemaphoreCount = signalSemaphoreCount_;
      return *this;
    }

    BindSparseInfo& setPSignalSemaphores( const Semaphore* pSignalSemaphores_ )
    {
      pSignalSemaphores = pSignalSemaphores_;
      return *this;
    }

    operator VkBindSparseInfo const&() const
    {
      return *reinterpret_cast<const VkBindSparseInfo*>(this);
    }

    operator VkBindSparseInfo &()
    {
      return *reinterpret_cast<VkBindSparseInfo*>(this);
    }

    bool operator==( BindSparseInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( waitSemaphoreCount == rhs.waitSemaphoreCount )
          && ( pWaitSemaphores == rhs.pWaitSemaphores )
          && ( bufferBindCount == rhs.bufferBindCount )
          && ( pBufferBinds == rhs.pBufferBinds )
          && ( imageOpaqueBindCount == rhs.imageOpaqueBindCount )
          && ( pImageOpaqueBinds == rhs.pImageOpaqueBinds )
          && ( imageBindCount == rhs.imageBindCount )
          && ( pImageBinds == rhs.pImageBinds )
          && ( signalSemaphoreCount == rhs.signalSemaphoreCount )
          && ( pSignalSemaphores == rhs.pSignalSemaphores );
    }

    bool operator!=( BindSparseInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindSparseInfo;

  public:
    const void* pNext = nullptr;
    uint32_t waitSemaphoreCount;
    const Semaphore* pWaitSemaphores;
    uint32_t bufferBindCount;
    const SparseBufferMemoryBindInfo* pBufferBinds;
    uint32_t imageOpaqueBindCount;
    const SparseImageOpaqueMemoryBindInfo* pImageOpaqueBinds;
    uint32_t imageBindCount;
    const SparseImageMemoryBindInfo* pImageBinds;
    uint32_t signalSemaphoreCount;
    const Semaphore* pSignalSemaphores;
  };
  static_assert( sizeof( BindSparseInfo ) == sizeof( VkBindSparseInfo ), "struct and wrapper have different size!" );

  enum class PipelineStageFlagBits
  {
    eTopOfPipe = VK_PIPELINE_STAGE_TOP_OF_PIPE_BIT,
    eDrawIndirect = VK_PIPELINE_STAGE_DRAW_INDIRECT_BIT,
    eVertexInput = VK_PIPELINE_STAGE_VERTEX_INPUT_BIT,
    eVertexShader = VK_PIPELINE_STAGE_VERTEX_SHADER_BIT,
    eTessellationControlShader = VK_PIPELINE_STAGE_TESSELLATION_CONTROL_SHADER_BIT,
    eTessellationEvaluationShader = VK_PIPELINE_STAGE_TESSELLATION_EVALUATION_SHADER_BIT,
    eGeometryShader = VK_PIPELINE_STAGE_GEOMETRY_SHADER_BIT,
    eFragmentShader = VK_PIPELINE_STAGE_FRAGMENT_SHADER_BIT,
    eEarlyFragmentTests = VK_PIPELINE_STAGE_EARLY_FRAGMENT_TESTS_BIT,
    eLateFragmentTests = VK_PIPELINE_STAGE_LATE_FRAGMENT_TESTS_BIT,
    eColorAttachmentOutput = VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT,
    eComputeShader = VK_PIPELINE_STAGE_COMPUTE_SHADER_BIT,
    eTransfer = VK_PIPELINE_STAGE_TRANSFER_BIT,
    eBottomOfPipe = VK_PIPELINE_STAGE_BOTTOM_OF_PIPE_BIT,
    eHost = VK_PIPELINE_STAGE_HOST_BIT,
    eAllGraphics = VK_PIPELINE_STAGE_ALL_GRAPHICS_BIT,
    eAllCommands = VK_PIPELINE_STAGE_ALL_COMMANDS_BIT,
    eTransformFeedbackEXT = VK_PIPELINE_STAGE_TRANSFORM_FEEDBACK_BIT_EXT,
    eConditionalRenderingEXT = VK_PIPELINE_STAGE_CONDITIONAL_RENDERING_BIT_EXT,
    eCommandProcessNVX = VK_PIPELINE_STAGE_COMMAND_PROCESS_BIT_NVX,
    eShadingRateImageNV = VK_PIPELINE_STAGE_SHADING_RATE_IMAGE_BIT_NV,
    eRayTracingShaderNV = VK_PIPELINE_STAGE_RAY_TRACING_SHADER_BIT_NV,
    eAccelerationStructureBuildNV = VK_PIPELINE_STAGE_ACCELERATION_STRUCTURE_BUILD_BIT_NV,
    eTaskShaderNV = VK_PIPELINE_STAGE_TASK_SHADER_BIT_NV,
    eMeshShaderNV = VK_PIPELINE_STAGE_MESH_SHADER_BIT_NV,
    eFragmentDensityProcessEXT = VK_PIPELINE_STAGE_FRAGMENT_DENSITY_PROCESS_BIT_EXT
  };

  using PipelineStageFlags = Flags<PipelineStageFlagBits, VkPipelineStageFlags>;

  VULKAN_HPP_INLINE PipelineStageFlags operator|( PipelineStageFlagBits bit0, PipelineStageFlagBits bit1 )
  {
    return PipelineStageFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE PipelineStageFlags operator~( PipelineStageFlagBits bits )
  {
    return ~( PipelineStageFlags( bits ) );
  }

  template <> struct FlagTraits<PipelineStageFlagBits>
  {
    enum
    {
      allFlags = VkFlags(PipelineStageFlagBits::eTopOfPipe) | VkFlags(PipelineStageFlagBits::eDrawIndirect) | VkFlags(PipelineStageFlagBits::eVertexInput) | VkFlags(PipelineStageFlagBits::eVertexShader) | VkFlags(PipelineStageFlagBits::eTessellationControlShader) | VkFlags(PipelineStageFlagBits::eTessellationEvaluationShader) | VkFlags(PipelineStageFlagBits::eGeometryShader) | VkFlags(PipelineStageFlagBits::eFragmentShader) | VkFlags(PipelineStageFlagBits::eEarlyFragmentTests) | VkFlags(PipelineStageFlagBits::eLateFragmentTests) | VkFlags(PipelineStageFlagBits::eColorAttachmentOutput) | VkFlags(PipelineStageFlagBits::eComputeShader) | VkFlags(PipelineStageFlagBits::eTransfer) | VkFlags(PipelineStageFlagBits::eBottomOfPipe) | VkFlags(PipelineStageFlagBits::eHost) | VkFlags(PipelineStageFlagBits::eAllGraphics) | VkFlags(PipelineStageFlagBits::eAllCommands) | VkFlags(PipelineStageFlagBits::eTransformFeedbackEXT) | VkFlags(PipelineStageFlagBits::eConditionalRenderingEXT) | VkFlags(PipelineStageFlagBits::eCommandProcessNVX) | VkFlags(PipelineStageFlagBits::eShadingRateImageNV) | VkFlags(PipelineStageFlagBits::eRayTracingShaderNV) | VkFlags(PipelineStageFlagBits::eAccelerationStructureBuildNV) | VkFlags(PipelineStageFlagBits::eTaskShaderNV) | VkFlags(PipelineStageFlagBits::eMeshShaderNV) | VkFlags(PipelineStageFlagBits::eFragmentDensityProcessEXT)
    };
  };

  struct QueueFamilyCheckpointPropertiesNV
  {
    operator VkQueueFamilyCheckpointPropertiesNV const&() const
    {
      return *reinterpret_cast<const VkQueueFamilyCheckpointPropertiesNV*>(this);
    }

    operator VkQueueFamilyCheckpointPropertiesNV &()
    {
      return *reinterpret_cast<VkQueueFamilyCheckpointPropertiesNV*>(this);
    }

    bool operator==( QueueFamilyCheckpointPropertiesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( checkpointExecutionStageMask == rhs.checkpointExecutionStageMask );
    }

    bool operator!=( QueueFamilyCheckpointPropertiesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eQueueFamilyCheckpointPropertiesNV;

  public:
    void* pNext = nullptr;
    PipelineStageFlags checkpointExecutionStageMask;
  };
  static_assert( sizeof( QueueFamilyCheckpointPropertiesNV ) == sizeof( VkQueueFamilyCheckpointPropertiesNV ), "struct and wrapper have different size!" );

  struct CheckpointDataNV
  {
    operator VkCheckpointDataNV const&() const
    {
      return *reinterpret_cast<const VkCheckpointDataNV*>(this);
    }

    operator VkCheckpointDataNV &()
    {
      return *reinterpret_cast<VkCheckpointDataNV*>(this);
    }

    bool operator==( CheckpointDataNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( stage == rhs.stage )
          && ( pCheckpointMarker == rhs.pCheckpointMarker );
    }

    bool operator!=( CheckpointDataNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCheckpointDataNV;

  public:
    void* pNext = nullptr;
    PipelineStageFlagBits stage;
    void* pCheckpointMarker;
  };
  static_assert( sizeof( CheckpointDataNV ) == sizeof( VkCheckpointDataNV ), "struct and wrapper have different size!" );

  enum class CommandPoolCreateFlagBits
  {
    eTransient = VK_COMMAND_POOL_CREATE_TRANSIENT_BIT,
    eResetCommandBuffer = VK_COMMAND_POOL_CREATE_RESET_COMMAND_BUFFER_BIT,
    eProtected = VK_COMMAND_POOL_CREATE_PROTECTED_BIT
  };

  using CommandPoolCreateFlags = Flags<CommandPoolCreateFlagBits, VkCommandPoolCreateFlags>;

  VULKAN_HPP_INLINE CommandPoolCreateFlags operator|( CommandPoolCreateFlagBits bit0, CommandPoolCreateFlagBits bit1 )
  {
    return CommandPoolCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE CommandPoolCreateFlags operator~( CommandPoolCreateFlagBits bits )
  {
    return ~( CommandPoolCreateFlags( bits ) );
  }

  template <> struct FlagTraits<CommandPoolCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(CommandPoolCreateFlagBits::eTransient) | VkFlags(CommandPoolCreateFlagBits::eResetCommandBuffer) | VkFlags(CommandPoolCreateFlagBits::eProtected)
    };
  };

  struct CommandPoolCreateInfo
  {
    CommandPoolCreateInfo( CommandPoolCreateFlags flags_ = CommandPoolCreateFlags(),
                           uint32_t queueFamilyIndex_ = 0 )
      : flags( flags_ )
      , queueFamilyIndex( queueFamilyIndex_ )
    {
    }

    CommandPoolCreateInfo( VkCommandPoolCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandPoolCreateInfo ) );
    }

    CommandPoolCreateInfo& operator=( VkCommandPoolCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandPoolCreateInfo ) );
      return *this;
    }
    CommandPoolCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CommandPoolCreateInfo& setFlags( CommandPoolCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    CommandPoolCreateInfo& setQueueFamilyIndex( uint32_t queueFamilyIndex_ )
    {
      queueFamilyIndex = queueFamilyIndex_;
      return *this;
    }

    operator VkCommandPoolCreateInfo const&() const
    {
      return *reinterpret_cast<const VkCommandPoolCreateInfo*>(this);
    }

    operator VkCommandPoolCreateInfo &()
    {
      return *reinterpret_cast<VkCommandPoolCreateInfo*>(this);
    }

    bool operator==( CommandPoolCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( queueFamilyIndex == rhs.queueFamilyIndex );
    }

    bool operator!=( CommandPoolCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCommandPoolCreateInfo;

  public:
    const void* pNext = nullptr;
    CommandPoolCreateFlags flags;
    uint32_t queueFamilyIndex;
  };
  static_assert( sizeof( CommandPoolCreateInfo ) == sizeof( VkCommandPoolCreateInfo ), "struct and wrapper have different size!" );

  enum class CommandPoolResetFlagBits
  {
    eReleaseResources = VK_COMMAND_POOL_RESET_RELEASE_RESOURCES_BIT
  };

  using CommandPoolResetFlags = Flags<CommandPoolResetFlagBits, VkCommandPoolResetFlags>;

  VULKAN_HPP_INLINE CommandPoolResetFlags operator|( CommandPoolResetFlagBits bit0, CommandPoolResetFlagBits bit1 )
  {
    return CommandPoolResetFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE CommandPoolResetFlags operator~( CommandPoolResetFlagBits bits )
  {
    return ~( CommandPoolResetFlags( bits ) );
  }

  template <> struct FlagTraits<CommandPoolResetFlagBits>
  {
    enum
    {
      allFlags = VkFlags(CommandPoolResetFlagBits::eReleaseResources)
    };
  };

  enum class CommandBufferResetFlagBits
  {
    eReleaseResources = VK_COMMAND_BUFFER_RESET_RELEASE_RESOURCES_BIT
  };

  using CommandBufferResetFlags = Flags<CommandBufferResetFlagBits, VkCommandBufferResetFlags>;

  VULKAN_HPP_INLINE CommandBufferResetFlags operator|( CommandBufferResetFlagBits bit0, CommandBufferResetFlagBits bit1 )
  {
    return CommandBufferResetFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE CommandBufferResetFlags operator~( CommandBufferResetFlagBits bits )
  {
    return ~( CommandBufferResetFlags( bits ) );
  }

  template <> struct FlagTraits<CommandBufferResetFlagBits>
  {
    enum
    {
      allFlags = VkFlags(CommandBufferResetFlagBits::eReleaseResources)
    };
  };

  enum class SampleCountFlagBits
  {
    e1 = VK_SAMPLE_COUNT_1_BIT,
    e2 = VK_SAMPLE_COUNT_2_BIT,
    e4 = VK_SAMPLE_COUNT_4_BIT,
    e8 = VK_SAMPLE_COUNT_8_BIT,
    e16 = VK_SAMPLE_COUNT_16_BIT,
    e32 = VK_SAMPLE_COUNT_32_BIT,
    e64 = VK_SAMPLE_COUNT_64_BIT
  };

  using SampleCountFlags = Flags<SampleCountFlagBits, VkSampleCountFlags>;

  VULKAN_HPP_INLINE SampleCountFlags operator|( SampleCountFlagBits bit0, SampleCountFlagBits bit1 )
  {
    return SampleCountFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SampleCountFlags operator~( SampleCountFlagBits bits )
  {
    return ~( SampleCountFlags( bits ) );
  }

  template <> struct FlagTraits<SampleCountFlagBits>
  {
    enum
    {
      allFlags = VkFlags(SampleCountFlagBits::e1) | VkFlags(SampleCountFlagBits::e2) | VkFlags(SampleCountFlagBits::e4) | VkFlags(SampleCountFlagBits::e8) | VkFlags(SampleCountFlagBits::e16) | VkFlags(SampleCountFlagBits::e32) | VkFlags(SampleCountFlagBits::e64)
    };
  };

  struct ImageFormatProperties
  {
    operator VkImageFormatProperties const&() const
    {
      return *reinterpret_cast<const VkImageFormatProperties*>(this);
    }

    operator VkImageFormatProperties &()
    {
      return *reinterpret_cast<VkImageFormatProperties*>(this);
    }

    bool operator==( ImageFormatProperties const& rhs ) const
    {
      return ( maxExtent == rhs.maxExtent )
          && ( maxMipLevels == rhs.maxMipLevels )
          && ( maxArrayLayers == rhs.maxArrayLayers )
          && ( sampleCounts == rhs.sampleCounts )
          && ( maxResourceSize == rhs.maxResourceSize );
    }

    bool operator!=( ImageFormatProperties const& rhs ) const
    {
      return !operator==( rhs );
    }

    Extent3D maxExtent;
    uint32_t maxMipLevels;
    uint32_t maxArrayLayers;
    SampleCountFlags sampleCounts;
    DeviceSize maxResourceSize;
  };
  static_assert( sizeof( ImageFormatProperties ) == sizeof( VkImageFormatProperties ), "struct and wrapper have different size!" );

  struct ImageCreateInfo
  {
    ImageCreateInfo( ImageCreateFlags flags_ = ImageCreateFlags(),
                     ImageType imageType_ = ImageType::e1D,
                     Format format_ = Format::eUndefined,
                     Extent3D extent_ = Extent3D(),
                     uint32_t mipLevels_ = 0,
                     uint32_t arrayLayers_ = 0,
                     SampleCountFlagBits samples_ = SampleCountFlagBits::e1,
                     ImageTiling tiling_ = ImageTiling::eOptimal,
                     ImageUsageFlags usage_ = ImageUsageFlags(),
                     SharingMode sharingMode_ = SharingMode::eExclusive,
                     uint32_t queueFamilyIndexCount_ = 0,
                     const uint32_t* pQueueFamilyIndices_ = nullptr,
                     ImageLayout initialLayout_ = ImageLayout::eUndefined )
      : flags( flags_ )
      , imageType( imageType_ )
      , format( format_ )
      , extent( extent_ )
      , mipLevels( mipLevels_ )
      , arrayLayers( arrayLayers_ )
      , samples( samples_ )
      , tiling( tiling_ )
      , usage( usage_ )
      , sharingMode( sharingMode_ )
      , queueFamilyIndexCount( queueFamilyIndexCount_ )
      , pQueueFamilyIndices( pQueueFamilyIndices_ )
      , initialLayout( initialLayout_ )
    {
    }

    ImageCreateInfo( VkImageCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageCreateInfo ) );
    }

    ImageCreateInfo& operator=( VkImageCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageCreateInfo ) );
      return *this;
    }
    ImageCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ImageCreateInfo& setFlags( ImageCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    ImageCreateInfo& setImageType( ImageType imageType_ )
    {
      imageType = imageType_;
      return *this;
    }

    ImageCreateInfo& setFormat( Format format_ )
    {
      format = format_;
      return *this;
    }

    ImageCreateInfo& setExtent( Extent3D extent_ )
    {
      extent = extent_;
      return *this;
    }

    ImageCreateInfo& setMipLevels( uint32_t mipLevels_ )
    {
      mipLevels = mipLevels_;
      return *this;
    }

    ImageCreateInfo& setArrayLayers( uint32_t arrayLayers_ )
    {
      arrayLayers = arrayLayers_;
      return *this;
    }

    ImageCreateInfo& setSamples( SampleCountFlagBits samples_ )
    {
      samples = samples_;
      return *this;
    }

    ImageCreateInfo& setTiling( ImageTiling tiling_ )
    {
      tiling = tiling_;
      return *this;
    }

    ImageCreateInfo& setUsage( ImageUsageFlags usage_ )
    {
      usage = usage_;
      return *this;
    }

    ImageCreateInfo& setSharingMode( SharingMode sharingMode_ )
    {
      sharingMode = sharingMode_;
      return *this;
    }

    ImageCreateInfo& setQueueFamilyIndexCount( uint32_t queueFamilyIndexCount_ )
    {
      queueFamilyIndexCount = queueFamilyIndexCount_;
      return *this;
    }

    ImageCreateInfo& setPQueueFamilyIndices( const uint32_t* pQueueFamilyIndices_ )
    {
      pQueueFamilyIndices = pQueueFamilyIndices_;
      return *this;
    }

    ImageCreateInfo& setInitialLayout( ImageLayout initialLayout_ )
    {
      initialLayout = initialLayout_;
      return *this;
    }

    operator VkImageCreateInfo const&() const
    {
      return *reinterpret_cast<const VkImageCreateInfo*>(this);
    }

    operator VkImageCreateInfo &()
    {
      return *reinterpret_cast<VkImageCreateInfo*>(this);
    }

    bool operator==( ImageCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( imageType == rhs.imageType )
          && ( format == rhs.format )
          && ( extent == rhs.extent )
          && ( mipLevels == rhs.mipLevels )
          && ( arrayLayers == rhs.arrayLayers )
          && ( samples == rhs.samples )
          && ( tiling == rhs.tiling )
          && ( usage == rhs.usage )
          && ( sharingMode == rhs.sharingMode )
          && ( queueFamilyIndexCount == rhs.queueFamilyIndexCount )
          && ( pQueueFamilyIndices == rhs.pQueueFamilyIndices )
          && ( initialLayout == rhs.initialLayout );
    }

    bool operator!=( ImageCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageCreateInfo;

  public:
    const void* pNext = nullptr;
    ImageCreateFlags flags;
    ImageType imageType;
    Format format;
    Extent3D extent;
    uint32_t mipLevels;
    uint32_t arrayLayers;
    SampleCountFlagBits samples;
    ImageTiling tiling;
    ImageUsageFlags usage;
    SharingMode sharingMode;
    uint32_t queueFamilyIndexCount;
    const uint32_t* pQueueFamilyIndices;
    ImageLayout initialLayout;
  };
  static_assert( sizeof( ImageCreateInfo ) == sizeof( VkImageCreateInfo ), "struct and wrapper have different size!" );

  struct PipelineMultisampleStateCreateInfo
  {
    PipelineMultisampleStateCreateInfo( PipelineMultisampleStateCreateFlags flags_ = PipelineMultisampleStateCreateFlags(),
                                        SampleCountFlagBits rasterizationSamples_ = SampleCountFlagBits::e1,
                                        Bool32 sampleShadingEnable_ = 0,
                                        float minSampleShading_ = 0,
                                        const SampleMask* pSampleMask_ = nullptr,
                                        Bool32 alphaToCoverageEnable_ = 0,
                                        Bool32 alphaToOneEnable_ = 0 )
      : flags( flags_ )
      , rasterizationSamples( rasterizationSamples_ )
      , sampleShadingEnable( sampleShadingEnable_ )
      , minSampleShading( minSampleShading_ )
      , pSampleMask( pSampleMask_ )
      , alphaToCoverageEnable( alphaToCoverageEnable_ )
      , alphaToOneEnable( alphaToOneEnable_ )
    {
    }

    PipelineMultisampleStateCreateInfo( VkPipelineMultisampleStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineMultisampleStateCreateInfo ) );
    }

    PipelineMultisampleStateCreateInfo& operator=( VkPipelineMultisampleStateCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineMultisampleStateCreateInfo ) );
      return *this;
    }
    PipelineMultisampleStateCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineMultisampleStateCreateInfo& setFlags( PipelineMultisampleStateCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineMultisampleStateCreateInfo& setRasterizationSamples( SampleCountFlagBits rasterizationSamples_ )
    {
      rasterizationSamples = rasterizationSamples_;
      return *this;
    }

    PipelineMultisampleStateCreateInfo& setSampleShadingEnable( Bool32 sampleShadingEnable_ )
    {
      sampleShadingEnable = sampleShadingEnable_;
      return *this;
    }

    PipelineMultisampleStateCreateInfo& setMinSampleShading( float minSampleShading_ )
    {
      minSampleShading = minSampleShading_;
      return *this;
    }

    PipelineMultisampleStateCreateInfo& setPSampleMask( const SampleMask* pSampleMask_ )
    {
      pSampleMask = pSampleMask_;
      return *this;
    }

    PipelineMultisampleStateCreateInfo& setAlphaToCoverageEnable( Bool32 alphaToCoverageEnable_ )
    {
      alphaToCoverageEnable = alphaToCoverageEnable_;
      return *this;
    }

    PipelineMultisampleStateCreateInfo& setAlphaToOneEnable( Bool32 alphaToOneEnable_ )
    {
      alphaToOneEnable = alphaToOneEnable_;
      return *this;
    }

    operator VkPipelineMultisampleStateCreateInfo const&() const
    {
      return *reinterpret_cast<const VkPipelineMultisampleStateCreateInfo*>(this);
    }

    operator VkPipelineMultisampleStateCreateInfo &()
    {
      return *reinterpret_cast<VkPipelineMultisampleStateCreateInfo*>(this);
    }

    bool operator==( PipelineMultisampleStateCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( rasterizationSamples == rhs.rasterizationSamples )
          && ( sampleShadingEnable == rhs.sampleShadingEnable )
          && ( minSampleShading == rhs.minSampleShading )
          && ( pSampleMask == rhs.pSampleMask )
          && ( alphaToCoverageEnable == rhs.alphaToCoverageEnable )
          && ( alphaToOneEnable == rhs.alphaToOneEnable );
    }

    bool operator!=( PipelineMultisampleStateCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineMultisampleStateCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineMultisampleStateCreateFlags flags;
    SampleCountFlagBits rasterizationSamples;
    Bool32 sampleShadingEnable;
    float minSampleShading;
    const SampleMask* pSampleMask;
    Bool32 alphaToCoverageEnable;
    Bool32 alphaToOneEnable;
  };
  static_assert( sizeof( PipelineMultisampleStateCreateInfo ) == sizeof( VkPipelineMultisampleStateCreateInfo ), "struct and wrapper have different size!" );

  struct GraphicsPipelineCreateInfo
  {
    GraphicsPipelineCreateInfo( PipelineCreateFlags flags_ = PipelineCreateFlags(),
                                uint32_t stageCount_ = 0,
                                const PipelineShaderStageCreateInfo* pStages_ = nullptr,
                                const PipelineVertexInputStateCreateInfo* pVertexInputState_ = nullptr,
                                const PipelineInputAssemblyStateCreateInfo* pInputAssemblyState_ = nullptr,
                                const PipelineTessellationStateCreateInfo* pTessellationState_ = nullptr,
                                const PipelineViewportStateCreateInfo* pViewportState_ = nullptr,
                                const PipelineRasterizationStateCreateInfo* pRasterizationState_ = nullptr,
                                const PipelineMultisampleStateCreateInfo* pMultisampleState_ = nullptr,
                                const PipelineDepthStencilStateCreateInfo* pDepthStencilState_ = nullptr,
                                const PipelineColorBlendStateCreateInfo* pColorBlendState_ = nullptr,
                                const PipelineDynamicStateCreateInfo* pDynamicState_ = nullptr,
                                PipelineLayout layout_ = PipelineLayout(),
                                RenderPass renderPass_ = RenderPass(),
                                uint32_t subpass_ = 0,
                                Pipeline basePipelineHandle_ = Pipeline(),
                                int32_t basePipelineIndex_ = 0 )
      : flags( flags_ )
      , stageCount( stageCount_ )
      , pStages( pStages_ )
      , pVertexInputState( pVertexInputState_ )
      , pInputAssemblyState( pInputAssemblyState_ )
      , pTessellationState( pTessellationState_ )
      , pViewportState( pViewportState_ )
      , pRasterizationState( pRasterizationState_ )
      , pMultisampleState( pMultisampleState_ )
      , pDepthStencilState( pDepthStencilState_ )
      , pColorBlendState( pColorBlendState_ )
      , pDynamicState( pDynamicState_ )
      , layout( layout_ )
      , renderPass( renderPass_ )
      , subpass( subpass_ )
      , basePipelineHandle( basePipelineHandle_ )
      , basePipelineIndex( basePipelineIndex_ )
    {
    }

    GraphicsPipelineCreateInfo( VkGraphicsPipelineCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( GraphicsPipelineCreateInfo ) );
    }

    GraphicsPipelineCreateInfo& operator=( VkGraphicsPipelineCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( GraphicsPipelineCreateInfo ) );
      return *this;
    }
    GraphicsPipelineCreateInfo& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setFlags( PipelineCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setStageCount( uint32_t stageCount_ )
    {
      stageCount = stageCount_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setPStages( const PipelineShaderStageCreateInfo* pStages_ )
    {
      pStages = pStages_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setPVertexInputState( const PipelineVertexInputStateCreateInfo* pVertexInputState_ )
    {
      pVertexInputState = pVertexInputState_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setPInputAssemblyState( const PipelineInputAssemblyStateCreateInfo* pInputAssemblyState_ )
    {
      pInputAssemblyState = pInputAssemblyState_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setPTessellationState( const PipelineTessellationStateCreateInfo* pTessellationState_ )
    {
      pTessellationState = pTessellationState_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setPViewportState( const PipelineViewportStateCreateInfo* pViewportState_ )
    {
      pViewportState = pViewportState_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setPRasterizationState( const PipelineRasterizationStateCreateInfo* pRasterizationState_ )
    {
      pRasterizationState = pRasterizationState_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setPMultisampleState( const PipelineMultisampleStateCreateInfo* pMultisampleState_ )
    {
      pMultisampleState = pMultisampleState_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setPDepthStencilState( const PipelineDepthStencilStateCreateInfo* pDepthStencilState_ )
    {
      pDepthStencilState = pDepthStencilState_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setPColorBlendState( const PipelineColorBlendStateCreateInfo* pColorBlendState_ )
    {
      pColorBlendState = pColorBlendState_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setPDynamicState( const PipelineDynamicStateCreateInfo* pDynamicState_ )
    {
      pDynamicState = pDynamicState_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setLayout( PipelineLayout layout_ )
    {
      layout = layout_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setRenderPass( RenderPass renderPass_ )
    {
      renderPass = renderPass_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setSubpass( uint32_t subpass_ )
    {
      subpass = subpass_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setBasePipelineHandle( Pipeline basePipelineHandle_ )
    {
      basePipelineHandle = basePipelineHandle_;
      return *this;
    }

    GraphicsPipelineCreateInfo& setBasePipelineIndex( int32_t basePipelineIndex_ )
    {
      basePipelineIndex = basePipelineIndex_;
      return *this;
    }

    operator VkGraphicsPipelineCreateInfo const&() const
    {
      return *reinterpret_cast<const VkGraphicsPipelineCreateInfo*>(this);
    }

    operator VkGraphicsPipelineCreateInfo &()
    {
      return *reinterpret_cast<VkGraphicsPipelineCreateInfo*>(this);
    }

    bool operator==( GraphicsPipelineCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( stageCount == rhs.stageCount )
          && ( pStages == rhs.pStages )
          && ( pVertexInputState == rhs.pVertexInputState )
          && ( pInputAssemblyState == rhs.pInputAssemblyState )
          && ( pTessellationState == rhs.pTessellationState )
          && ( pViewportState == rhs.pViewportState )
          && ( pRasterizationState == rhs.pRasterizationState )
          && ( pMultisampleState == rhs.pMultisampleState )
          && ( pDepthStencilState == rhs.pDepthStencilState )
          && ( pColorBlendState == rhs.pColorBlendState )
          && ( pDynamicState == rhs.pDynamicState )
          && ( layout == rhs.layout )
          && ( renderPass == rhs.renderPass )
          && ( subpass == rhs.subpass )
          && ( basePipelineHandle == rhs.basePipelineHandle )
          && ( basePipelineIndex == rhs.basePipelineIndex );
    }

    bool operator!=( GraphicsPipelineCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eGraphicsPipelineCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineCreateFlags flags;
    uint32_t stageCount;
    const PipelineShaderStageCreateInfo* pStages;
    const PipelineVertexInputStateCreateInfo* pVertexInputState;
    const PipelineInputAssemblyStateCreateInfo* pInputAssemblyState;
    const PipelineTessellationStateCreateInfo* pTessellationState;
    const PipelineViewportStateCreateInfo* pViewportState;
    const PipelineRasterizationStateCreateInfo* pRasterizationState;
    const PipelineMultisampleStateCreateInfo* pMultisampleState;
    const PipelineDepthStencilStateCreateInfo* pDepthStencilState;
    const PipelineColorBlendStateCreateInfo* pColorBlendState;
    const PipelineDynamicStateCreateInfo* pDynamicState;
    PipelineLayout layout;
    RenderPass renderPass;
    uint32_t subpass;
    Pipeline basePipelineHandle;
    int32_t basePipelineIndex;
  };
  static_assert( sizeof( GraphicsPipelineCreateInfo ) == sizeof( VkGraphicsPipelineCreateInfo ), "struct and wrapper have different size!" );

  struct PhysicalDeviceLimits
  {
    operator VkPhysicalDeviceLimits const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceLimits*>(this);
    }

    operator VkPhysicalDeviceLimits &()
    {
      return *reinterpret_cast<VkPhysicalDeviceLimits*>(this);
    }

    bool operator==( PhysicalDeviceLimits const& rhs ) const
    {
      return ( maxImageDimension1D == rhs.maxImageDimension1D )
          && ( maxImageDimension2D == rhs.maxImageDimension2D )
          && ( maxImageDimension3D == rhs.maxImageDimension3D )
          && ( maxImageDimensionCube == rhs.maxImageDimensionCube )
          && ( maxImageArrayLayers == rhs.maxImageArrayLayers )
          && ( maxTexelBufferElements == rhs.maxTexelBufferElements )
          && ( maxUniformBufferRange == rhs.maxUniformBufferRange )
          && ( maxStorageBufferRange == rhs.maxStorageBufferRange )
          && ( maxPushConstantsSize == rhs.maxPushConstantsSize )
          && ( maxMemoryAllocationCount == rhs.maxMemoryAllocationCount )
          && ( maxSamplerAllocationCount == rhs.maxSamplerAllocationCount )
          && ( bufferImageGranularity == rhs.bufferImageGranularity )
          && ( sparseAddressSpaceSize == rhs.sparseAddressSpaceSize )
          && ( maxBoundDescriptorSets == rhs.maxBoundDescriptorSets )
          && ( maxPerStageDescriptorSamplers == rhs.maxPerStageDescriptorSamplers )
          && ( maxPerStageDescriptorUniformBuffers == rhs.maxPerStageDescriptorUniformBuffers )
          && ( maxPerStageDescriptorStorageBuffers == rhs.maxPerStageDescriptorStorageBuffers )
          && ( maxPerStageDescriptorSampledImages == rhs.maxPerStageDescriptorSampledImages )
          && ( maxPerStageDescriptorStorageImages == rhs.maxPerStageDescriptorStorageImages )
          && ( maxPerStageDescriptorInputAttachments == rhs.maxPerStageDescriptorInputAttachments )
          && ( maxPerStageResources == rhs.maxPerStageResources )
          && ( maxDescriptorSetSamplers == rhs.maxDescriptorSetSamplers )
          && ( maxDescriptorSetUniformBuffers == rhs.maxDescriptorSetUniformBuffers )
          && ( maxDescriptorSetUniformBuffersDynamic == rhs.maxDescriptorSetUniformBuffersDynamic )
          && ( maxDescriptorSetStorageBuffers == rhs.maxDescriptorSetStorageBuffers )
          && ( maxDescriptorSetStorageBuffersDynamic == rhs.maxDescriptorSetStorageBuffersDynamic )
          && ( maxDescriptorSetSampledImages == rhs.maxDescriptorSetSampledImages )
          && ( maxDescriptorSetStorageImages == rhs.maxDescriptorSetStorageImages )
          && ( maxDescriptorSetInputAttachments == rhs.maxDescriptorSetInputAttachments )
          && ( maxVertexInputAttributes == rhs.maxVertexInputAttributes )
          && ( maxVertexInputBindings == rhs.maxVertexInputBindings )
          && ( maxVertexInputAttributeOffset == rhs.maxVertexInputAttributeOffset )
          && ( maxVertexInputBindingStride == rhs.maxVertexInputBindingStride )
          && ( maxVertexOutputComponents == rhs.maxVertexOutputComponents )
          && ( maxTessellationGenerationLevel == rhs.maxTessellationGenerationLevel )
          && ( maxTessellationPatchSize == rhs.maxTessellationPatchSize )
          && ( maxTessellationControlPerVertexInputComponents == rhs.maxTessellationControlPerVertexInputComponents )
          && ( maxTessellationControlPerVertexOutputComponents == rhs.maxTessellationControlPerVertexOutputComponents )
          && ( maxTessellationControlPerPatchOutputComponents == rhs.maxTessellationControlPerPatchOutputComponents )
          && ( maxTessellationControlTotalOutputComponents == rhs.maxTessellationControlTotalOutputComponents )
          && ( maxTessellationEvaluationInputComponents == rhs.maxTessellationEvaluationInputComponents )
          && ( maxTessellationEvaluationOutputComponents == rhs.maxTessellationEvaluationOutputComponents )
          && ( maxGeometryShaderInvocations == rhs.maxGeometryShaderInvocations )
          && ( maxGeometryInputComponents == rhs.maxGeometryInputComponents )
          && ( maxGeometryOutputComponents == rhs.maxGeometryOutputComponents )
          && ( maxGeometryOutputVertices == rhs.maxGeometryOutputVertices )
          && ( maxGeometryTotalOutputComponents == rhs.maxGeometryTotalOutputComponents )
          && ( maxFragmentInputComponents == rhs.maxFragmentInputComponents )
          && ( maxFragmentOutputAttachments == rhs.maxFragmentOutputAttachments )
          && ( maxFragmentDualSrcAttachments == rhs.maxFragmentDualSrcAttachments )
          && ( maxFragmentCombinedOutputResources == rhs.maxFragmentCombinedOutputResources )
          && ( maxComputeSharedMemorySize == rhs.maxComputeSharedMemorySize )
          && ( memcmp( maxComputeWorkGroupCount, rhs.maxComputeWorkGroupCount, 3 * sizeof( uint32_t ) ) == 0 )
          && ( maxComputeWorkGroupInvocations == rhs.maxComputeWorkGroupInvocations )
          && ( memcmp( maxComputeWorkGroupSize, rhs.maxComputeWorkGroupSize, 3 * sizeof( uint32_t ) ) == 0 )
          && ( subPixelPrecisionBits == rhs.subPixelPrecisionBits )
          && ( subTexelPrecisionBits == rhs.subTexelPrecisionBits )
          && ( mipmapPrecisionBits == rhs.mipmapPrecisionBits )
          && ( maxDrawIndexedIndexValue == rhs.maxDrawIndexedIndexValue )
          && ( maxDrawIndirectCount == rhs.maxDrawIndirectCount )
          && ( maxSamplerLodBias == rhs.maxSamplerLodBias )
          && ( maxSamplerAnisotropy == rhs.maxSamplerAnisotropy )
          && ( maxViewports == rhs.maxViewports )
          && ( memcmp( maxViewportDimensions, rhs.maxViewportDimensions, 2 * sizeof( uint32_t ) ) == 0 )
          && ( memcmp( viewportBoundsRange, rhs.viewportBoundsRange, 2 * sizeof( float ) ) == 0 )
          && ( viewportSubPixelBits == rhs.viewportSubPixelBits )
          && ( minMemoryMapAlignment == rhs.minMemoryMapAlignment )
          && ( minTexelBufferOffsetAlignment == rhs.minTexelBufferOffsetAlignment )
          && ( minUniformBufferOffsetAlignment == rhs.minUniformBufferOffsetAlignment )
          && ( minStorageBufferOffsetAlignment == rhs.minStorageBufferOffsetAlignment )
          && ( minTexelOffset == rhs.minTexelOffset )
          && ( maxTexelOffset == rhs.maxTexelOffset )
          && ( minTexelGatherOffset == rhs.minTexelGatherOffset )
          && ( maxTexelGatherOffset == rhs.maxTexelGatherOffset )
          && ( minInterpolationOffset == rhs.minInterpolationOffset )
          && ( maxInterpolationOffset == rhs.maxInterpolationOffset )
          && ( subPixelInterpolationOffsetBits == rhs.subPixelInterpolationOffsetBits )
          && ( maxFramebufferWidth == rhs.maxFramebufferWidth )
          && ( maxFramebufferHeight == rhs.maxFramebufferHeight )
          && ( maxFramebufferLayers == rhs.maxFramebufferLayers )
          && ( framebufferColorSampleCounts == rhs.framebufferColorSampleCounts )
          && ( framebufferDepthSampleCounts == rhs.framebufferDepthSampleCounts )
          && ( framebufferStencilSampleCounts == rhs.framebufferStencilSampleCounts )
          && ( framebufferNoAttachmentsSampleCounts == rhs.framebufferNoAttachmentsSampleCounts )
          && ( maxColorAttachments == rhs.maxColorAttachments )
          && ( sampledImageColorSampleCounts == rhs.sampledImageColorSampleCounts )
          && ( sampledImageIntegerSampleCounts == rhs.sampledImageIntegerSampleCounts )
          && ( sampledImageDepthSampleCounts == rhs.sampledImageDepthSampleCounts )
          && ( sampledImageStencilSampleCounts == rhs.sampledImageStencilSampleCounts )
          && ( storageImageSampleCounts == rhs.storageImageSampleCounts )
          && ( maxSampleMaskWords == rhs.maxSampleMaskWords )
          && ( timestampComputeAndGraphics == rhs.timestampComputeAndGraphics )
          && ( timestampPeriod == rhs.timestampPeriod )
          && ( maxClipDistances == rhs.maxClipDistances )
          && ( maxCullDistances == rhs.maxCullDistances )
          && ( maxCombinedClipAndCullDistances == rhs.maxCombinedClipAndCullDistances )
          && ( discreteQueuePriorities == rhs.discreteQueuePriorities )
          && ( memcmp( pointSizeRange, rhs.pointSizeRange, 2 * sizeof( float ) ) == 0 )
          && ( memcmp( lineWidthRange, rhs.lineWidthRange, 2 * sizeof( float ) ) == 0 )
          && ( pointSizeGranularity == rhs.pointSizeGranularity )
          && ( lineWidthGranularity == rhs.lineWidthGranularity )
          && ( strictLines == rhs.strictLines )
          && ( standardSampleLocations == rhs.standardSampleLocations )
          && ( optimalBufferCopyOffsetAlignment == rhs.optimalBufferCopyOffsetAlignment )
          && ( optimalBufferCopyRowPitchAlignment == rhs.optimalBufferCopyRowPitchAlignment )
          && ( nonCoherentAtomSize == rhs.nonCoherentAtomSize );
    }

    bool operator!=( PhysicalDeviceLimits const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t maxImageDimension1D;
    uint32_t maxImageDimension2D;
    uint32_t maxImageDimension3D;
    uint32_t maxImageDimensionCube;
    uint32_t maxImageArrayLayers;
    uint32_t maxTexelBufferElements;
    uint32_t maxUniformBufferRange;
    uint32_t maxStorageBufferRange;
    uint32_t maxPushConstantsSize;
    uint32_t maxMemoryAllocationCount;
    uint32_t maxSamplerAllocationCount;
    DeviceSize bufferImageGranularity;
    DeviceSize sparseAddressSpaceSize;
    uint32_t maxBoundDescriptorSets;
    uint32_t maxPerStageDescriptorSamplers;
    uint32_t maxPerStageDescriptorUniformBuffers;
    uint32_t maxPerStageDescriptorStorageBuffers;
    uint32_t maxPerStageDescriptorSampledImages;
    uint32_t maxPerStageDescriptorStorageImages;
    uint32_t maxPerStageDescriptorInputAttachments;
    uint32_t maxPerStageResources;
    uint32_t maxDescriptorSetSamplers;
    uint32_t maxDescriptorSetUniformBuffers;
    uint32_t maxDescriptorSetUniformBuffersDynamic;
    uint32_t maxDescriptorSetStorageBuffers;
    uint32_t maxDescriptorSetStorageBuffersDynamic;
    uint32_t maxDescriptorSetSampledImages;
    uint32_t maxDescriptorSetStorageImages;
    uint32_t maxDescriptorSetInputAttachments;
    uint32_t maxVertexInputAttributes;
    uint32_t maxVertexInputBindings;
    uint32_t maxVertexInputAttributeOffset;
    uint32_t maxVertexInputBindingStride;
    uint32_t maxVertexOutputComponents;
    uint32_t maxTessellationGenerationLevel;
    uint32_t maxTessellationPatchSize;
    uint32_t maxTessellationControlPerVertexInputComponents;
    uint32_t maxTessellationControlPerVertexOutputComponents;
    uint32_t maxTessellationControlPerPatchOutputComponents;
    uint32_t maxTessellationControlTotalOutputComponents;
    uint32_t maxTessellationEvaluationInputComponents;
    uint32_t maxTessellationEvaluationOutputComponents;
    uint32_t maxGeometryShaderInvocations;
    uint32_t maxGeometryInputComponents;
    uint32_t maxGeometryOutputComponents;
    uint32_t maxGeometryOutputVertices;
    uint32_t maxGeometryTotalOutputComponents;
    uint32_t maxFragmentInputComponents;
    uint32_t maxFragmentOutputAttachments;
    uint32_t maxFragmentDualSrcAttachments;
    uint32_t maxFragmentCombinedOutputResources;
    uint32_t maxComputeSharedMemorySize;
    uint32_t maxComputeWorkGroupCount[3];
    uint32_t maxComputeWorkGroupInvocations;
    uint32_t maxComputeWorkGroupSize[3];
    uint32_t subPixelPrecisionBits;
    uint32_t subTexelPrecisionBits;
    uint32_t mipmapPrecisionBits;
    uint32_t maxDrawIndexedIndexValue;
    uint32_t maxDrawIndirectCount;
    float maxSamplerLodBias;
    float maxSamplerAnisotropy;
    uint32_t maxViewports;
    uint32_t maxViewportDimensions[2];
    float viewportBoundsRange[2];
    uint32_t viewportSubPixelBits;
    size_t minMemoryMapAlignment;
    DeviceSize minTexelBufferOffsetAlignment;
    DeviceSize minUniformBufferOffsetAlignment;
    DeviceSize minStorageBufferOffsetAlignment;
    int32_t minTexelOffset;
    uint32_t maxTexelOffset;
    int32_t minTexelGatherOffset;
    uint32_t maxTexelGatherOffset;
    float minInterpolationOffset;
    float maxInterpolationOffset;
    uint32_t subPixelInterpolationOffsetBits;
    uint32_t maxFramebufferWidth;
    uint32_t maxFramebufferHeight;
    uint32_t maxFramebufferLayers;
    SampleCountFlags framebufferColorSampleCounts;
    SampleCountFlags framebufferDepthSampleCounts;
    SampleCountFlags framebufferStencilSampleCounts;
    SampleCountFlags framebufferNoAttachmentsSampleCounts;
    uint32_t maxColorAttachments;
    SampleCountFlags sampledImageColorSampleCounts;
    SampleCountFlags sampledImageIntegerSampleCounts;
    SampleCountFlags sampledImageDepthSampleCounts;
    SampleCountFlags sampledImageStencilSampleCounts;
    SampleCountFlags storageImageSampleCounts;
    uint32_t maxSampleMaskWords;
    Bool32 timestampComputeAndGraphics;
    float timestampPeriod;
    uint32_t maxClipDistances;
    uint32_t maxCullDistances;
    uint32_t maxCombinedClipAndCullDistances;
    uint32_t discreteQueuePriorities;
    float pointSizeRange[2];
    float lineWidthRange[2];
    float pointSizeGranularity;
    float lineWidthGranularity;
    Bool32 strictLines;
    Bool32 standardSampleLocations;
    DeviceSize optimalBufferCopyOffsetAlignment;
    DeviceSize optimalBufferCopyRowPitchAlignment;
    DeviceSize nonCoherentAtomSize;
  };
  static_assert( sizeof( PhysicalDeviceLimits ) == sizeof( VkPhysicalDeviceLimits ), "struct and wrapper have different size!" );

  struct PhysicalDeviceProperties
  {
    operator VkPhysicalDeviceProperties const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceProperties*>(this);
    }

    operator VkPhysicalDeviceProperties &()
    {
      return *reinterpret_cast<VkPhysicalDeviceProperties*>(this);
    }

    bool operator==( PhysicalDeviceProperties const& rhs ) const
    {
      return ( apiVersion == rhs.apiVersion )
          && ( driverVersion == rhs.driverVersion )
          && ( vendorID == rhs.vendorID )
          && ( deviceID == rhs.deviceID )
          && ( deviceType == rhs.deviceType )
          && ( memcmp( deviceName, rhs.deviceName, VK_MAX_PHYSICAL_DEVICE_NAME_SIZE * sizeof( char ) ) == 0 )
          && ( memcmp( pipelineCacheUUID, rhs.pipelineCacheUUID, VK_UUID_SIZE * sizeof( uint8_t ) ) == 0 )
          && ( limits == rhs.limits )
          && ( sparseProperties == rhs.sparseProperties );
    }

    bool operator!=( PhysicalDeviceProperties const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t apiVersion;
    uint32_t driverVersion;
    uint32_t vendorID;
    uint32_t deviceID;
    PhysicalDeviceType deviceType;
    char deviceName[VK_MAX_PHYSICAL_DEVICE_NAME_SIZE];
    uint8_t pipelineCacheUUID[VK_UUID_SIZE];
    PhysicalDeviceLimits limits;
    PhysicalDeviceSparseProperties sparseProperties;
  };
  static_assert( sizeof( PhysicalDeviceProperties ) == sizeof( VkPhysicalDeviceProperties ), "struct and wrapper have different size!" );

  struct PhysicalDeviceProperties2
  {
    operator VkPhysicalDeviceProperties2 const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceProperties2*>(this);
    }

    operator VkPhysicalDeviceProperties2 &()
    {
      return *reinterpret_cast<VkPhysicalDeviceProperties2*>(this);
    }

    bool operator==( PhysicalDeviceProperties2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( properties == rhs.properties );
    }

    bool operator!=( PhysicalDeviceProperties2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceProperties2;

  public:
    void* pNext = nullptr;
    PhysicalDeviceProperties properties;
  };
  static_assert( sizeof( PhysicalDeviceProperties2 ) == sizeof( VkPhysicalDeviceProperties2 ), "struct and wrapper have different size!" );

  using PhysicalDeviceProperties2KHR = PhysicalDeviceProperties2;

  struct ImageFormatProperties2
  {
    operator VkImageFormatProperties2 const&() const
    {
      return *reinterpret_cast<const VkImageFormatProperties2*>(this);
    }

    operator VkImageFormatProperties2 &()
    {
      return *reinterpret_cast<VkImageFormatProperties2*>(this);
    }

    bool operator==( ImageFormatProperties2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( imageFormatProperties == rhs.imageFormatProperties );
    }

    bool operator!=( ImageFormatProperties2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eImageFormatProperties2;

  public:
    void* pNext = nullptr;
    ImageFormatProperties imageFormatProperties;
  };
  static_assert( sizeof( ImageFormatProperties2 ) == sizeof( VkImageFormatProperties2 ), "struct and wrapper have different size!" );

  using ImageFormatProperties2KHR = ImageFormatProperties2;

  struct PhysicalDeviceSparseImageFormatInfo2
  {
    PhysicalDeviceSparseImageFormatInfo2( Format format_ = Format::eUndefined,
                                          ImageType type_ = ImageType::e1D,
                                          SampleCountFlagBits samples_ = SampleCountFlagBits::e1,
                                          ImageUsageFlags usage_ = ImageUsageFlags(),
                                          ImageTiling tiling_ = ImageTiling::eOptimal )
      : format( format_ )
      , type( type_ )
      , samples( samples_ )
      , usage( usage_ )
      , tiling( tiling_ )
    {
    }

    PhysicalDeviceSparseImageFormatInfo2( VkPhysicalDeviceSparseImageFormatInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceSparseImageFormatInfo2 ) );
    }

    PhysicalDeviceSparseImageFormatInfo2& operator=( VkPhysicalDeviceSparseImageFormatInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( PhysicalDeviceSparseImageFormatInfo2 ) );
      return *this;
    }
    PhysicalDeviceSparseImageFormatInfo2& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PhysicalDeviceSparseImageFormatInfo2& setFormat( Format format_ )
    {
      format = format_;
      return *this;
    }

    PhysicalDeviceSparseImageFormatInfo2& setType( ImageType type_ )
    {
      type = type_;
      return *this;
    }

    PhysicalDeviceSparseImageFormatInfo2& setSamples( SampleCountFlagBits samples_ )
    {
      samples = samples_;
      return *this;
    }

    PhysicalDeviceSparseImageFormatInfo2& setUsage( ImageUsageFlags usage_ )
    {
      usage = usage_;
      return *this;
    }

    PhysicalDeviceSparseImageFormatInfo2& setTiling( ImageTiling tiling_ )
    {
      tiling = tiling_;
      return *this;
    }

    operator VkPhysicalDeviceSparseImageFormatInfo2 const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceSparseImageFormatInfo2*>(this);
    }

    operator VkPhysicalDeviceSparseImageFormatInfo2 &()
    {
      return *reinterpret_cast<VkPhysicalDeviceSparseImageFormatInfo2*>(this);
    }

    bool operator==( PhysicalDeviceSparseImageFormatInfo2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( format == rhs.format )
          && ( type == rhs.type )
          && ( samples == rhs.samples )
          && ( usage == rhs.usage )
          && ( tiling == rhs.tiling );
    }

    bool operator!=( PhysicalDeviceSparseImageFormatInfo2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceSparseImageFormatInfo2;

  public:
    const void* pNext = nullptr;
    Format format;
    ImageType type;
    SampleCountFlagBits samples;
    ImageUsageFlags usage;
    ImageTiling tiling;
  };
  static_assert( sizeof( PhysicalDeviceSparseImageFormatInfo2 ) == sizeof( VkPhysicalDeviceSparseImageFormatInfo2 ), "struct and wrapper have different size!" );

  using PhysicalDeviceSparseImageFormatInfo2KHR = PhysicalDeviceSparseImageFormatInfo2;

  struct SampleLocationsInfoEXT
  {
    SampleLocationsInfoEXT( SampleCountFlagBits sampleLocationsPerPixel_ = SampleCountFlagBits::e1,
                            Extent2D sampleLocationGridSize_ = Extent2D(),
                            uint32_t sampleLocationsCount_ = 0,
                            const SampleLocationEXT* pSampleLocations_ = nullptr )
      : sampleLocationsPerPixel( sampleLocationsPerPixel_ )
      , sampleLocationGridSize( sampleLocationGridSize_ )
      , sampleLocationsCount( sampleLocationsCount_ )
      , pSampleLocations( pSampleLocations_ )
    {
    }

    SampleLocationsInfoEXT( VkSampleLocationsInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( SampleLocationsInfoEXT ) );
    }

    SampleLocationsInfoEXT& operator=( VkSampleLocationsInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( SampleLocationsInfoEXT ) );
      return *this;
    }
    SampleLocationsInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    SampleLocationsInfoEXT& setSampleLocationsPerPixel( SampleCountFlagBits sampleLocationsPerPixel_ )
    {
      sampleLocationsPerPixel = sampleLocationsPerPixel_;
      return *this;
    }

    SampleLocationsInfoEXT& setSampleLocationGridSize( Extent2D sampleLocationGridSize_ )
    {
      sampleLocationGridSize = sampleLocationGridSize_;
      return *this;
    }

    SampleLocationsInfoEXT& setSampleLocationsCount( uint32_t sampleLocationsCount_ )
    {
      sampleLocationsCount = sampleLocationsCount_;
      return *this;
    }

    SampleLocationsInfoEXT& setPSampleLocations( const SampleLocationEXT* pSampleLocations_ )
    {
      pSampleLocations = pSampleLocations_;
      return *this;
    }

    operator VkSampleLocationsInfoEXT const&() const
    {
      return *reinterpret_cast<const VkSampleLocationsInfoEXT*>(this);
    }

    operator VkSampleLocationsInfoEXT &()
    {
      return *reinterpret_cast<VkSampleLocationsInfoEXT*>(this);
    }

    bool operator==( SampleLocationsInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( sampleLocationsPerPixel == rhs.sampleLocationsPerPixel )
          && ( sampleLocationGridSize == rhs.sampleLocationGridSize )
          && ( sampleLocationsCount == rhs.sampleLocationsCount )
          && ( pSampleLocations == rhs.pSampleLocations );
    }

    bool operator!=( SampleLocationsInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eSampleLocationsInfoEXT;

  public:
    const void* pNext = nullptr;
    SampleCountFlagBits sampleLocationsPerPixel;
    Extent2D sampleLocationGridSize;
    uint32_t sampleLocationsCount;
    const SampleLocationEXT* pSampleLocations;
  };
  static_assert( sizeof( SampleLocationsInfoEXT ) == sizeof( VkSampleLocationsInfoEXT ), "struct and wrapper have different size!" );

  struct AttachmentSampleLocationsEXT
  {
    AttachmentSampleLocationsEXT( uint32_t attachmentIndex_ = 0,
                                  SampleLocationsInfoEXT sampleLocationsInfo_ = SampleLocationsInfoEXT() )
      : attachmentIndex( attachmentIndex_ )
      , sampleLocationsInfo( sampleLocationsInfo_ )
    {
    }

    AttachmentSampleLocationsEXT( VkAttachmentSampleLocationsEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentSampleLocationsEXT ) );
    }

    AttachmentSampleLocationsEXT& operator=( VkAttachmentSampleLocationsEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentSampleLocationsEXT ) );
      return *this;
    }
    AttachmentSampleLocationsEXT& setAttachmentIndex( uint32_t attachmentIndex_ )
    {
      attachmentIndex = attachmentIndex_;
      return *this;
    }

    AttachmentSampleLocationsEXT& setSampleLocationsInfo( SampleLocationsInfoEXT sampleLocationsInfo_ )
    {
      sampleLocationsInfo = sampleLocationsInfo_;
      return *this;
    }

    operator VkAttachmentSampleLocationsEXT const&() const
    {
      return *reinterpret_cast<const VkAttachmentSampleLocationsEXT*>(this);
    }

    operator VkAttachmentSampleLocationsEXT &()
    {
      return *reinterpret_cast<VkAttachmentSampleLocationsEXT*>(this);
    }

    bool operator==( AttachmentSampleLocationsEXT const& rhs ) const
    {
      return ( attachmentIndex == rhs.attachmentIndex )
          && ( sampleLocationsInfo == rhs.sampleLocationsInfo );
    }

    bool operator!=( AttachmentSampleLocationsEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t attachmentIndex;
    SampleLocationsInfoEXT sampleLocationsInfo;
  };
  static_assert( sizeof( AttachmentSampleLocationsEXT ) == sizeof( VkAttachmentSampleLocationsEXT ), "struct and wrapper have different size!" );

  struct SubpassSampleLocationsEXT
  {
    SubpassSampleLocationsEXT( uint32_t subpassIndex_ = 0,
                               SampleLocationsInfoEXT sampleLocationsInfo_ = SampleLocationsInfoEXT() )
      : subpassIndex( subpassIndex_ )
      , sampleLocationsInfo( sampleLocationsInfo_ )
    {
    }

    SubpassSampleLocationsEXT( VkSubpassSampleLocationsEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( SubpassSampleLocationsEXT ) );
    }

    SubpassSampleLocationsEXT& operator=( VkSubpassSampleLocationsEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( SubpassSampleLocationsEXT ) );
      return *this;
    }
    SubpassSampleLocationsEXT& setSubpassIndex( uint32_t subpassIndex_ )
    {
      subpassIndex = subpassIndex_;
      return *this;
    }

    SubpassSampleLocationsEXT& setSampleLocationsInfo( SampleLocationsInfoEXT sampleLocationsInfo_ )
    {
      sampleLocationsInfo = sampleLocationsInfo_;
      return *this;
    }

    operator VkSubpassSampleLocationsEXT const&() const
    {
      return *reinterpret_cast<const VkSubpassSampleLocationsEXT*>(this);
    }

    operator VkSubpassSampleLocationsEXT &()
    {
      return *reinterpret_cast<VkSubpassSampleLocationsEXT*>(this);
    }

    bool operator==( SubpassSampleLocationsEXT const& rhs ) const
    {
      return ( subpassIndex == rhs.subpassIndex )
          && ( sampleLocationsInfo == rhs.sampleLocationsInfo );
    }

    bool operator!=( SubpassSampleLocationsEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t subpassIndex;
    SampleLocationsInfoEXT sampleLocationsInfo;
  };
  static_assert( sizeof( SubpassSampleLocationsEXT ) == sizeof( VkSubpassSampleLocationsEXT ), "struct and wrapper have different size!" );

  struct RenderPassSampleLocationsBeginInfoEXT
  {
    RenderPassSampleLocationsBeginInfoEXT( uint32_t attachmentInitialSampleLocationsCount_ = 0,
                                           const AttachmentSampleLocationsEXT* pAttachmentInitialSampleLocations_ = nullptr,
                                           uint32_t postSubpassSampleLocationsCount_ = 0,
                                           const SubpassSampleLocationsEXT* pPostSubpassSampleLocations_ = nullptr )
      : attachmentInitialSampleLocationsCount( attachmentInitialSampleLocationsCount_ )
      , pAttachmentInitialSampleLocations( pAttachmentInitialSampleLocations_ )
      , postSubpassSampleLocationsCount( postSubpassSampleLocationsCount_ )
      , pPostSubpassSampleLocations( pPostSubpassSampleLocations_ )
    {
    }

    RenderPassSampleLocationsBeginInfoEXT( VkRenderPassSampleLocationsBeginInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( RenderPassSampleLocationsBeginInfoEXT ) );
    }

    RenderPassSampleLocationsBeginInfoEXT& operator=( VkRenderPassSampleLocationsBeginInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( RenderPassSampleLocationsBeginInfoEXT ) );
      return *this;
    }
    RenderPassSampleLocationsBeginInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    RenderPassSampleLocationsBeginInfoEXT& setAttachmentInitialSampleLocationsCount( uint32_t attachmentInitialSampleLocationsCount_ )
    {
      attachmentInitialSampleLocationsCount = attachmentInitialSampleLocationsCount_;
      return *this;
    }

    RenderPassSampleLocationsBeginInfoEXT& setPAttachmentInitialSampleLocations( const AttachmentSampleLocationsEXT* pAttachmentInitialSampleLocations_ )
    {
      pAttachmentInitialSampleLocations = pAttachmentInitialSampleLocations_;
      return *this;
    }

    RenderPassSampleLocationsBeginInfoEXT& setPostSubpassSampleLocationsCount( uint32_t postSubpassSampleLocationsCount_ )
    {
      postSubpassSampleLocationsCount = postSubpassSampleLocationsCount_;
      return *this;
    }

    RenderPassSampleLocationsBeginInfoEXT& setPPostSubpassSampleLocations( const SubpassSampleLocationsEXT* pPostSubpassSampleLocations_ )
    {
      pPostSubpassSampleLocations = pPostSubpassSampleLocations_;
      return *this;
    }

    operator VkRenderPassSampleLocationsBeginInfoEXT const&() const
    {
      return *reinterpret_cast<const VkRenderPassSampleLocationsBeginInfoEXT*>(this);
    }

    operator VkRenderPassSampleLocationsBeginInfoEXT &()
    {
      return *reinterpret_cast<VkRenderPassSampleLocationsBeginInfoEXT*>(this);
    }

    bool operator==( RenderPassSampleLocationsBeginInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( attachmentInitialSampleLocationsCount == rhs.attachmentInitialSampleLocationsCount )
          && ( pAttachmentInitialSampleLocations == rhs.pAttachmentInitialSampleLocations )
          && ( postSubpassSampleLocationsCount == rhs.postSubpassSampleLocationsCount )
          && ( pPostSubpassSampleLocations == rhs.pPostSubpassSampleLocations );
    }

    bool operator!=( RenderPassSampleLocationsBeginInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eRenderPassSampleLocationsBeginInfoEXT;

  public:
    const void* pNext = nullptr;
    uint32_t attachmentInitialSampleLocationsCount;
    const AttachmentSampleLocationsEXT* pAttachmentInitialSampleLocations;
    uint32_t postSubpassSampleLocationsCount;
    const SubpassSampleLocationsEXT* pPostSubpassSampleLocations;
  };
  static_assert( sizeof( RenderPassSampleLocationsBeginInfoEXT ) == sizeof( VkRenderPassSampleLocationsBeginInfoEXT ), "struct and wrapper have different size!" );

  struct PipelineSampleLocationsStateCreateInfoEXT
  {
    PipelineSampleLocationsStateCreateInfoEXT( Bool32 sampleLocationsEnable_ = 0,
                                               SampleLocationsInfoEXT sampleLocationsInfo_ = SampleLocationsInfoEXT() )
      : sampleLocationsEnable( sampleLocationsEnable_ )
      , sampleLocationsInfo( sampleLocationsInfo_ )
    {
    }

    PipelineSampleLocationsStateCreateInfoEXT( VkPipelineSampleLocationsStateCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineSampleLocationsStateCreateInfoEXT ) );
    }

    PipelineSampleLocationsStateCreateInfoEXT& operator=( VkPipelineSampleLocationsStateCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineSampleLocationsStateCreateInfoEXT ) );
      return *this;
    }
    PipelineSampleLocationsStateCreateInfoEXT& setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineSampleLocationsStateCreateInfoEXT& setSampleLocationsEnable( Bool32 sampleLocationsEnable_ )
    {
      sampleLocationsEnable = sampleLocationsEnable_;
      return *this;
    }

    PipelineSampleLocationsStateCreateInfoEXT& setSampleLocationsInfo( SampleLocationsInfoEXT sampleLocationsInfo_ )
    {
      sampleLocationsInfo = sampleLocationsInfo_;
      return *this;
    }

    operator VkPipelineSampleLocationsStateCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkPipelineSampleLocationsStateCreateInfoEXT*>(this);
    }

    operator VkPipelineSampleLocationsStateCreateInfoEXT &()
    {
      return *reinterpret_cast<VkPipelineSampleLocationsStateCreateInfoEXT*>(this);
    }

    bool operator==( PipelineSampleLocationsStateCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( sampleLocationsEnable == rhs.sampleLocationsEnable )
          && ( sampleLocationsInfo == rhs.sampleLocationsInfo );
    }

    bool operator!=( PipelineSampleLocationsStateCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineSampleLocationsStateCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    Bool32 sampleLocationsEnable;
    SampleLocationsInfoEXT sampleLocationsInfo;
  };
  static_assert( sizeof( PipelineSampleLocationsStateCreateInfoEXT ) == sizeof( VkPipelineSampleLocationsStateCreateInfoEXT ), "struct and wrapper have different size!" );

  struct PhysicalDeviceSampleLocationsPropertiesEXT
  {
    operator VkPhysicalDeviceSampleLocationsPropertiesEXT const&() const
    {
      return *reinterpret_cast<const VkPhysicalDeviceSampleLocationsPropertiesEXT*>(this);
    }

    operator VkPhysicalDeviceSampleLocationsPropertiesEXT &()
    {
      return *reinterpret_cast<VkPhysicalDeviceSampleLocationsPropertiesEXT*>(this);
    }

    bool operator==( PhysicalDeviceSampleLocationsPropertiesEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( sampleLocationSampleCounts == rhs.sampleLocationSampleCounts )
          && ( maxSampleLocationGridSize == rhs.maxSampleLocationGridSize )
          && ( memcmp( sampleLocationCoordinateRange, rhs.sampleLocationCoordinateRange, 2 * sizeof( float ) ) == 0 )
          && ( sampleLocationSubPixelBits == rhs.sampleLocationSubPixelBits )
          && ( variableSampleLocations == rhs.variableSampleLocations );
    }

    bool operator!=( PhysicalDeviceSampleLocationsPropertiesEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePhysicalDeviceSampleLocationsPropertiesEXT;

  public:
    void* pNext = nullptr;
    SampleCountFlags sampleLocationSampleCounts;
    Extent2D maxSampleLocationGridSize;
    float sampleLocationCoordinateRange[2];
    uint32_t sampleLocationSubPixelBits;
    Bool32 variableSampleLocations;
  };
  static_assert( sizeof( PhysicalDeviceSampleLocationsPropertiesEXT ) == sizeof( VkPhysicalDeviceSampleLocationsPropertiesEXT ), "struct and wrapper have different size!" );

  enum class AttachmentDescriptionFlagBits
  {
    e