#include "std_standardPkg.hh"
#include "Signal.hh"
#include "ImplicitSignal.hh"
#include "Variable.hh"
#include "SignalNetinfo.hh"
#include "STDTypes.hh"
#include <sstream>

using std::ostringstream;

PhysicalType::PhysicalType(const PhysicalType& value): ScalarType(value) {
  range = value.range;
}

PhysicalType::PhysicalType(ObjectBase::ObjectType objType,
			   const TypeInfo &typeInfo) {
  ASSERT ( typeInfo.getKind() == PHY_INFO );
  UniversalLongLongInteger initialValue(((const PhysicalTypeInfo &) typeInfo).get_left());
  
  switch(objType) {
  case ObjectBase::SIGNAL: 
    setObject( new Signal<UniversalLongLongInteger>(initialValue) );
    break;
  case ObjectBase::IMPLICIT_SIGNAL:
    setObject( new ImplicitSignal<UniversalLongLongInteger>(initialValue) );
    break;
  case ObjectBase::VARIABLE:
    setObject( new Variable<UniversalLongLongInteger>(initialValue) );
    break;
  case ObjectBase::SIGNAL_NETINFO:
    setObject( new SignalNetinfo );
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }
  range = (const PhysicalTypeInfo &) typeInfo;
}

PhysicalType::PhysicalType(ObjectBase::ObjectType objType, 
			   const UniversalLongLongInteger& val,
			   const TypeInfo& tInfo):
  ScalarType(objType) {
  switch(objType) {
  case ObjectBase::SIGNAL : 
    setObject( new Signal<UniversalLongLongInteger>(val)); 
    break;
  case ObjectBase::IMPLICIT_SIGNAL :
    setObject( new ImplicitSignal<UniversalLongLongInteger>(val)); 
    break;
  case ObjectBase::VARIABLE :
    setObject( new Variable<UniversalLongLongInteger>(val) ); 
    break;
  case ObjectBase::SIGNAL_NETINFO :
    setObject( new SignalNetinfo ); 
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }
  ASSERT ( tInfo.getKind() == PHY_INFO );
  range = (const PhysicalTypeInfo &) tInfo;
}

PhysicalType::PhysicalType(ObjectBase::ObjectType objType, 
			   const PhysicalType& value,
			   const TypeInfo& pInfo): 
  ScalarType(objType) {
  UniversalLongLongInteger val = (UniversalLongLongInteger&) value.readVal();
  switch(objType) {
  case ObjectBase::SIGNAL : 
    setObject( new Signal<UniversalLongLongInteger>(val) );
    break;
  case ObjectBase::IMPLICIT_SIGNAL :
    setObject( new ImplicitSignal<UniversalLongLongInteger>(val) ); 
    break;
  case ObjectBase::VARIABLE :
    setObject( new Variable<UniversalLongLongInteger>(val) );
    break;
  case ObjectBase::SIGNAL_NETINFO :
    setObject( new SignalNetinfo );
    break;
  default:
    cerr << "ERROR: unknown object type" << endl;
    exit(-1);
  }

  ASSERT ( pInfo.getKind() == PHY_INFO );
  range = (const PhysicalTypeInfo &) pInfo;
}

PhysicalType::PhysicalType(bool alias, const PhysicalType& actual,
			   const TypeInfo& pInfo) 
  : ScalarType(actual.getObject()->getKind(), alias) {
  setObject( actual.getObject() );

  ASSERT ( pInfo.getKind() == PHY_INFO );
  range = (const PhysicalTypeInfo &) pInfo;
}

PhysicalType::PhysicalType(const PhysicalType& value,
			   const TypeInfo& pInfo):
  ScalarType(value) {
  ASSERT ( pInfo.getKind() == PHY_INFO );
  range = (const PhysicalTypeInfo &) pInfo;
}

PhysicalType&
PhysicalType::operator = (const PhysicalType& value) {
  ((ScalarType &) *this) = (const ScalarType &) value;
  range = (const PhysicalTypeInfo &) value.range;
  return *this;
}
    
Type 
PhysicalType::get_kind() const {
  return PHYSICAL_TYPE;
}

PhysicalType
PhysicalType::vhdlPlus( const RValue &rhs ) const {
  UniversalLongLongInteger retval( getInt64Value() + rhs.getInt64Value() );
  return PhysicalType( ObjectBase::VARIABLE, retval, getTypeInfo() );
}

PhysicalType 
PhysicalType::vhdlMinus( const RValue &rhs ) const {
  UniversalLongLongInteger retval( getInt64Value() - rhs.getInt64Value() );
  return PhysicalType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

PhysicalType
PhysicalType::vhdlUnaryPlus() const {
  return *this;
}

PhysicalType 
PhysicalType::vhdlUnaryMinus() const {
  UniversalLongLongInteger retval( (LONG)-1 * getInt64Value() );
  return PhysicalType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

PhysicalType
PhysicalType::vhdlMultiply( const RValue &rhs ) const {
  UniversalLongLongInteger retval( getInt64Value() * rhs.getInt64Value() );
  return PhysicalType( ObjectBase::VARIABLE, retval, getTypeInfo() );
}

IntegerType
PhysicalType::vhdlDivide( const PhysicalType &rhs ) const {
  UniversalLongLongInteger retval( getInt64Value() / rhs.getInt64Value() );
  return IntegerType(ObjectBase::VARIABLE, retval.getIntValue(), SavantintegerType_info);
}

PhysicalType
PhysicalType::vhdlDivide( const RValue &rhs ) const {
  UniversalLongLongInteger retval( getInt64Value() / rhs.getInt64Value() );
  return PhysicalType(ObjectBase::VARIABLE, retval, getTypeInfo() );
}

PhysicalType
PhysicalType::vhdlAbs() const {
  UniversalLongLongInteger retval( dynamic_cast<const UniversalLongLongInteger &>(readVal()).vhdlAbs() );
  return PhysicalType(ObjectBase::VARIABLE, retval, getTypeInfo());
}

PhysicalType
PhysicalType::LEFT(const TypeInfo& tInfo) {
  ASSERT ( tInfo.getKind() == PHY_INFO );
  const PhysicalTypeInfo &pInfo = (const PhysicalTypeInfo &) tInfo;
  
  return PhysicalType(ObjectBase::VARIABLE,
		      UniversalLongLongInteger(pInfo.get_left()),
		      tInfo);
}

PhysicalType
PhysicalType::RIGHT(const TypeInfo& tInfo) {
  ASSERT ( tInfo.getKind() == PHY_INFO );
  const PhysicalTypeInfo &pInfo = (const PhysicalTypeInfo &) tInfo;
  
  return PhysicalType(ObjectBase::VARIABLE, pInfo.get_left(), tInfo);
}

PhysicalType
PhysicalType::HIGH(const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == PHY_INFO );
  const PhysicalTypeInfo &pInfo = (const PhysicalTypeInfo &) tInfo;
  
  if (pInfo.get_left() > pInfo.get_right()) {
    return PhysicalType(ObjectBase::VARIABLE,
			UniversalLongLongInteger(pInfo.get_left()), tInfo);
  }
  else {
    return PhysicalType(ObjectBase::VARIABLE,
			UniversalLongLongInteger(pInfo.get_right()), tInfo);
  }
}

PhysicalType
PhysicalType::LOW(const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == PHY_INFO );
  const PhysicalTypeInfo &pInfo = (const PhysicalTypeInfo &) tInfo;
  
  if (pInfo.get_left() > pInfo.get_right()) {
    return PhysicalType(ObjectBase::VARIABLE,
			UniversalLongLongInteger(pInfo.get_right()), tInfo);
  }
  else {
    return PhysicalType(ObjectBase::VARIABLE,
			UniversalLongLongInteger(pInfo.get_left()), tInfo);
  }
}

EnumerationType
PhysicalType::ASCENDING(const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == PHY_INFO );
  const PhysicalTypeInfo &pInfo = (const PhysicalTypeInfo &) tInfo; 
 
  if (pInfo.get_direction() == ArrayInfo::to){
    return SAVANT_BOOLEAN_TRUE;
  }
  else {
    return SAVANT_BOOLEAN_FALSE;
  }
}


PhysicalType
PhysicalType::SUCC(const PhysicalType& x){
  UniversalLongLongInteger val = x.getObject()->readVal();
  if(x.range.get_direction() == ArrayInfo::to && val >= x.range.get_left() &&
     val < x.range.get_right()){
    val = val + 1;
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else if ((x.range.get_direction() == ArrayInfo::downto) &&
	   (val < x.range.get_left()) &&
	   (val >= x.range.get_right())) {
    val = val + 1;
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else {
    cerr << "The parameter of `SUCC attribute not within range ";abort();
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
}

PhysicalType
PhysicalType::PRED(const IntegerType& x){
  UniversalLongLongInteger val = x.getObject()->readVal();
  if(x.range.get_direction() == ArrayInfo::to && val > x.range.get_left() &&
     val <= x.range.get_right()){
    val = val - 1;
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else if ((x.range.get_direction() == ArrayInfo::downto) &&
	   (val <= x.range.get_left()) &&
	   (val > x.range.get_right())) {
    val = val - 1;
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else {
    cerr << "The parameter of `PRED attribute not within range ";abort();
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
}

PhysicalType
PhysicalType::LEFTOF(const PhysicalType& x){
  UniversalLongLongInteger val = x.getObject()->readVal();
  if(x.range.get_direction() == ArrayInfo::to && val > x.range.get_left() &&
     val <= x.range.get_right()){
    val = val - 1;
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else if ((x.range.get_direction() == ArrayInfo::downto) &&
	   (val <= x.range.get_left()) &&
	   (val > x.range.get_right())) {
    val = val + 1;
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else {
    cerr << "The parameter of `LEFTOF attribute not within range ";abort();
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
}


PhysicalType
PhysicalType::RIGHTOF(const PhysicalType& x){
  UniversalLongLongInteger val = x.getObject()->readVal();
  if(x.range.get_direction() == ArrayInfo::to && val >= x.range.get_left() &&
     val < x.range.get_right()){
    val = val + 1;
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else if ((x.range.get_direction() == ArrayInfo::downto) &&
	   (val < x.range.get_left()) &&
	   (val >= x.range.get_right())) {
    val = val - 1;
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else {
    cerr << "The parameter of `RIGHTOF attribute not within range ";abort();
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
}

ArrayType 
PhysicalType::IMAGE(const PhysicalType& x) {
  UniversalLongLongInteger value;
  value = x.getObject()->readVal();
  ostringstream valString;
  valString << value << x.range.get_unit(0);
  ArrayType image(ObjectBase::VARIABLE, SavantstringType_info, -1, valString.str().c_str() );
  return image;
}

IntegerType
PhysicalType::POS( const PhysicalType &x, const TypeInfo &tInfo ){
  ASSERT( tInfo.getKind() == PHY_INFO );
  
  UniversalLongLongInteger val;
  val = x.getObject()->readVal();
  return IntegerType(ObjectBase::VARIABLE, (int) val.val, SavantintegerType_info);
}

PhysicalType
PhysicalType::VAL(const PhysicalType& x, const TypeInfo& tInfo) {
  ASSERT ( tInfo.getKind() == PHY_INFO );
  const PhysicalTypeInfo &pInfo = (const PhysicalTypeInfo &) tInfo;
  
  UniversalLongLongInteger val;
  val = x.getObject()->readVal();
  if(val >= pInfo.get_left() && val <= pInfo.get_right()){
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
  else {
    cerr << "The Result of `VAL attribute not withing range";
    abort();
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
}

PhysicalType
PhysicalType::VAL(const IntegerType& x, const TypeInfo& tInfo) {
  ASSERT ( tInfo.getKind() == PHY_INFO );
  const PhysicalTypeInfo &pInfo = (const PhysicalTypeInfo &) tInfo;
  
  UniversalLongLongInteger val;
  val = x.getObject()->readVal();
  if(val >= pInfo.get_left() && val <= pInfo.get_right()){
    return PhysicalType(ObjectBase::VARIABLE, val, pInfo);
  }
  else {
    cerr << "The Result of `VAL attribute not withing range";
    abort();
    return PhysicalType(ObjectBase::VARIABLE, val, x.getTypeInfo());
  }
}

const VHDLData&
PhysicalType::leftValue() {
  static UniversalLongLongInteger retValue(0);

  retValue = UniversalLongLongInteger(range.get_left());

  return retValue;
}

const VHDLVTime
PhysicalType::getVTime() const {
  return VHDLVTime(longlongint(dynamic_cast<const UniversalLongLongInteger &>(getObject()->readVal())));
}

TypeInfo &
PhysicalType::getTypeInfo() const {
  return const_cast<PhysicalTypeInfo &>(range);
}
