/* 
 * Miscellaneous utility functions
 *
 * This file is part of GTick
 * 
 *
 * Copyright (c) 1999, Alex Roberts
 * Copyright (c) 2003, Roland Stigge <stigge@antcom.de>
 *
 * GTick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GTick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GTick; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>

/* regular GNU system includes */
#include <stdio.h>
#include <unistd.h>
#ifdef HAVE_STDLIB_H
  #include <stdlib.h>
#endif
#ifdef HAVE_SYS_TIME_H
  #include <sys/time.h>
#endif
#include <string.h>
#include <time.h>
#include <pwd.h>
#include <sys/stat.h>
#include <errno.h>

/* own headers */
#include "globals.h"
#include "util.h"

/*
 * Subtract the `struct timeval' values X and Y,
 * storing the result in RESULT.
 * Return 1 if the difference is negative, otherwise 0.
 *
 * This function comes from the GNU C Library documentation.
 */
int timeval_subtract(struct timeval* result,
                     struct timeval* x, struct timeval* y)
{
  /* Perform the carry for the later subtraction by updating y. */
  if (x->tv_usec < y->tv_usec) {
    int nsec = (y->tv_usec - x->tv_usec) / 1000000 + 1;
    y->tv_usec -= 1000000 * nsec;
    y->tv_sec += nsec;
  }
  if (x->tv_usec - y->tv_usec > 1000000) {
    int nsec = (x->tv_usec - y->tv_usec) / 1000000;
    y->tv_usec += 1000000 * nsec;
    y->tv_sec -= nsec;
  }

  /* Compute the time remaining to wait.
     tv_usec is certainly positive. */
  result->tv_sec = x->tv_sec - y->tv_sec;
  result->tv_usec = x->tv_usec - y->tv_usec;

  /* Return 1 if result is negative. */
  return x->tv_sec < y->tv_sec;
}

/*
 * returns a string where all occurences of the character c are removed
 * from the input string s
 * NOTE: caller has to free() the result
 */
char* stripchr(char* s, char c) {
  int count = 1; /* trailing space */
  char* temp;
  char* result;

  for (temp = s; *temp; temp++) {
    if (*temp != c)
      count++;
  }
  if ((result = (char*) malloc (count))) {
    temp = result;
    while (*s) {
      if (*s != c) {
	*temp = *s;
	temp ++;
      }
      s++;
    }
    *temp = '\0';
  }

  return result;
}

/*
 * return home dir, returned buffer must not be freed but will be overwritten
 * by subsequent calls (or calls to getpwuid())
 * returns NULL on error
 */
char* get_homedir(void) {
  struct passwd* user;

  if (!(user = getpwuid(getuid()))) {
    return NULL;
  }
  
  if (user->pw_dir == NULL || !strcmp(user->pw_dir, "") ||
      user->pw_dir[strlen(user->pw_dir) - 1] == '/') {
    return NULL;
  }

  return user->pw_dir;
}

/*
 * returns name of rc file for this program, NULL on error
 *
 * NOTE: caller has to free the returned buffer himself
 */
char* get_rc_filename(void) {
  char* homedir = get_homedir();
  char* filename;
  
  if (!homedir || asprintf(&filename, "%s/.%src", homedir, PACKAGE) < 0) {
    return NULL;
  } else {
    return filename;
  }
}

/*
 * forks and executes given command (with arguments) via sh
 */
void execute(char *command) {
  pid_t result = fork();
  char *argv[] = {"sh", "-c", command, NULL};

  if (result == -1) { /* error */
    fprintf(stderr, "Fork error.\n");
  } else if (result == 0) { /* we are the child */
    if (execvp("sh", argv)) {
      fprintf(stderr, "Exec error.\n");
      exit(1);
    }
  }
}

/*
 * Return a double value from a string in "C" locale format
 */
double strtod_c(const char* s) {
  char* saved_locale;
  double result;

  saved_locale = strdup (setlocale (LC_NUMERIC, NULL));
  if (saved_locale == NULL) {
    fprintf(stderr, "strtod_c: Out of memory.\n");
    exit(1);
  }
  setlocale (LC_NUMERIC, "C");

  result = strtod(s, NULL);

  setlocale (LC_NUMERIC, saved_locale);
  free(saved_locale);

  return result;
}

/*
 * Return a string value from a double in "C" locale format
 * NOTE: caller has to free() the result himself
 */
char* dtostr_c(double d, int digits) {
  char* saved_locale;
  char* result;

  saved_locale = strdup (setlocale (LC_NUMERIC, NULL));
  if (saved_locale == NULL) {
    fprintf(stderr, "dtostr_c: Out of memory.\n");
    exit(1);
  }
  setlocale (LC_NUMERIC, "C");

  if (asprintf(&result, "%.*f", digits, d) == -1) {
    fprintf(stderr, "dtostr_c: Out of memory.\n");
    exit(1);
  }

  setlocale (LC_NUMERIC, saved_locale);
  free(saved_locale);

  return result;
}

