########################################################################
#
# File Name:            Common.py
#
# Documentation:        http://docs.4suite.org/4Rdf/Inference/Common.py.html
#
"""

WWW: http://4suite.org/4RDF         e-mail: support@4suite.org

Copyright (c) 1999 Fourthought Inc, USA.   All Rights Reserved.
See  http://4suite.org/COPYRIGHT  for license and copyright information
"""
import types,string

class ArgumentTypes:
    STRING_LIST = 1
    SKOLEM_VARIABLE = 2
    VARIABLE = 3
    QUERY = 4
    ACTION = 5

class Context:
    def __init__(self,
                 variables = None,
                 external_params = None,
                 fire_rules = 0,
                 nss = None):
        if variables == None:
            variables = {}
        if external_params == None:
            external_params = {}
        if nss == None:
            nss = {}
        self.variables = variables
        self.nss = nss
        self.external_params = external_params
        self.fireRules = fire_rules

    def __repr__(self):
        sv = ""
        if hasattr(self,"skolemVariables"):
            sv = ", skolemVariables %s" % self.skolemVariables
        st = "<Context at %s with variables %s, fireRules = %d nss %s%s>" % (id(self),self.variables,self.fireRules,self.nss,sv)
        return st

class Argument:
    def __init__(self,name,typ):
        self.name = name
        self.type = typ
    def execute(self,context):
        raise "Musr Override"

class SkolemVariableArgument(Argument):
    def __init__(self,name):
        Argument.__init__(self,name,ArgumentTypes.SKOLEM_VARIABLE)
    def execute(self,infEng,context):
        if not hasattr(context,'skolemVariables'):
            return None
        return context.skolemVariables.get(self.name)

    def _4rdf_dump(self,indent = 0):
        iStr = "\t"*indent
        rt = iStr + "<ril:variable name='%s'/>\n" % self.name
        return rt



class LiteralArgument(Argument):
    def __init__(self,name,typ,value):
        self.value = value
        Argument.__init__(self,name,typ)
    def execute(self,infEng,context):
        return self.value

class StringListLiteralArgument(LiteralArgument):
    def __init__(self,name,value):
        if type(value) != types.ListType:
            raise "Must be a list"
        LiteralArgument.__init__(self,name,ArgumentTypes.STRING_LIST,value)

    def _4rdf_dump(self,indent = 0):
        iStr = "\t"*indent
        rt = iStr + "<ril:string-list>\n"
        for st in self.value:
            rt = rt + iStr  + "\t<ril:string>%s</ril:string>\n" % st
        rt = rt + iStr + "</ril:string-list>\n"
        return rt

def StringLiteralArgument(name,value):
    return StringListLiteralArgument(name,[value])

class VariableReferenceArgument(Argument):
    def __init__(self,name):
        Argument.__init__(self,name,ArgumentTypes.VARIABLE)
        self.prefix,self.localName = SplitQName(name)
        
    def execute(self,infEng,context):
        fullName = self.name
        if self.prefix:
            uri = context.resolvePrefix(self.prefix)
            fullName = uri + '#' + self.localName
        v = context.variables.get(fullName,[])
        if type(v) not in [types.ListType,types.TupleType]:
            return (v,)
        return v

    def _4rdf_dump(self,indent = 0):
        iStr = "\t"*indent
        rt = iStr + "<ril:variable-ref name='%s'/>\n" % self.name
        return rt



def SplitQName(qname):
    index = string.find(qname, ':')
    if index == -1:
        sName = ('', qname)
    else:
        (part1, part2) = (qname[:index], qname[index+1:])
        sName = (part1, part2)    
    return sName

class Predicate:
    def __init__(self, id_, negative):
        self.id = id_
        self.negative = negative
        return


