/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "MGCellRendererBlob.h"
#include "myx_public_interface.h"
#include "myg_gtkutils.h"
#include "myg_utils.h"
#include "myqb.h"
#include "MGBlobEditor.h"

#include <gtkmm.h>

CellRendererBlob::CellRendererBlob(bool binary)
  : Glib::ObjectBase(typeid(CellRendererBlob)),
  Gtk::CellRenderer(),
  _property_field(*this, "field", ""),
  _property_editable(*this, "editable", false),
  _property_background(*this, "background", ""),
  _binary(binary)
{
  _max_text_length= 100;

  property_mode()= Gtk::CELL_RENDERER_MODE_ACTIVATABLE;
}


CellRendererBlob::~CellRendererBlob()
{
}


void CellRendererBlob::set_blob_icon(const Glib::RefPtr<Gdk::Pixbuf> &icon)
{
  _blob_icon= icon;
}


void CellRendererBlob::set_overlay_icons(const Glib::RefPtr<Gdk::Pixbuf> &clear_icon,
                                         const Glib::RefPtr<Gdk::Pixbuf> &edit_icon,
                                         const Glib::RefPtr<Gdk::Pixbuf> &load_icon,
                                         const Glib::RefPtr<Gdk::Pixbuf> &save_icon,
                                         const Glib::RefPtr<Gdk::Pixbuf> &view_icon)
{
  _clear_icon= clear_icon;
  _edit_icon= edit_icon;
  _load_icon= load_icon;
  _save_icon= save_icon;
  _view_icon= view_icon;
}


void CellRendererBlob::set_max_text_width(guint width)
{
  _max_text_length= width;
}


Glib::PropertyProxy<std::string> CellRendererBlob::property_field()
{
  return _property_field.get_proxy();
}


Glib::PropertyProxy<bool> CellRendererBlob::property_editable()
{
  return _property_editable.get_proxy();
}


Glib::PropertyProxy<std::string> CellRendererBlob::property_background()
{
  return _property_background.get_proxy();
}


void CellRendererBlob::get_size_vfunc(Gtk::Widget& widget,
                                      const Gdk::Rectangle* cell_area,
                                      int* x_offset, int* y_offset,
                                      int* width,    int* height)
{
  gpointer fdata;
  gsize fsize;
  
  str2data(((std::string)property_field()).c_str(), &fdata, &fsize);
  
  if (!_layout)
  {
    _layout= widget.create_pango_layout("ABC123j");
    _layout->get_pixel_size(_text_char_width, _text_height);
    _text_char_width/= 7;
  }
  
  if (_binary)
  {
    if (width)
      *width= _blob_icon->get_width()+4;
    if (height)
      *height= _blob_icon->get_height();
    if (x_offset)
      *x_offset= 0;
    if (y_offset)
      *y_offset= 0;
  }
  else
  {
    //XXX
    if (width)
    {
      if (_max_text_width >= _max_text_length * _text_char_width)
        *width= _max_text_length * _text_char_width;
      else
      {
        int h;
        Glib::ustring text;
    
        if (fdata)
        {
          if (fsize > _max_text_length)
            text= Glib::ustring((char*)fdata, 0, _max_text_length)+"...";
          else
            text= Glib::ustring((char*)fdata);

          text= strreplace(text, "\r\n", "\xc2\xab\xc2\xb6");
          text= strreplace(text, "\n", "\xc2\xb6");

          _layout->set_text(text);

          _layout->get_pixel_size(*width, h);
        
          if (*width > (int)_max_text_width)
            _max_text_width= *width;
        }
        else
          *width= _max_text_length * _text_char_width;
      }
    }
    
    if (height)
      *height= _text_height;
  }

  if (width)
    *width+= 5 * (_clear_icon->get_width()+4);
}


void CellRendererBlob::render_vfunc(const Glib::RefPtr<Gdk::Window>& window, Gtk::Widget& widget,
                                    const Gdk::Rectangle& background_area, const Gdk::Rectangle& cell_area,
                                    const Gdk::Rectangle& expose_area, Gtk::CellRendererState flags)
{
  gpointer fdata; 
  gsize fsize; 
  Gdk::Color color;
  std::string bgcolor= property_background();

  if (!_gc)
    _gc= Gdk::GC::create(window);

  if (!bgcolor.empty())
  {
    color= Gdk::Color(bgcolor.c_str());
    widget.get_colormap()->alloc_color(color);
    _gc->set_foreground(color);

    window->draw_rectangle(_gc, true,
                           background_area.get_x(), background_area.get_y(),
                           background_area.get_width(), background_area.get_height());
  }

  str2data(((std::string)property_field()).c_str(), &fdata, &fsize);

  if (_binary)
  {
    if (fdata && fsize > 0)
      window->draw_pixbuf(_gc, _blob_icon, 0, 0, 
                          cell_area.get_x(), cell_area.get_y()+(cell_area.get_height()-_blob_icon->get_height())/2, -1, -1, 
                          Gdk::RGB_DITHER_NONE, 0, 0);
  }
  else
  {    
    if (fdata)
    {
      Glib::ustring text;
      
      if (fsize > _max_text_length)
        text= Glib::ustring((char*)fdata, 0, _max_text_length)+"...";
      else
        text= Glib::ustring((char*)fdata);

      text= strreplace(text, "\r\n", "\xc2\xab\xc2\xb6");
      text= strreplace(text, "\n", "\xc2\xb6");

      _layout->set_text(text);

      Gtk::StateType state= Gtk::STATE_NORMAL;

      if ((flags & Gtk::CELL_RENDERER_SELECTED) != 0)
        state= (widget.has_focus()) ? Gtk::STATE_SELECTED : Gtk::STATE_ACTIVE;

      window->draw_layout(widget.get_style()->get_text_gc(state), cell_area.get_x(), cell_area.get_y() + (cell_area.get_height() - _text_height)/2, _layout);
    }
  }

  if (property_editable())
  {
    int x= cell_area.get_x()+cell_area.get_width();
    
    if (fdata)
    {
      x-= _clear_icon->get_width()+4;
      window->draw_pixbuf(_gc, _clear_icon, 0, 0, 
                          x, cell_area.get_y()+(cell_area.get_height()-_clear_icon->get_height())/2, -1, -1, 
                          Gdk::RGB_DITHER_NONE, 0, 0);
    }

    x-= _load_icon->get_width()+4;
    window->draw_pixbuf(_gc, _load_icon, 0, 0,
                        x, cell_area.get_y()+(cell_area.get_height()-_load_icon->get_height())/2, -1, -1,
                        Gdk::RGB_DITHER_NONE, 0, 0);
/*
    x-= _edit_icon->get_width()+4;
    window->draw_pixbuf(_gc, _edit_icon, 0, 0,
                        x, cell_area.get_y()+(cell_area.get_height()-_edit_icon->get_height())/2, -1, -1,
                        Gdk::RGB_DITHER_NONE, 0, 0);
 */

    x-= _save_icon->get_width()+4;
    window->draw_pixbuf(_gc, _save_icon, 0, 0,
                        x, cell_area.get_y()+(cell_area.get_height()-_save_icon->get_height())/2, -1, -1,
                        Gdk::RGB_DITHER_NONE, 0, 0);
    
    x-= _view_icon->get_width()+4;
    window->draw_pixbuf(_gc, _view_icon, 0, 0,
                        x, cell_area.get_y()+(cell_area.get_height()-_view_icon->get_height())/2, -1, -1,
                        Gdk::RGB_DITHER_NONE, 0, 0);
  }
  else
  {
    int x= cell_area.get_x()+cell_area.get_width();

    x-= _save_icon->get_width()+4;
    window->draw_pixbuf(_gc, _save_icon, 0, 0,
                        x, cell_area.get_y()+(cell_area.get_height()-_save_icon->get_height())/2, -1, -1,
                        Gdk::RGB_DITHER_NONE, 0, 0);
    
    x-= _view_icon->get_width()+4;
    window->draw_pixbuf(_gc, _view_icon, 0, 0,
                        x, cell_area.get_y()+(cell_area.get_height()-_view_icon->get_height())/2, -1, -1,
                        Gdk::RGB_DITHER_NONE, 0, 0);
  }
}


bool CellRendererBlob::activate_vfunc(GdkEvent* event,
                                      Gtk::Widget& widget,
                                      const Glib::ustring& path,
                                      const Gdk::Rectangle& background_area,
                                      const Gdk::Rectangle& cell_area,
                                      Gtk::CellRendererState flags)
{
  gpointer fdata; 
  gsize fsize; 
  int cx;
  
  str2data(((std::string)property_field()).c_str(), &fdata, &fsize);

  if (event->type == GDK_BUTTON_PRESS)
  {
    cx= (int)event->button.x;

    if (property_editable())
    {
      int x= cell_area.get_x()+cell_area.get_width();

      if (fdata)
      {
        x-= _clear_icon->get_width()+4;
        if (cx >= x && cx < x+_clear_icon->get_width())
        {
          clear_clicked(widget.get_parent(), path);
          return true;
        }
      }
        
      x-= _load_icon->get_width()+4;
      if (cx >= x && cx < x+_load_icon->get_width())
      {
        load_clicked(widget.get_parent(), path);
        return true;
      }
/*
      x-= _edit_icon->get_width()+4;
      if (cx >= x && cx < x+_edit_icon->get_width())
      {
        edit_clicked(widget.get_parent(), path);
        return true;
      }
 */
      
      x-= _save_icon->get_width()+4;
      if (cx >= x && cx < x+_save_icon->get_width())
      {
        save_clicked(widget.get_parent(), path);
        return true;
      }

      x-= _view_icon->get_width()+4;
      if (cx >= x && cx < x+_view_icon->get_width())
      {
        view_clicked(widget.get_parent(), path);
        return true;
      }
    }
    else
    {
      int x= cell_area.get_x()+cell_area.get_width();
      
      x-= _save_icon->get_width()+4;
      if (cx >= x && cx < x+_save_icon->get_width())
      {
        save_clicked(widget.get_parent(), path);
        return true;
      }

      x-= _view_icon->get_width()+4;
      if (cx >= x && cx < x+_view_icon->get_width())
      {
        view_clicked(widget.get_parent(), path);
        return true;
      }
    }
  }
  return false;
}


void CellRendererBlob::clear_clicked(Gtk::Widget *parent, const Glib::ustring &path)
{
  _signal_edited.emit(path, NULL, 0);
}


void CellRendererBlob::save_clicked(Gtk::Widget *parent, const Glib::ustring &path)
{
  gpointer fdata; 
  gsize fsize; 
  str2data(((std::string)property_field()).c_str(), &fdata, &fsize);

  Gtk::FileSelection fsel(_("Save Field Data"));

  if (fsel.run() == Gtk::RESPONSE_OK)
  {
    gsize written;

    try 
    {
      Glib::RefPtr<Glib::IOChannel> channel= Glib::IOChannel::create_from_file(fsel.get_filename(),"w+");
      channel->set_encoding();
      channel->write((char*)fdata, fsize, written);
      channel->close();
    } 
    catch (Glib::FileError &exc)
    {
      myg_show_error(ufmt(_("Could not save data to file '%s'.\n"), fsel.get_filename().c_str())+exc.what());
    }
  }
}


void CellRendererBlob::load_clicked(Gtk::Widget *parent, const Glib::ustring &path)
{
  Gtk::FileSelection fsel(_("Load Field Data"));

  if (fsel.run() == Gtk::RESPONSE_OK)
  {
    Glib::RefPtr<Glib::IOChannel> channel;
    try 
    {
      channel= Glib::IOChannel::create_from_file(fsel.get_filename(),"r");
      channel->set_encoding();
    } 
    catch (Glib::Error &exc)
    {
      myg_show_error(ufmt(_("Could not open file '%s'"), fsel.get_filename().c_str())+"\n"+exc.what());
      return;
    }

    try
    {
      gsize size= get_file_size(fsel.get_filename().c_str());
      gchar *buffer= g_new0(gchar, size);

      channel->read(buffer, size, size);

      _signal_edited.emit(path, buffer, size);
      
      g_free(buffer);
    }
    catch (Glib::Error &exc)
    {
      myg_show_error(ufmt(_("Could not read data from file '%s'"), fsel.get_filename().c_str())+"\n"+exc.what());
    }
    channel->close();
  }
}


static MGBlobEditor::ViewType get_flags_for_data(char *fdata, gsize fsize,
                                                 bool binary)
{
  MGBlobEditor::ViewType flags;
  
  if (fsize > 0 && myx_guess_image_format(fdata, fsize)!=MYX_IMG_UNKNOWN)
    flags= MGBlobEditor::ViewType(MGBlobEditor::VImage|MGBlobEditor::VBinary);
  else if (binary)
    flags= MGBlobEditor::VBinary;
  else
    flags= MGBlobEditor::ViewType(MGBlobEditor::VText|MGBlobEditor::VBinary);

  return flags;
}


void CellRendererBlob::view_clicked(Gtk::Widget *parent, const Glib::ustring &path)
{
  gpointer fdata; 
  gsize fsize; 

  str2data(((std::string)property_field()).c_str(), &fdata, &fsize);

  MGBlobEditor *editor= new MGBlobEditor(get_flags_for_data((char*)fdata, fsize, _binary));
  editor->set_data(fdata, fsize);
  editor->resize(400, 300);
  editor->set_delete_on_close();

  editor->set_position(Gtk::WIN_POS_CENTER_ON_PARENT);
  editor->set_transient_for(*(Gtk::Window*)parent->get_toplevel());
  editor->property_destroy_with_parent()=true;

  editor->show();
}


void CellRendererBlob::editor_save(MGBlobEditor *editor)
{
  gpointer buffer;
  gsize size;
  Glib::ustring path;
  
  editor->get_data(buffer, size);
  
  path= (char*)((Glib::Object*)editor)->get_data("path");
  _signal_edited.emit(path, buffer, size);
  
  g_free(buffer);
}


void CellRendererBlob::edit_clicked(Gtk::Widget *parent, const Glib::ustring &path)
{
  gpointer fdata; 
  gsize fsize; 

  str2data(((std::string)property_field()).c_str(), &fdata, &fsize);

  MGBlobEditor *editor= new MGBlobEditor(get_flags_for_data((char*)fdata, fsize, _binary), true);
  editor->set_data(fdata, fsize);
  editor->resize(400, 300);
  editor->set_delete_on_close();

  ((Glib::Object*)editor)->set_data("path", g_strdup(path.c_str()), g_free);
  
  editor->set_position(Gtk::WIN_POS_CENTER_ON_PARENT);
  editor->set_transient_for(*(Gtk::Window*)parent->get_toplevel());
  editor->property_destroy_with_parent()=true;
  editor->signal_save().connect(SigC::bind<MGBlobEditor*>(SigC::slot(*this,&CellRendererBlob::editor_save),editor));

  editor->show();
}
