unit AuxFuncsGrt;

interface

uses
  gnugettext, SysUtils, Classes,
  myx_public_interface, myx_grt_public_interface,
  TntClasses, TntSysUtils;

type
  EGrtError = class(Exception)
  private
    FErrorNumber: Integer;
    FDescription: WideString;
  public
    constructor Create(ErrorNumber: Integer; Description: WideString);

    property ErrorNumber: Integer read FErrorNumber;
    property Description: WideString read FDescription;
  end;

  GrtMethodExecutionThread = class(TThread)
  private
    FExecutionFinished: Boolean;
    MultiReadSync: TMultiReadExclusiveWriteSynchronizer;

    FStartTime,
      FEndTime: TDateTime;
    FGrt: Pointer;
    FModulName,
      FFunctionName: WideString;
    FSearchParent: Boolean;
    FFunctionArgument: Pointer;
    FError: PMYX_GRT_ERROR;
    FResult: Pointer;

    function GetExecutionFinished: Boolean;
    function GetExecutionTime: TDateTime;
    function GetResult: Pointer;
  public
    constructor Create(Grt: Pointer; ModulName: WideString;
      FunctionName: WideString; SearchParent: Boolean;
      FunctionArgument: Pointer; Error: PMYX_GRT_ERROR);
    destructor Destroy; override;

    procedure Execute; override;
    property ExecutionFinished: Boolean read GetExecutionFinished;
    property ExecutionTime: TDateTime read GetExecutionTime;
    property Result: Pointer read GetResult;
  end;

function CallGrtModuleMethod(Grt: Pointer;
  ModulName: WideString; FunctionName: WideString;
  FunctionArgument: Pointer; MessageOutput: TTntStrings = nil;
  SearchParent: Boolean = True; AllowNullAsResult: Boolean = False): Pointer;

function BuildGrtParamList(Params: array of const): Pointer;

function GetListMemberCount(Grt: Pointer; StructName: WideString;
  OnlyCheck: Boolean): Integer;
function GetListMember(Grt: Pointer; StructName: WideString;
  Index: Integer): Pointer;

implementation

// -----------------------------------------------------------------------------

constructor EGrtError.Create(ErrorNumber: Integer; Description: WideString);

begin
  FErrorNumber := ErrorNumber;
  FDescription := Description;
end;

// -----------------------------------------------------------------------------

constructor GrtMethodExecutionThread.Create(
  Grt: Pointer; ModulName: WideString;
  FunctionName: WideString; SearchParent: Boolean;
  FunctionArgument: Pointer; Error: PMYX_GRT_ERROR);

begin
  inherited Create(True);

  FExecutionFinished := False;
  FGrt := Grt;
  FModulName := ModulName;
  FFunctionName := FunctionName;
  FSearchParent := SearchParent;
  FFunctionArgument := FunctionArgument;
  FError := Error;

  MultiReadSync := TMultiReadExclusiveWriteSynchronizer.Create;
end;

// -----------------------------------------------------------------------------

destructor GrtMethodExecutionThread.Destroy;

begin
  MultiReadSync.Free;

  inherited Destroy;
end;

// -----------------------------------------------------------------------------

function GrtMethodExecutionThread.GetExecutionTime: TDateTime;

begin
  Result := FEndTime - FStartTime;
end;

// -----------------------------------------------------------------------------

function GrtMethodExecutionThread.GetResult: Pointer;

begin
  Result := FResult;
end;

// -----------------------------------------------------------------------------

function GrtMethodExecutionThread.GetExecutionFinished: Boolean;

begin
  MultiReadSync.BeginRead;
  try
    Result := FExecutionFinished;
  finally
    MultiReadSync.EndRead;
  end;
end;

// -----------------------------------------------------------------------------

procedure GrtMethodExecutionThread.Execute;

begin
  FStartTime := Now;

  //Search for the function and call it with a list of params
  FResult := myx_grt_function_get_and_call(FGrt,
    FModulName, FFunctionName, Ord(FSearchParent),
    FFunctionArgument, @FError);

  FEndTime := Now;

  MultiReadSync.BeginWrite;
  try
    FExecutionFinished := True;
  finally
    MultiReadSync.EndWrite;
  end;
end;

// -----------------------------------------------------------------------------

function CallGrtModuleMethod(Grt: Pointer;
  ModulName: WideString; FunctionName: WideString;
  FunctionArgument: Pointer; MessageOutput: TTntStrings = nil;
  SearchParent: Boolean = True; AllowNullAsResult: Boolean = False): Pointer;

var
  Error: MYX_GRT_ERROR;
  FunctionErrorString: WideString;
  PMsgs: PMYX_GRT_MSGS;
  Msgs: TMYX_GRT_MSGS;
  i, j: Integer;
  s: WideString;
  ExecTime: TDateTime;
  StartTime: TDateTime;
  //ExecThread: GrtMethodExecutionThread;

begin
  {ExecThread := GrtMethodExecutionThread.Create(Grt,
    ModulName, FunctionName, SearchParent,
    FunctionArgument, @error);
  try
    //Start thread execution
    ExecThread.Resume;

    //Wait till the execution is completed
    while (not(ExecThread.ExecutionFinished)) do
      Sleep(100);

    Result := ExecThread.Result;

    ExecTime := ExecThread.ExecutionTime;
  finally
    ExecThread.Free;
  end;}

  StartTime := Now;

  //Search for the function and call it with a list of params
  Result := myx_grt_function_get_and_call(Grt,
    ModulName, FunctionName, Ord(SearchParent),
    FunctionArgument, @error);

  ExecTime := Now - StartTime;

  //This will return a dict either with a key value or with
  //two keys error and stack

  //Check the result for error keys
  FunctionErrorString := myx_grt_function_check_error(Result,
    Ord(AllowNullAsResult));

  //Show error
  if (Error<>MYX_GRT_NO_ERROR) or
    (FunctionErrorString<>'') then
    raise EGrtError.Create(Ord(Error), FunctionErrorString);

  //If there was no error we have a value key
  //Return the contents of the key as result
  if (Result<>nil) then
    Result := myx_grt_dict_item_get_value(Result, 'value');

  if (MessageOutput<>nil) then
  begin
    //Check for messages
    PMsgs := myx_grt_get_msgs(Grt, 0);
    if (PMsgs<>nil)then
    begin
      try
        Msgs := TMYX_GRT_MSGS.Create(PMsgs);
        try
          for i:=0 to Msgs.msgs.Count-1 do
          begin
            if (Msgs.msgs[i].msg_type=1) then
              s := 'ERROR: '
            else
              s := '';

            MessageOutput.Add(s +
              WideStringReplace(
                WideStringReplace(
                  Msgs.msgs[i].msg, #13#10, #10, [rfReplaceAll], False),
                #10, #13#10, [rfReplaceAll], False));

            for j:=0 to Msgs.msgs[i].msg_detail.strings.Count-1 do
            begin
              MessageOutput.Add(
                WideStringReplace(
                  WideStringReplace(
                    Msgs.msgs[i].msg_detail.strings[j], #13#10, #10, [rfReplaceAll], False),
                  #10, #13#10, [rfReplaceAll], False));
            end;

            MessageOutput.Add('');
          end;
        finally
          Msgs.Free;
        end;
      finally
        myx_grt_free_msgs(PMsgs);
      end;
    end;

    MessageOutput.Add(
      _('Executed in ') +
        FormatDateTime('n:ss:zzz', ExecTime) + '.' + #13#10);
  end;
end;

// -----------------------------------------------------------------------------

function BuildGrtParamList(Params: array of const): Pointer;

var
  ParamList: Pointer;
  Param: Pointer;
  I: Integer;

begin
  ParamList := myx_grt_list_new(MYX_ANY_VALUE, '');

  for I:=0 to High(Params) do
  begin
    with Params[I] do
      case VType of
        vtInteger:
          Param := myx_grt_value_from_int(VInteger);
        vtExtended:
          Param := myx_grt_value_from_real(VExtended^);
        vtWideString:
          Param := myx_grt_value_from_string(WideString(VWideString));
        vtString:
          Param := myx_grt_value_from_string(VString^);
        vtAnsiString:
          Param := myx_grt_value_from_string(string(VAnsiString));
        vtPChar:
          Param := myx_grt_value_from_string(VPChar);
        vtPointer:
          Param := VPointer;
      else
        raise EInOutError.Create(_('BuildGrtParamList called with unsupported parameter type.'));
      end;

    myx_grt_list_item_add(ParamList, Param);
  end;

  Result := ParamList;
end;

// -----------------------------------------------------------------------------

function GetListMemberCount(Grt: Pointer; StructName: WideString;
  OnlyCheck: Boolean): Integer;

var
  PStruct,
    PMember: Pointer;
  i,
    count: integer;

begin
  PStruct := myx_grt_struct_get(Grt, StructName);

  count := 0;
  for i:=0 to myx_grt_struct_get_member_count_total_excluding_struct(
    Grt, PStruct, 'db.DatabaseObject')-1 do
  begin
    PMember := myx_grt_struct_get_member_by_index_total(
      Grt, PStruct, i);

    if (myx_grt_struct_member_get_type(PMember) = MYX_LIST_VALUE) and
      (myx_grt_struct_member_get_content_type(PMember) = MYX_DICT_VALUE) then
    begin
      inc(count);

      if (OnlyCheck) then
        break;
    end;
  end;

  Result := count;
end;

// -----------------------------------------------------------------------------

function GetListMember(Grt: Pointer; StructName: WideString;
  Index: Integer): Pointer;

var
  PStruct: Pointer;
  i,
    Count: integer;

begin
  Result := nil;
  PStruct := myx_grt_struct_get(Grt, StructName);

  count := 0;
  for i:=0 to myx_grt_struct_get_member_count_total(Grt, PStruct)-1 do
  begin
    Result := myx_grt_struct_get_member_by_index_total(
      Grt, PStruct, i);

    if (myx_grt_struct_member_get_type(Result) = MYX_LIST_VALUE) and
      (myx_grt_struct_member_get_content_type(Result) = MYX_DICT_VALUE) then
    begin
      inc(Count);

      if (Index = Count-1) then
        break;
    end;
  end;
end;

// -----------------------------------------------------------------------------

end.
