/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#include <myx_public_interface.h>
#include <myx_grt_public_interface.h>
#include <myx_grt_builtin_module_public_interface.h>

#define DBUG(msg) if (getenv("DEBUG_GRT_MYSQL")) g_message("%s", msg)


static void get_connection_info(MYX_GRT_VALUE *value, MYX_USER_CONNECTION *conn)
{
  conn->username= (char*)myx_grt_dict_item_get_as_string(value, "username");
  conn->password= (char*)myx_grt_dict_item_get_as_string(value, "password");
  conn->hostname= (char*)myx_grt_dict_item_get_as_string(value, "hostname");
}



static MYX_GRT_VALUE *make_error_dict(const char *message, const char *detail)
{
  return myx_grt_dict_create(NULL,
    "error", MYX_STRING_VALUE, message,
    "detail", MYX_STRING_VALUE, detail,
    NULL);
}


static MYSQL *connect_mysql(MYX_GRT_VALUE *param, MYX_GRT_VALUE **retval)
{
  MYX_USER_CONNECTION conn;
  MYX_GRT_VALUE *value;
  MYSQL *mysql;

  memset(&conn, 0, sizeof(MYX_USER_CONNECTION));
  
  value= myx_grt_dict_item_get_value(param, "connect_info");
  if (!value)
  {
    *retval= make_error_dict("bad parameter", "missing connect_info value");
    return NULL;
  }

  get_connection_info(value, &conn);

  mysql= myx_mysql_init();
  if (!mysql)
  {
    *retval= make_error_dict("out of memory", "");
    return NULL;
  }

  DBUG("connecting to mysql");

  if (myx_connect_to_instance(&conn, mysql) < 0)
  {
    myx_mysql_close(mysql);
    *retval= make_error_dict("can't connect to server", mysql_error(mysql));
    return NULL;
  }

  *retval= NULL;

  return mysql;
}


MYX_GRT_VALUE *get_schema_list(MYX_GRT_VALUE *param, void *data)
{
  MYX_GRT_VALUE *result= NULL, *tmp;
  MYX_CATALOGS *catalogs;
  MYSQL *mysql;
  unsigned int i, s;

  if (!param)
  {
    result= myx_grt_dict_create(NULL,
      "@status", MYX_STRING_VALUE, "error",
      "@error", 0, tmp= make_error_dict("bad parameter", "missing parameter"),
      NULL);
    myx_grt_value_release(tmp);
    return result;
  }

  mysql= connect_mysql(param, &result);
  if (!mysql)
  {
    tmp= myx_grt_dict_create(NULL,
      "@status", MYX_STRING_VALUE, "error",
      "@error", 0, result,
      NULL);
    myx_grt_value_release(result);
    return tmp;
  }

  catalogs= myx_get_catalogs(mysql);
  if (!catalogs)
  {
    myx_mysql_close(mysql);
    result=  myx_grt_dict_create(NULL,
      "@status", MYX_STRING_VALUE, "error",
      "@error", 0, tmp= make_error_dict("can't fetch catalogs", ""),
      NULL);
    myx_grt_value_release(tmp);
    return result;
  }
  myx_mysql_close(mysql);

  result= myx_grt_list_new(MYX_STRING_VALUE, NULL);
  for (i= 0; i < catalogs->catalogs_num; i++)
  {
    MYX_CATALOG *cat= catalogs->catalogs+i;

    for (s= 0; s < cat->schemata_num; s++)
    {
      char *tmp= g_strdup_printf("%s", cat->schemata[s].schema_name);
      MYX_GRT_VALUE *value= myx_grt_value_from_string(tmp);
      myx_grt_list_item_add(result, value);
      myx_grt_value_release(value);
      g_free(tmp);
    }
  }

  myx_free_catalogs(catalogs);

  tmp= myx_grt_dict_create(NULL,
    "@status", MYX_STRING_VALUE, "ok",
    "schemas", 0, result, 
    NULL);
  myx_grt_value_release(result);

  return tmp;
}


static MYX_GRT_VALUE *make_dict_from_table(MYX_SCHEMA_TABLE *table)
{
  MYX_GRT_VALUE *dict, *clist;
  unsigned int i;
  
  clist= myx_grt_list_new(MYX_DICT_VALUE, "db.mysql.Column");
  for (i= 0; i < table->columns_num; i++)
  {
    MYX_SCHEMA_TABLE_COLUMN *column= table->columns+i;
    MYX_GRT_VALUE *col;

    col= myx_grt_dict_create("db.mysql.Column",
      "name", MYX_STRING_VALUE, column->column_name,
      "datatype", MYX_STRING_VALUE, column->column_type,
      "default", MYX_STRING_VALUE, column->default_value,
      "extra", MYX_STRING_VALUE, column->extra,
      "idprimarykey", MYX_INT_VALUE, column->primary_key,
      "notnull", MYX_INT_VALUE, column->not_null, 
      NULL);
    myx_grt_list_item_add(clist, col);
    myx_grt_value_release(col);
  }

  dict= myx_grt_dict_create("db.mysql.Table",
    "name", MYX_STRING_VALUE, table->table_name,
    "mytabletype", MYX_INT_VALUE, table->table_type,
    "columns", 0, clist,
    NULL);
  
  myx_grt_value_release(clist);
  
  return dict;
}


static MYX_GRT_VALUE *fetch_tables(MYSQL *mysql, const char *catalog, const char *schema)
{
  MYX_SCHEMA_TABLES *tables;
  MYX_GRT_VALUE *table_list;
  unsigned int i;

  tables= myx_get_schema_tables(mysql, catalog, schema);
  if (!tables)
    return NULL;

  table_list= myx_grt_list_new(MYX_DICT_VALUE, "db.mysql.Table");
  for (i= 0; i < tables->schema_tables_num; i++)
  {
    MYX_GRT_VALUE *value= make_dict_from_table(tables->schema_tables+i);
    myx_grt_list_item_add(table_list, value);
    myx_grt_value_release(value);
  }

  return table_list;
}


static MYX_GRT_VALUE *fetch_views(MYSQL *mysql, const char *catalog, const char *schema)
{
  MYX_GRT_VALUE *view_list= myx_grt_list_new(MYX_DICT_VALUE, NULL);
  MYX_GRT_VALUE *value;

  value= myx_grt_dict_create(NULL,
    "name", MYX_STRING_VALUE, "testview",
    "sql", MYX_STRING_VALUE, "CREATE VIEW testview AS SELECT * FROM test",
    "updateable", MYX_INT_VALUE, 0,
    NULL);
  myx_grt_list_item_add(view_list, value);
  myx_grt_value_release(value);

  return view_list;
}


MYX_GRT_VALUE *get_schema_assets(MYX_GRT_VALUE *param, void *data)
{
  MYX_GRT_VALUE *result= NULL, *error;
  const char *catalog_name, *schema_name;
  MYSQL *mysql;
  MYX_GRT_VALUE *tables;
  MYX_GRT_VALUE *views;
  
  error= NULL;
  if (!param)
    error= make_error_dict("bad parameter", "missing parameter");
  else
  {
    catalog_name= myx_grt_dict_item_get_as_string(param, "catalog");
    if (!catalog_name)
      error= make_error_dict("bad parameter", "missing catalog name");
    else
    {
      schema_name= myx_grt_dict_item_get_as_string(param, "schema");
      if (!schema_name)
        error= make_error_dict("bad parameter", "missing schema name");
    }
  }
  if (error)
  {
    result= myx_grt_dict_create(NULL,
      "@status", MYX_STRING_VALUE, "error",
      "@error", 0, error,
      NULL);
    myx_grt_value_release(error);
    return result;
  }

  mysql= connect_mysql(param, &result);
  if (!mysql)
  {
    MYX_GRT_VALUE *tmp= result;

    result= myx_grt_dict_create(NULL,
      "@status", MYX_STRING_VALUE, "error",
      "@error", 0, tmp,
      NULL);
    myx_grt_value_release(tmp);

    return result;
  }

  tables= fetch_tables(mysql, catalog_name, schema_name);

  views= fetch_views(mysql, catalog_name, schema_name);

  myx_mysql_close(mysql);
  
  result= myx_grt_dict_create(NULL,
    "@status", MYX_STRING_VALUE, "ok",
    "tables", 0, tables,
    "views", 0, views,
    NULL);
  myx_grt_value_release(tables);
  myx_grt_value_release(views);
  
  return result;
}
