/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_gl_helper.cpp
 * @brief Helper functions for creating OpenGL data and structures out of XML data.
 * 
 */

#ifdef _WINDOWS
  #include <windows.h>
#else
#endif // ifdef _WINDOWS

#include <gl/gl.h>
#include <gl/glu.h>
#include <xmlmemory.h>
#include <parser.h>
#include <vector>
#include <hash_map>

using namespace stdext;

#include "myx_gc_gl_helper.h"
#include "myx_gc_utilities.h"
#include "myx_gc_font_manager.h"
#include "myx_gc_texture.h"
#include "myx_gc_const.h"

//----------------------------------------------------------------------------------------------------------------------

char* Primitives[] = 
{
  "rect",       // Filled rectangle with border.
  "line",       // A simple straight line.
  "polyline",   // A sequence of straight lines (interior is not filled).
  "polygon",    // A sequence of straight lines filled with a color.
  "circle",     // A filled circle with a border.
  "text",       // Text.
  "g"           // A group of other primitives, including other groups.
};

#define PRIMITIVE_COUNT (sizeof(Primitives) / sizeof(Primitives[0]))

//----------------------------------------------------------------------------------------------------------------------

// Convert a primitive name to a number.
int FindPrimitive(xmlChar* Name)
{
  for (int I = 0; I < PRIMITIVE_COUNT; I ++)
    if (xmlStrcmp(Name, (xmlChar *) Primitives[I]) == 0)
      return I;

  return -1;
}

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve a float attribute.
bool GetFloatAttribute(xmlNodePtr Element, const char* Name, double& Value)
{
  xmlChar* Attribute = xmlGetProp(Element, (xmlChar*) Name);
  if (Attribute == NULL)
    return false;
  Value = atof((const char*) Attribute);
  xmlFree(Attribute);
  return true;
}

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve an integer attribute.
bool GetIntAttribute(xmlNodePtr Element, const char* Name, int& Value)
{
  xmlChar* Attribute = xmlGetProp(Element, (xmlChar*) Name);
  if (Attribute == NULL)
    return false;
  Value = atoi((const char*) Attribute);
  xmlFree(Attribute);
  return true;
}

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve an integer attribute. If it cannot be found a default value will be used instead.
int GetIntAttributeDef(xmlNodePtr Element, const char* Name, int Default)
{
  int Result;
  if (!GetIntAttribute(Element, Name, Result))
    Result = Default;

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve a string attribute.
// If the attribute could be found then true is returned and Value is set to the value of the attribute.
// Otherwise false is returned and Value is not touched.
bool GetStringAttribute(xmlNodePtr Element, const char* Name, string& Value)
{
  xmlChar* Attribute = xmlGetProp(Element, (xmlChar*) Name);
  if (Attribute == NULL)
    return false;
  else
  {
    Value.clear();
    Value.append((char*) Attribute);
    return true;
  }
}

//----------------------------------------------------------------------------------------------------------------------

// Helper method to retrieve a string attribute. If the attribute is empty or cannot be found then a default value is returned.
string GetStringAttributeDef(xmlNodePtr Element, const char* Name, const string Default)
{
  string Result;
  xmlChar* Value = xmlGetProp(Element, (xmlChar*) Name);
  if ((Value == NULL) || (*Value == '\0'))
    Result += Default;
  else
    Result += (char*) Value;

  if (Value != NULL)
    xmlFree(Value);

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

// Converts the two hex digits given by Upper and Lower to an unsigned byte.
unsigned char HexToByte(char Upper, char Lower)
{
  Upper -= '0';
  if (Upper > 9)
    Upper -= 7;
  Lower -= '0';
  if (Lower > 9)
    Lower -= 7;
  return Upper * 16 + Lower;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Searchs the predifined colors and tries to find one with the given name.
 *
 * @param Name The name of the color to find.
 * @param Color [out] The color data if it could be found.
 * @returns true if the color could be found, otherwise false;
 */
bool ColorByName(string Name, GLubyte* Color)
{
  static TColorMap PredefinedColors;

  // Fill the color map if not already done.
  if (PredefinedColors.size() == 0)
  {
    for (int I = 0; I < COLOR_COUNT; I++)
      PredefinedColors[Colors[I].Name] = Colors[I].Color;

    for (int I = 0; I < SYS_COLOR_COUNT; I++)
    {
  #ifdef _WINDOWS
      COLORREF Reference = GetSysColor(SystemColors[I].Value);
      SystemColors[I].Color[0] = GetRValue(Reference);
      SystemColors[I].Color[1] = GetGValue(Reference);
      SystemColors[I].Color[2] = GetBValue(Reference);
      SystemColors[I].Color[3] = 1;
  #else
  #endif // #ifdef _WINDOWS
      PredefinedColors[SystemColors[I].Name] = SystemColors[I].Color;
    };
  };

  TColorMapIterator Iterator = PredefinedColors.find(Name);
  bool Result = Iterator != PredefinedColors.end();
  if (Result)
  {
    Color[0] = Iterator->second[0];
    Color[1] = Iterator->second[1];
    Color[2] = Iterator->second[2];
    Color[3] = Iterator->second[3];
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Reads attribute Name from Element and tries to treat the string as a color.
 * The allowed syntax for colors is (as given by the SVG specification) either an HTML like
 * value (e.g. #FFFFFF, #FFF) or a function like form (e.g. rgb(100, 255, 255), rgb(10%, 100%, 0%)).
 * In order to support translucent figures the function tries to find an alpha value by looking
 * at the opacity attribute (if given), which is (according to SVG 1.1) supported
 * for "fill" and "stroke". The opacity value is clamped to [0..1] and then converted to byte range.
 *
 * @param Element The XML element to parse.
 * @param Name The name of the color attribute.
 * @param OpacityName The name of the attribute that carries the alpha value (if any).
 * @param Color [out] The converted color.
 * @return true if conversion went ok, otherwise false.
 */

bool ConvertColor(xmlNodePtr Element, const char* Name, const char* OpacityName, GLubyte* Color)
{
  xmlChar* Attribute = xmlGetProp(Element, (xmlChar*) Name);
  if (Attribute == NULL)
    return false;
  
  Color[0] = 0;
  Color[1] = 0; 
  Color[2] = 0; 
  Color[3] = 255; // Make this fully opaque by default.
  xmlChar* Head = Attribute;
  xmlChar* Tail;
  bool FoundColor = false;

  // Start by skipping leading white spaces. We have only simple ASCII compatible strings here,
  // so we don't need to care for conversion from UTF-8.
  while ((*Head != '\0') && ((*Head == ' ') || (*Head == '\t')))
    Head++;

  if (*Head != '\0')
  {
    if (*Head == '#')
    {
      // HTML color.
      Head++;
      Tail = Head;
      while ((*Tail >= '0' && *Tail <= '9') || (*Tail >= 'a' && *Tail <= 'f') || (*Tail >= 'A' && *Tail <= 'F'))
        Tail++;
      switch (Tail - Head)
      {
        // Support only the two defined cases. Other values are simply ignored.
        case 3:
          {
            Color[0] = HexToByte(*Head, *Head);
            Head++;
            Color[1] = HexToByte(*Head, *Head);
            Head++;
            Color[2] = HexToByte(*Head, *Head);
            FoundColor = true;
            break;
          };
        case 6:
          {
            Tail = Head + 1;
            Color[0] = HexToByte(*Head, *Tail);
            Head += 2; Tail += 2;
            Color[1] = HexToByte(*Head, *Tail);
            Head += 2; Tail += 2;
            Color[2] = HexToByte(*Head, *Tail);
            FoundColor = true;
            break;
          };
      }
    }
    else
      if (xmlStrncasecmp(Head, (xmlChar*) "rgb(", 4)== 0)
      {
        // Found a function call like specification. Split the entries and look if they are absolute
        // or percentage values.
        Head += 4;
        int Index = 0;
        unsigned char Value;
        while (Index < 3)
        {
          Value = 0;
          
          // Skip leading white spaces.
          while ((*Head != '\0') && ((*Head == ' ') || (*Head == '\t')))
            Head++;
          while ((*Head >= '0') && (*Head <= '9'))
            Value = Value * 10 + *Head++ - '0';

          if (Value < 0)
            Value = 0;
          if (*Head == '%')
          {
            if (Value > 100)
              Value = 100;
            Value = (Value * 255) / 100;
            Head++;
          }
          else
          {
            if (Value > 255)
              Value = 255;
          };

          Color[Index++] = Value;

          // Skip trailing white spaces.
          while ((*Head != '\0') && ((*Head == ' ') || (*Head == '\t')))
            Head++;

          // If there is no comma or closing parenthesis then there is something wrong.
          if (*Head == ')') 
            break;
          if (*Head != ',')
            return false;

          Head++;
        };
        FoundColor = true;
      }
      else
      {
        // Last chance are color names. Try to find the text in the color constants.
        FoundColor = ColorByName((char*) Head, Color);
      }
  }

  xmlFree(Attribute);

  // Look if there is an opacitiy attribute.
  if (OpacityName != NULL)
  {
    double Value;
    if (GetFloatAttribute(Element, OpacityName, Value))
    {
      if (Value < 0)
        Value = 0;
      if (Value > 1)
        Value = 1;
      Color[3] = GLubyte(Value * 255); // Round the float value to a byte.
    };
  };

  return FoundColor;
}

//----------------------------------------------------------------------------------------------------------------------

typedef struct tagVertex
{
  double X;
  double Y;
} TVertex;

typedef std::vector<TVertex> CVertexVector;

// Scans the given string for coordinate values and fills the Points vector with them.
void ParsePoints(xmlChar* Raw, CVertexVector* Points)
{
  Points->clear();

  char Separators[] = " ,\t\n";
  char* Token = strtok((char*) Raw, Separators);

  while (Token != NULL)
  {
    // Invalid input results in a value of 0, so we don't need to take special
    // care for such cases.
    TVertex V;
    V.X = atof(Token);
    Token = strtok(NULL, Separators);
    if (Token != NULL)
    {
      V.Y = atof(Token);
      Points->push_back(V);

      Token = strtok(NULL, Separators);
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

// Converts the given string into a font weight value.
// Allowed values are: normal | bold | bolder | lighter | 100 | 200 | 300| 400 | 500 | 600 | 700 | 800 | 900 | inherit
int ConvertFontWeight(const string Weight)
{
  int Result = 0;
  if (Weight == "normal")
    Result = 400;
  else
    if (Weight == "bold")
      Result = 700;
    else
      if (Weight == "bolder")
        Result = 800;
      else
        if (Weight == "lighter")
          Result = 300;
        else
          if (Weight == "inherit")
            Result = 400;
          else
            Result = atoi(Weight.c_str());


  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Reads the content of the given node and interprets it as an SVG definition. This definition is then turned into OpenGL
 * commands. This function can recursively be called if groups are defined.
 *
 * @param Parent Points to the svg element that must be parsed, but can be NULL (in which case nothing is parsed).
 *               This element must either be svg:svg or svg:g.
 */
void ParseSVGElement(xmlNodePtr Parent)
{
  xmlNodePtr Child = Parent->children;
  while (Child != NULL)
  {
    if (Child->type == XML_ELEMENT_NODE)
    {
      int Index = FindPrimitive((xmlChar*) Child->name);
      CGCTexture* Texture = NULL;

      if (Index > -1)
      {
        xmlChar* TextureAttribute = xmlGetProp(Child, (xmlChar*) "texture");
        if (TextureAttribute != NULL)
        {
          Texture = TextureManager()->FindTexture(string((char*) TextureAttribute));
          xmlFree(TextureAttribute);
        };
      };

      switch (Index) 
      {
        case 0: // rect
          {
            double X, Y, Width, Height;

            // We are going to switch off antialiasing for polygons. Keep the last state and restore it later.
            glPushAttrib(GL_ENABLE_BIT);
            glDisable(GL_POLYGON_SMOOTH);

            if (!GetFloatAttribute(Child, "x", X))
              X = 0;
            if (!GetFloatAttribute(Child, "y", Y))
              Y = 0;
            if (!GetFloatAttribute(Child, "width", Width))
              Width = 1;
            if (!GetFloatAttribute(Child, "height", Height))
              Height = 1;

            if (Texture != NULL)
              Texture->ActivateTexture();

            GLubyte Color[4];

            if (ConvertColor(Child, "fill", "fill-opacity", Color))
              glColor4ubv(Color);
            glBegin(GL_POLYGON);
              glNormal3d(X, Y, 1);
              glTexCoord2d(0, Height);
              glVertex2d(X, Y);

              glNormal3d(X, Y + Height, 1);
              glTexCoord2d(0, 0);
              glVertex2d(X, Y + Height);

              glNormal3d(X + Height, Y + Height, 1);
              glTexCoord2d(Width, 0);
              glVertex2d(X + Width, Y + Height);

              glNormal3d(X + Height, Y, 1);
              glTexCoord2d(Width, Height);
              glVertex2d(X + Width, Y);
            glEnd();
            if (Texture != NULL)
              Texture->DeactivateTexture();
            
            if (ConvertColor(Child, "stroke", "stroke-opacity", Color))
            {
              // If a stroke color was given then render the edges of the figure in that color.
              glColor4ubv(Color);

              // Enable line mode for polygons.
              glPolygonOffset(-1, -1);
              glEnable(GL_POLYGON_OFFSET_LINE);
              glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
              glBegin(GL_POLYGON);
                glVertex2d(X, Y);
                glVertex2d(X, Y + Height);
                glVertex2d(X + Width, Y + Height);
                glVertex2d(X + Width, Y);
              glEnd();

              // Restore fill settings.
              glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
              glDisable(GL_POLYGON_OFFSET_LINE);
            };
            
            // Finally restore polygon smooth mode.
            glPopAttrib();

            break;
          };
        case 1: // line
          {
            double X1, Y1, X2, Y2;
            
            if (!GetFloatAttribute(Child, "x1", X1))
              X1 = 0;
            if (!GetFloatAttribute(Child, "y1", Y1))
              Y1 = 0;
            if (!GetFloatAttribute(Child, "x2", X2))
              X2 = 1;
            if (!GetFloatAttribute(Child, "y2", Y2))
              Y2 = 0;

            if (Texture != NULL)
              Texture->ActivateTexture();

            GLubyte Color[4];
            if (ConvertColor(Child, "stroke", "stroke-opacity", Color))
              glColor4ubv(Color);

            glBegin(GL_LINES);
              glNormal3d(X1, Y1, 1);
              glTexCoord1d(0);
              glVertex2d(X1, Y1);
              glNormal3d(X2, Y2, 1);
              glTexCoord1d(1);
              glVertex2d(X2, Y2);
            glEnd();

            if (Texture != NULL)
              Texture->DeactivateTexture();

            break;
          };
        case 2: // polyline
          {
            if (Texture != NULL)
              Texture->ActivateTexture();

            GLubyte Color[4];

            if (ConvertColor(Child, "stroke", "stroke-opacity", Color))
              glColor4ubv(Color);

            xmlChar* Raw = xmlGetProp(Child, (xmlChar*) "points");
            if (Raw != NULL)
            {
              CVertexVector Points;
              ParsePoints(Raw, &Points);

              // Find largest and smallest x and y coordinates for texture coordination creation.
              double MaxX = -1e6;
              double MaxY = -1e6;
              double MinX = 1e6;
              double MinY = 1e6;
              for (int I = 0; I < (int) Points.size(); I++)
              {
                if (Points[I].X < MinX)
                  MinX = Points[I].X;
                if (Points[I].Y < MinY)
                  MinY = Points[I].Y;
                if (Points[I].X > MaxX)
                  MaxX = Points[I].X;
                if (Points[I].Y > MaxY)
                  MaxY = Points[I].Y;
              };

              double dX = MaxX - MinX;
              double dY = MaxY - MinY;
              glBegin(GL_LINE_STRIP);
              for (int I = 0; I < (int) Points.size(); I++)
              {
                double X = Points[I].X;
                double Y = Points[I].Y;
                glNormal3d(X, Y, 1);
                glTexCoord2d(dX < EPSILON ? 1 : (X - MinX) / dX, dY < EPSILON ? 1 : (Y - MinY) / dY);
                glVertex2d(X, Y);
              }
              glEnd();
              xmlFree(Raw);
            };

            if (Texture != NULL)
              Texture->DeactivateTexture();

            break;
          };
        case 3: // polygon
          {
            GLubyte Color[4];

            xmlChar* Raw = xmlGetProp(Child, (xmlChar*) "points");
            if (Raw != NULL)
            {
              CVertexVector Points;
              ParsePoints(Raw, &Points);

              // Find largest and smallest x and y coordinates for texture coordination creation.
              double MaxX = -1e6;
              double MaxY = -1e6;
              double MinX = 1e6;
              double MinY = 1e6;
              for (int I = 0; I < (int) Points.size(); I++)
              {
                if (Points[I].X < MinX)
                  MinX = Points[I].X;
                if (Points[I].Y < MinY)
                  MinY = Points[I].Y;
                if (Points[I].X > MaxX)
                  MaxX = Points[I].X;
                if (Points[I].Y > MaxY)
                  MaxY = Points[I].Y;
              };

              double dX = MaxX - MinX;
              double dY = MaxY - MinY;

              if (Texture != NULL)
                Texture->ActivateTexture();

              if (ConvertColor(Child, "fill", "fill-opacity", Color))
                glColor4ubv(Color);
              
              glBegin(GL_POLYGON);
              for (int I = 0; I < (int) Points.size(); I++)
              {
                double X = Points[I].X;
                double Y = Points[I].Y;
                glNormal3d(X, Y, 1);
                glTexCoord2d(dX < EPSILON ? 1 : (X - MinX) / dX, dY < EPSILON ? 1 : (Y - MinY) / dY);
                glVertex2d(X, Y);
              };
              glEnd();
              
              if (Texture != NULL)
                Texture->DeactivateTexture();

              if (ConvertColor(Child, "stroke", "stroke-opacity", Color))
              {
                // If there is a stroke color given then draw the edges of the polygon in that color.
                glColor4ubv(Color);

                // Enable line mode and depth offsets for polygons.
                glPolygonOffset(-1, -1);
                glEnable(GL_POLYGON_OFFSET_LINE);
                glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
                glBegin(GL_POLYGON);
                for (int I = 0; I < (int) Points.size(); I++)
                  glVertex2d(Points[I].X, Points[I].Y);
                glEnd();

                glPolygonOffset(0, 0);
                glDisable(GL_POLYGON_OFFSET_LINE);
                glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
              };

              xmlFree(Raw);
            };

            break;
          };
        case 4: // circle
          {
            GLubyte Color[4];
            GLdouble InnerRadius, OuterRadius, StartAngle, SweepAngle;
            GLint Slices, Loops;

            GLUquadricObj *Quadric = gluNewQuadric();
            if (!GetFloatAttribute(Child, "inner-radius", InnerRadius))
              InnerRadius = 0;
            if (!GetFloatAttribute(Child, "outer-radius", OuterRadius))
              OuterRadius = 1;
            if (!GetFloatAttribute(Child, "start-angle", StartAngle))
              StartAngle = 0;
            if (!GetFloatAttribute(Child, "sweep-angle", SweepAngle))
              SweepAngle = 360;
            if (!GetIntAttribute(Child, "slices", Slices))
              Slices = 10;
            if (!GetIntAttribute(Child, "rings", Loops))
              Loops = 3;

            gluQuadricOrientation(Quadric, GLU_INSIDE);
            if (ConvertColor(Child, "stroke", "stroke-opacity", Color))
            {
              glColor4ubv(Color);

              gluQuadricDrawStyle(Quadric, GLU_SILHOUETTE);
              gluPartialDisk(Quadric, InnerRadius, OuterRadius, Slices, Loops, StartAngle, SweepAngle);
            };

            if (Texture != NULL)
              Texture->ActivateTexture();

            if (ConvertColor(Child, "fill", "fill-opacity", Color))
              glColor4ubv(Color);

            gluQuadricDrawStyle(Quadric, GLU_FILL);
            gluQuadricNormals(Quadric, GLU_SMOOTH);
            gluQuadricTexture(Quadric, GL_TRUE);
            gluPartialDisk(Quadric, InnerRadius, OuterRadius, Slices, Loops, StartAngle, SweepAngle);
            gluDeleteQuadric(Quadric);

            if (Texture != NULL)
              Texture->DeactivateTexture();

            break;
          };
        case 5: // text
          {
            // The actual text is the (only) child node of this node.
            xmlNodePtr TextNode = Child->children;
            if (TextNode != NULL)
            {
              GLubyte Color[4];
              if (ConvertColor(Child, "fill", "fill-opacity", Color))
                glColor4ubv(Color);

              double X, Y;
              
              if (!GetFloatAttribute(Child, "x", X))
                X = 0;
              if (!GetFloatAttribute(Child, "y", Y))
                Y = 0;

              // Collect font information.
              string FontFamily = GetStringAttributeDef(Child, "font-family", DefaultFontFamily);
              string FontStyle = GetStringAttributeDef(Child, "font-style", DefaultFontStyle);
              string FontWeightString = GetStringAttributeDef(Child, "font-weight", DefaultFontWeight);
              int Weight = ConvertFontWeight(FontWeightString);
              string FontDecoration = GetStringAttributeDef(Child, "font-decoration", DefaultFontDecoration);

              GLuint Base = FontManager()->GetFontBase(FontFamily, FontStyle, Weight, FontDecoration);
              glListBase(Base);

              string Source((char*) TextNode->content);
              wstring Output = Utf8ToUtf16(Source);
              glPushMatrix();
              glPushAttrib(GL_POLYGON_BIT);

              if (Texture != NULL)
                Texture->ActivateTexture();

              glCallLists((GLsizei) Output.size(), GL_UNSIGNED_SHORT, Output.c_str());

              if (Texture != NULL)
                Texture->DeactivateTexture();

              glPopAttrib();
              glPopMatrix();
              glListBase(0);
            };

            break;
          };
        case 6: // A group.
          {
            ParseSVGElement(Child);
            break;
          };
      }
    };
    Child = Child->next;
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Reads a SVG definition returns a display list of it so it can be used in figures for rendering.
 *
 * @param XML Points to the svg:svg element that must be parsed, but can be NULL.
 * @return Returns the newly created display list for the template.
 */
GLuint ReadTemplateDefinition(xmlNodePtr XML)
{
  GLuint List = glGenLists(1); 
  glNewList(List, GL_COMPILE);
  ParseSVGElement(XML);
  glEndList();

  return List;
}

//----------------------------------------------------------------------------------------------------------------------

// Parses the given XML node for font information and creates a new entry in the font manager.
void ParseFontEntry(xmlNodePtr XML)
{
  // Collect font information.
  string FontFamily = GetStringAttributeDef(XML, "font-family", DefaultFontFamily);
  string FontStyle  = GetStringAttributeDef(XML, "font-style", DefaultFontStyle);
  string FontWeightString = GetStringAttributeDef(XML, "font-weight", DefaultFontWeight);
  int Weight = ConvertFontWeight(FontWeightString);
  string FontDecoration = GetStringAttributeDef(XML, "font-decoration", DefaultFontDecoration);

  FontManager()->CreateFontEntry(FontFamily, FontStyle, Weight, FontDecoration);
}

//----------------------------------------------------------------------------------------------------------------------

// Parses the given XML node for texture information and creates a new entry in the texture manager.
void ParseTextureEntry(xmlNodePtr XML)
{
  // Collect font information.
  string ID;
  string FileName;
  if ((GetStringAttribute(XML, "id", ID)) && (GetStringAttribute(XML, "location", FileName)))
  {
    string WrapModeH  = GetStringAttributeDef(XML, "wrap-mode-horizontal", DefaultTextureWrapMode);
    string WrapModeV  = GetStringAttributeDef(XML, "wrap-mode-vertical", DefaultTextureWrapMode);
    string MinFilter = GetStringAttributeDef(XML, "min-filter", DefaultTextureMinFilter);
    string MaxFilter = GetStringAttributeDef(XML, "mag-filter", DefaultTextureMagFilter);
    int Dimensions = GetIntAttributeDef(XML, "dimensions", DefaultTextureDimensions);
    string Mode = GetStringAttributeDef(XML, "mode", DefaultTextureMode);

    TextureManager()->CreateTextureEntry(FileName, ID, WrapModeH, WrapModeV, MinFilter, MaxFilter, Dimensions, Mode);
  };
}

//----------------------------------------------------------------------------------------------------------------------

