/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"
#if HAVE_IBASE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "udm_common.h"
#include "udm_db.h"
#include "udm_db_int.h"
#include "udm_utils.h"
#include "udm_vars.h"
#include "udm_sqldbms.h"

#include "udm_xmalloc.h"
#ifdef WIN32
#include <process.h>
#endif

#include <ibase.h>

#define SQL_VARCHAR(len) struct {short vary_length; char vary_string[(len)+1];}
typedef struct
{
  short  len;
  char  str[1];
} UDM_IBASE_VARY;

typedef struct
{
  isc_db_handle DBH;        /* database handle    */
  ISC_STATUS    status[20]; /* status vector      */
  isc_tr_handle tr_handle;  /* transaction handle */ 
} UDM_IB;

static void UdmIBaseDisplayError(UDM_DB *db)
{
  char *s= db->errstr;
  UDM_IB *ib= (UDM_IB*) db->specific;
  ISC_STATUS *ibstatus= ib->status;
  
  while(isc_interprete(s ,&ibstatus))
  {
    strcat(s," ");
    s=db->errstr+strlen(db->errstr);
  }
}

static int UdmIBaseInitDB(UDM_DB *db)
{
  char dpb_buffer[256], *dpb, *e;
  const char *p;
  int dpb_length, len;
  char connect_string[256];
  const char* DBUser= UdmVarListFindStr(&db->Vars,"DBUser",NULL);
  const char* DBPass= UdmVarListFindStr(&db->Vars,"DBPass",NULL);
  const char* DBHost= UdmVarListFindStr(&db->Vars, "DBHost", "localhost");
  UDM_IB *ib= (UDM_IB*) malloc(sizeof(UDM_IB));
  bzero(ib, sizeof(*ib));
  db->specific= (void*) ib;
  
  dpb = dpb_buffer;
  *dpb++ = isc_dpb_version1;

  if (DBUser != NULL && (len = strlen(DBUser)))
  {
    *dpb++ = isc_dpb_user_name;
    *dpb++ = len;
    for (p = DBUser; *p;)
      *dpb++ = *p++;
  }
  if (DBPass != NULL && (len = strlen(DBPass)))
  {
    *dpb++ = isc_dpb_password;
    *dpb++ = len;
    for (p = DBPass; *p;)
      *dpb++ = *p++;
  }
  /*
  if (charset != NULL && (len = strlen(charset))) {
    *dpb++ = isc_dpb_lc_ctype;
    *dpb++ = strlen(charset);
    for (p = charset; *p;) {
      *dpb++ = *p++;
    }
  }
#ifdef isc_dpb_sql_role_name
  if (role != NULL && (len = strlen(role))) {
    *dpb++ = isc_dpb_sql_role_name;
    *dpb++ = strlen(role);
    for (p = role; *p;) {
      *dpb++ = *p++;
    }
  }
#endif
  */

  dpb_length = dpb - dpb_buffer;
  
  if(strcmp(DBHost,"localhost"))
    udm_snprintf(connect_string,sizeof(connect_string)-1,"%s:%s",DBHost, db->DBName);
  else
    udm_snprintf(connect_string,sizeof(connect_string)-1,"%s",db->DBName);
  
  /* Remove possible trailing slash */
  e= connect_string+strlen(connect_string);
  if (e>connect_string && e[-1]=='/')
    e[-1]='\0';
  
#ifdef DEBUG_SQL
  fprintf(stderr, "SQL Connect to: '%s'\n",connect_string);
#endif  
  if(isc_attach_database(ib->status, strlen(connect_string), connect_string,
                         &(ib->DBH), dpb_length, dpb_buffer))
  {
    db->errcode=1;
    return(1);
  }
  return(0);
}


static void UdmIBaseClose(UDM_DB*db)
{
  UDM_IB *ib= (UDM_IB*) db->specific;
  if(db->connected)
  {
    if (isc_detach_database(ib->status, &(ib->DBH)))
    {
      db->errcode=1;
    }
    UDM_FREE(db->specific);
  }
}


static int sql_ibase_query(UDM_DB *db, UDM_SQLRES *res, const char *query)
{
  ISC_STATUS  status[20]; /* To not override db->status */
  isc_stmt_handle  query_handle = NULL;
  char    query_info[] = { isc_info_sql_stmt_type };
  char    info_buffer[18];
  long    query_type;
  int     autocommit=1;
  char    shortdata[64];
  short   sqlind_array[128];
  int     rc= UDM_OK;
  UDM_IB  *ib;  

  if(!db->connected)
  {
    UdmIBaseInitDB(db);
    if(db->errcode)
    {
      UdmIBaseDisplayError(db);
      db->errcode=1;
      return UDM_ERROR;
    }
    else
    {
      db->connected=1;
    }
  }
  
  ib= (UDM_IB*) db->specific;
  
  if(!strcmp(query,"BEGIN"))
  {
    if(!ib->tr_handle)
    {
      if (isc_start_transaction(ib->status, &ib->tr_handle, 1,
                                &(ib->DBH), 0, NULL))
      {
        db->errcode=1;
        rc= UDM_ERROR;
      }
    }
    else
    {
      db->errcode=1;
      udm_snprintf(db->errstr,sizeof(db->errstr)-1,"Wrong call order: begin");
      rc= UDM_ERROR;
    }
    return rc;
  }
  
  if(!strcmp(query,"COMMIT"))
  {
    if(ib->tr_handle)
    {
      if (isc_commit_transaction(ib->status, &ib->tr_handle))
      {
        db->errcode=1;
        rc= UDM_ERROR;
      }
      ib->tr_handle=NULL;
    }
    else
    {
      db->errcode=1;
      udm_snprintf(db->errstr,sizeof(db->errstr)-1,"Wrong call order: commit");
      rc= UDM_ERROR;
    }
    return rc;
  }
  
  if(!ib->tr_handle)
  {
    if (isc_start_transaction(ib->status, &ib->tr_handle, 1,
                              &(ib->DBH), 0, NULL))
    {
      db->errcode=1;
      return UDM_ERROR;
    }
    autocommit=1;
  }
  else
  {
    autocommit=0;
  }
  
  if (isc_dsql_allocate_statement(ib->status, &(ib->DBH), &query_handle))
  {
    db->errcode=1; 
    return UDM_ERROR;
  }
  if (isc_dsql_prepare(ib->status, &ib->tr_handle, &query_handle,
                       0, query, 1, NULL))
  {
    db->errcode=1; 
    return UDM_ERROR;
  }
  if (!isc_dsql_sql_info(ib->status, &query_handle, sizeof(query_info),
                         query_info, sizeof(info_buffer), info_buffer))
  {
    short l;
    l = (short) isc_vax_integer((char ISC_FAR *) info_buffer + 1, 2);
    query_type = isc_vax_integer((char ISC_FAR *) info_buffer + 3, l);
  }
  
  /* Find out what kind of query is to be executed */
  if (query_type == isc_info_sql_stmt_select ||
      query_type == isc_info_sql_stmt_select_for_upd)
  {
    XSQLDA *osqlda=NULL;
    long fetch_stat;
    int i;
    
    /* Select, need to allocate output sqlda and and prepare it for use */
    
    osqlda = (XSQLDA *) UdmXmalloc(XSQLDA_LENGTH(0));
    osqlda->sqln = 0;
    osqlda->version = SQLDA_VERSION1;
    
    /* Fetch column information */
    if (isc_dsql_describe(ib->status, &query_handle, 1, osqlda))
    {
      UDM_FREE(osqlda);
      db->errcode=1;
      isc_rollback_transaction(status, &ib->tr_handle);
      return UDM_ERROR;
    }
    
    if (osqlda->sqld)
    {
      osqlda = (XSQLDA *) UdmXrealloc(osqlda, XSQLDA_LENGTH(osqlda->sqld));
      osqlda->sqln = osqlda->sqld;
      osqlda->version = SQLDA_VERSION1;
      if (isc_dsql_describe(ib->status, &query_handle, 1, osqlda))
      {
        UDM_FREE(osqlda);
        db->errcode=1;
        isc_rollback_transaction(status, &ib->tr_handle);
        return UDM_ERROR;
      }
    }
    
    if (!res)
    {
      db->errcode=1;
      udm_snprintf(db->errstr, sizeof(db->errstr)-1,
                   "ibase_query with empty 'res' returned rows");
      return UDM_ERROR;
    }
    
    res->nCols = osqlda->sqld;
    res->Fields= (UDM_SQLFIELD*)UdmXmalloc(res->nCols*sizeof(UDM_SQLFIELD));
    
    for (i = 0; i < osqlda->sqld; i++)
    {
      XSQLVAR *var=&osqlda->sqlvar[i];
      int coltype = osqlda->sqlvar[i].sqltype & ~1;
      
      osqlda->sqlvar[i].sqlind=&sqlind_array[i];
      osqlda->sqlvar[i].sqlind[0]=0;
      res->Fields[i].sqlname = (char*)UdmStrdup(osqlda->sqlvar[i].sqlname);
      res->Fields[i].sqllen  = osqlda->sqlvar[i].sqllen;
      
      
      switch(coltype)
      {
        case SQL_SHORT:
          var->sqldata = (char*)UdmMalloc(sizeof(short));
          break;
        case SQL_LONG:
          var->sqldata = (char*)UdmMalloc(sizeof(long));
          break;
        case SQL_FLOAT:
          var->sqldata = (char*)UdmMalloc(sizeof(float));
          break;
        case SQL_DOUBLE:
          var->sqldata = (char*)UdmMalloc(sizeof(double));
          break;
        case SQL_DATE:
        case SQL_BLOB:
        case SQL_ARRAY:
          var->sqldata = (char*)UdmMalloc(sizeof(ISC_QUAD));
          break;
        case SQL_TEXT:
          var->sqldata = (char*)UdmMalloc((size_t)(osqlda->sqlvar[i].sqllen));
          break;
        case SQL_VARYING:
          osqlda->sqlvar[i].sqldata = (char*)UdmMalloc((size_t)(osqlda->sqlvar[i].sqllen+sizeof(short)));
          break;
      }
    }
    
    if (isc_dsql_execute(ib->status, &ib->tr_handle, &query_handle, 1, NULL))
    {
      UDM_FREE(osqlda);
      db->errcode=1;
      isc_rollback_transaction(status, &ib->tr_handle);
      return UDM_ERROR;
    }
    
    while ((fetch_stat= isc_dsql_fetch(ib->status, &query_handle, 1, osqlda)) == 0)
    {
      res->Items= (UDM_PSTR *)UdmRealloc(res->Items,(res->nRows+1)*(res->nCols)*sizeof(UDM_PSTR));
      
      for(i=0;i<osqlda->sqld; i++)
      {
        UDM_IBASE_VARY *vary;
        XSQLVAR *var=osqlda->sqlvar+i;
        char *p=NULL;
        size_t len;
        
        if(*var->sqlind==-1) /* NULL data */
        {
          p= (char*)UdmStrdup("");
          len= 0;
        }
        else
        switch(var->sqltype & ~1)
        {
          case SQL_TEXT:
            p=(char*)UdmMalloc((size_t)(var->sqllen+1));
            strncpy(p,(char*)var->sqldata,(size_t)(var->sqllen));
            p[var->sqllen]='\0';
            len= var->sqllen;
            break;
          case SQL_VARYING:
            vary=(UDM_IBASE_VARY*)var->sqldata;
            p=(char*)UdmMalloc((size_t)(vary->len+1));
            strncpy(p,vary->str,(size_t)(vary->len));
            p[vary->len]='\0';
            len= vary->len;
            break;
          case SQL_LONG:
            len= sprintf(shortdata,"%ld",*(long*)(var->sqldata));
            p = (char*)UdmStrdup(shortdata);
            break;
          case SQL_SHORT:
            len= sprintf(shortdata,"%d",*(short*)(var->sqldata));
            p = (char*)UdmStrdup(shortdata);
            break;
          case SQL_FLOAT:
            len= sprintf(shortdata,"%f",*(float*)(var->sqldata));
            p = (char*)UdmStrdup(shortdata);
            break;
          case SQL_DOUBLE:
            len= sprintf(shortdata,"%f",*(double*)(var->sqldata));
            p = (char*)UdmStrdup(shortdata);
            break;
          default:
            len= sprintf(shortdata,"Unknown SQL type");
            p = (char*)UdmStrdup(shortdata);
            break; 
        }
        /*UdmRTrim(p," ");*/
        res->Items[res->nRows*res->nCols+i].val= p;
        res->Items[res->nRows*res->nCols+i].len= len;
      }
      res->nRows++;
      if (db->res_limit && res->nRows >= db->res_limit)
      {
        fetch_stat = 100L;
        break;
      }
    }
    
    db->res_limit= 0;
    
    /* Free fetch buffers */
    for (i = 0; i < osqlda->sqld; i++)
      UDM_FREE(osqlda->sqlvar[i].sqldata);

    UDM_FREE(osqlda);
    
    if (fetch_stat != 100L)
    {
      db->errcode=1; 
      isc_rollback_transaction(status, &ib->tr_handle);
      return UDM_ERROR;
    }
  }
  else
  {
    /* Not select */
    if (isc_dsql_execute(ib->status, &ib->tr_handle, &query_handle, 1, NULL))
    {
      db->errcode=1;
      isc_rollback_transaction(status, &ib->tr_handle);
      isc_dsql_free_statement(status, &query_handle, DSQL_drop);
      return UDM_ERROR;
    }
  }
  
  if (isc_dsql_free_statement(ib->status, &query_handle, DSQL_drop))
  {
    db->errcode=1; 
    isc_rollback_transaction(status, &ib->tr_handle);
    return UDM_ERROR;
  }
  
  if(autocommit)
  {
    if (isc_commit_transaction(ib->status, &ib->tr_handle))
    {
      db->errcode=1; 
      ib->tr_handle=NULL;
      return UDM_ERROR;
    }
    ib->tr_handle=NULL;
  }
  return rc;
}

static int UdmIBaseQuery(UDM_DB *db, UDM_SQLRES *res, const char *query)
{
  int rc;
  db->errcode= 0;
    
  if (res)
  {
    bzero((void*) res, sizeof(UDM_SQLRES));
    res->db= db;
  }

  if(UDM_OK != (rc= sql_ibase_query(db,res,query)))
  {
    UdmIBaseDisplayError(db);
    
    if(strstr(db->errstr,"uplicat") || strstr(db->errstr,"UNIQUE"))
    {
      db->errcode=0;
      rc= UDM_OK;
    }
    else
    {
      /*strcat(db->errstr," ");
      strcat(db->errstr,query);*/
    }
    return rc;
  }
  return rc;
}

static int UdmIBBegin(UDM_DB *db)
{
  int rc= UdmIBaseQuery(db,NULL,"BEGIN");
  db->commit_fl = 1;
  return rc;
}

static int UdmIBCommit(UDM_DB *db)
{
  int rc= UdmIBaseQuery(db,NULL,"COMMIT");
  db->commit_fl = 1;
  return rc;
}

UDM_SQLDB_HANDLER udm_sqldb_ibase_handler =
{
  NULL,
  UdmIBaseQuery,
  UdmIBaseClose,
  UdmIBBegin,
  UdmIBCommit,
  NULL,
  NULL,
  NULL,
  UdmSQLFetchRowSimple,
  UdmSQLStoreResultSimple,
  UdmSQLFreeResultSimple,
  UdmIBaseQuery
};

#endif
