/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: tentacle.c,v 1.30 2003/02/05 04:40:14 kos Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#include <errno.h>
#include <stdarg.h>
#include <signal.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "common.h"
#include "configfile.h"
#include "xmalloc.h"
#include "llog.h"
#include "tentacle.h"
#include "net.h"
#include "keyspace.h"
#include "generator.h"

void
usage(char *argv0, int code) {
  fprintf(stderr, "\
Tentacle version %s [%s] usage options\n\
\t-V Print version string\n\
\t-v verbose level (default: %s)\n\
\t-s server (default: %s)\n\
\t-p port (default: %s)\n\
\t-t cracking performance test\n\
\t-i local name (default: %s)\n\
\t-d conn retry delay (default: %s)\n\
\t-n niceness value (default: %s)\n\
\t-T configuration dump\n\
", 
	  TENTACLE_VERSION,
	  HOSTTYPE,
	  DEF_VERBOSE,
	  DEF_SERVER,
	  DEF_SERVERPORT,
	  "hostname",
	  DEF_CONNDELAY,
	  DEF_NICENESS);

  exit(code);
}
      
void
packet_log(int level, char *caption, char **argv, int argc) {
  int i;

  llog(level, "%s", caption);
  for(i=0; i<argc; i++)
    llog(level, "%s ", argv[i]);
  llog(level, "\n");
}

int
packet_send(net_conn_t *c, ...) {
  va_list vl;
  char *p;
  msg *packet;

  va_start(vl, c);
  packet = msg_new();
  while((p = va_arg(vl, char *)))
    msg_add(packet, p, 0);  
  va_end(vl);
  net_send(c, packet);
  msg_destroy(packet);
  return 0;
}

int 
process_payload(tentacle_t *me) {
  int slice, method_limit;
  int i, j;
  key_index_t s, f;
  char key[TEN_LINELEN];
  int relindex;
  int keylen;

  keyspace_init(s);
  keyspace_init(f);
  keyspace_fromstr(s, me->i1);
  keyspace_fromstr(f, me->i2);
  keyspace_sub(f, f, s);
  slice = keyspace_toint(f);
  keyspace_fromstr(f, me->i2);
  method_limit = method_siz(me->method);
  method_sethash(me->method, me->hash.p, me->hash.l);
  generator_set(me->generator, s);

  for(i=0; i<slice; i+=method_limit) {    
    for(j=i; j<i+MIN(method_limit, slice-i); j++) {
      generator_fetch(me->generator, key, TEN_LINELEN, &keylen);
      method_add(me->method, key, keylen);
    }
    if(method_crypt(me->method)) {
      method_getkey(me->method, me->key.p, TEN_LINELEN, &me->key.l, &relindex);
      keyspace_fromint(relindex, f);
      keyspace_add(s, f, s);
      keyspace_tostr(me->keyindex, TEN_LINELEN, s);
      llog(1, "Found the key:\n");
      llog_hexdump(1, me->key.p, me->key.l);
      llog(1, "relative index %d, index %s\n", relindex, me->keyindex);
      me->keyfound = 1;
      break;
    }
  }
  
  keyspace_destroy(s);
  keyspace_destroy(f);
  return 1;
}

int
process_reply(tentacle_t *me, msg **reply, char *ident) {
  
  if(!(*reply = net_recv(me->conn))) {
    llog(5, "client_connect: server sent bad packet\n");
    return 1;
  }
  
  if(!msg_nelems(*reply)) {
    llog(5, "client_connect: server sent empty packet\n");
    msg_destroy(*reply);
    return 2;
  }
  
  if(atoi(msg_get(*reply, 0))) {
    llog(5, "client_connect: error: %s\n", msg_get(*reply, 1));
    packet_send(me->conn, "quit", NULL);
    msg_destroy(*reply);
    net_conn_destroy(me->conn);
    return 3;
  }

  return 0;
}

int
client_reset_keystatus(tentacle_t *me) {
  me->keyfound = 0;
  bstring_zero(&me->key);
  memset(me->keyindex, '\0', TEN_LINELEN);	
  return 0;
}

int
client_reset_all(tentacle_t *me) {
  memset(me->i1, '\0', TEN_LINELEN);
  memset(me->i2, '\0', TEN_LINELEN);
  bstring_zero(&me->hash);
  bstring_zero(&me->key);
  return client_reset_keystatus(me);
}

int
client_connect(tentacle_t *me) {
  msg *reply;

  if(!(me->conn = net_connect(me->n)))
    return 1;

  if(process_reply(me, &reply, "client_connect"))
    return 2;
  
  if(strncmp(msg_get(reply, 2), PROTO_VERSION, strlen(PROTO_VERSION))) {
    llog(1, "client_connect: Server uses unsupported protocol version: %s, should be %s\n",
	 msg_get(reply, 2),
	 PROTO_VERSION);
    msg_destroy(reply);
    return 3;
  }

  msg_destroy(reply);
  return 0;
}

int
client_initial(tentacle_t *me) {
  msg *reply;
      
  packet_send(me->conn, "ident", NULL);
  if(process_reply(me, &reply, "client_initial"))
    return 1;
  llog(4, "Server ident: %s\n", msg_get(reply, 1));
  msg_destroy(reply);

  packet_send(me->conn, "motd", NULL);
  if(process_reply(me, &reply, "client_initial"))
    return 2;
  llog(4, "Server motd: %s\n", msg_get(reply, 1));
  msg_destroy(reply);
  
  packet_send(me->conn, "client", config_char_get(me->conf, "ident"), NULL);
  if(process_reply(me, &reply, "client_initial"))
    return 3;
  msg_destroy(reply);
  return 0;
}

int
client_getspace(tentacle_t *me) {
  msg *reply;

  packet_send(me->conn, "gimme", config_char_get(me->conf, "ident"), NULL);
  if(process_reply(me, &reply, "client_getspace"))
    return 1;

  if(msg_nelems(reply) < 6) {
    llog(5, "client_getspace: bad response from server\n");
    msg_destroy(reply);
    packet_send(me->conn, "quit", NULL);
    net_conn_destroy(me->conn);
    return 2;
  }

  strncpy(me->i1, msg_get(reply, 5), TEN_LINELEN);
  strncpy(me->i2, msg_get(reply, 6), TEN_LINELEN);
  memcpy(me->hash.p, msg_get(reply, 3), MIN(msg_getlen(reply, 3), BSTRING_MAXLEN));
  me->hash.l = MIN(msg_getlen(reply, 3), BSTRING_MAXLEN);

  llog(2, "New keyspace from server: %s %s\n", me->i1, me->i2);  
  if(strcmp(msg_get(reply, 1), me->generator_name) ||
     strcmp(msg_get(reply, 2), me->generator_params)) {
    
    llog(4, "Creating new generator %s(%s)\n", msg_get(reply, 1), msg_get(reply, 2));
    if(me->generator)
      generator_destroy(me->generator);

    if(!(me->generator = generator_init(msg_get(reply, 1), msg_get(reply, 2)))) {
      llog(2, "client_getspace: bad generator %s\n", msg_get(reply, 1));
      msg_destroy(reply);
      packet_send(me->conn, "cancel", config_char_get(me->conf, "ident"), me->i1, me->i2, NULL);
      packet_send(me->conn, "quit", NULL);
      net_conn_destroy(me->conn);
      return 3;
    }

    strncpy(me->generator_name, msg_get(reply, 1), TEN_LINELEN);
    strncpy(me->generator_params, msg_get(reply, 2), TEN_LINELEN);
  }
  
  if(strcmp(msg_get(reply, 4), me->method_name)) {
    llog(4, "Creating new method %s\n", msg_get(reply, 4));
    if(me->method)
      method_destroy(me->method);
    if(!(me->method = method_init(msg_get(reply, 4), ""))) {
      llog(2, "client_getspace: bad method %s\n", msg_get(reply, 4));
      msg_destroy(reply);
      packet_send(me->conn, "cancel", config_char_get(me->conf, "ident"), me->i1, me->i2, NULL);
      packet_send(me->conn, "quit", NULL);
      net_conn_destroy(me->conn);
      return 4;
    }
    strncpy(me->method_name, msg_get(reply, 4), TEN_LINELEN);
  }
  msg_destroy(reply);
  return 0;
}

int
client_disconnect(tentacle_t *me) {
  msg *reply;
  
  packet_send(me->conn, "quit", NULL);  
  if(process_reply(me, &reply, "client_disconnect"))
    return 1;
  msg_destroy(reply);
  net_conn_destroy(me->conn);
  me->conn = (net_conn_t *)NULL;
  return 0;
}

int
client_send_gotit(tentacle_t *me) {
  msg *packet;
  msg *reply;

  packet = msg_new();
  msg_add(packet, "gotit", 0);
  msg_add(packet, config_char_get(me->conf, "ident"), 0);
  msg_add(packet, me->i1, 0);
  msg_add(packet, me->i2, 0);
  msg_add(packet, me->key.p, me->key.l);
  msg_add(packet, me->keyindex, 0);
  net_send(me->conn, packet);
  msg_destroy(packet);

  if(process_reply(me, &reply, "client_send_gotit"))
    return 1;
  msg_destroy(reply);
  return 0;
}

int
client_send_zilch(tentacle_t *me) {
  msg *reply;

  packet_send(me->conn, "zilch", config_char_get(me->conf, "ident"), me->i1, me->i2, NULL);      

  if(process_reply(me, &reply, "client_send_zilch"))
    return 1;
  msg_destroy(reply);
  return 0;
}

int
client_send_cancel(tentacle_t *me) {
  msg *reply;

  packet_send(me->conn, "cancel", config_char_get(me->conf, "ident"), me->i1, me->i2, NULL);      

  if(process_reply(me, &reply, "client_send_cancel"))
    return 1;
  msg_destroy(reply);
  return 0;
}

int
play_client(tentacle_t *me) {
#ifdef DO_PROFILE
  int i=0;
#endif
  
  while(1) {
    if(client_connect(me)) {
      sleep(config_int_get(me->conf, "conndelay"));
      continue;
    }

    while(1) {
      client_reset_all(me);
      if(client_initial(me)) {
	sleep(config_int_get(me->conf, "conndelay"));
	break;
      }
    
      if(client_getspace(me)) {
	sleep(config_int_get(me->conf, "conndelay"));
	break;
      }

      if(client_disconnect(me))
	break;

      process_payload(me);
#ifdef DO_PROFILE
      if(i++ == 4)
	exit(0);
#endif

      if(client_connect(me)) {
	sleep(config_int_get(me->conf, "conndelay"));
	break;
      }

      if(client_initial(me)) {
	break;
      }

      if(me->keyfound) {
	if(client_send_gotit(me))
	  break;
	client_reset_keystatus(me);
      } else {
	if(client_send_zilch(me))
	  break;
      }    
    }
  }
  
  return 0;
}

int
run_test(char *op) {
  method_t *m;
  time_t now, then;
  int i;
  int limit;
  char *type;
  char *opts;

  if(!(type = strchr(op, ':'))) {
    fprintf(stderr, "testing argument is method:options\n");
    return 2;
  }
  *type = '\0';
  opts = type+1;
  type = op;

  if(!(m = method_init(type, opts))) {
    fprintf(stderr, "method_init(%s, %s): failed\n", type, opts);
    return 1;
  }

  fprintf(stderr, "method_init(%s, %s). Running test...\n", type, opts);
  limit = method_siz(m);
  method_sethash(m, "aaaaa", 5);
  now = time(NULL);
  for(i=0; i<limit; i++)
    method_add(m, "foo", 4);
  method_crypt(m);
  then = time(NULL);
  fprintf(stderr, "crypt() finished. %d crypts per second\n", limit/((int)(then-now)));
  method_destroy(m);
  return 0;
}

void
logging_init(config_t *c) {

  if(!strcmp(config_char_get(c, "logmethod"), "file"))
    llog_init(LLOG_FILE, config_char_get(c, "logfile"), "w");
  else if(!strcmp(config_char_get(c, "logmethod"), "syslog"))
    llog_init(LLOG_SYSLOG, "tentacle", LOG_PID, LOG_DAEMON, LOG_NOTICE);
  else
    llog_init(LLOG_STDERR);
  llog_level(config_int_get(c, "verbose"));
}

void
tentacle_init(tentacle_t *t) {

  t->conf = (config_t *)NULL;
  memset(t->i1, '\0', TEN_LINELEN);
  memset(t->i2, '\0', TEN_LINELEN);
  t->keyfound = 0;
  memset(t->keyindex, '\0', TEN_LINELEN);
  bstring_zero(&t->key);
  bstring_zero(&t->hash);
  memset(t->hashtype, '\0', TEN_LINELEN);
  t->method = (method_t *)NULL;
  memset(t->method_name, '\0', TEN_LINELEN);
  memset(t->method_params, '\0', TEN_LINELEN);
  t->n = (net_t *)NULL;
  t->conn = (net_conn_t *)NULL;
}

int
tentacle_destroy(tentacle_t *t) {
  if(t->method)
    method_destroy(t->method);
  if(t->generator)
    generator_destroy(t->generator);
  if(t->conn)
    net_conn_destroy(t->conn);
  free(t);
  return 0;
}

int
main(int argc, char **argv) {
  tentacle_t *t;
  char c;
  char configclass[TEN_LINELEN];
  char configfile[TEN_LINELEN];
  char temp[TEN_LINELEN];

  t = xcalloc(1, sizeof(tentacle_t));
  tentacle_init(t);

  /* basic defaults */
  strncpy(configclass, DEF_CONFIGCLASS, TEN_LINELEN);
  strncpy(configfile, DEF_CONFIGFILE, TEN_LINELEN);

#if 0
  /* Configfile command line override */
  while((c = getopt(argc, argv, "v:s:p:t:i:d:n:T")) != EOF) {
    switch(c) {
    case 'f':
      strncpy(configfile, optarg, TEN_LINELEN);
      break;      
    }
  }
#endif

  /* defaults */
  if(!(t->conf = config_init(configclass))) {
    fprintf(stderr, "config_init(%s): Failed\n", configclass);
    exit(12);
  }

  gethostname(temp, TEN_LINELEN);
  config_set(t->conf, "ident", temp);
  config_set(t->conf, "server", DEF_SERVER);
  config_set(t->conf, "serverport", DEF_SERVERPORT);
  config_set(t->conf, "configclass", configclass);
  config_set(t->conf, "verbose", DEF_VERBOSE);
  config_set(t->conf, "conndelay", DEF_CONNDELAY);
  config_set(t->conf, "niceness", DEF_NICENESS);
  config_set(t->conf, "logmethod", DEF_LOGMETHOD);
  sprintf(temp, "%u", getpid());
  config_set(t->conf, "pid", temp);

  /* config file */
  if(config_load(t->conf, configfile))
    fprintf(stderr, "config_load(%s): %s\n", configfile, config_perror(t->conf));

  /* command line options */
  while((c = getopt(argc, argv, "v:s:p:t:i:d:n:TV")) != EOF) {
    switch(c) {
    case 'v':
      config_set(t->conf, "verbose", optarg);
      break;
    case 's':
      config_set(t->conf, "server", optarg);
      break;
    case 'p':
      config_set(t->conf, "serverport", optarg);
      break;
    case 't':
      run_test(optarg);
      exit(0);
    case 'i':
      config_set(t->conf, "ident", optarg);
      break;
    case 'd':
      config_set(t->conf, "conndelay", optarg);
      break;
    case 'n':
      config_set(t->conf, "niceness", optarg);
      break;
    case 'T':
      config_dump(t->conf);
      exit(0);
    case 'V':
      printf("tentacle version %s\n", VERSION);
      exit(0);
    case '?':
      usage(argv[0], 1);
      break;
    }
  }

  logging_init(t->conf);
  if(!(t->n = net_init(NET_CLIENT, 
		       config_char_get(t->conf, "server"), 
		       config_int_get(t->conf, "serverport")))) {
    llog(1, "net_init(%s, %s) failed.\n", 
	 config_char_get(t->conf, "server"), 
	 config_char_get(t->conf, "serverport"));
    exit(1);
  }

  llog(1, "Tentacle %s starting up, loglevel %d, pid %d, nice %d\n", 
       TENTACLE_VERSION,
       config_int_get(t->conf, "verbose"),
       config_int_get(t->conf, "pid"),
       config_int_get(t->conf, "niceness"));

  nice(config_int_get(t->conf, "niceness"));  
  play_client(t);
  net_destroy(t->n);
  tentacle_destroy(t);
  llog_close();
  exit(0);
}


