/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: mecon.c,v 1.19 2003/02/05 04:40:14 kos Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "common.h"
#include "array.h"
#include "cli.h"
#include "configfile.h"
#include "mecon.h"
#include "net.h"
#include "llog.h"

char *
state_abbrev(char *state) {  
  if(!strcmp(state, "ready"))
    return "S";
  else if(!strcmp(state, "done"))
    return "D";
  else if(!strcmp(state, "crunching"))
    return "R";
  else if(!strcmp(state, "found"))
    return "F";
  else if(!strcmp(state, "disabled"))
    return "Z";
  else if(!strcmp(state, "deceased"))
    return "Z";
  else
    return "?";
}

int
packet_send(net_conn_t *c, ...) {
  va_list vl;
  char *p;
  msg *packet;

  va_start(vl, c);
  packet = msg_new();
  while((p = va_arg(vl, char *)))
    msg_add(packet, p, 0);  
  va_end(vl);
  net_send(c, packet);
  msg_destroy(packet);
  return 0;
}

int
packet_recv_msg(net_conn_t *c, msg *params) {
  msg *reply;
  int type;
  int i;

  msg_zero(params);
  while(1) {
    if(!(reply = net_recv(c))) {
      printf("server sent bad packet");
      return 1;
    }
    
    type = atoi(msg_get(reply, 0));
    if(type == P_ERR) {
      for(i=1; i<msg_nelems(reply); i++)
	printf("%s ", msg_get(reply, i));
      printf("\n");
      msg_destroy(reply);
      break;
    }
    
    if(type == P_OK) {
      msg_destroy(reply);
      break;
    }

    msg_add_str(params, msg_get(reply, 1), msg_get(reply, 2), msg_getlen(reply, 2));  
    msg_destroy(reply);
  }
  return 0;
}

int
packet_recv(net_conn_t *c, char *string, int len) {
  msg *reply;
  int type;
  int i;

  if(!(reply = net_recv(c))) {
    snprintf(string, len, "server sent bad packet: recv is NULL");
    return 1;
  }
  
  type = atoi(msg_get(reply, 0));
  if(type == P_ERR) {
    for(i=1; i<msg_nelems(reply); i++)
      printf("%s ", msg_get(reply, i));
    printf("\n");
    msg_destroy(reply);
    return 2;
  }

  snprintf(string, len, "%s", msg_get(reply, 1));
  return 0;
}

/* implementation */
     
int
cmd_close(mecon_t *m, int argc, char **argv) {
  
  if(!m->net) {
    fprintf(stderr, "Not connected.\n");
    return 0;
  }

  packet_send(m->conn, "quit", NULL);
  net_conn_destroy(m->conn);
  net_destroy(m->net);
  m->net = (net_t *)NULL;
  m->conn = (net_conn_t *)NULL;
  cli_set(m->cli, CLI_PROMPT, "mecon> ");
  return 0;
}

int
cmd_loglevel(mecon_t *m, int argc, char **argv) {
  
  if(argc==2)
    llog_level(atoi(argv[1]));
  else
    fprintf(stderr, "loglevel = %d\n", llog_getlevel());
  return 0;
}

int
cmd_suspend(mecon_t *m, int argc, char **argv) {
  char string[MECON_LINELEN];
  
  if(!m->net) {
    printf("Not connected.\n");
    return 0;
  }

  packet_send(m->conn, "pool", "disable", NULL);
  if(packet_recv(m->conn, string, MECON_LINELEN))
    printf("suspend: %s\n", string);
  return 0;
}

int
cmd_resume(mecon_t *m, int argc, char **argv) {
  char string[MECON_LINELEN];
  
  if(!m->net) {
    printf("Not connected.\n");
    return 0;
  }

  packet_send(m->conn, "pool", "enable", NULL);
  if(packet_recv(m->conn, string, MECON_LINELEN))
    printf("suspend: %s\n", string);
  return 0;
}

int
cmd_unset(mecon_t *m, int argc, char **argv) {
  config_unset(m->conf, argv[1]);
  return 0;
}

int
cmd_set(mecon_t *m, int argc, char **argv) {
  int i;
  config_elem_t *el;
  
  if(argc==1) {
    for(i=0; i<config_nelems(m->conf); i++) {
      el = config_getbynum(m->conf, i);
      fprintf(stdout, "%-20s %s\n", el->lhs, el->rhs);
    }
  } else if(argc==2) {
    for(i=0; i<config_nelems(m->conf); i++) {
      el = config_getbynum(m->conf, i);
      if(!strcmp(argv[1], el->lhs))
	fprintf(stdout, "%-20s %s\n", el->lhs, el->rhs);
    }
  } else {
    config_set(m->conf, argv[1], argv[2]);
  }
  return 0;
}

int
cmd_stats(mecon_t *m, int argc, char **argv) {
  char lhs[MECON_LINELEN];
  char rhs[MECON_LINELEN];
  msg *params;
  int i;
  int len;
  
  if(!m->net) {
    printf("Not connected.\n");
    return 0;
  }

  params = msg_new();

  packet_send(m->conn, "stats", "eta", NULL);
  packet_recv_msg(m->conn, params);
  for(i=0; i<msg_nelems(params); i++) {
    msg_get_index(params, i, lhs, MECON_LINELEN, rhs, MECON_LINELEN, &len);
    printf("%s: %s\n", lhs, rhs);
  }

  packet_send(m->conn, "stats", "percent", NULL);
  packet_recv_msg(m->conn, params);
  for(i=0; i<msg_nelems(params); i++) {
    msg_get_index(params, i, lhs, MECON_LINELEN, rhs, MECON_LINELEN, &len);
    printf("%s: %s\n", lhs, rhs);
  }

  packet_send(m->conn, "stats", "totalcps", NULL);
  packet_recv_msg(m->conn, params);
  for(i=0; i<msg_nelems(params); i++) {
    msg_get_index(params, i, lhs, MECON_LINELEN, rhs, MECON_LINELEN, &len);
    printf("%s: %s\n", lhs, rhs);
  }

  msg_destroy(params);  

  return 0;
}

int
cmd_parameter(mecon_t *m, int argc, char **argv) {
  char junk[MECON_LINELEN];
  char index[MECON_LINELEN];
  int i;
  int j;
  msg *p1, *p2;
  
  if(!m->net) {
    printf("Not connected.\n");
    return 0;
  }

  p1 = msg_new();
  p2 = msg_new();

  if(argc == 1) {
    packet_send(m->conn, "parameter", "list", NULL);
    packet_recv_msg(m->conn, p1);
    
    for(i=0; i<msg_nelems(p1); i++) {
      msg_get_index(p1, i, index, MECON_LINELEN, junk, MECON_LINELEN, &j);
      packet_send(m->conn, "parameter", "get", "name", index, NULL);
      packet_recv_msg(m->conn, p2);    
      printf("%-20s %s\n",
	   index,
	   msg_get_strp(p2, index));
    }
  } else if(argc == 2) {
    packet_send(m->conn, "parameter", "get", "name", argv[1], NULL);
    packet_recv_msg(m->conn, p2);    
    printf("%-20s %s\n",
	 index,
	 msg_get_strp(p2, argv[1])?(char *)msg_get_strp(p2, argv[1]):"unknown");
  } else if(argc == 3) {
    packet_send(m->conn, "parameter", "set", "name", argv[1], "value", argv[2], NULL);
    packet_recv_msg(m->conn, p2);
  }
  
  msg_destroy(p1);  
  msg_destroy(p2);
  return 0;
}

int
cmd_show(mecon_t *m, int argc, char **argv) {
  msg *p1, *p2;
  int i;
  int j;
  char index[MECON_LINELEN];
  char junk[MECON_LINELEN];
  char t1[MECON_LINELEN];
  char t2[MECON_LINELEN];
  char localname[MECON_LINELEN];

  if(!m->net) {
    fprintf(stderr, "Not connected.\n");
    return 0;
  }

  p1 = msg_new();
  p2 = msg_new();
  
  if(argc == 1) {
    packet_send(m->conn, "node", "list", NULL);
    packet_recv_msg(m->conn, p1);
    printf("%d nodes, ", msg_nelems(p1));
    msg_zero(p1);
    packet_send(m->conn, "pool", "info", NULL);
    packet_recv_msg(m->conn, p1);
    buildtime(msg_get_strp(p1, "start time"), t1, MECON_LINELEN);
    printf("current schedule %s, state %s, start time %s\n",
	 msg_get_strp(p1, "current schedule"),
	 msg_get_strp(p1, "state"),
	 t1);

  } else if(argc == 2 && !strncmp(argv[1], "s", 1)) {
    packet_send(m->conn, "schedule", "list", NULL);
    packet_recv_msg(m->conn, p1);
    printf("schedule        startime         endtime           S:I:E generator(parameters)\n");

    for(i=0; i<msg_nelems(p1); i++) {
      msg_get_index(p1, i, index, MECON_LINELEN, junk, MECON_LINELEN, &j);
      packet_send(m->conn, "schedule", "info", "name", index, NULL);
      packet_recv_msg(m->conn, p2);

      snprintf(localname, MECON_LINELEN, "[%3s] %s %-8s", 
	       msg_get_strp(p2, "name"),
	       state_abbrev(msg_get_strp(p2, "state")),
	       msg_get_strp(p2, "hash"));
      buildtime(msg_get_strp(p2, "time_start"), t1, MECON_LINELEN);
      buildtime(msg_get_strp(p2, "time_end"), t2, MECON_LINELEN);
     
      printf("%-10s%-17s%-17s %s:%s:%s %-10s(%s)\n", 
	   localname,
	   t1,
	   t2,
	   msg_get_strp(p2, "start"),
	   msg_get_strp(p2, "index"),
	   msg_get_strp(p2, "finish"),
	   msg_get_strp(p2, "generator"),
	   msg_get_strp(p2, "generator_params"));
    }

  } else if(argc == 2 && !strncmp(argv[1], "h", 1)) {
    packet_send(m->conn, "hash", "list", NULL);
    packet_recv_msg(m->conn, p1);
    printf("hash       state      method          hash\n");
    for(i=0; i<msg_nelems(p1); i++) {
      msg_get_index(p1, i, index, MECON_LINELEN, junk, MECON_LINELEN, &j);
      packet_send(m->conn, "hash", "info", "name", index, NULL);
      packet_recv_msg(m->conn, p2);    
      printf("%-10s %-10s %-15s %s\n",
	   msg_get_strp(p2, "name"),
	   msg_get_strp(p2, "state"),
	   msg_get_strp(p2, "method"),
	   msg_get_strp(p2, "hash"));
    }

  } else if(argc == 2 && !strncmp(argv[1], "n", 1)) {
    packet_send(m->conn, "node", "list", NULL);
    packet_recv_msg(m->conn, p1);
    printf("node                   cps       start time       total time   current slice\n");

    for(i=0; i<msg_nelems(p1); i++) {
      msg_get_index(p1, i, index, MECON_LINELEN, junk, MECON_LINELEN, &j);
      packet_send(m->conn, "node", "info", "name", index, NULL);
      packet_recv_msg(m->conn, p2);    
      buildtime(msg_get_strp(p2, "time_created"), t1, MECON_LINELEN);
      buildlapse(msg_get_strp(p2, "time_total"), t2, MECON_LINELEN);
      printf("%-20s %s %-8s %-17s %-12s %s:%s\n",
	   msg_get_strp(p2, "name"),
	   state_abbrev(msg_get_strp(p2, "state")),
	   msg_get_strp(p2, "cps"),
	   t1, t2,
	   msg_get_strp(p2, "start"),
	   msg_get_strp(p2, "finish"));
    }
  }
  
  msg_destroy(p1);
  msg_destroy(p2);
  return 0;
}
     
int
cmd_quote(mecon_t *m, int argc, char **argv) {
  int i;
  msg *p;
  int type;
  int done;
  
  if(!m->net) {
    fprintf(stderr, "Not connected.\n");
    return 0;
  }
  
  p = msg_new();
  for(i=1; i<argc; i++)
    msg_add(p, argv[i], strlen(argv[i]));
  net_send(m->conn, p);
  msg_destroy(p);
  
  done = 0;
  while(!done) {
    if(!(p = net_recv(m->conn))) {
      printf("server dropped the connection\n");
      cmd_close(m, 0, NULL);
      break;
    }
    
    type = atoi(msg_get(p, 0));
    if(type == P_OK || type == P_ERR)
      done = 1;
    for(i=0; i<msg_nelems(p); i++)
      printf("%s ", msg_get(p, i));
    printf("\n");
  }
  return 0;
}

int
cmd_open(mecon_t *m, int argc, char **argv) {
  char *host;
  char *port;
  char prompt[MECON_LINELEN];
  msg *reply;
  
  if(m->net) {
    fprintf(stderr, "Already connected.\n");
    return 0;
  }

  host = config_char_get(m->conf, "host");
  port = config_char_get(m->conf, "port");
  if(argc == 2 || argc == 3)
    host = argv[1];
  if(argc == 3)
    port = argv[2];

  if(!(m->net = net_init(NET_CLIENT, host, atoi(port)))) {
    printf("net_init(%s, %s) failed\n", host, port);
    return 0;
  }

  if(!(m->conn = net_connect(m->net))) {
    printf("net_connect(%s, %s) failed\n", host, port);
    net_destroy(m->net);    
    m->net = (net_t *)NULL;
    return 0;
  }
  
  if(!(reply = net_recv(m->conn))) {
    printf("server sent bad packet: reply was NULL\n");
    cmd_close(m, 0, NULL);
    return 0;
  }

  if(msg_nelems(reply) != 5) {
    printf("server sent bad packet: nelems = %d\n", msg_nelems(reply));
    cmd_close(m, 0, NULL);
    return 0;
  }

  if(msg_getlen(reply, 0) != 3 || atoi(msg_get(reply, 0)) != P_OK) {
    printf("server sent bad packet: Initial code is %s\n", msg_get(reply, 0));
    cmd_close(m, 0, NULL);
    return 0;
  }

  if(strncmp(PROTO_VERSION, msg_get(reply, 2), strlen(PROTO_VERSION)-1)) {
    printf("server sent bad packet: proto version is %s, I support %s\n", 
	 msg_get(reply, 2), PROTO_VERSION);
    cmd_close(m, 0, NULL);
    return 0;
  }

  if(msg_getlen(reply, 4) != NONCE_LEN) {
    printf("server sent bad packet: length of nonce is %d\n", msg_getlen(reply, 4));
    cmd_close(m, 0, NULL);
    return 0;
  }

  bstring_set_str(&m->nonce, msg_get(reply, 4), msg_getlen(reply, 4));
  snprintf(prompt, MECON_LINELEN, "mecon@%s> ", host);
  cli_set(m->cli, CLI_PROMPT, prompt);
  printf("%s\n", msg_get(reply, 3));
  return 0;
}
     
int
cmd_quit(mecon_t *m, int argc, char **argv) {
  
  if(m->net)
    cmd_close(m, 0, (char **)NULL);
  return 1;
}

int
cmd_disable(mecon_t *m, int argc, char **argv) {
  char string[MECON_LINELEN];

  if(!m->net) {
    printf("not connected.\n");
    return 0;
  }

  packet_send(m->conn, argv[1], "disable", "name", argv[2], NULL);
  packet_recv(m->conn, string, MECON_LINELEN);
  return 0;
}

int
cmd_enable(mecon_t *m, int argc, char **argv) {
  char string[MECON_LINELEN];

  if(!m->net) {
    printf("not connected.\n");
    return 0;
  }

  packet_send(m->conn, argv[1], "enable", "name", argv[2], NULL);
  packet_recv(m->conn, string, MECON_LINELEN);
  return 0;
}

int
cmd_delete(mecon_t *m, int argc, char **argv) {
  char string[MECON_LINELEN];

  if(!m->net) {
    printf("not connected.\n");
    return 0;
  }

  packet_send(m->conn, argv[1], "delete", "name", argv[2], NULL);
  packet_recv(m->conn, string, MECON_LINELEN);
  return 0;
}

int
cmd_reset(mecon_t *m, int argc, char **argv) {
  char string[MECON_LINELEN];

  if(!m->net) {
    printf("not connected.\n");
    return 0;
  }

  packet_send(m->conn, argv[1], "reset", "name", argv[2], NULL);
  packet_recv(m->conn, string, MECON_LINELEN);
  return 0;
}
     
int
cmd_admin(mecon_t *m, int argc, char **argv) {
  msg *p;
  bstring hash;

  if(!m->net) {
    printf("not connected.\n");
    return 0;
  }

  chresp_init(argv[1], &m->nonce, &hash);
  p = msg_new();
  msg_add(p, "admin", 0);
  msg_add(p, hash.p, hash.l);
  net_send(m->conn, p);
  msg_destroy(p);

  if(!(p = net_recv(m->conn))) {
    printf("server sent bad packet: recv gave NULL\n");
    cmd_close(m, 0, NULL);
    return 0;
  }

  if(atoi(msg_get(p, 0)) != P_OK)
    printf("server didnt like the password\n");
  msg_destroy(p);
  return 0;
}

int
cmd_add(mecon_t *m, int argc, char **argv) {
  msg *params;

  if(!m->net) {
    printf("not connected.\n");
    return 0;
  }
  
  if(!strcmp(argv[1], "hash")) {
    if(argc != 5) {
      printf("usage: add hash name method hash\n");
      return 0;
    }
    packet_send(m->conn, "hash", "add", "name", argv[2], "method", argv[3], "hash", argv[4], NULL);
    params = msg_new();
    packet_recv_msg(m->conn, params);
    msg_destroy(params);
  } else if(!strcmp(argv[1], "schedule")) {
    if(argc != 5) {
      printf("usage: add schedule hashname generator generator_params\n");
      return 0;
    }
    packet_send(m->conn, "schedule", "add", "hash", argv[2], "generator", argv[3], "generator_params", argv[4], NULL);
    params = msg_new();
    packet_recv_msg(m->conn, params);
    msg_destroy(params);
  } else {
    printf("usage: add { hash | schedule }\n");
  }
    
  return 0;
}

int
cmd_key(mecon_t *m, int argc, char **argv) {
  msg *p1;
  msg *p2;
  bstring key;
  char index[MECON_LINELEN];
  char junk[MECON_LINELEN];
  int i;
  int j;

  if(!m->net) {
    printf("not connected.\n");
    return 0;
  }

  p1 = msg_new();
  p2 = msg_new();

  if(argc == 1) {
    packet_send(m->conn, "hash", "list", NULL);
    packet_recv_msg(m->conn, p1);
    for(i=0; i<msg_nelems(p1); i++) {
      msg_get_index(p1, i, index, MECON_LINELEN, junk, MECON_LINELEN, &j);
      packet_send(m->conn, "hash", "info", "name", index, NULL);
      packet_recv_msg(m->conn, p2);    
      if(msg_get_strp(p2, "key")) {
	printf("hash %s:\n", msg_get_strp(p2, "name"));
	msg_get_str(p2, "key", key.p, BSTRING_MAXLEN, &key.l);
	llog_hexdump(1, key.p, key.l);
      } else {
	printf("hash %s: None\n", msg_get_strp(p2, "name"));
      }
    }
  } else {    
    packet_send(m->conn, "hash", "info", "name", argv[1], NULL);
    packet_recv_msg(m->conn, p2);    
    if(msg_get_strp(p2, "key")) {
      printf("hash %s:\n", msg_get_strp(p2, "name"));
      msg_get_str(p2, "key", key.p, BSTRING_MAXLEN, &key.l);
      llog_hexdump(1, key.p, key.l);
    } else {
      printf("hash %s: None\n", msg_get_strp(p2, "name"));
    }
  }

  msg_destroy(p1);
  msg_destroy(p2);
  return 0;
}

int
play_mecon(mecon_t *m) {

  llog_init(LLOG_STDERR);
  llog_level(config_int_get(m->conf, "verbose"));
  m->cli = cli_init((void *)m);
  m->conn = (net_conn_t *)NULL;
  m->net = (net_t *)NULL;

  cli_set(m->cli, CLI_PROMPT, "mecon> ");
  cli_set(m->cli, CLI_PARSESEP, " ");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_show, "show", 1, 2, "Produces information about hashes, schedules and nodes");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_set, "set", 1, 3, "Messes with the internal variable store");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_unset, "unset", 2, 2, "UnMesses with the internal variable store");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_loglevel, "loglevel", 1, 2, "Gets/Sets the loglevel");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_open, "open", 1, 3, "Opens a connection to a host");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_close, "close", 1, 1, "Closes the active connection");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_quit, "quit", 1, 1, "Quits this contraption");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_admin, "admin", 2, 2, "Attempts to switch to administration mode. Takes a password");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_quote, "quote", 0, 0, "Sends some stuff verbatim to the server");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_disable, "disable", 3, 3, "Disables hashes/schedules/nodes");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_enable, "enable", 3, 3, "Enables hashes/schedules/nodes");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_delete, "delete", 3, 3, "Deletes hashes/schedules/nodes");  
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_reset, "reset", 3, 3, "Resets hash index back to start");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_suspend, "suspend", 1, 1, "Suspends the pool, ie all clients");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_resume, "resume", 1, 1, "Resumes normal operation");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_stats, "stats", 1, 1, "Gives some linearly extrapolated stats");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_parameter, "parameter", 1, 3, "Gets/Sets hashpool parameters");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_add, "add", 2, 0, "Adds hashes/schedules");
  cli_register(m->cli, CLI_PARSE, (cmd_proc)cmd_key, "key", 1, 2, "Lists all keys found");

  while(cli_main(m->cli) != 1)
    ;

  if(m->conn)
    net_conn_destroy(m->conn);
  if(m->net)
    net_destroy(m->net);
  cli_destroy(m->cli);
  config_destroy(m->conf);  
  return 0;
}

void
usage(char *argv0, int code) {
  fprintf(stderr, "\
Medussa Remote Console usage options\n\
\t-V Print version string\n\
\t-v verbose level (default: %s)\n\
\t-T configuration dump\n\
\t-f configuration file\n\
\t-h host to connect to (default: %s)\n\
\t-p port to use (default: %s)\n\
",
	  MECON_DEF_VERBOSE,
	  MECON_DEF_HOST,
	  MECON_DEF_PORT);
  exit(code);
}

int
main(int argc, char **argv) {
  char c;
  char configclass[MECON_LINELEN];
  char configfile[MECON_LINELEN];
  mecon_t m;
  
  /* basic defaults */
  strncpy(configclass, MECON_CONFIGCLASS, MECON_LINELEN);
  strncpy(configfile, MECON_CONFIGFILE, MECON_LINELEN);

  /* defaults */
  if(!(m.conf = config_init(configclass))) {
    fprintf(stderr, "config_init(%s): %s\n", configclass, config_perror(m.conf));
    exit(2);
  }
  
  config_set(m.conf, "host", MECON_DEF_HOST);
  config_set(m.conf, "port", MECON_DEF_PORT);
  config_set(m.conf, "verbose", MECON_DEF_VERBOSE);
  
  /* config file options */
  config_load(m.conf, configfile);
  
  /* command line options */
  while((c = getopt(argc, argv, "f:h:p:v:TV")) != EOF) {
    switch(c) {
    case 'f':
      strncpy(configfile, optarg, MECON_LINELEN);
      break;
    case 'h':
      config_set(m.conf, "host", optarg);
      break;
    case 'p':
      config_set(m.conf, "port", optarg);
      break;
    case 'v':
      config_set(m.conf, "verbose", optarg);
      break;
    case 'T':
      config_dump(m.conf);
      exit(0);
    case 'V':
      printf("mecon version %s\n", VERSION);
      exit(0);
    case '?':
      usage(argv[0], 1);
      break;
    }
  }

  play_mecon(&m);
  exit(0);
}
