/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: hashpool.h,v 1.21 2003/02/05 04:40:14 kos Exp $
 *
 */

#ifndef _HASHPOOL_H
#define _HASHPOOL_H

#include <time.h>
#include <pthread.h>
#include "array.h"
#include "common.h"
#include "keyspace.h"

#define HASH_LINELEN 1024

#define DEF_KEYSLICE 70000
#define MAX_KEYSLICE 1000000
#define MIN_KEYSLICE  50000
#define DEF_TIMETOCRUNCH 20
#define DEF_TIMETODIE (5*DEF_TIMETOCRUNCH)

#define STATS_ETA 1
#define STATS_PERCENT 2
#define STATS_TOTALCPS 3

#define SCHED_CURRENT (-1)
#define SCHED_ALL (-2)

#define HASHPOOL_SCHEDULE 1
#define HASHPOOL_NODE 2
#define HASHPOOL_HASH 3
#define HASHPOOL_ITSELF 4

#define CMD_DISABLE 1
#define CMD_DELETE 2
#define CMD_ENABLE 3
#define CMD_ADD 4
#define CMD_GET_NPARAMS 5
#define CMD_GET_PARAM_NAME 6
#define CMD_GET_PARAM_VALUE 7
#define CMD_SET_PARAM_VALUE 8

#define ACTION_ADD 1
#define ACTION_DELETE 2
#define ACTION_DISABLE 3
#define ACTION_ENABLE 4
#define ACTION_GET 5
#define ACTION_SET 6
#define ACTION_ETA 7
#define ACTION_PERCENT 8
#define ACTION_LIST 9
#define ACTION_INFO 10
#define ACTION_TOTALCPS 11
#define ACTION_RESET 12

typedef enum { ready, done, crunching, found, disabled, deceased } state_t;

typedef struct node_t {
  char name[HASH_LINELEN];
  key_index_t start;
  key_index_t finish;
  int slice;
  state_t state;
  /* */
  int cracks_per_sec;
  time_t time_created;
  time_t time_start;
  time_t time_total;
} node_t;

typedef struct hash_t {
  state_t state;
  char name[HASH_LINELEN];
  char method[HASH_LINELEN];
  bstring hash;
  bstring key;
} hash_t;

typedef struct schedule_t {
  int serial;
  state_t state;
  hash_t *hash;
  char generator[HASH_LINELEN];
  char generator_params[HASH_LINELEN];
  char method[HASH_LINELEN];
  time_t time_start;
  time_t time_end;
  key_index_t start;
  key_index_t finish;
  key_index_t index;
} schedule_t;

typedef struct workload_t {
  char method[HASH_LINELEN];
  bstring hash;
  char generator[HASH_LINELEN];
  char generator_params[HASH_LINELEN];
  char start[HASH_LINELEN];
  char finish[HASH_LINELEN];
  int slice;
} workload_t;

typedef struct hashpool_t {
  array *nodes;
  array *hashes;
  array *schedules;
  int schedule_serial;
  schedule_t *cur_schedule;
  time_t time_start;
  pthread_mutex_t *mutex;
  msg *param;
  state_t state;  
  char errstring[HASH_LINELEN];
} hashpool_t;

workload_t *workload_init(void);
int workload_destroy(workload_t *);

hashpool_t *hashpool_init(void);
int hash_manage(hashpool_t *h, int action, msg *params);
int stats_manage(hashpool_t *h, int action, msg *s);
int node_manage(hashpool_t *h, int action, msg *s);
int schedule_manage(hashpool_t *h, int action, msg *s);
int hashpool_destroy(hashpool_t *h);
int hashpool_done(hashpool_t *h);
int hashpool_schedule(hashpool_t *h);
int hashpool_node_register(hashpool_t *h, char *name, workload_t *w);
int hashpool_node_update(hashpool_t *h, char *name, char *s, char *f, unsigned char *key, int len, unsigned char *keyindex);
int hashpool_node_exists(hashpool_t *h, char *name);
char *hashpool_stats(hashpool_t *h, int type, int schedserial, char *buf, int len);
int hashpool_manage(hashpool_t *h, char *object, char *action, msg *s);
char *hashpool_state_ntoa(state_t);
state_t hashpool_state_aton(char *);

char *hashpool_geterr(hashpool_t *h);

int hashpool_checkpoint(hashpool_t *h, char *fname);
int hashpool_restore(hashpool_t *h, char *fname);

#define hashpool_hash(x) ((x)->cur_schedule->hash->hash)
#define hashpool_hashtype(x) ((x)->cur_schedule->hash->method)
#define hashpool_generator(x) ((x)->cur_schedule->generator)
#define hashpool_genopts(x) ((x)->cur_schedule->generator_params)

#endif /* _HASHPOOL_H */
