/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */
/*
 * $Id: configfile.c,v 1.12 1999/12/20 19:46:35 kos Exp $
 *
 */

#include <string.h>
#include <strings.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>

#include "llog.h"
#include "array.h"
#include "configfile.h"
#include "xmalloc.h"
#include "common.h"

int
countem(char *line, char c) {
  char *p;
  int i;
  
  i = 0;
  while(1) {
    if(!(p = index(line, c)))
      return i;
    else
      line = p+1;
  }  
}

char *
isinit(char *line, char *set) {
  char *c;

  while(*line++) {
    for(c = set; *c; c++)
      if(*line == *c)
	return line;
  }
  return NULL;
}

int
config_seterror(config_t *c, char *fmt, ...) {
  va_list vl;
  
  va_start(vl, fmt);
  vsprintf(c->errstring, fmt, vl);
  va_end(vl);
  return 0;
}

char *
config_perror(config_t *c) {
  return c->errstring;
}

int
config_noerror(config_t *c) {
  return config_seterror(c, "");
}

config_t *
config_init(char *class) {
  config_t *c;

  if(!class)
    return (config_t *)NULL;

  c = xcalloc(1, sizeof(config_t));
  c->classes = array_init(sizeof(config_class_t));
  strncpy(c->configclass, class, CONFIG_LINELEN);
  strncpy(c->separators, CONFIG_SEPARATORS, CONFIG_LINELEN);
  config_seterror(c, "");
  return c;
}

int
config_destroy(config_t *c) {  
  config_class_t *cl;
  int i;

  for(i=0; (cl=array_get(c->classes, i)); i++)
    array_nuke(cl->elements);
  array_nuke(c->classes);
  xfree(c);
  return 0;
}

void
config_dump(config_t *c) {
  config_class_t *cl;
  config_elem_t *e;
  int i;
  int j;
  
  printf("configclass %s, separators \"%s\"\n", c->configclass, c->separators);  
  for(i=0; (cl=array_get(c->classes, i)); i++) {
    printf("class %s\n", cl->name);
    for(j=0; (e=array_get(cl->elements, j)); j++) {
      printf("  %-20s: %s\n", e->lhs, e->rhs);
    }
  }
}

  
int
config_class_set(config_t *c, char *class, char *what, char *towhat) {
  config_elem_t elem;
  config_elem_t *pelem;
  config_class_t *pcl;
  config_class_t cl;
  int i;

  config_noerror(c);

  pcl = (config_class_t *)NULL;
  for(i=0; (pcl=array_get(c->classes, i)); i++) {
    if(!strncmp(pcl->name, class, CONFIG_LINELEN))
      break;
  }
	
  if(!pcl) {
    strncpy(cl.name, class, CONFIG_LINELEN);
    cl.elements = array_init(sizeof(config_elem_t));
    array_add(c->classes, &cl);
    return config_class_set(c, class, what, towhat);
  }

  for(i=0; (pelem=array_get(pcl->elements, i)); i++) {
    if(!strncmp(pelem->lhs, what, CONFIG_LINELEN)) {
      strncpy(pelem->rhs, towhat, CONFIG_LINELEN);
      return 0;
    }    
  }

  strncpy(elem.lhs, what, CONFIG_LINELEN);
  strncpy(elem.rhs, towhat, CONFIG_LINELEN);
  array_add(pcl->elements, &elem);
  return 0;
}

int
config_set(config_t *c, char *what, char *towhat) {
  return config_class_set(c, DEFAULT_CLASS, what, towhat);
}

int
config_class_get(config_t *c, char *class, char *what, char *res, int reslen) {
  config_class_t *cl;
  config_elem_t *e;
  int i;

  config_noerror(c);
  cl = (config_class_t *)NULL;
  for(i=0; (cl=array_get(c->classes, i)); i++) {
    if(!strncmp(cl->name, class, CONFIG_LINELEN))
      break;
  }  
  if(!cl) {
    config_seterror(c, "no such class %s", class);
    return 1;
  }
  
  for(i=0; (e=array_get(cl->elements, i)); i++) {
    if(!strncmp(e->lhs, what, CONFIG_LINELEN)) {
      strncpy(res, e->rhs, MIN(reslen, CONFIG_LINELEN));
      return 0;
    }
  }
  config_seterror(c, "no such parameter %s:%s", class, what);
  return 2;
}

int
config_get(config_t *c, char *what, char *res, int reslen) {
  return config_class_get(c, DEFAULT_CLASS, what, res, reslen);
}

int
config_int_get(config_t *c, char *what) {
  char res[CONFIG_LINELEN];

  if(config_get(c, what, res, CONFIG_LINELEN))
    return -1;
  return atoi(res);  
}

char *
config_class_char_get(config_t *c, char *class, char *what) {
  config_class_t *cl;
  config_elem_t *e;
  int i;

  config_noerror(c);
  cl = (config_class_t *)NULL;
  for(i=0; (cl=array_get(c->classes, i)); i++) {
    if(!strncmp(cl->name, class, CONFIG_LINELEN))
      break;
  }
  if(!cl) {
    config_seterror(c, "no such class %s", class);
    return NULL;
  }
  
  for(i=0; (e=array_get(cl->elements, i)); i++) {
    if(!strncmp(e->lhs, what, CONFIG_LINELEN))
      return e->rhs;
  }
  config_seterror(c, "no such parameter %s:%s", class, what);
  return NULL;
}

char *
config_char_get(config_t *c, char *what) {
  return config_class_char_get(c, DEFAULT_CLASS, what);  
}

int
config_nelems(config_t *c) {
  return config_class_nelems(c, DEFAULT_CLASS);
}

config_elem_t *
config_getbynum(config_t *c, int which) {
  config_class_t *cl;
  int i;

  config_noerror(c);
  cl = (config_class_t *)NULL;
  for(i=0; (cl=array_get(c->classes, i)); i++) {
    if(!strncmp(cl->name, DEFAULT_CLASS, CONFIG_LINELEN))
      break;
  }  
  if(!cl) {
    config_seterror(c, "no such class %s", DEFAULT_CLASS);
    return NULL;
  }
  return array_get(cl->elements, which);
}

int
process_file(config_t *c, char *fname) {
  FILE *fp;
  char class[CONFIG_LINELEN];
  int state;
  char line[CONFIG_LINELEN];
  char lhs[CONFIG_LINELEN];
  char rhs[CONFIG_LINELEN];  
  char *p;
  
  if(!(fp = fopen(fname, "r")))
    return 1;

  state = 0;
  while(fgets(line, CONFIG_LINELEN, fp)) {
    zap_whitespace(line, CONFIG_LINELEN, ZAP_LEFT|ZAP_RIGHT);

    if(!strcmp(line, "}")) {
      state = 0;
      continue;
    }

    if(*line == '#')
      continue;

    if(!(p = isinit(line, c->separators)))
      continue;

    *p++ = '\0';
    strncpy(lhs, line, CONFIG_LINELEN);
    strncpy(rhs, p, CONFIG_LINELEN);    

    if(!strcmp(rhs, "{")) {
      state = 1;
      strncpy(class, lhs, CONFIG_LINELEN);
      continue;
    }

    switch(state) {
    case 0:
      config_set(c, lhs, rhs);
      break;
    case 1:
      config_class_set(c, class, lhs, rhs);
      break;
    }
  }
  fclose(fp);
  return 0;
}

int
config_load(config_t *c, char *fname) {
  char homedir[CONFIG_LINELEN];
  char filename[CONFIG_LINELEN];
  char cwd[CONFIG_LINELEN];
  char *phomedir;  

  if(!(phomedir = getenv("HOME")))
    homedir[0] = '\0';
  else
    strncpy(homedir, phomedir, CONFIG_LINELEN);
  getcwd(cwd, CONFIG_LINELEN);

  /* Class-wide config */
  snprintf(filename, CONFIG_LINELEN, "%s/%s", "/etc", c->configclass);
  process_file(c, filename);
  snprintf(filename, CONFIG_LINELEN, "%s/.%s", homedir, c->configclass);
  process_file(c, filename);
  snprintf(filename, CONFIG_LINELEN, "%s/.%s", cwd, c->configclass);
  process_file(c, filename);

  /* Application config */
  snprintf(filename, CONFIG_LINELEN, "%s/%s", "/etc", fname);
  process_file(c, filename);
  snprintf(filename, CONFIG_LINELEN, "%s/%s", homedir, fname);
  process_file(c, filename);
  snprintf(filename, CONFIG_LINELEN, "%s/.%s", homedir, fname);
  process_file(c, filename);
  snprintf(filename, CONFIG_LINELEN, "%s/.%s", cwd, fname);  
  process_file(c, filename);
  snprintf(filename, CONFIG_LINELEN, "%s", fname);
  process_file(c, filename);

  config_seterror(c, "");
  return 0;
}

int
config_save(config_t *c, char *fname) {
  config_seterror(c, "Not implemented");
  return 1;
}

int
config_nclasses(config_t *c) {
  return array_nelems(c->classes);
}

char *
config_class_name(config_t *c, int which) {
  config_class_t *cl;
  
  cl = array_get(c->classes, which);
  return cl->name;
}

int 
config_class_nelems(config_t *c, char *class) {
  config_class_t *cl;
  int i;

  config_noerror(c);
  for(i=0; (cl=array_get(c->classes, i)); i++) {
    if(!strncmp(cl->name, class, CONFIG_LINELEN))
      return array_nelems(cl->elements);
  }
  
  config_seterror(c, "no such class %s", class);
  return -1;
}

char *
config_class_elem_name(config_t *c, char *class, int which) {
  config_class_t *cl;
  config_elem_t *e;
  int i;
  
  config_noerror(c);
  for(i=0; (cl=array_get(c->classes, i)); i++) {
    if(!strncmp(cl->name, class, CONFIG_LINELEN)) {
      if(!(e=array_get(cl->elements, which))) {
	config_seterror(c, "no such element %d", which);
	return NULL;
      }
      return e->lhs;
    }
  }
  
  config_seterror(c, "no such class %s", class);
  return NULL;
}

int
config_class_unset(config_t *c, char *class, char *which) {
  config_class_t *cl;
  config_elem_t *e;
  int i;
  int j;
  
  config_noerror(c);
  for(i=0; (cl=array_get(c->classes, i)); i++) {
    if(!strncmp(cl->name, class, CONFIG_LINELEN)) {
      for(j=0; (e=array_get(cl->elements, j)); j++) {
	if(!strncmp(which, e->lhs, CONFIG_LINELEN))
	  array_delete(cl->elements, j);
      }
    }
  }
  
  return 0;  
}

int
config_unset(config_t *c, char *which) {
  return config_class_unset(c, DEFAULT_CLASS, which);
}

#ifdef DEBUG_CONFIG
int
main(int argc, char **argv) {
  config_t *c;
  int i;

  if(!(c = config_init("sample"))) {
    fprintf(stderr, "config_init: Failed\n");
    exit(1);
  }
  
  config_set(c, "ldapip", "127.0.0.1");
  config_set(c, "ldapip", "127.0.0.2");
  config_set(c, "ldapport", "389");
  config_set(c, "proxyserver", "www.foo.com");
  config_set(c, "proxyip", "1111");
  config_save(c, ".samplerc");
  
  for(i=0; i<config_nvalues(c, "ldapip"); i++)
    printf("value #%d of ldapip: %s\n", i, config_char_nget(c, "ldapip", i));
  config_destroy(c);
  
  c=config_init("sample");
  config_load(c, ".samplerc");  
}

#endif
  
