/*
 * Math3d - The 3D Computer Graphics Math Library
 * Copyright (C) 1996-2000 by J.E. Hoffmann <je-h@gmx.net>
 * All rights reserved.
 *
 * This program is  free  software;  you can redistribute it and/or modify it
 * under the terms of the  GNU Lesser General Public License  as published by 
 * the  Free Software Foundation;  either version 2.1 of the License,  or (at 
 * your option) any later version.
 *
 * This  program  is  distributed in  the  hope that it will  be useful,  but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or  FITNESS FOR A  PARTICULAR PURPOSE.  See the  GNU Lesser General Public  
 * License for more details.
 *
 * You should  have received  a copy of the GNU Lesser General Public License
 * along with  this program;  if not, write to the  Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: mrot.cpp,v 1.5 2000/10/09 12:17:54 jeh Exp $
 */
#define _MATH3D_EXPORT
#include <math3d/mrot.h>
#include <math3d/mquat.h>
#include <cmath>


/*!
 *
 */
Math3d::MRot::MRot()
  : d_axis(1,0,0)
{
  d_angle=0.0;
}


/*!
 *
 */
Math3d::MRot::MRot(const M3d& axis, double angle)
  : d_axis(axis)
{
  d_angle=angle;
}


/*!
 *
 */
Math3d::MRot::MRot(const MQuat& quat)
{
  set(quat);
}


/*!
 *
 */
Math3d::MRot::MRot(const MRot& rot)
  : d_axis(rot.d_axis)
{
  d_angle=rot.d_angle;
}


/*!
 *
 */
const Math3d::MRot& 
Math3d::MRot::operator=(const MRot& rot)
{
  d_axis=rot.d_axis;
  d_angle=rot.d_angle;
  return(*this);
}


/*!
 *
 */
void
Math3d::MRot::set(double x, double y, double z, double angle)
{
  d_axis[0]=x;
  d_axis[1]=y;
  d_axis[2]=z;
  d_angle=angle;
}


/*!
 *
 */
void
Math3d::MRot::set(const M3d& axis, double angle)
{
  d_axis=axis;
  d_angle=angle;
}


/*!
 *
 */
void
Math3d::MRot::set(const MQuat& quat)
{
  double omega=acos(quat[3]);
  d_angle=-2.0*omega;
  double s=sin(omega);
  if (fabs(s)>EPSILON) {
    d_axis[0]=quat[0]/s;
    d_axis[1]=quat[1]/s;
    d_axis[2]=quat[2]/s;
  }
}


/*!
 *
 */
void
Math3d::MRot::copy(const MRot& rot)
{
  d_axis=rot.d_axis;
  d_angle=rot.d_angle;
}


/*!
 *
 */
Math3d::MQuat
Math3d::MRot::quat() const
{
  return(MQuat(d_axis,d_angle));
}


/*!
 *
 */
bool
Math3d::MRot::operator==(const MRot& rot) const
{
  return(cmp(rot));
}


/*!
 *
 */
bool
Math3d::MRot::operator!=(const MRot& rot) const
{
  return(!cmp(rot));
}


/*!
 *
 */
bool
Math3d::MRot::cmp(const MRot& rot, double epsilon) const
{
  return(
    (fabs(d_axis[0]-rot.d_axis[0])<epsilon) &&
    (fabs(d_axis[1]-rot.d_axis[1])<epsilon) &&
    (fabs(d_axis[2]-rot.d_axis[2])<epsilon) &&
    (fabs(d_angle-rot.d_angle)<epsilon)
  );
}




