/*  cssed (c) Iago Rubio 2003, 2004 - A tiny CSS editor.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* 
	This parser is implemented using a subset of the libcroco (the parser)
	and using just it's SAC handler. Don't use it as a libcroco example as
	the libcroco used IS MODIFIED. Some functions doesn't exists or are private
	in the original libcroco. 
	
	Download the original libcroco from http://www.freespiders.org/
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#define PLAT_GTK 1

#include <stdio.h>
#include <gtk/gtk.h>

#ifdef WITH_TERMINAL
 #include <vte/vte.h>
#endif

#include "cssedwindow.h"
#include "document.h"
#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "utils.h"

#include <libcroco.h>
#include <string.h>

#define OUT_COLOR "navy"
#define ERR_COLOR "red"
#define SEL_COLOR "blue"
#define PRO_COLOR "maroon"
#define MED_COLOR "dark green"

typedef struct _CssedCssParserData {
	CssedDoc* doc;
	gchar* current_selector;
	gboolean in_media;
	gint err_count;
} CssedCssParserData;
	
/* *** UTILS *** */
CssedWindow *
get_window_fom_sac_handler_data (CRDocHandler * a_handler)
{
	CssedCssParserData* data;
	CssedDoc *document;
	CssedWindow *win;

	data = (CssedCssParserData*) a_handler->app_data;
	document = (CssedDoc *) data->doc;
	win = document_get_window ( document );

	return win;
}

void 
change_current_sac_handler_selector(CRDocHandler * a_handler,gchar* selector)
{
	CssedCssParserData* data;	
	data = (CssedCssParserData*) a_handler->app_data;
	
	if( data->current_selector )
		g_free(data->current_selector);
	
	data->current_selector = g_strdup( selector );	
}

/* *** SAC HANDLERS START *** */
static void
sac_handler_start_document (CRDocHandler * a_handler)
{
	CssedWindow* window;
	g_return_if_fail (a_handler);

	window = get_window_fom_sac_handler_data (a_handler);
	cssed_window_output_clear( window );
}

static void
sac_handler_end_document (CRDocHandler * a_handler)
{
	g_return_if_fail (a_handler);
}

static void
sac_handler_import_style (CRDocHandler * a_handler, GList * a_media_list,
			  GString * a_uri, GString * a_uri_default_ns)
{
	GList *cur = NULL;
	guchar *str;
	guchar *out;
	CssedCssParserData* data;

	g_return_if_fail (a_handler);
	data = (CssedCssParserData*) a_handler->app_data	;

	document_write_error_to_program_output ((CssedDoc *) data->
						doc,
						"Input style from import found.",
						OUT_COLOR);

	if (a_media_list)
	{
		document_write_error_to_program_output ((CssedDoc *)
							data->doc,
							"\tMedia list",
							OUT_COLOR);

		for (cur = a_media_list; cur; cur = cur->next)
		{
			if (cur->data)
			{
				str = g_strndup
					(((GString *) cur->data)->str,
					 ((GString *) cur->data)->len);

				if (str)
				{
					out = g_strdup_printf ("\t\t%s", str);
					document_write_error_to_program_output
						((CssedDoc *) data->
						 doc, out, OUT_COLOR);
					g_free (str);
					g_free (out);
					str = NULL;
					out = NULL;
				}
			}
		}

		document_write_error_to_program_output ((CssedDoc *)
							data->doc,
							"\tDefault namespace",
							OUT_COLOR);
		if (a_uri_default_ns && a_uri_default_ns->str)
		{
			str = g_strndup (a_uri_default_ns->str,
					 a_uri_default_ns->len);
			if (str)
			{
				out = g_strdup_printf ("\t\t%s", str);
				document_write_error_to_program_output ((CssedDoc *) data->doc, out, OUT_COLOR);
				g_free (str);
				g_free (out);
				str = NULL;
				out = NULL;
			}
		}
	}
	a_uri = NULL;		/*keep compiler happy */
}


static void
sac_handler_namespace_declaration (CRDocHandler * a_handler,
				   GString * a_prefix, GString * a_uri)
{
	guchar *prefix = NULL;
	guchar *uri = NULL;
	guchar *out = NULL;
	CssedCssParserData* data;

	g_return_if_fail (a_handler);
	data = (CssedCssParserData*) a_handler->app_data;	

	document_write_error_to_program_output ((CssedDoc *) data->
						doc,
						"parser: Namespace declaration",
						OUT_COLOR);
	if (a_prefix && a_prefix->str)
	{
		prefix = g_strndup (a_prefix->str, a_prefix->len);

		if (prefix)
		{
			out = g_strdup_printf ("\t\tprefix: %s", prefix);
			document_write_error_to_program_output ((CssedDoc *)
								data->
								doc, out,
								OUT_COLOR);
			g_free (prefix);
			g_free (out);
			out = NULL;
			prefix = NULL;
		}
	}

	if (a_uri && a_uri->str)
	{

		uri = g_strndup (a_uri->str, a_uri->len);

		if (uri)
		{
			out = g_strdup_printf ("\t\turi: %s", uri);
			document_write_error_to_program_output ((CssedDoc *)
								data->
								doc, out,
								OUT_COLOR);
			g_free (prefix);
			g_free (out);
			out = NULL;
			prefix = NULL;
		}
	}
}


static void
sac_handler_comment (CRDocHandler * a_handler, GString * a_comment)
{
	gchar *out = NULL;
	guchar *comment = NULL;
	CssedCssParserData* data;

	g_return_if_fail (a_handler);
	data = (CssedCssParserData*) a_handler->app_data;	
	
	if (a_comment && a_comment->str)
	{
		comment = g_strndup (a_comment->str, a_comment->len);

		if (comment)
		{
			out = g_strdup_printf ("comment -> %s",
					       comment);
			document_write_error_to_program_output ((CssedDoc *)
								data->
								doc, out,
								OUT_COLOR);
			g_free (comment);
			g_free (out);
			out = NULL;
			comment = NULL;
		}
	}
}



static void
sac_handler_start_selector (CRDocHandler * a_handler,
			    CRSelector * a_selector_list)
{
	guchar *tmp_buf = NULL;
	gchar *out = NULL;
	CssedCssParserData* data;

	g_return_if_fail (a_handler);
	data = (CssedCssParserData*) a_handler->app_data;	

	if (a_selector_list)
	{
		tmp_buf = cr_selector_to_string (a_selector_list);
		if (tmp_buf)
		{
			data->current_selector = g_strdup(tmp_buf);
			out = g_strdup_printf ("%s", tmp_buf);
			document_write_error_to_program_output ((CssedDoc *)
								data->
								doc, out,
								SEL_COLOR);
			g_free (out);
			g_free (tmp_buf);
			tmp_buf = NULL;
			out = NULL;
		}
	}
}


static void
sac_handler_end_selector (CRDocHandler * a_handler,
			  CRSelector * a_selector_list)
{
	guchar *tmp_buf = NULL;
	gchar *out = NULL;
	CssedCssParserData* data;

	g_return_if_fail (a_handler);
	data = (CssedCssParserData*) a_handler->app_data;	

	if (a_selector_list)
	{
		tmp_buf = cr_selector_to_string (a_selector_list);
		if (tmp_buf)
		{
			out = g_strdup_printf ("%s end", tmp_buf);
			document_write_error_to_program_output ((CssedDoc *)
								data->
								doc, out,
								SEL_COLOR);
			g_free (out);
			g_free (tmp_buf);
			tmp_buf = NULL;
			out = NULL;
		}
	}
}


static void
sac_handler_property (CRDocHandler * a_handler, GString * a_name,
		      CRTerm * a_expr)
{
	gchar *out = NULL;
	guchar* val;	
	guchar *name ;
	CssedCssParserData* data;

	g_return_if_fail (a_handler);
	data = (CssedCssParserData*) a_handler->app_data;	


	if (a_name && a_name->str)
	{
		name = g_strndup (a_name->str, a_name->len);

		if (name && a_expr)
		{
			val = cr_term_to_string (a_expr) ;
			out=g_strdup_printf("\t%s : %s",name,val);
			document_write_error_to_program_output ((CssedDoc *) data->
						doc,
						out,
						PRO_COLOR);
			g_free( out );
			g_free( val );
		}
		else
		{
			if( name ){
				out=g_strdup_printf("\tempty property -> %s",name);
				document_write_error_to_program_output ((CssedDoc *) data->
							doc,
							out,
							ERR_COLOR);
				g_free( out );
				
			}
			if( a_expr ){
				val=g_strdup_printf("\tempty value -> %s",name);
				document_write_error_to_program_output ((CssedDoc *) data->
							doc,
							val,
							ERR_COLOR);
				g_free( val );				
			}
		}

		if (name)
		{
			g_free (name);
			name = NULL;
		}
	}
}


static void
sac_handler_start_font_face (CRDocHandler * a_handler)
{
	g_return_if_fail (a_handler);

}


static void
sac_handler_end_font_face (CRDocHandler * a_handler)
{
	g_return_if_fail (a_handler);
}

static void
sac_handler_start_media (CRDocHandler * a_handler, GList * a_media_list)
{
	CssedCssParserData* data;

	g_return_if_fail (a_handler);
	data = (CssedCssParserData*) a_handler->app_data	;


	if (a_media_list)
	{
		GList *cur = NULL;
		guchar *medium = NULL;
		gchar *out = NULL; 

		for (cur = a_media_list; cur; cur = cur->next)
		{
			if (cur->data == NULL)
				continue;

			medium = g_strndup (((GString *) cur->data)->str,
					    ((GString *) cur->data)->len);

			if (medium == NULL)
				continue;

			out=g_strdup_printf("Media %s",medium);
			document_write_error_to_program_output ((CssedDoc *) data->
							doc,
							out,
							MED_COLOR);
			g_free( out );	

			if (medium)
			{
				g_free (medium);
				medium = NULL;
			}
		}
	}
}


static void
sac_handler_end_media (CRDocHandler * a_handler, GList * a_media_list)
{
	CssedCssParserData* data;
	gchar* out;

	g_return_if_fail (a_handler);
	data = (CssedCssParserData*) a_handler->app_data	;

	if (a_media_list)
	{
		GList *cur = NULL;
		guchar *medium = NULL;

		for (cur = a_media_list; cur; cur = cur->next)
		{
			if (cur->data == NULL)
				continue;

			medium = g_strndup (((GString *) cur->data)->str,
					    ((GString *) cur->data)->len);

			if (medium == NULL)
				continue;
			out = g_strdup_printf("end Media %s",medium );
			document_write_error_to_program_output ((CssedDoc *) data->
							doc,
							out,
							MED_COLOR);
			g_free( out );	

			if (medium)
			{
				g_free (medium);
				medium = NULL;
			}
		}
	}
}

static void
sac_handler_start_page (CRDocHandler * a_handler, GString * a_name,
			GString * a_pseudo_page)
{
	guchar *name = NULL, *pseudo_page = NULL;
	CssedCssParserData* data;
	gchar* out;

	g_return_if_fail (a_handler);
	data = (CssedCssParserData*) a_handler->app_data	;

	if (a_name && a_name->str)
	{
		name = g_strndup (a_name->str, a_name->len);
	}

	if (a_pseudo_page && a_pseudo_page->str)
	{
		pseudo_page = g_strndup (a_pseudo_page->str,
					 a_pseudo_page->len);
	}

	if (name && pseudo_page)
	{
			out = g_strdup_printf("Paged Media <span color='navy'>%s : %s</span>",name, pseudo_page);
			document_write_error_to_program_output ((CssedDoc *) data->
							doc,
							out,
							MED_COLOR);
			g_free( out );	
	}else{
		if (name)
		{
			out = g_strdup_printf("Paged Media <span color='navy'>%s</span>",name);
			document_write_error_to_program_output ((CssedDoc *) data->
							doc,
							out,
							MED_COLOR);
			g_free( out );	
		}
		if (pseudo_page)
		{
			out = g_strdup_printf("Paged Media <span color='navy'>%s </span>", pseudo_page);
			document_write_error_to_program_output ((CssedDoc *) data->doc,	out, MED_COLOR);
			g_free( out );	
		}
	}
	if (name)
	{
		g_free (name);
		name = NULL;
	}

	if (pseudo_page)
	{
		g_free (pseudo_page);
		pseudo_page = NULL;
	}
}

static void
sac_handler_end_page (CRDocHandler * a_handler, GString * a_name,
		      GString * a_pseudo_page)
{

	guchar *name = NULL, *pseudo_page = NULL;
	CssedCssParserData* data;
	gchar* out;

	g_return_if_fail (a_handler);
	data = (CssedCssParserData*) a_handler->app_data	;

	if (a_name && a_name->str)
	{
		name = g_strndup (a_name->str, a_name->len);
	}

	if (a_pseudo_page && a_pseudo_page->str)
	{
		pseudo_page = g_strndup (a_pseudo_page->str,
					 a_pseudo_page->len);
	}

	if (name && pseudo_page)
	{
			out = g_strdup_printf("end Paged Media <span color='navy'>%s : %s</span>",name, pseudo_page);
			document_write_error_to_program_output ((CssedDoc *) data->
							doc,
							out,
							MED_COLOR);
			g_free( out );	
	}else{
		if (name)
		{
			out = g_strdup_printf("end Paged Media <span color='navy'>%s</span>",name);
			document_write_error_to_program_output ((CssedDoc *) data->
							doc,
							out,
							MED_COLOR);
			g_free( out );	
		}
		if (pseudo_page)
		{
			out = g_strdup_printf("end Paged Media <span color='navy'>%s </span>", pseudo_page);
			document_write_error_to_program_output ((CssedDoc *) data->
							doc,
							out,
							MED_COLOR);
			g_free( out );	
		}
	}

	if (name)
	{
		g_free (name);
		name = NULL;
	}

	if (pseudo_page)
	{
		g_free (pseudo_page);
		pseudo_page = NULL;
	}
}


static void
sac_handler_ignorable_at_rule (CRDocHandler * a_handler, GString * a_name)
{
	guchar *name = NULL;

	g_return_if_fail (a_handler);

	fprintf (stdout, "*********************\n");
	fprintf (stdout, "ignorable_at_rule\n");

	if (a_name && a_name->str)
	{
		name = g_strndup (a_name->str, a_name->len);
	}

	if (name)
	{
		fprintf (stdout, "%s\n", name);
	}

	fprintf (stdout, "*********************\n\n");
}

void
sac_handler_parse_error (CRDocHandler * a_handler, gint line)
{
	CssedCssParserData* data;
	gchar* err;
	
	data = (CssedCssParserData*) a_handler->app_data	;
	data->err_count++;	
	err = g_strdup_printf("[%d] : syntax error (%d)",line+1,data->err_count);
	document_write_error_to_program_output ((CssedDoc *) data->
						doc,
						err,
						ERR_COLOR);
	g_free(err);
}

void
sac_handler_fatal_error (CRDocHandler * a_handler)
{
	CssedCssParserData* data;
	data = (CssedCssParserData*) a_handler->app_data	;

	document_write_error_to_program_output ((CssedDoc *) data->
						doc,
						"FATAL error",
						ERR_COLOR);
}

static void
init_sac_handler (CRDocHandler * a_handler)
{
	a_handler->start_document = sac_handler_start_document;
	a_handler->end_document = sac_handler_end_document;
	a_handler->import_style = sac_handler_import_style;
	a_handler->namespace_declaration = sac_handler_namespace_declaration;
	a_handler->comment = sac_handler_comment;
	a_handler->start_selector = sac_handler_start_selector;
	a_handler->end_selector = sac_handler_end_selector;
	a_handler->property = sac_handler_property;
	a_handler->start_font_face = sac_handler_start_font_face;
	a_handler->end_font_face = sac_handler_end_font_face;
	a_handler->start_media = sac_handler_start_media;
	a_handler->end_media = sac_handler_end_media;
	a_handler->start_page = sac_handler_start_page;
	a_handler->end_page = sac_handler_end_page;
	a_handler->ignorable_at_rule = sac_handler_ignorable_at_rule;
	a_handler->error = sac_handler_parse_error;
	a_handler->unrecoverable_error = sac_handler_fatal_error;
}

void
init_no_dump_sac_handler (CRDocHandler * a_handler)
{
	a_handler->start_document = sac_handler_start_document; // clean previous output
	a_handler->end_document = NULL;
	a_handler->import_style = NULL;
	a_handler->namespace_declaration = NULL;
	a_handler->comment = NULL;
	a_handler->start_selector = NULL;
	a_handler->end_selector = NULL;
	a_handler->property = NULL;
	a_handler->start_font_face = NULL;
	a_handler->end_font_face = NULL;
	a_handler->start_media = NULL;
	a_handler->end_media = NULL;
	a_handler->start_page = NULL;
	a_handler->end_page = NULL;
	a_handler->ignorable_at_rule = NULL;
	a_handler->error = sac_handler_parse_error;
	a_handler->unrecoverable_error = NULL;
}
/* *** SAC HANDLERS END *** */




void
cssed_cr_parser_parse_buffer (CssedDoc * doc, gchar * buffer, gint len, gboolean dump)
{
	enum CRStatus status = CR_OK;
	CRDocHandler *handler = { 0 };
	CRParser *parser = NULL;
	CssedCssParserData* data;
	gchar* err_buf;
	gchar* val_buf;
	GList* err_list;
	CRParserError* cr_err;
	
	data = g_malloc(sizeof(CssedCssParserData));	
	data->doc = doc;
	data->err_count = 0;

	//g_return_val_if_fail (buffer, CR_BAD_PARAM_ERROR);

	parser = cr_parser_new_from_buf (buffer, len, CR_UTF_8, FALSE);
	handler = cr_doc_handler_new ();
	if( dump ){
		init_sac_handler (handler);
	}else{
		init_no_dump_sac_handler (handler);
	}
	handler->app_data = data;
	status = cr_parser_set_sac_handler (parser, handler);

	if (status != CR_OK)
	{//FIXME say somethig to user
		cr_parser_destroy (parser);
		return;
	}

	//g_print("validating buffer: %s\n", buffer);
	status = cr_parser_parse (parser);
	switch( status ){		
		case CR_OK:
			if( data->err_count > 0 ){
				err_buf = g_strdup_printf(
						_("The CSS document can be parsed, but has syntax errors (%d).\nCheck the program output window."),
						data->err_count);
				cssed_error_message(err_buf,_("Validation result ok with errors"));
				g_free(err_buf);
			}else{
				cssed_error_message(
					_("The CSS document is well formed and\ncan be parsed by a browser."),
					_("Validation result ok")
					);
			}
			break;
		case  CR_ERROR:	
			cssed_error_message(_("The CSS document is not well formed"),
								_("Validation fails"));
			break;	
		case  CR_BAD_PARAM_ERROR:
		case  CR_INSTANCIATION_FAILED_ERROR:
		case  CR_EMPTY_PARSER_INPUT_ERROR:
		case  CR_ENCODING_ERROR:
		case  CR_ENCODING_NOT_FOUND_ERROR:
			cssed_error_message(_("The validator found an internal error\nand is unable to continue."),
								_("Validation fails"));
			break;
		default:
			cssed_error_message(_("The CSS document is not well formed"),
								_("Validation fails"));
			break;
	}
	
	 err_list = cr_parser_get_error_stack(parser);

    for ( ;err_list;err_list = err_list->next )
    {		
		cr_err = (CRParserError*) err_list->data;
		if ( cr_err != NULL ){
			val_buf = g_strdup_printf("[%d] - %s ",((int) cr_err->line + 1) ,cr_err->msg	);
			document_write_error_to_program_output(doc,val_buf,ERR_COLOR);
			g_free(val_buf);
		}
	}	
	cr_parser_destroy (parser);
	if( data->current_selector )
		g_free(data->current_selector);
	g_free(data);
	handler = NULL;
}

