/*-----------------------------------------------------------------------
  
    File        : tutorial.cpp

    Description : View intensity profile on an image, along the X-axis

    Author      : David Tschumperl David

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    ---------------------------------------------------------------------*/

#include "../CImg.h"
using namespace cimg_library;

int main(int argc,char **argv) {

  // Define program usage and read command line parameters
  //-------------------------------------------------------
  cimg_usage("View intensity profile of an image along the X axis");
  const char* file_i = cimg_option("-i","img/parrot_original.ppm","Input image");
  const double sigma = cimg_option("-blur",1,"Variance of gaussian pre-blurring");

  // Init variables
  //----------------
  
  // Load image and blur it with the parameter sigma.
  const CImg<unsigned char> image = CImg<unsigned char>(file_i).blur((float)sigma);

  // Create two display window, one to display the image, the other to display the intensity profiles.
  CImgDisplay 
    main_disp(image,"Move mouse over the image",0,0x12),
    draw_disp(500,400,"Intensity profile of the X-axis",0,0x11);

  // Define colors for plotting the intensity profiles, and hatch for drawing a hatched line
  unsigned long hatch=0xF0F0F0F0;
  const unsigned char 
    red  [3] = {255,  0,  0}, 
    green[3] = {  0,255,  0},
    blue [3] = {  0,  0,255},
    white[3] = {255,255,255},
    grey [3] = {128,128,128};

  // Enter event loop
  while (!main_disp.closed && !draw_disp.closed) {

    // Handle display window resizing (if any)
    main_disp.resize(main_disp).wait(40);
    draw_disp.resize();

    if (main_disp.mousex>=0 && main_disp.mousey>=0) {
      // if mouse pointer is over the main image display, plot the profiles in the profile display window
      const int x = main_disp.mousex*draw_disp.dimx()/main_disp.dimx(), y = main_disp.mousey*image.dimy()/main_disp.dimy();
      CImg<unsigned char>(draw_disp.width,draw_disp.height,1,3,0).
        draw_axeXY(0.0,(double)image.width,255,0,grey).
        draw_graph(image.ref_line(y,0,0),red,0,0,255).
        draw_graph(image.ref_line(y,0,1),green,0,0,255).
        draw_graph(image.ref_line(y,0,2),blue,0,0,255).
        draw_text(15,5,white,NULL,1,"(x,y)=(%d,%d)",main_disp.mousex,main_disp.mousey).
        draw_line(x,0,x,draw_disp.dimy()-1,white,hatch=cimg::rol(hatch),0.5).
        display(draw_disp);
    } else
      // else display a text in the profile display window.
      CImg<unsigned char>(draw_disp.width,draw_disp.height).fill(0).
        draw_text("Mouse pointer is outside the image",draw_disp.dimx()/2-110,draw_disp.dimy()/2-5,white).display(draw_disp);
  }
  exit(0);
  return 0;
}
