/*-----------------------------------------------------------------------------------------

 File        : pde_TschumperleDeriche3D.cpp

 Description : Implementation of the Tschumperle-Deriche's Regularization
               PDE, for 3D multivalued volumes, as described in the following articles :
 
 (1) PDE-Based Regularization of Multivalued Images and Applications.
     (D. Tschumperl). PhD Thesis. University of Nice-Sophia Antipolis, December 2002.
 (2) Diffusion PDE's on Vector-valued Images : Local Approach and Geometric Viewpoint.
     (D. Tschumperl and R. Deriche). IEEE Signal Processing Magazine, October 2002.
 (3) Vector-Valued Image Regularization with PDE's : A Common Framework for Different Applications.
     (D. Tschumperl and R. Deriche). CVPR'2003, Computer Vision and Pattern Recognition, Madison, United States, June 2003.  
 
 This code can be used to perform image restoration, inpainting, magnification or flow visualization.
 
 Author      : David Tschumperl

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

------------------------------------------------------------------------------------*/

#define cimg_lapack 1
#include "../CImg.h"
// The lines below are not necessary in your own code, it simply allows 
// the source compilation with compilers that do not respect the C++ standart.
#if ( defined(_MSC_VER) && _MSC_VER<=1200 ) || defined(__DMC__)
#define std
#endif
using namespace cimg_library;

int main(int argc,char **argv) {

  // Read command line arguments
  //-----------------------------
  cimg_usage("Tschumperle-Deriche's flow for 2D Image Restoration, Inpainting, Magnification or Flow visualization");
  const char *file_i  = cimg_option("-i",(char*)NULL,"Input image");
  const char *file_m  = cimg_option("-m",(char*)NULL,"Mask image (if Inpainting)");
  const char *file_f  = cimg_option("-f",(char*)NULL,"Flow image (if Flow visualization)");
  const char *file_o  = cimg_option("-o",(char*)NULL,"Output file");
  const double zoom   = cimg_option("-zoom",1.0,"Image magnification");

  const unsigned int nb_iter  = cimg_option("-iter",100000,"Number of iterations");
  const double dt     = cimg_option("-dt",20,"Adapting time step");
  const double alpha  = cimg_option("-alpha",0,"Gradient smoothing");
  const double sigma  = cimg_option("-sigma",0.5,"Structure tensor smoothing");
  const double a1     = cimg_option("-a1",1.0,"Diffusion coef along minimal variations");
  const double a2     = cimg_option("-a2",0.5,"Diffusion coef along medium variations");
  const double a3     = cimg_option("-a3",0.1,"Diffusion coef along maximal variations");
  const double noiseg = cimg_option("-ng",0.0,"Add gauss noise before aplying the algorithm");
  const double noiseu = cimg_option("-nu",0.0,"Add uniform noise before applying the algorithm");
  const double noises = cimg_option("-ns",0.0,"Add salt&pepper noise before applying the algorithm");
  const bool filter   = cimg_option("-filter",false,"Filtering-based scheme");

  const bool stflag   = cimg_option("-stats",false,"Display image statistics at each iteration");
  const unsigned int save     = cimg_option("-save",0,"Iteration saving step");
  const unsigned int visu     = cimg_option("-visu",10,"Visualization step (0=no visualization)");
  const unsigned int init     = cimg_option("-init",2,"Inpainting initialization (0=black, 1=white, 2=noise, 3=unchanged)");
  const unsigned int skip     = cimg_option("-skip",1,"Step of image geometry computation");
  const double na1 = -1.0/a1, na2 = -1.0/a2, na3 = -1.0/a3;
  double xdt = 0;

  // Variable initialization
  //-------------------------
  CImg<> img, flow;
  CImg<int> mask;

  if (file_i) {
    img = CImg<>(file_i);
    if (file_m) mask = CImg<unsigned char>(file_m).resize(img.width,img.height,img.depth,1);
    else if (zoom>1) {
      mask = CImg<int>(img.width,img.height,img.depth,1,-1).
        resize((int)(img.width*zoom),(int)(img.height*zoom),(int)(img.depth*zoom),1,4)+1;
      img.resize((int)(img.width*zoom),(int)(img.height*zoom),(int)(img.depth*zoom),-100,3);
    }
  } else {
    if (file_f) {
      flow = CImg<>(file_f);
      img = CImg<>((int)(flow.width*zoom),(int)(flow.height*zoom),(int)(flow.depth*zoom),1,0).noise(100,2);
      flow.resize(img.width,img.height,img.depth,3,3);      
    } else throw CImgException("You need to specify at least one input image (option -i), or one flow image (option -f)");
  }
  img.noise(noiseg,0).noise(noiseu,1).noise(noises,2);
  CImgStats initial_stats(img,false);
  if (mask.data && init!=3) 
    cimg_mapXYZV(img,x,y,z,k) if (mask(x,y,z)) 
      img(x,y,z,k)=(float)((init?
                    (init==1?initial_stats.max:((initial_stats.max-initial_stats.min)*cimg::rand())):
                    initial_stats.min));
  
  CImgDisplay *disp = visu?img.new_display("Iterated Image",2,0x11):NULL;
  CImg<> G(img.width,img.height,img.depth,6,0), T(G,false), veloc(img,false), val(3), vec(3,3);
  
  // PDE main iteration loop
  //-------------------------
  for (unsigned int iter=0; iter<nb_iter && (!disp || !disp->closed && disp->key!=cimg::keyQ && disp->key!=cimg::keyESC); iter++) {
    std::printf("\riter %d , xdt = %g               ",iter,xdt); fflush(stdout);
    if (stflag) img.print();

    if (!(iter%skip)) {
      // Compute the tensor field T, used to drive the diffusion
      //---------------------------------------------------------
      
      // When using PDE for flow visualization
      if (flow.data) cimg_mapXYZ(flow,x,y,z) {
        const float
          u = flow(x,y,z,0),
          v = flow(x,y,z,1),
          w = flow(x,y,z,2),
          n = (float)std::sqrt((double)(u*u+v*v+w*w)),
          nn = (n!=0)?n:1;
        T(x,y,0) = u*u/nn;
        T(x,y,1) = u*v/nn;
        T(x,y,2) = u*w/nn;
        T(x,y,3) = v*v/nn;
        T(x,y,4) = v*w/nn;
        T(x,y,5) = w*w/nn;
      } else {
        
        // Compute structure tensor field G
        const CImgl<> grad = img.get_gradientXYZ(3);
        if (alpha!=0) cimgl_map(grad,l) grad[l].blur((float)alpha);
        cimg_mapXYZV(img,x,y,z,k) {
          const float ix = grad[0](x,y,z,k), iy = grad[1](x,y,z,k), iz = grad[2](x,y,z,k);
          G(x,y,z,0) += ix*ix;
          G(x,y,z,1) += ix*iy;
          G(x,y,z,2) += ix*iz;
          G(x,y,z,3) += iy*iy;
          G(x,y,z,4) += iy*iz;
          G(x,y,z,5) += iz*iz;          
        }
        if (sigma!=0) G.blur((float)sigma);
        
        if (mask.data) {
          
          // When using PDE for image inpainting or zooming
          cimg_mapXYZ(G,x,y,z) if (mask(x,y,z)) {
            G.get_tensor(x,y,z).symeigen(val,vec);
            const double ng = std::sqrt((double)(1e-7+val[0]+val[1]+val[2]));
            const float
              ux = vec[0],
              uy = vec[1],
              uz = vec[2];
            T(x,y,z,0) = ux*ux;
            T(x,y,z,1) = ux*uy;
            T(x,y,z,2) = ux*uz;
            T(x,y,z,3) = uy*uy;
            T(x,y,z,4) = uy*uz;
            T(x,y,z,5) = uz*uz;
          }
        } else {
          
          // When using for image restoration
          cimg_mapXYZ(G,x,y,z) { 
            G.get_tensor(x,y,z).symeigen(val,vec);
            const double ng = std::sqrt((double)(1e-7+val[0]+val[1]+val[2]));
            const float
              l1 = (float)std::pow(ng,na1),
              l2 = (float)std::pow(ng,na2),
              l3 = (float)std::pow(ng,na3),
              ux = vec[0],
              uy = vec[1],
              uz = vec[2],
              vx = vec[3],
              vy = vec[4],
              vz = vec[5],
              wx = vec[6],
              wy = vec[7],
              wz = vec[8];
            T(x,y,z,0) = l1*ux*ux + l2*vx*vx + l3*wx*wx;
            T(x,y,z,1) = l1*ux*uy + l2*vx*vy + l3*wx*wy;
            T(x,y,z,2) = l1*ux*uz + l2*vx*vz + l3*wx*wz;
            T(x,y,z,3) = l1*uy*uy + l2*vy*vy + l3*wy*wy;
            T(x,y,z,4) = l1*uy*uz + l2*vy*vz + l3*wy*wz;
            T(x,y,z,5) = l1*uz*uz + l2*vz*vz + l3*wz*wz;
          }
        }
      }
    }
    
    // Compute the PDE velocity and update the iterated image
    //--------------------------------------------------------
    CImg_3x3x3(I,float);
    veloc.fill(0);
    cimg_mapV(img,k) cimg_map3x3x3(img,x,y,z,k,I) if (!mask.data || mask(x,y,z)) {
      const float 
        a = T(x,y,z,0), 
        b = T(x,y,z,1),
        c = T(x,y,z,2),
        d = T(x,y,z,3),
        e = T(x,y,z,4),
        f = T(x,y,z,5),
        ixx = Incc+Ipcc-2*Iccc,       
        iyy = Icnc+Icpc-2*Iccc,
        izz = Iccn+Iccp-2*Iccc,
        ixy = 0.25f*(Ippc+Innc-Ipnc-Inpc),
        ixz = 0.25f*(Ipcp+Incn-Ipcn-Incp),
        iyz = 0.25f*(Icpp+Icnn-Icpn-Icnp);
      veloc(x,y,z,k) = a*ixx + 2*b*ixy + 2*c*ixz + d*iyy + 2*e*iyz + f*izz;
    }
    const CImgStats stats(veloc,false);
    xdt = dt/cimg::max(std::fabs(stats.min),std::fabs(stats.max));
    img+=veloc*xdt;
    img.cut((float)initial_stats.min,(float)initial_stats.max);

    // Display and save iterations
    if (disp && !(iter%visu)) img.display(*disp);
    if (save && file_o && !(iter%save)) img.save(file_o,iter);
    if (disp) disp->resize().display(img);
  }

  // Save result and exit.
  if (file_o) img.save(file_o);
  return 0;
}
