/*
 * main.c
 *
 * Copyright (C) 2003 Bastian Blank <waldi@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $LastChangedBy: bastian $
 * $LastChangedDate: 2005-01-05 22:48:37 +0100 (Mi, 05 Jan 2005) $
 * $LastChangedRevision: 785 $
 */

#include <config.h>

#include "download.h"
#include "execute.h"
#include "frontend.h"
#include "install.h"
#include "log.h"
#include "suite.h"

#include <debian-installer.h>

#include <getopt.h>
#include <libgen.h>
#include <limits.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <stdarg.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

const char *mirror;
int debug = 0;
int quiet = 0;
int verbose = 0;

#ifdef DEB_ARCH
char *const default_arch = DEB_ARCH;
#endif
char *const default_configdir = CONFIGDIR;
char *const default_flavour = "standard";
char *const default_mirror = "http://ftp.debian.org/debian";

enum
{
  GETOPT_FIRST = CHAR_MAX + 1,
  GETOPT_DEBUG,
  GETOPT_DEVFS,
  GETOPT_VERSION,
};

static struct option const long_opts[] =
{
  {"arch", required_argument, 0, 'a'},
  {"configdir", required_argument, 0, 'c'},
  {"debug", no_argument, 0, GETOPT_DEBUG},
  {"download-only", no_argument, 0, 'd'},
  {"flavour", required_argument, 0, 'f'},
  {"helperdir", required_argument, 0, 'H'},
  {"include", required_argument, 0, 'i'},
  {"quiet", no_argument, 0, 'q'},
  {"verbose", no_argument, 0, 'v'},
  {"help", no_argument, 0, 'h'},
  {"version", no_argument, 0, GETOPT_VERSION},
  {0, 0, 0, 0}
};

char *program_name;

int frontend_download (const char *source, const char *target)
{
  char buf[1024];
  int ret;

  if (!strncmp (mirror, "http://", 7) || !strncmp (mirror, "ftp://", 6))
    snprintf (buf, 1024, "wget -q -O %s %s/%s", target, mirror, source);
  else if (!strncmp (mirror, "file://", 7))
    snprintf (buf, 1024, "cp %s/%s %s", mirror + sizeof ("file://") - 1, source, target);
  ret = execute (buf);

  return WEXITSTATUS (ret);
}

static int check_mirror (void)
{
  if (!strncmp (mirror, "http://", 7))
    return 0;
  else if (!strncmp (mirror, "ftp://", 6))
    return 0;
  else if (!strncmp (mirror, "file://", 7))
    return 0;
  return 1;
}

static void usage (int status)
{
  if (status != 0)
    fprintf (stderr, "Try `%s --help' for more information.\n", program_name);
  else
  {
    fprintf (stdout, "\
Usage: %s [OPTION]... SUITE TARGET [MIRROR]\n\
\n\
", program_name);
    fputs ("\
Mandatory arguments to long options are mandatory for short options too.\n\
", stdout);
    fputs ("\
  -a, --arch=ARCH              set the target architecture\n\
  -c, --configdir=DIR          set the config directory\n\
      --debug                  enable debug\n\
  -d, --download-only          download packages, but don't perform installation\n\
  -f, --flavour=FLAVOUR        select the flavour to use\n\
  -H, --helperdir=DIR          set the helper directory\n\
  -q, --quiet                  be quiet\n\
  -v, --verbose                be verbose\n\
  -h, --help                   display this help and exit\n\
      --version                output version information and exit\n\
", stdout);
    fputs ("\n\
Defines:\n\
target architecture: " 
#ifdef DEB_ARCH
DEB_ARCH 
#else
"(no default)"
#endif
"\n", stdout);
  }
  exit (status);
}

int frontend_main (int argc, char **argv, char **envp)
{
  int c;
  char *arch = NULL, *configdir = default_configdir, *flavour = default_flavour, *helperdir = default_configdir, *suite = NULL, *target = NULL;
  bool download_only = false;
  const char **include = NULL;
  unsigned int include_len = 0, include_size = 0;

  program_name = argv[0];

  while ((c = getopt_long (argc, argv, "a:c:df:hH:i:qv", long_opts, NULL)) != -1)
  {
    switch (c)
    {
      case 0:
        break;
      case 'a':
        arch = optarg;
        break;
      case 'c':
        configdir = optarg;
        break;
      case 'd':
        download_only = true;
        break;
      case 'f':
        flavour = optarg;
        break;
      case 'h':
        usage (EXIT_SUCCESS);
        break;
      case 'H':
        helperdir = optarg;
        break;
      case 'i':
        {
          if (include_size <= include_len || include_size == 0)
          {
            if (include_size == 0)
            {
              include = di_new (const char *, 2);
              include_size = 2;
            }
            else
            {
              include = di_renew (const char *, include, include_size * 2);
              include_size *= 2;
            }
          }
          include[include_len++] = strdup (optarg);
        }
        break;
      case 'q':
        quiet++;
        break;
      case 'v':
        verbose++;
        break;
      case GETOPT_DEBUG:
        debug++;
        break;
      case GETOPT_VERSION:
        fputs (PACKAGE_STRING, stdout);
        fputs ("\n\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n\
", stdout);
        exit (EXIT_SUCCESS);
        break;
      default:
        usage (EXIT_FAILURE);
    }
  }

  if ((argc - optind) <= 0)
  {
    fprintf (stderr, "%s: missing suite argument\n", program_name);
    usage (EXIT_FAILURE);
  }
  else if ((argc - optind) == 1)
  {
    fprintf (stderr, "%s: missing target argument\n", program_name);
    usage (EXIT_FAILURE);
  }
  else if ((argc - optind) > 3)
  {
    fprintf (stderr, "%s: too much arguments\n", program_name);
    usage (EXIT_FAILURE);
  }
  else if ((argc - optind) == 3)
  {
    mirror = argv[optind + 2];
  }
  else
  {
    mirror = default_mirror;
  }

  suite = argv[optind];
  target = argv[optind + 1];

  if (debug)
  {
    quiet = 0;
    verbose = 1;
  }

  if (!arch)
#ifdef DEB_ARCH
    arch = default_arch;
#else
  {
    fprintf (stderr, "%s: missing architecture\n", program_name);
    usage (EXIT_FAILURE);
  }
#endif

  di_init (basename (program_name));

  if (check_mirror ())
    log_text (DI_LOG_LEVEL_ERROR, "invalid mirror");

  if (!download_only && getuid ())
    log_text (DI_LOG_LEVEL_ERROR, "need root privileges");

  log_init ();

  if (install_target_check (target))
    log_text (DI_LOG_LEVEL_ERROR, "target check");
  if (suite_init (suite, arch, flavour, include, include_len, configdir))
    log_text (DI_LOG_LEVEL_ERROR, "suite init");
  if (execute_init (envp))
    log_text (DI_LOG_LEVEL_ERROR, "execute init");
  if (download_init ())
    log_text (DI_LOG_LEVEL_ERROR, "download init");
  if (!download_only && install_init (helperdir))
    log_text (DI_LOG_LEVEL_ERROR, "install init");
  if (backend_main (download_only))
    log_text (DI_LOG_LEVEL_ERROR, "backend");
  return 0;
}

