// ScriptCommands.C  -*- C++ -*-
// Copyright (c) 1998 Etienne BERNARD

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "Utils.H"
#include "Server.H"
#include "ServerList.H"
#include "ScriptCommands.H"
#include "Interp.H"
extern "C" {
#include <libguile.h>
#include <guile/gh.h>
}

#define VERIFY_STRING(par) if (!gh_string_p((par))) \
                             return gh_long2scm(-17)

#define VERIFY_NUMBER(par) if (!gh_number_p((par))) \
                             return gh_long2scm(-17)

SCM
ScriptCommands::Action(SCM channel, SCM message)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(message);
  Message m = Commands::Action(Interp::bot, Utils::scm2String(channel),
                               Utils::scm2String(message));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::AddUser(SCM who, SCM maskChannel, SCM level, SCM prot, SCM aop, SCM expire, SCM password)
{
  // FIXME: Todo
  return gh_long2scm(0);
}

SCM
ScriptCommands::AddServer(SCM servername, SCM port)
{
  int p = 6667;
  if (gh_number_p(port))
    p = gh_scm2long(port);
  Message m = Commands::AddServer(Interp::bot,
                                  Utils::scm2String(servername),
                                  p);
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::AddShit(SCM mask, SCM maskChannel, SCM level, SCM expiration, SCM reason)
{
  // FIXME: Todo
  return gh_long2scm(0);
}

SCM
ScriptCommands::Ban(SCM channel, SCM who)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(who);
  Message m = Commands::Ban(Interp::bot, Utils::scm2String(channel),
                            Utils::scm2String(who));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Cycle(SCM channel)
{
  VERIFY_STRING(channel);
  Message m = Commands::Cycle(Interp::bot, Utils::scm2String(channel));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Deban(SCM channel, SCM who)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(who);
  Message m = Commands::Deban(Interp::bot, Utils::scm2String(channel),
                              Utils::scm2String(who));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::DelServer(SCM number)
{
  VERIFY_NUMBER(number);
  Message m = Commands::DelServer(Interp::bot, gh_scm2long(number));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::DelUser(SCM who, SCM maskChannel)
{
  VERIFY_STRING(who);
  VERIFY_STRING(maskChannel);
  Message m = Commands::DelUser(Interp::bot, Utils::scm2String(who),
                                Utils::scm2String(maskChannel));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::DelShit(SCM who, SCM maskChannel)
{
  VERIFY_STRING(who);
  VERIFY_STRING(maskChannel);
  Message m = Commands::DelShit(Interp::bot, Utils::scm2String(who),
                                Utils::scm2String(maskChannel));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Deop(SCM channel, SCM who)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(who);
  Message m = Commands::Deop(Interp::bot, Utils::scm2String(channel),
                             Utils::scm2String(who));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Die(SCM reason)
{
  String r = "Leaving";
  if (gh_string_p(reason))
    r = Utils::scm2String(reason);
  Message m = Commands::Die(Interp::bot, r);
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Do(SCM command)
{
  VERIFY_STRING(command);
  Message m = Commands::Do(Interp::bot, Utils::scm2String(command));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Invite(SCM channel, SCM who)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(who);
  Message m = Commands::Invite(Interp::bot, Utils::scm2String(channel),
                               Utils::scm2String(who));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Join(SCM channel, SCM key)
{
  VERIFY_STRING(channel);
  String k = "";
  if (gh_string_p(key))
    k = Utils::scm2String(key);
  Message m = Commands::Join(Interp::bot, Utils::scm2String(channel),
                             k);
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Keep(SCM channel, SCM modes)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(modes);
  Message m = Commands::Keep(Interp::bot, Utils::scm2String(channel),
                             Utils::scm2String(modes));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Kick(SCM channel, SCM who, SCM reason)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(who);

  String r = "";
  if (gh_string_p(reason))
    r = Utils::scm2String(reason);

  Message m = Commands::Kick(Interp::bot, Utils::scm2String(channel),
                             Utils::scm2String(who), r);
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::KickBan(SCM channel, SCM who, SCM reason)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(who);
  String r = "";
  if (gh_string_p(reason))
    r = Utils::scm2String(reason);
  Message m = Commands::KickBan(Interp::bot, Utils::scm2String(channel),
                                Utils::scm2String(who), r);
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Lock(SCM channel)
{
  VERIFY_STRING(channel);
  Message m = Commands::Lock(Interp::bot, Utils::scm2String(channel));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::LogPort(void)
{
  return Interp::bot->botInterp->logPort;
}

SCM
ScriptCommands::Mode(SCM channel, SCM mode)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(mode);
  Message m = Commands::Mode(Interp::bot, Utils::scm2String(channel),
                             Utils::scm2String(mode));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Msg(SCM nick, SCM message)
{
  VERIFY_STRING(nick);
  VERIFY_STRING(message);
  Message m = Commands::Msg(Interp::bot, Utils::scm2String(nick),
                            Utils::scm2String(message));
  return gh_long2scm(m.getCode());

}

SCM
ScriptCommands::NextServer(void)
{
  Message m = Commands::NextServer(Interp::bot);
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Nick(SCM nick)
{
  VERIFY_STRING(nick);
  Message m = Commands::Nick(Interp::bot, Utils::scm2String(nick));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Op(SCM channel, SCM who)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(who);
  Message m = Commands::Op(Interp::bot, Utils::scm2String(channel),
                           Utils::scm2String(who));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Part(SCM channel)
{
  VERIFY_STRING(channel);
  Message m = Commands::Part(Interp::bot, Utils::scm2String(channel));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Reconnect(void)
{
  Message m = Commands::Reconnect(Interp::bot);
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Say(SCM channel, SCM message)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(message);
  Message m = Commands::Say(Interp::bot, Utils::scm2String(channel),
                            Utils::scm2String(message));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Server(SCM number)
{
  VERIFY_NUMBER(number);
  Message m = Commands::Server(Interp::bot, gh_scm2long(number));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::SetVersion(SCM version)
{
  Message m = Commands::SetVersion(Interp::bot, Utils::scm2String(version));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::TBan(SCM channel, SCM who, SCM seconds)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(who);
  VERIFY_NUMBER(seconds);
  Message m = Commands::TBan(Interp::bot, Utils::scm2String(channel),
                             Utils::scm2String(who), gh_scm2long(seconds));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::TKBan(SCM channel, SCM who, SCM seconds, SCM reason)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(who);
  VERIFY_NUMBER(seconds);
  String r = "";
  if (gh_string_p(reason))
    r = Utils::scm2String(reason);
  Message m = Commands::TKBan(Interp::bot, Utils::scm2String(channel),
                              Utils::scm2String(who),
                              gh_scm2long(seconds), r);
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Topic(SCM channel, SCM topic)
{
  VERIFY_STRING(channel);
  VERIFY_STRING(topic);
  Message m = Commands::Topic(Interp::bot, Utils::scm2String(channel),
                              Utils::scm2String(topic));
  return gh_long2scm(m.getCode());
}

SCM
ScriptCommands::Unlock(SCM channel)
{
  VERIFY_STRING(channel);
  Message m = Commands::Unlock(Interp::bot, Utils::scm2String(channel));
  return gh_long2scm(m.getCode());
}




SCM
ScriptCommands::getNickname(void)
{
  return Utils::string2SCM(Interp::bot->nickName);
}

SCM
ScriptCommands::getServer(void)
{
  ::Server *serv = Interp::bot->serverList->currentServer();
  int serverNumber = Interp::bot->serverList->currentNumber;

  return gh_list(gh_long2scm(serverNumber),
                 Utils::string2SCM(serv->getHostName()),
                 gh_long2scm(serv->getPort()),
                 Utils::string2SCM(serv->getPassword()),
                 SCM_UNDEFINED);
}

SCM
ScriptCommands::getServerList(void)
{
  SCM res = gh_list(SCM_UNDEFINED);
  ::Server *s;
  int i = 0;
  vector<std::Server *>::iterator it = Interp::bot->serverList->begin();

  for ( ; it != Interp::bot->serverList->end(); ++it) {
    s = (*it);
    res = gh_append2(res,
                     gh_list(gh_list(gh_long2scm(i++),
                                     Utils::string2SCM(s->getHostName()),
                                     gh_long2scm(s->getPort()),
                                     Utils::string2SCM(s->getPassword()),
                                     SCM_UNDEFINED), SCM_UNDEFINED));
  }
  return res;
}

SCM
ScriptCommands::flushQueue(void)
{
  if (!Interp::bot->serverConnection)
    return SCM_BOOL_F;
  Interp::bot->serverConnection->queue->flush();
  return SCM_BOOL_T;
}

SCM
ScriptCommands::flushPort(void)
{
  scm_flush(Interp::bot->botInterp->logPort);
  return SCM_BOOL_T;
}

SCM
ScriptCommands::random(SCM scm_max)
{
  int max = 0;
  srand(time(NULL));
  if (gh_number_p(scm_max))
    max = gh_scm2int(scm_max);
  return gh_long2scm(max ? rand() % max : 0);
}

SCM
ScriptCommands::addCommand(SCM scm_commandName, SCM scm_function,
                           SCM scm_needsChannel, SCM scm_args,
                           SCM scm_minLevel)
{
  // We check that commandName is a string
  if (!gh_string_p(scm_commandName))
    return SCM_BOOL_F;

  // We check that the command does not exist
  String commandName = Utils::scm2String(scm_commandName).toUpper();
  list<class userFunction *>::iterator it =
    Interp::bot->userFunctions.begin();
  for ( ; it != Interp::bot->userFunctions.end(); ++it) {
    if ((*it)->name == commandName)
      return SCM_BOOL_F;
  }

  // Next we check that needsChannel is a boolean
  if (!gh_boolean_p(scm_needsChannel))
    return SCM_BOOL_F;
  bool needsChannel = (bool)gh_scm2bool(scm_needsChannel);

  // We check that minLevel is an integer and that it's
  // a valid level
  if (!gh_number_p(scm_minLevel))
    return SCM_BOOL_F;

  int minLevel = gh_scm2long(scm_minLevel);
  if (minLevel < User::NONE || minLevel > User::MASTER)
    return SCM_BOOL_F;

  // We check that "scm_function" is a Scheme procedure
  if (!gh_procedure_p(scm_function))
    return SCM_BOOL_F;

  // We check that args is an integer and is between 0 and 20 (arbitrary limit)
  if (!gh_number_p(scm_args))
    return SCM_BOOL_F;

  int args = gh_scm2long(scm_args);
  if (args < 0 || args > 20)
    return SCM_BOOL_F;

  // We add the command in the commands list
  Interp::bot->userFunctions.push_back(new userFunction(commandName, (void (*)(ServerConnection *, Person *, String, String))0, minLevel, needsChannel, args, scm_function));

  return SCM_BOOL_T;
}

SCM
ScriptCommands::delCommand(SCM scm_commandName)
{
  // We check that commandName is a string
  if (!gh_string_p(scm_commandName))
    return SCM_BOOL_F;

  // We check that the command does exist
  String commandName = Utils::scm2String(scm_commandName).toUpper();
  list<class userFunction *>::iterator it =
    Interp::bot->userFunctions.begin();
  for ( ; it != Interp::bot->userFunctions.end(); ++it) {
    if ((*it)->name == commandName)
      break;
  }

  if (it == Interp::bot->userFunctions.end())
    return SCM_BOOL_F;

  // We delete the command
  Interp::bot->userFunctions.erase(it);
  delete (*it);

  return SCM_BOOL_T;
}

SCM
ScriptCommands::AddHook(SCM type, SCM regex, SCM function)
{
  return gh_bool2scm(Interp::bot->botInterp->AddHook(gh_scm2long(type),
                                                     regex, function));
}

SCM
ScriptCommands::AddTimer(SCM when, SCM function)
{
  return Interp::bot->botInterp->AddTimer(gh_scm2long(when), function);
}

SCM
ScriptCommands::DelTimer(SCM timer)
{
  return gh_bool2scm(Interp::bot->botInterp->DelTimer(timer));
}
