/**
 * ========================================
 * JFreeReport : a free Java report library
 * ========================================
 *
 * Project Info:  http://www.object-refinery.com/jfreereport/index.html
 * Project Lead:  Thomas Morgner (taquera@sherito.org);
 *
 * (C) Copyright 2000-2003, by Simba Management Limited and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * ---------------------------------
 * DrawableContentFactoryModule.java
 * ---------------------------------
 * (C)opyright 2003, by Thomas Morgner and Contributors.
 *
 * Original Author:  Thomas Morgner;
 * Contributor(s):   David Gilbert (for Simba Management Limited);
 *
 * $Id: DrawableContentFactoryModule.java,v 1.8 2003/06/10 16:07:54 taqua Exp $
 *
 * Changes
 * -------
 * 09-Apr-2003 : Added standard header (DG);
 *
 */
package com.jrefinery.report.targets.base.content;

import java.awt.geom.Dimension2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import com.jrefinery.report.DrawableContainer;
import com.jrefinery.report.Element;
import com.jrefinery.report.targets.base.ElementLayoutInformation;
import com.jrefinery.report.targets.base.layout.LayoutSupport;

/**
 * A factory module for drawable content.
 *
 * @author Thomas Morgner.
 */
public class DrawableContentFactoryModule implements ContentFactoryModule
{
  /**
   * Returns <code>true</code> if the module can handle the specified content type, and
   * <code>false</code> otherwise.
   *
   * @param contentType  the content type.
   *
   * @return <code>true</code> or <code>false</code>.
   */
  public boolean canHandleContent(String contentType)
  {
    return contentType.startsWith("drawable/");
  }

  /**
   * Creates content for an element.
   *
   * @param e  the element.
   * @param bounds  the bounds.
   * @param ot  the output target.
   *
   * @return the content.
   *
   * @throws ContentCreationException if there is a problem with the Content creation.
   */
  public Content createContentForElement(Element e, ElementLayoutInformation bounds,
                                         LayoutSupport ot)
      throws ContentCreationException
  {
    DrawableContainer drawable = (DrawableContainer) e.getValue();
    if (drawable == null)
    {
      return null;
    }

    Point2D point = bounds.getAbsolutePosition();
    Dimension2D iBounds = ElementLayoutInformation.unionMin(bounds.getMaximumSize(),
                                                            bounds.getPreferredSize());
    if (iBounds.getWidth() == 0 && iBounds.getHeight() == 0)
    {
      return null;
    }
    // basic drawable object don't have own bounds, so they cannot define
    // scaling or keep-aspect ratio.
    //
    // this could be a show-stopper for WMF-Drawables, so we'll start subclassing
    // the drawable stuff soon ...

    Rectangle2D drawableBounds = new Rectangle2D.Float(0, 0,
                                                       (float) iBounds.getWidth(),
                                                       (float) iBounds.getHeight());
    DrawableContainer con = new DrawableContainer(drawable.getDrawable(), iBounds, drawableBounds);
    return new DrawableContent(con, point);
  }
}
