/***************************************************************************
*	Brutal Chess
*	- board.cpp
*
*	Authors: Maxwell Lazaroff, Michael Cook, and Joe Flint
*	Date Created : May 3rd, 2005
*	Last Modified: November 5th, 2005
*
*	- description - Implements a chessboard, with methods for checking
*   bounds, etc.
***************************************************************************/

#include <iostream>

#include "board.h"
#include "boardmove.h"
#include "chessplayer.h"

using namespace std;

/***************************************************************************
*	Piece Class Member Functions
***************************************************************************/

//******** ACCESSORS ********

Color        Piece::getColor()     const { return this->_color; }
Unit         Piece::getUnit()      const { return this->_piece; }
unsigned int Piece::getMovecount() const { return this->_movecount; }
bool         Piece::getJustMoved() const { return this->_justMoved; }

string 
Piece::getName() const
{
	switch( this->_piece ) {
		case EMPTY:
			return "Empty";
		case PAWN:
			return "Pawn";
		case KNIGHT:
			return "Knight";
		case BISHOP:
			return "Bishop";
		case ROOK:
			return "Rook";
		case QUEEN:
			return "Queen";
		case KING:
			return "King";
		default:
			exit(1);
	}
}

//******** UTILITIES ********

void Piece::setColor(Color color) { this->_color = color; }
void Piece::setUnit(Unit unit)    { this->_piece = unit;  }
void Piece::setMovecount(unsigned int movecount) { this->_movecount = movecount; }
void Piece::setJustMoved()        { this->_justMoved = true; }
void Piece::unsetJustMoved()      { this->_justMoved = false; }

/***************************************************************************
*	Board Class Member Functions
***************************************************************************/

void 
Board::reset()
/***************************************************************************
*	Board::reset()
*	
*	- Resets the board to its initial setup.
***************************************************************************/
{
	// Clearing the board
    for(uint i = 0; i < 8; i++) {
        for(uint j = 0; j < 8; j++) {
           _board[i][j].setColor(BLANK);
           _board[i][j].setUnit(EMPTY);
		   _board[i][j].setMovecount( 0 );
		   _board[i][j].unsetJustMoved();
        }
    }

	// Setting up the pawns
    for(int i = 0; i < 8; i++)
    {
        _board[0][i].setColor(BLACK);
        _board[1][i].setColor(BLACK);
        _board[1][i].setUnit(PAWN);
        _board[6][i].setColor(WHITE);
        _board[6][i].setUnit(PAWN);
        _board[7][i].setColor(WHITE);
    }

	// Setting up the remaining pieces
	_board[0][0].setUnit(ROOK);
	_board[0][1].setUnit(KNIGHT);
	_board[0][2].setUnit(BISHOP);
	_board[0][3].setUnit(QUEEN);
	_board[0][4].setUnit(KING);
	_board[0][5].setUnit(BISHOP);
	_board[0][6].setUnit(KNIGHT);
	_board[0][7].setUnit(ROOK);

	_board[7][0].setUnit(ROOK);
	_board[7][1].setUnit(KNIGHT);
	_board[7][2].setUnit(BISHOP);
	_board[7][3].setUnit(QUEEN);
	_board[7][4].setUnit(KING);
	_board[7][5].setUnit(BISHOP);
	_board[7][6].setUnit(KNIGHT);
	_board[7][7].setUnit(ROOK);

	return;
}

void 
Board::print()
/***************************************************************************
*	Board::print()
*	
*	- Prints the board to the console.  This is a throw back to the early
*   development days of Brutal Chess when it was all still text based.  
*   We're keeping this around in case we want to add support later for
*   playing over a network with the text based version vs. the graphical
*   version.
***************************************************************************/
{
	// A matrix that we'll fill with characters to visually represent the
	// current state of the chess board.
    char printerboard[25][25];

	// Setting the edges of the squares.
    for(int i = 0; i < 25; i++) {
        for(int j = 0; j < 25; j++) {
            if(i%3 == 0)
                printerboard[i][j] = '-';
            else
                printerboard[i][j] = '|';
        }
    }
//---
    for(int i = 1; i < 25; i += 6) {
        for(int j = 2; j < 25; j += 6)
            printerboard[i][j] = '-';
    }
    for(int i = 2; i < 25; i += 6) {
        for(int j = 1; j < 25; j += 6)
            printerboard[i][j] = '-';
    }
    for(int i = 4; i < 25; i += 6) {
        for(int j = 5; j < 25; j += 6)
            printerboard[i][j] = '-';
    }
    for(int i = 5; i < 25; i += 6) {
        for(int j = 4; j < 25; j += 6)
            printerboard[i][j] = '-';
    }
//---
    for(int i = 4; i < 25; i += 6) {
        for(int j = 2; j < 25; j += 6)
            printerboard[i][j] = ' ';
    }
    for(int i = 5; i < 25; i += 6) {
        for(int j = 1; j < 25; j += 6)
            printerboard[i][j] = ' ';
    }
    for(int i = 1; i < 25; i += 6) {
        for(int j = 5; j < 25; j += 6)
            printerboard[i][j] = ' ';
    }
    for(int i = 2; i < 25; i += 6) {
        for(int j = 4; j < 25; j += 6)
            printerboard[i][j] = ' ';
    }

	// Setting the piece type for non-empty squares.
    for(int i = 1; i < 25; i += 3) {
        for(int j = 1; j < 25; j += 3) {
            if(_board[i/3][j/3].getUnit() == PAWN)
                printerboard[i][j] = 'P';
            if(_board[i/3][j/3].getUnit() == KNIGHT)
                printerboard[i][j] = 'N';
            if(_board[i/3][j/3].getUnit() == BISHOP)
                printerboard[i][j] = 'B';
            if(_board[i/3][j/3].getUnit() == ROOK)
                printerboard[i][j] = 'R';
            if(_board[i/3][j/3].getUnit() == QUEEN)
                printerboard[i][j] = 'Q';
            if(_board[i/3][j/3].getUnit() == KING)
                printerboard[i][j] = 'K';
            if(_board[i/3][j/3].getUnit() == EMPTY) {
                if((i+2)%6 != 0) {
                    if((j+2)%6 != 0)
                        printerboard[i][j] = '-';
                    else
                        printerboard[i][j] = ' ';
                } else {
                    if((j+2)%6 !=0)
                        printerboard[i][j] = ' ';
                    else
                        printerboard[i][j] = '-';
                }
            }
        }
    }
    
	// Setting the piece colors.
    for(int i = 2; i < 25; i += 3) {
        for(int j = 2; j < 25; j += 3) {
            if(_board[i/3][j/3].getColor() == BLACK)
                printerboard[i][j] = 'B';
            else if(_board[i/3][j/3].getColor() == WHITE)
                printerboard[i][j] = 'W';
            else {
                if((i+1)%6 != 0) {
                    if((j+1)%6 != 0)
                        printerboard[i][j] = '-';
                    else
                        printerboard[i][j] = ' ';
                } else {
                    if((j+1)%6 != 0)
                        printerboard[i][j] = ' ';
                    else
                        printerboard[i][j] = '-';
                }
            }
        }
    }

	// Setting up the file indicators.
    cout << endl;
	cout << " 0  1  2  3  4  5  6  7" << endl;
    for(int i = 0; i < 25; i++) {
        for(int j = 0; j < 25; j++)
            cout << printerboard[i][j];
		if ((i+2)%3 == 0)
			cout << ((i+2)/3)-1;
        cout << endl;
    }

	return;
}

void
Board::pawnPromotion()
/***************************************************************************
*	Board::pawnPromotion
*
*	- Looks over board and promotes any pawns that make it to the last rank
*     a queen. Need to implement a way to choose, however.
***************************************************************************/
{
	for(int i = 0; i < 8; i++) {
		if ((_board[7][i].getColor() == BLACK) &&
		    (_board[7][i].getUnit()==PAWN))
			_board[7][i].setUnit(QUEEN);
		if ((_board[0][i].getColor() == WHITE) &&
		    (_board[0][i].getUnit() == PAWN))
			_board[0][i].setUnit(QUEEN);
	}

	return;
}

void 
Board::updateBoard(BoardMove move)
/***************************************************************************
*	Board::updateBoard
*
*	- Takes in a BoardMove, 'move', which has already validated
*	hopefully, and takes care of moving the pieces, updating movecounts, and
*	when neccessary, taking pieces.
***************************************************************************/
{
	// Getting Move Coordinates.
	uint srow = move.getStartx();
	uint scol = move.getStarty();
	uint erow = move.getEndx();
	uint ecol = move.getEndy();

	move.checkFlags( *this );

	// Check for En Passant and handle it.
	if (move.getIsEnPassant()) {
		move.unsetIsEnPassant();
		_board[srow][ecol].setUnit(EMPTY);
		_board[srow][ecol].setColor(BLANK);
		_board[srow][ecol].setMovecount(0);
	}

	// Check for Castling and handle it.
	if (move.getCastling()) {
        move.unsetCastling();

		if ((_board[srow][ecol-2].getUnit() == ROOK) && 
			(_board[srow][ecol-2].getMovecount() == 0)) {
            _board[srow][ecol+1].setUnit(_board[srow][ecol-2].getUnit());
            _board[srow][ecol+1].setColor(_board[srow][ecol-2].getColor());
            _board[srow][ecol+1].setMovecount(_board[srow][ecol-2].getMovecount()+1);
            _board[srow][ecol-2].setUnit(EMPTY);
            _board[srow][ecol-2].setColor(BLANK);
            _board[srow][ecol-2].setMovecount(0);
        } else if((_board[srow][ecol+1].getUnit() == ROOK) && 
			      (_board[srow][ecol+1].getMovecount() == 0)) {
            _board[srow][ecol-1].setUnit(_board[srow][ecol+1].getUnit());
            _board[srow][ecol-1].setColor(_board[srow][ecol+1].getColor());
            _board[srow][ecol-1].setMovecount(_board[srow][ecol+1].getMovecount()+1);
            _board[srow][ecol+1].setUnit(EMPTY);
            _board[srow][ecol+1].setColor(BLANK);
            _board[srow][ecol+1].setMovecount(0);
        }
    }

	// Move to the current 'end' coordinates.
	_board[erow][ecol].setUnit(_board[srow][scol].getUnit());
	_board[erow][ecol].setColor(_board[srow][scol].getColor());
	_board[erow][ecol].setMovecount(_board[srow][scol].getMovecount()+1);
	_board[srow][scol].setUnit(EMPTY);
	_board[srow][scol].setColor(BLANK);
	_board[srow][scol].setMovecount(0);
	
	//Check for any neccessary pawns to promote
	pawnPromotion();
	
	// Do this here, or flags don't get set right
	// Updating 'last moved piece'
	// Unseting them all is slower, but safer
	for( int i = 0; i < BOARDSIZE; i++ )
		for( int j = 0; j < BOARDSIZE; j++ )
			_board[i][j].unsetJustMoved();
	
	_board[erow][ecol].setJustMoved();
	
	return;
}

bool
Board::inDanger(bool is_white, int row, int col) const
/***************************************************************************
*	inDanger
*	
*	- Checks to make sure that the legal move the king made is not illegal
*     by placing himself into check
****************************************************************************/
{   

    //Look up the column for an enemy Rook or Queen
    if (col != 7) {
        for (int i = col+1; i < 8; i++) {
            if (is_white) {
                if (_board[row][i].getColor() == WHITE)
                    break;
                if ((_board[row][i].getColor() == BLACK) && 
				    ((_board[row][i].getUnit() == ROOK) || 
				     (_board[row][i].getUnit() == QUEEN)))
                    return true;
                if (_board[row][i].getColor() == BLACK)
                    break;
            } else {
                if (_board[row][i].getColor() == BLACK)
                    break;
                if ((_board[row][i].getColor() == WHITE) && 
					((_board[row][i].getUnit() == ROOK) || 
					 (_board[row][i].getUnit() == QUEEN)))
                    return true;
                if(_board[row][i].getColor() == WHITE)
                    break;
            }
        }
    }
    //Look down the column for an enemy Rook or Queen
    if(col != 0) {
        for(int i = col-1; i >= 0; i--) {
            if(is_white) {
                if(_board[row][i].getColor() == WHITE)
                    break;
                if(   _board[row][i].getColor() == BLACK
                   &&(_board[row][i].getUnit() == ROOK
                   || _board[row][i].getUnit() == QUEEN) )
                    return true;
                if(_board[row][i].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[row][i].getColor() == BLACK)
                    break;
                if(   _board[row][i].getColor() == WHITE
                   &&(_board[row][i].getUnit() == ROOK
                   || _board[row][i].getUnit() == QUEEN) )
                    return true;
                if(_board[row][i].getColor() == WHITE)
                    break;
            }
        }
    }
    //Look right across the rows for an enemy Rook or Queen
    if(row != 7)
    {
        for(int i=row+1; i < 8; i++)
        {
            if(is_white)
            {
                if(_board[i][col].getColor() == WHITE)
                    break;
                if(   _board[i][col].getColor() == BLACK
                   &&(_board[i][col].getUnit() == ROOK
                   || _board[i][col].getUnit() == QUEEN) )
                    return true;
                if(_board[i][col].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[i][col].getColor() == BLACK)
                    break;
                if(   _board[i][col].getColor() == WHITE
                   &&(_board[i][col].getUnit() == ROOK
                   || _board[i][col].getUnit() == QUEEN) )
                    return true;
                if(_board[i][col].getColor() == WHITE)
                    break;
            }
        }
    }
    //Look left across the rows for an enemy Rook or Queen
    if(row != 0)
    {
        for(int i=row-1; i >= 0; i--)
        {
            if(is_white)
            {
                if(_board[i][col].getColor() == WHITE)
                    break;
                if(   _board[i][col].getColor() == BLACK
                   &&(_board[i][col].getUnit() == ROOK
                   || _board[i][col].getUnit() == QUEEN) )
                    return true;
                if(_board[i][col].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[i][col].getColor() == BLACK)
                    break;
                if(   _board[i][col].getColor() == WHITE
                   &&(_board[i][col].getUnit() == ROOK
                   || _board[i][col].getUnit() == QUEEN) )
                    return true;
                if(_board[i][col].getColor() == WHITE)
                    break;
            }
        }
    }
    
    //Diagonal
    if(row != 7 && col != 7)
    {
        for(int i = 1; (row+i != 8 && col+i !=8); i++)
        {
            if(is_white)
            {
                if(_board[row+i][col+i].getColor() == WHITE)
                    break;
                if(   _board[row+i][col+i].getColor() == BLACK
                   &&(_board[row+i][col+i].getUnit() == BISHOP
                   || _board[row+i][col+i].getUnit() == QUEEN) )
                    return true;
                if(_board[row+i][col+i].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[row+i][col+i].getColor() == BLACK)
                    break;
                if(   _board[row+i][col+i].getColor() == WHITE
                   &&(_board[row+i][col+i].getUnit() == BISHOP
                   || _board[row+i][col+i].getUnit() == QUEEN) )
                    return true;
                if(_board[row+i][col+i].getColor() == WHITE)
                    break;
            }
        }
    }
    if(row != 0 && col != 7)
    {
        for(int i = 1; (row-i >= 0 && col+i !=8); i++)
        {
            if(is_white)
            {
                if(_board[row-i][col+i].getColor() == WHITE)
                    break;
                if(   _board[row-i][col+i].getColor() == BLACK
                   &&(_board[row-i][col+i].getUnit() == BISHOP
                   || _board[row-i][col+i].getUnit() == QUEEN) )
                    return true;
                if(_board[row-i][col+i].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[row-i][col+i].getColor() == BLACK)
                    break;
                if(   _board[row-i][col+i].getColor() == WHITE
                   &&(_board[row-i][col+i].getUnit() == BISHOP
                   || _board[row-i][col+i].getUnit() == QUEEN) )
                    return true;
                if(_board[row-i][col+i].getColor() == WHITE)
                    break;
            }
        }
    }
    if(row != 7 && col != 0)
    {
        for(int i = 1; (row+i != 8 && col-i >=0); i++)
        {
            if(is_white)
            {
                if(_board[row+i][col-i].getColor() == WHITE)
                    break;
                if(   _board[row+i][col-i].getColor() == BLACK
                   &&(_board[row+i][col-i].getUnit() == BISHOP
                   || _board[row+i][col-i].getUnit() == QUEEN) )
                    return true;
                if(_board[row+i][col-i].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[row+i][col-i].getColor() == BLACK)
                    break;
                if(   _board[row+i][col-i].getColor() == WHITE
                   &&(_board[row+i][col-i].getUnit() == BISHOP
                   || _board[row+i][col-i].getUnit() == QUEEN) )
                    return true;
                if(_board[row+i][col-i].getColor() == WHITE)
                    break;
            }
        }
    }
    if(row != 0 && col != 0)
    {
        for(int i = 1; (row-i >=0 && col-i >=0); i++)
        {
            if(is_white)
            {
                if(_board[row-i][col-i].getColor() == WHITE)
                    break;
                if(   _board[row-i][col-i].getColor() == BLACK
                   &&(_board[row-i][col-i].getUnit() == BISHOP
                   || _board[row-i][col-i].getUnit() == QUEEN) )
                    return true;
                if(_board[row-i][col-i].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[row-i][col-i].getColor() == BLACK)
                    break;
                if(   _board[row-i][col-i].getColor() == WHITE
                   &&(_board[row-i][col-i].getUnit() == BISHOP
                   || _board[row-i][col-i].getUnit() == QUEEN) )
                    return true;
                if(_board[row-i][col-i].getColor() == WHITE)
                    break;
            }
        }
    }
    
    //Pawns
    if(is_white)
    {
        if(row != 0 && col != 0)
        {
            if(_board[row-1][col-1].getUnit() == PAWN
               && _board[row-1][col-1].getColor() == BLACK)
               return true;
        }
        if(row != 0 && col != 7)
        {
            if(_board[row-1][col+1].getUnit() == PAWN
               && _board[row-1][col+1].getColor() == BLACK)
               return true;
        }
    }
    else
    {
        if(row != 7 && col != 0)
        {
            if(_board[row+1][col-1].getUnit() == PAWN
               && _board[row+1][col-1].getColor() == WHITE)
               return true;
        }
        if(row != 7 && col != 7)
        {
            if(_board[row+1][col+1].getUnit() == PAWN
               && _board[row+1][col+1].getColor() == WHITE)
               return true;
        }
    }
	
	// Checking to see if any Knights are putting the king into check.
	uint startrow = 0;
	uint startcol = 0;
	BoardMove knightOne(startrow, startcol, row+1, col+2);	
	BoardMove knightTwo(startrow, startcol, row+1, col-2);	
	BoardMove knightThr(startrow, startcol, row+2, col+1);	
	BoardMove knightFou(startrow, startcol, row+2, col-1);	
	BoardMove knightFiv(startrow, startcol, row-1, col+2);	
	BoardMove knightSix(startrow, startcol, row-1, col-2);	
	BoardMove knightSev(startrow, startcol, row-2, col+1);	
	BoardMove knightEig(startrow, startcol, row-2, col-1);	
	if (is_white)
	{
		if (Board::is_in(knightOne))
		{
			if (   _board[row+1][col+2].getUnit() == KNIGHT
				&& _board[row+1][col+2].getColor() == BLACK)
				return true;
		}	
		if (Board::is_in(knightTwo))
		{
			if (   _board[row+1][col-2].getUnit() == KNIGHT
				&& _board[row+1][col-2].getColor() == BLACK)
				return true;
		}		
		if (Board::is_in(knightThr))
		{
			if (   _board[row+2][col+1].getUnit() == KNIGHT
				&& _board[row+2][col+1].getColor() == BLACK)
				return true;
		}
		if (Board::is_in(knightFou))
		{
			if (   _board[row+2][col-1].getUnit() == KNIGHT
				&& _board[row+2][col-1].getColor() == BLACK)
				return true;
		}
		if (Board::is_in(knightFiv))
		{
			if (   _board[row-1][col+2].getUnit() == KNIGHT
				&& _board[row-1][col+2].getColor() == BLACK)
				return true;
		}
		if (Board::is_in(knightSix))
		{
			if (   _board[row-1][col-2].getUnit() == KNIGHT
				&& _board[row-1][col-2].getColor() == BLACK)
				return true;
		}
		if (Board::is_in(knightSev))
		{
			if (   _board[row-2][col+1].getUnit() == KNIGHT
				&& _board[row-2][col+1].getColor() == BLACK)
				return true;
		}
		if (Board::is_in(knightEig))
		{
			if (   _board[row-2][col-1].getUnit() == KNIGHT
				&& _board[row-2][col-1].getColor() == BLACK)
				return true;
		}			
	}
	else
	{
		if (Board::is_in(knightOne))
		{
			if (   _board[row+1][col+2].getUnit() == KNIGHT
				&& _board[row+1][col+2].getColor() == WHITE)
				return true;
		}	
		if (Board::is_in(knightTwo))
		{
			if (   _board[row+1][col-2].getUnit() == KNIGHT
				&& _board[row+1][col-2].getColor() == WHITE)
				return true;
		}		
		if (Board::is_in(knightThr))
		{
			if (   _board[row+2][col+1].getUnit() == KNIGHT
				&& _board[row+2][col+1].getColor() == WHITE)
				return true;
		}
		if (Board::is_in(knightFou))
		{
			if (   _board[row+2][col-1].getUnit() == KNIGHT
				&& _board[row+2][col-1].getColor() == WHITE)
				return true;
		}
		if (Board::is_in(knightFiv))
		{
			if (   _board[row-1][col+2].getUnit() == KNIGHT
				&& _board[row-1][col+2].getColor() == WHITE)
				return true;
		}
		if (Board::is_in(knightSix))
		{
			if (   _board[row-1][col-2].getUnit() == KNIGHT
				&& _board[row-1][col-2].getColor() == WHITE)
				return true;
		}
		if (Board::is_in(knightSev))
		{
			if (   _board[row-2][col+1].getUnit() == KNIGHT
				&& _board[row-2][col+1].getColor() == WHITE)
				return true;
		}
		if (Board::is_in(knightEig))
		{
			if (   _board[row-2][col-1].getUnit() == KNIGHT
				&& _board[row-2][col-1].getColor() == WHITE)
				return true;
		}
	}
	
	// Check to make sure a king isn't trying to move next to another king
	if (is_white)
	{
		if (col != 7)
		{
			if (row != 7)
			{
				if(   _board[row+1][col+1].getUnit() == KING
				   && _board[row+1][col+1].getColor() == BLACK)
					return true;
			}
			if (   _board[row][col+1].getUnit() == KING
				&& _board[row][col+1].getColor() == BLACK)
				return true;
			if (row != 0)
			{
				if(   _board[row-1][col+1].getUnit() == KING
				   && _board[row-1][col+1].getColor() == BLACK)
					return true;
			}
		}
		if (row != 0)
		{
			if(   _board[row-1][col].getUnit() == KING
			   && _board[row-1][col].getColor() == BLACK)
				return true;
			if (col != 0)
			{
				if (   _board[row-1][col-1].getUnit() == KING
					&& _board[row-1][col-1].getColor() == BLACK)
					return true;
			}
		}
		if (col != 0)
		{
			if(   _board[row][col-1].getUnit() == KING
			   && _board[row][col-1].getColor() == BLACK)
				return true;
			if (row != 7)
			{
				if(   _board[row+1][col-1].getUnit() == KING
				   && _board[row+1][col-1].getColor() == BLACK)
					return true;
			}
		}
		if (row != 7)
		{ 
			if(   _board[row+1][col].getUnit() == KING
			   && _board[row+1][col].getColor() == BLACK)
				return true;  
		}
	}
	else
	{
		if (col != 7)
		{
			if (row != 7)
			{
				if(   _board[row+1][col+1].getUnit() == KING
				   && _board[row+1][col+1].getColor() == WHITE)
					return true;
			}
			if (   _board[row][col+1].getUnit() == KING
				&& _board[row][col+1].getColor() == WHITE)
				return true;
			if (row != 0)
			{
				if(   _board[row-1][col+1].getUnit() == KING
				   && _board[row-1][col+1].getColor() == WHITE)
					return true;
			}
		}
		if (row != 0)
		{
			if(   _board[row-1][col].getUnit() == KING
			   && _board[row-1][col].getColor() == WHITE)
				return true;
			if (col != 0)
			{
				if (   _board[row-1][col-1].getUnit() == KING
					&& _board[row-1][col-1].getColor() == WHITE)
					return true;
			}
		}
		if (col != 0)
		{
			if(   _board[row][col-1].getUnit() == KING
			   && _board[row][col-1].getColor() == WHITE)
				return true;
			if (row != 7)
			{
				if(   _board[row+1][col-1].getUnit() == KING
				   && _board[row+1][col-1].getColor() == WHITE)
					return true;
			}
		}
		if (row != 7)
		{ 
			if(   _board[row+1][col].getUnit() == KING
			   && _board[row+1][col].getColor() == WHITE)
				return true;  
		}
	}
	
    return false;
}

uint
Board::inDangerCount(bool is_white, int row, int col) const
/***************************************************************************
*	inDanger
*	
*	- Checks to make sure that the legal move the king made is not illegal
*     by placing himself into check
****************************************************************************/
{   
	uint count = 0;
	
    //Look up the column for an enemy Rook or Queen
    
    if(col != 7)
    {
        for(int i=col+1; i < 8; i++)
        {
            if(is_white)
            {
                if(_board[row][i].getColor() == WHITE)
                    break;
                if(   _board[row][i].getColor() == BLACK
                   &&(_board[row][i].getUnit() == ROOK
                   || _board[row][i].getUnit() == QUEEN) )
                    count++;
                if(_board[row][i].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[row][i].getColor() == BLACK)
                    break;
                if(   _board[row][i].getColor() == WHITE
                   &&(_board[row][i].getUnit() == ROOK
                   || _board[row][i].getUnit() == QUEEN) )
                    count++;
                if(_board[row][i].getColor() == WHITE)
                    break;
            }
        }
    }
    //Look down the column for an enemy Rook or Queen
    if(col != 0)
    {
        for(int i=col-1; i >= 0; i--)
        {
            if(is_white)
            {
                if(_board[row][i].getColor() == WHITE)
                    break;
                if(   _board[row][i].getColor() == BLACK
                   &&(_board[row][i].getUnit() == ROOK
                   || _board[row][i].getUnit() == QUEEN) )
                    count++;
                if(_board[row][i].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[row][i].getColor() == BLACK)
                    break;
                if(   _board[row][i].getColor() == WHITE
                   &&(_board[row][i].getUnit() == ROOK
                   || _board[row][i].getUnit() == QUEEN) )
                    count++;
                if(_board[row][i].getColor() == WHITE)
                    break;
            }
        }
    }
    //Look right across the rows for an enemy Rook or Queen
    if(row != 7)
    {
        for(int i=row+1; i < 8; i++)
        {
            if(is_white)
            {
                if(_board[i][col].getColor() == WHITE)
                    break;
                if(   _board[i][col].getColor() == BLACK
                   &&(_board[i][col].getUnit() == ROOK
                   || _board[i][col].getUnit() == QUEEN) )
                    count++;
                if(_board[i][col].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[i][col].getColor() == BLACK)
                    break;
                if(   _board[i][col].getColor() == WHITE
                   &&(_board[i][col].getUnit() == ROOK
                   || _board[i][col].getUnit() == QUEEN) )
                    count++;
                if(_board[i][col].getColor() == WHITE)
                    break;
            }
        }
    }
    //Look left across the rows for an enemy Rook or Queen
    if(row != 0)
    {
        for(int i=row-1; i >= 0; i--)
        {
            if(is_white)
            {
                if(_board[i][col].getColor() == WHITE)
                    break;
                if(   _board[i][col].getColor() == BLACK
                   &&(_board[i][col].getUnit() == ROOK
                   || _board[i][col].getUnit() == QUEEN) )
                    count++;
                if(_board[i][col].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[i][col].getColor() == BLACK)
                    break;
                if(   _board[i][col].getColor() == WHITE
                   &&(_board[i][col].getUnit() == ROOK
                   || _board[i][col].getUnit() == QUEEN) )
                    count++;
                if(_board[i][col].getColor() == WHITE)
                    break;
            }
        }
    }
    
    //Diagonal
    if(row != 7 && col != 7)
    {
        for(int i = 1; (row+i != 8 && col+i !=8); i++)
        {
            if(is_white)
            {
                if(_board[row+i][col+i].getColor() == WHITE)
                    break;
                if(   _board[row+i][col+i].getColor() == BLACK
                   &&(_board[row+i][col+i].getUnit() == BISHOP
                   || _board[row+i][col+i].getUnit() == QUEEN) )
                    count++;
                if(_board[row+i][col+i].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[row+i][col+i].getColor() == BLACK)
                    break;
                if(   _board[row+i][col+i].getColor() == WHITE
                   &&(_board[row+i][col+i].getUnit() == BISHOP
                   || _board[row+i][col+i].getUnit() == QUEEN) )
                    count++;
                if(_board[row+i][col+i].getColor() == WHITE)
                    break;
            }
        }
    }
    if(row != 0 && col != 7)
    {
        for(int i = 1; (row-i >= 0 && col+i !=8); i++)
        {
            if(is_white)
            {
                if(_board[row-i][col+i].getColor() == WHITE)
                    break;
                if(   _board[row-i][col+i].getColor() == BLACK
                   &&(_board[row-i][col+i].getUnit() == BISHOP
                   || _board[row-i][col+i].getUnit() == QUEEN) )
                    count++;
                if(_board[row-i][col+i].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[row-i][col+i].getColor() == BLACK)
                    break;
                if(   _board[row-i][col+i].getColor() == WHITE
                   &&(_board[row-i][col+i].getUnit() == BISHOP
                   || _board[row-i][col+i].getUnit() == QUEEN) )
                    count++;
                if(_board[row-i][col+i].getColor() == WHITE)
                    break;
            }
        }
    }
    if(row != 7 && col != 0)
    {
        for(int i = 1; (row+i != 8 && col-i >=0); i++)
        {
            if(is_white)
            {
                if(_board[row+i][col-i].getColor() == WHITE)
                    break;
                if(   _board[row+i][col-i].getColor() == BLACK
                   &&(_board[row+i][col-i].getUnit() == BISHOP
                   || _board[row+i][col-i].getUnit() == QUEEN) )
                    count++;
                if(_board[row+i][col-i].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[row+i][col-i].getColor() == BLACK)
                    break;
                if(   _board[row+i][col-i].getColor() == WHITE
                   &&(_board[row+i][col-i].getUnit() == BISHOP
                   || _board[row+i][col-i].getUnit() == QUEEN) )
                    count++;
                if(_board[row+i][col-i].getColor() == WHITE)
                    break;
            }
        }
    }
    if(row != 0 && col != 0)
    {
        for(int i = 1; (row-i >=0 && col-i >=0); i++)
        {
            if(is_white)
            {
                if(_board[row-i][col-i].getColor() == WHITE)
                    break;
                if(   _board[row-i][col-i].getColor() == BLACK
                   &&(_board[row-i][col-i].getUnit() == BISHOP
                   || _board[row-i][col-i].getUnit() == QUEEN) )
                    count++;
                if(_board[row-i][col-i].getColor() == BLACK)
                    break;
            }
            else
            {
                if(_board[row-i][col-i].getColor() == BLACK)
                    break;
                if(   _board[row-i][col-i].getColor() == WHITE
                   &&(_board[row-i][col-i].getUnit() == BISHOP
                   || _board[row-i][col-i].getUnit() == QUEEN) )
                    count++;
                if(_board[row-i][col-i].getColor() == WHITE)
                    break;
            }
        }
    }
    
    //Pawns
    if(is_white)
    {
        if(row != 0 && col != 0)
        {
            if(_board[row-1][col-1].getUnit() == PAWN
               && _board[row-1][col-1].getColor() == BLACK)
               count++;
        }
        if(row != 0 && col != 7)
        {
            if(_board[row-1][col+1].getUnit() == PAWN
               && _board[row-1][col+1].getColor() == BLACK)
               count++;
        }
    }
    else
    {
        if(row != 7 && col != 0)
        {
            if(_board[row+1][col-1].getUnit() == PAWN
               && _board[row+1][col-1].getColor() == WHITE)
               count++;
        }
        if(row != 7 && col != 7)
        {
            if(_board[row+1][col+1].getUnit() == PAWN
               && _board[row+1][col+1].getColor() == WHITE)
               count++;
        }
    }
	
	// Checking to see if any Knights are putting the king into check.
	uint startrow = 0;
	uint startcol = 0;
	BoardMove knightOne(startrow, startcol, row+1, col+2);	
	BoardMove knightTwo(startrow, startcol, row+1, col-2);	
	BoardMove knightThr(startrow, startcol, row+2, col+1);	
	BoardMove knightFou(startrow, startcol, row+2, col-1);	
	BoardMove knightFiv(startrow, startcol, row-1, col+2);	
	BoardMove knightSix(startrow, startcol, row-1, col-2);	
	BoardMove knightSev(startrow, startcol, row-2, col+1);	
	BoardMove knightEig(startrow, startcol, row-2, col-1);	
	if (is_white)
	{
		if (Board::is_in(knightOne))
		{
			if (   _board[row+1][col+2].getUnit() == KNIGHT
				&& _board[row+1][col+2].getColor() == BLACK)
				count++;
		}	
		if (Board::is_in(knightTwo))
		{
			if (   _board[row+1][col-2].getUnit() == KNIGHT
				&& _board[row+1][col-2].getColor() == BLACK)
				count++;
		}		
		if (Board::is_in(knightThr))
		{
			if (   _board[row+2][col+1].getUnit() == KNIGHT
				&& _board[row+2][col+1].getColor() == BLACK)
				count++;
		}
		if (Board::is_in(knightFou))
		{
			if (   _board[row+2][col-1].getUnit() == KNIGHT
				&& _board[row+2][col-1].getColor() == BLACK)
				count++;
		}
		if (Board::is_in(knightFiv))
		{
			if (   _board[row-1][col+2].getUnit() == KNIGHT
				&& _board[row-1][col+2].getColor() == BLACK)
				count++;
		}
		if (Board::is_in(knightSix))
		{
			if (   _board[row-1][col-2].getUnit() == KNIGHT
				&& _board[row-1][col-2].getColor() == BLACK)
				count++;
		}
		if (Board::is_in(knightSev))
		{
			if (   _board[row-2][col+1].getUnit() == KNIGHT
				&& _board[row-2][col+1].getColor() == BLACK)
				count++;
		}
		if (Board::is_in(knightEig))
		{
			if (   _board[row-2][col-1].getUnit() == KNIGHT
				&& _board[row-2][col-1].getColor() == BLACK)
				count++;
		}			
	}
	else
	{
		if (Board::is_in(knightOne))
		{
			if (   _board[row+1][col+2].getUnit() == KNIGHT
				&& _board[row+1][col+2].getColor() == WHITE)
				count++;
		}	
		if (Board::is_in(knightTwo))
		{
			if (   _board[row+1][col-2].getUnit() == KNIGHT
				&& _board[row+1][col-2].getColor() == WHITE)
				count++;
		}		
		if (Board::is_in(knightThr))
		{
			if (   _board[row+2][col+1].getUnit() == KNIGHT
				&& _board[row+2][col+1].getColor() == WHITE)
				count++;
		}
		if (Board::is_in(knightFou))
		{
			if (   _board[row+2][col-1].getUnit() == KNIGHT
				&& _board[row+2][col-1].getColor() == WHITE)
				count++;
		}
		if (Board::is_in(knightFiv))
		{
			if (   _board[row-1][col+2].getUnit() == KNIGHT
				&& _board[row-1][col+2].getColor() == WHITE)
				count++;
		}
		if (Board::is_in(knightSix))
		{
			if (   _board[row-1][col-2].getUnit() == KNIGHT
				&& _board[row-1][col-2].getColor() == WHITE)
				count++;
		}
		if (Board::is_in(knightSev))
		{
			if (   _board[row-2][col+1].getUnit() == KNIGHT
				&& _board[row-2][col+1].getColor() == WHITE)
				count++;
		}
		if (Board::is_in(knightEig))
		{
			if (   _board[row-2][col-1].getUnit() == KNIGHT
				&& _board[row-2][col-1].getColor() == WHITE)
				count++;
		}
	}
	
	// Check to make sure a king isn't trying to move next to another king
	if (is_white)
	{
		if (col != 7)
		{
			if (row != 7)
			{
				if(   _board[row+1][col+1].getUnit() == KING
				   && _board[row+1][col+1].getColor() == BLACK)
					count++;
			}
			if (   _board[row][col+1].getUnit() == KING
				&& _board[row][col+1].getColor() == BLACK)
				count++;
			if (row != 0)
			{
				if(   _board[row-1][col+1].getUnit() == KING
				   && _board[row-1][col+1].getColor() == BLACK)
					count++;
			}
		}
		if (row != 0)
		{
			if(   _board[row-1][col].getUnit() == KING
			   && _board[row-1][col].getColor() == BLACK)
				count++;
			if (col != 0)
			{
				if (   _board[row-1][col-1].getUnit() == KING
					&& _board[row-1][col-1].getColor() == BLACK)
					count++;
			}
		}
		if (col != 0)
		{
			if(   _board[row][col-1].getUnit() == KING
			   && _board[row][col-1].getColor() == BLACK)
				count++;
			if (row != 7)
			{
				if(   _board[row+1][col-1].getUnit() == KING
				   && _board[row+1][col-1].getColor() == BLACK)
					count++;
			}
		}
		if (row != 7)
		{ 
			if(   _board[row+1][col].getUnit() == KING
			   && _board[row+1][col].getColor() == BLACK)
				count++;  
		}
	}
	else
	{
		if (col != 7)
		{
			if (row != 7)
			{
				if(   _board[row+1][col+1].getUnit() == KING
				   && _board[row+1][col+1].getColor() == WHITE)
					count++;
			}
			if (   _board[row][col+1].getUnit() == KING
				&& _board[row][col+1].getColor() == WHITE)
				count++;
			if (row != 0)
			{
				if(   _board[row-1][col+1].getUnit() == KING
				   && _board[row-1][col+1].getColor() == WHITE)
					count++;
			}
		}
		if (row != 0)
		{
			if(   _board[row-1][col].getUnit() == KING
			   && _board[row-1][col].getColor() == WHITE)
				count++;
			if (col != 0)
			{
				if (   _board[row-1][col-1].getUnit() == KING
					&& _board[row-1][col-1].getColor() == WHITE)
					count++;
			}
		}
		if (col != 0)
		{
			if(   _board[row][col-1].getUnit() == KING
			   && _board[row][col-1].getColor() == WHITE)
				count++;
			if (row != 7)
			{
				if(   _board[row+1][col-1].getUnit() == KING
				   && _board[row+1][col-1].getColor() == WHITE)
					count++;
			}
		}
		if (row != 7)
		{ 
			if(   _board[row+1][col].getUnit() == KING
			   && _board[row+1][col].getColor() == WHITE)
				count++;  
		}
	}
	
    return count;
}


bool
Board::is_in(BoardMove move) const
{
    if(    move.getStartx() < 8 && move.getStartx() >=0
		&& move.getStarty() < 8 && move.getStarty() >=0
	    && move.getEndx() < 8 && move.getEndy() >= 0
		&& move.getEndy() < 8 && move.getEndy() >= 0   )
		return true;
	else
		return false;
}

// End of file board.cpp
