﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>
#include <aws/iot/model/AuditMitigationActionsTaskStatus.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoT {
namespace Model {

/**
 */
class ListAuditMitigationActionsTasksRequest : public IoTRequest {
 public:
  AWS_IOT_API ListAuditMitigationActionsTasksRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAuditMitigationActionsTasks"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  AWS_IOT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>Specify this filter to limit results to tasks that were applied to results
   * for a specific audit.</p>
   */
  inline const Aws::String& GetAuditTaskId() const { return m_auditTaskId; }
  inline bool AuditTaskIdHasBeenSet() const { return m_auditTaskIdHasBeenSet; }
  template <typename AuditTaskIdT = Aws::String>
  void SetAuditTaskId(AuditTaskIdT&& value) {
    m_auditTaskIdHasBeenSet = true;
    m_auditTaskId = std::forward<AuditTaskIdT>(value);
  }
  template <typename AuditTaskIdT = Aws::String>
  ListAuditMitigationActionsTasksRequest& WithAuditTaskId(AuditTaskIdT&& value) {
    SetAuditTaskId(std::forward<AuditTaskIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify this filter to limit results to tasks that were applied to a specific
   * audit finding.</p>
   */
  inline const Aws::String& GetFindingId() const { return m_findingId; }
  inline bool FindingIdHasBeenSet() const { return m_findingIdHasBeenSet; }
  template <typename FindingIdT = Aws::String>
  void SetFindingId(FindingIdT&& value) {
    m_findingIdHasBeenSet = true;
    m_findingId = std::forward<FindingIdT>(value);
  }
  template <typename FindingIdT = Aws::String>
  ListAuditMitigationActionsTasksRequest& WithFindingId(FindingIdT&& value) {
    SetFindingId(std::forward<FindingIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify this filter to limit results to tasks that are in a specific
   * state.</p>
   */
  inline AuditMitigationActionsTaskStatus GetTaskStatus() const { return m_taskStatus; }
  inline bool TaskStatusHasBeenSet() const { return m_taskStatusHasBeenSet; }
  inline void SetTaskStatus(AuditMitigationActionsTaskStatus value) {
    m_taskStatusHasBeenSet = true;
    m_taskStatus = value;
  }
  inline ListAuditMitigationActionsTasksRequest& WithTaskStatus(AuditMitigationActionsTaskStatus value) {
    SetTaskStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return at one time. The default is 25.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListAuditMitigationActionsTasksRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next set of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAuditMitigationActionsTasksRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify this filter to limit results to tasks that began on or after a
   * specific date and time.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  ListAuditMitigationActionsTasksRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify this filter to limit results to tasks that were completed or canceled
   * on or before a specific date and time.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  ListAuditMitigationActionsTasksRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_auditTaskId;

  Aws::String m_findingId;

  AuditMitigationActionsTaskStatus m_taskStatus{AuditMitigationActionsTaskStatus::NOT_SET};

  int m_maxResults{0};

  Aws::String m_nextToken;

  Aws::Utils::DateTime m_startTime{};

  Aws::Utils::DateTime m_endTime{};
  bool m_auditTaskIdHasBeenSet = false;
  bool m_findingIdHasBeenSet = false;
  bool m_taskStatusHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_startTimeHasBeenSet = false;
  bool m_endTimeHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
