/* $Id: ai_scanner.hpp 15060 2009-01-13 15:44:36Z smatz $ */

/** @file ai_scanner.hpp declarations of the class for AI scanner */

#ifndef AI_SCANNER_HPP
#define AI_SCANNER_HPP

#include <map>

class AIScanner {
public:
	AIScanner();
	~AIScanner();

	/**
	 * Import a library inside the Squirrel VM.
	 */
	bool ImportLibrary(const char *library, const char *class_name, int version, HSQUIRRELVM vm, class AIController *controller);

	/**
	 * Register a library to be put in the available list.
	 */
	void RegisterLibrary(class AILibrary *library);

	/**
	 * Register an AI to be put in the available list.
	 */
	void RegisterAI(class AIInfo *info);

	void SetDummyAI(class AIInfo *info) { this->info_dummy = info; }

	/**
	 * Remove an AI from the available list.
	 */
	void UnregisterAI(class AIInfo *info);

	/**
	 * Select a Random AI.
	 */
	class AIInfo *SelectRandomAI();

	/**
	 * Find an AI by name.
	 */
	class AIInfo *FindInfo(const char *name, int version);

	/**
	 * Get the list of available AIs for the console.
	 */
	char *GetAIConsoleList(char *p, const char *last);

	/**
	 * Get the list of all registered AIs.
	 */
	const AIInfoList *GetAIInfoList() { return &this->info_list; }

	/**
	 * Get the engine of the main squirrel handler (it indexes all avialable squirrels).
	 */
	class Squirrel *GetEngine() { return this->engine; }

	/**
	 * Get the current script the ScanDir is looking at.
	 */
	const char *GetCurrentScript() { return this->current_script; }

	/**
	 * Get the directory of the current script the ScanDir is looking at.
	 */
	const char *GetCurrentDirName() { return this->current_dir; }

	/**
	 * Rescan the AI dir for scripts.
	 */
	void RescanAIDir();

private:
	typedef std::map<const char *, class AILibrary *, ltstr> AILibraryList;

	/**
	 * Scan the AI dir for scripts.
	 */
	void ScanAIDir();

	/**
	 * Scan a dir for AIs.
	 *  For non-library-scan, if an AI is found, AIInfo is created, and the AI
	 *    is registered to the central system.
	 *  For library-scan, if a library is found, AILibrary is created, and the
	 *    library is registered to the central system.
	 */
	void ScanDir(const char *dirname, bool library_dir, char *library_depth = NULL);

	AIInfoList info_list;
	AIInfo *info_dummy;
	AILibraryList library_list;
	class Squirrel *engine;
	char *current_script;
	char *current_dir;
};

#endif /* AI_SCANNER_HPP */
