/* $Id: 8bpp_optimized.cpp 13354 2008-06-01 19:36:55Z smatz $ */

/** @file 8bpp_optimized.cpp Implementation of the optimized 8 bpp blitter. */

#include "../stdafx.h"
#include "../zoom_func.h"
#include "../debug.h"
#include "../core/alloc_func.hpp"
#include "../core/math_func.hpp"
#include "8bpp_optimized.hpp"

static FBlitter_8bppOptimized iFBlitter_8bppOptimized;

void Blitter_8bppOptimized::Draw(Blitter::BlitterParams *bp, BlitterMode mode, ZoomLevel zoom)
{
	/* Find the offset of this zoom-level */
	uint offset = ((const uint8 *)bp->sprite)[(int)(zoom - ZOOM_LVL_BEGIN) * 2] | ((const byte *)bp->sprite)[(int)(zoom - ZOOM_LVL_BEGIN) * 2 + 1] << 8;

	/* Find where to start reading in the source sprite */
	const uint8 *src = (const uint8 *)bp->sprite + offset;
	uint8 *dst_line = (uint8 *)bp->dst + bp->top * bp->pitch + bp->left;

	/* Skip over the top lines in the source image */
	for (int y = 0; y < bp->skip_top; y++) {
		for (;;) {
			uint trans = *src++;
			uint pixels = *src++;
			if (trans == 0 && pixels == 0) break;
			src += pixels;
		}
	}

	const uint8 *src_next = src;

	for (int y = 0; y < bp->height; y++) {
		uint8 *dst = dst_line;
		dst_line += bp->pitch;

		uint skip_left = bp->skip_left;
		int width = bp->width;

		for (;;) {
			src = src_next;
			uint trans = *src++;
			uint pixels = *src++;
			src_next = src + pixels;
			if (trans == 0 && pixels == 0) break;
			if (width <= 0) continue;

			if (skip_left != 0) {
				if (skip_left < trans) {
					trans -= skip_left;
					skip_left = 0;
				} else {
					skip_left -= trans;
					trans = 0;
				}
				if (skip_left < pixels) {
					src += skip_left;
					pixels -= skip_left;
					skip_left = 0;
				} else {
					src += pixels;
					skip_left -= pixels;
					pixels = 0;
				}
			}
			if (skip_left != 0) continue;

			/* Skip transparent pixels */
			dst += trans;
			width -= trans;
			if (width <= 0) continue;
			pixels = min<uint>(pixels, (uint)width);
			width -= pixels;

			switch (mode) {
				case BM_COLOUR_REMAP:
					for (uint x = 0; x < pixels; x++) {
						if (bp->remap[*src] != 0) *dst = bp->remap[*src];
						dst++; src++;
					}
					break;

				case BM_TRANSPARENT:
					for (uint x = 0; x < pixels; x++) {
						*dst = bp->remap[*dst];
						dst++; src++;
					}
					break;

				default:
					memcpy(dst, src, pixels);
					dst += pixels; src += pixels;
					break;
			}
		}
	}
}

Sprite *Blitter_8bppOptimized::Encode(SpriteLoader::Sprite *sprite, Blitter::AllocatorProc *allocator)
{
	/* Make memory for all zoom-levels */
	uint memory = (int)(ZOOM_LVL_END - ZOOM_LVL_BEGIN) * sizeof(uint16);

	for (ZoomLevel i = ZOOM_LVL_BEGIN; i < ZOOM_LVL_END; i++) {
		memory += UnScaleByZoom(sprite->height, i) * UnScaleByZoom(sprite->width, i);
	}

	/* We have no idea how much memory we really need, so just guess something */
	memory *= 5;
	byte *temp_dst = MallocT<byte>(memory);
	byte *dst = &temp_dst[(ZOOM_LVL_END - ZOOM_LVL_BEGIN) * 2];

	/* Make the sprites per zoom-level */
	for (ZoomLevel i = ZOOM_LVL_BEGIN; i < ZOOM_LVL_END; i++) {
		/* Store the index table */
		uint index = dst - temp_dst;
		temp_dst[i * 2] = index & 0xFF;
		temp_dst[i * 2 + 1] = (index >> 8) & 0xFF;

		/* cache values, because compiler can't cache it */
		int scaled_height = UnScaleByZoom(sprite->height, i);
		int scaled_width  = UnScaleByZoom(sprite->width,  i);
		int scaled_1      =   ScaleByZoom(1,              i);

		for (int y = 0; y < scaled_height; y++) {
			uint trans = 0;
			uint pixels = 0;
			uint last_colour = 0;
			byte *count_dst = NULL;

			/* Store the scaled image */
			const SpriteLoader::CommonPixel *src = &sprite->data[ScaleByZoom(y, i) * sprite->width];
			const SpriteLoader::CommonPixel *src_end = &src[sprite->width];

			for (int x = 0; x < scaled_width; x++) {
				uint colour = 0;

				/* Get the color keeping in mind the zoom-level */
				for (int j = 0; j < scaled_1; j++) {
					if (src->m != 0) colour = src->m;
					/* Because of the scaling it might happen we read outside the buffer. Avoid that. */
					if (++src == src_end) break;
				}

				if (last_colour == 0 || colour == 0 || pixels == 255) {
					if (count_dst != NULL) {
						/* Write how many non-transparent bytes we get */
						*count_dst = pixels;
						pixels = 0;
						count_dst = NULL;
					}
					/* As long as we find transparency bytes, keep counting */
					if (colour == 0) {
						last_colour = 0;
						trans++;
						continue;
					}
					/* No longer transparency, so write the amount of transparent bytes */
					*dst = trans;
					dst++;
					trans = 0;
					/* Reserve a byte for the pixel counter */
					count_dst = dst;
					dst++;
				}
				last_colour = colour;
				pixels++;
				*dst = colour;
				dst++;
			}

			if (count_dst != NULL) *count_dst = pixels;

			/* Write line-ending */
			*dst = 0; dst++;
			*dst = 0; dst++;
		}
	}

	uint size = dst - temp_dst;

	/* Safety check, to make sure we guessed the size correctly */
	assert(size < memory);

	/* Allocate the exact amount of memory we need */
	Sprite *dest_sprite = (Sprite *)allocator(sizeof(*dest_sprite) + size);

	dest_sprite->height = sprite->height;
	dest_sprite->width  = sprite->width;
	dest_sprite->x_offs = sprite->x_offs;
	dest_sprite->y_offs = sprite->y_offs;
	memcpy(dest_sprite->data, temp_dst, size);
	free(temp_dst);

	return dest_sprite;
}
