/* $Id: texteff.cpp 11777 2008-01-07 14:23:25Z rubidium $ */

/** @file texteff.cpp */

#include "stdafx.h"
#include "openttd.h"
#include "tile_cmd.h"
#include "landscape.h"
#include "viewport.h"
#include "saveload.h"
#include "console.h"
#include "variables.h"
#include "table/sprites.h"
#include "blitter/factory.hpp"
#include <stdarg.h> /* va_list */
#include "texteff.hpp"
#include "video/video_driver.hpp"
#include "transparency.h"
#include "strings_func.h"
#include "core/alloc_func.hpp"
#include "date_func.h"
#include "functions.h"

enum {
	MAX_TEXTMESSAGE_LENGTH = 200,
	INIT_NUM_TEXT_MESSAGES =  20,
	MAX_CHAT_MESSAGES      =  10,
};

struct TextEffect {
	StringID string_id;
	int32 x;
	int32 y;
	int32 right;
	int32 bottom;
	uint16 duration;
	uint64 params_1;
	uint64 params_2;
	TextEffectMode mode;
};


struct ChatMessage {
	char message[MAX_TEXTMESSAGE_LENGTH];
	uint16 color;
	Date end_date;
};

/* used for text effects */
static TextEffect *_text_effect_list = NULL;
static uint16 _num_text_effects = INIT_NUM_TEXT_MESSAGES;

/* used for chat window */
static ChatMessage _chatmsg_list[MAX_CHAT_MESSAGES];
static bool _chatmessage_dirty = false;
static bool _chatmessage_visible = false;

/* The chatbox grows from the bottom so the coordinates are pixels from
 * the left and pixels from the bottom. The height is the maximum height */
static const PointDimension _chatmsg_box = {10, 30, 500, 150};
static uint8 _chatmessage_backup[150 * 500 * 6]; // (height * width)

static inline uint GetChatMessageCount()
{
	uint i;

	for (i = 0; i < MAX_CHAT_MESSAGES; i++) {
		if (_chatmsg_list[i].message[0] == '\0') break;
	}

	return i;
}

/* Add a text message to the 'chat window' to be shown
 * @param color The colour this message is to be shown in
 * @param duration The duration of the chat message in game-days
 * @param message message itself in printf() style */
void CDECL AddChatMessage(uint16 color, uint8 duration, const char *message, ...)
{
	char buf[MAX_TEXTMESSAGE_LENGTH];
	const char *bufp;
	va_list va;
	uint msg_count;
	uint16 lines;

	va_start(va, message);
	vsnprintf(buf, lengthof(buf), message, va);
	va_end(va);


	Utf8TrimString(buf, MAX_TEXTMESSAGE_LENGTH);

	/* Force linebreaks for strings that are too long */
	lines = GB(FormatStringLinebreaks(buf, _chatmsg_box.width - 8), 0, 16) + 1;
	if (lines >= MAX_CHAT_MESSAGES) return;

	msg_count = GetChatMessageCount();
	/* We want to add more chat messages than there is free space for, remove 'old' */
	if (lines > MAX_CHAT_MESSAGES - msg_count) {
		int i = lines - (MAX_CHAT_MESSAGES - msg_count);
		memmove(&_chatmsg_list[0], &_chatmsg_list[i], sizeof(_chatmsg_list[0]) * (msg_count - i));
		msg_count = MAX_CHAT_MESSAGES - lines;
	}

	for (bufp = buf; lines != 0; lines--) {
		ChatMessage *cmsg = &_chatmsg_list[msg_count++];
		ttd_strlcpy(cmsg->message, bufp, sizeof(cmsg->message));

		/* The default colour for a message is player colour. Replace this with
		 * white for any additional lines */
		cmsg->color = (bufp == buf && color & IS_PALETTE_COLOR) ? color : (0x1D - 15) | IS_PALETTE_COLOR;
		cmsg->end_date = _date + duration;

		bufp += strlen(bufp) + 1; // jump to 'next line' in the formatted string
	}

	_chatmessage_dirty = true;
}

void InitChatMessage()
{
	uint i;

	for (i = 0; i < MAX_CHAT_MESSAGES; i++) {
		_chatmsg_list[i].message[0] = '\0';
	}
}

/** Hide the chatbox */
void UndrawChatMessage()
{
	if (_chatmessage_visible) {
		Blitter *blitter = BlitterFactoryBase::GetCurrentBlitter();
		/* Sometimes we also need to hide the cursor
		 *   This is because both textmessage and the cursor take a shot of the
		 *   screen before drawing.
		 *   Now the textmessage takes his shot and paints his data before the cursor
		 *   does, so in the shot of the cursor is the screen-data of the textmessage
		 *   included when the cursor hangs somewhere over the textmessage. To
		 *   avoid wrong repaints, we undraw the cursor in that case, and everything
		 *   looks nicely ;)
		 * (and now hope this story above makes sense to you ;))
		 */

		if (_cursor.visible) {
			if (_cursor.draw_pos.x + _cursor.draw_size.x >= _chatmsg_box.x &&
				_cursor.draw_pos.x <= _chatmsg_box.x + _chatmsg_box.width &&
				_cursor.draw_pos.y + _cursor.draw_size.y >= _screen.height - _chatmsg_box.y - _chatmsg_box.height &&
				_cursor.draw_pos.y <= _screen.height - _chatmsg_box.y) {
				UndrawMouseCursor();
			}
		}

		int x      = _chatmsg_box.x;
		int y      = _screen.height - _chatmsg_box.y - _chatmsg_box.height;
		int width  = _chatmsg_box.width;
		int height = _chatmsg_box.height;
		if (y < 0) {
			height = max(height + y, min(_chatmsg_box.height, _screen.height));
			y = 0;
		}
		if (x + width >= _screen.width) {
			width = _screen.width - x;
		}
		if (width <= 0 || height <= 0) return;

		_chatmessage_visible = false;
		/* Put our 'shot' back to the screen */
		blitter->CopyFromBuffer(blitter->MoveTo(_screen.dst_ptr, x, y), _chatmessage_backup, width, height);
		/* And make sure it is updated next time */
		_video_driver->MakeDirty(x, y, width, height);

		_chatmessage_dirty = true;
	}
}

/** Check if a message is expired every day */
void ChatMessageDailyLoop()
{
	uint i;

	for (i = 0; i < MAX_CHAT_MESSAGES; i++) {
		ChatMessage *cmsg = &_chatmsg_list[i];
		if (cmsg->message[0] == '\0') continue;

		/* Message has expired, remove from the list */
		if (cmsg->end_date < _date) {
			/* Move the remaining messages over the current message */
			if (i != MAX_CHAT_MESSAGES - 1) memmove(cmsg, cmsg + 1, sizeof(*cmsg) * (MAX_CHAT_MESSAGES - i - 1));

			/* Mark the last item as empty */
			_chatmsg_list[MAX_CHAT_MESSAGES - 1].message[0] = '\0';
			_chatmessage_dirty = true;

			/* Go one item back, because we moved the array 1 to the left */
			i--;
		}
	}
}

/** Draw the chat message-box */
void DrawChatMessage()
{
	Blitter *blitter = BlitterFactoryBase::GetCurrentBlitter();
	if (!_chatmessage_dirty) return;

	/* First undraw if needed */
	UndrawChatMessage();

	if (_iconsole_mode == ICONSOLE_FULL) return;

	/* Check if we have anything to draw at all */
	uint count = GetChatMessageCount();
	if (count == 0) return;

	int x      = _chatmsg_box.x;
	int y      = _screen.height - _chatmsg_box.y - _chatmsg_box.height;
	int width  = _chatmsg_box.width;
	int height = _chatmsg_box.height;
	if (y < 0) {
		height = max(height + y, min(_chatmsg_box.height, _screen.height));
		y = 0;
	}
	if (x + width >= _screen.width) {
		width = _screen.width - x;
	}
	if (width <= 0 || height <= 0) return;

	assert(blitter->BufferSize(width, height) < (int)sizeof(_chatmessage_backup));

	/* Make a copy of the screen as it is before painting (for undraw) */
	blitter->CopyToBuffer(blitter->MoveTo(_screen.dst_ptr, x, y), _chatmessage_backup, width, height);

	_cur_dpi = &_screen; // switch to _screen painting

	/* Paint a half-transparent box behind the chat messages */
	GfxFillRect(
			_chatmsg_box.x,
			_screen.height - _chatmsg_box.y - count * 13 - 2,
			_chatmsg_box.x + _chatmsg_box.width - 1,
			_screen.height - _chatmsg_box.y - 2,
			PALETTE_TO_TRANSPARENT | (1 << USE_COLORTABLE) // black, but with some alpha for background
		);

	/* Paint the chat messages starting with the lowest at the bottom */
	for (uint y = 13; count-- != 0; y += 13) {
		DoDrawString(_chatmsg_list[count].message, _chatmsg_box.x + 3, _screen.height - _chatmsg_box.y - y + 1, _chatmsg_list[count].color);
	}

	/* Make sure the data is updated next flush */
	_video_driver->MakeDirty(x, y, width, height);

	_chatmessage_visible = true;
	_chatmessage_dirty = false;
}

/* Text Effects */
/**
 * Mark the area of the text effect as dirty.
 *
 * This function marks the area of a text effect as dirty for repaint.
 *
 * @param te The TextEffect to mark the area dirty
 * @ingroup dirty
 */
static void MarkTextEffectAreaDirty(TextEffect *te)
{
	/* Width and height of the text effect are doubled, so they are correct in both zoom out levels 1x and 2x. */
	MarkAllViewportsDirty(
		te->x,
		te->y - 1,
		(te->right - te->x)*2 + te->x + 1,
		(te->bottom - (te->y - 1)) * 2 + (te->y - 1) + 1
	);
}

TextEffectID AddTextEffect(StringID msg, int x, int y, uint16 duration, TextEffectMode mode)
{
	TextEffect *te;
	int w;
	char buffer[100];
	TextEffectID i;

	if (_game_mode == GM_MENU) return INVALID_TE_ID;

	/* Look for a free spot in the text effect array */
	for (i = 0; i < _num_text_effects; i++) {
		if (_text_effect_list[i].string_id == INVALID_STRING_ID) break;
	}

	/* If there is none found, we grow the array */
	if (i == _num_text_effects) {
		_num_text_effects += 25;
		_text_effect_list = ReallocT<TextEffect>(_text_effect_list, _num_text_effects);
		for (; i < _num_text_effects; i++) _text_effect_list[i].string_id = INVALID_STRING_ID;
		i = _num_text_effects - 1;
	}

	te = &_text_effect_list[i];

	/* Start defining this object */
	te->string_id = msg;
	te->duration = duration;
	te->y = y - 5;
	te->bottom = y + 5;
	te->params_1 = GetDParam(0);
	te->params_2 = GetDParam(4);
	te->mode = mode;

	GetString(buffer, msg, lastof(buffer));
	w = GetStringBoundingBox(buffer).width;

	te->x = x - (w >> 1);
	te->right = x + (w >> 1) - 1;
	MarkTextEffectAreaDirty(te);

	return i;
}

void UpdateTextEffect(TextEffectID te_id, StringID msg)
{
	assert(te_id < _num_text_effects);
	TextEffect *te;

	/* Update details */
	te = &_text_effect_list[te_id];
	te->string_id = msg;
	te->params_1 = GetDParam(0);
	te->params_2 = GetDParam(4);

	/* Update width of text effect */
	char buffer[100];
	GetString(buffer, msg, lastof(buffer));
	int w = GetStringBoundingBox(buffer).width;

	/* Only allow to make it broader, so it completely covers the old text. That avoids remnants of the old text. */
	int right_new = te->x + w;
	if (te->right < right_new) te->right = right_new;

	MarkTextEffectAreaDirty(te);
}

void RemoveTextEffect(TextEffectID te_id)
{
	assert(te_id < _num_text_effects);
	TextEffect *te;

	te = &_text_effect_list[te_id];
	MarkTextEffectAreaDirty(te);
	te->string_id = INVALID_STRING_ID;
}

static void MoveTextEffect(TextEffect *te)
{
	/* Never expire for duration of 0xFFFF */
	if (te->duration == 0xFFFF) return;
	if (te->duration < 8) {
		te->string_id = INVALID_STRING_ID;
	} else {
		te->duration -= 8;
		te->y--;
		te->bottom--;
	}
	MarkTextEffectAreaDirty(te);
}

void MoveAllTextEffects()
{
	for (TextEffectID i = 0; i < _num_text_effects; i++) {
		TextEffect *te = &_text_effect_list[i];
		if (te->string_id != INVALID_STRING_ID && te->mode == TE_RISING) MoveTextEffect(te);
	}
}

void InitTextEffects()
{
	if (_text_effect_list == NULL) _text_effect_list = MallocT<TextEffect>(_num_text_effects);

	for (TextEffectID i = 0; i < _num_text_effects; i++) _text_effect_list[i].string_id = INVALID_STRING_ID;
}

void DrawTextEffects(DrawPixelInfo *dpi)
{
	switch (dpi->zoom) {
		case ZOOM_LVL_NORMAL:
			for (TextEffectID i = 0; i < _num_text_effects; i++) {
				TextEffect *te = &_text_effect_list[i];
				if (te->string_id != INVALID_STRING_ID &&
						dpi->left <= te->right &&
						dpi->top  <= te->bottom &&
						dpi->left + dpi->width  > te->x &&
						dpi->top  + dpi->height > te->y) {
					if (te->mode == TE_RISING || (_patches.loading_indicators && !IsTransparencySet(TO_LOADING))) {
						AddStringToDraw(te->x, te->y, te->string_id, te->params_1, te->params_2);
					}
				}
			}
			break;

		case ZOOM_LVL_OUT_2X:
			for (TextEffectID i = 0; i < _num_text_effects; i++) {
				TextEffect *te = &_text_effect_list[i];
				if (te->string_id != INVALID_STRING_ID &&
						dpi->left <= te->right  * 2 - te->x &&
						dpi->top  <= te->bottom * 2 - te->y &&
						dpi->left + dpi->width  > te->x &&
						dpi->top  + dpi->height > te->y) {
					if (te->mode == TE_RISING || (_patches.loading_indicators && !IsTransparencySet(TO_LOADING))) {
						AddStringToDraw(te->x, te->y, (StringID)(te->string_id - 1), te->params_1, te->params_2);
					}
				}
			}
			break;

		case ZOOM_LVL_OUT_4X:
		case ZOOM_LVL_OUT_8X:
			break;

		default: NOT_REACHED();
	}
}

/** The table/list with animated tiles. */
TileIndex *_animated_tile_list = NULL;
/** The number of animated tiles in the current state. */
uint _animated_tile_count = 0;
/** The number of slots for animated tiles allocated currently. */
static uint _animated_tile_allocated = 0;

/**
 * Removes the given tile from the animated tile table.
 * @param tile the tile to remove
 */
void DeleteAnimatedTile(TileIndex tile)
{
	for (TileIndex *ti = _animated_tile_list; ti < _animated_tile_list + _animated_tile_count; ti++) {
		if (tile == *ti) {
			/* Remove the hole
			 * The order of the remaining elements must stay the same, otherwise the animation loop
			 * may miss a tile; that's why we must use memmove instead of just moving the last element.
			 */
			memmove(ti, ti + 1, (_animated_tile_list + _animated_tile_count - (ti + 1)) * sizeof(*ti));
			_animated_tile_count--;
			MarkTileDirtyByTile(tile);
			return;
		}
	}
}

/**
 * Add the given tile to the animated tile table (if it does not exist
 * on that table yet). Also increases the size of the table if necessary.
 * @param tile the tile to make animated
 */
void AddAnimatedTile(TileIndex tile)
{
	MarkTileDirtyByTile(tile);

	for (const TileIndex *ti = _animated_tile_list; ti < _animated_tile_list + _animated_tile_count; ti++) {
		if (tile == *ti) return;
	}

	/* Table not large enough, so make it larger */
	if (_animated_tile_count == _animated_tile_allocated) {
		_animated_tile_allocated *= 2;
		_animated_tile_list = ReallocT<TileIndex>(_animated_tile_list, _animated_tile_allocated);
	}

	_animated_tile_list[_animated_tile_count] = tile;
	_animated_tile_count++;
}

/**
 * Animate all tiles in the animated tile list, i.e.\ call AnimateTile on them.
 */
void AnimateAnimatedTiles()
{
	const TileIndex *ti = _animated_tile_list;
	while (ti < _animated_tile_list + _animated_tile_count) {
		const TileIndex curr = *ti;
		AnimateTile(curr);
		/* During the AnimateTile call, DeleteAnimatedTile could have been called,
		 * deleting an element we've already processed and pushing the rest one
		 * slot to the left. We can detect this by checking whether the index
		 * in the current slot has changed - if it has, an element has been deleted,
		 * and we should process the current slot again instead of going forward.
		 * NOTE: this will still break if more than one animated tile is being
		 *       deleted during the same AnimateTile call, but no code seems to
		 *       be doing this anyway.
		 */
		if (*ti == curr) ++ti;
	}
}

/**
 * Initialize all animated tile variables to some known begin point
 */
void InitializeAnimatedTiles()
{
	_animated_tile_list = ReallocT<TileIndex>(_animated_tile_list, 256);
	_animated_tile_count = 0;
	_animated_tile_allocated = 256;
}

/**
 * Save the ANIT chunk.
 */
static void Save_ANIT()
{
	SlSetLength(_animated_tile_count * sizeof(*_animated_tile_list));
	SlArray(_animated_tile_list, _animated_tile_count, SLE_UINT32);
}

/**
 * Load the ANIT chunk; the chunk containing the animated tiles.
 */
static void Load_ANIT()
{
	/* Before version 80 we did NOT have a variable length animated tile table */
	if (CheckSavegameVersion(80)) {
		/* In pre version 6, we has 16bit per tile, now we have 32bit per tile, convert it ;) */
		SlArray(_animated_tile_list, 256, CheckSavegameVersion(6) ? (SLE_FILE_U16 | SLE_VAR_U32) : SLE_UINT32);

		for (_animated_tile_count = 0; _animated_tile_count < 256; _animated_tile_count++) {
			if (_animated_tile_list[_animated_tile_count] == 0) break;
		}
		return;
	}

	_animated_tile_count = SlGetFieldLength() / sizeof(*_animated_tile_list);

	/* Determine a nice rounded size for the amount of allocated tiles */
	_animated_tile_allocated = 256;
	while (_animated_tile_allocated < _animated_tile_count) _animated_tile_allocated *= 2;

	_animated_tile_list = ReallocT<TileIndex>(_animated_tile_list, _animated_tile_allocated);
	SlArray(_animated_tile_list, _animated_tile_count, SLE_UINT32);
}

/**
 * "Definition" imported by the saveload code to be able to load and save
 * the animated tile table.
 */
extern const ChunkHandler _animated_tile_chunk_handlers[] = {
	{ 'ANIT', Save_ANIT, Load_ANIT, CH_RIFF | CH_LAST},
};
